#ifdef __GNUG__
#include <values.h>
#else
#define MAXDOUBLE 1.701411834604692293e+38
#endif
#include "random.h"
#include "mx.h"

// BLAS routines
extern "C" double ddot(int *n,double *dx,int *incx,double *dy,int *incy);

// LINPACK routines
extern "C" void dgefa(double *a,int *lda,int *n,int *ipvt,int *info);
extern "C" void dgedi(double *a,int *lda,int *n,int *ipvt,
                      double *d,double *work,int *job);

// error-routine
void mx::error (char *msg)
               {
               cout << "Fatal error in MX-routines : " << msg << "\n";
               exit(0);
               }

// constructor for matrix
mx::mx ()
       {
       x = new double[MAX_MX_DIM*MAX_MX_DIM];
       r = 0;
       c = 0;
       }
    
// constructor for matrix
mx::mx (int rr,int cc)
       {
       if((rr>MAX_MX_DIM)||(cc>MAX_MX_DIM)) error("Too big dimension");
       x = new double[rr*cc];
       r = rr;
       c = cc;
       for (int j=0;j<cc;j++) for (int i=0;i<rr;i++) x[j*rr+i]=0;
       }

// constructor for matrix with constant value
mx::mx (int rr,int cc,double xx)
       {
       if((rr>MAX_MX_DIM)||(cc>MAX_MX_DIM)) error("Too big dimension");
       x = new double[rr*cc];
       r = rr;
       c = cc;
       for (int j=0;j<cc;j++) for (int i=0;i<rr;i++) x[j*rr+i]=xx;
       }

// constructor for matrix loaded from file
mx::mx (int rr,int cc,char *fnm)
       {
       ifstream data(fnm);
       if((rr>MAX_MX_DIM)||(cc>MAX_MX_DIM)) error("Too big dimension");
       x = new double[rr*cc];
       r = rr;
       c = cc;
       for (int i=0;i<r;i++) for (int j=0;j<c;j++) data >> x[j*r+i];
       }
// constructor with copy-initializer
mx::mx (mx& z)
       {
       x = new double[z.r*z.c];
       r = z.r;
       c = z.c;
       for (int i=0;i<r*c;i++) x[i] = z.x[i];
       }

// destructor
mx::~mx ()
        {
        delete x;
        }

#ifdef DEBUG
// dump-routine
void mx::dump ()
              {
              cout << "R = " << r << " C = " << c << " X = " << (int)x << "\n";
              }
#endif

// assignment
mx mx::operator=(mx& z)
                {
                   r = z.r;
                   c = z.c;
                   for (int i=0;i<r*c;i++) x[i] = z.x[i];
                   return *this;
                }

// matrix + matrix
mx operator+(mx& a,mx& b)
{
   if ((a.r!=b.r)||(a.c!=b.c)) a.error("not equal dimensions in addition");
   mx tmp(a.r,a.c);
   for (int j=0;j<tmp.c;j++) for (int i=0;i<tmp.r;i++)
      tmp.x[j*tmp.r+i] = a.x[j*tmp.r+i] + b.x[j*tmp.r+i];
   return tmp;
}

// number + matrix
mx operator+(double a,mx& b)
{
   mx tmp(b.r,b.c);
   for (int j=0;j<tmp.c;j++) for (int i=0;i<tmp.r;i++)
      tmp.x[j*tmp.r+i] = a + b.x[j*tmp.r+i];
   return tmp;
}

// matrix + number
mx operator+(mx& a,double b)
{
   mx tmp(a.r,a.c);
   for (int j=0;j<tmp.c;j++) for (int i=0;i<tmp.r;i++)
      tmp.x[j*tmp.r+i] = a.x[j*tmp.r+i] + b;
   return tmp;
}

// matrix - matrix
mx operator-(mx& a,mx& b)
{
   if ((a.r!=b.r)||(a.c!=b.c)) a.error("not equal dimensions");
   mx tmp(a.r,a.c);
   for (int j=0;j<tmp.c;j++) for (int i=0;i<tmp.r;i++)
      tmp.x[j*tmp.r+i] = a.x[j*tmp.r+i] - b.x[j*tmp.r+i];
   return tmp;
}

// number - matrix
mx operator-(double a,mx& b)
{
   mx tmp(b.r,b.c);
   for (int j=0;j<tmp.c;j++) for (int i=0;i<tmp.r;i++)
      tmp.x[j*tmp.r+i] = a - b.x[j*tmp.r+i];
   return tmp;
}

// matrix - number
mx operator-(mx& a,double b)
{
   mx tmp(a.r,a.c);
   for (int j=0;j<tmp.c;j++) for (int i=0;i<tmp.r;i++)
      tmp.x[j*tmp.r+i] = a.x[j*tmp.r+i] - b;
   return tmp;
}

// matrix * matrix
mx operator*(mx& a,mx& b)
{
   if (a.c!=b.r) a.error("not appropriate dimensions for multiplication");
   int one = 1;
   mx tmp(a.r,b.c);
   for (int j=0;j<tmp.c;j++) for (int i=0;i<tmp.r;i++)
      tmp.x[j*tmp.r+i] = ddot((int *)&a.c,(double *)&a.x[i],(int *)&a.r,
                              (double *)&b.x[j*b.r],(int *)&one);
   return tmp;
}

// number * matrix
mx operator*(double a,mx& b)
{
   mx tmp(b.r,b.c);
   for (int j=0;j<tmp.c;j++) for (int i=0;i<tmp.r;i++)
      tmp.x[j*tmp.r+i] = a * b.x[j*tmp.r+i];
   return tmp;
}

// matrix * number
mx operator*(mx& a,double b)
{
   mx tmp(a.r,a.c);
   for (int j=0;j<tmp.c;j++) for (int i=0;i<tmp.r;i++)
      tmp.x[j*tmp.r+i] = a.x[j*tmp.r+i] * b;
   return tmp;
}

// write
ostream& operator<<(ostream& s,mx& o)
{
   s << "Dimensions : " << o.r << "*" << o.c << "\n";
   for (int j=0;j<o.c;j++) for (int i=0;i<o.r;i++)
      s << "(" << (i+1) << "," << (j+1) << ") = " << o.x[j*o.r+i] << "\n";
   return s;
}

// transpose
mx tran(mx& z)
{
   mx tmp(z.c,z.r);
   for (int j=0;j<tmp.c;j++) for (int i=0;i<tmp.r;i++)
      tmp.x[j*tmp.r+i] = z.x[i*z.r+j];
   return tmp;
}

// inverse
mx inv(mx& z)
{
   if(z.c!=z.r) z.error("non quadratic matrix - can not invert");
   mx tmp(z.r,z.c);
   for (int i=0;i<tmp.c*tmp.r;i++) tmp.x[i] = z.x[i];
   int pivot[MAX_MX_DIM];
   int status;
   dgefa(tmp.x,(int *)&tmp.r,(int *)&tmp.c,(int *)&pivot,(int *)&status);
   if (status!=0) z.error("singular matrix - can not invert");
   double d[2];
   double work[MAX_MX_DIM];
   int code = 11;
   dgedi(tmp.x,(int *)&tmp.r,(int *)&tmp.c,(int *)&pivot,
         (double *)&d,(double *)&work,(int *)&code);
   return tmp;
}

// determinant
double det(mx& z)
{
   if(z.c!=z.r) z.error("non quadratic matrix - can not calculate determinant");
   mx tmp(z.r,z.c);
   for (int i=0;i<tmp.c*tmp.r;i++) tmp.x[i] = z.x[i];
   int pivot[MAX_MX_DIM];
   int status;
   dgefa(tmp.x,(int *)&tmp.r,(int *)&tmp.c,(int *)&pivot,(int *)&status);
   if (status!=0) z.error("singular matrix - can not calculate determinant");
   double d[2];
   double work[MAX_MX_DIM];
   int code = 11;
   dgedi(tmp.x,(int *)&tmp.r,(int *)&tmp.c,(int *)&pivot,
         (double *)&d,(double *)&work,(int *)&code);
   if ((log(d[0])+d[1]*log(10.0))>log(MAXDOUBLE))
      z.error("determinant too large");
   return (d[0]*pow(10.0,d[1]));
}

// number of rows function
int row(mx& z)
{
   return (z.r);
}

// number of columns function
int col(mx& z)
{
   return (z.c);
}

// address of data function
double *data(mx& z)
{
   return (z.x);
}

// random function
void random(mx& z,double m,double sd)
{
   for (int i=0;i<z.r*z.c;i++) random(&z.x[i],m,sd);
   return;
}
