/*****************************************************************************/
/*
                                   Conan.c


CGI-compliant script providing access to VMS help and text libraries.

"Conan The Librarian"


HELP LIBRARIES
--------------

Help library modules have a reasonably complex internal structure described in 
the "Utility Routines Manual" and the "Command Definition, Librarian, and 
Message Utilities Manual".  The structure will not be described here. 

KeyWord[1]...KeyWord[10] represent the help keywords 1...10 that are used as 
keys into help information.


PS
--
This program is not a paradigm of software engineering virtue  :^)


LOGICAL NAMES
-------------
HTTPD$GMT              timezone offset from Greenwich Mean Time (e.g. "+09:30")


QUALIFIERS
----------
/DBUG                  turns on all "if (Debug)" statements
/ICONS=                location (in URL fomat) of Conan icons


BUILD DETAILS
-------------
See BUILD_CONAN.COM procedure.


VERSION HISTORY (update SoftwareID as well!)
---------------
23-FEB-96  MGD  v2.6.1, bugfix, after modifying the HTTPD$GMT format for the
                        HTTPd server I had forgotten about some scripts
12-OCT-95  MGD  v2.6.0, added 'Referer:', 'Last-Modified:'/'If-Modified-Since:'
18-AUG-95  MGD  v2.5.0, added "search" form at appropriate places
20-JUN-95  MGD  v2.4.0, added "explode" feature, listing all help levels below
                        the key specified in a single page
24-MAY-95  MGD  v2.3.0, minor changes for AXP compatibility
31-MAR-95  MGD  v2.2.0, ongoing maintenance
02-FEB-95  MGD  v2.1.0, add "Conan the Librarian" icon
13-DEC-94  MGD  v2.0.0, made CGI compliant
09-JUL-94  MGD  v1.0.0, initial development
*/
/*****************************************************************************/

#ifdef __ALPHA
   char SoftwareID [] = "CONAN AXP-2.6.1";
#else
   char SoftwareID [] = "CONAN VAX-2.6.1";
#endif

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <errno.h>

/* VMS related header files */
#include <descrip.h>
#include <iodef.h>
#include <jpidef.h>
#include <libdef.h>
#include <libdtdef.h>
#include <lbrdef.h>
#include <lhidef.h>
#include <lnmdef.h>
#include <rms.h>
#include <ssdef.h>
#include <stsdef.h>
#include <syidef.h>

#ifdef __ALPHA
#   pragma nomember_alignment
#endif

#define boolean int
#define true 1
#define false 0
 
#define VMSok(x) ((x) & STS$M_SUCCESS)
#define VMSnok(x) !(((x) & STS$M_SUCCESS))

#define DefaultLibrary "SYS$HELP:HELPLIB.HLB"
 
char  Utility [] = "CONAN";

char  Http404Header [] =
"HTTP/1.0 404 Error report follows.\r\n\
Content-Type: text/html\r\n\
\r\n";

boolean  Debug,
         ExplodeHelp,
         DoHeaderInformation,
         DoListLibraries,
         DoLibraryMenu,
         DoSearch,
         DoSearchStatistics = true,
         FormatAsList,
         HttpHasBeenOutput,
         KeyNameWildcard,
         TimeAheadOfGmt;
         
int  SearchHitCount,
     KeyWordCount,
     RecordsSearched,
     SearchStringLength;

unsigned long  LibraryIndex,
               IndexNumber;

unsigned long  IfModifiedSinceBinaryTime [2],
               TimeGmtDeltaBinary [2];

char  CgiHttpIfModifiedSince [256],
      CgiPathInfo [256],
      CgiPathTranslated [256],
      CgiRequestMethod [256],
      CgiScriptName [256],
      ClientHostName [128],
      ConanIconUrl [256],
      SpacerIconUrl [256],
      FormDo [256],
      FormExplode [16],
      FormFormat [256],
      FormKey [256],
      FormTitle [256],
      GmDateTime [32],
      HtmlFormTitle [256],
      HtmlLibraryTitle [256],
      HtmlReferer [512],
      HtmlSearchString [512],
      LastModifiedGmDateTime [32],
      LibraryName [256],
      LibraryPathInfo [256],
      LibrarySpec [256],
      LibraryTitle [256],
      OriginalReferer [256],
      SearchString [512],
      TimeGmtString [32],
      TimeGmtVmsString [32],
      UriLibraryName [256],
      UriFormTitle [256],
      UriReferer [512];

/* element 0 is not used, keywords number from 1 to 10 */
char  KeyWord [11][256],
      KeyWordHtml [11][256],
      KeyName [11][256],
      KeyNameHtml [11][256];
      
FILE  *HttpOut;

struct lhidef  LibraryHeader;

struct {
   unsigned long  lo32;
   unsigned long  hi32;
} ModuleRFA;

/* required prototypes */
char* SearchText (char*, char*, boolean);
int ListHelpModule (struct dsc$descriptor_s*, void*);
int ListTextModule (struct dsc$descriptor_s*, void*);
int OpenHelpModule (struct dsc$descriptor_s*, void*);
int OpenTextModule (struct dsc$descriptor_s*, void*);
int SearchHelpModule (struct dsc$descriptor_s*, void*);
int SearchTextModule (struct dsc$descriptor_s*, void*);

/**********************/
/* external functions */
/**********************/

char* MapUrl (char*, char*, char*, char*);

/*****************************************************************************/
/*
*/

main (int argc, char *argv[])

{
   register int  acnt, idx;
   register char  *cptr, *sptr;

   int  status;
   char  IconLocation [256];

   /*********/
   /* begin */
   /*********/

   /* open another output stream so that the "\n" are not filtered */
#ifdef __DECC
   if ((HttpOut = fopen ("SYS$OUTPUT", "w", "ctx=bin")) == NULL)
      exit (vaxc$errno);
#else
   if ((HttpOut = fopen ("SYS$OUTPUT", "w", "rfm=udf")) == NULL)
      exit (vaxc$errno);
#endif

   /***********************************/
   /* get the command line parameters */
   /***********************************/

   for (acnt = 1; acnt < argc; acnt++)
   {
      if (Debug) fprintf (stdout, "argv[%d] |%s|\n", acnt, argv[acnt]);
      if (strsame (argv[acnt], "/DBUG", -1))
      {
         Debug = true;
         continue;
      }
      if (strsame (argv[acnt], "/ICONS=", 5))
      {
         for (cptr = argv[acnt]+4; *cptr && *cptr != '='; cptr++);
         if (*cptr) cptr++;
         sptr = IconLocation;
         while (*cptr) *sptr++ = *cptr++;
         *sptr = '\0';
         sprintf (ConanIconUrl,
                  "<IMG ALIGN=bottom SRC=\"%sconan.xbm\" ALT=\"\">",
                  IconLocation);
         sprintf (SpacerIconUrl,
                  "<IMG ALIGN=bottom SRC=\"%sspacer.xbm\" ALT=\"\">",
                  IconLocation);
         continue;
      }
      fprintf (stdout, "%%%s-E-IVQUAL, unrecognized qualifier\n \\%s\\\n",
               Utility, argv[acnt]+1);
      exit (STS$K_ERROR | STS$M_INHIB_MSG);
   }

   /*************************/
   /* get the CGI variables */
   /*************************/

   GetCgiVariable ("WWW_PATH_INFO", CgiPathInfo, sizeof(CgiPathInfo));
   GetCgiVariable ("WWW_PATH_TRANSLATED", CgiPathTranslated,
                   sizeof(CgiPathTranslated));
   GetCgiVariable ("WWW_REQUEST_METHOD", CgiRequestMethod,
                   sizeof(CgiRequestMethod));
   GetCgiVariable ("WWW_SCRIPT_NAME", CgiScriptName, sizeof(CgiScriptName));

   /* numeric equivalent of "GET\0" (just a bit more efficient, that's all!) */
   if (*(unsigned long*)CgiRequestMethod != 0x00544547) exit (SS$_NORMAL);

   FormDo[0] = FormFormat[0] = FormKey[0] = SearchString[0] = '\0';
   GetCgiVariable ("WWW_FORM_DO", FormDo, sizeof(FormDo));
   GetCgiVariable ("WWW_FORM_EXPLODE", FormExplode, sizeof(FormExplode));
   GetCgiVariable ("WWW_FORM_FORMAT", FormFormat, sizeof(FormFormat));
   GetCgiVariable ("WWW_FORM_KEY", FormKey, sizeof(FormKey));
   GetCgiVariable ("WWW_FORM_SEARCH", SearchString, sizeof(SearchString));
   GetCgiVariable ("WWW_FORM_TITLE", FormTitle, sizeof(FormTitle));

   GetCgiVariable ("WWW_HTTP_IF_MODIFIED_SINCE", CgiHttpIfModifiedSince,
                   sizeof(CgiHttpIfModifiedSince));

   if (VMSnok (status = TimeSetGmt ()))
   {
      if (status != SS$_NOLOGNAM)
      {
         ErrorVmsStatus (status, CgiPathInfo, LibrarySpec, __FILE__, __LINE__);
         exit (SS$_NORMAL);
      }
   }
   if (CgiHttpIfModifiedSince[0])
   {
      if (VMSnok (HttpGmTime (CgiHttpIfModifiedSince, 
                              &IfModifiedSinceBinaryTime)))
      {
         if (Debug) fprintf (stdout, "If-Modified-Since: NBG!\n");
         IfModifiedSinceBinaryTime[0] = IfModifiedSinceBinaryTime[0] = 0;
         CgiHttpIfModifiedSince[0] = '\0';
      }
   }

   GetCgiVariable ("WWW_FORM_REFERER", OriginalReferer, sizeof(OriginalReferer));
   if (!OriginalReferer[0])
      GetCgiVariable ("WWW_HTTP_REFERER", OriginalReferer,
                      sizeof(OriginalReferer));
   if (OriginalReferer[0])
   {
      /* re-escape the URL-escaped percentages */
      CopyIntoUri (UriReferer, OriginalReferer, -1);
      CopyIntoHtml (HtmlReferer, OriginalReferer, -1);
   }

   /**************/
   /* initialize */
   /**************/

   strcpy (LibraryPathInfo, CgiPathInfo);
   strcpy (LibrarySpec, CgiPathTranslated);
   for (sptr = LibrarySpec; *sptr && *sptr != '*' && *sptr != '%'; sptr++);
   if (*sptr) DoListLibraries = true;

   idx = 0;
   cptr = FormKey;
   while (*cptr)
   {
      if (idx <= 10) idx++;
      if (*cptr == '~') cptr++;
      sptr = KeyWord[idx];
      while (*cptr && *cptr != '~') *sptr++ = *cptr++;
      *sptr = '\0';
      CopyIntoHtml (KeyWordHtml[idx], KeyWord[idx], -1);
      if (Debug)
         fprintf (stdout, "KeyWord[idx] |%s|%s|\n",
                  KeyWord[idx], KeyWordHtml[idx]);
   }
   KeyWordCount = idx;
   while (idx < 10) KeyWord[++idx][0] = KeyWordHtml[idx][0] = '\0';

   switch (toupper(FormDo[0]))
   {
      case 'H' :  DoHeaderInformation = true;  break;
      case 'L' :  DoListLibraries = true;  break;
      case 'M' :  DoLibraryMenu = true;  break;
      case 'S' :  DoSearch = true;  break;
      default :
         for (cptr = CgiPathTranslated; *cptr; cptr++)
            if (*cptr == '*' || *cptr == '%') DoListLibraries = true;
   }

   if (FormExplode[0])
      ExplodeHelp = true;
   else
      ExplodeHelp = false;

   if (FormFormat[0])
   {
      if (toupper(FormFormat[0]) == 'L')
         FormatAsList = true;
      else
         FormatAsList = false;
   }

   if (SearchString[0])
   {
      DoSearch = true;
      SearchStringLength = strlen(SearchString);
      CopyIntoHtml (HtmlSearchString, SearchString, -1);
   }

   if (!KeyWord[1][0])
   {
      /* no module keyname supplied, wildcard to list all modules */
      strcpy (KeyWord[1], "*");
      KeyNameWildcard = true;
   }
   else
   {
      /* check for a wildcard in the keyname, list all modules if there is */
      for (sptr = KeyWord[1]; *sptr && *sptr != '*' && *sptr != '%'; sptr++);
      if (*sptr) KeyNameWildcard = true;
   }

   if (FormTitle[0])
   {
      strcpy (LibraryTitle, FormTitle);
      CopyIntoUri (UriFormTitle, LibraryTitle, -1);
      CopyIntoHtml (HtmlFormTitle, LibraryTitle, -1);
      CopyIntoHtml (HtmlLibraryTitle, LibraryTitle, -1);
   }
   else
      sprintf (HtmlLibraryTitle, "<TT>%s</TT>", LibraryPathInfo);

   /***********/
   /* process */
   /***********/

   HttpGmTimeString (GmDateTime, 0);

   if (DoSearch && !SearchString[0])
   {
      ErrorGeneral ("Search string not supplied.", __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }

   if (DoListLibraries)
      ListLibraries ();
   else
   if (DoLibraryMenu)
      LibraryMenu ();
   else
      Librarian ();
 
   if (OriginalReferer[0])
      fprintf (HttpOut, "<P>\n<A HREF=\"%s\"><I>Close</A> Librarian.</I>\n",
               OriginalReferer);

   fclose (HttpOut);

   exit (SS$_NORMAL);
}

/****************************************************************************/
/*
Search for library files according to the the supplied specification (defaults 
to help libraries; SYS$HELP:*.HLB).  Display the name of each library in lower 
case as a list item.
*/

ListLibraries ()

{
   register int  len;
   register char  *cptr, *sptr;

   boolean  SupportedLibraryType;
   int  status,
        FileCount = 0;
   char  FileName [256],
         ExpandedFileSpec [256],
         String [1024];
   struct FAB  FileFab = cc$rms_fab;
   struct NAM  FileNam = cc$rms_nam;

   if (Debug) fputs ("ListLibraries()\n", stdout);

   /* if no library specifcation provided then default to help libraries */
   if (!LibrarySpec[0]) strcpy (LibrarySpec, "SYS$HELP:*.HLB;0");

   /* initialize the file access block (FAB) */
   FileFab.fab$l_fna = LibrarySpec;
   FileFab.fab$b_fns = strlen(LibrarySpec);
   FileFab.fab$l_fop = FAB$V_NAM;
   FileFab.fab$l_nam = &FileNam;

   /* initialize the file name block (NAM) */
   FileNam.nam$l_esa = ExpandedFileSpec;
   FileNam.nam$b_ess = sizeof(ExpandedFileSpec)-1;
   FileNam.nam$l_rsa = FileName;
   FileNam.nam$b_rss = sizeof(FileName)-1;

   if (VMSnok (status = sys$parse (&FileFab, 0, 0)))
   {
      ErrorVmsStatus (status, CgiPathInfo, LibrarySpec, __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }

   fprintf (HttpOut,
"HTTP/1.0 200 Document follows.\r\n\
Date: %s\r\n\
Content-Type: text/html\r\n\
\r\n\
<TITLE>Index of %s</TITLE>\n\
<H1>%sIndex of <TT>%s</TT></H1>\n\
<P>\n",
   GmDateTime,
   LibraryPathInfo,
   ConanIconUrl, LibraryPathInfo);

   while (VMSok (status = sys$search (&FileFab, 0, 0)))
   {
      if (!FileCount++) fputs ("<OL>\n", HttpOut);

      /* ignore directories if the file type includes them (numeric ".DIR") */
      if (*(unsigned long*)FileNam.nam$l_type == 0x5249442e) continue;

      /* for efficiency; numeric equivalent of ".HLB" and ".TLB" */
      if (*(unsigned long*)FileNam.nam$l_type == 0x424c482e ||
          *(unsigned long*)FileNam.nam$l_type == 0x424c542e)
         SupportedLibraryType = true;
      else
         SupportedLibraryType = false;

      sptr = String;
      strcpy (sptr, "<LI>");
      sptr += 4;
      if (SupportedLibraryType)
      {
         *(char*)FileNam.nam$l_ver = '\0';
         sptr += sprintf (sptr,
            "<A HREF=\"%s%s?do=menu&title=%s&referer=%s\">",
            CgiScriptName, MapUrl(NULL,FileName,NULL,NULL),
            UriFormTitle, UriReferer);
         *(char*)FileNam.nam$l_ver = ';';
      }

      /* if there was wildcard in the file type then include it */
      if (FileNam.nam$l_fnb & NAM$M_WILD_TYPE)
         *(char*)FileNam.nam$l_ver = '\0';
      else
         *(char*)FileNam.nam$l_type = '\0';
      cptr = (char*)FileNam.nam$l_name;
      while (*cptr) *sptr++ = tolower(*cptr++);
      *sptr = '\0';
      if (FileNam.nam$l_fnb & NAM$M_WILD_TYPE)
         *(char*)FileNam.nam$l_ver = ';';
      else
         *(char*)FileNam.nam$l_type = '.';

      if (SupportedLibraryType)
      {
         strcpy (sptr, "</A>\n");
         sptr += 5;
      }
      *sptr = '\0';

      fputs (String, HttpOut);
   }

   if (FileCount)
      fputs ("</OL>\n", HttpOut);
   else
      fputs ("<P>\nNo files found.\n", HttpOut);

   if (status == RMS$_FNF || status == RMS$_NMF) status = SS$_NORMAL;
   if (VMSnok (status))
   {
      ErrorVmsStatus (status, CgiPathInfo, LibrarySpec, __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }

   return (status);
}

/*****************************************************************************/
/*
For the specified library file provide a menu of four services; librarian, 
searching, library header and help.
*/ 
 
LibraryMenu ()
 
{
   register char  *cptr, *sptr;
   char  *TypePtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "LibraryMenu() |%s|\n", LibrarySpec);

   for (cptr = LibrarySpec; *cptr; cptr++);
   while (*cptr != '.' && cptr > LibrarySpec) cptr--;
   /* for efficiency; numeric equivalent of ".HLB" */
   if (*(unsigned long*)cptr == 0x424c482e)
      TypePtr = "HELP ";
   else
   /* for efficiency; numeric equivalent of ".TLB" */
   if (*(unsigned long*)cptr == 0x424c542e)
      TypePtr = "TEXT ";
   else
      TypePtr = "";

   cptr = sptr = MapUrl (NULL, LibrarySpec, NULL, NULL);
   while (*cptr && *cptr != ';') cptr++;
   *cptr = '\0';

   fprintf (HttpOut,
"HTTP/1.0 200 Document follows.\r\n\
Date: %s\r\n\
Content-Type: text/html\r\n\
\r\n\
<!-- SoftwareID: %s -->\n\
<TITLE>Library %s</TITLE>\n\
<H1>%sLibrary <TT>%s</TT></H1>\n\
<P>\n\
<UL>\n\
<LI><A HREF=\"%s%s?title=%s&referer=%s\">Open %sLibrary</A>\n\
<LI><A HREF=\"%s%s?do=header&title=%s&referer=%s\">Library Header</A>\n\
</UL>\n\
<FORM ACTION=\"%s%s\">\n\
<INPUT TYPE=hidden NAME=\"referer\" VALUE=\"%s\">\n\
<INPUT TYPE=hidden NAME=\"title\" VALUE=\"%s\">\n\
<INPUT TYPE=hidden NAME=\"do\" VALUE=\"search\">\n\
<INPUT TYPE=\"submit\" VALUE=\"Search for: \">\n\
<INPUT TYPE=\"text\" NAME=\"search\">\n\
<INPUT TYPE=\"reset\" VALUE=\"Reset\">\n\
</FORM>\n",
   GmDateTime,
   SoftwareID,
   HtmlLibraryTitle, ConanIconUrl, LibraryPathInfo,
   CgiScriptName, sptr, UriFormTitle, UriReferer, TypePtr,
   CgiScriptName, sptr, UriFormTitle, UriReferer,
   CgiScriptName, sptr,
   HtmlReferer, HtmlFormTitle);

   fflush (HttpOut);
   HttpHasBeenOutput = true;
}

/*****************************************************************************/
/*
Initialise library control.  Open the library specified by LibrarySpec.  Get 
header information from the library so its internal format can be determined 
(help or text, other are not supported).  Call ProcessLibrary() to perform the 
requested access according to the library type.
*/ 
 
Librarian ()

{
   register char  *cptr;
   int  status;
   unsigned long  Function = LBR$C_READ,
                  Length;
   char  String [1024];
   $DESCRIPTOR (LibrarySpecDsc, "");
   static struct dsc$descriptor_s 
   LibraryDefDsc = { sizeof(DefaultLibrary)-1, DSC$K_DTYPE_T,
                     DSC$K_CLASS_S, DefaultLibrary },
   LibraryNameDsc = { sizeof(LibraryName)-1, DSC$K_DTYPE_T,
                      DSC$K_CLASS_S, LibraryName };

   /****************************************/
   /* initialize, open library, get header */
   /****************************************/

   if (Debug) fputs ("Librarian()\n", stdout);

   if (VMSnok (status = lbr$ini_control (&LibraryIndex, &Function, 0, 0)))
   {
      ErrorVmsStatus (status, CgiPathInfo, LibrarySpec, __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }
 
   LibrarySpecDsc.dsc$w_length = strlen(LibrarySpec);
   LibrarySpecDsc.dsc$a_pointer = LibrarySpec;
 
   if (VMSnok (status = lbr$open (&LibraryIndex,
                                  &LibrarySpecDsc,
                                  0,
                                  &LibraryDefDsc,
                                  0,
                                  &LibraryNameDsc,
                                  &Length)))
   {
      if (Debug) fprintf (stdout, "lib$open() %%X%08.08X\n", status);
      ErrorVmsStatus (status, CgiPathInfo, LibrarySpec, __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }
 
   /* terminate, removing the trailing version number */
   LibraryName[Length] = '\0';
   for (cptr = LibraryName; *cptr && *cptr != ';'; cptr++);
   *cptr = '\0';

   *(cptr = UriLibraryName) = '\0';
   MapUrl(cptr, LibraryName, NULL, NULL);
   if (Debug)
      fprintf (stdout, "LibraryName |%s|%s|\n", LibraryName, UriLibraryName);
 
   if (VMSnok (status = lbr$get_header (&LibraryIndex, &LibraryHeader)))
   {
      ErrorVmsStatus (status, CgiPathInfo, LibrarySpec, __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }

   if (CgiHttpIfModifiedSince[0])
   {
      if (VMSnok (ModifiedSince (&IfModifiedSinceBinaryTime,
                                 &LibraryHeader.lhi$l_updtim)))
      {
         /* library has not been modified since the date/time, don't send */
         status = lbr$close (&LibraryIndex);
         exit (SS$_NORMAL);
      }
   }

   HttpGmTimeString (LastModifiedGmDateTime, &LibraryHeader.lhi$l_updtim);

   /*******************/
   /* process library */
   /*******************/

   if (DoHeaderInformation)
      HeaderInformation ();
   else
   if (LibraryHeader.lhi$l_type == LBR$C_TYP_TXT)
   {
      if (DoSearch)
         SearchTextLibraryModules ();
      else
      if (KeyNameWildcard)
         ListTextLibraryModules ();
      else
         OpenTextLibraryModule ();
   }
   else
   if (LibraryHeader.lhi$l_type == LBR$C_TYP_HLP)
   {
      if (DoSearch)
         SearchHelpLibraryModules ();
      else
      if (KeyNameWildcard)
         ListHelpLibraryModules ();
      else
         OpenHelpLibraryModule ();
   }
   else
   {
      sprintf (String,
      "Library type %d not supported. (Only HELP and TEXT libraries are valid)",
      LibraryHeader.lhi$l_type);
      ErrorGeneral (String, __FILE__, __LINE__);
   }
 
   status = lbr$close (&LibraryIndex);
   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
Provide selected library header information.
*/ 
 
int HeaderInformation ()
 
{
   static char  *LibraryTypes[] =
          { "Unknown", "VAX Object", "Macro", "Help", "Text",
            "VAX Sharable Image", "NCS",
            "Alpha Object", "Alpha Sharable Image", "?" };
   static  $DESCRIPTOR (CreatedFaoDsc, "<P>Created: <TT>!%D</TT>\n");
   static  $DESCRIPTOR (RevisedFaoDsc, "<P>Revised: <TT>!%D</TT>\n");

   unsigned short  Length;
   char  String [1024];
   $DESCRIPTOR (StringDsc, String);

   /*********/
   /* begin */
   /*********/

   fprintf (HttpOut,
"HTTP/1.0 200 Document follows.\r\n\
Date: %s\r\n\
Last-Modified: %s\r\n\
Content-Type: text/html\r\n\
\r\n\
<!-- SoftwareID: %s -->\n\
<TITLE>Library %s</TITLE>\n\
<H1>%sLibrary <TT>%s</TT></H1>\n",
   GmDateTime, LastModifiedGmDateTime,
   SoftwareID,
   LibraryPathInfo, ConanIconUrl, LibraryPathInfo);

   fflush (HttpOut);
   HttpHasBeenOutput = true;

   if (LibraryHeader.lhi$l_type > 8) LibraryHeader.lhi$l_type = 9;
   fprintf (HttpOut, "<P>Type: <TT>%s</TT>\n",
   LibraryTypes[LibraryHeader.lhi$l_type]);

   fprintf (HttpOut, "<P>Creator: <TT>%-*s</TT>\n",
   *(char*)LibraryHeader.lhi$t_lbrver, (char*)LibraryHeader.lhi$t_lbrver+1);

   fprintf (HttpOut, "<P>Format: <TT>%d.%d</TT>\n",
   LibraryHeader.lhi$l_majorid, LibraryHeader.lhi$l_minorid);

   sys$fao (&CreatedFaoDsc, &Length, &StringDsc, &LibraryHeader.lhi$l_credat);
   String[Length] = '\0';
   fputs (String, HttpOut);

   sys$fao (&RevisedFaoDsc, &Length, &StringDsc, &LibraryHeader.lhi$l_updtim);
   String[Length] = '\0';
   fputs (String, HttpOut);

   if (LibraryHeader.lhi$l_libstatus)
      fputs ("<P>Status: <TT>OK</TT>\n", HttpOut);
   else
      fputs ("<P>Status: <TT>PROBLEM</TT>\n", HttpOut);

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
*/
 
int ListHelpLibraryModules ()

{
   register char  *cptr, *sptr;
   int  status;
   char  KeyName [256],
         String [1024];
   $DESCRIPTOR (KeyNameDsc, KeyName);
 
   if (Debug) fputs ("ListHelpLibraryModules()\n", stdout);
 
   if (!LibraryTitle[0])
   {
      sprintf (LibraryTitle, "Library <TT>%s</TT>", LibraryPathInfo);
      strcpy (HtmlLibraryTitle, LibraryTitle);
      if (Debug) fprintf (stdout, "LibraryTitle |%s|\n", LibraryTitle);
   }

   cptr = KeyName;
   for (sptr = KeyWord[1]; *sptr; *cptr++ = toupper(*sptr++));
   *cptr = '\0';
   KeyNameDsc.dsc$w_length = cptr - KeyName;

   fprintf (HttpOut,
"HTTP/1.0 200 Document follows.\r\n\
Date: %s\r\n\
Last-Modified: %s\r\n\
Content-Type: text/html\r\n\
\r\n\
<!-- SoftwareID: %s -->\n\
<TITLE>Library %s</TITLE>\n\
<H1>%s%s</H1>\n\
<HR>\n\
<FORM ACTION=\"%s%s\">\n\
<INPUT TYPE=hidden NAME=\"referer\" VALUE=\"%s\">\n\
<INPUT TYPE=hidden NAME=\"title\" VALUE=\"%s\">\n\
<INPUT TYPE=hidden NAME=\"do\" VALUE=\"search\">\n\
<INPUT TYPE=\"submit\" VALUE=\"Search for: \">\n\
<INPUT TYPE=\"text\" NAME=\"search\">\n\
<INPUT TYPE=\"reset\" VALUE=\"Reset\">\n\
</FORM>\n",
   GmDateTime, LastModifiedGmDateTime,
   SoftwareID,
   LibraryPathInfo,
   ConanIconUrl, HtmlLibraryTitle,
   CgiScriptName, CgiPathInfo,
   HtmlReferer, HtmlFormTitle);

   fflush (HttpOut);
   HttpHasBeenOutput = true;

   if (FormatAsList)
      fputs ("<UL>\n", HttpOut);
   else
      fputs ("<P>\n", HttpOut);
   
   IndexNumber = 1;

   if (VMSnok (status = lbr$get_index (&LibraryIndex,
                                       &IndexNumber,
                                       &ListHelpModule,
                                       &KeyNameDsc)))
   {
      if (Debug) fprintf (stdout, "lib$get_index() %%X%08.08X\n", status);
      ErrorVmsStatus (status, CgiPathInfo, LibrarySpec, __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }

   if (FormatAsList)
      fputs ("\n</UL>\n<P>\n<HR>\n", HttpOut);
   else
      fputs ("\n<P>\n<HR>\n", HttpOut);

   return (status);
}
 
/*****************************************************************************/
/*
*/ 
 
int ListHelpModule
(
struct dsc$descriptor_s  *KeyNameDscPtr,
void  *RFAptr
)
{
   static boolean  FirstCall = true;
   static char  PreviousAlphabetic = '\0';

   char  String [1024],
         Uri [1024];
 
   /*********/
   /* begin */
   /*********/

   if (Debug) fputs ("ListHelpModule()\n", stdout);

   strncpy (KeyWord[1],
            KeyNameDscPtr->dsc$a_pointer, KeyNameDscPtr->dsc$w_length);
   KeyWord[1][KeyNameDscPtr->dsc$w_length] = '\0';
   strcpy (KeyName[1], KeyWord[1]);
   CopyIntoHtml (KeyNameHtml[1], KeyWord[1], -1);

   if (Debug) fprintf (stdout, "KeyNameDsc |%s|\n", KeyWord[1]);

   if (FormatAsList)
   {
      if (FirstCall)
         fputs ("<LI>", HttpOut);
      else
         fputs ("\n<LI>", HttpOut);
   }
   else
   {
      if (!isalpha(KeyName[1][0]) ||
          toupper(KeyName[1][0]) == PreviousAlphabetic)
         if (!FirstCall)
            fputs (",\n", HttpOut);
         else;
      else
      {
         if (!FirstCall) fputs ("\n<BR><BR>", HttpOut);
         PreviousAlphabetic = toupper(KeyName[1][0]);
      }
   }
   FormatHelpUri (Uri, NULL, 1);
   fprintf (HttpOut, "<A HREF=\"%s%s?key=%s&title=%s&referer=%s\">%s</A>",
            CgiScriptName, UriLibraryName, Uri, UriFormTitle, UriReferer,
            KeyNameHtml[1]);
   FirstCall = false;
   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
*/
 
int OpenHelpLibraryModule ()

{
   register char  *cptr, *sptr;
   int  status;
   char  KeyName [256],
         String [512];
   $DESCRIPTOR (KeyNameDsc, KeyName);
 
   if (Debug) fputs ("OpenHelpLibraryModule()\n", stdout);
 
   cptr = KeyName;
   for (sptr = KeyWord[1]; *sptr; *cptr++ = toupper(*sptr++));
   *cptr = '\0';
   KeyNameDsc.dsc$w_length = cptr - KeyName;
   
   IndexNumber = 1;

   if (VMSnok (status = lbr$get_index (&LibraryIndex,
                                       &IndexNumber,
                                       &OpenHelpModule,
                                       &KeyNameDsc)))
   {
      if (Debug) fprintf (stdout, "lib$get_index() %%X%08.08X\n", status);
      ErrorVmsStatus (status, CgiPathInfo, LibrarySpec, __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }

   return (status);
}
 
/*****************************************************************************/
/*
This is a reasonably complex function, and reflects the complexities internal 
to help library modules!
*/ 
 
int OpenHelpModule
(
struct dsc$descriptor_s  *KeyNameDscPtr,
void  *RFAptr
)
{
   static  MatchedCount = 0;

   register int  Count;
   register char  *cptr, *sptr;

   boolean  AdditionalInformation = true,
            ExplodedHeading = false,
            OutputModuleTitle = true;
   int  status,
        AdditionalInformationCount = 0,
        BlankLineCount = 0,
        ExplodeLevel = 0,
        HelpLevel = 0,
        Length,
        MatchedLevel = 0,
        MatchesToLevel = 0,
        PreviousHelpLevel = 0,
        TextLineCount = 0;
   int  KeyCount [11];
   char  PreviousFirstBufferChar = '\0';
   char  Buffer [1024],
         KeyString [1024],
         KeyUri [1024],
         String [1024];
   $DESCRIPTOR (InBufferDsc, Buffer);
   $DESCRIPTOR (OutBufferDsc, Buffer);
 
   /*********/
   /* begin */
   /*********/

   if (Debug) fputs ("OpenHelpModule()\n", stdout);

   strncpy (KeyName[1],
            KeyNameDscPtr->dsc$a_pointer, KeyNameDscPtr->dsc$w_length);
   KeyName[1][KeyNameDscPtr->dsc$w_length] = '\0';

   if (Debug) fprintf (stdout, "KeyNameDsc |%s|\n", KeyName[1]);

   CopyIntoHtml (KeyWordHtml[1], KeyName[1], -1);
   CopyIntoHtml (KeyNameHtml[1], KeyName[1], -1);
   if (!KeyWordCount)
   {
      strcpy (KeyWord[1], KeyName[1]);
      KeyWordCount = 1;
   }

   if (ExplodeHelp)
      for (Count = 0; Count < 11; KeyCount[Count++] = 0);

   if (VMSnok (status = lbr$lookup_key (&LibraryIndex,
                                        KeyNameDscPtr,
                                        &ModuleRFA)))
   {
      if (Debug) fprintf (stdout, "lib$lookup_key() %%X%08.08X\n", status);
      ErrorVmsStatus (status, CgiPathInfo, LibrarySpec, __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }

   /**************************************/
   /* loop through all records in module */
   /**************************************/

   for (;;)
   {
      status = lbr$get_record (&LibraryIndex,
                               &InBufferDsc,
                               &OutBufferDsc);
      if (Debug) fprintf (stdout, "lbr$get_record() %%X%08.08X\n", status);
      if (VMSnok (status)) break;
 
      /* terminate output buffer, removing any trailing white space */
      cptr = Buffer+OutBufferDsc.dsc$w_length-1;
      while (cptr >= Buffer && isspace(*cptr)) cptr--;
      *++cptr = '\0';
 
      /* comment record (line) in module */
      if (Buffer[0] == '!') continue;

      if (Debug) fprintf (stdout, "Buffer |%s|\n", Buffer);

      if ((isdigit (Buffer[0]) && Buffer[1] == ' ') || Buffer[0] == '/')
      {
         /***************************/
         /* help level or qualifier */
         /***************************/
 
         PreviousHelpLevel = HelpLevel;
         if (Buffer[0] == '/')
         {
            /* if no previous qualifier encountered at this help level */
            /* then the qualifier effectively creates a new help level */
            if (PreviousFirstBufferChar != '/') HelpLevel++;
            cptr = Buffer;
         }
         else
         {
            /* help level topic (line begins with a digit then a space) */
            HelpLevel = Buffer[0] - '0';
            for (cptr = Buffer + 1; isspace(*cptr); cptr++);
         }

         strcpy (KeyName[HelpLevel], cptr);
         CopyIntoHtml (KeyNameHtml[HelpLevel], cptr, -1);
         PreviousFirstBufferChar = Buffer[0];

         if (MatchesToLevel >= HelpLevel)
         {
            /* if the topic's been matched and output, finish up */
            if (MatchedLevel) break;

            MatchesToLevel = ExplodeLevel = 0;
            for (Count = 1; Count <= HelpLevel; Count++)
            {
               if (strsame (KeyWord[Count], KeyName[Count], -1))
                  MatchesToLevel = Count;
               else
                  break;
            }
         }
         else
         {
            for (Count = MatchesToLevel + 1; Count <= HelpLevel; Count++)
            {
               if (strsame (KeyWord[Count], KeyName[Count], -1))
                  MatchesToLevel = Count;
               else
                  break;
            }
         }

         if (MatchesToLevel == KeyWordCount && MatchesToLevel == HelpLevel)
         {
            /****************/
            /* module title */
            /****************/

            MatchedCount++;

            if (!LibraryTitle[0])
            {
               sprintf (LibraryTitle, "Library <TT>%s</TT>", LibraryPathInfo);
               strcpy (HtmlLibraryTitle, LibraryTitle);
               if (Debug) fprintf (stdout, "LibraryTitle |%s|\n", LibraryTitle);
            }

            fprintf (HttpOut,
"HTTP/1.0 200 Document follows.\r\n\
Date: %s\r\n\
Last-Modified: %s\r\n\
Content-Type: text/html\r\n\
\r\n\
<!-- SoftwareID: %s -->\n",
   GmDateTime, LastModifiedGmDateTime,
   SoftwareID);

            fflush (HttpOut);
            HttpHasBeenOutput = true;

            sptr = String;
            sptr += sprintf (sptr,
               "<TITLE>Library %s</TITLE>\n<H1>%s%s</H1>\n",
               LibraryPathInfo, ConanIconUrl, HtmlLibraryTitle);

            sptr += sprintf (sptr, "<H2>%s", SpacerIconUrl);

            for (Count = 1; Count <= HelpLevel; Count++)
            {
               if (Count > 1)
               {
                  *sptr++ = ',';
                  *sptr++ = ' ';
               }
               strcpy (sptr, KeyName[Count]);
               while (*sptr) sptr++;
            }
            strcpy (sptr, "</H2>\n<HR>\n");
            sptr += 11;

            fputs (String, HttpOut);

            if (ExplodeHelp)
            {
               ExplodedHeading = true;
               ExplodeLevel = HelpLevel + 1;
            }

            MatchedLevel = HelpLevel;
         }
         else
         if (ExplodeLevel && HelpLevel >= ExplodeLevel)
         {
            /******************/
            /* exploding help */
            /******************/

            if (TextLineCount)
               if (PreviousHelpLevel == ExplodeLevel - 1)
                  fputs ("</PRE>\n<P><HR>\n", HttpOut);
               else
                  fputs ("</PRE>\n", HttpOut);
            TextLineCount = 0;

            ExplodedHeading = true;
            for (Count = HelpLevel+1; Count < 11; KeyCount[Count++] = 0);
            KeyCount[HelpLevel]++;

            fputs ("<P>\n", HttpOut);
            if (HelpLevel == 2)
               fputs ("<H1>", HttpOut);
            else
            if (HelpLevel == 3)
               fputs ("<H2>", HttpOut);
            else
               fputs ("<H3>", HttpOut);
            for (Count = ExplodeLevel; Count <= HelpLevel; Count++)
            {
               if (Count < HelpLevel)
                  fprintf (HttpOut, "%d.", KeyCount[Count]);
               else
                  fprintf (HttpOut, "%d", KeyCount[Count]);
            }
            fprintf (HttpOut, " - %s", KeyNameHtml[HelpLevel]);
            if (HelpLevel == 2)
               fputs ("</H1>\n", HttpOut);
            else
            if (HelpLevel == 3)
               fputs ("</H2>\n", HttpOut);
            else
               fputs ("</H3>\n", HttpOut);
         }
         else
         if (MatchesToLevel == KeyWordCount && HelpLevel == KeyWordCount+1)
         {
            /*************/
            /* subtopics */
            /*************/

            if (AdditionalInformation)
            {
               FormatHelpUri (KeyUri, KeyString, HelpLevel-1);
               if (TextLineCount) fputs ("</PRE>\n<HR>\n", HttpOut);
               if (HelpLevel == 2)
               {
                  fprintf (HttpOut,
"<P>\n\
<FORM ACTION=\"%s%s\">\n\
<INPUT TYPE=hidden NAME=\"referer\" VALUE=\"%s\">\n\
<INPUT TYPE=hidden NAME=\"title\" VALUE=\"%s\">\n\
<INPUT TYPE=hidden NAME=\"do\" VALUE=\"search\">\n\
<INPUT TYPE=hidden NAME=\"key\" VALUE=\"%s\">\n\
<INPUT TYPE=\"submit\" VALUE=\"Search `%s' for: \">\n\
<INPUT TYPE=\"text\" NAME=\"search\">\n\
<INPUT TYPE=\"reset\" VALUE=\"Reset\">\n\
</FORM>\n",
                  CgiScriptName, CgiPathInfo,
                  HtmlReferer, HtmlFormTitle,
                  KeyString, KeyWordHtml[1]);
               }
               fprintf (HttpOut,
"<P>\n\
Additional Information\
 <A HREF=\"%s%s?key=%s&explode=yes&title=%s&referer=%s\"><I>(explode)</I></A> \
:\n\
<P>\n\
<UL>\n",
               CgiScriptName, UriLibraryName, KeyUri,
               UriFormTitle, UriReferer);
               AdditionalInformation = false;
            }

            FormatHelpUri (KeyUri, NULL, HelpLevel);
            fprintf (HttpOut,
               "<LI><A HREF=\"%s%s?key=%s&title=%s&referer=%s\">%s</A>\n",
               CgiScriptName, UriLibraryName, KeyUri, UriFormTitle,
               UriReferer, KeyNameHtml[HelpLevel]);
            AdditionalInformationCount++;
            TextLineCount = 0;
         }
      }
      else
      {
         /*************/
         /* help text */
         /*************/
 
         if ((ExplodeLevel && HelpLevel >= ExplodeLevel) || 
             (MatchesToLevel == KeyWordCount && HelpLevel == KeyWordCount))
         {
            /* check to see if this module record has any non-space chars */
            for (cptr = Buffer; *cptr && !isspace(*cptr); cptr++);
            if (*cptr)
            {
               if (!TextLineCount) fputs ("<P>\n<PRE>", HttpOut);
               sptr = String;
               if (ExplodedHeading)
               {
                  ExplodedHeading = false;
                  BlankLineCount = 0;
               }
               if (BlankLineCount && TextLineCount)
               {
                  *sptr++ = '\r';
                  *sptr++ = '\n';
               }
               if (BlankLineCount) BlankLineCount = 0;
               sptr += CopyIntoHtml (sptr, cptr, -1);
               *sptr++ = '\n';
               *sptr = '\0';
               fputs (String, HttpOut);
               TextLineCount++;
            }
            else
               BlankLineCount++;
         }
      }
   }
 
   if (AdditionalInformationCount)
      fputs ("</UL>\n", HttpOut);
   else
   if (TextLineCount)
      fputs ("</PRE>\n<P><HR>\n", HttpOut);
   else
   if (!MatchedCount)
      ErrorGeneral ("Topic not found!", __FILE__, __LINE__);

   return (SS$_NORMAL);
}
 
/*****************************************************************************/
/*
*/
 
int SearchHelpLibraryModules ()

{
   register char  *cptr, *sptr;
   int  status;
   char  KeyName [256],
         String [512];
   $DESCRIPTOR (KeyNameDsc, KeyName);
 
   /*********/
   /* begin */
   /*********/

   if (Debug) fputs ("SearchHelpLibraryModules()\n", stdout);
 
   cptr = KeyName;
   for (sptr = KeyWord[1]; *sptr; *cptr++ = toupper(*sptr++));
   *cptr = '\0';
   KeyNameDsc.dsc$w_length = cptr - KeyName;
   
   fprintf (HttpOut,
"HTTP/1.0 200 Document follows.\r\n\
Date: %s\r\n\
Last-Modified: %s\r\n\
Content-Type: text/html\r\n\
\r\n\
<!-- SoftwareID: %s -->\n\
<TITLE>Search ``%s'' for ``%s''</TITLE>\n\
<H1>%sSearch ``%s'' for ``<TT>%s</TT>''</H1>\n",
   GmDateTime, LastModifiedGmDateTime,
   SoftwareID,
   LibraryPathInfo, HtmlSearchString,
   ConanIconUrl, HtmlLibraryTitle, HtmlSearchString);

   fflush (HttpOut);
   HttpHasBeenOutput = true;

   SearchHitCount = 0;
   IndexNumber = 1;

   if (DoSearchStatistics) lib$init_timer (0);

   if (VMSnok (status = lbr$get_index (&LibraryIndex,
                                       &IndexNumber,
                                       &SearchHelpModule,
                                       &KeyNameDsc)))
   {
      if (Debug) fprintf (stdout, "lib$get_index() %%X%08.08X\n", status);
      ErrorVmsStatus (status, CgiPathInfo, LibrarySpec, __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }

   if (!SearchHitCount) fputs ("<P><HR>\n<P><I>(not found)</I>\n", HttpOut);

   fputs ("<P><HR>\n", HttpOut);

   if (DoSearchStatistics) SearchStatistics ();

   return (status);
}

/*****************************************************************************/
/*
This is a reasonably complex function, and reflects the complexities internal 
to help library modules!
*/ 
 
int SearchHelpModule
(
struct dsc$descriptor_s  *KeyNameDscPtr,
void  *RFAptr
)
{
   static boolean  HitThisModule = false,
                   HitThisTopic = false;
   static char  PreviousFirstBufferChar = '\0';

   register int  Count;
   register char  *cptr, *hptr, *sptr;
   int  status,
        HelpLevel = 0,
        Length;
   char  Buffer [1024],
         String [8192],
         Uri [1024];
   $DESCRIPTOR (InBufferDsc, Buffer);
   $DESCRIPTOR (OutBufferDsc, Buffer);
 
   /*********/
   /* begin */
   /*********/

   if (Debug) fputs ("SearchHelpModule()\n", stdout);

   strncpy (KeyWord[1],
            KeyNameDscPtr->dsc$a_pointer, KeyNameDscPtr->dsc$w_length);
   KeyWord[1][KeyNameDscPtr->dsc$w_length] = '\0';
   strcpy (KeyName[1], KeyWord[1]);
   CopyIntoHtml (KeyWordHtml[1], KeyWord[1], -1);
   CopyIntoHtml (KeyNameHtml[1], KeyWord[1], -1);
   if (!KeyWordCount) KeyWordCount = 1;

   if (Debug) fprintf (stdout, "KeyNameDsc |%s|\n", KeyWord[1]);

   if (VMSnok (status = lbr$lookup_key (&LibraryIndex,
                                        KeyNameDscPtr,
                                        &ModuleRFA)))
   {
      if (Debug) fprintf (stdout, "lib$lookup_key() %%X%08.08X\n", status);
      ErrorVmsStatus (status, CgiPathInfo, LibrarySpec, __FILE__, __LINE__);
      return (SS$_NORMAL);
   }

   /**************************************/
   /* loop through all records in module */
   /**************************************/

   for (;;)
   {
      status = lbr$get_record (&LibraryIndex,
                               &InBufferDsc,
                               &OutBufferDsc);
      if (Debug) fprintf (stdout, "lbr$get_record() %%X%08.08X\n", status);
      if (VMSnok (status)) break;
 
      /* terminate output buffer, removing any trailing white space */
      cptr = Buffer+OutBufferDsc.dsc$w_length-1;
      while (cptr >= Buffer && isspace(*cptr)) cptr--;
      *++cptr = '\0';
 
      /* comment record (line) in module */
      if (Buffer[0] == '!') continue;

      if (Debug) fprintf (stdout, "Buffer |%s|\n", Buffer);

      if ((isdigit (Buffer[0]) && Buffer[1] == ' ') || Buffer[0] == '/')
      {
         /***************************/
         /* help level or qualifier */
         /***************************/
 
         if (Buffer[0] == '/')
         {
            /* if no previous qualifier encountered at this help level */
            /* then the qualifier effectively creates a new help level */
            if (PreviousFirstBufferChar != '/') HelpLevel++;
            cptr = Buffer;
         }
         else
         {
            /* help level topic (line begins with a digit then a space) */
            HelpLevel = Buffer[0] - '0';
            for (cptr = Buffer + 1; isspace(*cptr); cptr++);
         }

         PreviousFirstBufferChar = Buffer[0];
         strcpy (KeyName[HelpLevel], cptr);
         CopyIntoHtml (KeyNameHtml[HelpLevel], cptr, -1);

         /* if previous topic had a hit then end the list of the hit(s) */
         if (HitThisTopic) fputs ("</UL>\n", HttpOut);

         if (HelpLevel == 1) HitThisModule = false;
         HitThisTopic = false;
      }
      else
      {
         /*************/
         /* help text */
         /*************/
 
         /* check to see if this module record has any non-space chars */
         for (cptr = Buffer; *cptr && isspace(*cptr); cptr++);
         if (*cptr)
         {
            if (*(hptr = SearchText (cptr, SearchString, false)))
            {
               /********/
               /* hit! */
               /********/

               if (Debug) fprintf (stdout, "Hit |%s|\n", hptr);
               SearchHitCount++;
               sptr = String;
               if (!HitThisModule)
               {
                  strcpy (sptr, "<HR>");
                  sptr += 4;
                  HitThisModule = true;
               }
               if (!HitThisTopic)
               {
                  strcpy (sptr, "<H2>");
                  sptr += 4;
                  for (Count = 1; Count <= HelpLevel; Count++)
                  {
                     if (Count > 1) { *sptr++ = ','; *sptr++ = ' '; }
                     FormatHelpUri (Uri, NULL, Count);
                     sptr += sprintf (sptr,
                        "<A HREF=\"%s%s?key=%s&title=%s&referer=%s\">%s</A>",
                        CgiScriptName, UriLibraryName, Uri, UriFormTitle,
                        UriReferer, KeyNameHtml[Count]);
                  }
                  strcpy (sptr, "</H2><UL>\n<LI>");
                  sptr += 14;
                  HitThisTopic = true;
               }
               else
               {
                  /* same topic, new hit */
                  strcpy (sptr, "<LI>");
                  sptr += 4;
               }
               sptr += CopyIntoHtml (sptr, cptr, hptr-cptr);
               strcpy (sptr, "<TT>");
               sptr += 4;
               strncpy (sptr, hptr, SearchStringLength);
               sptr += SearchStringLength;
               strcpy (sptr, "</TT>");
               sptr += 5;
               sptr += CopyIntoHtml (sptr, hptr+SearchStringLength, -1);
               *sptr++ = '\n';
               *sptr = '\0';

               fputs (String, HttpOut);
            }
         }
      }
   }

   if (HitThisTopic) fputs ("</UL>\n", HttpOut);

   return (SS$_NORMAL);
}
 
/*****************************************************************************/
/*
String together the keys associated with this help item.
*/ 
 
FormatHelpUri
(
register char *uptr,
register char *sptr,
int ToLevel
)
{
   register int  idx;

   for (idx = 1; idx <= ToLevel; idx++)
   {
      uptr += CopyIntoUri (uptr, KeyName[idx], -1);
      *uptr++ = '~';
      if (sptr != NULL)
      {
         strcpy (sptr, KeyName[idx]);
         while (*sptr) sptr++;
         *sptr++ = '~';
      }
   }
   if (idx > 1)
   {
      uptr[-1] = '\0';
      if (sptr != NULL)
         sptr[-1] = '\0';
      else;
   }
   else
   {
      *uptr = '\0';
      if (sptr != NULL) *sptr = '\0';
   }
}
 
/*****************************************************************************/
/*
*/
 
int ListTextLibraryModules ()

{
   register char  *cptr, *sptr;
   int  status;
   char  KeyName [256],
         String [1024];
   $DESCRIPTOR (KeyNameDsc, KeyName);
 
   /*********/
   /* begin */
   /*********/

   if (Debug) fputs ("ListTextLibraryModules()\n", stdout);
 
   if (!LibraryTitle[0])
   {
      sprintf (LibraryTitle, "Library <TT>%s</TT>", LibraryPathInfo);
      strcpy (HtmlLibraryTitle, LibraryTitle);
      if (Debug) fprintf (stdout, "LibraryTitle |%s|\n", LibraryTitle);
   }

   cptr = KeyName;
   for (sptr = KeyWord[1]; *sptr; *cptr++ = toupper(*sptr++));
   *cptr = '\0';
   KeyNameDsc.dsc$w_length = cptr - KeyName;
   
   fprintf (HttpOut,
"HTTP/1.0 200 Document follows.\r\n\
Date: %s\r\n\
Last-Modified: %s\r\n\
Content-Type: text/html\r\n\
\r\n\
<!-- SoftwareID: %s -->\n\
<TITLE>Library %s</TITLE>\n\
<H1>%s%s</H1>\n\
<HR>\n\
<FORM ACTION=\"%s%s\">\n\
<INPUT TYPE=hidden NAME=\"referer\" VALUE=\"%s\">\n\
<INPUT TYPE=hidden NAME=\"title\" VALUE=\"%s\">\n\
<INPUT TYPE=hidden NAME=\"do\" VALUE=\"search\">\n\
<INPUT TYPE=\"submit\" VALUE=\"Search for: \">\n\
<INPUT TYPE=\"text\" NAME=\"search\">\n\
<INPUT TYPE=\"reset\" VALUE=\"Reset\">\n\
</FORM>\n",
   GmDateTime, LastModifiedGmDateTime,
   SoftwareID,
   LibraryPathInfo,
   ConanIconUrl, HtmlLibraryTitle,
   CgiScriptName, CgiPathInfo,
   HtmlReferer, HtmlFormTitle);

   fflush (HttpOut);
   HttpHasBeenOutput = true;

   if (FormatAsList)
      fputs ("<UL>\n", HttpOut);
   else
      fputs ("<P>\n", HttpOut);
   
   IndexNumber = 1;

   if (VMSnok (status = lbr$get_index (&LibraryIndex,
                                       &IndexNumber,
                                       &ListTextModule,
                                       &KeyNameDsc)))
   {
      if (Debug) fprintf (stdout, "lib$get_index() %%X%08.08X\n", status);
      ErrorVmsStatus (status, CgiPathInfo, LibrarySpec, __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }

   if (FormatAsList)
      fputs ("\n</UL>\n<P>\n<HR>\n", HttpOut);
   else
      fputs ("\n<P>\n<HR>\n", HttpOut);

   return (status);
}
 
/*****************************************************************************/
/*
*/ 
 
int ListTextModule
(
struct dsc$descriptor_s  *KeyNameDscPtr,
void  *RFAptr
)
{
   static boolean  FirstCall = true;
   static char  PreviousAlphabetic = '\0';

   char  String [1024],
         UriKeyName [256];
 
   /*********/
   /* begin */
   /*********/

   if (Debug) fputs ("ListTextModule()\n", stdout);

   strncpy (KeyWord[1],
            KeyNameDscPtr->dsc$a_pointer, KeyNameDscPtr->dsc$w_length);
   KeyWord[1][KeyNameDscPtr->dsc$w_length] = '\0';
   strcpy (KeyName[1], KeyWord[1]);
   CopyIntoUri (UriKeyName, KeyWord[1], -1);
   CopyIntoHtml (KeyNameHtml[1], KeyWord[1], -1);

   if (Debug) fprintf (stdout, "KeyNameDsc |%s|\n", KeyWord[1]);

   if (FormatAsList)
   {
      if (FirstCall)
         fputs ("<LI>", HttpOut);
      else
         fputs ("\n<LI>", HttpOut);
   }
   else
   {
      if (!isalpha(KeyName[1][0]) || KeyName[1][0] == PreviousAlphabetic)
         if (!FirstCall)
            fputs (",\n", HttpOut);
         else;
      else
      {
         if (!FirstCall) fputs ("\n<BR><BR>", HttpOut);
         PreviousAlphabetic = KeyName[1][0];
      }
   }
   fprintf (HttpOut, "<A HREF=\"%s%s?key=%s&title=%s&referer=%s\">%s</A>",
            CgiScriptName, UriLibraryName, UriKeyName,
            UriFormTitle, UriReferer, KeyNameHtml[1]);
   FirstCall = false;
   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
*/
 
int OpenTextLibraryModule ()

{
   register char  *cptr, *sptr;
   int  status;
   char  KeyName [256],
         String [512];
   $DESCRIPTOR (KeyNameDsc, KeyName);
 
   /*********/
   /* begin */
   /*********/

   if (Debug) fputs ("OpenTextLibraryModule()\n", stdout);
 
   cptr = KeyName;
   for (sptr = KeyWord[1]; *sptr; *cptr++ = toupper(*sptr++));
   *cptr = '\0';
   KeyNameDsc.dsc$w_length = cptr - KeyName;
   
   fprintf (HttpOut,
"HTTP/1.0 200 Document follows.\r\n\
Date: %s\r\n\
Last-Modified: %s\r\n\
Content-Type: text/html\r\n\
\r\n\
<!-- SoftwareID: %s -->\n\
<TITLE>Module ``%s'' of ``%s''</TITLE>\n",
   GmDateTime, LastModifiedGmDateTime,
   SoftwareID,
   KeyName, LibraryPathInfo);

   fflush (HttpOut);
   HttpHasBeenOutput = true;

   IndexNumber = 1;

   if (VMSnok (status = lbr$get_index (&LibraryIndex,
                                       &IndexNumber,
                                       &OpenTextModule,
                                       &KeyNameDsc)))
   {
      if (Debug) fprintf (stdout, "lib$get_index() %%X%08.08X\n", status);
      ErrorVmsStatus (status, CgiPathInfo, LibrarySpec, __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }

   return (status);
}
 
/*****************************************************************************/
/*
*/ 
 
int OpenTextModule
(
struct dsc$descriptor_s  *KeyNameDscPtr,
void  *RFAptr
)
{
   register char  *sptr;

   int  status;
   char  Buffer [1024],
         String [1024];
   $DESCRIPTOR (InBufferDsc, Buffer);
   $DESCRIPTOR (OutBufferDsc, Buffer);
 
   /*********/
   /* begin */
   /*********/

   if (Debug) fputs ("OpenTextModule()\n", stdout);

   strncpy (KeyWord[1],
            KeyNameDscPtr->dsc$a_pointer, KeyNameDscPtr->dsc$w_length);
   KeyWord[1][KeyNameDscPtr->dsc$w_length] = '\0';
   if (Debug) fprintf (stdout, "KeyNameDsc |%s|\n", KeyWord[1]);

   if (VMSnok (status = lbr$lookup_key (&LibraryIndex,
                                        KeyNameDscPtr,
                                        &ModuleRFA)))
   {
      if (Debug) fprintf (stdout, "lib$lookup_key() %%X%08.08X\n", status);
      ErrorVmsStatus (status, CgiPathInfo, LibrarySpec, __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }

   /**************************************/
   /* loop through all records in module */
   /**************************************/

   fputs ("<PRE>", HttpOut);

   for (;;)
   {
      status = lbr$get_record (&LibraryIndex,
                               &InBufferDsc,
                               &OutBufferDsc);
      if (Debug) fprintf (stdout, "lbr$get_record() %%X%08.08X\n", status);
      if (VMSnok (status)) break;
 
      Buffer[OutBufferDsc.dsc$w_length] = '\0';
      sptr = String;
      sptr += CopyIntoHtml (sptr, Buffer, -1);
      /* add HTTP required carriage-control */
      *sptr++ = '\n';
      *sptr = '\0';
      fputs (String, HttpOut);
  }
 
   fputs ("</PRE>\n", HttpOut);

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
*/
 
int SearchTextLibraryModules ()

{
   register char  *cptr, *sptr;
   int  status;
   char  KeyName [256],
         String [512];
   $DESCRIPTOR (KeyNameDsc, KeyName);
 
   /*********/
   /* begin */
   /*********/

   if (Debug) fputs ("SearchTextLibraryModules()\n", stdout);
 
   cptr = KeyName;
   for (sptr = KeyWord[1]; *sptr; *cptr++ = toupper(*sptr++));
   *cptr = '\0';
   KeyNameDsc.dsc$w_length = cptr - KeyName;
   
   fprintf (HttpOut,
"HTTP/1.0 200 Document follows.\r\n\
Date: %s\r\n\
Last-Modified: %s\r\n\
Content-Type: text/html\r\n\
\r\n\
<!-- SoftwareID: %s -->\n\
<TITLE>Search ``%s'' for ``%s''</TITLE>\n\
<H1>%sSearch ``%s'' for ``<TT>%s</TT>''</H1>\n",
   GmDateTime, LastModifiedGmDateTime,
   SoftwareID,
   LibraryPathInfo, HtmlSearchString,
   ConanIconUrl, HtmlLibraryTitle, HtmlSearchString);

   fflush (HttpOut);
   HttpHasBeenOutput = true;

   SearchHitCount = 0;
   IndexNumber = 1;

   if (DoSearchStatistics) lib$init_timer (0);

   if (VMSnok (status = lbr$get_index (&LibraryIndex,
                                       &IndexNumber,
                                       &SearchTextModule,
                                       &KeyNameDsc)))
   {
      if (Debug) fprintf (stdout, "lib$get_index() %%X%08.08X\n", status);
      ErrorVmsStatus (status, CgiPathInfo, LibrarySpec, __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }

   if (!SearchHitCount) fputs ("<P><HR>\n<P><I>(not found)</I>\n", HttpOut);

   fputs ("<P><HR>\n", HttpOut);

   if (DoSearchStatistics) SearchStatistics ();

   return (status);
}

/*****************************************************************************/
/*
*/ 
 
int SearchTextModule
(
struct dsc$descriptor_s  *KeyNameDscPtr,
void  *RFAptr
)
{
   register char  *cptr, *hptr, *sptr;

   boolean  HitThisModule = false;
   int  status;
   char  Buffer [1024],
         String [1024];
   $DESCRIPTOR (InBufferDsc, Buffer);
   $DESCRIPTOR (OutBufferDsc, Buffer);
 
   /*********/
   /* begin */
   /*********/

   if (Debug) fputs ("SearchTextModule()\n", stdout);

   strncpy (KeyName[1],
            KeyNameDscPtr->dsc$a_pointer, KeyNameDscPtr->dsc$w_length);
   KeyName[1][KeyNameDscPtr->dsc$w_length] = '\0';
   CopyIntoHtml (KeyNameHtml[1], KeyName[1], -1);
   if (Debug) fprintf (stdout, "KeyNameDsc |%s|\n", KeyName[1]);

   if (VMSnok (status = lbr$lookup_key (&LibraryIndex,
                                        KeyNameDscPtr,
                                        &ModuleRFA)))
   {
      if (Debug) fprintf (stdout, "lib$get_index() %%X%08.08X\n", status);
      ErrorVmsStatus (status, CgiPathInfo, LibrarySpec, __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }

   /**************************************/
   /* loop through all records in module */
   /**************************************/

   for (;;)
   {
      status = lbr$get_record (&LibraryIndex,
                               &InBufferDsc,
                               &OutBufferDsc);
      if (Debug) fprintf (stdout, "lbr$get_record() %%X%08.08X\n", status);
      if (VMSnok (status)) break;
 
      Buffer[OutBufferDsc.dsc$w_length] = '\0';
      if (Debug) fprintf (stdout, "Buffer |%s|\n", Buffer);

      /* check to see if this module record has any non-space chars */
      for (cptr = Buffer; *cptr && isspace(*cptr); cptr++);
      if (*cptr)
      {
         if (*(hptr = SearchText (cptr, SearchString, false)))
         {
            /********/
            /* hit! */
            /********/

            if (Debug) fprintf (stdout, "Hit |%s|\n", sptr);
            SearchHitCount++;
            sptr = String;
            if (!HitThisModule)
            {
               sptr += sprintf (sptr,
"<HR>\n\
<A HREF=\"%s%s?key=%s&title=%s&referer=%s\"><H2>%s</H2></A>\n<UL>\n",
               CgiScriptName, MapUrl(NULL, LibraryName, NULL, NULL),
               KeyName[1], UriFormTitle, UriReferer, KeyNameHtml[1]);
               HitThisModule = true;
            }
            strcpy (sptr, "<LI>");
            sptr += 4;
            sptr += CopyIntoHtml (sptr, cptr, hptr-cptr);
            strcpy (sptr, "<TT>");
            sptr += 4;
            strncpy (sptr, hptr, SearchStringLength);
            sptr += SearchStringLength;
            strcpy (sptr, "</TT>");
            sptr += 5;
            sptr += CopyIntoHtml (sptr, hptr+SearchStringLength, -1);
            *sptr++ = '\n';
            *sptr = '\0';

            fputs (String, HttpOut);
         }
      }
   }

   if (HitThisModule) fputs ("</UL>\n", HttpOut);

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
Case sensistive or case insensistive search of the string pointed to by 'sptr' 
for the string pointed to by 'SearchString'.
*/ 

char* SearchText
( 
register char *tptr,
register char *SearchString,
register boolean CaseSensitive
)
{
   register char  *cptr, *sptr;

   RecordsSearched++;

   sptr = SearchString;
   if (CaseSensitive)
   {
      while (*tptr)
      {
         if (*tptr++ != *sptr) continue;
         /* first character of search string matched record character */
         sptr++;
         cptr = tptr;
         while (*cptr && *sptr)
         {
            if (*cptr != *sptr) break;
            cptr++;
            sptr++;
         }
         if (!*sptr)
         {
            tptr--;
            break;
         }
         sptr = SearchString;
      }
   }
   else
   {
      while (*tptr)
      {
         if (toupper(*tptr++) != toupper(*sptr)) continue;
         /* first character of search string matched record character */
         sptr++;
         cptr = tptr;
         while (*cptr && *sptr)
         {
            if (toupper(*cptr) != toupper(*sptr)) break;
            cptr++;
            sptr++;
         }
         if (!*sptr)
         {
            tptr--;
            break;
         }
         sptr = SearchString;
      }
   }
   return (tptr);
}

/*****************************************************************************/
/*
*/ 

SearchStatistics ()

{
   static long  LibElapsedTime = 1,
                LibCpuTime = 2,
                LibDirectIO = 4;
   static $DESCRIPTOR (ElapsedFaoDsc, "!%T");
   static $DESCRIPTOR (SearchStatsFaoDsc,
"<I>Elapsed: <TT>!AZ</TT> \
CPU: <TT>!2ZL:!2ZL.!ZL</TT> \
I/O: <TT>!UL</TT> \
Records (lines): <TT>!UL</TT></I>\n");

   unsigned long  ElapsedTime [2];
   unsigned long  CpuTime,
                  DirectIO;
   unsigned short  Length;
   char  Elapsed [32],
         String [512];
   $DESCRIPTOR (ElapsedDsc, Elapsed);
   $DESCRIPTOR (StringDsc, String);

   lib$stat_timer (&LibElapsedTime, &ElapsedTime, 0);
   lib$stat_timer (&LibCpuTime, &CpuTime, 0);
   lib$stat_timer (&LibDirectIO, &DirectIO, 0);

   sys$fao (&ElapsedFaoDsc, &Length, &ElapsedDsc,
            &ElapsedTime);
   Elapsed[Length] = '\0';
   sys$fao (&SearchStatsFaoDsc, &Length, &StringDsc,
            Elapsed+3, CpuTime/6000, CpuTime/100, CpuTime%100,
            DirectIO, RecordsSearched);
   String[Length] = '\0';
   fputs (String, HttpOut);
}

/*****************************************************************************/
/*
Get the contents of the DCL symbol corresponding to the 'VariableName'.
*/

int GetCgiVariable
(
char *VariableName,
char *VariableValue,
int VariableValueSize
)
{
   static $DESCRIPTOR (VariableNameDsc, "");
   static $DESCRIPTOR (VariableValueDsc, "");

   register int  status;

   unsigned short  Length;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "GetCgiVariable() |%s|\n", VariableName);

   VariableNameDsc.dsc$w_length = strlen(VariableName);
   VariableNameDsc.dsc$a_pointer = VariableName;
   VariableValueDsc.dsc$w_length = VariableValueSize-1;
   VariableValueDsc.dsc$a_pointer = VariableValue;

   if (VMSok (status =
       lib$get_symbol (&VariableNameDsc, &VariableValueDsc, &Length, 0)))
      VariableValue[Length] = '\0';
   else
      VariableValue[0] = '\0';

   if (status == LIB$_NOSUCHSYM) return (-1);
   if (VMSnok (status))
   {
      if (Debug) fprintf (stdout, "lib$get_symbol() %%X%08.08X\n", status);
      exit (status);
   }
   if (Debug) fprintf (stdout, "|%s|\n", VariableValue);
   return (Length);
}

/*****************************************************************************/
/*
*/

ErrorGeneral
(
char *Text,
char *SourceFileName,
int SourceLineNumber
)
{
   register char  *cptr;

   /*********/
   /* begin */
   /*********/

   /* 
      The source file format provided by the "__FILE__" macro will be
      "device:[directory]name.type;ver".  Reduce that to "name.type".
   */
   for (cptr = SourceFileName; *cptr && *cptr != ';'; cptr++);
   if (*cptr)
   {
      while (*cptr != '.') cptr--;
      *cptr-- = '\0';
   }
   while (*cptr != ']') cptr--;
   cptr++;

   if (!HttpHasBeenOutput) fputs (Http404Header, HttpOut);
   fprintf (HttpOut,
"<!-- SoftwareID: %s Module: %s Line: %d -->\n\
<H1>ERROR!</H1>\n\
<P>Reported by server.\n\
<P>%s\n",
   SoftwareID, cptr, SourceLineNumber, Text);
}

/*****************************************************************************/
/*
*/

ErrorVmsStatus
(
int StatusValue,
char *Text,
char *HiddenText,
char *SourceFileName,
int SourceLineNumber
)
{
   static char  Message [256];
   static $DESCRIPTOR (MessageDsc, Message);

   register char  *cptr;
   int  status;
   short int  Length;

   /*********/
   /* begin */
   /*********/

   if (VMSok (status = sys$getmsg (StatusValue, &Length, &MessageDsc, 1, 0))) 
   {
      Message[Length] = '\0';
      Message[0] = toupper(Message[0]);
   }
   else
      exit (status);

   /* 
      The source file format provided by the "__FILE__" macro will be
      "device:[directory]name.type;ver".  Reduce that to "name.type".
   */
   for (cptr = SourceFileName; *cptr && *cptr != ';'; cptr++);
   if (*cptr)
   {
      while (*cptr != '.') cptr--;
      *cptr-- = '\0';
   }
   while (*cptr != ']') cptr--;
   cptr++;

   if (!HttpHasBeenOutput) fputs (Http404Header, HttpOut);
   fprintf (HttpOut,
"<!-- SoftwareID: %s Module: %s Line: %d -->\n\
<H1>ERROR!</H1>\n\
<P>Reported by server.\n\
<P>%s ... <TT>%s</TT>\n\
<!-- %%X%08.08X \"%s\" -->\n",
   SoftwareID, cptr, SourceLineNumber, Message, Text, StatusValue, HiddenText);
}

/*****************************************************************************/
/*
If the object has been modified since the specified date and time then return 
a normal status indicating that the data transfer is to continue.  If not 
modified then send a "not modified" HTTP header and return an error status to 
indicate the object should not be sent.
*/ 
 
int ModifiedSince
(
unsigned long *SinceBinaryTimePtr,
unsigned long *BinaryTimePtr
)
{
   static unsigned long  OneSecondDelta [2] = { -10000000, -1 };
   static char  Http304Header [] = "HTTP/1.0 304 Not modified.\r\n\r\n";

   int  status;
   unsigned long  AdjustedBinTime [2],
                  ScratchBinTime [2];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ModifiedSince()\n");

   /*
      Add one second to the modified time.  Ensures a negative time
      for VMS where fractional seconds may result in inconclusive
      results when the target time is being specified in whole seconds.
   */ 
   if (VMSnok (status =
       lib$add_times (SinceBinaryTimePtr, &OneSecondDelta, &AdjustedBinTime)))
   {
      ErrorVmsStatus (status, "If-Modified-Since:", "", __FILE__, __LINE__);
      return (status);
   }
   if (Debug) fprintf (stdout, "sys$add_times() %%X%08.08X\n", status);

   /* if a positive time results the file has been modified */
   if (VMSok (status =
       lib$sub_times (BinaryTimePtr, &AdjustedBinTime, &ScratchBinTime)))
      return (status);

   if (Debug) fprintf (stdout, "sys$sub_times() %%X%08.08X\n", status);
   if (status != LIB$_NEGTIM)
   {
      ErrorVmsStatus (status, "If-Modified-Since:", "", __FILE__, __LINE__);
      return (status);
   }

   fputs (Http304Header, HttpOut);

   return (LIB$_NEGTIM);
}

/*****************************************************************************/
/*
Create an HTTP format Greenwich Mean Time (UTC) time string in the storage 
pointed at by 'TimeString', e.g. "Fri, 25 Aug 1995 17:32:40 GMT" (RFC 1123). 
This must be at least 30 characters capacity.  If 'BinTimePtr' is null the 
time string represents the current time.  If it points to a quadword, VMS time 
value the string represents that time.  'TimeString' must point to storage 
large enough for 31 characters.
*/

int HttpGmTimeString
(
char *TimeString,
unsigned long *BinTimePtr
)
{
   static char  *DayNames [] =
      { "", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat", "Sun" };
   static char  *MonthName [] =
      { "", "Jan", "Feb", "Mar", "Apr", "May", "Jun",
            "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };

   static $DESCRIPTOR (HttpTimeFaoDsc, "!AZ, !2ZW !AZ !4ZW !2ZW:!2ZW:!2ZW GMT");
   static $DESCRIPTOR (TimeStringDsc, "");

   int  status;
   unsigned long  BinTime [2],
                  GmTime [2];
   unsigned short  Length;
   unsigned short  NumTime [7];
   unsigned long  DayOfWeek;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HttpGmTimeString()\n");

   if (BinTimePtr == NULL)
      sys$gettim (&GmTime);
   else
   {
      GmTime[0] = BinTimePtr[0];
      GmTime[1] = BinTimePtr[1];
   }
   if (VMSnok (status = TimeAdjustGMT (true, &GmTime)))
      return (status);

   status = sys$numtim (&NumTime, &GmTime);
   if (Debug)
      fprintf (stdout, "sys$numtim() %%X%08.08X %d %d %d %d %d %d %d\n",
               status, NumTime[0], NumTime[1], NumTime[2],
               NumTime[3], NumTime[4], NumTime[5], NumTime[6]);

   if (VMSnok (status = lib$day_of_week (&GmTime, &DayOfWeek)))
      return (status);
   if (Debug)
      fprintf (stdout, "lib$day_of_week() %%X%08.08X is %d\n",
               status, DayOfWeek);

   /* set the descriptor address and size of the resultant time string */
   TimeStringDsc.dsc$w_length = 30;
   TimeStringDsc.dsc$a_pointer = TimeString;

   if (VMSnok (status =
       sys$fao (&HttpTimeFaoDsc, &Length, &TimeStringDsc,
                DayNames[DayOfWeek], NumTime[2], MonthName[NumTime[1]],
                NumTime[0], NumTime[3], NumTime[4], NumTime[5])))
   {
      if (Debug) fprintf (stdout, "sys$fao() %%X%08.08X\n", status);
      TimeString[0] = '\0';
   }
   else
      TimeString[Length] = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", TimeString);

   return (status);
}

/*****************************************************************************/
/*
Given a string such as "Fri, 25 Aug 1995 17:32:40 GMT" (RFC 1123), or "Friday,
25-Aug-1995 17:32:40 GMT" (RFC 1036), create an internal, local, binary time
with the current GMT offset.  See complementary function HttpGmTimeString().
*/ 

int HttpGmTime
(
char *TimeString,
unsigned long *BinTimePtr
)
{
   static char  *MonthName [] =
      { "", "Jan", "Feb", "Mar", "Apr", "May", "Jun",
            "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };

   register char  *tptr;

   int  status;
   unsigned short  Length;
   unsigned short  NumTime [7] = { 0,0,0,0,0,0,0 };
   unsigned long  DayOfWeek;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HttpGmTime() |%s|\n", TimeString);

   tptr = TimeString;
   /* hunt straight for the comma after the weekday name! */
   while (*tptr && *tptr != ',') tptr++;
   if (*tptr) tptr++;
   /* span white space between weekday name and date */
   while (*tptr && isspace(*tptr)) tptr++;
   if (Debug) fprintf (stdout, "tptr |%s|\n", tptr);
   if (!*tptr) return (STS$K_ERROR);

   /* get the date and then skip to month name */
   if (isdigit(*tptr)) NumTime[2] = atoi (tptr);
   while (*tptr && isdigit(*tptr)) tptr++;
   while (*tptr && (*tptr == '-' || isspace(*tptr))) tptr++;
   if (Debug) fprintf (stdout, "tptr |%s|\n", tptr);
   if (!*tptr) return (STS$K_ERROR);

   /* get the month number from the name and skip to the year */
   for (NumTime[1] = 1; NumTime[1] <= 12; NumTime[1]++)
      if (strsame (tptr, MonthName[NumTime[1]], 3)) break;
   if (NumTime[1] > 12) return (STS$K_ERROR);
   while (*tptr && isalpha(*tptr)) tptr++;
   while (*tptr && (*tptr == '-' || isspace(*tptr))) tptr++;
   if (Debug) fprintf (stdout, "tptr |%s|\n", tptr);
   if (!*tptr) return (STS$K_ERROR);

   /* get the year and then skip to the hour */
   if (isdigit(*tptr))
   {
      NumTime[0] = atoi (tptr);
      if (NumTime[0] < 100) NumTime[0] += 1900;
   }
   while (*tptr && isdigit(*tptr)) tptr++;
   while (*tptr && isspace(*tptr)) tptr++;
   if (Debug) fprintf (stdout, "tptr |%s|\n", tptr);
   if (!*tptr) return (STS$K_ERROR);

   /* get the hour, minute and second */
   if (isdigit(*tptr)) NumTime[3] = atoi (tptr);
   while (*tptr && isdigit(*tptr)) tptr++;
   if (*tptr == ':') tptr++;
   if (isdigit(*tptr)) NumTime[4] = atoi (tptr);
   while (*tptr && isdigit(*tptr)) tptr++;
   if (*tptr == ':') tptr++;
   if (isdigit(*tptr)) NumTime[5] = atoi (tptr);
   while (*tptr && isdigit(*tptr)) tptr++;
   if (*tptr == ':') tptr++;
   if (!*tptr) return (STS$K_ERROR);

   /* the only thing remaining should be the "GMT" */
   while (*tptr && isspace(*tptr)) tptr++;
   if (Debug) fprintf (stdout, "tptr |%s|\n", tptr);
   if (!strsame (tptr, "GMT", 3)) return (STS$K_ERROR);

   /*******************************************/
   /* convert what looks like legitimate GMT! */
   /*******************************************/

   if (Debug)
      fprintf (stdout, "NumTime[] %d %d %d %d %d %d %d\n",
               NumTime[0], NumTime[1], NumTime[2], NumTime[3], 
               NumTime[4], NumTime[5], NumTime[6]); 
   status = lib$cvt_vectim (&NumTime, BinTimePtr);
   if (VMSnok (status)) return (status);
   if (Debug) fprintf (stdout, "lib$cvt_vectim() %%X%08.08X\n", status);

   return (TimeAdjustGMT (false, BinTimePtr));
}

/*****************************************************************************/
/*
Translate the logical HTTPD$GMT (defined to be something like "+10:30" or "-
01:15") and convert it into a delta time structure and store in 
'TimeGmtDeltaBinary'.  Store whether it is in advance or behind GMT in boolean 
'TimeAheadOfGmt'.  Store the logical string in 'TimeGmtString'.
*/

int TimeSetGMT ()

{
   register char  *cptr, *sptr;

   int  status;
   unsigned short  Length;
   $DESCRIPTOR (GmtLogicalNameDsc, "HTTPD$GMT");
   $DESCRIPTOR (LnmFileDevDsc, "LNM$FILE_DEV");
   $DESCRIPTOR (TimeGmtVmsStringDsc, "");
   struct {
      short int  buf_len;
      short int  item;
      unsigned char   *buf_addr;
      unsigned short  *ret_len;
   } LnmItems [] =
   {
      { sizeof(TimeGmtString)-1, LNM$_STRING, TimeGmtString, &Length },
      { 0,0,0,0 }
   };

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "TimeSetGMT()\n");

   status = sys$trnlnm (0, &LnmFileDevDsc, &GmtLogicalNameDsc, 0, &LnmItems);
   if (Debug) fprintf (stdout, "sys$trnlnm() %%X%08.08X\n", status);
   if (VMSnok (status)) return (status);

   TimeGmtString[Length] = '\0';
   if (Debug) fprintf (stdout, "TimeGmtString |%s|\n", TimeGmtString);
   sptr = TimeGmtVmsString;
   if (*(cptr = TimeGmtString) == '-')
      TimeAheadOfGmt = false;
   else
      TimeAheadOfGmt = true;
   *sptr++ = *cptr++;
   *sptr++ = '0';
   *sptr++ = ' ';
   while (*cptr && *cptr != ' ') *sptr++ = *cptr++;
   *sptr++ = '\0';
   if (Debug) fprintf (stdout, "TimeGmtVmsString |%s|\n", TimeGmtVmsString);

   /* set the descriptor address and size of the offset time string */
   if (TimeGmtVmsString[0] == '+' || TimeGmtVmsString[0] == '-')
   {
      TimeGmtVmsStringDsc.dsc$w_length = Length-1;
      TimeGmtVmsStringDsc.dsc$a_pointer = TimeGmtVmsString+1;
   }
   else
   {
      TimeGmtVmsStringDsc.dsc$w_length = Length;
      TimeGmtVmsStringDsc.dsc$a_pointer = TimeGmtVmsString;
   }

   return (sys$bintim (&TimeGmtVmsStringDsc, &TimeGmtDeltaBinary));
}

/*****************************************************************************/
/*
The GMT is generated by calculating using an offset using 
'TimeGmtDeltaOffset' and boolean 'TimeAheadOfGmt'.  Adjust either to or from 
GMT.
*/ 

int TimeAdjustGMT
(
boolean ToGmTime,
unsigned long *BinTimePtr
)
{
   int  status;
   unsigned long  AdjustedTime [2];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "TimeAdjustGMT() ToGmTime: %d\n", ToGmTime);

   if ((ToGmTime && TimeAheadOfGmt) || (!ToGmTime && !TimeAheadOfGmt))
   {
      /* to GMT from local and ahead of GMT, or to local from GMT and behind */
      status = lib$sub_times (BinTimePtr, &TimeGmtDeltaBinary, &AdjustedTime);
      if (Debug) fprintf (stdout, "lib$sub_times() %%X%08.08X\n", status);
   }
   else
   {
      /* to GMT from local and behind GMT, or to local from GMT and ahead */
      status = lib$add_times (BinTimePtr, &TimeGmtDeltaBinary, &AdjustedTime);
      if (Debug) fprintf (stdout, "lib$add_times() %%X%08.08X\n", status);
   }

   if (Debug)
   {
      unsigned short  Length;
      char  String [64];
      $DESCRIPTOR (AdjustedTimeFaoDsc, "AdjustedTime: |!%D|\n");
      $DESCRIPTOR (StringDsc, String);

      sys$fao (&AdjustedTimeFaoDsc, &Length, &StringDsc, &AdjustedTime);
      String[Length] = '\0';
      fputs (String, stdout);
   }

   BinTimePtr[0] = AdjustedTime[0];
   BinTimePtr[1] = AdjustedTime[1];

   return (status);
}

/*****************************************************************************/
/*
Copy text from one string to another, converting characters forbidden to 
appear as plain-text in HTML.  For example the '<', '&', etc.  Convert these 
to the corresponding HTML character entities.  Returns number of characters 
copied into HTML string.
*/ 

int CopyIntoHtml
( 
register char *hptr,
register char *sptr,
register int ccnt
)
{
   char  *FirstPtr;

   /*********/
   /* begin */
   /*********/

   FirstPtr = hptr;
   while (*sptr && ccnt--)
   {
      switch (*sptr)
      {
         case '<' : strcpy (hptr, "&lt;"); hptr += 4; sptr++; break;
         case '>' : strcpy (hptr, "&gt;"); hptr += 4; sptr++; break;
         case '&' : strcpy (hptr, "&amp;"); hptr += 5; sptr++; break;
         case '\"' : strcpy (hptr, "&quot;"); hptr += 6; sptr++; break;
         case '\t' : *hptr++ = *sptr++; break;
         case '\n' : *hptr++ = *sptr++; break;
         case '\r' : *hptr++ = *sptr++; break;
         default : if (isprint(*sptr)) *hptr++ = *sptr++; else sptr++;
      }
   }
   *hptr = '\0';
   return (hptr-FirstPtr);
}

/*****************************************************************************/
/*
Copy text from one string to another, converting characters forbidden to 
appear as plain-text text in an HTTP URL.  For example the '?', '+', '&', etc.  
Convert these to "%nn" hexadecimal escaped characters.
*/ 

int CopyIntoUri
( 
register char *uptr,
register char *sptr,
register int  ccnt
)
{
   char  *FirstPtr;

   /*********/
   /* begin */
   /*********/

   FirstPtr = uptr;
   while (*sptr && ccnt--)
   {  
      switch (*sptr)
      {
         case '\t' : strcpy (uptr, "%09"); uptr += 3; sptr++; break;
         case ' ' : strcpy (uptr, "%20"); uptr += 3; sptr++; break;
         case '!' : strcpy (uptr, "%21"); uptr += 3; sptr++; break;
         case '\"' : strcpy (uptr, "%22"); uptr += 3; sptr++; break;
         case '#' : strcpy (uptr, "%23"); uptr += 3; sptr++; break;
         case '$' : strcpy (uptr, "%24"); uptr += 3; sptr++; break;
         case '%' : strcpy (uptr, "%25"); uptr += 3; sptr++; break;
         case '&' : strcpy (uptr, "%26"); uptr += 3; sptr++; break;
         case '\'' : strcpy (uptr, "%27"); uptr += 3; sptr++; break;
         case '(' : strcpy (uptr, "%28"); uptr += 3; sptr++; break;
         case ')' : strcpy (uptr, "%29"); uptr += 3; sptr++; break;
         case '*' : strcpy (uptr, "%2a"); uptr += 3; sptr++; break;
         case '+' : strcpy (uptr, "%2b"); uptr += 3; sptr++; break;
         case ',' : strcpy (uptr, "%2c"); uptr += 3; sptr++; break;
         case '-' : strcpy (uptr, "%2d"); uptr += 3; sptr++; break;
         case '.' : strcpy (uptr, "%2e"); uptr += 3; sptr++; break;
         case '/' : strcpy (uptr, "%2f"); uptr += 3; sptr++; break;
         case ':' : strcpy (uptr, "%3a"); uptr += 3; sptr++; break;
         case ';' : strcpy (uptr, "%3b"); uptr += 3; sptr++; break;
         case '<' : strcpy (uptr, "%3c"); uptr += 3; sptr++; break;
         case '=' : strcpy (uptr, "%3d"); uptr += 3; sptr++; break;
         case '>' : strcpy (uptr, "%3e"); uptr += 3; sptr++; break;
         case '?' : strcpy (uptr, "%3f"); uptr += 3; sptr++; break;
         case '[' : strcpy (uptr, "%5b"); uptr += 3; sptr++; break;
         case '\\' : strcpy (uptr, "%5c"); uptr += 3; sptr++; break;
         case ']' : strcpy (uptr, "%5d"); uptr += 3; sptr++; break;
         case '^' : strcpy (uptr, "%5e"); uptr += 3; sptr++; break;
         case '_' : strcpy (uptr, "%5f"); uptr += 3; sptr++; break;
         case '{' : strcpy (uptr, "%7b"); uptr += 3; sptr++; break;
         case '|' : strcpy (uptr, "%7c"); uptr += 3; sptr++; break;
         case '}' : strcpy (uptr, "%7d"); uptr += 3; sptr++; break;
         case '~' : strcpy (uptr, "%7e"); uptr += 3; sptr++; break;
         default : if (isprint(*sptr)) *uptr++ = *sptr++; else sptr++;
      }
   }
   *uptr = '\0';
   return (uptr-FirstPtr);
}

/****************************************************************************/
/*
Does a case-insensitive, character-by-character string compare and returns 
true if two strings are the same, or false if not.  If a maximum number of 
characters are specified only those will be compared, if the entire strings 
should be compared then specify the number of characters as 0.
*/ 

boolean strsame
(
register char *sptr1,
register char *sptr2,
register int  count
)
{
   while (*sptr1 && *sptr2)
   {
      if (toupper (*sptr1++) != toupper (*sptr2++)) return (false);
      if (count)
         if (!--count) return (true);
   }
   if (*sptr1 || *sptr2)
      return (false);
   else
      return (true);
}

/*****************************************************************************/

