/*****************************************************************************/
/*
                                  Dtree.c

A CGI-compliant script.

Given a directory node to start from, display a directory tree with links 
allowing any node to be selected and displayed as an "Index of ..." directory 
listing by the HTTPd.  Honours the HTTPd ".WWW_HIDDEN" file method of 
preventing directory listings being taken of a directory. 

In line with the HFRD HTTPd, appending a query string of "&format=vms" (note
the leading null query element represented by the leading "&") to the directory
path will result in a VMS-style directory tree, and with this query string
being propagated to the HTTPd server where it will likewise result in the
'directory.c' module producing a VMS-style directory listing.


QUALIFIERS
----------
/DBUG           turns on all "if (Debug)" statements
/ICON=          directory icon URL


BUILD DETAILS
-------------
See BUILD_DTREE.COM procedure.


VERSION HISTORY (update SoftwareID as well!)
---------------
19-SEP-95  MGD  v1.2.1, replace <CR><LF> carriage-control with single newline,
                        still acceptable for HTTP, and slightly more efficient
08-AUG-95  MGD  v1.2.0, VMS-style directory listings, propagate query string
24-MAY-95  MGD  v1.1.0, minor changes for AXP compatibility
10-APR-95  MGD  v1.0.0, initial development
*/
/*****************************************************************************/

#ifdef __ALPHA
   char SoftwareID [] = "DTREE AXP-1.2.1";
#else
   char SoftwareID [] = "DTREE VAX-1.2.1";
#endif

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <errno.h>
#include <unixio.h>

/* VMS related header files */
#include <descrip.h>
#include <libdef.h>
#include <rmsdef.h>
#include <rms.h>
#include <ssdef.h>
#include <stsdef.h>

#ifdef __ALPHA
#   pragma nomember_alignment
#endif

#define boolean int
#define true 1
#define false 0
 
#define VMSok(x) ((x) & STS$M_SUCCESS)
#define VMSnok(x) !(((x) & STS$M_SUCCESS))
 
#define LeftIndent 3

char  Utility [] = "DTREE";

char DirectoryHiddenFileName [] = ".WWW_HIDDEN";
int DirectoryHiddenFileNameLength = sizeof(DirectoryHiddenFileName)-1;

char  Http200Header [] =
"HTTP/1.0 200 Document follows.\n\
Content-Type: text/html\n\
\n";

char  Http404Header [] =
"HTTP/1.0 404 Error report follows.\n\
Content-Type: text/html\n\
\n";

boolean  Debug,
         HttpHasBeenOutput;

char  CgiRequestMethod [16],
      CgiPathInfo [256],
      CgiPathTranslated [256],
      CgiQueryString [256],
      FileSpecification [256],
      DirIconUrl [256],
      DirIconImage [256];

char  *QuestionMarkPtr,
      *QueryStringPtr;

FILE  *HttpOut;

/* required function prototypes */
char* MapUrl (char*, char*, char*, char*);

/*****************************************************************************/
/*
*/

main (int argc, char *argv[])

{
   register int  acnt;
   register char  *cptr, *sptr;

   int  status;
   char  String [256];

   /*********/
   /* begin */
   /*********/

   /* open another output stream so that the "\n" are not filtered */
#ifdef __DECC
   if ((HttpOut = fopen ("SYS$OUTPUT", "w", "ctx=bin")) == NULL)
      exit (vaxc$errno);
#else
   if ((HttpOut = fopen ("SYS$OUTPUT", "w", "rfm=udf")) == NULL)
      exit (vaxc$errno);
#endif

   /***********************************/
   /* get the command line parameters */
   /***********************************/

   /* using this mechanism parameters must be space-separated! */
   for (acnt = 1; acnt < argc; acnt++)
   {
      if (Debug) fprintf (stdout, "argv[%d] |%s|\n", acnt, argv[acnt]);
      if (strsame (argv[acnt], "/DBUG", -1))
      {
         Debug = true;
         continue;
      }
      if (strsame (argv[acnt], "/ICON=", 5))
      {
         for (cptr = argv[acnt]+5; *cptr && *cptr != '='; cptr++);
         if (*cptr) cptr++;
         sptr = DirIconUrl;
         while (*cptr) *sptr++ = *cptr++;
         *sptr = '\0';
         sprintf (DirIconImage, "<IMG ALIGN=top SRC=\"%s\" ALT=\"[DIR]\"> ",
                  DirIconUrl);
         continue;
      }
      fprintf (stdout, "%%%s-E-IVQUAL, unrecognized qualifier\n \\%s\\\n",
               Utility, argv[acnt]+1);
      exit (STS$K_ERROR | STS$M_INHIB_MSG);
   }

   /*********************/
   /* get CGI variables */
   /*********************/

   GetCgiVariable ("WWW_REQUEST_METHOD", CgiRequestMethod,
                   sizeof(CgiRequestMethod));

   /* numeric equivalent of "GET\0" (just a bit more efficient, that's all!) */
   if (*(unsigned long*)CgiRequestMethod != 0x00544547) exit (SS$_NORMAL);

   GetCgiVariable ("WWW_QUERY_STRING", CgiQueryString, sizeof(CgiQueryString));
   GetCgiVariable ("WWW_PATH_INFO", CgiPathInfo, sizeof(CgiPathInfo));
   GetCgiVariable ("WWW_PATH_TRANSLATED", CgiPathTranslated, 
                   sizeof(CgiPathTranslated));

   DirectoryTree ();

   exit (SS$_NORMAL);
}

/*****************************************************************************/
/*
Recursively calls function DirectoryTreeRecursion() to implement the 
hierarchical display.
*/ 

DirectoryTree ()

{
   register char  *cptr, *sptr;
   int  status,
        DirectoryLength;
   char  Directory [256],
         DirectoryPath [256],
         DirName [256];
   char  *TreeOfPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "DirectoryTree() |%s|%s|\n",
               CgiPathInfo, CgiPathTranslated);

   /* if a query string is supplied then be sure to propagate it */
   if (CgiQueryString[0])
   {
      QuestionMarkPtr = "?";
      QueryStringPtr = CgiQueryString;
   }
   else
      QuestionMarkPtr = QueryStringPtr = "";

   /* get the directory part from what may be a file specification */
   sptr = DirectoryPath;
   for (cptr = CgiPathInfo; *cptr; *sptr++ = *cptr++);
   *sptr = '\0';
   while (sptr > DirectoryPath && *sptr != '/') sptr--;
   if (*sptr == '/')
   {
      sptr++;
      strcpy (FileSpecification, sptr);
   }
   *sptr = '\0';
   if (Debug) fprintf (stdout, "|%s|%s|\n", DirectoryPath, FileSpecification);
   /* map the URL-style directory to a VMS file specification */
   Directory[0] = '\0';
   if (!*(sptr = MapUrl (DirectoryPath, Directory, NULL, NULL)))
   {
      ErrorGeneral (sptr+1, __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }
   if (Debug) fprintf (stdout, "Directory |%s|\n", Directory);
   DirectoryLength = strlen(Directory);

   if (DirectoryHidden (Directory, DirectoryLength))
   {
      /* a bit of dis-information :^) if the directory is hidden */
      ErrorVmsStatus (RMS$_DNF, MapUrl(NULL,Directory,NULL,NULL), Directory,
                      __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }

   /* get the name of the last directory in the path */
   for (cptr = DirectoryPath; *cptr; cptr++);
   cptr--;
   if (*cptr == '/') cptr--;
   while (cptr > DirectoryPath && *cptr != '/') cptr--;
   if (*cptr == '/') cptr++; 
   sptr = DirName;
   while (*cptr && *cptr != '/') *sptr++ = toupper(*cptr++);
   *sptr = '\0';

   /* check if the the tree should be presented in a VMS-style format */
   TreeOfPtr = DirectoryPath;
   if (CgiQueryString[0] == '&')
   {
      cptr = CgiQueryString + 1;
      while (*cptr)
      {
         sptr = cptr;
         while (*cptr && *cptr != '&') cptr++;
         if (strsame (sptr, "format=vms", cptr-sptr)) TreeOfPtr = Directory;
         if (*cptr) cptr++;
      }
   }

   fprintf (HttpOut,
"%s\
<!-- SoftwareID: %s -->\n\
<!-- %s -->\n\
<HTML>\n\
<TITLE>Tree of %s</TITLE>\n\
<H1>Tree of <TT>%s</TT></H1>\n\
<PRE><HR>%s<A HREF=\"%s%s%s\">%s</A>\n",
   Http200Header,
   SoftwareID,
   Directory,
   TreeOfPtr,
   TreeOfPtr,
   DirIconImage, CgiPathInfo, QuestionMarkPtr, QueryStringPtr, DirName);
   fflush (HttpOut);
   HttpHasBeenOutput = true;

   DirectoryTreeRecursion (Directory, DirectoryLength);

   fputs ("</PRE>\n</HTML>\n", HttpOut);
}

/*****************************************************************************/
/*
Searches the directory of the supplied file specification for other 
directories, calling itself to recursively display a nested hierarchy of 
directories.
*/ 

DirectoryTreeRecursion
(
char *Directory,
int DirectoryLength
)
{
   static int  RecursionLevel = 0;

   register char  *cptr, *sptr;

   int  status,
        SubDirectoryLength;
   char  ExpandedFileSpec [256],
         FileName [256],
         DirName [256],
         SubDirectory [256],
         SubDirectoryPath [256];
   struct FAB  SearchFab;
   struct NAM  SearchNam;

   /*********/
   /* begin */
   /*********/

   RecursionLevel++;
   if (Debug)
      fprintf (stdout, "DirectoryTreeRecursion() level %d |%s|%d|\n",
               RecursionLevel, Directory, DirectoryLength);

   /****************************/
   /* parse file specification */
   /****************************/

   SearchFab = cc$rms_fab;
   SearchFab.fab$l_dna = Directory;
   SearchFab.fab$b_dns = DirectoryLength;
   SearchFab.fab$l_fna = "*.DIR;";
   SearchFab.fab$b_fns = 6;
   SearchFab.fab$l_fop = FAB$M_NAM;
   SearchFab.fab$l_nam = &SearchNam;
   SearchNam = cc$rms_nam;
   SearchNam.nam$l_esa = ExpandedFileSpec;
   SearchNam.nam$b_ess = sizeof(ExpandedFileSpec)-1;
   SearchNam.nam$l_rsa = FileName;
   SearchNam.nam$b_rss = sizeof(FileName)-1;

   if (VMSnok (status = sys$parse (&SearchFab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$parse() %%X%08.08X\n", status);

      if ((SearchNam.nam$l_fnb & NAM$M_SEARCH_LIST) && status == RMS$_DNF)
         status = RMS$_FNF;

      ErrorVmsStatus (status, MapUrl(NULL,Directory,NULL,NULL), Directory,
                      __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }
   ExpandedFileSpec[SearchNam.nam$b_esl] = '\0';
   if (Debug) fprintf (stdout, "ExpandedFileSpec |%s|\n", ExpandedFileSpec);

   /******************************/
   /* directory file search loop */
   /******************************/

   while (VMSok (status = sys$search (&SearchFab, 0, 0)))
   {
      FileName[SearchNam.nam$b_rsl] = '\0';
      if (Debug) fprintf (stdout, "FileName |%s|\n", FileName);

      /* create a device:[dir.subdir]" from a "device:[dir]subdir.dir" */
      sptr = SubDirectory;
      for (cptr = FileName; *cptr && *cptr != ';'; *sptr++ = *cptr++);
      *sptr = '\0';
      while (sptr > SubDirectory && *sptr != '.') sptr--;
      *sptr++ = ']';
      *sptr = '\0';
      SubDirectoryLength = sptr - SubDirectory + 1;
      sptr -= 2;
      while (sptr > SubDirectory && *sptr != ']') sptr--;
      *sptr = '.';
      if (Debug) fprintf (stdout, "SubDirectory |%s|\n", SubDirectory);

      /* if the directory is hidden then just ignore it! */
      if (DirectoryHidden (SubDirectory, SubDirectoryLength))
         continue;

      /* get the name of the last directory in the path */
      cptr = sptr + 1;
      sptr = DirName;
      while (*cptr && *cptr != ']') *sptr++ = *cptr++;
      *sptr = '\0';
      if (Debug) fprintf (stdout, "DirName |%s|\n", DirName);

      /* map the VMS file specification to a URL-style directory */
      SubDirectoryPath[0] = '\0';
      if (!*(sptr = MapUrl (SubDirectoryPath, SubDirectory, NULL, NULL)))
      {
         ErrorGeneral (sptr+1, __FILE__, __LINE__);
         exit (SS$_NORMAL);
      }
      if (Debug) fprintf (stdout, "SubDirectoryPath |%s|\n", SubDirectoryPath);

      fprintf (HttpOut, "%*.s%s<A HREF=\"%s%s%s%s\">%s</A>\n",
              RecursionLevel*LeftIndent, "", DirIconImage,
              SubDirectoryPath, FileSpecification,
              QuestionMarkPtr, QueryStringPtr, DirName);

      /* recursively call this function to check for any subdirectories */
      DirectoryTreeRecursion (SubDirectory, SubDirectoryLength);
   }

   /*******************/
   /* end search loop */
   /*******************/

   if ((SearchNam.nam$l_fnb & NAM$M_SEARCH_LIST) && status == RMS$_DNF)
      status = RMS$_FNF;
   if (status == RMS$_FNF || status == RMS$_NMF) status = SS$_NORMAL;
   if (VMSnok (status))
   {
      ErrorVmsStatus (status, MapUrl(NULL,Directory,NULL,NULL), Directory,
                      __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }
   if (Debug)
      fprintf (stdout, "DirectoryTreeRecursion() returning to level %d\n",
               RecursionLevel-1);
   RecursionLevel--;
}

/*****************************************************************************/
/*
Returns SS$_NORMAL if directory is not "hidden", RMS$_DNF if it is "hidden", 
any other status if an error occurs.
*/ 

boolean DirectoryHidden
(
char *Directory,
int DirectoryLength
)
{
   int  status;
   char  ExpandedFileName [256];
   struct FAB  SearchFab;
   struct NAM  SearchNam;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DirectoryHidden() |%s|\n", Directory);

   SearchFab = cc$rms_fab;
   SearchFab.fab$l_dna = Directory;
   SearchFab.fab$b_dns = DirectoryLength;
   SearchFab.fab$l_fna = DirectoryHiddenFileName;
   SearchFab.fab$b_fns = DirectoryHiddenFileNameLength;
   SearchFab.fab$l_fop = FAB$M_NAM;
   SearchFab.fab$l_nam = &SearchNam;

   SearchNam = cc$rms_nam;
   SearchNam.nam$l_esa = ExpandedFileName;
   SearchNam.nam$b_ess = sizeof(ExpandedFileName)-1;

   if (VMSnok (status = sys$parse (&SearchFab, 0, 0)))
   {
      if ((SearchNam.nam$l_fnb & NAM$M_SEARCH_LIST) && status == RMS$_DNF)
         status = RMS$_FNF;

      ErrorVmsStatus (status, MapUrl(NULL,Directory,NULL,NULL), Directory,
                      __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }

   status = sys$search (&SearchFab, 0, 0);

   /* release parse and search internal data structures */
   SearchNam.nam$b_nop = NAM$M_SYNCHK;
   sys$parse (&SearchFab, 0, 0);

   /* if found return a little dis-information :^) */
   if (VMSok (status)) return (true);
   /* if the file was not found then its not a "hidden" directory */
   if (status == RMS$_FNF) return (false);

   /* if its a search list treat directory not found as if file not found */
   if ((SearchNam.nam$l_fnb & NAM$M_SEARCH_LIST) && status == RMS$_DNF)
      return (false);

   ErrorVmsStatus (status, MapUrl(NULL,Directory,NULL,NULL), Directory,
                   __FILE__, __LINE__);
   exit (SS$_NORMAL);
}

/*****************************************************************************/
/*
*/

boolean GetCgiVariable
(
char *VariableName,
char *VariableValue,
int VariableValueSize
)
{
   static $DESCRIPTOR (VariableNameDsc, "");
   static $DESCRIPTOR (VariableValueDsc, "");

   int  status;
   unsigned short  Length;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "GetCgiVariable() |%s|\n", VariableName);

   VariableNameDsc.dsc$w_length = strlen(VariableName);
   VariableNameDsc.dsc$a_pointer = VariableName;
   VariableValueDsc.dsc$w_length = VariableValueSize-1;
   VariableValueDsc.dsc$a_pointer = VariableValue;

   if (VMSok (status =
       lib$get_symbol (&VariableNameDsc, &VariableValueDsc, &Length, 0)))
   {
      VariableValue[Length] = '\0';
      if (Debug) fprintf (stdout, "|%s|\n", VariableValue);
      return (true);
   }
   else
   {
      if (Debug) fprintf (stdout, "lib$get_symbol() %%X%08.08X\n", status);
      VariableValue[0] = '\0';
      if (status == LIB$_NOSUCHSYM) return (false);
      exit (status);
   }
}

/*****************************************************************************/
/*
*/

ErrorGeneral
(
char *Text,
char *SourceFileName,
int SourceLineNumber
)
{
   register char  *cptr;

   /*********/
   /* begin */
   /*********/

   /* 
      The source file format provided by the "__FILE__" macro will be
      "device:[directory]name.type;ver".  Reduce that to "name.type".
   */
   for (cptr = SourceFileName; *cptr && *cptr != ';'; cptr++);
   if (*cptr)
   {
      while (*cptr != '.') cptr--;
      *cptr-- = '\0';
   }
   while (*cptr != ']') cptr--;
   cptr++;

   if (!HttpHasBeenOutput) fputs (Http404Header, HttpOut);
   fprintf (HttpOut,
"<!-- SoftwareID: %s Module: %s Line: %d -->\n\
<H1>ERROR!</H1>\n\
<P>Reported by server.\n\
<P>%s\n",
   SoftwareID, cptr, SourceLineNumber, Text);
}

/*****************************************************************************/
/*
*/

ErrorVmsStatus
(
int StatusValue,
char *Text,
char *HiddenText,
char *SourceFileName,
int SourceLineNumber
)
{
   static char  Message [256];
   static $DESCRIPTOR (MessageDsc, Message);

   register char  *cptr;
   int  status;
   short int  Length;

   /*********/
   /* begin */
   /*********/

   if (VMSok (status = sys$getmsg (StatusValue, &Length, &MessageDsc, 1, 0))) 
   {
      Message[Length] = '\0';
      Message[0] = toupper(Message[0]);
   }
   else
      exit (status);

   /* 
      The source file format provided by the "__FILE__" macro will be
      "device:[directory]name.type;ver".  Reduce that to "name.type".
   */
   for (cptr = SourceFileName; *cptr && *cptr != ';'; cptr++);
   if (*cptr)
   {
      while (*cptr != '.') cptr--;
      *cptr-- = '\0';
   }
   while (*cptr != ']') cptr--;
   cptr++;

   if (!HttpHasBeenOutput) fputs (Http404Header, HttpOut);
   fprintf (HttpOut,
"<!-- SoftwareID: %s Module: %s Line: %d -->\n\
<H1>ERROR!</H1>\n\
<P>Reported by server.\n\
<P>%s ... <TT>%s</TT>\n\
<!-- %%X%08.08X \"%s\" -->\n",
   SoftwareID, cptr, SourceLineNumber, Message, Text, StatusValue, HiddenText);
}

/****************************************************************************/
/*
Does a case-insensitive, character-by-character string compare and returns 
true if two strings are the same, or false if not.  If a maximum number of 
characters are specified only those will be compared, if the entire strings 
should be compared then specify the number of characters as 0.
*/ 

boolean strsame
(
register char *sptr1,
register char *sptr2,
register int  count
)
{
   while (*sptr1 && *sptr2)
   {
      if (toupper (*sptr1++) != toupper (*sptr2++)) return (false);
      if (count)
         if (!--count) return (true);
   }
   if (*sptr1 || *sptr2)
      return (false);
   else
      return (true);
}

/*****************************************************************************/


