/*****************************************************************************/
/*
                                Extract.c


CGI-compliant script to extract a specified range of lines from a plain text
file.


QUALIFIERS
----------
/DBUG                   turns on all "if (Debug)" statements
/HTML=                  list of comma separated HTML file types
/TEXT=                  list of comma separated TEXT file types


BUILD DETAILS
-------------
See BUILD_EXTRACT.COM procedure.


VERSION HISTORY (update SoftwareID as well!)
---------------
19-SEP-95  MGD  v2.2.1, replace <CR><LF> carriage-control with single newline,
                        still acceptable for HTTP, and slightly more efficient;
                        added 'previous' and 'next' number of lines link
24-MAY-95  MGD  v2.2.0, minor changes for AXP compatibility
27-MAR-95  MGD  v2.1.0, modifications to CGI interface
05-DEC-94  MGD  v2.0.0, major revision, URL mapping, CGI-like interface
10-JUN-94  MGD  v1.0.0, initial development
*/
/*****************************************************************************/

#ifdef __ALPHA
   char SoftwareID [] = "EXTRACT AXP-2.2.1";
#else
   char SoftwareID [] = "EXTRACT VAX-2.2.1";
#endif

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <errno.h>

/* VMS-related header files */
#include <descrip.h>
#include <libdef.h>
#include <rmsdef.h>
#include <rms.h>
#include <ssdef.h>
#include <stsdef.h>

#ifdef __ALPHA
#   pragma nomember_alignment
#endif

#define boolean int
#define true 1
#define false 0
 
#define VMSok(x) ((x) & STS$M_SUCCESS)
#define VMSnok(x) !(((x) & STS$M_SUCCESS))

#define DefaultExtractNumberOfRecords 20

char  Utility [] = "EXTRACT";

char  Http200Header [] =
"HTTP/1.0 200 Document follows.\n\
Content-Type: text/html\n\
\n";

char  Http404Header [] =
"HTTP/1.0 404 Error report follows.\n\
Content-Type: text/html\n\
\n";

boolean  Debug,
         HttpHasBeenOutput;

int  ExtractNumberOfRecords;

char  FormExact [16],
      HtmlFileTypes [256],
      CgiPathInfo [256],
      CgiPathTranslated [256],
      CgiRequestMethod [256],
      CgiServerName [256],
      CgiScriptName [256],
      TextFileTypes [256];
      
FILE  *HttpOut;

/*****************************************************************************/
/*
*/

main (int argc, char *argv[])

{
   register int  acnt;
   register char  *cptr, *sptr;

   boolean  CaseSensitive = false,
            DocumentOnly = false;
   int  status,
        EndRecordNumber,
        StartRecordNumber;
   char  FormCase [16],
         FormEnd [16],
         FormHighlight [256],
         FormSearch [256],
         FormStart [16],
         KeyCount [16];

   /*********/
   /* begin */
   /*********/

   /* open another output stream so that the "\n" are not filtered */
#ifdef __DECC
   if ((HttpOut = fopen ("SYS$OUTPUT", "w", "ctx=bin")) == NULL)
      exit (vaxc$errno);
#else
   if ((HttpOut = fopen ("SYS$OUTPUT", "w", "rfm=udf")) == NULL)
      exit (vaxc$errno);
#endif

   /***********************************/
   /* get the command line parameters */
   /***********************************/

   for (acnt = 1; acnt < argc; acnt++)
   {
      if (Debug) fprintf (stdout, "argv[%d] |%s|\n", acnt, argv[acnt]);
      if (strsame (argv[acnt], "/DBUG", -1))
      {
         Debug = true;
         continue;
      }
      if (strsame (argv[acnt], "/HTML=", 5))
      {
         for (cptr = argv[acnt]+4; *cptr && *cptr != '='; cptr++);
         if (*cptr) cptr++;
         sptr = HtmlFileTypes;
         while (*cptr) *sptr++ = toupper(*cptr++);
         *sptr = '\0';
         continue;
      }
      if (strsame (argv[acnt], "/TEXT=", 5))
      {
         for (cptr = argv[acnt]+4; *cptr && *cptr != '='; cptr++);
         if (*cptr) cptr++;
         sptr = TextFileTypes;
         while (*cptr) *sptr++ = toupper(*cptr++);
         *sptr = '\0';
         continue;
      }
      fprintf (stdout, "%%%s-E-IVQUAL, unrecognized qualifier\n \\%s\\\n",
               Utility, argv[acnt]+1);
      exit (STS$K_ERROR | STS$M_INHIB_MSG);
   }

   if (Debug)
      fprintf (stdout,
               "HtmlFileTypes |%s|\nTextFileTypes |%s|\n",
               HtmlFileTypes, TextFileTypes);
   if (!HtmlFileTypes[0] || !TextFileTypes[0])
   {
      ErrorGeneral ("File extensions not configured.", __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }

   /*************************/
   /* get the CGI variables */
   /*************************/

   GetCgiVariable ("WWW_SERVER_NAME", CgiServerName, sizeof(CgiServerName));
   GetCgiVariable ("WWW_PATH_INFO", CgiPathInfo, sizeof(CgiPathInfo));
   GetCgiVariable ("WWW_PATH_TRANSLATED", CgiPathTranslated,
                   sizeof(CgiPathTranslated));
   GetCgiVariable ("WWW_REQUEST_METHOD", CgiRequestMethod,
                   sizeof(CgiRequestMethod));
   GetCgiVariable ("WWW_SCRIPT_NAME", CgiScriptName, sizeof(CgiScriptName));

   FormCase[0] = FormExact[0] = FormHighlight[0] = FormEnd[0] =
      FormStart[0] = '\0';
   GetCgiVariable ("WWW_FORM_CASE", FormCase, sizeof(FormCase));
   GetCgiVariable ("WWW_FORM_END", FormEnd, sizeof(FormEnd));
   GetCgiVariable ("WWW_FORM_EXACT", FormExact, sizeof(FormExact));
   GetCgiVariable ("WWW_FORM_HIGHLIGHT", FormHighlight, sizeof(FormHighlight));
   GetCgiVariable ("WWW_FORM_START", FormStart, sizeof(FormStart));

   /***********/
   /* process */
   /***********/

   /* numeric equivalent of "GET\0" (just a bit more efficient, that's all!) */
   if (*(unsigned long*)CgiRequestMethod != 0x00544547) exit (SS$_NORMAL);

   if (toupper(FormCase[0]) == 'Y')
      CaseSensitive = true;
   else
      CaseSensitive = false;

   FormExact[0] = toupper(FormExact[0]);

   StartRecordNumber = atoi (FormStart);
   EndRecordNumber = atoi (FormEnd);

   for (sptr = CgiPathTranslated; *sptr; sptr++);
   while (*sptr != '.') sptr--;
   if (SameFileType (TextFileTypes, sptr))
   {
      status = ExtractFile (CgiPathTranslated, CgiPathInfo,
                            FormHighlight,
                            StartRecordNumber, EndRecordNumber,
                            CaseSensitive);

      if (VMSnok (status))
         ErrorVmsStatus (status, CgiPathInfo, CgiPathTranslated, __FILE__, __LINE__);
   }
   else
      ErrorGeneral ("Only plain text files may be <I>extracted</I>.",
                    __FILE__, __LINE__);

   exit (SS$_NORMAL);
}

/*****************************************************************************/
/*
Get the contents of the DCL symbol corresponding to the 'VariableName'.
*/

int GetCgiVariable
(
char *VariableName,
char *VariableValue,
int VariableValueSize
)
{
   static $DESCRIPTOR (VariableNameDsc, "");
   static $DESCRIPTOR (VariableValueDsc, "");

   register int  status;

   unsigned short  Length;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "GetCgiVariable() |%s|\n", VariableName);

   VariableNameDsc.dsc$w_length = strlen(VariableName);
   VariableNameDsc.dsc$a_pointer = VariableName;
   VariableValueDsc.dsc$w_length = VariableValueSize-1;
   VariableValueDsc.dsc$a_pointer = VariableValue;

   if (VMSok (status =
       lib$get_symbol (&VariableNameDsc, &VariableValueDsc, &Length, 0)))
      VariableValue[Length] = '\0';
   else
      VariableValue[0] = '\0';

   if (status == LIB$_NOSUCHSYM) return (-1);
   if (VMSnok (status))
   {
      if (Debug) fprintf (stdout, "lib$get_symbol() %%X%08.08X\n", status);
      exit (status);
   }
   if (Debug) fprintf (stdout, "|%s|\n", VariableValue);
   return (Length);
}

/*****************************************************************************/
/*
*/

ErrorGeneral
(
char *Text,
char *SourceFileName,
int SourceLineNumber
)
{
   register char  *cptr;

   /*********/
   /* begin */
   /*********/

   /* 
      The source file format provided by the "__FILE__" macro will be
      "device:[directory]name.type;ver".  Reduce that to "name.type".
   */
   for (cptr = SourceFileName; *cptr && *cptr != ';'; cptr++);
   if (*cptr)
   {
      while (*cptr != '.') cptr--;
      *cptr-- = '\0';
   }
   while (*cptr != ']') cptr--;
   cptr++;

   if (!HttpHasBeenOutput) fputs (Http404Header, HttpOut);
   fprintf (HttpOut,
"<!-- SoftwareID: %s Module: %s Line: %d -->\n\
<H1>ERROR!</H1>\n\
<P>Reported by server.\n\
<P>%s\n",
   SoftwareID, cptr, SourceLineNumber, Text);
}

/*****************************************************************************/
/*
*/

ErrorVmsStatus
(
int StatusValue,
char *Text,
char *HiddenText,
char *SourceFileName,
int SourceLineNumber
)
{
   static char  Message [256];
   static $DESCRIPTOR (MessageDsc, Message);

   register char  *cptr;
   int  status;
   short int  Length;

   /*********/
   /* begin */
   /*********/

   if (VMSok (status = sys$getmsg (StatusValue, &Length, &MessageDsc, 1, 0))) 
   {
      Message[Length] = '\0';
      Message[0] = toupper(Message[0]);
   }
   else
      exit (status);

   /* 
      The source file format provided by the "__FILE__" macro will be
      "device:[directory]name.type;ver".  Reduce that to "name.type".
   */
   for (cptr = SourceFileName; *cptr && *cptr != ';'; cptr++);
   if (*cptr)
   {
      while (*cptr != '.') cptr--;
      *cptr-- = '\0';
   }
   while (*cptr != ']') cptr--;
   cptr++;

   if (!HttpHasBeenOutput) fputs (Http404Header, HttpOut);
   fprintf (HttpOut,
"<!-- SoftwareID: %s Module: %s Line: %d -->\n\
<H1>ERROR!</H1>\n\
<P>Reported by server.\n\
<P>%s ... <TT>%s</TT>\n\
<!-- %%X%08.08X \"%s\" -->\n",
   SoftwareID, cptr, SourceLineNumber, Message, Text, StatusValue, HiddenText);
}

/****************************************************************************/
/*
Does a case-insensitive, character-by-character string compare and returns 
true if two strings are the same, or false if not.  If a maximum number of 
characters are specified only those will be compared, if the entire strings 
should be compared then specify the number of characters as 0.
*/ 

boolean strsame
(
register char *sptr1,
register char *sptr2,
register int  count
)
{
   while (*sptr1 && *sptr2)
   {
      if (toupper (*sptr1++) != toupper (*sptr2++)) return (false);
      if (count)
         if (!--count) return (true);
   }
   if (*sptr1 || *sptr2)
      return (false);
   else
      return (true);
}

/*****************************************************************************/
/*
This function accepts a comma-separated list of file types (extensions, e.g.
"TXT,TEXT,COM,C,PAS,FOR") and a VMS file type (e.g. ".TXT;", ".TXT", "TXT").
It returns true if the file type is in the list, false if not.
*/

boolean SameFileType
(
char *TypeList,
char *FileType
)
{
   register char  *cptr, *sptr;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "SameFileType() |%s|%s|\n", TypeList, FileType);

   if (*FileType == '.') FileType++;
   cptr = TypeList;
   while (*cptr)
   {
      sptr = FileType;
      while (*sptr && *sptr != ';' && *cptr && *cptr != ',' &&
             toupper(*sptr) == toupper(*cptr))
      {
         cptr++;
         sptr++;
      }
      if ((!*sptr || *sptr == ';') && (!*cptr || *cptr == ',')) return (true);
      while (*cptr && *cptr != ',') cptr++;
      if (*cptr) cptr++;
   }
   return (false);
}

/*****************************************************************************/

