/*****************************************************************************/
/*
                              ExtractFile.c

For a plain-text file will extract the specified range of records (lines).


VERSION HISTORY
---------------
19-SEP-95  MGD  carraige-control changed, 'previous' and 'next' lines added
28-MAR-95  MGD  minor revision to support 'HttpOut' stream
05-DEC-94  MGD  major revision, extract functions placed in a CGI-style script
10-JUN-94  MGD  initial development
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <ctype.h>

/* VMS related header files */
#include <rmsdef.h>
#include <rms.h>
#include <ssdef.h>
#include <stsdef.h>

#define boolean int
#define true 1
#define false 0

#define VMSok(x) ((x) & STS$M_SUCCESS)
#define VMSnok(x) (!((x) & STS$M_SUCCESS))
 
#define TextFileRecordSize 1024 

/* external declarations */
extern boolean  Debug;
extern boolean  HttpHasBeenOutput;
extern char  CgiPathInfo[];
extern char  CgiScriptName[];
extern char  FormExact[];
extern char  Http200Header[];
extern char  SoftwareID[];
FILE  *HttpOut;

/*********************************/
/* functions in this source file */
/*********************************/

int ExtractTextFile (char*, char*, char*, int, int, boolean);
GenerateDocumentName (char*, char*, int);
char* SearchTextString (char*, char*, boolean);

/******************************************/
/* external functions used in this module */
/******************************************/

char* CopyTextIntoHtml (char*, char*, int);
char* CopyTextIntoUri (char*, char*, int);

/*****************************************************************************/
/*
Extract the specified record (line) range from a plain-text file.  If the 
search string is supplied then highlight the first matched string in any line.
*/ 

int ExtractFile
(
char *FileName,
char *UriFileName,
char *HighlightString,
int StartRecordNumber,
int EndRecordNumber,
boolean CaseSensitive
)
{
   register char  *rptr, *sptr;
   register int  RecordNumber = 0;

   boolean  OutputHttpHeader = true; 
   int  status,
        NumberOfRecords,
        HighlightStringLength = 0;
   char  DocumentName [256] = "",
         ExpandedFileName [256],
         Record [TextFileRecordSize+1],
         String [2048],
         UriHighlightString [256] = "";
   struct FAB  FileFab;
   struct RAB  FileRab;
   struct NAM  FileNam;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "ExtractFile() |%s|%s|\n", FileName, HighlightString);

   if (EndRecordNumber > StartRecordNumber)
      NumberOfRecords = EndRecordNumber - StartRecordNumber;
   else
      NumberOfRecords = 0;

   if (HighlightString[0])
   {
      HighlightStringLength = strlen(HighlightString);
      CopyTextIntoUri (UriHighlightString, HighlightString, -1);
   }

   FileFab = cc$rms_fab;
   FileFab.fab$b_fac = FAB$M_GET;
   FileFab.fab$l_fna = FileName;  
   FileFab.fab$b_fns = strlen(FileName);
   FileFab.fab$b_shr = FAB$M_SHRGET;
   FileFab.fab$l_nam = &FileNam;

   FileNam = cc$rms_nam;
   FileNam.nam$l_esa = ExpandedFileName;
   FileNam.nam$b_ess = sizeof(ExpandedFileName)-1;

   if (VMSnok (status = sys$open (&FileFab, 0, 0)))
   {
      /* if its a search list treat directory not found as if file not found */
      if ((FileNam.nam$l_fnb & NAM$M_SEARCH_LIST) && status == RMS$_DNF)
         status = RMS$_FNF;

      if (Debug) fprintf (stdout, "sys$open() %%X%08.08X\n", status);
      return (status);
   }

   FileRab = cc$rms_rab;
   FileRab.rab$l_fab = &FileFab;
   /* 2 buffers and read ahead performance option */
   FileRab.rab$b_mbf = 2;
   FileRab.rab$l_rop = RAB$M_RAH;
   FileRab.rab$l_ubf = Record;
   FileRab.rab$w_usz = sizeof(Record)-1;

   if (VMSnok (status = sys$connect (&FileRab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$connect() %%X%08.08X\n", status);
      sys$close (&FileFab, 0, 0);
      return (status);
   }

   /*******************/
   /* extract records */
   /*******************/

   while (VMSok (status = sys$get (&FileRab, 0, 0)))
   {
      RecordNumber++;
      Record[FileRab.rab$w_rsz] = '\0';
      if (Debug) fprintf (stdout, "Record |%d|%s|\n", RecordNumber, Record);

      /* terminate at any carriage control that may be in the record */
      for (rptr = Record; *rptr && *rptr != '\r' && *rptr != '\n'; rptr++);
      *rptr = '\0';

      /* if necessary generate document name */
      if (!DocumentName[0])
         GenerateDocumentName (Record, DocumentName, sizeof(DocumentName));

      if (StartRecordNumber && RecordNumber < StartRecordNumber) continue;

      if (OutputHttpHeader && !Record[0]) continue;

      /**********************/
      /* HTML document text */
      /**********************/

      if (OutputHttpHeader)
      {
         if (!DocumentName[0])
         {
            /* no document name could have been generated, use the file name */
            *FileNam.nam$l_ver = '\0';
            strcpy (DocumentName, FileNam.nam$l_name);
         }

         fprintf (HttpOut,
"%s\
<HTML>\n\
<!-- SoftwareID: %s -->\n\
<TITLE>%s</TITLE>\n",
         Http200Header, SoftwareID, DocumentName);

         if (StartRecordNumber)
         {
            if (EndRecordNumber && StartRecordNumber - NumberOfRecords > 0)
            {
               fprintf (HttpOut,
"<I>Extract begins at line %d; retrieve \
<A HREF=\"%s%s?highlight=%s&start=%d&end=%d\">previous %d lines</A> or \n\
<A HREF=\"%s%s?highlight=%s\">entire document</A>.</I>\n\
<HR>\n\
<PRE>",
               StartRecordNumber, 
               CgiScriptName, CgiPathInfo, UriHighlightString,
               StartRecordNumber-NumberOfRecords-1,
               StartRecordNumber-1, NumberOfRecords+1,
               CgiScriptName, CgiPathInfo, UriHighlightString);
            }
            else
            {
               fprintf (HttpOut,
"<I>Extract Begins at line %d; retrieve \
<A HREF=\"%s%s?highlight=%s\">entire document</A>.</I>\n\
<HR>\n\
<PRE>",
               RecordNumber,
               CgiScriptName, CgiPathInfo, UriHighlightString);
            }
         }
         else
         {
            if (EndRecordNumber)
            {
               fprintf (HttpOut,
"<I>Extract (and document) begins at line %d; retrieve \
<A HREF=\"%s%s?highlight=%s\">entire document</A>.</I>\n\
<HR>\n\
<PRE>",
               RecordNumber,
               CgiScriptName, CgiPathInfo, UriHighlightString);
            }
            else
               fputs ("<PRE>", HttpOut);
         }
         HttpHasBeenOutput = true;
         OutputHttpHeader = false;
      }

      /******************/
      /* process record */
      /******************/

      if (EndRecordNumber && RecordNumber > EndRecordNumber)
      {
         if (FormExact[0] == 'T' || FormExact[0] == 'Y') break;
         /* if this is an empty (blank) line then end-of-section */
         if (!FileRab.rab$w_rsz) break;
         /* if there is only white-space in this line then end-of-section */
         for (rptr = Record; *rptr && isspace(*rptr); rptr++);
         if (!*rptr) break;
      }

      if (HighlightString[0])
         rptr = SearchTextString (Record, HighlightString, CaseSensitive);
      else
         rptr = "";
      if (*rptr)
      {
         /***********************************/
         /* hit! - Highlight matched string */
         /***********************************/

         /* copy the record up to the first character of the search string */
         sptr = CopyTextIntoHtml (String, Record, rptr-Record);
         /* emphasize the matched search string */
         strcpy (sptr, "<I>"); sptr += 3;
         sptr = CopyTextIntoHtml (sptr, rptr, HighlightStringLength);
         strcpy (sptr, "</I>"); sptr += 4;
         /* rest of record after the matched search string */
         sptr = CopyTextIntoHtml (sptr, rptr+HighlightStringLength, -1);
         *sptr++ = '\n'; *sptr = '\0';
      }
      else
      {
         /**********/
         /* no hit */
         /**********/

         sptr = CopyTextIntoHtml (String, Record, -1);
         *sptr++ = '\n'; *sptr = '\0';
      }

      if (Debug) fprintf (stdout, "String |%s|\n", String);
      fputs (String, HttpOut);
   }

   /***************/
   /* end of file */
   /***************/

   if (status == RMS$_EOF) status = SS$_NORMAL;
   sys$close (&FileFab, 0, 0);

   /**********************/
   /* HTML document text */
   /**********************/

   if (StartRecordNumber)
   {
      if (EndRecordNumber && RecordNumber > EndRecordNumber)
      {
         fprintf (HttpOut,
"</PRE>\n\
<P><HR>\n\
<I>Extract ends at line %d; retrieve \
<A HREF=\"%s%s?Highlight=%s&start=%d&end=%d\">next %d lines</A> or \n\
<A HREF=\"%s%s?Highlight=%s&start=%d\">rest of document</A>.</I>\n\
</HTML>\n",
         RecordNumber-1, 
         CgiScriptName, CgiPathInfo, UriHighlightString,
         RecordNumber, RecordNumber+NumberOfRecords, NumberOfRecords+1,
         CgiScriptName, CgiPathInfo, UriHighlightString, RecordNumber);
      }
      else
      {
         fprintf (HttpOut,
"</PRE>\n\
<P><HR>\n\
<I>Extract (and document) ends at line %d; retrieve \
<A HREF=\"%s%s?Highlight=%s\">entire document</A>.</I>\n\
</HTML>\n",
         RecordNumber, CgiScriptName, CgiPathInfo, UriHighlightString);
      }
   }
   else
      fputs ("</PRE>\n</HTML>\n", HttpOut);

   return (status);
}

/*****************************************************************************/
/*
This function serves to generate a document name string (for use in the HTML
<TITLE> tag) for a plain-text file from the first line containing alpha-
numeric characters.  Copy the string pointed to by 'sptr' into the string 
pointed to by 'NamePtr', compressing white-space.
*/ 

GenerateDocumentName
(
register char *sptr,
register char *nptr,
int SizeOfDocumentName
)
{
   register char  *zptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "GenerateDocumentName() |%s|\n", sptr);

   zptr = nptr + SizeOfDocumentName - 1;

   /* skip leading white-space */
   while (*sptr && isspace(*sptr)) sptr++;
   while (*sptr)
   {
      /* copy an element */
      while (*sptr && !isspace(*sptr) && nptr < zptr) *nptr++ = *sptr++;
      if (!*sptr) break;
      if (nptr == zptr) break;
      /* skip intervening/trailing white-space */
      while (*sptr && isspace(*sptr)) sptr++;
      if (!*sptr) break;
      /* add a single space between elements */
      if (nptr < zptr) *nptr++ = ' ';
      if (nptr == zptr) break;
   }
   *nptr = '\0';
}

/*****************************************************************************/
/*
Case sensistive or insensitive search of the string pointed to by 'sptr' for 
the string pointed to by 'SearchString'.
*/ 

char* SearchTextString
( 
register char *tptr,
register char *SearchString,
register boolean CaseSensitive
)
{
   register char  *cptr, *sptr;

   /*********/
   /* begin */
   /*********/

   sptr = SearchString;
   if (CaseSensitive)
   {
      while (*tptr)
      {
         if (*tptr++ != *sptr) continue;
         /* first character of search string matched record character */
         sptr++;
         cptr = tptr;
         while (*cptr && *sptr)
         {
            if (*cptr != *sptr) break;
            cptr++;
            sptr++;
         }
         if (!*sptr)
         {
            tptr--;
            break;
         }
         sptr = SearchString;
      }
   }
   else
   {
      while (*tptr)
      {
         if (toupper(*tptr++) != toupper(*sptr)) continue;
         /* first character of search string matched record character */
         sptr++;
         cptr = tptr;
         while (*cptr && *sptr)
         {
            if (toupper(*cptr) != toupper(*sptr)) break;
            cptr++;
            sptr++;
         }
         if (!*sptr)
         {
            tptr--;
            break;
         }
         sptr = SearchString;
      }
   }
   return (tptr);
}

/*****************************************************************************/

