/*****************************************************************************
/*
                                   CLI.c

Module to process (interpret) the HTTPD command line.  Allows images activated
by a "foreign verb" to behave in a way that approximates the CLI$ (Command
Line Interpreter) utility calls.


VERSION HISTORY
---------------
01-DEC-95  MGD  HTTPd version 3
20-DEC-94  MGD  multi-threaded daemon
20-JUN-94  MGD  single-threaded daemon
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <ctype.h>

/* VMS header files */
#include <descrip.h>
#include <stsdef.h>
#include <syidef.h>
#include <ssdef.h>
#include <stsdef.h>

#include "httpd.h"

/**********/
/* macros */
/**********/

#define boolean int
#define true 1
#define false 0
 
#define VMSok(x) ((x) & STS$M_SUCCESS)
#define VMSnok(x) (!((x) & STS$M_SUCCESS))
 
/******************/
/* global storage */
/******************/

int  CliServerPort = 0;
boolean  CliLoggingDisabled = false,
         CliLoggingEnabled = false,
         CliMonitorDisabled = false,
         CliMonitorEnabled = false;

/********************/
/* external storage */
/********************/

extern boolean  Debug;
extern boolean  NoSwapOut;
extern int  DclSysOutputSize;
extern int  DclCgiVariablePrefixLength;
extern int  FileBufferSize;
extern int  MaxConcurrentConnections;
extern int  NetReadBufferSize;
extern int  OutputBufferSize;
extern int  ProcessPriority;
extern char  CommandLine[];
extern char  ControlBuffer[];
extern char  DclCgiVariablePrefix[];
extern char  LoggingFileName[];
extern char  SoftwareID[];
extern char  Utility[];
 
/****************************************************************************/
/*
This function allows images activated by a "foreign verb" to behave in a way 
that approximates the CLI$ (Command Line Interpreter) utility calls.  Get the 
entire command line following the verb that activated the image.  The command 
line is returned in uppercase, space compressed (i.e. maximum of one space 
between text elements, trimmed of leading and trailing spaces).  Returns a 
warning status if there were no parameters/qualifiers on the command line.
The variable CommandLine is global.
*/ 
 
int ParseCommandLine ()
 
{
   int  status;
   unsigned short  Length;
   unsigned long  Flags = 0;
   struct dsc$descriptor_s 
          CommandLineDsc = { 255, DSC$K_DTYPE_T,
                             DSC$K_CLASS_S, CommandLine };
 
   /* get the entire command line following the verb */
   if (VMSnok (status = lib$get_foreign (&CommandLineDsc, 0, &Length, &Flags)))
      return (status);
   CommandLine[Length] = '\0';
 
   if (ParseCommand (CommandLine))
      return (SS$_NORMAL);
   else
      return (STS$K_ERROR | STS$M_INHIB_MSG);
}
 
/****************************************************************************/
/*
This function allows images activated by a "foreign verb" to behave in a way 
that approximates the CLI$ (Command Line Interpreter) utility calls.  Quoted 
strings are always indicated by being parsed to include a single leading 
quote.
*/ 
 
boolean ParseCommand (char *CommandLine)
 
{
   register int  QuoteCount = 0;
   register char  *cptr, *eptr;
   boolean  CommandLineOK = true;
   char  Entity [256] = "";
 
   /* set up any argument defaults */
   ParseCommandEntity (NULL);
 
   cptr = CommandLine;
   eptr = Entity;
 
   for (;;)
   {
      if (*cptr == '\"')
      {
         QuoteCount++;
         *eptr++ = *cptr++;
         continue;
      }
 
      if (QuoteCount & 1 && *cptr)
      {
         /* inside quoted text, copy all characters as literals */
         *eptr++ = *cptr++;
         continue;
      }
 
      if (*cptr == '/' || isspace (*cptr) || !*cptr)
      {
         if (isspace (*cptr))
         {
            /* span the white space */
            while (*cptr && isspace (*cptr)) cptr++;
            if (*cptr == '=')
            {
               /* part of a qualifier, continue to get the value */
               *eptr++ = *cptr++;
               /* span any intervening white space */
               while (*cptr && isspace (*cptr)) cptr++;
               continue;
            }
         }
 
         if (Entity[0])
         {
            *eptr = '\0';
            if (!ParseCommandEntity (Entity)) CommandLineOK = false;
         }
 
         /* if end of command line then break from loop */
         if (!*cptr) break;
 
         /* start of new entity */
         eptr = Entity;
         /* if start of qualifier ensure slash is copied */
         if (*cptr == '/') *eptr++ = *cptr++;
 
         continue;
      }
 
      /* any other character, just copy, ensure upper case */
      *eptr++ = toupper(*cptr++);
   }
 
   return (CommandLineOK);
}
 
/*****************************************************************************/
/*
Get a string value from a qualifier, e.g. '/EXAMPLE=TEST'.
*/
 
boolean ParseCommandString
(
char *Entity,
char *String,
boolean Qualifier,
boolean ReportErrors,
boolean EnsureUpperCase
)
{
   register int  QuoteCount = 0;
   register char  *eptr, *sptr;
 
   if (Debug) fprintf (stdout, "ParseCommandString()\nEntity: '%s'\n", Entity);
 
   eptr = Entity;
 
   if (Qualifier)
   {
      /* scan down to equate symbol */
      while (*eptr && *eptr != '=') eptr++;
      if (*eptr) eptr++;
      if (!*eptr)
      {
         if (ReportErrors)
         {
            fprintf (stdout,
            "%%%s-E-VALREQ, missing qualifier or keyword value\n \\%s\\\n",
            Utility, Entity+1);
         }
         return (false);
      }
   }
 
   sptr = String;
   while (*eptr)
   {
      if (*eptr == '\"')
      {
         if (QuoteCount & 1)
         {
            /* are inside quotes, check for escaped quotes ("") */
            if (*++eptr != '\"')
            {
               /* now outside quotes */
               QuoteCount++;
            }
            /* drop thru to character copy */
         }
         else
         {
            /* now inside quotes */
            QuoteCount++;
            eptr++;
            continue;
         }
      }
 
      if (EnsureUpperCase)
         *sptr++ = toupper(*eptr++);
      else
         *sptr++ = *eptr++;
   }
   *sptr = '\0';
 
   if (Debug) fprintf (stdout, "String: '%s'\n", String);
 
   return (true);
}
 
/*****************************************************************************/
/*
Get an integer value from a qualifier, e.g. '/EXAMPLE=99'.
*/
 
boolean ParseCommandInteger
(
char *Entity,
int *IntegerPtr,
int Base,
boolean ReportErrors
)
{
   register char  *eptr;
   char  *sptr;
 
   if (Debug)
      fprintf (stdout, "ParseCommandInteger() '%s' Base: %d\n", Entity, Base);
 
   for (eptr = Entity; *eptr && *eptr != '='; eptr++);
   if (*eptr) eptr++;
   if (*eptr)
   {
      *IntegerPtr = strtol (eptr, &sptr, Base);
      if (sptr > eptr && !*sptr)
         return (true);
      else
      {
         if (ReportErrors)
         {
            fprintf (stdout,
            "%%%s-E-BADVALUE, '%s' is an invalid keyword value\n",
            Utility, eptr);
         }
         return (false);
      }
   }
   else
   {
      if (ReportErrors)
      {
         fprintf (stdout,
         "%%%s-E-VALREQ, missing qualifier or keyword value\n \\%s\\\n",
         Utility, Entity+1);
      }
      return (false);
   }
}
 
/*****************************************************************************/
/*
A single command line "entity" has been parsed, check if its recognised.  This 
function is the one modified for the individual requirements of each program.
*/
 
boolean ParseCommandEntity (char *Entity)
 
{
   if (Entity == NULL)
   {
      /* set up any argument defaults */
      return (true);
   }

   if (Debug) fprintf (stdout, "ParseCommandEntity() Entity: '%s'\n", Entity);
 
   if (Entity[0] == '/')
   {
      if (strsame (Entity, "/CGI_PREFIX=", 4))
      {
         boolean  ok;

         ok = ParseCommandString (Entity, DclCgiVariablePrefix,
                                  true, true, true);
         DclCgiVariablePrefixLength = strlen(DclCgiVariablePrefix);
         return (ok);
      }

      if (strsame (Entity, "/DO=", 4))
         return (ParseCommandString (Entity, ControlBuffer, true, true, true));

      /* turns on all "if (Debug)" statements */
      if (strsame (Entity, "/DBUG", -1))
         return (Debug = true);

      if (strsame (Entity, "/FILBUF=", 5))
         return (ParseCommandInteger (Entity, &FileBufferSize, 10, true));

      if (strsame (Entity, "/LOG=", 4))
      {
         CliLoggingEnabled = true;
         CliLoggingDisabled = false;
         if (Entity[4] == '=')
            return (ParseCommandString (Entity, LoggingFileName,
                                        true, true, true));
         else
            return (true);
      }
      if (strsame (Entity, "/NOLOG", 6))
      {
         CliLoggingEnabled = false;
         CliLoggingDisabled = true;
         return (true);
      }

      if (strsame (Entity, "/MONITOR", -1))
      {
         CliMonitorEnabled = true;
         CliMonitorDisabled = false;
         return (true);
      }
      if (strsame (Entity, "/NOMONITOR", -1))
      {
         CliMonitorEnabled = false;
         CliMonitorDisabled = true;
         return (true);
      }

      if (strsame (Entity, "/NETBUF=", 5))
         return (ParseCommandInteger (Entity, &NetReadBufferSize, 10, true));

      if (strsame (Entity, "/OUTBUF=", 5))
         return (ParseCommandInteger (Entity, &OutputBufferSize, 10, true));

      if (strsame (Entity, "/PORT=", 4))
         return (ParseCommandInteger (Entity, &CliServerPort, 10, true));

      if (strsame (Entity, "/PRIORITY=", 4))
         return (ParseCommandInteger (Entity, &ProcessPriority, 10, true));

      if (strsame (Entity, "/SUBBUF=", 5))
         return (ParseCommandInteger (Entity, &DclSysOutputSize, 10, true));

      if (strsame (Entity, "/SWAP", -1))
      {
         NoSwapOut = false;
         return (true);
      }
      if (strsame (Entity, "/NOSWAP", -1))
         return (NoSwapOut = true);

      fprintf (stdout,
      "%%%s-E-IVQUAL, unrecognised qualifier\n \\%s\\\n", Utility, Entity+1);
      return (false);
   }
 
   fprintf (stdout,
   "%%%s-E-MAXPARM, too many parameters\n \\%s\\\n", Utility, Entity);
   return (false);
}
   
/*****************************************************************************/

