/*****************************************************************************/
/*
                                Control.c

This module implements the HTTPd controlling functionality.  That is, the 
control of the HTTPd process from another, with functions such as reloading of 
configuration and elegant exiting (without disrupting client data transfers).

The module comprises two components:

  1.  The functions used by the serving HTTPd when being controlled
  2.  The function used by another process to control the serving HTTPd

Current control commands:

  o  ABORT      exit immediately
  o  AUTH       series of authorization functions
  o  EXIT       exit after all client activity complete
  o  LOG        series of logging functions
  o  MAP        reload mapping information
  o  RESTART    restart the image, effectively exit and re-execute
  o  ZERO       reset the server counters

These commands are entered at the DCL command line (interpreted in the
CommandLine.c module) in the following manner:

  $ HTTPD /DO=command

For example:

  $ HTTPD /DO=ABORT
  $ HTTPD /DO=AUTH=ALL
  $ HTTPD /DO=AUTH=FAIL
  $ HTTPD /DO=AUTH=RESET=realm:username
  $ HTTPD /DO=EXIT
  $ HTTPD /DO=MAP
  $ HTTPD /DO=RESTART
  $ HTTPD /DO=ZERO


VERSION HISTORY
---------------
01-DEC-95  MGD  HTTPd version 3
12-APR-95  MGD  support logging
03-APR-95  MGD  support authorization
20-DEC-94  MGD  initial development for multi-threaded version of HTTPd
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <ctype.h>

/* VMS related header files */
#include <descrip.h>
#include <iodef.h>
#include <prvdef.h>
#include <psldef.h>
#include <ssdef.h>
#include <stsdef.h>

/* application-related header files */
#include "httpd.h"

/******************/
/* global storage */
/******************/

/* 5 seconds */
$DESCRIPTOR (ControlCommandTimeoutDsc, "0 00:00:05.00");

boolean  ControlExitRequested,
         ControlRestartRequested;

unsigned short  ControlMbxChannel;

char  ControlBuffer [256],
      ControlMbxName [32];
      
struct AnIOsb  ControlMbxReadIOsb,
               ControlMbxWriteIOsb;

$DESCRIPTOR (ControlMbxNameDsc, ControlMbxName);
$DESCRIPTOR (ControlMbxNameFaoDsc, "HTTPD!UL$CONTROL");

/********************/
/* external storage */
/********************/

extern boolean  Debug;
extern boolean  MonitorEnabled;
extern int  CurrentConnectCount;
extern int  ExitStatus;
extern int  ServerPort;
extern char  SoftwareID[];
extern char  Utility[];
extern struct AccountingStruct Accounting;

/****************************/
/* functions in this module */
/****************************/

int ControlCommand (char*);
ControlCommandTimeoutAST ();
int ControlHttpd ();
ControlMbxReadAST ();
ControlMbxOutput (char*, int);
ControlMbxWrite (char*, int);
ControlMbxWriteAST ();

/*************************************/
/* prototypes for external functions */
/*************************************/

int AuthAllList ();
int AuthFailList ();
int AuthFailResetCount (char*, char*);
int DefineMonitorLogicals (struct RequestStruct*);
HttpdExit ();
int Logging (struct RequestStruct*, int);
char* MapUrl (char*, char*, char*, char*);
char* SysGetMsg (int);

/*****************************************************************************/
/*
This, and the other supporting functions, are used by the serving HTTPd 
process.  Create a system-permanent mailbox for the receipt of control 
messages sent by another process.
*/ 

int ControlHttpd ()

{
   /* no world or group access, full owner and system access */
#  define ControlMbxProtectionMask 0xff00

   /*
       PRMMBX to allow a permanent mailbox to be created.
       SYSNAM to create its logical name in the system table.
       SYSPRV just in case it was originally created by a privileged,
              non-HTTPd account (as has happened to me when developing!)
   */
   static unsigned long  CreMbxPrivMask [2] =
          { PRV$M_PRMMBX | PRV$M_SYSNAM | PRV$M_SYSPRV, 0 };

   int  status,
        SetPrvStatus;
   unsigned short  Length;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ControlHttpd()\n");

   if (VMSnok (status =
       sys$fao (&ControlMbxNameFaoDsc, &Length, &ControlMbxNameDsc,
                ServerPort)))
   {
      if (Debug) fprintf (stdout, "sys$fao() %%X%08.08X\n", status);
      return (status);
   }
   ControlMbxName[ControlMbxNameDsc.dsc$w_length = Length] = '\0';
   if (Debug) fprintf (stdout, "ControlMbxName |%s|\n", ControlMbxName);

   /* turn on privileges to allow access to the permanent mailbox */
   if (VMSnok (status = sys$setprv (1, &CreMbxPrivMask, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$setprv() %%X%08.08X\n", status);
      return (status);
   }

   status = sys$crembx (1,
                        &ControlMbxChannel,
                        sizeof(ControlBuffer),
                        sizeof(ControlBuffer),
                        ControlMbxProtectionMask,
                        PSL$C_USER,
                        &ControlMbxNameDsc,
                        0);
   if (Debug) fprintf (stdout, "sys$crembx() %%X%08.08X\n", status);

   if (VMSok (status))
   {
     /* pro-actively mark the mailbox for deletion */
      status = sys$delmbx (ControlMbxChannel);
      if (Debug) fprintf (stdout, "sys$delmbx() %%X%08.08X\n", status);
   }

   if (VMSnok (SetPrvStatus = sys$setprv (0, &CreMbxPrivMask, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$setprv() %%X%08.08X\n", SetPrvStatus);
      /* do NOT keep processing if there is a problem turning them off! */
      ErrorExitVmsStatus (SetPrvStatus, "disabling SYSNAM and SYSPRV",
                          __FILE__, __LINE__);
   }

   if (VMSnok (status)) return (status);

   /* queue up the first asynchronous read from the control mailbox */
   return (sys$qio (0, ControlMbxChannel, IO$_READVBLK, &ControlMbxReadIOsb,
                    &ControlMbxReadAST, 0,
                    ControlBuffer, sizeof(ControlBuffer)-1, 0, 0, 0, 0));
}

/*****************************************************************************/
/*
An asynchronous read from the system-permanent control mailbox has completed.  
Act on the command string read into the buffer.
*/ 

ControlMbxReadAST ()

{
   register char  *cptr, *sptr;
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout,
               "ControlMbxReadAST() IO %d Status %%X%08.08X\n",
               ControlMbxReadIOsb.Count, ControlMbxReadIOsb.Status);

   if (VMSnok (ControlMbxReadIOsb.Status) &&
       ControlMbxReadIOsb.Status != SS$_ABORT &&
       ControlMbxReadIOsb.Status != SS$_CANCEL)
   {
      fprintf (stdout, "%%%s-E-CONTROLMBX, read failed\n-%s\n",
               Utility, SysGetMsg(ControlMbxReadIOsb.Status)+1);
      exit (ControlMbxReadIOsb.Status | STS$M_INHIB_MSG);
   }

   ControlBuffer[ControlMbxReadIOsb.Count] = '\0';
   if (Debug) fprintf (stdout, "ControlBuffer |%s|\n", ControlBuffer);

   if (strsame (ControlBuffer, "MAP", -1))
   {
      /*********************************/
      /* re-read the mapping rule file */
      /*********************************/

      /* ensure the mapping rules are re-read with the next mapping */
      MapUrl (NULL, NULL, NULL, NULL);
      ControlMbxWrite ("!new map loaded", 15);
      return;
   }
   else
   if (strsame (ControlBuffer, "ZERO", -1))
   {
      /*****************************/
      /* reset the server counters */
      /*****************************/

      memset ((char*)&Accounting + sizeof(Accounting.ZeroedCount),
              0,
              sizeof(Accounting) - sizeof(Accounting.ZeroedCount));
      Accounting.ZeroedCount++;
      if (MonitorEnabled) DefineMonitorLogicals (NULL);
      ControlMbxWrite ("!zeroed", 7);
      return;
   }
   else
   if (strsame (ControlBuffer, "RESTART", -1))
   {
      /******************/
      /* server restart */
      /******************/

      /* relies on the supporting DCL procedure looping immediately */
      if (CurrentConnectCount)
      {
         ControlRestartRequested = true;
         ControlMbxWrite ("!restarting when all clients disconnected", 41);
         return;
      }
      else
      {
         fprintf (stdout, "%%%s-I-CONTROL, server restart\n", Utility);
         ControlMbxWrite ("!restarting now", 15);
         sleep (1);
         exit (SS$_NORMAL);
      }
   }
   else
   if (strsame (ControlBuffer, "EXIT", -1))
   {
      /*******************/
      /* server shutdown */
      /*******************/

      if (CurrentConnectCount)
      {
         ControlExitRequested = true;
         ControlMbxWrite ("!exiting when all clients disconnected", 38);
         return;
      }
      else
      {
         fprintf (stdout, "%%%s-I-CONTROL, server exit\n", Utility);
         ControlMbxWrite ("!exiting now", 12);
         ExitStatus = SS$_NORMAL;
         HttpdExit ();
         sys$delprc (0, 0);
      }
   }
   else
   if (strsame (ControlBuffer, "ABORT", -1))
   {
      /*********************************/
      /* unconditional server shutdown */
      /*********************************/

      fprintf (stdout, "%%%s-I-CONTROL, server abort\n", Utility);
      ControlMbxWrite ("!exiting now", 12);
      ExitStatus = SS$_NORMAL;
      HttpdExit ();
      sys$delprc (0, 0);
   }
   else
   if (strsame (ControlBuffer, "AUTHENTICATION", 4))
   {
      /******************/
      /* authentication */
      /******************/

      for (cptr = ControlBuffer; *cptr && *cptr != '='; cptr++);
      if (*cptr)
      {
         cptr++;
         if (strsame (cptr, "ALL", 3))
         {
            if (AuthAllList ())
               ControlMbxWrite ("!ok", 3);
            else
               ControlMbxWrite ("!no authentication records", 33);
            return;
         }
         else
         if (strsame (cptr, "FAIL", 4))
         {
            if (AuthFailList ())
               ControlMbxWrite ("!ok", 3);
            else
               ControlMbxWrite ("!no authentication fail records", 33);
            return;
         }
         else
         if (strsame (cptr, "RESET", 3))
         {
            while (*cptr && *cptr != '=') cptr++;
            if (*cptr)
            {
               cptr++;
               sptr = cptr;
               while (*cptr && *cptr != ':') cptr++;
               if (*cptr)
               {
                  *cptr++ = '\0';
                  if (AuthFailResetCount (sptr, cptr))
                     ControlMbxWrite ("!ok", 3);
                  else
                     ControlMbxWrite ("!record not found", 17);
                  return;
               }
            }
         }
         ControlMbxWrite ("!syntax error", 13);
         return;
      }                      
   }
   else
   if (strsame (ControlBuffer, "LOG", 3))
   {
      /***********/
      /* logging */
      /***********/

      for (cptr = ControlBuffer; *cptr && *cptr != '='; cptr++);
      if (*cptr)
      {
         cptr++;
         if (strsame (cptr, "OPEN", 4))
         {
            if (VMSok (Logging (NULL, LOGGING_OPEN)))
               ControlMbxWrite ("!log opened", 11);
            else
               ControlMbxWrite ("!log not opened", 15);
            return;
         }
         if (strsame (cptr, "CLOSE", 4))
         {
            if (VMSok (Logging (NULL, LOGGING_CLOSE)))
               ControlMbxWrite ("!log closed", 11);
            else
               ControlMbxWrite ("!log not closed", 15);
            return;
         }
         if (strsame (cptr, "FLUSH", 5))
         {
            if (VMSok (Logging (NULL, LOGGING_FLUSH)))
               ControlMbxWrite ("!log flushed", 12);
            else
               ControlMbxWrite ("!log not flushed", 16);
            return;
         }
         ControlMbxWrite ("!syntax error", 13);
         return;
      }
   }

   /*********/
   /* what? */
   /*********/

   /* all handled situations must return/exit before here */
   ControlMbxWrite ("!what?", 6);
}

/*****************************************************************************/
/*
This is the function used by the controlling process.  Synchronously write the 
command to the HTTPd control mailbox, then synchronously read the response.  
Set a timer to limit the wait for the command to be read, and then response to 
be written, by the HTTPd process.
*/ 

int ControlCommand (char *Command)

{
   int  status;
   unsigned short  Length;
   unsigned long  CommandTimeout[2];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ControlCommand() |%s|\n", Command);

   if (VMSnok (status =
       sys$fao (&ControlMbxNameFaoDsc, &Length, &ControlMbxNameDsc,
                ServerPort)))
   {
      if (Debug) fprintf (stdout, "sys$fao() %%X%08.08X\n", status);
      return (status);
   }
   ControlMbxName[ControlMbxNameDsc.dsc$w_length = Length] = '\0';
   if (Debug) fprintf (stdout, "ControlMbxName |%s|\n", ControlMbxName);

   if (VMSnok (status =
       sys$assign (&ControlMbxNameDsc, &ControlMbxChannel, 0, 0, 0)))
   {
      if (status == SS$_NOSUCHDEV)
      {
         fprintf (stdout,
         "%%%s-E-NOSUCHDEV, server (control mailbox) does not exist\n",
         Utility);
         return (status | STS$M_INHIB_MSG);
      }
      return (status);
   }

   /* initialize a delta-time seconds structure used in sys$setimr() */
   if (VMSnok (status =
       sys$bintim (&ControlCommandTimeoutDsc, &CommandTimeout)))
   {
      if (Debug) fprintf (stdout, "sys$bintim() %%X%08.08X\n", status);
      exit (status);
   }

   status = sys$setimr (0, &CommandTimeout, &ControlCommandTimeoutAST, 0, 0);
   if (Debug) fprintf (stdout, "sys$setimr() %%X%08.08X\n", status);

   /* synchronously write the command to the HTTPd process */
   status = sys$qiow (0, ControlMbxChannel, IO$_WRITEVBLK,
                      &ControlMbxReadIOsb, 0, 0,
                      Command, strlen(Command), 0, 0, 0, 0);
   if (Debug)
      fprintf (stdout, "sys$qiow() %%X%08.08X IOsb.Status: %%X%08.08X\n",
               status, ControlMbxReadIOsb.Status);
   if (VMSok (status) && VMSnok (ControlMbxReadIOsb.Status))
      status = ControlMbxReadIOsb.Status;
   if (status == SS$_ABORT)
   {
      /* timer has expired */
      fprintf (stdout,
      "%%%s-E-TIMEOUT, server failed to respond within 5 seconds\n",
      Utility);
      return (SS$_TIMEOUT | STS$M_INHIB_MSG);
   }
   if (VMSnok (status)) return (status);

   for (;;)
   {
      /* read the acknowledgement from the HTTPd process */
      status = sys$qiow (0, ControlMbxChannel, IO$_READVBLK,
                         &ControlMbxReadIOsb, 0, 0,
                         ControlBuffer, sizeof(ControlBuffer)-1, 0, 0, 0, 0);
      if (Debug)
         fprintf (stdout, "sys$qiow() %%X%08.08X IOsb.Status: %%X%08.08X\n",
                  status, ControlMbxReadIOsb.Status);
      if (VMSok (status) && VMSnok (ControlMbxReadIOsb.Status))
         status = ControlMbxReadIOsb.Status;
      if (VMSnok (status)) break;
      ControlBuffer[ControlMbxReadIOsb.Count] = '\0';
      if (ControlBuffer[0] == '!') break;
      fprintf (stdout, "%s\n", ControlBuffer);
   }

   if (status == SS$_ABORT)
   {
      /* timer has expired */
      fprintf (stdout,
      "%%%s-E-TIMEOUT, server failed to respond within 5 seconds\n",
      Utility);
      return (SS$_TIMEOUT | STS$M_INHIB_MSG);
   }
   if (VMSnok (status)) return (status);

   /* cancel the timer */
   sys$cantim (0, 0);

   /* present the response to the user */
   fprintf (stdout, "%%%s-I-RESPONSE, %s\n", Utility, ControlBuffer+1);
   return (status);
}

/*****************************************************************************/
/*
Queue an I/O to the mailbox without AST.  This allows multiple output I/Os
without queuing another read from the mailbox (which the AST routine does).
*/

ControlMbxOutput
(
char *String,
int Length
)
{
   int  status;

   if (Debug) fprintf (stdout, "ControlMbxOutput() %d\n", Length);
   status = sys$qiow (0, ControlMbxChannel, IO$_WRITEVBLK,
                      &ControlMbxReadIOsb, 0, 0,
                      String, Length, 0, 0, 0, 0);
   if (VMSok (status) && VMSnok (ControlMbxReadIOsb.Status))
      status = ControlMbxReadIOsb.Status;
   if (Debug) fprintf (stdout, "sys$qio() %%X%08.08X\n", status);
}

/*****************************************************************************/
/*
Queue an I/O to the mailbox with AST.  When the I/O completes the AST routine
queues another read from the control mailbox, setting the situation for another
control command to be sent to the HTTPd.  This function should always be called
as the final I/O of control output.
*/

ControlMbxWrite
(
char *String,
int Length
)
{
   int  status;

   if (Debug) fprintf (stdout, "ControlMbxWrite() %d\n", Length);
   status = sys$qio (0, ControlMbxChannel, IO$_WRITEVBLK,
                     &ControlMbxWriteIOsb, &ControlMbxWriteAST, 0,
                     String, Length, 0, 0, 0, 0);
   if (Debug) fprintf (stdout, "sys$qio() %%X%08.08X\n", status);
}

/*****************************************************************************/
/*
This function is called after the HTTPd process has asynchronously written a 
response back to the controlling process, and that I/O has been read.  It 
merely queues another asynchronous read from the control mailbox.
*/ 

ControlMbxWriteAST ()

{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout,
               "ControlMbxWriteAST() IO %d Status %%X%08.08X\n",
               ControlMbxWriteIOsb.Count, ControlMbxWriteIOsb.Status);

   /* queue up another read from the control mailbox */
   sys$qio (0, ControlMbxChannel, IO$_READVBLK, &ControlMbxReadIOsb,
            &ControlMbxReadAST, 0,
            ControlBuffer, sizeof(ControlBuffer)-1, 0, 0, 0, 0);
}

/*****************************************************************************/
/*
When the timer expires this function is called as an AST routine.  It merely 
cancels the outstanding I/O on the control mailbox channel.  This is detected 
by an I/O return code of SS$_CANCEL.
*/

ControlCommandTimeoutAST ()

{
   if (Debug) fprintf (stdout, "ControlCommandTimeoutAST()\n");

   sys$cancel (ControlMbxChannel);
}

/*****************************************************************************/

