/*****************************************************************************
/*
                                  Dcl.c

Provides multi-threaded, concurrent HTTPd subprocesses executing DCL.

The DCL can either be in the form of a command, or a procedure or executable 
image file specification.  Both should not be supplied, but file 
specifications have precedence.  If a file specfication is supplied the module 
verifies its existance, and if not qualified with an extension, looks for a 
procedure first (".COM"), then an executable image (".EXE").  Furthermore, the 
DCL can be executed either standalone or as a CGI script (indicated by the 
presence of a script name in the request data structure).  If a CGI script, 
then the output stream is parsed for header information, if not a script then 
the stream is just checked for correct carriage control (terminated by a 
newline).  CGI variables are created for both standalone DCL and scripts, 
although some (e.g. WWW_SCRIPT_NAME) will be empty, and meaningless, for 
standalone DCL. 

The AST-driven nature makes the code a little more difficult to follow, but 
creates a powerful, event-driven, multi-threaded server.  All of the 
necessary functions implementing this module are designed to be non-blocking. 

All of these functions are designed to be, and should be, called at AST 
delivery level, which means they cannot be interrupted at the same level (in 
this case USER mode), and so their actions are essentially atomic at that 
level, meaning no additional synchronization is required for such activities 
as thread disposal, etc.

HTTPD can maintain a number of subprocesses limited only by its process 
quotas, memory is dynamically allocated and there are no fixed data structures 
related to subprocess management. 

HTTPD maintains a network connection with the client for each of the 
subprocesses, receiving data from the network socket and if a script passing 
it to the subprocess via a mailbox, and accepting data from the subprocess via 
another mailbox, then sending this to the client via its network socket.  
Because HTTP interactions are byte-stream only this works without additional 
IP socket operations. 

The use of byte-streams (effectively "pipes") allows even DCL procedures to
output as HTTP servers, without the need for explicit network I/O. 

Two, or three if a script, mailboxes are created for each subprocess:

  1.  A mailbox connected to its SYS$INPUT.  This mailbox is used to
      pass DCL commands to the subprocess.  It effectively allows HTTPD 
      to control the activities of the subprocess this way.

  2.  A mailbox connected to its SYS$OUTPUT.  This recieves records from
      the subprocess, if required appends the HTTP-required carriage-control
      (<CR><LF>), then sends the record to the client via the network socket.
      This allows even DCL procedures to supply a correct output stream to
      the client (see next paragraph).

      If the first line from a script is an HTTP status line (e.g.
      "HTTP/1.0 200 ok") then HTTPD assumes the script will be supplying
      a complete HTTP data stream, including full header and required
      carriage control (<CR><LF> terminating each line).  If the first
      line is not a HTTP status line it assumes CGI script output compliance
      and also ensures each record (line) received has correct HTTP
      carriage-control.

      This stream also attempts to maintain compliance with CGI scripting.
      If the first line output by a script is not an HTTP status line it
      creates and sends one to the client before sending the first line.

  3.  Only if the subprocess is executing as a CGI script.
      A mailbox defined for the subprocess by the name HTTP$INPUT.
      This may be used to read the data steam sent by the client.

      A program executing in the subprocess may easily use this stream 
      by doing a "DEFINE /USER_MODE SYS$INPUT HTTP$INPUT" before
      activating the image, or by explicitly opening a stream to that
      logical name.

The script subprocesses can use the basic CGI variables (VMS CERN-like) and 
behave very much like a CGI script.

That is, if a script wants to be CGI-compliant it provides as the first line a 
"Status:", "Content-type:" or a "Location:" then a blank line.   If the first 
line output by a script is a "Content-Type:" header line an HTTP "200" status 
line is prepended.  If the first line output by a script is a "Location:" 
redirection header line the redirection is processed to ensure CERN HTTPD/CGI 
behaviour.  An HTTP "302" status line is prepended if not a local redirection. 
If none of these, HTTPD creates a complete HTTP header comprising status line, 
"Content-Type: text/plain" and blank line (this is an extension of CERN HTTPD 
behaviour). 

If the first characters are "HTTP/1.0 ..." the script will be considered to be 
supplying the raw HTTP stream and record boundaries, carriage-control, etc., 
are of no further concern to the module.  This is the equivalent of a
"no-parse-header" script.  If CGI-compliant each record should represent a 
line of output.   That is lines should not be buffered together and sent as a 
block unless the script is supplying a raw HTTP data stream. 


CGI VARIABLES
-------------
These CGI variable names are those supported by the INTERNET-DRAFT authored by 
D.Robinson (drtr@ast.cam.ac.uk), 8 January 1996, plus some "convenience" 
variables, breaking the query string into its components (KEY_, FORM_, etc.)  
The CGI symbols (CERN/VMS-HTTPd-like DCL symbols instead of Unix environment 
variables) are created by the SYS$INPUT stream of each subprocess before the 
script DCL procedure is invoked.  By default each variable name is prefixed by 
"WWW_" (similar to CERN HTTPd), although this can be modified at the command 
line when starting the server.  CGI variable (symbol) values are limited in 
size to approximately 1000 characters. 

  o  WWW_AUTH_TYPE ...........  authentication type (or empty)

  o  WWW_CONTENT_LENGTH ....... "Content-Length:" header header
  o  WWW_CONTENT_TYPE ......... "Content-Type:" from header

  o  WWW_GATEWAY_INTERFACE .... "CGI/n.n"

  o  WWW_HTTP_IF_NOT_MODIFIED . GMT time string
  o  WWW_HTTP_REFERER ......... source document URL for this request
  o  WWW_HTTP_USER_AGENT ...... client/browser identification string

  o  WWW_PATH_INFO ............ virtual path of data requested in URL
  o  WWW_PATH_TRANSLATED ...... VMS file path of data requested in URL

  o  WWW_QUERY_STRING ......... string following "?" in URL

  o  WWW_REMOTE_ADDR .......... IP host address of HTTP client
  o  WWW_REMOTE_HOST .......... IP host name of HTTP client
  o  WWW_REMOTE_IDENT ......... RFC931 remote user (always empty)
  o  WWW_REMOTE_USER .......... authenticated remote user name (or empty)
  o  WWW_REQUEST_METHOD ....... "GET", "PUT", etc.

  o  WWW_SCRIPT_NAME .......... name of script being executed (e.g. "/query")
  o  WWW_SERVER_NAME ... ...... IP host name of server system
  o  WWW_SERVER_PROTOCOL ...... HTTP protocol version
  o  WWW_SERVER_PORT .......... IP port request was received on
  o  WWW_SERVER_SOFTWARE ...... software ID of the HTTPD daemon

  o  WWW_KEY_n ................ query string "+" separated elements
  o  WWW_KEY_COUNT ............ number of "+" separated elements
  o  WWW_FORM_field ........... query string "&" separated form elements


VERSION HISTORY
---------------
01-DEC-95  MGD  HTTPd version 3
19-SEP-95  MGD  changed carriage-control on records from <CR><LF> (the strict
                HTTP requirement) to single newline (<LF>, de facto standard)
                This will be slightly more efficient, and "more compliant"!
21-APR-95  MGD  bugfix to DclSysOutputAST()
03-APR-95  MGD  added remote user authentication and CGI symbol
20-MAR-95  MGD  bugfix to DclQioHttpInput()
20-DEC-94  MGD  initial development as a module for multi-threaded daemon
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <ctype.h>

/* VMS related header files */

/* cmbdef.h is not defined for VAXC 3.n */
#define CMB$M_READONLY 0x01
#define CMB$M_WRITEONLY 0x02

#include <descrip.h>
#include <dvidef.h>

#include <iodef.h>
/* these should be, but are not defined for VAXC 3.n */
#define IO$M_READERCHECK 0x100
#define IO$M_WRITERCHECK 0x200
#define IO$M_READERWAIT 0x400
#define IO$M_WRITERWAIT 0x800

#include <jpidef.h>
#include <prvdef.h>
#include <rms.h>

#include <ssdef.h>
/* these should be, but are not defined for VAXC 3.n */
#define SS$_NOREADER 9412
#define SS$_NOWRITER 9420

#include <stsdef.h>
#include <syidef.h>

/* application header file */
#include "httpd.h"

/**********************/
/* module definitions */
/**********************/

/* no world or group access */
#define SubprocessMbxProtectionMask 0xff00

/******************/
/* global storage */
/******************/

#define DefaultCgiVariablePrefix "WWW_"
char DclCgiVariablePrefix [32] = DefaultCgiVariablePrefix;
int DclCgiVariablePrefixLength = sizeof(DefaultCgiVariablePrefix)-1;

/********************/
/* external storage */
/********************/

extern boolean  Debug;
extern int  DclSysOutputSize;
extern int  NetReadBufferSize;
extern int  ServerPort;
extern char  SoftwareID[];
extern char  ServerHostName[];
extern struct AccountingStruct Accounting;

/****************************/
/* functions in this module */
/****************************/

DclAbortSubprocess (struct RequestStruct*);
DclDispose (struct RequestStruct*);
DclEnd (struct RequestStruct*);
DclBegin (struct RequestStruct*, char*, char*); 
DclFindProcedure (char*, int);
DclHttpInputAST (struct RequestStruct*);
DclQioSysInput (struct RequestStruct*, char*, int);
DclQioSysOutput (struct RequestStruct*);
DclQioSysOutputToClientAST (struct RequestStruct*);
DclSysInputCommands (struct RequestStruct*, char*, char*, char*, boolean);
DclSysInputAST (struct RequestStruct*);
DclSubprocessCompletionAST (struct RequestStruct*);
DclSysOutputHeader (struct RequestStruct*);
DclSysOutputLocation (struct RequestStruct*);
DclSysOutputAST (struct RequestStruct*);
DclSysOutputWaitAST (struct RequestStruct*);

/*********************/
/* external functions */
/**********************/

BufferOutput (struct RequestStruct*, void*, char*, int);
ConcludeProcessing (struct RequestStruct*);
ErrorGeneral (struct RequestStruct*, char*, char*, int);
ErrorVmsStatus (struct RequestStruct*, int, char*, int);
unsigned char*  HeapAlloc (struct RequestStruct*, int);
int IfModifiedSince (struct RequestStruct*, unsigned long*, unsigned long*);
QioNetWrite (struct RequestStruct*, void*, char*, int);

/*****************************************************************************/
/*
Create a subprocess to execute the DCL.

'RequestPtr->ScriptName' empty, or contains name if CGI script execution.
'RequestPtr->SysInputDevName' empty, or contains device name.
'RequestPtr->SysInputDevNameDsc' empty, or contains device name descriptor.
'RequestPtr->SysOutputDevName' empty, or contains device name.
'RequestPtr->SysOutputDevNameDsc' empty, or contains device name descriptor.
'RequestPtr->HttpInputDevName' empty, or contains device name.
'RequestPtr->HttpInputDevNameDsc' empty, or contains device name descriptor.
*/ 

DclBegin
(
struct RequestStruct *RequestPtr,
char *DclCommand,
char *DclFileName
)
{
   static unsigned long  DevNamItem = DVI$_DEVNAM,
                         /* nowait,noclisym,nolognam,nokey */
                         SpawnFlags = 0x0f;
   static int  DclSubprocessNumber;
   static char  SubprocessName [16];
   static $DESCRIPTOR (SubprocessNameFaoDsc, "!15<HTTPd:!UL_!UL!>");
   static $DESCRIPTOR (SubprocessNameDsc, SubprocessName);

   boolean  DclFileNameIsProcedure;
   int  status,
        Count;
   unsigned short  Length;

   /*********/
   /* begin */
   /*********/

   if (DclCommand == NULL) DclCommand = "";
   if (DclFileName == NULL) DclFileName = "";
   RequestPtr->DclHasBeenExecuting = true;

   if (Debug)
   {
      fprintf (stdout,
"DclBegin()\n\
DclCommand |%s|\n\
DclFileName |%s|\n\
ScriptName |%s|\n\
PathInfoPtr |%s|\n\
QueryStringPtr |%s|\n",
      DclCommand, DclFileName,
      RequestPtr->ScriptName, RequestPtr->PathInfoPtr,
      RequestPtr->QueryStringPtr);
   }

   /* a little accounting before we start! */
   Accounting.DclExecutedCount++;
   if (RequestPtr->DclSysInputChannel)
      Accounting.DclExecutedMoreThanOnceCount++;
   if (RequestPtr->ScriptName[0])
      Accounting.DoScriptCount++;

   if (RequestPtr->HttpAuthorizationPtr != NULL &&
       !RequestPtr->RemoteUser[0])
   {
      /********************************/
      /* authenticate client username */
      /********************************/

      /* authorization supplied a remote username, check its ok */
      strcpy (RequestPtr->AuthRealm, "local");
      if (VMSnok (Authenticate (RequestPtr)))
      {
         ConcludeProcessing (RequestPtr);
         return;
      }
      /*
         The AuthVerifyRemoteUser() function has nulled the remote username
         if it did not verify, otherwise left it like intact.  If the script
         requires authorization it now checks the appropriate CGI symbol,
         if there is a string in it then it is an authorized username.
      */
   }

   if (DclFileName[0])
   {
      /************************************/
      /* check DCL procedure/image exists */
      /************************************/

      if (VMSnok (status =
          DclFindScript (DclFileName, &DclFileNameIsProcedure)))
      {
         if (status == RMS$_FNF && RequestPtr->ScriptName[0])
         {
            RequestPtr->ResponseStatusCode = 404;
            ErrorGeneral (RequestPtr, "Script not found.", __FILE__, __LINE__);
         }
         else
         {
            RequestPtr->ErrorTextPtr = "finding file";
            RequestPtr->ErrorHiddenTextPtr = DclFileName;
            ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
         }

         ConcludeProcessing (RequestPtr);
         return;
      }
   }

   /* allocate heap memory for SYS$OUTPUT buffer */
   if (RequestPtr->DclSysOutputPtr == NULL)
   {
      /* allow two bytes for carriage control and terminating null */
      if ((RequestPtr->DclSysOutputPtr =
          HeapAlloc (RequestPtr, DclSysOutputSize+2)) == NULL)
      {
         ErrorHeapAlloc (RequestPtr, __FILE__, __LINE__);
         DclEnd (RequestPtr);
         return (SS$_NORMAL);
      }
      RequestPtr->DclSysOutputSize = DclSysOutputSize;
   }

   if (RequestPtr->ScriptName[0])
   {
      /****************************/
      /* script is being executed */
      /****************************/

      if (!RequestPtr->DclHttpInputChannel)
      {
         /*********************************/
         /* create the HTTP$INPUT mailbox */
         /*********************************/

         if (VMSnok (status =
             sys$crembx (0,
                         &RequestPtr->DclHttpInputChannel,
                         NetReadBufferSize, NetReadBufferSize,
                         SubprocessMbxProtectionMask,
                         0, 0, CMB$M_WRITEONLY)))
         {
            if (Debug) fprintf (stdout, "sys$crembx() %%X%08.08X\n", status);
            ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
            DclAbortSubprocess (RequestPtr);
            return;
         }

         RequestPtr->HttpInputDevNameDsc.dsc$w_length =
            sizeof(RequestPtr->HttpInputDevName);
         RequestPtr->HttpInputDevNameDsc.dsc$a_pointer =
            RequestPtr->HttpInputDevName;
         RequestPtr->HttpInputDevNameDsc.dsc$b_class = DSC$K_CLASS_S;
         RequestPtr->HttpInputDevNameDsc.dsc$b_dtype = DSC$K_DTYPE_T;

         if (VMSnok (status =
             lib$getdvi (&DevNamItem, &RequestPtr->DclHttpInputChannel,
                         0, 0, &RequestPtr->HttpInputDevNameDsc, &Length)))
         {
            ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
            DclAbortSubprocess (RequestPtr);
            return;
         }
         RequestPtr->HttpInputDevName[
            RequestPtr->HttpInputDevNameDsc.dsc$w_length =
            Length] = '\0';
         if (Debug)
            fprintf (stdout, "HttpInputDevName |%s|\n",
                     RequestPtr->HttpInputDevName);
      }

      /****************************************/
      /* provide request header to subprocess */
      /****************************************/

      /*
         Be sure to supply the actual bytes received, not header length,
         this may be a "post" and additional data may have been received.
         If the subprocess reads the initial record the transfer from the
         client to the subprocess becomes I/O AST driven.
      */

      if (Debug) 
         fprintf (stdout, "|%d|%s|\n",
                  RequestPtr->BytesRx,
                  RequestPtr->NetReadBufferPtr);

      if (VMSnok (status =
          sys$qio (0, RequestPtr->DclHttpInputChannel,
                   IO$_WRITELBLK, &RequestPtr->DclHttpInputIOsb,
                   &DclHttpInputAST, RequestPtr,
                   RequestPtr->NetReadBufferPtr, RequestPtr->BytesRx,
                   0, 0, 0, 0)))
      {
         RequestPtr->ErrorTextPtr = "providing HTTP header to script";
         ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
         DclAbortSubprocess (RequestPtr);
         return;
      }
      if (Debug) fprintf (stdout, "sys$qio() %%X%08.08X\n", status);
      RequestPtr->DclOutstandingIO++;

      RequestPtr->DclCheckSubprocessResponseHeader = true;
   }
   else
   {
      /********************************/
      /* script is NOT being executed */
      /********************************/

      RequestPtr->DclCheckSubprocessResponseHeader = false;
      RequestPtr->DclAddNewline = true;
   }

   if (!RequestPtr->DclSysInputChannel)
   {
      /********************************/
      /* create the SYS$INPUT mailbox */
      /********************************/

      if (VMSnok (status =
          sys$crembx (0,
                      &RequestPtr->DclSysInputChannel,
                      DclSysOutputSize, DclSysOutputSize,
                      SubprocessMbxProtectionMask,
                      0, 0, CMB$M_WRITEONLY)))
      {
         if (Debug) fprintf (stdout, "sys$crembx() %%X%08.08X\n", status);
         ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
         DclAbortSubprocess (RequestPtr);
         return;
      }

      RequestPtr->SysInputDevNameDsc.dsc$w_length =
         sizeof(RequestPtr->SysInputDevName);
      RequestPtr->SysInputDevNameDsc.dsc$a_pointer =
         RequestPtr->SysInputDevName;
      RequestPtr->SysInputDevNameDsc.dsc$b_class = DSC$K_CLASS_S;
      RequestPtr->SysInputDevNameDsc.dsc$b_dtype = DSC$K_DTYPE_T;

      if (VMSnok (status =
          lib$getdvi (&DevNamItem, &RequestPtr->DclSysInputChannel,
                      0, 0, &RequestPtr->SysInputDevNameDsc, &Length)))
      {
         ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
         DclAbortSubprocess (RequestPtr);
         return;
      }
      RequestPtr->SysInputDevName[
         RequestPtr->SysInputDevNameDsc.dsc$w_length =
         Length] = '\0';
      if (Debug)
         fprintf (stdout, "SysInputDevName |%s|\n",
                  RequestPtr->SysInputDevName);
   }

   if (!RequestPtr->DclSysOutputChannel)
   {
      /*********************************/
      /* create the SYS$OUTPUT mailbox */
      /*********************************/

      if (VMSnok (status =
          sys$crembx (0,
                      &RequestPtr->DclSysOutputChannel,
                      DclSysOutputSize, DclSysOutputSize,
                      SubprocessMbxProtectionMask,
                      0, 0, CMB$M_READONLY)))
      {
         if (Debug) fprintf (stdout, "sys$crembx() %%X%08.08X\n", status);
         ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
         DclAbortSubprocess (RequestPtr);
         return;
      }

      RequestPtr->SysOutputDevNameDsc.dsc$w_length =
         sizeof(RequestPtr->SysOutputDevName);
      RequestPtr->SysOutputDevNameDsc.dsc$a_pointer =
         RequestPtr->SysOutputDevName;
      RequestPtr->SysOutputDevNameDsc.dsc$b_class = DSC$K_CLASS_S;
      RequestPtr->SysOutputDevNameDsc.dsc$b_dtype = DSC$K_DTYPE_T;

      if (VMSnok (status =
          lib$getdvi (&DevNamItem, &RequestPtr->DclSysOutputChannel,
                      0, 0, &RequestPtr->SysOutputDevNameDsc, &Length)))
      {
         ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
         DclAbortSubprocess (RequestPtr);
         return;
      }
      RequestPtr->SysOutputDevName[
         RequestPtr->SysOutputDevNameDsc.dsc$w_length =
         Length] = '\0';
      if (Debug)
         fprintf (stdout, "SysOutputDevName |%s|\n",
                  RequestPtr->SysOutputDevName);
   }

   /*********************************************/
   /* provide commands to subprocess' SYS$INPUT */
   /*********************************************/

   RequestPtr->ErrorTextPtr = "when creating CGI symbols";
   if (VMSnok (status =
       DclSysInputCommands (RequestPtr, RequestPtr->HttpInputDevName,
                            DclCommand, DclFileName, DclFileNameIsProcedure)))
   {
      /* some error messages are not generated inside DclSysInputCommands() */
      if (RequestPtr->ErrorMessagePtr == NULL)
         ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
      DclAbortSubprocess (RequestPtr);
      return;
   }
   RequestPtr->ErrorTextPtr = NULL;

   /****************************************/
   /* queue wait-for subprocess SYS$OUTPUT */
   /****************************************/
 
   /*
      Queue up an asynchronous wait for a writer channel to be assigned
      to the mailbox as the subprocess' SYS$OUTPUT.  When the channel is
      assigned the AST function will queue the first read from the mailbox.
   */
   if (VMSnok (status =
       sys$qio (0, RequestPtr->DclSysOutputChannel,
                IO$_SETMODE | IO$M_WRITERWAIT,
                &RequestPtr->DclSysOutputIOsb,
                &DclSysOutputWaitAST, RequestPtr,
                0, 0, 0, 0, 0, 0)))
   {
      ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
      DclAbortSubprocess (RequestPtr);
      return;
   }
   if (Debug) fprintf (stdout, "sys$qio() READERWAIT %%X%08.08X\n", status);
   RequestPtr->DclOutstandingIO++;

   /************************/
   /* spawn the subprocess */
   /************************/

   for (Count = 100; Count; Count--)
   {
      /* subprocesses are consecutively numbered from 1..999 */
      SubprocessNameDsc.dsc$w_length = sizeof(SubprocessName);
      sys$fao (&SubprocessNameFaoDsc, &Length, &SubprocessNameDsc, 
               ServerPort, DclSubprocessNumber++ % 1000 + 1);
      SubprocessName[SubprocessNameDsc.dsc$w_length = Length] = '\0';
      if (Debug) fprintf (stdout, "SubprocessName |%s|\n", SubprocessName);

      status = lib$spawn (0,
                          &RequestPtr->SysInputDevNameDsc,
                          &RequestPtr->SysOutputDevNameDsc,
                          &SpawnFlags,
                          &SubprocessNameDsc,
                          &RequestPtr->DclSubprocessPid,
                          &RequestPtr->DclSubprocessCompletionStatus,
                          0,
                          &DclSubprocessCompletionAST, RequestPtr,
                          0, 0, 0);

      if (Debug)
         fprintf (stdout, "lib$spawn() %%X%08.08X PID: %08.08X\n",
                  status, RequestPtr->DclSubprocessPid);

      if (status != SS$_DUPLNAM) break;
   }

   if (VMSnok (status))
   {
      RequestPtr->ErrorTextPtr = "creating subprocess";
      ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
      DclAbortSubprocess (RequestPtr);
      return;
   }

   /*
      Return to the calling routine now.  Subsequent processing is
      event-driven.  Routine completion ASTs drives the script output.
   */
}

/*****************************************************************************/
/*
*/

DclEnd (struct RequestStruct *RequestPtr)

{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DclEnd()\n");

   ConcludeProcessing (RequestPtr);
}

/*****************************************************************************/
/*
Check if the script procedure or executable file exists.
*/ 

int DclFindScript
(
char *FileName,
boolean *FileNameIsProcedurePtr
)
{
   int  status;
   char  ExpandedFileName [256];
   struct FAB  SearchFab;
   struct NAM  SearchNam;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DclFindScript() |%s|\n", FileName);

   SearchFab = cc$rms_fab;
   SearchFab.fab$l_fna = FileName;
   SearchFab.fab$b_fns = strlen(FileName);
   SearchFab.fab$l_fop = FAB$M_NAM;
   SearchFab.fab$l_nam = &SearchNam;
   SearchNam = cc$rms_nam;
   SearchNam.nam$l_esa = ExpandedFileName;
   SearchNam.nam$b_ess = sizeof(ExpandedFileName)-1;

   /* first, default to a DCL procedure */
   SearchFab.fab$l_dna = ".COM;";
   SearchFab.fab$b_dns = 5;

   if (VMSnok (status = sys$parse (&SearchFab, 0, 0)))
      return (status);

   if ((status = sys$search (&SearchFab, 0, 0)) == RMS$_FNF)
   {
      /* next, default to an executable */
      SearchFab.fab$l_dna = ".EXE;";
      SearchFab.fab$b_dns = 5;

      if (VMSok (status = sys$parse (&SearchFab, 0, 0)))
         status = sys$search (&SearchFab, 0, 0);
   }

   if (VMSok (status))
   {
      /* numeric equivalent of "COM;" */
      if (*(unsigned long*)(SearchNam.nam$l_type+1) == 0x3b4d4f43)
         *FileNameIsProcedurePtr = true;
      else
         *FileNameIsProcedurePtr = false;
   }

   /* release parse and search internal data structures */
   SearchNam.nam$b_nop = NAM$M_SYNCHK;
   sys$parse (&SearchFab, 0, 0);

   return (status);
}

/*****************************************************************************/
/*
Send DCL commands to the subprocess' SYS$INPUT.  This sets up the DCL 
environment (defines logical names, assigns symbols) then executes the 
specified DCL command, procedure or image.

Some CGI variables have a potential to be larger than DCL symbols can be 
created on the DCL command line (which is limited to 255 characters).  For 
example, PATH_INFO, QUERY_STRING, etc.  These variables are handled by a 
function that allows variables (DCL symbols) to have up to 1000 characters via 
a symbol assignment kludge.
*/ 

DclSysInputCommands
(
struct RequestStruct *RequestPtr,
char *HttpInputDevName,
char *DclCommand,
char *DclFileName,
boolean DclFileNameIsProcedure
)
{
   static $DESCRIPTOR (DefineHttpInputFaoDsc,
                       "DEFINE /EXECUTIVE HTTP$INPUT !AZ");

   static $DESCRIPTOR (AuthTypeFaoDsc, "!AZAUTH_TYPE==\"!AZ\"");

   static $DESCRIPTOR (ContentLengthFaoDsc, "!AZCONTENT_LENGTH==\"!AZ\"");
   static $DESCRIPTOR (ContentTypeFaoDsc, "!AZCONTENT_TYPE==\"!AZ\"");

   static $DESCRIPTOR (GatewayFaoDsc, "!AZGATEWAY_INTERFACE==\"CGI/1.1\"");

   static $DESCRIPTOR (IfModifiedSinceFaoDsc,
                       "!AZHTTP_IF_MODIFIED_SINCE==\"!AZ\"");
   /* HTTP_REFERER is created using DclSysInputStringSymbol() */
   static $DESCRIPTOR (UserAgentFaoDsc, "!AZHTTP_USER_AGENT==\"!AZ\"");

   static $DESCRIPTOR (RequestMethodFaoDsc, "!AZREQUEST_METHOD==\"!AZ\"");

   /* PATH_INFO is created using DclSysInputStringSymbol() */
   static $DESCRIPTOR (PathTranslatedFaoDsc, "!AZPATH_TRANSLATED==\"!AZ\"");

   /* QUERY_STRING is created using DclSysInputStringSymbol() */

   static $DESCRIPTOR (RemoteAddressFaoDsc, "!AZREMOTE_ADDR==\"!AZ\"");
   static $DESCRIPTOR (RemoteHostNameFaoDsc, "!AZREMOTE_HOST==\"!AZ\"");
   /* REMOTE_IDENT is always empty */
   static $DESCRIPTOR (RemoteIdentFaoDsc, "!AZREMOTE_IDENT==\"\"");
   static $DESCRIPTOR (RemoteUserFaoDsc, "!AZREMOTE_USER==\"!AZ\"");

   static $DESCRIPTOR (ScriptNameFaoDsc, "!AZSCRIPT_NAME==\"!AZ\"");
   static $DESCRIPTOR (ServerNameFaoDsc, "!AZSERVER_NAME==\"!AZ\"");
   static $DESCRIPTOR (ServerSoftwareFaoDsc, "!AZSERVER_SOFTWARE==\"!AZ\"");
   static $DESCRIPTOR (ServerProtocolFaoDsc, "!AZSERVER_PROTOCOL==\"!AZ\"");
   static $DESCRIPTOR (ServerPortFaoDsc, "!AZSERVER_PORT==\"!UL\"");

   /* extensions to INTERNET-DRAFT CGI variables */
   static $DESCRIPTOR (EncodingFaoDsc, "!AZENCODING==\"!AZ\"");
   static $DESCRIPTOR (KeyWordFieldNameFaoDsc, "KEY_!UL");
   static $DESCRIPTOR (KeyWordCountFaoDsc, "!AZKEY_COUNT==\"!UL\"");

   register char  c;
   register char  *cptr, *sptr, *zptr;

   int  status,
        Count;
   unsigned short  Length;
   char  FormFieldName [256],
         KeyWordFieldName [16],
         Scratch [1024],
         DclLine [256];
   $DESCRIPTOR (KeyWordFieldNameDsc, KeyWordFieldName);
   $DESCRIPTOR (DclLineDsc, DclLine);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DclSysInputCommands()\n");

   /*
      If the logical name HTTPD$VERIFY is defined to DCL true then
      DCL verify will be turned on.  If NOT defined, or defined to
      DCL false then DCL verify will be turned off.
   */
   if (VMSnok (status = DclQioSysInput (RequestPtr,
                        "!'F$VERIFY(F$TRNLNM(\"HTTPD$VERIFY\"))", 36)))
      return (status);

   if (VMSnok (status = DclQioSysInput (RequestPtr, "SET NOON", 8)))
      return (status);

   if (RequestPtr->ScriptName[0])
   {
      sys$fao (&DefineHttpInputFaoDsc, &Length, &DclLineDsc, HttpInputDevName);
      DclLine[Length] = '\0';
      if (VMSnok (status = DclQioSysInput (RequestPtr, DclLine, Length)))
         return (status);
   }

   /***************************/ 
   /* CGI compliant variables */
   /***************************/ 

   sys$fao (&AuthTypeFaoDsc, &Length, &DclLineDsc,
            DclCgiVariablePrefix, RequestPtr->AuthType);
   DclLine[Length] = '\0';
   if (VMSnok (status = DclQioSysInput (RequestPtr, DclLine, Length)))
      return (status);

   if (RequestPtr->PostContentTypePtr != NULL)
      sys$fao (&ContentTypeFaoDsc, &Length, &DclLineDsc,
               DclCgiVariablePrefix, RequestPtr->PostContentTypePtr);
   else
   if (RequestPtr->ContentTypePtr != NULL)
      sys$fao (&ContentTypeFaoDsc, &Length, &DclLineDsc,
               DclCgiVariablePrefix, RequestPtr->ContentTypePtr);
   else
      sys$fao (&ContentTypeFaoDsc, &Length, &DclLineDsc,
               DclCgiVariablePrefix, "");
   DclLine[Length] = '\0';
   if (VMSnok (status = DclQioSysInput (RequestPtr, DclLine, Length)))
      return (status);

   sys$fao (&ContentLengthFaoDsc, &Length, &DclLineDsc,
            DclCgiVariablePrefix, RequestPtr->ContentLength);
   DclLine[Length] = '\0';
   if (VMSnok (status = DclQioSysInput (RequestPtr, DclLine, Length)))
      return (status);

   sys$fao (&GatewayFaoDsc, &Length, &DclLineDsc, DclCgiVariablePrefix);
   DclLine[Length] = '\0';
   if (VMSnok (status = DclQioSysInput (RequestPtr, DclLine, Length)))
      return (status);

   if (RequestPtr->HttpIfModifiedSincePtr != NULL)
   {
      sys$fao (&IfModifiedSinceFaoDsc, &Length, &DclLineDsc,
               DclCgiVariablePrefix, RequestPtr->HttpIfModifiedSincePtr);
      DclLine[Length] = '\0';
      if (VMSnok (status = DclQioSysInput (RequestPtr, DclLine, Length)))
         return (status);
   }

   if (RequestPtr->HttpRefererPtr != NULL)
   {
      if (VMSnok (status =
          DclSysInputStringSymbol (RequestPtr, "HTTP_REFERER",
                                   RequestPtr->HttpRefererPtr)))
      {
         ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
         return (status);
      }
   }

   if (RequestPtr->HttpUserAgentPtr != NULL)
   {
      sys$fao (&UserAgentFaoDsc, &Length, &DclLineDsc,
               DclCgiVariablePrefix, RequestPtr->HttpUserAgentPtr);
      DclLine[Length] = '\0';
      if (VMSnok (status = DclQioSysInput (RequestPtr, DclLine, Length)))
         return (status);
   }

   sys$fao (&RequestMethodFaoDsc, &Length, &DclLineDsc,
            DclCgiVariablePrefix, RequestPtr->Method);
   DclLine[Length] = '\0';
   if (VMSnok (status = DclQioSysInput (RequestPtr, DclLine, Length)))
      return (status);

   if (VMSnok (status =
       DclSysInputStringSymbol (RequestPtr, "PATH_INFO",
                                RequestPtr->PathInfoPtr)))
   {
      ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
      return (status);
   }

   sys$fao (&PathTranslatedFaoDsc, &Length, &DclLineDsc,
            DclCgiVariablePrefix, RequestPtr->FileName);
   DclLine[Length] = '\0';
   if (VMSnok (status = DclQioSysInput (RequestPtr, DclLine, Length)))
      return (status);

   if (VMSnok (status =
       DclSysInputStringSymbol (RequestPtr, "QUERY_STRING",
                                RequestPtr->QueryStringPtr)))
   {
      ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
      return (status);
   }

   sys$fao (&RemoteHostNameFaoDsc, &Length, &DclLineDsc,
            DclCgiVariablePrefix, RequestPtr->ClientHostName);
   DclLine[Length] = '\0';
   if (VMSnok (status = DclQioSysInput (RequestPtr, DclLine, Length)))
      return (status);

   sys$fao (&RemoteAddressFaoDsc, &Length, &DclLineDsc,
            DclCgiVariablePrefix, RequestPtr->ClientInternetAddress);
   DclLine[Length] = '\0';
   if (VMSnok (status = DclQioSysInput (RequestPtr, DclLine, Length)))
      return (status);

   sys$fao (&RemoteIdentFaoDsc, &Length, &DclLineDsc,
            DclCgiVariablePrefix);
   DclLine[Length] = '\0';
   if (VMSnok (status = DclQioSysInput (RequestPtr, DclLine, Length)))
      return (status);

   sys$fao (&RemoteUserFaoDsc, &Length, &DclLineDsc,
            DclCgiVariablePrefix, RequestPtr->RemoteUser);
   DclLine[Length] = '\0';
   if (VMSnok (status = DclQioSysInput (RequestPtr, DclLine, Length)))
      return (status);

   sys$fao (&ScriptNameFaoDsc, &Length, &DclLineDsc,
            DclCgiVariablePrefix, RequestPtr->ScriptName);
   DclLine[Length] = '\0';
   if (VMSnok (status = DclQioSysInput (RequestPtr, DclLine, Length)))
      return (status);

   sys$fao (&ServerNameFaoDsc, &Length, &DclLineDsc,
            DclCgiVariablePrefix, ServerHostName);
   DclLine[Length] = '\0';
   if (VMSnok (status = DclQioSysInput (RequestPtr, DclLine, Length)))
      return (status);

   sys$fao (&ServerProtocolFaoDsc, &Length, &DclLineDsc,
            DclCgiVariablePrefix, HttpProtocol);
   DclLine[Length] = '\0';
   if (VMSnok (status = DclQioSysInput (RequestPtr, DclLine, Length)))
      return (status);

   sys$fao (&ServerSoftwareFaoDsc, &Length, &DclLineDsc,
            DclCgiVariablePrefix, SoftwareID);
   DclLine[Length] = '\0';
   if (VMSnok (status = DclQioSysInput (RequestPtr, DclLine, Length)))
      return (status);

   sys$fao (&ServerPortFaoDsc, &Length, &DclLineDsc,
            DclCgiVariablePrefix, ServerPort);
   DclLine[Length] = '\0';
   if (VMSnok (status = DclQioSysInput (RequestPtr, DclLine, Length)))
      return (status);

   if (RequestPtr->BinaryEncoding)
      sys$fao (&EncodingFaoDsc, &Length, &DclLineDsc,
               DclCgiVariablePrefix, "BINARY");
   else
      sys$fao (&EncodingFaoDsc, &Length, &DclLineDsc,
               DclCgiVariablePrefix, "TEXT");
   DclLine[Length] = '\0';
   if (VMSnok (status = DclQioSysInput (RequestPtr, DclLine, Length)))
      return (status);

   /***************************/
   /* query string components */
   /***************************/

   if (RequestPtr->QueryStringPtr[0])
   {
      Count = 0;

      cptr = RequestPtr->QueryStringPtr;
      while (*cptr && *cptr != '=') cptr++;
      /* if an equal symbol was found then its a form not a keyword search */
      if (*cptr)
      {
         /***************/
         /* form fields */
         /***************/

         memcpy (FormFieldName, "FORM_", 5);
         cptr = RequestPtr->QueryStringPtr;
         while (*cptr)
         {
            sptr = FormFieldName + 5;
            zptr = FormFieldName + sizeof(FormFieldName);
            while (*cptr && *cptr != '=' && *cptr != '&' && sptr < zptr)
            {
               if (isalnum(*cptr))
                  *sptr++ = toupper(*cptr++);
               else
               {
                  *sptr++ = '_';
                  cptr++;
               }
            }
            if (sptr >= zptr)
            {
                ErrorVmsStatus (RequestPtr, SS$_BUFFEROVF, __FILE__, __LINE__);
                return (status);
            }
            *sptr = '\0';

            if (!FormFieldName[0] || *cptr != '=')
            {
               /* error; back-to-back '&' and/or '=', or no equate */
               if (*cptr) cptr++;
               continue;
            }

            /* must have encountered an '=' */
            if (*cptr) cptr++;
            zptr = (sptr = Scratch) + sizeof(Scratch);
            while (*cptr && *cptr != '&' && sptr < zptr)
            {
               if (*cptr == '+')
               {
                  *sptr++ = ' ';
                  cptr++;
               }
               else
               if (*cptr == '%')
               {
                  /* an escaped character ("%xx" where xx is a hex number) */
                  cptr++;
                  c = 0;
                  if (*cptr >= '0' && *cptr <= '9')
                     { c = (*cptr - (int)'0') << 4; cptr++; }
                  else
                  if (toupper(*cptr) >= 'A' && toupper(*cptr) <= 'F')
                     { c = (toupper(*cptr) - (int)'A' + 10) << 4; cptr++; }
                  if (*cptr >= '0' && *cptr <= '9')
                     { c += (*cptr - (int)'0'); cptr++; }
                  else
                  if (toupper(*cptr) >= 'A' && toupper(*cptr) <= 'F')
                     { c += (toupper(*cptr) - (int)'A' + 10); cptr++; }
                  if (c == '\"' && sptr < zptr)
                  {
                     /* ensure enough quotes are passed for a DCL procedure */
                     *sptr++ = '\"';
                  }
                  if (sptr < zptr) *sptr++ = c;
               }
               else
                  *sptr++ = *cptr++;
            }
            if (sptr >= zptr)
            {
                ErrorVmsStatus (RequestPtr, SS$_BUFFEROVF, __FILE__, __LINE__);
                return (status);
            }
            *sptr = '\0';
            if (*cptr) cptr++;

            if (VMSnok (status =
                DclSysInputStringSymbol (RequestPtr, FormFieldName, Scratch)))
            {
                ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
                return (status);
            }
         }

         sys$fao (&KeyWordCountFaoDsc, &Length, &DclLineDsc,
                  DclCgiVariablePrefix, 0);
         DclLine[Length] = '\0';
         if (VMSnok (status = DclQioSysInput (RequestPtr, DclLine, Length)))
            return (status);
      }
      else
      {
         /******************/
         /* query keywords */
         /******************/

         memcpy (FormFieldName, "KEY_", 4);
         cptr = RequestPtr->QueryStringPtr;
         while (*cptr)
         {
            zptr = (sptr = Scratch) + sizeof(Scratch);
            while (*cptr && *cptr != '+' && sptr < zptr)
            {
               if (*cptr == '%')
               {
                  /* an escaped character ("%xx" where xx is a hex number) */
                  cptr++;
                  c = 0;
                  if (*cptr >= '0' && *cptr <= '9')
                     { c = (*cptr - (int)'0') << 4; cptr++; }
                  else
                  if (toupper(*cptr) >= 'A' && toupper(*cptr) <= 'F')
                     { c = (toupper(*cptr) - (int)'A' + 10) << 4; cptr++; }
                  if (*cptr >= '0' && *cptr <= '9')
                     { c += (*cptr - (int)'0'); cptr++; }
                  else
                  if (toupper(*cptr) >= 'A' && toupper(*cptr) <= 'F')
                     { c += (toupper(*cptr) - (int)'A' + 10); cptr++; }
                  if (c == '\"' && sptr < zptr)
                  {
                     /* ensure enough quotes are passed for a DCL procedure */
                     *sptr++ = '\"';
                  }
                  if (sptr < zptr) *sptr++ = c;
               }
               else
                  *sptr++ = *cptr++;
            }
            if (sptr >= zptr)
            {
                ErrorVmsStatus (RequestPtr, SS$_BUFFEROVF, __FILE__, __LINE__);
                return (status);
            }
            *sptr = '\0';
            if (*cptr) cptr++;

            /* just use 'DclLine' to contain the name of the symbol here! */
            sys$fao (&KeyWordFieldNameFaoDsc, &Length, &KeyWordFieldNameDsc,
                     ++Count);
            KeyWordFieldName[Length] = '\0';
            if (VMSnok (status =
                DclSysInputStringSymbol (RequestPtr, KeyWordFieldName,
                                         Scratch)))
            {
                ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
                return (status);
            }
         }

         sys$fao (&KeyWordCountFaoDsc, &Length, &DclLineDsc,
                  DclCgiVariablePrefix, Count);
         DclLine[Length] = '\0';
         if (VMSnok (status = DclQioSysInput (RequestPtr, DclLine, Length)))
            return (status);
      }
   }
   else
   {
      sys$fao (&KeyWordCountFaoDsc, &Length, &DclLineDsc,
               DclCgiVariablePrefix, 0);
      DclLine[Length] = '\0';
      if (VMSnok (status = DclQioSysInput (RequestPtr, DclLine, Length)))
         return (status);
   }

   /*******************************/
   /* DCL command/procedure/image */
   /*******************************/

   if (DclFileName[0])
   {
      sptr = DclLine;
      if (DclFileNameIsProcedure)
         *sptr++ = '@';
      else
      {
         cptr = "RUN ";
         while (*cptr) *sptr++ = *cptr++;
      }
      cptr = DclFileName;
      while (*cptr) *sptr++ = *cptr++;
      *sptr = '\0';
      Length = sptr - (cptr = DclLine);
   }
   else
   {
      for (cptr = sptr = DclCommand; *sptr; sptr++);
      Length = sptr - cptr;
   }
   if (VMSnok (status = DclQioSysInput (RequestPtr, cptr, Length)))
      return (status);

   /*********************************/
   /* ensure subprocess terminates! */
   /*********************************/

   return (DclQioSysInput (RequestPtr, "STOP/ID=0", 9));
}

/*****************************************************************************/
/*
DCL symbol creation at the command line is limited by the CLI command line 
length (255 characters).  Symbol values however can be up to approximately 
1000 characters, probably enough for any CGI variable value.  If a CGI value 
is too large for for a single command-line assignment then build it up using 
multiple assignments, a symbol assignment kludge!
*/ 

DclSysInputStringSymbol
(
struct RequestStruct *RequestPtr,
char *SymbolName,
char *SymbolValue
)
{
#  define MaxDclLineSize 255

   static char  DclLine [MaxDclLineSize+1];
   static $DESCRIPTOR (DclLineDsc, DclLine);
   static $DESCRIPTOR (AssignFaoDsc, "!AZ!AZ==X");
   static $DESCRIPTOR (SymbolFaoDsc, "!AZ!AZ==\"!AZ\"");

   int  status;
   unsigned short  Length;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DclSysInputStringSymbol() |%s|\n", SymbolName);

   status = sys$fao (&SymbolFaoDsc, &Length, &DclLineDsc,
                     DclCgiVariablePrefix, SymbolName, SymbolValue);
   if (VMSok (status) && status != SS$_BUFFEROVF)
   {
      DclLine[Length] = '\0';
      if (VMSnok (status = DclQioSysInput (RequestPtr, DclLine, Length)))
         return (status);
   }
   if (status != SS$_BUFFEROVF) return (status);

   /* buffer overflowed!  create it using multiple symbol assignments */
   {
      /* register storage */
      register char  *cptr, *sptr, *zptr;

      /* create, by assigning the temporary symbol with an empty string */
      if (VMSnok (status = DclQioSysInput (RequestPtr, "X=\"\"", 4)))
         return (status);

      memcpy (DclLine, "X=X+\"", 5);
      /* minus two, one each for the closing quote and the terminating null */
      zptr = DclLine + sizeof(DclLine) - 2;
      cptr = SymbolValue;
      while (*cptr)
      {
         sptr = DclLine + 5;
         while (*cptr && sptr < zptr) *sptr++ = *cptr++;
         *sptr++ = '\"';
         *sptr = '\0';
         if (VMSnok (status =
             DclQioSysInput (RequestPtr, DclLine, sptr-DclLine)))
            return (status);
      }

      /* assign the temporary symbol value to the CGI symbol */
      if (VMSnok (status =
          sys$fao (&AssignFaoDsc, &Length, &DclLineDsc,
                   DclCgiVariablePrefix, SymbolName)))
         return (status);
      DclLine[Length] = '\0';
      if (VMSnok (status = DclQioSysInput (RequestPtr, DclLine, Length)))
         return (status);

      /* not really necessary, but let's be tidy */
      if (VMSnok (status = DclQioSysInput (RequestPtr, "DELETE/SYMBOL X", 15)))
         return (status);
   }
}

/*****************************************************************************/
/*
This AST is called when the subprocesses exits.  Cancel any subprocess input 
IO, but importantly, let output from it complete normally before deleting the 
thread.
*/

DclSubprocessCompletionAST (struct RequestStruct *RequestPtr)

{
   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout,
"DclSubprocessCompletionAST() \
PID: %08.08X DclSubprocessCompletionStatus %%X%08.08X\n",
      RequestPtr->DclSubprocessPid,
      RequestPtr->DclSubprocessCompletionStatus);

   /* set the PID to zero to indicate the subprocess no longer exists */
   RequestPtr->DclSubprocessPid = 0;

   /* cancel IO on subprocess input mailbox channels */
   if (RequestPtr->DclSysInputChannel)
      sys$cancel (RequestPtr->DclSysInputChannel);
   if (RequestPtr->DclHttpInputChannel)
      sys$cancel (RequestPtr->DclHttpInputChannel);

   /*
      If the subprocess has gone, no outstanding IO, and SYS$OUTPUT is
      known to be empty (SS$_NOWRITER status), then dispose of the thread.
   */
   if (!(RequestPtr->DclOutstandingIO ||
         RequestPtr->DclOutstandingClientRead))
      DclEnd (RequestPtr);
}

/*****************************************************************************/
/*
This function can be called at any stage to abort the execution of a script 
subprocess.  It unconditionally cancels all IO associated with the subprocess, 
deletes it if it still exists and can delete the thread, or allow deletion 
after the cancellation of all outstanding IO.
*/

DclAbortSubprocess (struct RequestStruct *RequestPtr)

{
   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "DclAbortSubprocess() PID: %08.08X\n",
               RequestPtr->DclSubprocessPid);

   /* cancel IO on all channels */
   if (RequestPtr->ClientChannel)
      sys$cancel (RequestPtr->ClientChannel);
   if (RequestPtr->DclSysInputChannel)
      sys$cancel (RequestPtr->DclSysInputChannel);
   if (RequestPtr->DclSysOutputChannel)
      sys$cancel (RequestPtr->DclSysOutputChannel);
   if (RequestPtr->DclHttpInputChannel)
      sys$cancel (RequestPtr->DclHttpInputChannel);

   /* if the subprocess still exists then delete it */
   if (RequestPtr->DclSubprocessPid)
      sys$delprc (&RequestPtr->DclSubprocessPid, 0);

   /*
      If the subprocess has gone, no outstanding IO, and SYS$OUTPUT is
      known to be empty (SS$_NOWRITER status), then dispose of the thread.
   */
   if (!(RequestPtr->DclSubprocessPid ||
         RequestPtr->DclOutstandingIO ||
         RequestPtr->DclOutstandingClientRead))
      ConcludeProcessing (RequestPtr);
}

/*****************************************************************************/
/*
When a thread is about to be disposed of this function must be called to 
deassign any DCL-associated channels assigned.
*/ 

DclDispose (struct RequestStruct *RequestPtr)

{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DclDispose()\n");

   /* deassign script-associated channels */
   if (RequestPtr->DclSysInputChannel)
   {
      sys$dassgn (RequestPtr->DclSysInputChannel);
      RequestPtr->DclSysInputChannel = 0;
   }
   if (RequestPtr->DclSysOutputChannel)
   {
      sys$dassgn (RequestPtr->DclSysOutputChannel);
      RequestPtr->DclSysOutputChannel = 0;
   }
   if (RequestPtr->DclHttpInputChannel)
   {
      sys$dassgn (RequestPtr->DclHttpInputChannel);
      RequestPtr->DclHttpInputChannel = 0;
   }

   RequestPtr->DclHasBeenExecuting = false;
}

/*****************************************************************************/
/*
A channel has been assigned to the subprocess' SYS$OUTPUT mailbox.  Queue the 
first asynchronous read of the subprocess' output.
*/ 

DclSysOutputWaitAST (struct RequestStruct *RequestPtr)

{
   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout,
               "DclSysOutputWaitAST() IO %d Status %%X%08.08X\n",
               RequestPtr->DclOutstandingIO,
               RequestPtr->DclSysOutputIOsb.Status);

   if (RequestPtr->DclOutstandingIO) RequestPtr->DclOutstandingIO--;

   if (VMSnok (RequestPtr->DclSysOutputIOsb.Status))
   {
      if (RequestPtr->DclSysOutputIOsb.Status == SS$_ABORT ||
          RequestPtr->DclSysOutputIOsb.Status == SS$_CANCEL)
      {
         /* looks like the script has been cancelled before it started! */
         return;
      }

      ErrorVmsStatus (RequestPtr, RequestPtr->DclSysOutputIOsb.Status,
                      __FILE__, __LINE__);
      DclAbortSubprocess (RequestPtr);
      return;
   }

   /* queue the initial read of the subprocess' SYS$OUTPUT */
   DclQioSysOutput (RequestPtr);
}

/*****************************************************************************/
/*
A queued asynchronous write of subprocess SYS$OUTPUT (mailbox) to the client
over the network has completed.
*/ 

DclSysOutputToClientAST (struct RequestStruct *RequestPtr)

{
   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout,
               "DclSysOutputToClientAST() IO %d Status %%X%08.08X\n",
               RequestPtr->DclOutstandingIO,
               RequestPtr->DclSysOutputIOsb.Status);

   if (RequestPtr->DclOutstandingIO) RequestPtr->DclOutstandingIO--;

   if (VMSnok (RequestPtr->NetWriteIOsb.Status))
   {
      if (RequestPtr->NetWriteIOsb.Status != SS$_ABORT &&
          RequestPtr->NetWriteIOsb.Status != SS$_CANCEL)
      {
         ErrorVmsStatus (RequestPtr, RequestPtr->NetWriteIOsb.Status,
                         __FILE__, __LINE__);
         DclAbortSubprocess (RequestPtr);
         return;
      }
   }

   /* queue the next read of the subprocess' SYS$OUTPUT */
   DclQioSysOutput (RequestPtr);
}

/*****************************************************************************/
/*
Queue up a read from the subprocess "SYS$OUTPUT" mailbox.  When the read 
completes call function DclSysOutputAST(), do any post-processing 
required and write the data to the client over the network.  The next read 
from the subprocess via the mailbox will be queued by the network write 
completion AST function.
*/ 

DclQioSysOutput (struct RequestStruct *RequestPtr)

{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DclSysOutput()\n");

   if (VMSok (status =
       sys$qio (0, RequestPtr->DclSysOutputChannel,
                IO$_READLBLK | IO$M_WRITERCHECK,
                &RequestPtr->DclSysOutputIOsb,
                &DclSysOutputAST, RequestPtr,
                RequestPtr->DclSysOutputPtr, RequestPtr->DclSysOutputSize,
                0, 0, 0, 0)))
   {
      RequestPtr->DclOutstandingIO++;
      return;
   }
}

/*****************************************************************************/
/*
A queued asynchronous read from the subprocess "SYS$OUTPUT" mailbox has 
completed.  If this record is part of the HTTP header do some specific 
processing.  If required append carriage-control (newline) to this record to 
make it HTTP compliant.  Queue a write of this data to the client over the 
network.  The next read from the subprocess via the mailbox will be queued by 
the network write completion AST function, DclQioSysOutput().
*/ 

DclSysOutputAST (struct RequestStruct *RequestPtr)

{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
   {
      fprintf (stdout,
      "DclSysOutputAST() IO %d ClientRead %d Count %d Status %%X%08.08X\n",
      RequestPtr->DclOutstandingIO,
      RequestPtr->DclOutstandingClientRead,
      RequestPtr->DclSysOutputIOsb.Count,
      RequestPtr->DclSysOutputIOsb.Status);
   }

   if (RequestPtr->DclOutstandingIO) RequestPtr->DclOutstandingIO--;

   if (RequestPtr->DclSysOutputIOsb.Status == SS$_ENDOFFILE)
   {
      /* 
          If a script spawns multiple subprocesses each will
          terminate by queueing an end-of-file.  Ignore these.
          Queue the next read of the subprocess' SYS$OUTPUT, a
          SS$_NOWRITER signals the completion of the subprocess!
      */
      DclQioSysOutput (RequestPtr);
      return;
   }

   if (VMSnok (RequestPtr->DclSysOutputIOsb.Status))
   {
      if (RequestPtr->DclSysOutputIOsb.Status == SS$_NOWRITER)
      {
         /* subprocess has terminated, cancel I/O on client network channel */
         if (RequestPtr->DclOutstandingClientRead)
            sys$cancel (RequestPtr->ClientChannel);
         else;
      }
      else
      if (RequestPtr->DclSysOutputIOsb.Status != SS$_ABORT &&
          RequestPtr->DclSysOutputIOsb.Status != SS$_CANCEL)
      {
         ErrorVmsStatus (RequestPtr, RequestPtr->DclSysOutputIOsb.Status,
                         __FILE__, __LINE__);
         DclAbortSubprocess (RequestPtr);
         return;
      }

      /*
         If the subprocess has gone, no outstanding IO, and SYS$OUTPUT is
         known to be empty (SS$_NOWRITER status), then dispose of the thread.
      */
      if (!(RequestPtr->DclSysOutputIOsb.Count ||
            RequestPtr->DclSubprocessPid ||
            RequestPtr->DclOutstandingIO ||
            RequestPtr->DclOutstandingClientRead))
         DclEnd (RequestPtr);

      return;
   }

   if (RequestPtr->LocationPtr != NULL)
   {
      /*
         Absorb any output from script until the subprocess is terminates.
         The script has output a "Location:" redirection header for a
         partial URL that can be processed locally.  This request will be
         resubmitted to the server as soon as the current script subprocess
         terminates.
      */
      DclQioSysOutput (RequestPtr);
      return;
   }

   if (RequestPtr->ScriptName[0])
   {
      /********************/
      /* script execution */
      /********************/

      if (RequestPtr->DclCheckSubprocessResponseHeader)
      {
         if (!DclSysOutputHeader (RequestPtr))
         {
            DclAbortSubprocess (RequestPtr);
            return;
         }
      }

      if (RequestPtr->LocationPtr != NULL)
      {
         /* redirection, absorb any output */
         DclQioSysOutput (RequestPtr);
         return;
      }
   }

   if (RequestPtr->DclAddNewline)
   {
      /*************************/
      /* HTTP carriage-control */
      /*************************/

      if (RequestPtr->DclSysOutputIOsb.Count)
      {
         if (RequestPtr->
               DclSysOutputPtr[RequestPtr->DclSysOutputIOsb.Count-1] != '\n')
            RequestPtr->
               DclSysOutputPtr[RequestPtr->DclSysOutputIOsb.Count++] = '\n';
         else;
      }
      else
      {
         /* must be a blank line (empty record), add a newline */
         RequestPtr->
            DclSysOutputPtr[RequestPtr->DclSysOutputIOsb.Count++] = '\n';
      }
      if (Debug)
         RequestPtr->
            DclSysOutputPtr[RequestPtr->DclSysOutputIOsb.Count] = '\0';

      /*********************/
      /* buffer the record */
      /*********************/

      /*
         The AST routine address supplied ensures that at the completion
         of the network write to the client DclQioSysOutput() function
         is called to queue another read from the subprocess.
      */

      BufferOutput (RequestPtr, &DclSysOutputToClientAST,
                    RequestPtr->DclSysOutputPtr,
                    RequestPtr->DclSysOutputIOsb.Count);
      if (!RequestPtr->ErrorMessagePtr)
      {
         RequestPtr->DclOutstandingIO++;
         return;
      }

      DclAbortSubprocess (RequestPtr);
      return;
   }

   /********************/
   /* write the record */
   /********************/

   if (Debug)
   {
      RequestPtr->DclSysOutputPtr[RequestPtr->DclSysOutputIOsb.Count] = '\0';
      fprintf (stdout, "|%s|\n", RequestPtr->DclSysOutputPtr);
   }

   /*
      The AST routine address supplied ensures that at the completion
      of the network write to the client DclQioSysOutput() function
      is called to queue another read from the subprocess.
   */
   if (VMSok (status =
      QioNetWrite (RequestPtr, &DclSysOutputToClientAST,
                   RequestPtr->DclSysOutputPtr,
                   RequestPtr->DclSysOutputIOsb.Count)))
   {
      RequestPtr->DclOutstandingIO++;
      return;
   }

   ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
   DclAbortSubprocess (RequestPtr);
}

/*****************************************************************************/
/*
Queue up a write of data to the subprocess "SYS$INPUT" mailbox.  This is the 
subprocesses' "SYS$COMMAND", supplying the DCL commands to execute.
*/ 

DclQioSysInput 
(
struct RequestStruct *RequestPtr,
char *DataPtr,
int DataLength
)
{
   register int  status;

   if (Debug) fprintf (stdout, "DclQioSysInput() |%s|\n", DataPtr);

   if (VMSok (status =
       sys$qio (0, RequestPtr->DclSysInputChannel,
                IO$_WRITELBLK, 0,
                &DclSysInputAST, RequestPtr,
                DataPtr, DataLength,
                0, 0, 0, 0)))
      RequestPtr->DclOutstandingIO++;

   return (status);
}

/*****************************************************************************/
/*
A queued write to the subprocess "SYS$INPUT" mailbox has completed.  This is 
the subprocesses' "SYS$COMMAND", supplying the DCL commands to execute.
*/ 

DclSysInputAST (struct RequestStruct *RequestPtr)

{
   if (Debug)
      fprintf (stdout, "DclSysInputAST() IO %d\n",
               RequestPtr->DclOutstandingIO);

   if (RequestPtr->DclOutstandingIO) RequestPtr->DclOutstandingIO--;

   /* if the subprocess has exited and there is no outstanding I/O */
   if (!(RequestPtr->DclSubprocessPid ||
         RequestPtr->DclOutstandingIO ||
         RequestPtr->DclOutstandingClientRead))
      DclEnd (RequestPtr);
}

/*****************************************************************************/
/*
A queued read from the client over network has completed, write it to the
subprocess' HTTP$INPUT channel.
*/ 

DclQioHttpInput (struct RequestStruct *RequestPtr)

{
   int  status;

   if (Debug)
      fprintf (stdout,
               "DclQioHttpInput() IO %d Count %d Status %%X%08.08X\n",
               RequestPtr->DclOutstandingIO,
               RequestPtr->NetReadIOsb.Count,
               RequestPtr->NetReadIOsb.Status);

   if (RequestPtr->DclOutstandingIO) RequestPtr->DclOutstandingIO--;
   if (RequestPtr->DclOutstandingClientRead)
      RequestPtr->DclOutstandingClientRead--;

   if (VMSnok (RequestPtr->NetReadIOsb.Status))
   {
      if (RequestPtr->NetReadIOsb.Status != SS$_ABORT &&
          RequestPtr->NetReadIOsb.Status != SS$_CANCEL &&
          RequestPtr->NetReadIOsb.Status != SS$_ENDOFFILE)
      {
         ErrorVmsStatus (RequestPtr, RequestPtr->NetReadIOsb.Status,
                         __FILE__, __LINE__);
         DclAbortSubprocess (RequestPtr);
         return;
      }
   }

   /* if the subprocess has exited and there is no outstanding I/O */
   if (!(RequestPtr->DclSubprocessPid ||
         RequestPtr->DclOutstandingIO ||
         RequestPtr->DclOutstandingClientRead))
   {
      DclEnd (RequestPtr);
      return;
   }

   RequestPtr->BytesRx += RequestPtr->NetReadIOsb.Count;

   if (VMSok (status =
       sys$qio (0, RequestPtr->DclHttpInputChannel,
                IO$_WRITELBLK, &RequestPtr->DclHttpInputIOsb,
                &DclHttpInputAST, RequestPtr,
                RequestPtr->NetReadBufferPtr, RequestPtr->NetReadIOsb.Count,
                0, 0, 0, 0)))
   {
      RequestPtr->DclOutstandingIO++;
      return;
   }

   ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
   DclAbortSubprocess (RequestPtr);
}

/*****************************************************************************/
/*
A queued write to the subprocess "HTTP$INPUT" mailbox has completed, queue 
another read from the client over network.
*/ 

DclHttpInputAST (struct RequestStruct *RequestPtr)

{
   int  status;

   if (Debug)
   {
      fprintf (stdout,
"DclHttpInputAST() IO %d ClientRead %d Count %d Status %%X%08.08X\n",
      RequestPtr->DclOutstandingIO,
      RequestPtr->DclOutstandingClientRead,
      RequestPtr->DclHttpInputIOsb.Count,
      RequestPtr->DclHttpInputIOsb.Status);
   }

   if (RequestPtr->DclOutstandingIO) RequestPtr->DclOutstandingIO--;

   if (VMSnok (RequestPtr->DclHttpInputIOsb.Status))
   {
      if (RequestPtr->DclHttpInputIOsb.Status != SS$_ABORT &&
          RequestPtr->DclHttpInputIOsb.Status != SS$_CANCEL &&
          RequestPtr->DclHttpInputIOsb.Status != SS$_ENDOFFILE)
      {
         ErrorVmsStatus (RequestPtr, RequestPtr->DclHttpInputIOsb.Status,
                         __FILE__, __LINE__);
         DclAbortSubprocess (RequestPtr);
         return;
      }
   }

   /* if the subprocess has exited and there is no outstanding I/O */
   if (!(RequestPtr->DclSubprocessPid ||
         RequestPtr->DclOutstandingIO ||
         RequestPtr->DclOutstandingClientRead))
   {
      DclEnd (RequestPtr);
      return;
   }

   if (VMSnok (RequestPtr->DclHttpInputIOsb.Status)) return;

   if (VMSok (status =
      QioNetRead (RequestPtr,
                  &DclQioHttpInput,
                  RequestPtr->NetReadBufferPtr,
                  RequestPtr->NetReadBufferSize)))
   {
      RequestPtr->DclOutstandingClientRead++;
      return;
   }

   ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
   DclAbortSubprocess (RequestPtr);
}

/*****************************************************************************/
/*
For CGI-compliance the first line of the header should be a "Content-Type:", 
"Location:" or "Status:".  If a "Content-Type:" prepend a "200" header status 
line.  If a "Status:" use this to create a unique header status line.  If a 
"Location:" redirection header line call a function to establish the level of 
redirection, and do any associated processing.  For non-parse-header style CGI 
scripts check if the first line is a full HTTP status line.  If it is then do 
nothing, the script will supply the raw HTTP stream.  If none of the above 
then consider the script will supply nothing but plain text, none of the HTTP 
header, etc., and prepend a "200" status line and a "Content-Type: text/plain" 
line to create a complete HTTP header.  This is an extension to CGI behaviour, 
but makes simple, plain-text DCL scripts very easy!
*/ 

DclSysOutputHeader (struct RequestStruct *RequestPtr)

{
   static int  Http200StatusLength,
               Http302StatusLength;

   static char  Http200Status [256],
                Http302Status [256];

   static $DESCRIPTOR (Http200StatusFaoDsc,
"!AZ 200 CGI data follows.\r\n\
Server: !AZ\r\n");
   static $DESCRIPTOR (Http302StatusFaoDsc,
"!AZ 302 CGI redirection.\r\n\
Server: !AZ\r\n");

   static $DESCRIPTOR (Http200StatusDsc, Http200Status);
   static $DESCRIPTOR (Http302StatusDsc, Http302Status);

   register char  *cptr, *sptr;

   char  Scratch [256];
   unsigned short  Length;

   /*********/
   /* begin */
   /*********/

   if (Debug)
   {
      RequestPtr->DclSysOutputPtr[RequestPtr->DclSysOutputIOsb.Count] = '\0';
      fprintf (stdout, "DclSysOutputHeader()\n|%s|\n",
               RequestPtr->DclSysOutputPtr);
   }

   RequestPtr->DclCheckSubprocessResponseHeader = false;

   if (!Http200StatusLength)
   {
      /******************/
      /* initialization */
      /******************/

      /* create a HTTP status line responses for CGI script compliance */
      sys$fao (&Http200StatusFaoDsc, &Length, &Http200StatusDsc,
               HttpProtocol, SoftwareID);
      Http200Status[Http200StatusLength = Length] = '\0';
      sys$fao (&Http302StatusFaoDsc, &Length, &Http302StatusDsc,
               HttpProtocol, SoftwareID);
      Http302Status[Http302StatusLength = Length] = '\0';
   }

   /************************************/
   /* first line(s) output from script */
   /************************************/

   RequestPtr->DclSysOutputPtr[RequestPtr->DclSysOutputIOsb.Count] = '\0';
   cptr = RequestPtr->DclSysOutputPtr;

   if (toupper(*cptr) == 'H' && strsame (cptr, "HTTP/", 5))
   {
      /*************************************************/
      /* script is supplying the full HTTP data stream */
      /*************************************************/

      if (Debug) fprintf (stdout, "(HTTP data stream)\n");
      RequestPtr->DclAddNewline = false;

      /* get the response status code for logging purposes */
      while (*cptr && !isspace(*cptr)) cptr++;
      while (*cptr && !isdigit(*cptr)) cptr++;
      if (isdigit(*cptr))
         RequestPtr->ResponseStatusCode = atoi(cptr);
      else;
   }
   else
   if (toupper(*cptr) == 'L' && strsame (cptr, "Location:", 9))
   {
      /*******************/
      /* redirection URL */
      /*******************/

      cptr += 9;
      /* locate the first character of the "Location:" URL */
      while (isspace(*cptr)) cptr++;
      if (*cptr == '/')
      {
         /* not a fully-specified URL */
         sptr = cptr;
         while (*cptr && !isspace(*cptr) && *cptr != '\r' && *cptr != '\n')
            cptr++;
         if ((RequestPtr->LocationPtr = HeapAlloc (RequestPtr, cptr-sptr+1))
             == NULL)
         {
            ErrorHeapAlloc (RequestPtr, __FILE__, __LINE__);
            return (false);
         }
         memcpy (RequestPtr->LocationPtr, sptr, cptr-sptr);
         RequestPtr->LocationPtr[cptr-sptr] = '\0';
         if (Debug)
            fprintf (stdout, "LocationPtr |%s|\n", RequestPtr->LocationPtr);
         else;
      }
      else;
   }
   else
   if (toupper(*cptr) == 'C' && strsame (cptr, "Content-Type:", 13))
   {

      /**************************/
      /* content-type specified */
      /**************************/

      /* make sure each record received has HTTP carriage control */
      if (Debug) fprintf (stdout, "(RMS record stream)\n");
      RequestPtr->DclAddNewline = true;
      /* should only occur at the beginning of output, so no AST is safe :^) */
      BufferOutput (RequestPtr, 0, Http200Status, Http200StatusLength);
      RequestPtr->ResponseStatusCode = 200;
   }
   else
   if (toupper(*cptr) == 'S' && strsame (cptr, "Status:", 7))
   {
      /********************/
      /* status specified */
      /********************/

      /* make sure each record received has HTTP carriage control */
      if (Debug) fprintf (stdout, "(RMS record stream)\n");
      RequestPtr->DclAddNewline = true;
      /* create HTTP header status line using supplied status */
      sptr = Scratch;
      strcpy (sptr, HttpProtocol);
      sptr += sizeof(HttpProtocol)-1;
      *sptr++ = ' ';
      cptr += 7;
      while (isspace(*cptr)) cptr++;
      /* get the response status code for logging purposes */
      if (isdigit(*cptr)) RequestPtr->ResponseStatusCode = atoi(cptr);
      /* append the rest of the line (e.g. "403 Forbidden") */
      while (*cptr && *cptr != '\r' && *cptr != '\n') *sptr++ = *cptr++;
      *sptr = '\0';
      /* should only occur at the beginning of output, so no AST is safe :^) */
      BufferOutput (RequestPtr, 0, Scratch, sptr-Scratch);
   }
   else
   {
      /******************/
      /* non-CGI output */
      /******************/

      /* make sure each record received has HTTP carriage control */
      if (Debug) fprintf (stdout, "(RMS record stream)\n");
      RequestPtr->DclAddNewline = true;
      /* should only occur at the beginning of output, so no AST is safe :^) */
      BufferOutput (RequestPtr, 0, Http200Status, Http200StatusLength);
      RequestPtr->ResponseStatusCode = 200;
      /* should only occur at the beginning of output, so no AST is safe :^) */
      BufferOutput (RequestPtr, 0, "Content-Type: text/plain\r\n\r\n", 28);
   }
}

/*****************************************************************************/

