/*****************************************************************************/
/*
                                  Dir.c

This module implements a full multi-threaded, AST-driven, asynchronous 
directory listing.  The AST-driven nature makes the code a little more 
difficult to follow, but creates a powerful, event-driven, multi-threaded 
server.  All of the necessary functions implementing this module are designed 
to be non-blocking. 

This module uses the BufferOutput() function to buffer any output it can into
larger chunks before sending it to the client.

A directory listing is recognised by the server whenever a wildcard is present 
in a specification and there is no query string directing another activity 
(e.g. a query/search).  Compliant with other HTTPD implementations, a 
directory listing is also generated if a URL specifies a directory only and 
the directory contains no home page. 

The directory listing is designed to look very much like the default layout of 
the CERN and NCSA HTTPD implementations, with the exception that with BHTS all 
directories are grouped at the top, not dispersed throughout the file names.  
This looks and functions better than the above (at least in the experience of 
the author). 

The layout of the listing can be controlled from the configuration file or via
a server directive.  A decimal number may optionally, immediately precede any
directive, this becomes the width of that particular field.  If a width is not
specified an appropriate default is used.  Information wider than the field
width is truncated without indication.  The following layout directives
provide:

  _     (underscore) each underscore provides one space between fields
  C     creation date
  D     content-type description (BEST be the last field specified)
  I     icon
  L     file anchor (including hot-spot name)
  N     file name (without anchor, why I can't imagine :^)
  O     file owner (optional, configuration file enable/disable)
  R     revision date
  S     file size

These may placed in any order ("D" is best last because it does not use a 
field width) and with any (reasonable) field-width.  For example see 
'DefaultDirLayout' below. 


VERSION HISTORY
---------------
01-DEC-95  MGD  HTTPd version 3
27-SEP-95  MGD  major revision of file and directory anchor generation
                (far fewer MapUrl()s yielding greater efficiency);
                added query string capability;
                added automatic script capability
07-AUG-95  MGD  include VMS-style directory layout and file sizes
16-JUN-95  MGD  file contents description for non-HTML files (see config.c)
25-MAR-95  MGD  bugfix for error handling with DirReadMe() and DirFileExists()
20-DEC-94  MGD  multi-threaded daemon (it suddenly got very complex!)
20-JUN-94  MGD  single-threaded daemon
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <string.h>

/* VMS related header files */
#include <atrdef.h>
#include <descrip.h>
#include <dvidef.h>
#include <fibdef.h>
#include <iodef.h>
#include <rmsdef.h>
#include <rms.h>
#include <ssdef.h>
#include <stsdef.h>

/* application related header file */
#include "httpd.h"

/**********/
/* macros */
/**********/

#define DefaultFieldWidthInterField 1
#define DefaultFieldWidthCdt 15
#define DefaultFieldWidthName 25
#define DefaultFieldWidthOwner 20
#define DefaultFieldWidthRdt 15
#define DefaultFieldWidthSize 5

/* look through the first 20 lines of an HTML file before quitting */
#define MaxDirDescriptionLineCount 20

#define DirBlankContentTypeIcon "x-internal/blank"
#define DirParentContentTypeIcon "x-internal/parent"
#define DirUnknownContentTypeIcon "x-internal/unknown"

/******************/
/* global storage */
/******************/

char  DirBrowsableFileName [] = ".WWW_BROWSABLE",
      DirHiddenFileName [] = ".WWW_HIDDEN",
      DirLayoutProblem [] = "Directory layout problem.",
      DirNopFileName [] = ".WWW_NOP",
      DirNosFileName [] = ".WWW_NOS",
      DirNopsFileName [] = ".WWW_NOPS",
      DirParentDirectoryDescription [] = "parent directory",
      DirSubDirectoryDescription [] = "subdirectory";

/********************/
/* external storage */
/********************/

extern boolean  Debug;
extern int  FileBufferSize;
extern char  ErrorFacilityDisabled[];
extern char  SoftwareID[];
extern char  ErrorStringSize[];
extern struct AccountingStruct  Accounting;
extern struct ConfigStruct  Config;

/****************************/
/* functions in this module */
/****************************/

DirBegin (struct RequestStruct*);
DirBeginDirectories (struct RequestStruct*);
DirBeginFiles (struct RequestStruct*);
DirDescription (struct RequestStruct*);
DirDescriptionHtml (struct RequestStruct*);
DirDescriptionHtmlRecord (struct RAB*);
DirDirectories (struct FAB*);
DirEnd (struct RequestStruct*);
DirEndOutput (struct RequestStruct*);
DirFiles (struct FAB*);
boolean DirFileExists (struct RequestStruct*, char*, char*);
DirFormat (struct RequestStruct*, void*, char*, boolean);
DirFormatLayout (struct RequestStruct*);
DirHeading (struct RequestStruct*);
int DirHttpHeader (struct RequestStruct*);
DirReadMe (struct RequestStruct*,
           void(*NextTaskFunction)(struct RequestStruct*));
DirReadMeTop (struct RequestStruct*);
DirReadMeBottom (struct RequestStruct*);
DirSearchDirectories (struct RequestStruct*);

/**********************/
/* external functions */
/**********************/

int BufferOutput (struct RequestStruct*, void*, char*, int);
char* ConfigReadMeFile (int);
ConcludeProcessing (struct RequestStruct*);
char* ConfigHomePage (int);
char* ConfigIconFor (struct RequestStruct*, char*);
char* ConfigReadMeFile (int);
ErrorGeneral (struct RequestStruct*, char*, char*, int);
ErrorVmsStatus (struct RequestStruct*, int, char*, int);
unsigned char*  HeapAlloc (struct RequestStruct*, int);
QioNetWrite (struct RequestStruct*, void*, char*, int);

/*****************************************************************************/
/*
Generate a directory listing for a client.
Needless-to-say, it does not block while generating the listing. 

As it is generally difficult to determine if the contents of a directory have 
changed without pre-processing that directory, no check of any
"If-Modified-Since:" request field date/time is made, and no "Last-Modified:" 
field is included in any response header. 

'RequestPtr->DirSpec' contains the VMS file specification.
'RequestPtr->DirPathInfoPtr' points at the virtual directory specification.
'RequestPtr->DirQueryStringPtr' points at any server directive query string.
'RequestPtr->SendHttpHeader' is true if an HTTP header should be generated.
'RequestPtr->DirLayoutPtr' points at a directory layout string (or NULL)
*/ 
 
DirBegin (struct RequestStruct *RequestPtr)

{
   static $DESCRIPTOR (IndexOfFaoDsc,
"<TITLE>Index of !AZ</TITLE>\n\
<H1>Index of <TT>!AZ</TT></H1>\n");

   register char  *cptr, *sptr;

   int  status;
   unsigned short  Length;
   char  Scratch [256],
         String [1024];
   void  *AstFunctionPtr;
   $DESCRIPTOR (StringDsc, String);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DirBegin() |%s|\n", RequestPtr->DirSpec);

   /* make sure any "If-Modified-Since:" request header line is ignored */
   RequestPtr->IfModifiedSinceBinaryTime[0] =
      RequestPtr->IfModifiedSinceBinaryTime[1] = 0;

   if (RequestPtr->AdjustAccounting)
   {
      Accounting.DoDirectoryCount++;
      RequestPtr->AdjustAccounting = 0;
   }

   if (!Config.DirAccess)
   {
      Accounting.RequestForbiddenCount++;
      RequestPtr->ResponseStatusCode = 501;
      ErrorGeneral (RequestPtr, ErrorFacilityDisabled, __FILE__, __LINE__);
      DirEnd (RequestPtr);
      return;
   }

   /********************************************/
   /* check if there are any server directives */
   /********************************************/

   if (Debug)
      fprintf (stdout, "DirQueryStringPtr |%s|\n",
               RequestPtr->DirQueryStringPtr);

   if (RequestPtr->DirQueryStringPtr[0])
   {
      cptr = RequestPtr->DirQueryStringPtr;
      while (*cptr)
      {
         if (strsame (cptr, "autoscript=", 11))
         {
            cptr += 11;
            /* if "false", "no" or "disabled" then turn off auto-scripting */
            if (toupper(*cptr) == 'F' || toupper(*cptr) == 'N' ||
                toupper(*cptr) == 'D')
               RequestPtr->AutoScriptEnabled = false;
            else
               RequestPtr->AutoScriptEnabled = true;
         }
         else
         if (strsame (cptr, "layout=", 7))
         {
            RequestPtr->DirLayoutPtr = cptr+7;
            while (*cptr && *cptr != '&') cptr++;
            if (*cptr) cptr++;
         }
         else
         if (strsame (cptr, "script=", 7))
         {
            /* local storage */
            register char  *zptr;

            cptr += 7;
            zptr = (sptr = RequestPtr->DirScriptName) +
                    sizeof(RequestPtr->DirScriptName);
            if (*cptr != '/') *sptr++ = '/';
            while (*cptr && *cptr != '&' && sptr < zptr) *sptr++ = *cptr++;
            if (sptr >= zptr)
            {
               RequestPtr->ResponseStatusCode = 500;
               ErrorGeneral (RequestPtr, ErrorStringSize, __FILE__, __LINE__);
               DirEnd (RequestPtr);
               return;
            }
            *sptr = '\0';
         }
         while (*cptr && *cptr != '&') cptr++;
         if (*cptr) cptr++;
      }
   }

   /*************************************/
   /* parse and get required components */
   /*************************************/

   /* look for an elipsis wildcard in specification (i.e. "...") */
   for (cptr = RequestPtr->DirSpec; *cptr; cptr++)
      if ((*(unsigned long*)cptr & 0x00ffffff) == 0x002e2e2e) break;
   if (*cptr)
   {
      /* detected, remove */
      sptr = cptr + 3;
      while (*sptr) *cptr++ = *sptr++;
      *cptr = '\0';
   }

   /* set up default error report information (just in case!) */
   RequestPtr->ErrorTextPtr = RequestPtr->DirPathInfoPtr;
   RequestPtr->ErrorHiddenTextPtr = RequestPtr->DirSpec;

   RequestPtr->SearchFab = cc$rms_fab;
   RequestPtr->SearchFab.fab$l_dna = "*.*";
   RequestPtr->SearchFab.fab$b_dns = 3;
   RequestPtr->SearchFab.fab$l_fna = RequestPtr->DirSpec;
   RequestPtr->SearchFab.fab$b_fns = strlen(RequestPtr->DirSpec);
   RequestPtr->SearchFab.fab$l_fop = FAB$M_NAM;
   RequestPtr->SearchFab.fab$l_nam = &RequestPtr->SearchNam;

   RequestPtr->SearchNam = cc$rms_nam;
   RequestPtr->SearchNam.nam$l_esa = RequestPtr->ExpandedDirSpec;
   RequestPtr->SearchNam.nam$b_ess = sizeof(RequestPtr->ExpandedDirSpec)-1;

   if (VMSnok (status = sys$parse (&RequestPtr->SearchFab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$parse() %%X%08.08X\n", status);
      ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
      DirEnd (RequestPtr);
      return;
   }

   memcpy (RequestPtr->DirDirectoryPart,
           RequestPtr->SearchNam.nam$l_dev,
           Length = RequestPtr->SearchNam.nam$l_name -
                    RequestPtr->SearchNam.nam$l_dev);
   RequestPtr->DirDirectoryPart[Length] = '\0';

   /* if a wildcard or explicit version number supplied the VMS format */
   if (RequestPtr->SearchNam.nam$l_fnb & NAM$M_WILD_VER ||
       RequestPtr->SearchNam.nam$l_fnb & NAM$M_EXP_VER)
   {
      memcpy (RequestPtr->DirFilePart,
              RequestPtr->SearchNam.nam$l_name,
              Length = RequestPtr->SearchNam.nam$l_ver -
                       RequestPtr->SearchNam.nam$l_name +
                       RequestPtr->SearchNam.nam$b_ver);
      RequestPtr->DirFormatVms = true;
   }
   else
      memcpy (RequestPtr->DirFilePart,
              RequestPtr->SearchNam.nam$l_name,
              Length = RequestPtr->SearchNam.nam$l_ver -
                       RequestPtr->SearchNam.nam$l_name);
   RequestPtr->DirFilePart[Length] = '\0';

   Scratch[0] = '\0';
   sptr = MapUrl (Scratch, RequestPtr->DirDirectoryPart, NULL, NULL);
   if (!sptr[0] && sptr[1])
   {
      /* mapping report */
      RequestPtr->ResponseStatusCode = 403;
      ErrorGeneral (RequestPtr, sptr+1, __FILE__, __LINE__);
      DirEnd (RequestPtr);
      return;
   }
   Length = strlen(Scratch) + 1;
   if ((RequestPtr->DirDirectoryPathPtr =
       HeapAlloc (RequestPtr, Length)) == NULL)
   {
      ErrorHeapAlloc (RequestPtr, __FILE__, __LINE__);
      DirEnd (RequestPtr);
      return;
   }
   memcpy (RequestPtr->DirDirectoryPathPtr, Scratch, Length);

   if (Debug)
      fprintf (stdout,
      "DirDirectoryPart |%s|\nDirFilePart |%s|\nDirDirectoryUrl |%s|\n",
      RequestPtr->DirDirectoryPart, RequestPtr->DirFilePart,
      RequestPtr->DirDirectoryPathPtr);

   /************************/
   /* directory access ok? */
   /************************/

   if ((Config.DirAccessSelective &&
        !DirFileExists (RequestPtr, RequestPtr->DirDirectoryPart,
                        DirBrowsableFileName)) ||
       DirFileExists (RequestPtr, RequestPtr->DirDirectoryPart,
                      DirHiddenFileName))
   {
      /* give some "disinformation"  ;^)  */
      status = RMS$_DNF;
      ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
      DirEnd (RequestPtr);
      return;
   }

   /********************/
   /* begin processing */
   /********************/

   RequestPtr->DirFileCount = RequestPtr->DirDirectoryCount = 0;

   if (VMSnok (DirFormatLayout (RequestPtr)))
   {
      DirEnd (RequestPtr);
      return;
   }

   if (RequestPtr->SendHttpHeader)
   {
      if (VMSnok (DirHttpHeader (RequestPtr)))
      {
         DirEnd (RequestPtr);
         return;
      }
      if (RequestPtr->MethodHead)
      {
         DirEnd (RequestPtr);
         return;
      }

      if (RequestPtr->DirFormatVms)
         sys$fao (&IndexOfFaoDsc, &Length, &StringDsc,
                  RequestPtr->DirSpec, RequestPtr->DirSpec);
      else
         sys$fao (&IndexOfFaoDsc, &Length, &StringDsc,
                  RequestPtr->PathInfoPtr, RequestPtr->PathInfoPtr);
      String[Length] = '\0';

      if (Config.DirReadMeTop)
         AstFunctionPtr = &DirReadMeTop;
      else
         AstFunctionPtr = &DirHeading;

      if (VMSnok (BufferOutput (RequestPtr, AstFunctionPtr, String, Length)))
      {
         DirEnd (RequestPtr);
         return;
      }
   }
   else
      DirHeading (RequestPtr);
}

/*****************************************************************************/
/*
Release any dynamic memory allocated for parse structures.
*/ 

DirEnd (struct RequestStruct *RequestPtr)

{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DirEnd()\n");

   /* ensure multiple dir() calls start off without a "hangover" layout */
   RequestPtr->DirLayoutPtr = NULL;

   /* ensure parse internal data structures are released */
   RequestPtr->SearchNam.nam$b_nop = NAM$M_SYNCHK;
   sys$parse (&RequestPtr->SearchFab, 0, 0);

   ConcludeProcessing (RequestPtr);
}

/*****************************************************************************/
/*
This function generates an HTTP header for the following directory index.
*/ 
 
int DirHttpHeader (struct RequestStruct *RequestPtr)

{
   static $DESCRIPTOR (StringDsc, "");

   static $DESCRIPTOR (CommentedInfoFaoDsc,
"<!!-- SoftwareID: !AZ -->\n\
<!!-- !AZ -->\n");

   static $DESCRIPTOR (Http200HeaderDsc,
"!AZ 200 Data follows.\r\n\
Server: !AZ\r\n\
Date: !AZ\r\n\
Content-Type: text/html\r\n\
\r\n");

   int  status;
   unsigned short  CommentLength,
                   Length,
                   HeaderLength;
   char  String [1024];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DirHttpHeader()\n");

   RequestPtr->ResponseStatusCode = 200;

   StringDsc.dsc$a_pointer = String;
   StringDsc.dsc$w_length = sizeof(String)-1;

   sys$fao (&Http200HeaderDsc, &HeaderLength, &StringDsc,
            HttpProtocol, SoftwareID, RequestPtr->GmDateTime);

   if (Config.IncludeCommentedInfo && !RequestPtr->MethodHead)
   {
      StringDsc.dsc$a_pointer = String + HeaderLength;
      StringDsc.dsc$w_length = sizeof(String) - HeaderLength - 1;

      status = sys$fao (&CommentedInfoFaoDsc, &CommentLength, &StringDsc,
                        SoftwareID, RequestPtr->DirSpec);
   }
   else
      CommentLength = 0;

   String[Length = HeaderLength+CommentLength] = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", String);

   /* allocate heap memory for the response header */
   if ((RequestPtr->ResponseHeaderPtr =
        HeapAlloc (RequestPtr, Length)) == NULL)
   {
      ErrorHeapAlloc (RequestPtr, __FILE__, __LINE__);
      return (SS$_INSFMEM);
   }
   memcpy (RequestPtr->ResponseHeaderPtr, String, Length);

   if (VMSnok (status =
       QioNetWrite (RequestPtr, 0, String, Length)))
      ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);

   RequestPtr->SendHttpHeader = false;

   return (status);
}

/*****************************************************************************/
/*
AST-driven output of column headings (these have already generated by 
DirFormatLayout()).
*/ 

DirHeading (struct RequestStruct *RequestPtr)

{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DirHeading()\n");

   if (VMSnok (BufferOutput (RequestPtr, &DirBeginDirectories,
                             RequestPtr->DirLayoutHeadingPtr,
                             RequestPtr->DirLayoutHeadingLength)))
      DirEnd (RequestPtr);
}

/*****************************************************************************/
/*
Begin a listing with directories if not expressly forbidden by configuration 
or local directory-specific configuration files.  If required check if there 
is an accessable parent directory and if so generate a formatted entry for it. 
*/ 
 
DirBeginDirectories (struct RequestStruct *RequestPtr)

{
   register char  *cptr, *sptr;

   boolean  ParentDirectory,
            SubDirectories;
   int  status;
   char  ParentDirPath [256],
         ParentDirVms [256];
   void  *AstFunctionPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DirBeginDirectories()\n");

   /*
      Determine if any parent directory or subdirectories will be
      displayed by checking for the presence of directory listing
      control files (and a kludge for user directory mapping).
   */
   ParentDirectory = SubDirectories = true;
   cptr = RequestPtr->DirDirectoryPart;
   if (DirFileExists (RequestPtr, cptr, DirNopsFileName))
      ParentDirectory = SubDirectories = false;
   else
   {
      if (DirFileExists (RequestPtr, cptr, DirNosFileName))
         SubDirectories = false;

      if (DirFileExists (RequestPtr, cptr, DirNopFileName))
         ParentDirectory = false;
      else
      if (*(unsigned short*)(cptr = RequestPtr->PathInfoPtr) == 0x7e2f)
      {
         /*
            This is a kludge to better support user directories ("/~username/").
            It prevents a "top-level" user directory from displaying a parent
            directory even if it is not a "top-level" VMS directory.  It does
            this by looking to see if there is any subdirectory to the "top-
            level" user directory (e.g. "/~username/subdirectory/").
         */

         /* step over the leading "/~" */
         cptr += 2;
         /* scan to the end of the username */
         while (*cptr && *cptr != '/') cptr++;
         if (*cptr)
         {
            /* check to see if there is a directory beyond the username */
            cptr++;
            while (*cptr && *cptr != '/') cptr++;
         }
         if (!*cptr) ParentDirectory = false;
      }
   }

   if (SubDirectories)
   {
      /* next function to be executed will be the search for directories */
      AstFunctionPtr = &DirSearchDirectories;

      /* set up ready to search for all files ending in ".DIR" (directories) */
      RequestPtr->SearchFab = cc$rms_fab;
      /* set the FAB user context to the client thread pointer */
      RequestPtr->SearchFab.fab$l_ctx = RequestPtr;
      RequestPtr->SearchFab.fab$l_dna = "*.DIR;";
      RequestPtr->SearchFab.fab$b_dns = 6;
      RequestPtr->SearchFab.fab$l_fna = RequestPtr->DirDirectoryPart;
      RequestPtr->SearchFab.fab$b_fns = strlen(RequestPtr->DirDirectoryPart);
      RequestPtr->SearchFab.fab$l_fop = FAB$M_NAM;
      RequestPtr->SearchFab.fab$l_nam = &RequestPtr->SearchNam;
      RequestPtr->SearchNam = cc$rms_nam;
      RequestPtr->SearchNam.nam$l_esa = RequestPtr->ExpandedDirSpec;
      RequestPtr->SearchNam.nam$b_ess = sizeof(RequestPtr->ExpandedDirSpec)-1;
      RequestPtr->SearchNam.nam$l_rsa = RequestPtr->DirSpec;
      RequestPtr->SearchNam.nam$b_rss = sizeof(RequestPtr->DirSpec)-1;

      if (VMSnok (status = sys$parse (&RequestPtr->SearchFab, 0, 0)))
      {
         if (Debug) fprintf (stdout, "sys$parse() %%X%08.08X\n", status);
         RequestPtr->ErrorHiddenTextPtr = RequestPtr->DirDirectoryPart;
         ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
         DirEnd (RequestPtr);
         return;
      }
   }
   else
   {
      /* next function to be executed will be to start the search for files */
      AstFunctionPtr = &DirBeginFiles;
   }

   if (ParentDirectory)
   {
      cptr = RequestPtr->DirDirectoryPart;
      sptr = ParentDirVms;
      while (*cptr && *cptr != '[') *sptr++ = *cptr++;
      /* these two longwords represent the string "[000000]" */
      if (*(unsigned long*)cptr == 0x3030305b &&
          *(unsigned long*)(cptr+4) == 0x5d303030)
      {
         /* in Master-File-Directory, therefore no parent */
         ParentDirectory = false;
      }
      else
      {
         /*
            Not in a Master-File-Directory, create a parent directory
            (e.g. "DEVICE:[DIR1.DIR2]" from "DEVICE:[DIR1.DIR2.DIR3]",
            and "DEVICE:[000000]" from "DEVICE:[DIR1]", etc.)
         */
         /* these two longwords represent the string "[000000." */
         if (*(unsigned long*)cptr == 0x3030305b &&
             *(unsigned long*)(cptr+4) == 0x2e303030)
         {
            /* skip over the Master-File-Directory in the specification */
            *sptr++ = *cptr++;
            cptr += 7;
         }
         while (*cptr) *sptr++ = *cptr++;
         *sptr = '\0';
         while (*sptr != ']') sptr--;
         while (*sptr != '.' && *sptr != '[') sptr--;
         if (*sptr == '.')
         {
            *sptr++ = ']';
            *sptr = '\0';
         }
         else
            memcpy (sptr, "[000000]", 9);
         if (Debug) fprintf (stdout, "ParentDirVms |%s|\n", ParentDirVms);

         /*
            If access is allowed display the details.
            Otherwise, its as if the directory just didn't exist!
         */
         if (DirFileExists (RequestPtr, ParentDirVms, DirHiddenFileName) ||
             (Config.DirAccessSelective &&
              !DirFileExists (RequestPtr, ParentDirVms, DirBrowsableFileName)))
            ParentDirectory = false;
      }

      if (ParentDirectory)
      {
         ParentDirPath[0] = '\0';
         sptr = MapUrl (ParentDirPath, ParentDirVms, NULL, NULL);
         if (sptr[0])
         {
            DirFormat (RequestPtr, AstFunctionPtr, ParentDirPath, false);
            return;
         }
         /*
            Mapping error, most probably access forbidden, we'll assume that.
            This can happen where a subdirectory is mapable but the parent is
            not (e.g. "pass /parent/subd/* /device/parent/subd/*"), making it
            forbidden to request "/parent/file" or "/parent/*.*".
            Hence, for directory listings, it's as if parent does not exist!
         */
         ParentDirectory = false;
      }
   }

   if (!ParentDirectory)
   {
      /* parent directory was not output, explicitly drive the search */
      if (VMSnok (status = sys$dclast (AstFunctionPtr, RequestPtr, 0)))
      {
         ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
         DirEnd (RequestPtr);
         return;
      }
   }
}

/*****************************************************************************/
/*
AST function to invoke another sys$search() call when listing directories.
*/ 

DirSearchDirectories (struct RequestStruct *RequestPtr)

{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DirSearchDirectories()\n");

   /* call RMS directory search routine, AST completion to DirDirectories() */
   status = sys$search (&RequestPtr->SearchFab,
                        &DirDirectories, &DirDirectories);
   if (Debug) fprintf (stdout, "sys$search() %%X%08.08X\n", status);
}

/*****************************************************************************/
/*
AST completion routine called each time sys$search() completes.  It will 
either point to another file name found or have "no more files found" status 
(or an error!).  If a file (directory) has been found check if its a "hidden" 
directory and if not format its details for the listing.
*/ 

DirDirectories (struct FAB *FabPtr)

{
   register char  *cptr, *sptr;
   register struct RequestStruct  *RequestPtr;

   int  status;
   char  LocalDirectory [256];

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout,
      "DirDirectories() sts: %%X%08.08X stv: %%X%08.08X\n",
      FabPtr->fab$l_sts, FabPtr->fab$l_stv);

   /* retrieve the pointer to the client thread from the FAB user context */
   RequestPtr = FabPtr->fab$l_ctx;

   if (VMSnok (status = RequestPtr->SearchFab.fab$l_sts))
   {
      /* if its a search list treat directory not found as if file not found */
      if ((RequestPtr->FileNam.nam$l_fnb & NAM$M_SEARCH_LIST) &&
          status == RMS$_DNF)
         status = RMS$_FNF;

      if (status == RMS$_FNF || status == RMS$_NMF)
      {
         /***************************/
         /* end of directory search */
         /***************************/

         /* now lets turn our attention to listing the non-directory files */
         DirBeginFiles (RequestPtr);
         return;
      }

      /**********************/
      /* sys$search() error */
      /**********************/

      RequestPtr->ErrorHiddenTextPtr = RequestPtr->SearchNam.nam$l_dev;
      ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
      DirEnd (RequestPtr);
      return;
   }

   /* terminate following the last character in the version number */
   RequestPtr->SearchNam.nam$l_ver[RequestPtr->SearchNam.nam$b_ver] = '\0';
   if (Debug) fprintf (stdout, "Directory |%s|\n", RequestPtr->DirSpec);

   /************************/
   /* directory access ok? */
   /************************/

   /*
      Create a directory specification from a directory file name.
      (e.g. "DEVICE:[DIR1.DIR2]" from "DEVICE:[DIR1]DIR2.DIR")
   */
   sptr = LocalDirectory;
   cptr = RequestPtr->SearchNam.nam$l_dev;
   *RequestPtr->SearchNam.nam$l_type = '\0';
   while (*cptr) *sptr++ = *cptr++;
   *RequestPtr->SearchNam.nam$l_type = '.';
   *sptr++ = ']';
   *sptr = '\0';
   sptr -= 2;
   while (*sptr != ']') sptr--;
   *sptr = '.';

   /*
      If access is allowed display the details.
      Otherwise, its as if the directory just didn't exist!
   */
   if (!DirFileExists (RequestPtr, LocalDirectory, DirHiddenFileName) &&
       !Config.DirAccessSelective ||
       (Config.DirAccessSelective &&
        DirFileExists (RequestPtr, LocalDirectory, DirBrowsableFileName)))
   {
      if (!RequestPtr->DirDirectoryCount)
      {
      }

      RequestPtr->InternalDescription[0] = '\0';
      ConfigContentType (RequestPtr, RequestPtr->SearchNam.nam$l_type);
      DirFormat (RequestPtr, &DirSearchDirectories, "", false);
      return;
   }

   /* just look for the next directory */
   DirSearchDirectories (RequestPtr);
}

/*****************************************************************************/
/*
Initiate the listing of non-directory files.
*/ 

DirBeginFiles (struct RequestStruct *RequestPtr)

{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DirBeginFiles()\n");

   RequestPtr->SearchFab = cc$rms_fab;
   /* set the FAB user context to the client thread pointer */
   RequestPtr->SearchFab.fab$l_ctx = RequestPtr;
   RequestPtr->SearchFab.fab$l_dna = RequestPtr->DirDirectoryPart;
   RequestPtr->SearchFab.fab$b_dns = strlen(RequestPtr->DirDirectoryPart);
   RequestPtr->SearchFab.fab$l_fna = RequestPtr->DirFilePart;
   RequestPtr->SearchFab.fab$b_fns = strlen(RequestPtr->DirFilePart);
   RequestPtr->SearchFab.fab$l_fop = FAB$M_NAM;
   RequestPtr->SearchFab.fab$l_nam = &RequestPtr->SearchNam;
   RequestPtr->SearchNam = cc$rms_nam;
   RequestPtr->SearchNam.nam$l_esa = RequestPtr->ExpandedDirSpec;
   RequestPtr->SearchNam.nam$b_ess = sizeof(RequestPtr->ExpandedDirSpec)-1;
   RequestPtr->SearchNam.nam$l_rsa = RequestPtr->DirSpec;
   RequestPtr->SearchNam.nam$b_rss = sizeof(RequestPtr->DirSpec)-1;

   if (VMSnok (status = sys$parse (&RequestPtr->SearchFab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$parse() %%X%08.08X\n", status);
      RequestPtr->ErrorHiddenTextPtr = RequestPtr->DirDirectoryPart;
      ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
      DirEnd (RequestPtr);
      return;
   }

   /***************/
   /* file search */
   /***************/

   /* call RMS directory search routine, AST completion to DirFiles() */
   sys$search (&RequestPtr->SearchFab, &DirFiles, &DirFiles);
}

/*****************************************************************************/
/*
AST function to invoke another sys$search() call when listing files.
*/ 

DirSearchFiles (struct RequestStruct *RequestPtr)

{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DirSearchFiles()\n");

   /* call RMS directory search routine, AST completion to DirFiles() */
   status = sys$search (&RequestPtr->SearchFab, &DirFiles, &DirFiles);
   if (Debug) fprintf (stdout, "sys$search() %%X%08.08X\n", status);
}

/*****************************************************************************/
/*
This is an AST completion routine called each time sys$search() completes.  It 
will either point to another file name found or have "no more files found" 
status (or an error!).  If a file call a function to determine if the file 
will contain a "description".  When that function returns just return to the 
AST interrupted routine.
*/ 

DirFiles (struct FAB *FabPtr)

{
   register struct RequestStruct  *RequestPtr;

   int  status;
   void *AstFunctionPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout,
               "DirFiles() sts: %%X%08.08X stv: %%X%08.08X\n",
               FabPtr->fab$l_sts,
               FabPtr->fab$l_stv);

   /* retrieve the pointer to the client thread from the FAB user context */
   RequestPtr = FabPtr->fab$l_ctx;

   if (VMSnok (status = RequestPtr->SearchFab.fab$l_sts))
   {
      /* if its a search list treat directory not found as if file not found */
      if ((RequestPtr->FileNam.nam$l_fnb & NAM$M_SEARCH_LIST) &&
          status == RMS$_DNF)
         status = RMS$_FNF;

      if (status == RMS$_FNF || status == RMS$_NMF)
      {
         /**********************/
         /* end of file search */
         /**********************/

         if (RequestPtr->DirFormatVms)
            DirFormat (RequestPtr, &DirEndOutput, "", true);
         else
            DirEndOutput (RequestPtr);

         return;
      }

      /**********************/
      /* sys$search() error */
      /**********************/

      RequestPtr->ErrorHiddenTextPtr = RequestPtr->SearchNam.nam$l_dev;
      ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
      DirEnd (RequestPtr);
      return;
   }

   /* terminate following the last character in the version number */
   RequestPtr->SearchNam.nam$l_ver[RequestPtr->SearchNam.nam$b_ver] = '\0';

   /* we've already output the directories, ignore them (number = "DIR;") */
   if (*(unsigned long*)(RequestPtr->SearchNam.nam$l_type+1) == 0x3b524944)
   {
      sys$search (&RequestPtr->SearchFab, &DirFiles, &DirFiles);
      return;
   }

   if (Debug) fprintf (stdout, "File |%s|\n", RequestPtr->SearchNam.nam$l_dev);

   /* in true Unix-style "hidden" files do not appear (those beginning ".") */
   if (*(char*)RequestPtr->SearchNam.nam$l_name == '.')
   {
      sys$search (&RequestPtr->SearchFab, &DirFiles, &DirFiles);
      return;
   }

   ConfigContentType (RequestPtr, RequestPtr->SearchNam.nam$l_type);

   DirDescription (RequestPtr);
}

/*****************************************************************************/
/*
Directories have been listed, files have been listed.  If appropriate provide 
a readme at the bottom, then conclude the HTML and the directory listing.
*/ 

boolean DirEndOutput (struct RequestStruct *RequestPtr)

{
   int  status;
   void *AstFunctionPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DirEndOutput()\n");

   if (Config.DirReadMeBottom)
      AstFunctionPtr = &DirReadMeBottom;
   else
      AstFunctionPtr = &DirEnd;
 
   if (RequestPtr->DirDirectoryCount || RequestPtr->DirFileCount)
   {
      if (VMSnok (BufferOutput (RequestPtr, AstFunctionPtr,
                                "</PRE>\n", 7)))
         DirEnd (RequestPtr);
   }
   else
   {
      if (VMSnok (BufferOutput (RequestPtr, AstFunctionPtr, "</PRE>\n", 7)))
         DirEnd (RequestPtr);
   }
}

/*****************************************************************************/
/*
Using the directive string pointed to by 'RequestPtr->DirLayoutPtr' format the 
directory listing column headings and sys$fao() directive string used to format
the information for each directory/file.  Return normal or error status.  This
function buffers the details of the default generic and VMS directory layout
the first time each is required.  There-after the buffered layout information
is used, reducing overhead.  Only if a user-supplied layout is required does
any format processing occur again.
*/ 
 
int DirFormatLayout (struct RequestStruct *RequestPtr)

{
#  define LayoutCapacity 256

   static int  FieldWidthCdt,
               FieldWidthName,
               FieldWidthOwner,
               FieldWidthRdt,
               FieldWidthSize,
               LayoutFaoLength,
               LayoutHeadingLength,
               VmsFieldWidthCdt,
               VmsFieldWidthName,
               VmsFieldWidthOwner,
               VmsFieldWidthRdt,
               VmsFieldWidthSize,
               VmsLayoutFaoLength,
               VmsLayoutHeadingLength;
   static char  LayoutFao [LayoutCapacity/2],
                LayoutHeading [LayoutCapacity],
                VmsLayoutFao [LayoutCapacity/2],
                VmsLayoutHeading [LayoutCapacity];

   register int  cnt;
   register char  *cptr, *lptr, *sptr, *wptr, *zptr;

   int  size;
   char  String [LayoutCapacity];
   char  *LayoutFaoPtr,
         *LayoutHeadingPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DirFormatLayout()\n");

   if (RequestPtr->DirLayoutPtr == NULL)
   {
      /******************/
      /* default layout */
      /******************/

      if (Config.DefaultDirLayout[0])
         RequestPtr->DirLayoutPtr = Config.DefaultDirLayout;
      else
         RequestPtr->DirLayoutPtr = DefaultDirListingLayout;

      if (!RequestPtr->DirFormatVms && LayoutFaoLength)
      {
         /* default generic heading and fao have already been generated */
         RequestPtr->DirLayoutFaoPtr = LayoutFao;
         RequestPtr->DirLayoutFaoLength = LayoutFaoLength;
         RequestPtr->DirLayoutHeadingPtr = LayoutHeading;
         RequestPtr->DirLayoutHeadingLength = LayoutHeadingLength;
         RequestPtr->DirFieldWidthCdt = FieldWidthCdt;
         RequestPtr->DirFieldWidthName = FieldWidthName;
         RequestPtr->DirFieldWidthOwner = FieldWidthOwner;
         RequestPtr->DirFieldWidthRdt = FieldWidthRdt;
         RequestPtr->DirFieldWidthSize = FieldWidthSize;

         if (!Debug) return (SS$_NORMAL);

         fprintf (stdout,"|%d|%s|\n|%d|%s|\n",
                  LayoutHeadingLength, LayoutHeading,
                  LayoutFaoLength, LayoutFao);
         return (SS$_NORMAL);
      }

      if (RequestPtr->DirFormatVms && VmsLayoutFaoLength)
      {
         /* default VMS heading and fao have already been generated */
         RequestPtr->DirLayoutFaoPtr = VmsLayoutFao;
         RequestPtr->DirLayoutFaoLength = VmsLayoutFaoLength;
         RequestPtr->DirLayoutHeadingPtr = VmsLayoutHeading;
         RequestPtr->DirLayoutHeadingLength = VmsLayoutHeadingLength;
         RequestPtr->DirFieldWidthCdt = VmsFieldWidthCdt;
         RequestPtr->DirFieldWidthName = VmsFieldWidthName;
         RequestPtr->DirFieldWidthOwner = VmsFieldWidthOwner;
         RequestPtr->DirFieldWidthRdt = VmsFieldWidthRdt;
         RequestPtr->DirFieldWidthSize = VmsFieldWidthSize;

         if (!Debug) return (SS$_NORMAL);

         fprintf (stdout,"|%d|%s|\n|%d|%s|\n",
                  VmsLayoutHeadingLength, VmsLayoutHeading,
                  VmsLayoutFaoLength, VmsLayoutFao);
         return (SS$_NORMAL);
      }

      /* default headings and layouts have not been generated ... do it! */
      if (RequestPtr->DirFormatVms)
      {
         /* use the function-internal VMS buffers */
         LayoutHeadingPtr = VmsLayoutHeading;
         LayoutFaoPtr = VmsLayoutFao;
      }
      else
      {
         /* use the function-internal generic buffers */
         LayoutHeadingPtr = LayoutHeading;
         LayoutFaoPtr = LayoutFao;
      }
   }
   else
   {
      /**********************/
      /* non-default layout */
      /**********************/

      /* use scratch space to generate the non-default heading and fao */
      LayoutHeadingPtr = LayoutFaoPtr = String;
   }

   /******************************/
   /* first, the column headings */
   /******************************/

   zptr = (sptr = LayoutHeadingPtr) + LayoutCapacity;
   lptr = RequestPtr->DirLayoutPtr;
   wptr = "";

   cptr = "<PRE>";
   while (*cptr && sptr < zptr) *sptr++ = *cptr++;

   while (*lptr && *lptr != '&' && sptr < zptr)
   {
      /** if (Debug) fprintf (stdout, "lptr |%s|\n", lptr); **/

      if (isdigit(*lptr))
      {
         wptr = lptr;
         while (*lptr && isdigit(*lptr)) lptr++;
         continue;
      }

      if (wptr[0])
      {
         cnt = atoi(wptr);
         if (cnt < 0) cnt = 0;
         wptr = "";
      }
      else
         cnt = 0;

      switch (toupper(*lptr))
      {
         case ' ' :
         case '_' :
            if (!cnt) cnt = DefaultFieldWidthInterField;
            break;

         case 'C' :
            cptr = "Created";
            if (!cnt) cnt = DefaultFieldWidthCdt;
            if (cnt < (size = 7)) cnt = size;
            if (RequestPtr->DirFormatVms) cnt += 5;
            RequestPtr->DirFieldWidthCdt = cnt;
            break;

         case 'D' :
            /* no flush either way, must always be the last field */
            cptr = "Description";
            break;

         case 'I' :  /* icon */
            break;

         case 'L' :
         case 'N' :
            cptr = "Name";
            if (!cnt) cnt = DefaultFieldWidthName;
            if (cnt < (size = 4)) cnt = size;
            RequestPtr->DirFieldWidthName = cnt;
            break;

         case 'O' :
            cptr = "Owner";
            if (!cnt) cnt = DefaultFieldWidthOwner;
            if (cnt < (size = 5)) cnt = size;
            RequestPtr->DirFieldWidthOwner = cnt;
            break;

         case 'R' :
            cptr = "Revised";
            if (!cnt) cnt = DefaultFieldWidthRdt;
            if (cnt < (size = 7)) cnt = size;
            if (RequestPtr->DirFormatVms) cnt += 5;
            RequestPtr->DirFieldWidthRdt = cnt;
            break;

         case 'S' :
            cptr = "Size";
            if (!cnt) cnt = DefaultFieldWidthSize;
            if (cnt < (size = 4)) cnt = size;
            /* expand size field width by 7 for blocks allocated/used */
            if (RequestPtr->DirFormatVms) cnt += 7;
            RequestPtr->DirFieldWidthSize = cnt;
            break;

         default :
            /* unknown layout directive character */
            RequestPtr->ResponseStatusCode = 501;
            ErrorGeneral (RequestPtr, DirLayoutProblem, __FILE__, __LINE__);
            return (STS$K_ERROR);
      }

      switch (toupper(*lptr))
      {
         case ' ' :
         case '_' :
            while (cnt-- && sptr < zptr) *sptr++ = ' ';
            break;

         case 'C' :
         case 'R' :
         case 'S' :
            /* flush the column heading right using spaces */
            cnt -= size;
            while (cnt-- && sptr < zptr) *sptr++ = ' ';
            while (*cptr && sptr < zptr) *sptr++ = *cptr++;
            break;

         case 'L' :
         case 'N' :
         case 'O' :
            /* flush the column heading left using spaces */
            cnt -= size;
            while (*cptr && sptr < zptr) *sptr++ = *cptr++;
            while (cnt-- && sptr < zptr) *sptr++ = ' ';
            break;

         case 'D' :
            /* no flush either way, must always be the last field */
            while (*cptr && sptr < zptr) *sptr++ = *cptr++;
            break;

         case 'I' :
            cptr = ConfigIconfor (RequestPtr, DirBlankContentTypeIcon);
            while (*cptr && sptr < zptr) *sptr++ = *cptr++;
            break;
      }
      lptr++;
   }

   cptr = "\n<HR>\n";
   while (*cptr && sptr < zptr) *sptr++ = *cptr++;

   if (sptr >= zptr)
   {
      /* should be enough for all but a deliberate formatting hack! */
      RequestPtr->ResponseStatusCode = 501;
      ErrorGeneral (RequestPtr, DirLayoutProblem, __FILE__, __LINE__);
      return (STS$K_ERROR);
   }
   *sptr = '\0';
   cnt = sptr - LayoutHeadingPtr;

   if (Debug)
      fprintf (stdout, "LayoutHeadingPtr |%d|%s|\n", cnt, LayoutHeadingPtr);

   if (LayoutHeadingPtr == LayoutHeading)
   {
      /* first time a generic layout has been required, buffer details */
      RequestPtr->DirLayoutHeadingPtr = LayoutHeading;
      RequestPtr->DirLayoutHeadingLength = LayoutHeadingLength = cnt;
      FieldWidthCdt = RequestPtr->DirFieldWidthCdt;
      FieldWidthName = RequestPtr->DirFieldWidthName;
      FieldWidthOwner = RequestPtr->DirFieldWidthOwner;
      FieldWidthRdt = RequestPtr->DirFieldWidthRdt;
      FieldWidthSize = RequestPtr->DirFieldWidthSize;
   }
   else
   if (LayoutHeadingPtr == VmsLayoutHeading)
   {
      /* first time a VMS layout has been required, buffer details */
      RequestPtr->DirLayoutHeadingPtr = VmsLayoutHeading;
      RequestPtr->DirLayoutHeadingLength = VmsLayoutHeadingLength = cnt;
      VmsFieldWidthCdt = RequestPtr->DirFieldWidthCdt;
      VmsFieldWidthName = RequestPtr->DirFieldWidthName;
      VmsFieldWidthOwner = RequestPtr->DirFieldWidthOwner;
      VmsFieldWidthRdt = RequestPtr->DirFieldWidthRdt;
      VmsFieldWidthSize = RequestPtr->DirFieldWidthSize;
   }
   else
   {
      /* this is a user-supplied layout, place it in the thread's heap */
      if ((RequestPtr->DirLayoutHeadingPtr = lptr =
          HeapAlloc (RequestPtr, cnt+1)) == NULL)
      {
         ErrorHeapAlloc (RequestPtr, __FILE__, __LINE__);
         return (SS$_INSFMEM);
      }
      memcpy (lptr, LayoutHeadingPtr, cnt+1);
      RequestPtr->DirLayoutHeadingLength = cnt;
   }

   /******************************************/
   /* second, the sys$fao() directive string */
   /******************************************/

   zptr = (sptr = LayoutFaoPtr) + LayoutCapacity / 2;
   lptr = RequestPtr->DirLayoutPtr;
   wptr = "";

   /* accomodate an optional leading new-line (for creating a blank line) */
   cptr = "!AZ";
   while (*cptr && sptr < zptr) *sptr++ = *cptr++;

   while (*lptr && *lptr != '&' && sptr < zptr)
   {
      /** if (Debug) fprintf (stdout, "lptr |%s|\n", lptr); **/

      if (isdigit(*lptr))
      {
         wptr = lptr;
         while (*lptr && isdigit(*lptr)) lptr++;
         continue;
      }

      if (wptr[0])
      {
         cnt = atoi(wptr);
         if (cnt < 0) cnt = 0;
         wptr = "";
      }
      else
         cnt = 0;

      switch (toupper(*lptr))
      {
         case ' ' :
         case '_' :
            if (!cnt) cnt = DefaultFieldWidthInterField;
            while (cnt-- && sptr < zptr) *sptr++ = ' ';
            break;

         case 'C' :
         case 'R' :
         case 'S' :
            /* right justify, then value */
            cptr = "!#<!#AZ!AZ!>";
            while (*cptr && sptr < zptr) *sptr++ = *cptr++;
            break;

         case 'O' :
            /* '!#!AZ' will left justify value with spaces */
            cptr = "!#AZ";
            while (*cptr && sptr < zptr) *sptr++ = *cptr++;
            break;

         case 'L' :
            /* left justify, then link, name, and name overflow */
            cptr = "!AZ!AZ!AZ!#<!AZ!AZ!AZ!>";
            while (*cptr && sptr < zptr) *sptr++ = *cptr++;
            break;

         case 'D' :
         case 'I' :
            /* nothing fancy here */
            cptr = "!AZ";
            while (*cptr && sptr < zptr) *sptr++ = *cptr++;
            break;

         case 'N' :
            /* left justify, string, plus overflow */
            cptr = "!#<!AZ!AZ!>";
            while (*cptr && sptr < zptr) *sptr++ = *cptr++;
            break;
      }
      lptr++;
   }
   if (sptr < zptr) *sptr++ = '\n';

   if (sptr >= zptr)
   {
      /* should be enough for all but a deliberate formatting hack! */
      RequestPtr->ResponseStatusCode = 501;
      ErrorGeneral (RequestPtr, DirLayoutProblem, __FILE__, __LINE__);
      return (STS$K_ERROR);
   }
   *sptr = '\0';
   cnt = sptr - LayoutFaoPtr;

   if (Debug) fprintf (stdout, "LayoutFaoPtr |%d|%s|\n", cnt, LayoutFaoPtr);

   if (LayoutFaoPtr == LayoutFao)
   {
      /* first time a generic layout has been required, buffer details */
      RequestPtr->DirLayoutFaoPtr = LayoutFao;
      RequestPtr->DirLayoutFaoLength = LayoutFaoLength = cnt;
   }
   else
   if (LayoutFaoPtr == VmsLayoutFao)
   {
      /* first time a VMS layout has been required, buffer details */
      RequestPtr->DirLayoutFaoPtr = VmsLayoutFao;
      RequestPtr->DirLayoutFaoLength = VmsLayoutFaoLength = cnt;
   }
   else
   {
      /* this is a user-supplied layout, place it in the thread's heap */
      if ((RequestPtr->DirLayoutFaoPtr = lptr =
          HeapAlloc (RequestPtr, cnt+1)) == NULL)
      {
         ErrorHeapAlloc (RequestPtr, __FILE__, __LINE__);
         return (SS$_INSFMEM);
      }
      memcpy (lptr, LayoutFaoPtr, cnt+1);
      RequestPtr->DirLayoutFaoLength = cnt;
   }

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
This overly-long (sorry) function formats and outputs a line of information 
for a parent directory, directory file, non-directory file, or for the final 
line giving used/allocated totals for a VMS format listing. 

For directory and ordinary files retrieve and display the specified file's 
statistics (name, size, modification  time, etc.)  This function completes 
synchronously, that is it waits for the single QIO on the file details to 
complete before assembling the details for the listing, because of this the 
function introduces slight granularity.  It is a candidate for future 
improvement. 

This function uses the ACP-QIO interface detailed in the "OpenVMS I/O User's 
Reference Manual".
*/ 

int DirFormat
(
struct RequestStruct *RequestPtr,
void *AstFunctionPtr,
char *ParentDirPath,
boolean BlockTotals
)
{
   static char  *MonthName [] =
      { "", "Jan", "Feb", "Mar", "Apr", "May", "Jun",
            "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };

   static $DESCRIPTOR (BlocksFaoDsc, "!UL/!UL");
   static $DESCRIPTOR (BytesFaoDsc, "!UL");
   static $DESCRIPTOR (CdtDsc, "");
   static $DESCRIPTOR (DateTimeFaoDsc, "!2ZW-!AZ-!2ZW !2ZW:!2ZW:!2ZW");
   static $DESCRIPTOR (DeviceDsc, "");
   static $DESCRIPTOR (DirLayoutFaoDsc, "");
   static $DESCRIPTOR (KBytesFaoDsc, "!ULK");
   static $DESCRIPTOR (MBytesFaoDsc, "!ULM");
   static $DESCRIPTOR (OwnerFaoDsc, "!%I");
   static $DESCRIPTOR (OwnerDsc, "");
   static $DESCRIPTOR (RdtDsc, "");
   static $DESCRIPTOR (SizeDsc, "");
   static $DESCRIPTOR (SizeFaoDsc, "!#* !AZ");
   static $DESCRIPTOR (StringDsc, "");
   static $DESCRIPTOR (TotalFilesDsc, "");
   static $DESCRIPTOR (TotalFilesFaoDsc, "!UL files");
   static $DESCRIPTOR (VmsDateTimeFaoDsc, "!%D");

   register int  idx, MaxIdx;
   register char  *cptr, *sptr, *zptr;

   boolean  ThisIsADirectory;
   int  status;
   unsigned short  AcpChannel,
                   Length;
   unsigned short  NumTime [7];
   unsigned long  Bytes,
                  AllocatedVbn,
                  EndOfFileVbn;
   unsigned long  FaoVector [64];
   char  *IconPtr,
         *AnchorNameOverflowPtr;
   char  AnchorLink [1024],
         AnchorName [128],
         Cdt [64],
         Description [256],
         Owner [128],
         Rdt [64],
         Scratch [256],
         Size [32],
         String [2048],
         TotalFiles [32];

   unsigned long  AtrUic;
   unsigned long  AtrCdt [2],
                  AtrRdt [2];
   struct {
      unsigned long  OfNoInterest1;
      unsigned short  AllocatedVbnHi;
      unsigned short  AllocatedVbnLo;
      unsigned short  EndOfFileVbnHi;
      unsigned short  EndOfFileVbnLo;
      unsigned short  FirstFreeByte;
      unsigned short  OfNoInterest2;
      unsigned long  OfNoInterestLots [4];
   } AtrRecord;

   struct fibdef  SearchFib;
   struct atrdef  SearchAtr [] =
   {
      { sizeof(AtrRecord), ATR$C_RECATTR, &AtrRecord },
      { sizeof(AtrCdt), ATR$C_CREDATE, &AtrCdt },
      { sizeof(AtrRdt), ATR$C_REVDATE, &AtrRdt },
      { sizeof(AtrUic), ATR$C_UIC, &AtrUic },
      { 0, 0, 0 }
   };

   struct {
      unsigned short  Length;
      unsigned short  Unused;
      unsigned long  Address;
   } FileNameAcpDsc,
     SearchFibAcpDsc,
     SearchAtrAcpDsc;

   struct {
      unsigned short  Status;
      unsigned short  Unused1;
      unsigned long  Unused2;
   } AcpIOsb;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "DirFormat() |%s|\n", RequestPtr->SearchNam.nam$l_dev);

   if (!(ParentDirPath[0] || BlockTotals))
   {
      /*******************************************/
      /* queue an ACP I/O to get file attributes */
      /*******************************************/

      /* assign a channel to the disk device containing the file */
      DeviceDsc.dsc$w_length = RequestPtr->SearchNam.nam$b_dev;
      DeviceDsc.dsc$a_pointer = RequestPtr->SearchNam.nam$l_dev;
      status = sys$assign (&DeviceDsc, &AcpChannel, 0, 0, 0);
      if (Debug) fprintf (stdout, "sys$assign() %%X%08.08X\n", status);
      if (VMSnok (status))
      {
         RequestPtr->ErrorHiddenTextPtr = RequestPtr->SearchNam.nam$l_dev;
         ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
         DirEnd (RequestPtr);
         return;
      }

      /* set up the File Information Block for the ACP interface */
      memset (&SearchFib, 0, sizeof(struct fibdef));
      SearchFibAcpDsc.Length = sizeof(SearchFib);
      SearchFibAcpDsc.Address = &SearchFib;
      /* quick work around, different syntax for this structure with DEC C? */
#ifdef __DECC
      SearchFib.fib$w_did[0] = RequestPtr->SearchNam.nam$w_did[0];
      SearchFib.fib$w_did[1] = RequestPtr->SearchNam.nam$w_did[1];
      SearchFib.fib$w_did[2] = RequestPtr->SearchNam.nam$w_did[2];
#else
      SearchFib.fib$r_did_overlay.fib$w_did[0] =
         RequestPtr->SearchNam.nam$w_did[0];
      SearchFib.fib$r_did_overlay.fib$w_did[1] =
         RequestPtr->SearchNam.nam$w_did[1];
      SearchFib.fib$r_did_overlay.fib$w_did[2] =
         RequestPtr->SearchNam.nam$w_did[2];
#endif

      FileNameAcpDsc.Length =
         RequestPtr->SearchNam.nam$b_name +
         RequestPtr->SearchNam.nam$b_type +
         RequestPtr->SearchNam.nam$b_ver;
      FileNameAcpDsc.Address = RequestPtr->SearchNam.nam$l_name;

      status = sys$qiow (0, AcpChannel, IO$_ACCESS, &AcpIOsb, 0, 0, 
                         &SearchFibAcpDsc, &FileNameAcpDsc, 0, 0,
                         &SearchAtr, 0);

      /* immediately deassign the channel in case we return on an error */
      sys$dassgn (AcpChannel);

      if (Debug)
         fprintf (stdout, "sys$qio() %%X%08.08X IOsb: %%X%08.08X\n",
                 status, AcpIOsb.Status);

      if (VMSok (status)) status = AcpIOsb.Status;
      if (VMSnok (status)) 
      {
         RequestPtr->ErrorHiddenTextPtr = RequestPtr->SearchNam.nam$l_dev;
         ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
         DirEnd (RequestPtr);
         return;
      }

      /* test for numeric equivalent of "DIR;" */
      if (ThisIsADirectory =
          (*(unsigned long*)(RequestPtr->SearchNam.nam$l_type+1) == 0x3b524944))
      {
         /* no need to display the MFD */
         if (RequestPtr->SearchNam.nam$l_name[0] == '0' &&
             strsame (RequestPtr->SearchNam.nam$l_name, "000000.DIR;", 11))
         {
            /* can't return without queueing the next bit of processing! */
            if (VMSnok (status = sys$dclast (AstFunctionPtr, RequestPtr, 0)))
               ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
            if (Debug) fprintf (stdout, "sys$dclast() %%X%08.08X\n", status);
            return;
         }
      }
   }

   /********************************/
   /* process file name components */
   /********************************/

   MaxIdx = sizeof(FaoVector);
   idx = 0;

   if (BlockTotals)
   {
      /****************/
      /* block totals */
      /****************/

      /* must be block totals, which requires a leading newline */
      if (idx < MaxIdx) FaoVector[idx++] = "\n";
   }
   else
   if (ParentDirPath[0])
   {
      /********************/
      /* parent directory */
      /********************/

      /* no leading newline required for the parent directory */
      if (idx < MaxIdx) FaoVector[idx++] = "";
      RequestPtr->DirDirectoryCount = -1;

      zptr = (sptr = AnchorLink) + sizeof(AnchorLink);
      /* copy in the directory portion of the URL */
      cptr = ParentDirPath;
      while (*cptr && sptr < zptr) *sptr++ = tolower(*cptr++);
      /* append anything from the original spec */
      cptr = RequestPtr->DirFilePart;
      while (*cptr && sptr < zptr) *sptr++ = tolower(*cptr++);
      /* propagate any query string */
      if (RequestPtr->DirQueryStringPtr[0])
      {
         if (sptr < zptr) *sptr++ = '?';
         for (cptr = RequestPtr->DirQueryStringPtr;
              *cptr && sptr < zptr;
              *sptr++ = *cptr++);
      }
      if (sptr >= zptr)
      {
         RequestPtr->ResponseStatusCode = 500;
         ErrorGeneral (RequestPtr, ErrorStringSize, __FILE__, __LINE__);
         DirEnd (RequestPtr);
         return;
      }
      *sptr = '\0';
      /** if (Debug) fprintf (stdout, "AnchorLink |%s|\n", AnchorLink); **/

      if (RequestPtr->DirFormatVms)
         memcpy (AnchorName, "[-]", 4);
      else
         memcpy (AnchorName, "../", 4);
      sptr = AnchorName + 4;
      /** if (Debug) fprintf (stdout, "AnchorName |%s|\n", AnchorName); **/
   }
   else
   if (ThisIsADirectory)
   {
      /***************/
      /* a directory */
      /***************/

      /* no leading newline required for directories */
      if (idx < MaxIdx) FaoVector[idx++] = "";
      if (RequestPtr->DirDirectoryCount <= 0)
         RequestPtr->DirDirectoryCount = 1;
      else
         RequestPtr->DirDirectoryCount++;

      /* terminate to remove the file type (".DIR") */
      *RequestPtr->SearchNam.nam$l_type = '\0';

      zptr = (sptr = AnchorLink) + sizeof(AnchorLink);
      /* copy in the directory portion of the URL */
      cptr = RequestPtr->DirDirectoryPathPtr;
      while (*cptr && sptr < zptr) *sptr++ = tolower(*cptr++);
      /* copy in the directory (file) name */
      cptr = RequestPtr->SearchNam.nam$l_name;
      while (*cptr && sptr < zptr) *sptr++ = tolower(*cptr++);
      /* end directory, then append anything from the original spec */
      if (sptr < zptr) *sptr++ = '/';
      cptr = RequestPtr->DirFilePart;
      while (*cptr && sptr < zptr) *sptr++ = tolower(*cptr++);
      /* propagate any query string */
      if (RequestPtr->DirQueryStringPtr[0])
      {
         if (sptr < zptr) *sptr++ = '?';
         for (cptr = RequestPtr->DirQueryStringPtr;
              *cptr && sptr < zptr;
              *sptr++ = *cptr++);
      }
      if (sptr >= zptr)
      {
         RequestPtr->ResponseStatusCode = 500;
         ErrorGeneral (RequestPtr, ErrorStringSize, __FILE__, __LINE__);
         DirEnd (RequestPtr);
         return;
      }
      *sptr = '\0';
      /** if (Debug) fprintf (stdout, "AnchorLink |%s|\n", AnchorLink); **/

      sptr = AnchorName;
      if (RequestPtr->DirFormatVms)
      {
         *sptr++ = '[';
         *sptr++ = '.';
      }
      cptr = RequestPtr->SearchNam.nam$l_name;
      while (*cptr) *sptr++ = *cptr++;
      if (RequestPtr->DirFormatVms)
         *sptr++ = ']';
      else
         *sptr++ = '/';
      *sptr = '\0';
      /** if (Debug) fprintf (stdout, "AnchorName |%s|\n", AnchorName); **/

      /* restore the type delimitter */
      *RequestPtr->SearchNam.nam$l_type = '.';
   }
   else
   {
      /**********/
      /* a file */
      /**********/

      /* leading newline required for first file after directories */
      if (RequestPtr->DirDirectoryCount && !RequestPtr->DirFileCount)
         { if (idx < MaxIdx) FaoVector[idx++] = "\n"; }
      else
         if (idx < MaxIdx) FaoVector[idx++] = "";
      RequestPtr->DirFileCount++;

      /* if appropriate, terminate to remove the version number */
      if (!RequestPtr->DirFormatVms) *RequestPtr->SearchNam.nam$l_ver = '\0';

      /* don't need string limit checking, no user supplied component here! */
      sptr = AnchorLink;
      if (RequestPtr->DirScriptName[0])
      {
         /* prepend the (optional) script component of the path */
         cptr = RequestPtr->DirScriptName;
         while (*cptr) *sptr++ = *cptr++;
      }
      /* copy in the directory portion of the URL */
      cptr = RequestPtr->DirDirectoryPathPtr;
      while (*cptr) *sptr++ = tolower(*cptr++);
      /* copy in the file name and type */
      cptr = RequestPtr->SearchNam.nam$l_name;
      while (*cptr) *sptr++ = tolower(*cptr++);
      /* disable auto-scripting by appending a (most-recent) version number */
      if (!RequestPtr->DirFormatVms && !RequestPtr->AutoScriptEnabled)
      {
         *sptr++ = ';';
         *sptr++ = '0';
      }
      *sptr = '\0';
      /** if (Debug) fprintf (stdout, "AnchorLink |%s|\n", AnchorLink); **/

      sptr = AnchorName;
      cptr = RequestPtr->SearchNam.nam$l_name;
      while (*cptr) *sptr++ = *cptr++;
      *sptr = '\0';
      /** if (Debug) fprintf (stdout, "AnchorName |%s|\n", AnchorName); **/

      /* if appropriate, restore the version delimitter */
      if (!RequestPtr->DirFormatVms) *RequestPtr->SearchNam.nam$l_ver = ';';
   }

   if (!BlockTotals)
   {
      /********************************/
      /* check if file name overflows */
      /********************************/

      /* assumes that the end of 'AnchorName' is still pointed to by 'sptr' */
      if (sptr - AnchorName > RequestPtr->DirFieldWidthName)
      {
         AnchorNameOverflowPtr = "+";
         AnchorName[RequestPtr->DirFieldWidthName-1] = '\0';
      }
      else
         AnchorNameOverflowPtr = "";
   }

   /***********************************/
   /* build the sys$faol() paremeters */
   /***********************************/

   for (cptr = RequestPtr->DirLayoutPtr; *cptr; cptr++)
   {
      if (!isalpha(*cptr)) continue;
      /** if (Debug) fprintf (stdout, "idx: %d cptr |%s|\n", idx, cptr); **/

      switch (toupper(*cptr))
      {
         case 'C' :

            /*****************/
            /* creation date */
            /*****************/

            if (ParentDirPath[0] || BlockTotals)
            {
               /* make it an empty field */
               if (idx < MaxIdx)
                  FaoVector[idx++] = RequestPtr->DirFieldWidthCdt;
               if (idx < MaxIdx)
                  FaoVector[idx++] = RequestPtr->DirFieldWidthCdt;
               if (idx < MaxIdx) FaoVector[idx++] = "";
               if (idx < MaxIdx) FaoVector[idx++] = "";
               break;
            }

            /* format the creation date/time */
            CdtDsc.dsc$a_pointer = Cdt;
            CdtDsc.dsc$w_length = sizeof(Cdt)-1;
            if (RequestPtr->DirFormatVms)
               sys$fao (&VmsDateTimeFaoDsc, &Length, &CdtDsc, &AtrCdt);
            else
            {
               sys$numtim (&NumTime, &AtrCdt);
               sys$fao (&DateTimeFaoDsc, &Length, &CdtDsc,
                        NumTime[2], MonthName[NumTime[1]], NumTime[0]%100,
                        NumTime[3], NumTime[4], NumTime[5]);
            }
            Cdt[Length] = '\0';
            /** if (Debug) fprintf (stdout, "Cdt |%s|\n", Cdt); **/
            if (idx < MaxIdx) FaoVector[idx++] = RequestPtr->DirFieldWidthCdt;
            /* next parameter right justifies the field */
            if (RequestPtr->DirFieldWidthCdt - Length > 0)
            {
               if (idx < MaxIdx)
                  FaoVector[idx++] = RequestPtr->DirFieldWidthCdt - Length;
            }
            else
               if (idx < MaxIdx) FaoVector[idx++] = 0;
            if (idx < MaxIdx) FaoVector[idx++] = "";
            if (idx < MaxIdx) FaoVector[idx++] = Cdt;
            break;

         case 'D' :

            /***************/
            /* description */
            /***************/

            if (ParentDirPath[0])
            {
               if (idx < MaxIdx)
                  FaoVector[idx++] = DirParentDirectoryDescription;
            }
            else
            if (BlockTotals)
            {
               if (RequestPtr->DirDirectoryCount < 0)
                  RequestPtr->DirDirectoryCount = 0;
               TotalFilesDsc.dsc$a_pointer = TotalFiles;
               TotalFilesDsc.dsc$w_length = sizeof(TotalFiles)-1;
               sys$fao (&TotalFilesFaoDsc, &Length, &TotalFilesDsc,
                        RequestPtr->DirDirectoryCount +
                        RequestPtr->DirFileCount);
               TotalFiles[Length] = '\0';
               if (idx < MaxIdx) FaoVector[idx++] = TotalFiles;
            }
            else
            if (ThisIsADirectory)
            {
               if (idx < MaxIdx)
                  FaoVector[idx++] = DirSubDirectoryDescription;
            }
            else
            if (RequestPtr->InternalDescription[0])
            {
               if (idx < MaxIdx)
                  FaoVector[idx++] = RequestPtr->InternalDescription;
            }
            else
            {
               if (idx < MaxIdx)
                  FaoVector[idx++] = RequestPtr->ContentDescriptionPtr;
            }
            break;

         case 'I' :

            /********/
            /* icon */
            /********/

            if (ParentDirPath[0])
               IconPtr = ConfigIconfor (RequestPtr, DirParentContentTypeIcon);
            else
            if (BlockTotals)
               IconPtr = ConfigIconfor (RequestPtr, DirBlankContentTypeIcon);
            else
               IconPtr = ConfigIconfor (RequestPtr, RequestPtr->ContentTypePtr);
            if (!IconPtr[0])
               IconPtr = ConfigIconfor (RequestPtr, DirUnknownContentTypeIcon);
            if (idx < MaxIdx) FaoVector[idx++] = IconPtr;
            break;

         case 'L' :

            /***************/
            /* link/anchor */
            /***************/

            if (BlockTotals)
            {
               /* make it an empty field */
               if (idx < MaxIdx) FaoVector[idx++] = "";
               if (idx < MaxIdx) FaoVector[idx++] = "";
               if (idx < MaxIdx) FaoVector[idx++] = "";
               if (idx < MaxIdx)
                  FaoVector[idx++] = RequestPtr->DirFieldWidthName;
               if (idx < MaxIdx) FaoVector[idx++] = "";
               if (idx < MaxIdx) FaoVector[idx++] = "";
               if (idx < MaxIdx) FaoVector[idx++] = "";
               break;
            }

            if (idx < MaxIdx) FaoVector[idx++] = "<A HREF=\"";
            if (idx < MaxIdx) FaoVector[idx++] = AnchorLink;
            if (idx < MaxIdx) FaoVector[idx++] = "\">";
            if (idx < MaxIdx)
               FaoVector[idx++] = RequestPtr->DirFieldWidthName +
                                  sizeof("</A>")-1;
            if (idx < MaxIdx) FaoVector[idx++] = AnchorName;
            if (idx < MaxIdx) FaoVector[idx++] = "</A>";
            if (idx < MaxIdx) FaoVector[idx++] = AnchorNameOverflowPtr;
            break;

         case 'N' :

            /********/
            /* name */
            /********/

            if (BlockTotals)
            {
               /* make it an empty field */
               if (idx < MaxIdx)
                  FaoVector[idx++] = RequestPtr->DirFieldWidthName;
               if (idx < MaxIdx) FaoVector[idx++] = "";
               if (idx < MaxIdx) FaoVector[idx++] = "";
               break;
            }

            if (idx < MaxIdx) FaoVector[idx++] = RequestPtr->DirFieldWidthName;
            if (idx < MaxIdx) FaoVector[idx++] = AnchorName;
            if (idx < MaxIdx) FaoVector[idx++] = AnchorNameOverflowPtr;
            break;

         case 'O' :

            /*********/
            /* owner */
            /*********/

            if (ParentDirPath[0] || BlockTotals)
            {
               /* make it an empty field */
               if (idx < MaxIdx)
                  FaoVector[idx++] = RequestPtr->DirFieldWidthOwner;
               if (idx < MaxIdx) FaoVector[idx++] = "";
               break;
            }

            if (Config.DirOwnerEnabled)
            {
               OwnerDsc.dsc$a_pointer = Owner;
               OwnerDsc.dsc$w_length = sizeof(Owner)-1;
               sys$fao (&OwnerFaoDsc, &Length, &OwnerDsc, AtrUic);
               Owner[Length] = '\0';
               if (idx < MaxIdx)
                  FaoVector[idx++] = RequestPtr->DirFieldWidthOwner;
               if (idx < MaxIdx) FaoVector[idx++] = Owner;
               break;
            }

            memcpy (Owner, "<I>(disabled)</I>", 18);
            if (idx < MaxIdx)
               FaoVector[idx++] = RequestPtr->DirFieldWidthOwner + 7;
            if (idx < MaxIdx) FaoVector[idx++] = Owner;
            break;

         case 'R' :

            /*****************/
            /* revision date */
            /*****************/

            if (ParentDirPath[0] || BlockTotals)
            {
               /* make it an empty field */
               if (idx < MaxIdx)
                  FaoVector[idx++] = RequestPtr->DirFieldWidthRdt;
               if (idx < MaxIdx)
                  FaoVector[idx++] = RequestPtr->DirFieldWidthRdt;
               if (idx < MaxIdx) FaoVector[idx++] = "";
               if (idx < MaxIdx) FaoVector[idx++] = "";
               break;
            }

            /* format the revision date/time */
            RdtDsc.dsc$a_pointer = Rdt;
            RdtDsc.dsc$w_length = sizeof(Rdt)-1;
            if (RequestPtr->DirFormatVms)
               sys$fao (&VmsDateTimeFaoDsc, &Length, &RdtDsc, &AtrRdt);
            else
            {
               sys$numtim (&NumTime, &AtrRdt);
               sys$fao (&DateTimeFaoDsc, &Length, &RdtDsc,
                        NumTime[2], MonthName[NumTime[1]], NumTime[0]%100,
                        NumTime[3], NumTime[4], NumTime[5]);
            }
            Rdt[Length] = '\0';
            /** if (Debug) fprintf (stdout, "Rdt |%s|\n", Rdt); **/
            if (idx < MaxIdx) FaoVector[idx++] = RequestPtr->DirFieldWidthRdt;
            /* next parameter right justifies the field */
            if (RequestPtr->DirFieldWidthRdt - Length > 0)
            {
               if (idx < MaxIdx)
                  FaoVector[idx++] = RequestPtr->DirFieldWidthRdt - Length;
            }
            else
               if (idx < MaxIdx) FaoVector[idx++] = 0;
            if (idx < MaxIdx) FaoVector[idx++] = "";
            if (idx < MaxIdx) FaoVector[idx++] = Rdt;
            break;

         case 'S' :

            /********/
            /* size */
            /********/

            if (ParentDirPath[0])
            {
               /* make it an empty field */
               if (idx < MaxIdx)
                  FaoVector[idx++] = RequestPtr->DirFieldWidthSize;
               if (idx < MaxIdx)
                  FaoVector[idx++] = RequestPtr->DirFieldWidthSize;
               if (idx < MaxIdx) FaoVector[idx++] = "";
               if (idx < MaxIdx) FaoVector[idx++] = "";
               break;
            }

            SizeDsc.dsc$a_pointer = Size;
            SizeDsc.dsc$w_length = sizeof(Size)-1;

            if (BlockTotals)
            {
               sys$fao (&BlocksFaoDsc, &Length, &SizeDsc,
                        RequestPtr->DirTotalUsedBlocks,
                        RequestPtr->DirTotalAllocatedBlocks);
            }
            else
            {
               AllocatedVbn = AtrRecord.AllocatedVbnLo +
                              (AtrRecord.AllocatedVbnHi << 16);
               EndOfFileVbn = AtrRecord.EndOfFileVbnLo +
                              (AtrRecord.EndOfFileVbnHi << 16);

/*
               if (Debug)
                  fprintf (stdout,
                  "AllocatedVbn: %d EndOfFileVbn: %d FirstFreeByte %d\n",
                  AllocatedVbn, EndOfFileVbn, AtrRecord.FirstFreeByte);
*/

               if (ThisIsADirectory && RequestPtr->DirFormatVms)
                  EndOfFileVbn--;
               else
               if (!ThisIsADirectory &&
                   EndOfFileVbn && !AtrRecord.FirstFreeByte)
                  EndOfFileVbn--;

               RequestPtr->DirTotalAllocatedBlocks += AllocatedVbn;
               RequestPtr->DirTotalUsedBlocks += EndOfFileVbn;

               if (RequestPtr->DirFormatVms)
               {
                  sys$fao (&BlocksFaoDsc, &Length, &SizeDsc,
                           EndOfFileVbn, AllocatedVbn);
               }
               else
               {
                  /* the "<< 9" is more efficient than equivalent "* 512" */
                  if (EndOfFileVbn <= 1)
                     Bytes = AtrRecord.FirstFreeByte;
                  else
                     Bytes = ((EndOfFileVbn - 1) << 9) +
                             AtrRecord.FirstFreeByte;
                  /** if (Debug) fprintf (stdout, "Bytes %d\n", Bytes); **/
                  if (Bytes >= 1048576)
                     sys$fao (&MBytesFaoDsc, &Length, &SizeDsc, Bytes/1048576);
                  else
                  if (Bytes >= 1024)
                     sys$fao (&KBytesFaoDsc, &Length, &SizeDsc, Bytes/1024);
                  else
                     sys$fao (&BytesFaoDsc, &Length, &SizeDsc, Bytes);
               }
            }

            Size[Length] = '\0';
            /** if (Debug) fprintf (stdout, "Size |%s|\n", Size); **/

            if (idx < MaxIdx) FaoVector[idx++] = RequestPtr->DirFieldWidthSize;
            /* next parameter right justifies the field */
            if (RequestPtr->DirFieldWidthSize - Length > 0)
            {
               if (idx < MaxIdx)
                  FaoVector[idx++] = RequestPtr->DirFieldWidthSize - Length;
            }
            else
               if (idx < MaxIdx) FaoVector[idx++] = 0;
            if (idx < MaxIdx) FaoVector[idx++] = "";
            if (idx < MaxIdx) FaoVector[idx++] = Size;
            break;
      }

      if (idx >= MaxIdx)
      {
         /* should be enough for all but a deliberate formatting hack! */
         RequestPtr->ResponseStatusCode = 501;
         ErrorGeneral (RequestPtr, DirLayoutProblem, __FILE__, __LINE__);
         DirEnd (RequestPtr);
         return;
      }
   }

   /** if (Debug) fprintf (stdout, "idx: %d\n", idx); **/

   /*********************/
   /* format and output */
   /*********************/

/*
   if (Debug)
      fprintf (stdout, "DirLayoutFaoPtr |%s|\n", RequestPtr->DirLayoutFaoPtr);
*/
   DirLayoutFaoDsc.dsc$a_pointer = RequestPtr->DirLayoutFaoPtr;
   DirLayoutFaoDsc.dsc$w_length = RequestPtr->DirLayoutFaoLength;
   StringDsc.dsc$a_pointer = String;
   StringDsc.dsc$w_length = sizeof(String)-1;

   if (VMSnok (status =
       sys$faol (&DirLayoutFaoDsc, &Length, &StringDsc, &FaoVector)))
   {
      ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
      DirEnd (RequestPtr);
      return;
   }
   String[Length] = '\0';
   /** if (Debug) fprintf (stdout, "String: %d |%s|\n", Length, String); **/

   /* empty any file-internal description previously generated */
   RequestPtr->InternalDescription[0] = '\0';

   if (VMSnok (BufferOutput (RequestPtr, AstFunctionPtr, String, Length)))
      DirEnd (RequestPtr);
}

/*****************************************************************************/
/*
Return true if the specified file name exists in the specified directory, 
false otherwise.  This function completes synchronously.
*/ 

boolean DirFileExists
(
struct RequestStruct *RequestPtr,
char *Directory,
char *AccessFileName
)
{
   int  status;
   char  ExpandedFileName [256];
   struct FAB  SearchFab;
   struct NAM  SearchNam;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "DirFileExists() |%s|%s|\n", Directory, AccessFileName);

   SearchFab = cc$rms_fab;
   SearchFab.fab$l_dna = Directory;
   SearchFab.fab$b_dns = strlen(Directory);
   SearchFab.fab$l_fna = AccessFileName;
   SearchFab.fab$b_fns = strlen(AccessFileName);
   SearchFab.fab$l_fop = FAB$M_NAM;
   SearchFab.fab$l_nam = &SearchNam;
   SearchNam = cc$rms_nam;
   SearchNam.nam$l_esa = ExpandedFileName;
   SearchNam.nam$b_ess = sizeof(ExpandedFileName)-1;

   if (VMSnok (status = sys$parse (&SearchFab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$parse() %%X%08.08X\n", status);
      return (false);
   }

   status = sys$search (&SearchFab, 0, 0);

   /* release parse and search internal data structures */
   SearchNam.nam$b_nop = NAM$M_SYNCHK;
   sys$parse (&SearchFab, 0, 0);

   if (VMSok(status)) return (true);

   return (false);
}

/*****************************************************************************/
/*
AST-callable function to try for a top readme.
*/ 
 
int DirReadMeTop (struct RequestStruct *RequestPtr)

{
   if (Debug) fprintf (stdout, "DirReadMeTop()\n");
   DirReadMe (RequestPtr, &DirHeading);
}

/*****************************************************************************/
/*
AST-callable function to try for a bottom readme.
*/ 
 
int DirReadMeBottom (struct RequestStruct *RequestPtr)

{
   if (Debug) fprintf (stdout, "DirReadMeBottom()\n");
   DirReadMe (RequestPtr, &DirEnd);
}

/*****************************************************************************/
/*
Attempt to send one of possible multiple files ("README.", et. al.) in the 
specified directory.  Then, provided there was not serious error with the 
send, execute the function specified by the address in 'NextTaskFunction'.  
The file can be HTML or plain-text.  With plain-text the contents will be 
encapsulated.
*/ 
 
int DirReadMe
(
struct RequestStruct *RequestPtr,
void (*NextTaskFunction)(struct RequestStruct*)
)
{
   register char  *cptr, *rptr, *sptr;

   int  status,
        Count;
   char  *TypePtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DirReadMe()\n");

   RequestPtr->NextTaskFunction = NextTaskFunction;

   for (Count = 0; *(rptr = ConfigReadMeFile(Count)); Count++)
   {
      sptr = RequestPtr->FileName;
      for (cptr = RequestPtr->DirDirectoryPart; *cptr; *sptr++ = *cptr++);
      TypePtr = NULL;
      for (/* 'rptr' initialized above! */; *rptr; *sptr++ = *rptr++)
         if (*rptr == '.') TypePtr = rptr;
      *sptr = '\0';
      if (TypePtr == NULL) TypePtr = rptr;

      ConfigContentType (RequestPtr, TypePtr);
      if (strsame (RequestPtr->ContentTypePtr, Config.PlainTextContentType, -1))
         RequestPtr->FileEncapsulateData = true;
      else
         RequestPtr->FileEncapsulateData = false;

      if (VMSok (status = FileBegin (RequestPtr)))
         return (status);

      RequestPtr->FileEncapsulateData = false;
   }

   RequestPtr->NextTaskFunction = NULL;

   if (!Count || status == RMS$_FNF)
   {
      if (NextTaskFunction == NULL)
         DirEnd (RequestPtr);
      else
         (*NextTaskFunction) (RequestPtr);
      return;
   }

   RequestPtr->ErrorTextPtr = "(checking for read-me file)";
   RequestPtr->ErrorHiddenTextPtr = RequestPtr->FileName;
   ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
   DirEnd (RequestPtr);
}

/*****************************************************************************/
/*
Determine whether the file can supply a "description" for for the listing.  At 
this stage this is only HTML files, where then description is taken from the 
<TITLE> element.  If it is an HTML file initiate the asynchronous retrieval of 
the <TITLE> elemment.  If not an HTML file call a function to format and 
provide the file details to the listing, then queue another asynchronous RMS 
directory search call to get the next file name.
*/

DirDescription (struct RequestStruct *RequestPtr)

{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DirDescription()\n");

   RequestPtr->InternalDescription[0] = '\0';

   if (toupper(RequestPtr->ContentTypePtr[0]) ==
       toupper(Config.HtmlContentType[0]) &&
       strsame (RequestPtr->ContentTypePtr, Config.HtmlContentType, -1))
   {
      /*
         Its an HTML file, search for the <TITLE> element.
         In any case the call to FileDetails() is made from the description
         functions, and the search for the next file initiated from these.
      */
      if (VMSnok (DirDescriptionHtml (RequestPtr)))
      {
         /*
            There was a problem with initiating the description retrieval.
            For example, file locked by another user (or something worse!).
            Provide a message and carry on regardless from here.
            If the file details were formatted ok then initiate the search for
            the next file, completion AST to DirFiles().
         */
         DirFormat (RequestPtr, &DirSearchFiles, "", false);
         return;
      }
   }
   else
   {
      /*
         Not an HTML file.  Format the file details now.
         If there is a problem with the file details do not proceed further.
         If the file details were formatted ok then initiate the search for
         the next file, completion AST to DirFiles().
      */
      DirFormat (RequestPtr, &DirSearchFiles, "", false);
      return;
   }
}

/*****************************************************************************/
/*
Initiate an asynchronous read of enough records of an HTML file to locate the 
<TITLE> element.  This is used as the file "description".  When the first read 
is queued return immediately to the calling routine.
*/

int DirDescriptionHtml (struct RequestStruct *RequestPtr)

{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "DirDescriptionHtml() |%s|\n", RequestPtr->DirSpec);

   RequestPtr->FileFab = cc$rms_fab;
   RequestPtr->FileFab.fab$b_fac = FAB$M_GET;
   RequestPtr->FileFab.fab$l_fna = RequestPtr->DirSpec;  
   RequestPtr->FileFab.fab$b_fns = strlen(RequestPtr->DirSpec);
   RequestPtr->FileFab.fab$b_shr = FAB$M_SHRGET;

   if (VMSnok (status = sys$open (&RequestPtr->FileFab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$open() %%X%08.08X\n", status);
      return (status);
   }

   /* allocate heap memory for file record buffer */
   if (RequestPtr->DirFileBufferPtr == NULL)
   {
      /* allow one byte for terminating null */
      if ((RequestPtr->DirFileBufferPtr =
          HeapAlloc (RequestPtr, FileBufferSize+1)) == NULL)
      {
         ErrorHeapAlloc (RequestPtr, __FILE__, __LINE__);
         DirEnd (RequestPtr);
         return (SS$_NORMAL);
      }
   }

   RequestPtr->FileRab = cc$rms_rab;
   RequestPtr->FileRab.rab$l_fab = &RequestPtr->FileFab;
   /* 2 buffers and read ahead performance option */
   RequestPtr->FileRab.rab$b_mbf = 2;
   RequestPtr->FileRab.rab$l_rop = RAB$M_RAH;
   RequestPtr->FileRab.rab$l_ubf = RequestPtr->DirFileBufferPtr;
   RequestPtr->FileRab.rab$w_usz = FileBufferSize;

   if (VMSnok (status = sys$connect (&RequestPtr->FileRab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$connect() %%X%08.08X\n", status);
      sys$close (&RequestPtr->FileFab, 0, 0);
      return (status);
   }

   *(RequestPtr->DirDescriptionPtr = RequestPtr->InternalDescription) = '\0';
   RequestPtr->DirInsideDescription =
      RequestPtr->DirDescriptionRetrieved = false;
   RequestPtr->DirOpeningTagCount = RequestPtr->DirClosingTagCount =
      RequestPtr->DirDescriptionLineCount = 0;

   /* set the RAB user context storage to the client thread pointer */
   RequestPtr->FileRab.rab$l_ctx = RequestPtr;

   /* queue up a read of the first record in the HTML file */
   sys$get (&RequestPtr->FileRab,
            &DirDescriptionHtmlRecord, &DirDescriptionHtmlRecord);

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
This is an AST completion routine called each time sys$get() completes.  It 
returns with the next record, end-of-file, or some other error.  Examine the 
record looking for text comprising the first of either the <TITLE></TITLE> 
element or a heading element (e.g. <H1></H1>), retrieving this and using it 
as the file description.
*/

DirDescriptionHtmlRecord (struct RAB *RabPtr)

{
   register char  *dptr, *rptr, *zptr;
   register struct RequestStruct  *RequestPtr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
   {
      fprintf (stdout,
         "DirDescriptionHtmlRecord() sts: %%X%08.08X stv: %%X%08.08X\n|%s|\n",
         RabPtr->rab$l_sts,
         RabPtr->rab$l_stv,
         ((struct RequestStruct*)(RabPtr->rab$l_ctx))->InternalDescription);
   }

   /* get the pointer to the thread from the RAB user context storage */
   RequestPtr = RabPtr->rab$l_ctx;

   if (VMSnok (RequestPtr->FileRab.rab$l_sts))
   {
      if (RequestPtr->FileRab.rab$l_sts == RMS$_EOF)
      {
         /***************/
         /* end-of-file */
         /***************/

         sys$close (&RequestPtr->FileFab, 0, 0);
         DirFormat (RequestPtr, &DirSearchFiles, "", false);
         return;
      }

      /**********************/
      /* error reading file */
      /**********************/

      RequestPtr->ErrorHiddenTextPtr = RequestPtr->SearchNam.nam$l_dev;
      ErrorVmsStatus (RequestPtr, RequestPtr->FileRab.rab$l_sts,
                      __FILE__, __LINE__);
      sys$close (&RequestPtr->FileFab, 0, 0);
      DirEnd (RequestPtr);
      return;
   }

   RequestPtr->DirDescriptionLineCount++;
   RequestPtr->FileRab.rab$l_ubf[RequestPtr->FileRab.rab$w_rsz] = '\0';

   /****************************************/
   /* look for the <TITLE> or <Hn> element */
   /****************************************/

   /* 
      'zptr' points at the last character of the description storage.
      It prevents non-closed <TITLE> or <Hn> structures from running off
      the end of the storage area.
   */

   dptr = RequestPtr->DirDescriptionPtr;
   zptr = RequestPtr->InternalDescription +
          sizeof(RequestPtr->InternalDescription) - 3;

   /*
      Add a space for any record boundary occuring in the title,
      except if it's a leading space (i.e. first after the leading quote).
   */
   if (RequestPtr->DirInsideDescription &&
       dptr > RequestPtr->InternalDescription+1 &&
       dptr < zptr)
      *dptr++ = ' ';

   for (rptr = RequestPtr->FileRab.rab$l_ubf; *rptr; rptr++)
   {
      if (*rptr == '<')
      {
         RequestPtr->DirOpeningTagCount++;
         if (toupper(rptr[1]) == 'T' && strsame (rptr+2, "ITLE>", 5))
         {
            RequestPtr->DirInsideDescription = true;
            rptr += 6;
            RequestPtr->DirClosingTagCount++;
            /* descriptions from HTML files have a leading quote */
            *dptr++ = '\"';
            continue;
         }
         if (toupper(rptr[1]) == 'H' && isdigit(rptr[2]) && rptr[3] == '>')
         {
            RequestPtr->DirInsideDescription = true;
            rptr += 3;
            RequestPtr->DirClosingTagCount++;
            /* descriptions from HTML files have a leading quote */
            *dptr++ = '\"';
            continue;
         }
         if (rptr[1] == '/')
         {
            if (toupper(rptr[2]) == 'T' && strsame (rptr+3, "ITLE>", 5))
            {
               RequestPtr->DirInsideDescription = false;
               RequestPtr->DirDescriptionRetrieved = true;
               rptr += 7;
               RequestPtr->DirClosingTagCount++;
               /*
                  Suppress any trailing white-space.
                  Descriptions from HTML files have a trailing quote.
               */
               if (dptr > RequestPtr->InternalDescription+1) dptr--;
               while (dptr > RequestPtr->InternalDescription+1 &&
                      isspace(*dptr)) dptr--;
               *++dptr = '\"';
               *++dptr = '\0';
               continue;
            }
            if (toupper(rptr[2]) == 'H' && isdigit(rptr[3]) && rptr[4] == '>')
            {
               RequestPtr->DirInsideDescription = false;
               RequestPtr->DirDescriptionRetrieved = true;
               rptr += 4;
               RequestPtr->DirClosingTagCount++;
               /*
                  Suppress any trailing white-space.
                  Descriptions from HTML files have a trailing quote.
               */
               if (dptr > RequestPtr->InternalDescription+1) dptr--;
               while (dptr > RequestPtr->InternalDescription+1 &&
                      isspace(*dptr)) dptr--;
               *++dptr = '\"';
               *++dptr = '\0';
               continue;
            }
         }
      }
      else
      if (*rptr == '>')
      {
         RequestPtr->DirClosingTagCount++;
         continue;
      }

      /* don't scan the line further if we've found what we're looking for */
      if (RequestPtr->DirDescriptionRetrieved) break;

      /* if we're currently inside a tag name (between "<" and ">") */
      if (RequestPtr->DirOpeningTagCount > RequestPtr->DirClosingTagCount)
         continue;

      /* suppress leading white-space (i.e immediately after leading quote) */
      if (RequestPtr->DirInsideDescription &&
          !(dptr == RequestPtr->InternalDescription+1 && isspace(*rptr)) &&
          dptr < zptr)
         *dptr++ = *rptr;
   }

   /* terminate the title/description */
   if (dptr > RequestPtr->InternalDescription) *dptr = '\0';
   RequestPtr->DirDescriptionPtr = dptr;

   if (RequestPtr->DirDescriptionRetrieved ||
       RequestPtr->DirDescriptionLineCount > MaxDirDescriptionLineCount)
   {
      /************************************/
      /* title found, or out-of-patience! */
      /************************************/

      sys$close (&RequestPtr->FileFab, 0, 0);

      if (Debug)
         fprintf (stdout, "RequestPtr->InternalDescription |%s|\n",
                  RequestPtr->InternalDescription);

      DirFormat (RequestPtr, &DirSearchFiles, "", false);
      return;
   }
   else
   {
      /*****************/
      /* still looking */
      /*****************/

      /* queue another read, completion AST back to this function again */
      sys$get (&RequestPtr->FileRab,
               &DirDescriptionHtmlRecord, &DirDescriptionHtmlRecord);
   }
}

/*****************************************************************************/

