/*****************************************************************************/
/*
                                 Error.c

Error reporting functions.

After a request thread is created and an error is generated the storage
'RequestPtr->ErrorMessagePtr' becomes non-NULL.  This mechanism can be used 
to detect whether an error has occured.  Most common errors create an error 
message in heap memory and point at this using 'RequestPtr->ErrorMessagePtr'. 
This message can be output at some time convenient to the task underway.  
After an error message is sent to the client the 'RequestPtr->ErrorMessagePtr' 
is returned to a NULL value to indicate no outstanding error (but usually the 
thread is disposed of after an error occurs).  Notification of other, more 
ciritical errors are sent directly to the client at the time they occur (for 
instance, failed heap memory allocation).  In this circumstance
'RequestPtr->ErrorMessagePtr' also becomes non-NULL but points to an empty 
string (meaning the message has already been sent).

Some error message functions have only a channel as the destination parameter 
(are not supplied with a 'RequestPtr').  These errors have occured early 
enough in connection acceptance processing that a thread structures has not 
been created.


VERSION HISTORY
---------------
01-DEC-95  MGD  HTTPd version 3
25-MAR-95  MGD  minor functional and cosmetic changes
20-DEC-94  MGD  multi-threaded daemon
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <string.h>

/* VMS related header files */
#include <descrip.h>
#include <iodef.h>
#include <ssdef.h>
#include <stsdef.h>

#include "httpd.h"

/*************************/
/* external declarations */
/*************************/

extern boolean  Debug;
extern char  SoftwareID[];
extern char  Utility[];
extern struct ConfigStruct  Config;

/***********************/
/* function prototypes */
/***********************/

ErrorHeapAlloc (struct RequestStruct*, char*, int);
ErrorExitVmsStatus (int, char*, char*, int);
ErrorGeneral (struct RequestStruct*, char*, char*, int);
ErrorSendToClient (struct RequestStruct*, void*);
ErrorServerShutdown (unsigned short);
ErrorVmsStatus (struct RequestStruct*, int, char*, int);

/**********************/
/* external functions */
/**********************/

unsigned char* HeapAlloc (struct RequestStruct*, int);

/*****************************************************************************/
/*
Exit the application after output of the module and line location reporting 
the error, a brief explanation, and then exiting generating a DCL-reported 
message.
*/

ErrorExitVmsStatus
(
int VmsStatus,
char *Explanation,
char *SourceFileName,
int SourceLineNumber
)
{
   register char  *cptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ErrorExitVmsStatus()\n");

   /* 
      The source file format provided by the "__FILE__" macro will be
      "device:[directory]name.type;ver".  Reduce that to "name".
   */
   for (cptr = SourceFileName; *cptr && *cptr != ';'; cptr++);
   if (*cptr)
   {
      while (*cptr != '.') cptr--;
      *cptr-- = '\0';
   }
   while (*cptr != ']') cptr--;
   cptr++;

   fprintf (stdout,
"%%%s-E-WHERE, module: %s line: %d\n\
%%%s-E-WHAT, %s\n",
            Utility, cptr, SourceLineNumber,
            Utility, Explanation);

   if (!VmsStatus) VmsStatus = STS$K_ERROR | STS$M_INHIB_MSG;
   exit (VmsStatus);
}

/****************************************************************************/
/*
If no bytes have been sent to the client (and therefore no header) generate
and send an HTTP response header, otherwise just send a <P> tag for a bit of
separation.
*/ 

ErrorHeader (struct RequestStruct *RequestPtr)

{
   static $DESCRIPTOR (StringDsc, "");

   static $DESCRIPTOR (HttpErrorHeaderFaoDsc,
"!AZ !UL Error report.\r\n\
Server: !AZ\r\n\
Content-Type: text/html\r\n\
\r\n");

   unsigned short  Length;
   char  String [1024];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ErrorHeader()\n");

   if (RequestPtr->BytesTx)
   {
      /* lets just stick a bit of separation in here */
      return (QioNetWrite (RequestPtr, 0, "<P>\n", 4));
   }
   else
   {
      /* nothing has been written to the client, need an HTTP header */
      StringDsc.dsc$a_pointer = String;
      StringDsc.dsc$w_length = sizeof(String)-1;
      sys$fao (&HttpErrorHeaderFaoDsc, &Length, &StringDsc,
               HttpProtocol, RequestPtr->ResponseStatusCode, SoftwareID);
      String[Length] = '\0';

      /* allocate heap memory for the response header */
      if ((RequestPtr->ResponseHeaderPtr =
           HeapAlloc (RequestPtr, Length)) == NULL)
      {
         ErrorHeapAlloc (RequestPtr, __FILE__, __LINE__);
         return (SS$_INSFMEM);
      }
      memcpy (RequestPtr->ResponseHeaderPtr, String, Length);

      return (QioNetWrite (RequestPtr, 0, String, Length));
   }
}

/****************************************************************************/
/*
If the error message pointer is non-NULL and the message is not empty then 
send it to the client.
*/ 

ErrorSendToClient
(
struct RequestStruct *RequestPtr,
void *AstFunctionPtr
)
{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ErrorSendToClient()\n");

   if (RequestPtr->ErrorMessagePtr != NULL)
   {
      if (RequestPtr->ErrorMessagePtr[0])
      {
         if (VMSok (ErrorHeader (RequestPtr)) && !RequestPtr->MethodHead)
            QioNetWrite (RequestPtr, AstFunctionPtr,
                         RequestPtr->ErrorMessagePtr,
                         RequestPtr->ErrorMessageLength);
      }
      /* indicate the error message has been sent */
      RequestPtr->ErrorMessagePtr = NULL;
   }
   RequestPtr->ErrorMessageLength = 0;
}

/*****************************************************************************/
/*
Generate an error message about a VMS status problem for subsequent reporting
to the client.
*/

ErrorVmsStatus
(
struct RequestStruct *RequestPtr,
int Status,
char *SourceFileName,
int SourceLineNumber
)
{
   static char  DocumentNotFound [] = "Document not found",
                FileNotFound [] = "File not found",
                DocumentProtVio [] = "Document protection violation",
                FileProtVio [] = "File protection violation";

   static $DESCRIPTOR (StatusFaoDsc,
"<!!-- SoftwareID: !AZ Module: !AZ Line: !UL -->\n\
<H1>ERROR!!</H1><!!-- !UL -->\n\
<P>Report generated by server.\n\
<P>!AZ ... <TT>!AZ</TT>\n\
<!!-- sts: %X!XL \"!AZ\" -->\n\
!AZ");

   register char  *cptr, *sptr;

   int  FaoStatus;
   unsigned short  Length;
   char  *ContentTypePtr,
         *MessagePtr,
         *RecommendPtr;
   char  Message [256],
         String [1024];
   $DESCRIPTOR (MessageDsc, Message);
   $DESCRIPTOR (StringDsc, String);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ErrorVmsStatus()\n");

   /* don't overwrite any existing error message */
   if (RequestPtr->ErrorMessagePtr != NULL) return;

   if (RequestPtr->ContentTypePtr == NULL)
      ContentTypePtr = "";
   else
      ContentTypePtr = RequestPtr->ContentTypePtr;

   if (Status == RMS$_FNF)
   {
      RequestPtr->ResponseStatusCode = 404;
      if (strsame (ContentTypePtr, "text/", 5))
         MessagePtr = DocumentNotFound;
      else
         MessagePtr = FileNotFound;
   }
   else
   if (Status == RMS$_PRV)
   {
      RequestPtr->ResponseStatusCode = 403;
      if (strsame (ContentTypePtr, "text/", 5))
         MessagePtr = DocumentProtVio;
      else
         MessagePtr = FileProtVio;
   }
   else
   if (VMSok (sys$getmsg (Status, &Length, &MessageDsc, 1, 0))) 
   {
      if (Status == RMS$_DNF)
         RequestPtr->ResponseStatusCode = 404;
      else
         RequestPtr->ResponseStatusCode = 500;
      Message[Length] = '\0';
      *(MessagePtr = Message) = *Message;
      cptr = sptr = Message;
      *cptr = toupper(*cptr);
      while (*cptr)
      {
         if (*cptr == '!')
         {
            /* improve the look of sys$fao() formatting (cover them up :^) */
            cptr++;
            *sptr++ = '?';
            /* step over any field width digits */
            while (isdigit(*cptr)) cptr++;
            /* usually two formatting characters */
            if (isalpha(*cptr)) cptr++;
            if (isalpha(*cptr)) cptr++;
         }
         else
            *sptr++ = *cptr++;
      }
      *sptr = '\0';
   }
   else
      strcpy (MessagePtr = Message, "<B>sys$getmsg() failed!</B>");

   if (RequestPtr->ErrorTextPtr == NULL)
      if (RequestPtr->PathInfoPtr == NULL)
         RequestPtr->ErrorTextPtr = "(no information available)";
      else
         RequestPtr->ErrorTextPtr = RequestPtr->PathInfoPtr;

   if (RequestPtr->ErrorHiddenTextPtr == NULL)
      RequestPtr->ErrorHiddenTextPtr = "(no information available)";

   if (!Config.ErrorRecommend)
      RecommendPtr = "";
   else
   if (Status == RMS$_FNF || Status == RMS$_DNF)
      RecommendPtr = 
         "<P><I>(document, or bookmark, requires revision)</I>\n";
   else
   if (Status == RMS$_PRV)
      RecommendPtr = "<P><I>(protection requires revision)</I>\n";
   else
   if (Status == RMS$_SYN)
      RecommendPtr = "<P><I>(document requires correction)</I>\n";
   else
   if (Status == RMS$_FLK)
      RecommendPtr = "<P><I>(try again shortly)</I>\n";
   else
      RecommendPtr = "<P><I>(notify system administrator)</I>\n";

   /* 
      The source file format provided by the "__FILE__" macro will be
      "device:[directory]name.type;ver".  Reduce that to "name".
   */
   for (cptr = SourceFileName; *cptr && *cptr != ';'; cptr++);
   if (*cptr)
   {
      while (*cptr != '.') cptr--;
      *cptr-- = '\0';
   }
   while (*cptr != ']') cptr--;
   cptr++;

   if (VMSnok (FaoStatus =
       sys$fao (&StatusFaoDsc, &Length, &StringDsc,
                SoftwareID, cptr, SourceLineNumber,
                RequestPtr->ResponseStatusCode,
                MessagePtr, RequestPtr->ErrorTextPtr,
                Status, RequestPtr->ErrorHiddenTextPtr,
                RecommendPtr)))
   {
      ErrorInternal (RequestPtr, FaoStatus, "sys$fao()", __FILE__, __LINE__);
      return;
   }

   String[RequestPtr->ErrorMessageLength = Length] = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", String);
   if ((RequestPtr->ErrorMessagePtr =
       HeapAlloc (RequestPtr, Length+1)) == NULL)
   {
      ErrorHeapAlloc (RequestPtr, __FILE__, __LINE__);
      return;
   }
   memcpy (RequestPtr->ErrorMessagePtr, String, Length+1);
}

/*****************************************************************************/
/*
Generate an error message about a general (non-VMS status) problem for
subsequent reporting to the client.
*/
 
ErrorGeneral
(
struct RequestStruct *RequestPtr,
char *Explanation,
char *SourceFileName,
int SourceLineNumber
)
{
   static $DESCRIPTOR (StatusErrorFaoDsc,
"<!!-- SoftwareID: !AZ Module: !AZ Line: !UL -->\n\
<H1>ERROR!!</H1><!!-- !UL -->\n\
<P>Report generated by server.\n\
<P>!AZ\n");

   static $DESCRIPTOR (ErrorFaoDsc,
"<!!-- SoftwareID: !AZ Module: !AZ Line: !UL -->\n\
<H1>ERROR!!</H1>\n\
<P>Report generated by server.\n\
<P>!AZ\n");

   register char  *cptr;

   int  FaoStatus;
   unsigned short  Length;
   char  String [1024];
   $DESCRIPTOR (StringDsc, String);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ErrorGeneral()\n");

   /* don't overwrite any existing error message */
   if (RequestPtr->ErrorMessagePtr != NULL) return;

   /* 
      The source file format provided by the "__FILE__" macro will be
      "device:[directory]name.type;ver".  Reduce that to "name".
   */
   for (cptr = SourceFileName; *cptr && *cptr != ';'; cptr++);
   if (*cptr)
   {
      while (*cptr != '.') cptr--;
      *cptr-- = '\0';
   }
   while (*cptr != ']') cptr--;
   cptr++;

   if (!RequestPtr->ResponseStatusCode)
      RequestPtr->ResponseStatusCode = 404;

   if (RequestPtr->ResponseStatusCode == 200)
      FaoStatus = sys$fao (&ErrorFaoDsc, &Length, &StringDsc,
                           SoftwareID, cptr, SourceLineNumber, Explanation);
   else
      FaoStatus = sys$fao (&StatusErrorFaoDsc, &Length, &StringDsc,
                           SoftwareID, cptr, SourceLineNumber,
                           RequestPtr->ResponseStatusCode, Explanation);
   if (VMSnok (FaoStatus))
   {
      ErrorInternal (RequestPtr, FaoStatus, "sys$fao()", __FILE__, __LINE__);
      return;
   }
   String[RequestPtr->ErrorMessageLength = Length] = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", String);
   if ((RequestPtr->ErrorMessagePtr =
        HeapAlloc (RequestPtr, Length+1)) == NULL)
   {
      ErrorHeapAlloc (RequestPtr, __FILE__, __LINE__);
      return;
   }
   memcpy (RequestPtr->ErrorMessagePtr, String, Length+1);
}

/*****************************************************************************/
/*
Report heap memory allocation problem IMMEDIATELY to the client (PS: can't use
dynamic memory to report, so do it in two parts, one that waits so that
automatic storage can be used, the second a static message).
*/
 
ErrorHeapAlloc
(
struct RequestStruct *RequestPtr,
char *SourceFileName,
int SourceLineNumber
)
{
   static $DESCRIPTOR (WhereFaoDsc,
          "<P>\n<!!-- SoftwareID: !AZ Module: !AZ Line: !UL -->\n");

   static char  ErrorReport [] =
"<H1>ERROR!!</H1><!!-- 500 -->\n\
<H2>Please report.</H2>\n\
<P>Heap allocation failed.\n";

   register char  *cptr;

   int  FaoStatus;
   unsigned short  Length;
   char  *RecommendPtr;
   char  String [256];
   $DESCRIPTOR (StringDsc, String);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ErrorHeapAlloc()\n");

   /* 
      The source file format provided by the "__FILE__" macro will be
      "device:[directory]name.type;ver".  Reduce that to "name".
   */
   for (cptr = SourceFileName; *cptr && *cptr != ';'; cptr++);
   if (*cptr)
   {
      while (*cptr != '.') cptr--;
      *cptr-- = '\0';
   }
   while (*cptr != ']') cptr--;
   cptr++;

   if (VMSnok (FaoStatus =
       sys$fao (&WhereFaoDsc, &Length, &StringDsc,
                SoftwareID, cptr, SourceLineNumber)))
   {
      ErrorInternal (RequestPtr, FaoStatus, "sys$fao()", __FILE__, __LINE__);
      return;
   }
   String[Length] = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", String);

   if (VMSok (ErrorHeader (RequestPtr)) && !RequestPtr->MethodHead)
      NetWrite (RequestPtr, 0, String, Length);

   RequestPtr->ErrorMessagePtr = ErrorReport;
   RequestPtr->ErrorMessageLength = sizeof(ErrorReport)-1;
   RequestPtr->ResponseStatusCode = 500;
}

/*****************************************************************************/
/*
*/
 
ErrorInternal
(
struct RequestStruct *RequestPtr,
int Status,
char *Explanation,
char *SourceFileName,
int SourceLineNumber
)
{
   static $DESCRIPTOR (StringFaoDsc,
"<!!-- SoftwareID: !AZ Module: !AZ Line: !UL -->\n\
<H1>SERVER INTERNAL ERROR!!</H1><!!-- 500 -->\n\
<H2>Please report.</H2>\n\
<!!-- !AZ %X!XL -->\n\
!AZ");

   register char  *cptr;
   unsigned short  Length;
   char  *RecommendPtr;
   char  String [1024];
   $DESCRIPTOR (StringDsc, String);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ErrorInternal()\n");

   /* 
      The source file format provided by the "__FILE__" macro will be
      "device:[directory]name.type;ver".  Reduce that to "name".
   */
   for (cptr = SourceFileName; *cptr && *cptr != ';'; cptr++);
   if (*cptr)
   {
      while (*cptr != '.') cptr--;
      *cptr-- = '\0';
   }
   while (*cptr != ']') cptr--;
   cptr++;

   if (!Config.ErrorRecommend)
      RecommendPtr = "";
   else
      RecommendPtr = "<P><I>(notify system administrator)</I>\n";

   sys$fao (&StringFaoDsc, &Length, &StringDsc,
            SoftwareID, cptr, SourceLineNumber, Explanation, Status,
            RecommendPtr);
   String[RequestPtr->ErrorMessageLength = Length] = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", String);
   if ((RequestPtr->ErrorMessagePtr =
        HeapAlloc (RequestPtr, Length+1)) == NULL)
   {
      ErrorHeapAlloc (RequestPtr, __FILE__, __LINE__);
      return;
   }
   memcpy (RequestPtr->ErrorMessagePtr, String, Length+1);
   RequestPtr->ResponseStatusCode = 500;
}

/*****************************************************************************/
/*
Error message specifically for the when there is no request data structure 
available (sent via the network channel).
*/

ErrorRequestCallocFailed
(
unsigned short ClientChannel,
int Status,
char *SourceFileName,
int SourceLineNumber
)
{
   static $DESCRIPTOR (ReportFaoDsc,
"!AZ 500 Error report follows.\r\n\
Server: !AZ\r\n\
Content-Type: text/html\r\n\
\r\n\
<!!-- SoftwareID: !AZ Module: !AZ Line: !UL -->\n\
<H1>ERROR!!</H1><!!-- 500 -->\n\
<P>Report generated by server.\n\
<P>Request data structure calloc() failed.\n\
<BR>Status: <TT>%X!XL</TT>\n");

   register char  *cptr;
   unsigned short  Length;
   char  *RecommendPtr;
   char  String [512];
   $DESCRIPTOR (StringDsc, String);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ErrorRequestCallocFailed()\n");

   /* 
      The source file format provided by the "__FILE__" macro will be
      "device:[directory]name.type;ver".  Reduce that to "name".
   */
   for (cptr = SourceFileName; *cptr && *cptr != ';'; cptr++);
   if (*cptr)
   {
      while (*cptr != '.') cptr--;
      *cptr-- = '\0';
   }
   while (*cptr != ']') cptr--;
   cptr++;

   if (!Config.ErrorRecommend)
      RecommendPtr = "";
   else
      RecommendPtr = "<P><I>(notify system administrator)</I>\n";

   sys$fao (&ReportFaoDsc, &Length, &StringDsc,
            SoftwareID,
            SoftwareID, cptr, SourceLineNumber,
            Status, RecommendPtr);
   String[Length] = '\0';
   NetWrite (ClientChannel, String, Length);
}

/*****************************************************************************/
/*
Report that client host name has been denied access to ther server.
*/
 
ErrorAccessDenied (unsigned short Channel)
 
{
   static char  ErrorMessage [] =
"HTTP/1.0 403 Server access denied.\r\n\
Content-Type: text/html\r\n\
\r\n\
Access denied.\n";

   NetWrite (Channel, ErrorMessage, sizeof(ErrorMessage)-1);
}

/*****************************************************************************/
/*
Report that there is already the limit of client connections.
*/
 
ErrorTooBusy (unsigned short Channel)
 
{
   static char  ErrorMessage [] =
"HTTP/1.0 502 Server too busy.\r\n\
Content-Type: text/html\r\n\
\r\n\
Server too busy right now ... try again shortly.\n";

   NetWrite (Channel, ErrorMessage, sizeof(ErrorMessage)-1);
}

/*****************************************************************************/
/*
Report that the server is currently shutting down (only continuing to service 
connections in progress).
*/ 
 
ErrorServerShutdown (unsigned short Channel)
 
{
   static char  ErrorMessage [] =
"HTTP/1.0 502 Server shutting down.\r\n\
Content-Type: text/html\r\n\
\r\n\
Server is shutting down ... try again shortly.\n";

   NetWrite (Channel, ErrorMessage, sizeof(ErrorMessage)-1);
}

/*****************************************************************************/
/*
*/
 
char* SysGetMsg (int StatusValue)
 
{
   static char  Message [256];
   short int  Length;
   $DESCRIPTOR (MessageDsc, Message);
 
   sys$getmsg (StatusValue, &Length, &MessageDsc, 0, 0);
   Message[Length] = '\0';
   if (Debug) fprintf (stdout, "SysGetMsg() |%s|\n", Message);
   return (Message);
}
 
/****************************************************************************/

