/*****************************************************************************/
/*
                                   File.c

This module implements a full multi-threaded, AST-driven, asynchronous file 
send.  The AST-driven nature makes the code a little more difficult to follow, 
but creates a powerful, event-driven, multi-threaded server.  All of the 
necessary functions implementing this module are designed to be non-blocking. 

It uses the same buffer space as, and interworks with, BufferOutput().  If 
there is already data (text) buffered in this area the file module will, for 
record-oriented, non-HTML-escaped transfers, continue to fill the area (using 
its own buffering function), flushing when and if necessary.  At end-of-file 
explicitly flush the buffer only if escaping HTML-forbidden characters, 
otherwise allow subsequent processing to do it as necessary.  For binary-mode 
files the buffer is explicitly flushed before commencing the file transfer. 

It handles both text (record-by-record) and binary (virtual block-oriented) 
transfers.  The text mode increases efficiency by building up a series of file 
records in available buffer space before transfering them as one I/O to the 
network.  The binary mode reads as many virtual blocks as will fit into the 
available buffer, then transfers these as one I/O to the network. 

For text mode transfers (record-oriented) the completion of each read of the 
next record using FileNextRecord() calls FileNextRecordAST().  This function 
checks if the record buffer is full (by detecting if the last read failed with 
a "record too big for user's buffer" error).  If full it writes the buffer to 
the network using QioNetWrite(), which when complete calls FileNextRecord() as 
an AST routine.  This begins by getting the next record again (the one that 
failed).  If the buffer is not full FileNextRecordAST() just calls 
FileNextRecord() to get the next record, until the buffer fills. 

For binary mode transfers (non-record oriented) the completion of each read of 
the next series of blocks using FileNextBlocks() calls FileNextBlocksAST().  
This function writes the blocks (binary) using QioNetWrite() which when 
complete calls FileNextBlocks() as an AST.  This in turn calls 
FileNextBlocksAST(), etc., which drives the transfer until EOF. 

Once the file has been opened, the transfer becomes event-driven.

The module can encapsulate plain-text and escape HTML-forbidden characters. 

The only non-success status codes returned to the calling function are RMS 
codes.  These can be used to determine the success of the the file access.


VERSION HISTORY
---------------
01-DEC-95  MGD  HTTPd version 3
27-SEP-95  MGD  added If-Modified-Since: functionality;
                changed carriage-control on records from <CR><LF> to single
                <LF> ('\n' ... newline), to better comply with some browsers
                (Netscape was spitting on X-bitmap files, for example!)
07-AUG-95  MGD  ConfigIncludeCommentedInfo to allow physical file
                specification to be included as commentary within an HTML file
13-JUL-95  MGD  bugfix where occasionally a record was re-read after flushing
                the records accumulated in the buffer NOT due to RMS$_RTB
20-DEC-94  MGD  initial development for multi-threaded daemon
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <string.h>

/* VMS related header files */
#include <descrip.h>
#include <iodef.h>
#include <libdef.h>
#include <rms.h>
#include <rmsdef.h>
#include <ssdef.h>
#include <stsdef.h>

/* application header file */
#include "httpd.h"

/********************/
/* external storage */
/********************/

extern boolean  Debug;
extern int  OutputBufferSize;
extern char  SoftwareID[];
extern struct AccountingStruct  Accounting;
extern struct ConfigStruct  Config;

/****************************/
/* functions in this module */
/****************************/

FileBegin (struct RequestStruct*);
FileEnd (struct RequestStruct*);
int FileHttpHeader (struct RequestStruct*);
FileFlushBuffer (struct RequestStruct*, void*);
FileFlushBufferEscapeHtml (struct RequestStruct*, void*);
FileNextBlocks (struct RequestStruct*);
FileNextBlocksAST (struct RAB*); 
FileNextRecord (struct RequestStruct*);
FileNextRecordAST (struct RAB*); 

/**********************/
/* external functions */
/**********************/

ConcludeProcessing (struct RequestStruct*);
ErrorHeapAlloc (struct RequestStruct*, char*, int);
ErrorVmsStatus (struct RequestStruct*, int, char*, int);
unsigned char* HeapAlloc (struct RequestStruct*, int);
unsigned char* HeapRealloc (struct RequestStruct*, unsigned char*, int);
int IfModifiedSince (struct RequestStruct*, unsigned long*, unsigned long*);
QioNetWrite (struct RequestStruct*, void*, char*, int);

/*****************************************************************************/
/*
Open the specified file.  If there is a problem opening it then just return 
the status, do not report an error or anything.  This is used to determine 
whether the file exists, as when attempting to open one of multiple, possible 
home pages.

As fixed documents (files) can have revision date/times easily checked, any
"If-Modified-Since:" request field date/time supplied in the request is 
processed and the file only sent if modified later than specified.
A "Last-Modified:" field is included in any response header. 

When successfully opened and connected generate an HTTP header, if required.
Once open and connected the transfer becomes I/O event-driven. 

'RequestPtr->FileName' contains the VMS file specification.
'RequestPtr->ContentTypePtr' contains the MIME content type.
'RequestPtr->BinaryEncoding' is true if the content type is binary.
'RequestPtr->IfModifiedSinceBinaryTime' contains an optional VMS time.
'RequestPtr->SendHttpHeader' is true if an HTTP header should be generated.
*/ 
 
FileBegin (struct RequestStruct *RequestPtr)

{
   int  status;
   void  *AstFunctionPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug)
   {
      fprintf (stdout, "File()\n|%s|%s|",
               RequestPtr->FileName, RequestPtr->ContentTypePtr);
      if (RequestPtr->BinaryEncoding)
         fprintf (stdout, "binary-encoding|\n");
      else
         fprintf (stdout, "record-encoding|\n");
   }

   /* set up default error report information (just in case!) */
   RequestPtr->ErrorTextPtr = RequestPtr->PathInfoPtr;
   RequestPtr->ErrorHiddenTextPtr = RequestPtr->FileName;

   /*************/
   /* open file */
   /*************/

   RequestPtr->FileFab = cc$rms_fab;
   if (RequestPtr->BinaryEncoding)
   {
      RequestPtr->FileFab.fab$b_fac = FAB$M_GET | FAB$M_BIO; 
      RequestPtr->FileFab.fab$b_shr = FAB$M_SHRGET;
   }
   else
   {
      RequestPtr->FileFab.fab$b_fac = FAB$M_GET;
      RequestPtr->FileFab.fab$b_shr = FAB$M_SHRGET | FAB$M_SHRPUT;
   }
   RequestPtr->FileFab.fab$l_fna = RequestPtr->FileName;  
   RequestPtr->FileFab.fab$b_fns = strlen(RequestPtr->FileName);

   /* initialize the NAM block */
   RequestPtr->FileNam = cc$rms_nam;
   RequestPtr->FileFab.fab$l_nam = &RequestPtr->FileNam;
   RequestPtr->FileNam.nam$l_esa = RequestPtr->ExpandedFileName;
   RequestPtr->FileNam.nam$b_ess = sizeof(RequestPtr->ExpandedFileName)-1;

   /* initialize the date and allocation extended attribute blocks */
   RequestPtr->FileXabDat = cc$rms_xabdat;
   RequestPtr->FileXabFhc = cc$rms_xabfhc;
   RequestPtr->FileFab.fab$l_xab = &RequestPtr->FileXabDat;
   RequestPtr->FileXabDat.xab$l_nxt = &RequestPtr->FileXabFhc;

   if (VMSnok (status = sys$open (&RequestPtr->FileFab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$open() %%X%08.08X\n", status);

      /* if its a search list treat directory not found as if file not found */
      if ((RequestPtr->FileNam.nam$l_fnb & NAM$M_SEARCH_LIST) &&
          status == RMS$_DNF)
         status = RMS$_FNF;

      if (RequestPtr->AdjustAccounting > 1)
      {
         /* greater than one adjusts counters even if there is an error */
         Accounting.DoFileCount++;
         RequestPtr->AdjustAccounting = 0;
      }
      return (status);
   }

   if (RequestPtr->MethodGet &&
       (RequestPtr->IfModifiedSinceBinaryTime[0] ||
        RequestPtr->IfModifiedSinceBinaryTime[1]))
   {
      /*********************/
      /* if modified since */
      /*********************/

      if (VMSnok (status =
          IfModifiedSince (RequestPtr,
                           &RequestPtr->FileXabDat.xab$q_rdt,
                           &RequestPtr->IfModifiedSinceBinaryTime)))
      {
         /* task status LIB$_NEGTIM if not modified/sent */
         if (RequestPtr->AdjustAccounting)
         {
            Accounting.DoFileCount++;
            Accounting.DoFileNotModifiedCount++;
            RequestPtr->AdjustAccounting = 0;
         }
         FileEnd (RequestPtr);
         return (SS$_NORMAL);
      }

      /* make sure the "If-Modified-Since:" request header line is ignored */
      RequestPtr->IfModifiedSinceBinaryTime[0] =
         RequestPtr->IfModifiedSinceBinaryTime[1] = 0;
   }

   /*****************/
   /* gonna send it */
   /*****************/

   if (RequestPtr->AdjustAccounting)
   {
      Accounting.DoFileCount++;
      RequestPtr->AdjustAccounting = 0;
   }

   if (RequestPtr->FileEncapsulateData)
      RequestPtr->FileEscapeHtml = true;

   /*
      This module uses the same buffer space as BufferOutput().
      Better make sure its been allocated before attempting to use it!
      If it does exist and exclusive use is required then ensure its empty.
   */
   if (RequestPtr->OutputBufferPtr == NULL)
   {
      /* doesn't exist, initialize it */
      if (VMSnok (BufferOutput (RequestPtr, 0, NULL, 0)))
      {
         FileEnd (RequestPtr);
         return (SS$_NORMAL);
      }
   }
   else
   if ((RequestPtr->BinaryEncoding || RequestPtr->FileEscapeHtml) &&
       RequestPtr->OutputBufferCurrentPtr > RequestPtr->OutputBufferPtr)
   {
      /* need exclusive use, flush the current contents */
      if (VMSnok (BufferOutput (RequestPtr, 0, NULL, 0)))
      {
         FileEnd (RequestPtr);
         return (SS$_NORMAL);
      }
   }

   /*******************************/
   /* connect record access block */
   /*******************************/

   RequestPtr->FileRab = cc$rms_rab;
   /* set the RAB context to contain the request thread pointer */
   RequestPtr->FileRab.rab$l_ctx = RequestPtr;
   RequestPtr->FileRab.rab$l_fab = &RequestPtr->FileFab;
   if (RequestPtr->BinaryEncoding)
   {
      RequestPtr->FileRab.rab$l_bkt = 0;
      RequestPtr->FileRab.rab$l_rop = RAB$M_RAH | RAB$M_BIO | RAB$M_ASY;
      RequestPtr->FileRab.rab$l_ubf = RequestPtr->OutputBufferPtr;
      /* make user buffer size an even number of 512 byte blocks */
      RequestPtr->FileRab.rab$w_usz = RequestPtr->OutputBufferSize & 0xfe00;
   }
   else
   {
      /* divide buffer size by 512 */
      RequestPtr->FileRab.rab$b_mbc = RequestPtr->OutputBufferSize >> 9;
      RequestPtr->FileRab.rab$b_mbf = 2;
      RequestPtr->FileRab.rab$l_rop = RAB$M_RAH | RAB$M_ASY;
   }

   if (VMSnok (status = sys$connect (&RequestPtr->FileRab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$connect() %%X%08.08X\n", status);
      ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
      FileEnd (RequestPtr);
      return (SS$_NORMAL);
   }

   /********************/
   /* begin processing */
   /********************/

   if (RequestPtr->SendHttpHeader)
   {
      if (VMSnok (status = FileHttpHeader (RequestPtr)))
      {
         FileEnd (RequestPtr);
         return (SS$_NORMAL);
      }
      if (RequestPtr->MethodHead)
      {
         FileEnd (RequestPtr);
         return (SS$_NORMAL);
      }
   }

   if (RequestPtr->FileEncapsulateData)
   {
      if (RequestPtr->BinaryEncoding)
         AstFunctionPtr = &FileNextBlocks;
      else
         AstFunctionPtr = &FileNextRecord;

      if (RequestPtr->FileEscapeHtml)
      {
         if (VMSnok (status =
             QioNetWrite (RequestPtr, AstFunctionPtr,
                          EncapsulateDataBegin,
                          sizeof(EncapsulateDataBegin)-1)))
         {
            ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
            FileEnd (RequestPtr);
            return (SS$_NORMAL);
         }
      }
      else
      {
         if (VMSnok (status =
             BufferOutput (RequestPtr, AstFunctionPtr,
                           EncapsulateDataBegin,
                           sizeof(EncapsulateDataBegin)-1)))
         {
            FileEnd (RequestPtr);
            return (SS$_NORMAL);
         }
      }
   }
   else
   {
      if (RequestPtr->BinaryEncoding)
         FileNextBlocks (RequestPtr);
      else
         FileNextRecord (RequestPtr);
   }

   /* further processing is AST-driven */
   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
*/

FileEnd (struct RequestStruct *RequestPtr)

{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "FileEnd()\n");

   if (RequestPtr->FileFab.fab$w_ifi)
   {
      sys$close (&RequestPtr->FileFab, 0, 0);
      RequestPtr->FileFab.fab$w_ifi = 0;
   }

   if (!RequestPtr->MethodHead && RequestPtr->FileEncapsulateData)
   {
      if (VMSnok (status = 
          BufferOutput (RequestPtr, &ConcludeProcessing,
                        EncapsulateDataEnd, sizeof(EncapsulateDataEnd)-1)))
      {
          /* if the buffered output failed assume the AST was not queued */
          ConcludeProcessing (RequestPtr);
      }
   }
   else
      ConcludeProcessing (RequestPtr);

   RequestPtr->FileEncapsulateData = RequestPtr->FileEscapeHtml = false;
}

/*****************************************************************************/
/*
Generates an HTTP header for the following file transfer.
*/ 
 
int FileHttpHeader (struct RequestStruct *RequestPtr)

{
   static $DESCRIPTOR (StringDsc, "");

   static $DESCRIPTOR (CommentedInfoFaoDsc,
"<!!-- SoftwareID: !AZ -->\n\
<!!-- !AZ -->\n");

   static $DESCRIPTOR (Http200HeaderBinaryFaoDsc,
"!AZ 200 Data follows.\r\n\
Server: !AZ\r\n\
Date: !AZ\r\n\
Last-Modified: !AZ\r\n\
Content-Type: !AZ\r\n\
Content-Length: !UL\r\n\
\r\n");

   static $DESCRIPTOR (Http200HeaderTextFaoDsc,
"!AZ 200 Data follows.\r\n\
Server: !AZ\r\n\
Date: !AZ\r\n\
Last-Modified: !AZ\r\n\
Content-Type: !AZ\r\n\
\r\n");

   int  status;
   unsigned short  CommentLength,
                   Length,
                   HeaderLength;
   char  LastModified [32],
         String [1024];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "FileHttpHeader()\n");

   RequestPtr->ResponseStatusCode = 200;

   if (VMSnok (status =
       HttpGmTimeString (LastModified, &RequestPtr->FileXabDat.xab$q_rdt)))
      return (status);

   StringDsc.dsc$a_pointer = String;
   StringDsc.dsc$w_length = sizeof(String)-1;

   if (RequestPtr->BinaryEncoding)
      sys$fao (&Http200HeaderBinaryFaoDsc, &HeaderLength, &StringDsc,
               HttpProtocol, SoftwareID, RequestPtr->GmDateTime,
               LastModified, RequestPtr->ContentTypePtr,
               ((RequestPtr->FileXabFhc.xab$l_ebk - 1) << 9) +
                RequestPtr->FileXabFhc.xab$w_ffb);
   else
      sys$fao (&Http200HeaderTextFaoDsc, &HeaderLength, &StringDsc,
               HttpProtocol, SoftwareID, RequestPtr->GmDateTime,
               LastModified, RequestPtr->ContentTypePtr);

   if (Config.IncludeCommentedInfo && !RequestPtr->MethodHead &&
       strsame (RequestPtr->ContentTypePtr, Config.HtmlContentType, -1))
   {
      StringDsc.dsc$a_pointer = String + HeaderLength;
      StringDsc.dsc$w_length = sizeof(String) - HeaderLength - 1;

      sys$fao (&CommentedInfoFaoDsc, &CommentLength, &StringDsc,
               SoftwareID, RequestPtr->FileName);
   }
   else
      CommentLength = 0;

   String[Length = HeaderLength+CommentLength] = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", String);

   /* allocate heap memory for the response header */
   if ((RequestPtr->ResponseHeaderPtr =
        HeapAlloc (RequestPtr, Length)) == NULL)
   {
      ErrorHeapAlloc (RequestPtr, __FILE__, __LINE__);
      return (SS$_INSFMEM);
   }
   memcpy (RequestPtr->ResponseHeaderPtr, String, Length);

   if (VMSnok (status =
       QioNetWrite (RequestPtr, 0, String, Length)))
      ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);

   RequestPtr->SendHttpHeader = false;

   return (status);
}

/*****************************************************************************/
/*
Queue a read of the next record from the file.  When the read completes call 
FileNextRecordAST() function to post-process the read and/or send the data 
to the client.  Don't bother to test any status here, the AST routine will do 
that!
*/ 

FileNextRecord (struct RequestStruct *RequestPtr)

{
   /*********/
   /* begin */
   /*********/

/*
   if (Debug) fprintf (stdout, "FileNextRecord()\n");
*/

   RequestPtr->FileRab.rab$l_ubf = RequestPtr->OutputBufferCurrentPtr;
   RequestPtr->FileRab.rab$w_usz = RequestPtr->OutputBufferRemaining;

   /* GET the next record (or last read if RFA is set) */
   sys$get (&RequestPtr->FileRab, &FileNextRecordAST, &FileNextRecordAST);
}

/*****************************************************************************/
/*
The read of the next record from the file has completed.  Post-process and/or 
queue a network write to the client.  When the network write completes it will 
call the function FileNextRecord() to queue a read of the next record. Record-
by-record processing builds up as many file records as possible into available 
buffer space before transfering them to the network as one I/O.  The 
sys$get()s use the buffer area pointed to by
'RequestPtr->OutputBufferCurrentPtr' and the size specified by
'RequestPtr->OutputBufferRemaining'.  When a sys$get() fails with a "record 
too big for user's buffer", because we're running out of buffer space, the 
accumulated records are flushed to the network and the pointer and available 
space values are reset.  The FileNextRecord() AST called when the 
QioNetWrite() completes will re-read a record that failed with the "record too 
big" error by forcing the read to be by RFA for that one record.
*/ 

FileNextRecordAST (struct RAB *RabPtr)

{
   register int  rsz;
   register struct RequestStruct  *RequestPtr;

   int  status;

   /*********/
   /* begin */
   /*********/

/*
   if (Debug)
      fprintf (stdout,
         "FileNextRecordAST() sts: %%X%08.08X stv: %%X%08.08X rsz: %d\n",
         RabPtr->rab$l_sts,
         RabPtr->rab$l_stv,
         RabPtr->rab$w_rsz);
*/

   RequestPtr = RabPtr->rab$l_ctx;

   if (VMSnok (RequestPtr->FileRab.rab$l_sts))
   {
      /*
          Test for "record too big for user's buffer", indicates that
          its time to flush the records accumulated in the buffer.
      */
      if (RequestPtr->FileRab.rab$l_sts == RMS$_RTB)
      {
         /** if (Debug) fprintf (stdout, "RMS$_RTB\n"); **/

         if (RequestPtr->OutputBufferCurrentPtr == RequestPtr->OutputBufferPtr)
         {
            /* legitimate "record too big for user's buffer" error */
            ErrorVmsStatus (RequestPtr, RMS$_RTB, __FILE__, __LINE__);
            FileEnd (RequestPtr);
            return;
         }

         /* insufficient space for next record, flush buffer contents */
         FileFlushBuffer (RequestPtr, &FileNextRecord);
         if (RequestPtr->ErrorMessagePtr != NULL)
         {
            FileEnd (RequestPtr);
            return;
         }
            
         /* re-read the record that just failed by forcing a read by RFA */
         RequestPtr->FileRab.rab$b_rac |= RAB$C_RFA;

         return;
      }

      if (RequestPtr->FileRab.rab$l_sts == RMS$_EOF)
      {
         /** if (Debug) fprintf (stdout, "RMS$_EOF\n"); **/

         /*
            If escaping HTML-forbidden characters and some remain in
            the buffer then flush them, otherwise rely on subsequent
            processing (even thread disposal) to do it as necessary.
         */
         if (RequestPtr->FileEscapeHtml &&
             RequestPtr->OutputBufferCurrentPtr > RequestPtr->OutputBufferPtr)
            FileFlushBuffer (RequestPtr, &FileEnd);
         else
            FileEnd (RequestPtr);

         return;
      }

      ErrorVmsStatus (RequestPtr, RequestPtr->FileRab.rab$l_sts,
                      __FILE__, __LINE__);
      FileEnd (RequestPtr);
      return;
   }

   /* force the next read to be sequential (in case the last was by RFA) */
   RequestPtr->FileRab.rab$b_rac &= ~RAB$C_RFA;

   if (rsz = RequestPtr->FileRab.rab$w_rsz)
   {
      /* add newline if last character in line is not a newline! */
      if (RequestPtr->FileRab.rab$l_ubf[rsz-1] != '\n')
         RequestPtr->FileRab.rab$l_ubf[rsz++] = '\n';
   }
   else
   {
      /* must be a blank line (empty record), add a newline */
      RequestPtr->FileRab.rab$l_ubf[rsz++] = '\n';
   }

   if (RequestPtr->OutputBufferRemaining > rsz)
   {
      /* has not filled or overflowed the buffer */
      RequestPtr->OutputBufferRemaining -= rsz;
      RequestPtr->OutputBufferCurrentPtr += rsz;
      FileNextRecord (RequestPtr);
   }
   else
   {
      /** if (Debug) fprintf (stdout, "insufficient space\n"); **/

      /* insufficient space for next record, flush buffer contents */
      FileFlushBuffer (RequestPtr, &FileNextRecord);

      /* re-read the record that just failed by forcing a read by RFA */
      RequestPtr->FileRab.rab$b_rac |= RAB$C_RFA;
   }
}

/*****************************************************************************/
/*
Flush the file records accumulated in the 'Buffer'.  If plain text and 
encapsulated then escape any HTML-forbidden characters using a separate 
function.
*/ 

int FileFlushBuffer
(
struct RequestStruct *RequestPtr,
void *AstFunctionPtr
)
{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "FileFlushBuffer()\n");

   if (RequestPtr->FileEncapsulateData)
      status = FileFlushBufferEscapeHtml (RequestPtr, AstFunctionPtr);
   else
      if (VMSnok (status =
          QioNetWrite (RequestPtr,
                       AstFunctionPtr,
                       RequestPtr->OutputBufferPtr,
                       RequestPtr->OutputBufferCurrentPtr -
                          RequestPtr->OutputBufferPtr)))
         ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
   
   RequestPtr->OutputBufferCurrentPtr = RequestPtr->OutputBufferPtr;
   RequestPtr->OutputBufferRemaining = RequestPtr->OutputBufferSize;

   return (status);
}

/*****************************************************************************/
/*
Send the buffer contents escaping any HTML-forbidden characters.  Do this by
progressively allocating dynamic buffer space to buffer the characters to be
escaped, and then copying the original buffer to the dynamic buffer escaping
any forbidden characters.  Output that buffer.
*/ 

int FileFlushBufferEscapeHtml
(
struct RequestStruct *RequestPtr,
void *AstFunctionPtr
)
{
   register int  bcnt, ecnt;
   unsigned register char  *bptr, *sptr, *zptr;

   int  status;
                         
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "FileFlushBufferEscapeHtml()\n");

   bcnt = RequestPtr->OutputBufferCurrentPtr -
          (bptr = RequestPtr->OutputBufferPtr);
   if (RequestPtr->FileEscapeHtmlSize)
   {
      sptr = RequestPtr->FileEscapeHtmlPtr;
      /* 'zptr' points where there is insufficient worst-case escape space */
      zptr = RequestPtr->FileEscapeHtmlPtr + RequestPtr->FileEscapeHtmlSize - 5;
   }
   else
   {
      /* force the initial allocation of escape buffer space */
      sptr = 1;
      zptr = 0;
   }

   while (bcnt)
   {
      if (sptr > zptr)
      {
         /* a realloc() may move the memory, calculate an offset */
         if (RequestPtr->FileEscapeHtmlSize)
            ecnt = sptr - RequestPtr->FileEscapeHtmlPtr;
         else
            ecnt = 0;
         /* (re)allocate buffer space for escaping html-forbidden characters */
         if (RequestPtr->FileEscapeHtmlSize)
            RequestPtr->FileEscapeHtmlSize += RequestPtr->OutputBufferSize;
         else
            RequestPtr->FileEscapeHtmlSize = RequestPtr->OutputBufferSize * 2;
         if ((RequestPtr->FileEscapeHtmlPtr = sptr =
              HeapRealloc (RequestPtr,
                           RequestPtr->FileEscapeHtmlPtr,
                           RequestPtr->FileEscapeHtmlSize+1)) == NULL)
         {
            RequestPtr->FileEscapeHtmlSize = 0;
            ErrorHeapAlloc (RequestPtr, __FILE__, __LINE__);
            return (SS$_INSFMEM);
         }
         /* a realloc() may move the memory, recalculate the pointer */
         sptr = RequestPtr->FileEscapeHtmlPtr + ecnt;
         /* 'zptr' points where there is insufficient worst-case escape space */
         zptr = RequestPtr->FileEscapeHtmlPtr +
                RequestPtr->FileEscapeHtmlSize - 5;
      }

      switch (*bptr)
      {
         case '<' :
            memcpy (sptr, "&lt;", 4); sptr += 4; bptr++; break;
         case '>' :
            memcpy (sptr, "&gt;", 4); sptr += 4; bptr++; break;
         case '&' :
            memcpy (sptr, "&amp;", 5); sptr += 5; bptr++; break;
         default :
            *sptr++ = *bptr++;
      }
      bcnt--;
   }

   /* flush the buffer contents */
   if (VMSnok (status =
       QioNetWrite (RequestPtr, AstFunctionPtr, 
                    RequestPtr->FileEscapeHtmlPtr,
                    sptr - RequestPtr->FileEscapeHtmlPtr)))
      ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);

   return (status);
}

/*****************************************************************************/
/*
Queue a read of the next series of Virtual Blocks from the file.  When the 
read completes call FileNextBlocksAST() function to post-process the read 
and/or send the data to the client.  Don't bother to test any status here, the 
AST routine will do that!
*/ 

FileNextBlocks (struct RequestStruct *RequestPtr)

{
   /*********/
   /* begin */
   /*********/

/*
   if (Debug) fprintf (stdout, "FileNextBlocks()\n");
*/

   if (RequestPtr->FileRab.rab$l_bkt)
      RequestPtr->FileRab.rab$l_bkt += RequestPtr->OutputBufferSize >> 9;
   else
      RequestPtr->FileRab.rab$l_bkt = 1;

   sys$read (&RequestPtr->FileRab, &FileNextBlocksAST, &FileNextBlocksAST);
}

/*****************************************************************************/
/*
The read of the next series of Virtual Blocks from the file has completed.  
Post-process and/or queue a network write to the client.  When the network 
write completes it will call the function FileNextBlocks() to queue a read 
of the next series of blocks.
*/ 

FileNextBlocksAST (struct RAB *RabPtr)

{
   register struct RequestStruct  *RequestPtr;
   register int  rsz;
   register char  *ubf;
   int  status;

   /*********/
   /* begin */
   /*********/

/*
   if (Debug)
      fprintf (stdout,
               "FileNextBlocksAST() sts: %%X%08.08X stv: %%X%08.08X\n",
               RabPtr->rab$l_sts,
               RabPtr->rab$l_stv);
*/

   RequestPtr = RabPtr->rab$l_ctx;

   if (VMSnok (RequestPtr->FileRab.rab$l_sts))
   {
      if (RequestPtr->FileRab.rab$l_sts == RMS$_EOF)
      {
         FileEnd (RequestPtr);
         return;
      }
      ErrorVmsStatus (RequestPtr, RequestPtr->FileRab.rab$l_sts,
                      __FILE__, __LINE__);
      FileEnd (RequestPtr);
      return;
   }

   /* queue a network write to the client, AST to FileNextBlocks() */
   if (VMSnok (status =
       QioNetWrite (RequestPtr,
                    &FileNextBlocks,
                    RequestPtr->FileRab.rab$l_ubf,
                    RequestPtr->FileRab.rab$w_rsz)))
   {
      ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
      FileEnd (RequestPtr);
   }
}

/*****************************************************************************/

