/*****************************************************************************/
/*
                                   HTTPd.c

HFRD VMS Hypertext Transfer Protocol daemon.

This module implements a full multi-threaded, event-driven (via ASTs) network 
interface for other modules comprising the server.  This module handles all 
the TCP/IP networking functionality, connection requests, network reads and 
writes, and connection closure.

When it first starts up it binds to the HTTP server port and listens for 
connection requests.  When a request is received it checks that the client 
host is allowed access and if so queues a network read from the client to 
obtain the HTTP request.

The networking essentials are based on DEC TCP/IP Services for OpenVMS.  The 
QIO interface was obviously chosen because of its asynchronous, non-blocking 
capabilities.  Although some functionalilty could have been implemented using 
the BSD sockets abstraction and only the asynchronous I/O done using the QIO I 
opted for working out the QIOs.  It wasn't too difficult and avoids the 
(slight) extra overhead of the sockets layer.  The TGV MultiNet port follows 
the basic outline of the DEC TCP/IP functionality.  With MultiNet I couldn't 
find a way of implementing the gethostbyaddr() functionality using QIOs, as 
with DEC TCP/IP.  A slightly less elegant kludge gets around that.  I 
deliberately differentiated the DEC and TGV interfaces, even when there was 
some functional overlap (e.g. IO$_WRITEVBLK and IO$_SEND).  It wasn't much 
extra work and helps keep obvious the fact of the two APIs. 


OBLIGATORY COPYRIGHT NOTICE
---------------------------
HFRD VMS Hypertext Services, Copyright (C) 1996 Mark G.Daniel.

This package is free software; you can redistribute it and/or modify it under 
the terms of the GNU General Public License as published by the Free Software 
Foundation; version 2 of the License, or any later version. 

This package is distributed in the hope that it will be useful, but WITHOUT 
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS 
FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more 
details. 

You should have received a copy of the GNU General Public License along with 
this package; if not, write to the Free Software Foundation, Inc., 675 Mass 
Ave, Cambridge, MA 02139, USA. 


QUALIFIERS  (implemented in the CLI.c module)
----------
/CGI_PREFIX=            prefix to CGI-script variable (symbol) names
/DBUG                   turn on all "if (Debug)" statements
/DO=                    command to be passed to server
/FILBUF=                number of bytes in record buffer
/[NO]LOG[=]             logging enabled/disabled, optional log file name
/[NO]MONITOR            monitor logicals enabled/disabled
/NETBUF=                number of bytes in network read buffer
/OUTBUF=                number of bytes in output buffer
/PORT=                  server IP port number
/PRIORITY=              process priority
/SUBBUF=                number of bytes in subprocess's SYS$OUTPUT buffer
/[NO]SWAP               allow/disallow process to be swapped out


VERSION HISTORY
---------------
15-FEB-96  MGD  v3.1.1, fixed rediculous :^( bug in 302 HTTP header;
                        minor changes to request accounting and server report;
                        minor changes for user directory support;
                        minor changes to error reporting
03-JAN-96  MGD  v3.1.0, support for both DEC TCP/IP Services and TGV MultiNet
01-DEC-95  MGD  v3.0.0, single heap for each thread's dynamic memory management;
                        extensive rework of DCL subprocess functionality;
                        HTML pre-processsing module (aka Server Side Includes);
                        NCSA/CERN compliant image-mapping module;
                        BufferOutput() for improving network IO;
                        miscellaneous reworks/rewrites
27-SEP-95  MGD  v2.3.0, carriage-control on non-header records from <CR><LF> 
                        to single <LF> ('\n' ... newline); some browsers expect
                        only this (e.g. Netscape 1.n was spitting on X-bitmaps)
                        added Greenwich Mean Time time-stamp functionality;
                        added 'Referer:', 'If-Modified-Since:', 'User-Agent:'
07-AUG-95  MGD  v2.2.2, optionally include commented VMS file specifications
                        in HTML documents and VMS-style directory listings
16-JUN-95  MGD  v2.2.1, added file type description to "Index of" (directory)
24-MAY-95  MGD  v2.2.0, minor changes to allow compilation on AXP platform
03-APR-95  MGD  v2.1.0, add SYSUAF authentication, POST method handling
20-DEC-94  MGD  v2.0.0, multi-threaded version
20-JUN-94  MGD  v1.0.0, single-threaded version
*/
/*****************************************************************************/

#define HttpdVersion "v3.1.1"

/* if not MultiNet then UCX! */
#ifndef IP_MULTINET
#  define IP_UCX 1
#  define HttpdIp "DEC TCP/IP Services"
#else
#  define HttpdIp "TGV MultiNet"
#endif

#ifdef __ALPHA
#  define HttpdArch "AXP"
#else
#  define HttpdArch "VAX"
#endif

/* standard C header files */
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>

/* VMS related header files */
#include <descrip.h>
#include <dvidef.h>
#include <iodef.h>
#include <jpidef.h>
#include <prvdef.h>
#include <psldef.h>
#include <rms.h>
#include <ssdef.h>
#include <stsdef.h>
#include <syidef.h>

/* Internet-related header files */
#ifdef IP_UCX
#   include <socket.h>
#   include <in.h>
#   include <netdb.h>
#   include <inet.h>
#   include <ucx$inetdef.h>
#endif

#ifdef IP_MULTINET
#   include "multinet_root:[multinet.include.sys]types.h"
#   include "multinet_root:[multinet.include.sys]socket.h"
#   include "multinet_root:[multinet.include.vms]inetiodef.h"
#   include "multinet_root:[multinet.include.netinet]in.h"
#   include "multinet_root:[multinet.include]netdb.h"
#endif

/* application-related header files */
#include "httpd.h"
#include "copyright.h"

/******************/
/* global storage */
/******************/

/*
   These are the required privileges of the executing HTTP server.
   The server ACCOUNT should only have TMPMBX and NETMBX (just for
   extra security, policy ... keep privileged accounts to a minimum).
   The image should be installed with SYSPRV, PRMMBX, PSWAPM and SYSNAM.
   Subprocesses are spawned only with the process's current privileges,
   which are always maintained at TMPMBX and NETMBX.  If additional
   privileges are required for any particular purpose (e.g. binding to
   a privileged IP port) then they are enabled, the action performed,
   and then they are disabled again immediately.
*/
unsigned long  AvJoePrivMask [2] = { PRV$M_NETMBX | PRV$M_TMPMBX, 0 },
               ResetPrivMask [2] = { 0xffffffff, 0xffffffff };

char  ErrorFacilityDisabled [] = "Facility disabled.",
      ErrorMemoryAllocation [80] = "Memory allocation failed.", 
      ErrorRecommendBrowser [] = "\n<P><I>(indicates browser problem)</I>",
      ErrorRecommendCorrect [] = "\n<P><I>(document requires correction)</I>",
      ErrorRecommendNotify [] = "\n<P><I>(notify systems administrator)</I>",
      ErrorRedirectionCount [80] = "Redirection loop detected.",
      ErrorRequestFormat [80] = "Cannot understand HTTP request.",
      ErrorRequestHttpMethod [] = "HTTP method not implemented.",
      ErrorRequestMaxAccept [80] = "Maximum header <I>Accept:</I> exceeded.",
      ErrorStringSize [80] = "String too small.",
      ErrorTimeFmt [80] = "Time format problem.",
      Utility [] = "HTTPD";

boolean  Debug,
         MonitorEnabled = true,
         NoSwapOut = true;

int  AcceptHostsLength,
     ClientPort,
     DclSysOutputSize = DefaultDclSysOutputSize,
     ExitStatus,
     FileBufferSize = DefaultFileBufferSize,
     NetReadBufferSize = DefaultNetReadBufferSize,
     OutputBufferSize = DefaultOutputBufferSize,
     ProcessPriority = 4,
     QueueLength = 5, 
     RejectHostsLength,
     ServerPort = 80;

unsigned short  ClientChannel,
                ServerChannel;

unsigned long  CurrentConnectCount,
               MaxConcurrentConnections = DefaultMaxConcurrentConnections;

char  ClientHostName [128],
      ClientInternetAddress [32],
      CommandLine [256],
      ServerHostName [128],
      SoftwareID [64];

char  *AcceptHostsPtr = NULL,
      *RejectHostsPtr = NULL;

struct AnExitHandler  ExitHandler;

struct AccountingStruct  Accounting;

#ifdef IP_UCX
   $DESCRIPTOR (InetDeviceDsc, "UCX$DEVICE");
#endif

#ifdef IP_MULTINET
   $DESCRIPTOR (InetDeviceDsc, "INET0:");
#endif

struct AnIOsb  ServerIOsb;

struct sockaddr_in  ServerSocketName,
                    ClientSocketName;

struct hostent  *HostEntryPtr;

int  OptionEnabled = 1;

#ifdef IP_UCX

struct {
   unsigned int  Length;
   char  *Address;
} ServerSocketNameItem =
   { sizeof(ServerSocketName), &ServerSocketName };

int  ClientSocketNameLength;
struct {
   unsigned int  Length;
   char  *Address;
   unsigned int  *LengthPtr;
} ClientSocketNameItem =
   { sizeof(ClientSocketName), &ClientSocketName, &ClientSocketNameLength };

struct {
   unsigned short  Length;
   unsigned short  Parameter;
   char  *Address;
} ReuseAddress =
   { sizeof(OptionEnabled), UCX$C_REUSEADDR, &OptionEnabled },
  ReuseAddressSocketOption =
   { sizeof(ReuseAddress), UCX$C_SOCKOPT, &ReuseAddress };

unsigned short  TcpSocket [2] =
   { UCX$C_TCP, INET_PROTYP$C_STREAM };

#endif /* IP_UCX */

/********************/
/* external storage */
/********************/

extern boolean  CliLoggingDisabled;
extern boolean  CliLoggingEnabled;
extern boolean  CliMonitorDisabled;
extern boolean  CliMonitorEnabled;
extern boolean  ControlExitRequested;
extern boolean  ControlRestartRequested;
extern boolean  LoggingEnabled;
extern int  CliServerPort;
extern char  CliLogFileName[];
extern char  ControlBuffer[];
extern struct ConfigStruct  Config;

/****************************/
/* functions in this module */
/****************************/

HttpdAcceptRequest ();
HttpdConnectRequestAst ();
HttpdExit ();
HttpdHostLookup ();
HttpdListen ();
HttpdRedirect (struct RequestStruct*);
HttpdShutdownSocket (unsigned short);
TimeoutAst (struct RequestStruct*);

/***************************/
/* functions used globally */
/***************************/

int BufferOutput (struct RequestStruct*, void*, char*, int);
ConcludeProcessing (struct RequestStruct*);
NetWrite (unsigned short, char*, int);
QioNetRead (struct RequestStruct*, void*, char*, int);
QioNetWrite (struct RequestStruct*, void*, char*, int);
int TimeSetGmt ();
int TimeoutSet (struct RequestStruct*, int);

/*************************************/
/* prototypes for external functions */
/*************************************/

boolean ConfigAcceptClientHostName (char*);
DclDispose (struct RequestStruct*);
DefineMonitorLogicals (struct RequestStruct*);
ErrorHeapAlloc (struct RequestStruct*, char*, int);
ErrorExitVmsStatus (int, char*, char*, int);
ErrorGeneral (struct RequestStruct*, char*, char*, int);
ErrorSendToClient (struct RequestStruct*, void*);
ErrorServerShutdown (unsigned short);
ErrorVmsStatus (struct RequestStruct*, int, char*, int);
unsigned char* HeapAlloc (struct RequestStruct*, int);
HeapFree (struct RequestStruct*);
Logging (struct RequestStruct*, int);
MapUrl_VirtualPath (char*, char*, char*, int);
RequestGet (struct RequestStruct*);
char* SysGetMsg (int);

/*****************************************************************************/
/*
*/

int main ()

{
   static unsigned long  PswapmMask [2] = { PRV$M_PSWAPM, 0 };

   register char  *cptr;

   int  status;
   unsigned short  Length;
   char  ProcessName [16],
         TimeScratch [32];
   $DESCRIPTOR (ProcessNameDsc, ProcessName);
   $DESCRIPTOR (ProcessNameFaoDsc, "HTTPd:!UL");

   /*********/
   /* begin */
   /*********/

   sprintf (SoftwareID, "HFRD VMS HTTPd %s, %s, %s",
            HttpdVersion, HttpdArch, HttpdIp);

   if (VMSnok (status = ParseCommandLine ()))
      exit (status);
   if (!Debug)
      if (getenv ("HTTPD$DBUG") != NULL)
         Debug = true;

   /* if this image is being used to control the HTTPD process */
   if (ControlBuffer[0])
   {
      if (CliServerPort) ServerPort = CliServerPort;
      if (ServerPort < 1 || ServerPort > 65535)
         ErrorExitVmsStatus (SS$_BADPARAM, "port out of range",
                             __FILE__, __LINE__);
      exit (ControlCommand (ControlBuffer));
   }

   /****************/
   /* HTTPd server */
   /****************/

   /* output the GNU GENERAL PUBLIC LICENSE message */
   fprintf (stdout, "%%%s-I-SOFTWAREID, %s\n%s",
            Utility, SoftwareID, CopyRightMessageBrief);

   /* make sure the process' privileges are those of a mere mortal */
   status = sys$setprv (0, &ResetPrivMask, 0, 0);
   if (Debug) fprintf (stdout, "sys$setprv() %%X%08.08X\n", status);
   if (VMSnok (status))
      ErrorExitVmsStatus (status, "reseting privileges", __FILE__, __LINE__);
   status = sys$setprv (1, &AvJoePrivMask, 0, 0);
   if (Debug) fprintf (stdout, "sys$setprv() %%X%08.08X\n", status);
   if (VMSnok (status))
      ErrorExitVmsStatus (status, "default privileges", __FILE__, __LINE__);

   /* get local (server) host details */
   if ((status = gethostname (ServerHostName, sizeof(ServerHostName))) != 0)
      ErrorExitVmsStatus (0, strerror(errno), __FILE__, __LINE__);
   if (Debug) fprintf (stdout, "ServerHostName |%s|\n", ServerHostName);
   /* looks better if its all in lower case (host name is sometimes upper) */
   for (cptr = ServerHostName; *cptr; cptr++) *cptr = tolower(*cptr);

   /* read the configuration file, set any relevant global storage */
   Configure ();
   if (Config.Busy) MaxConcurrentConnections = Config.Busy;
   if (CliServerPort)
      ServerPort = CliServerPort;
   else
   if (Config.ServerPort)
      ServerPort = Config.ServerPort;
   if (ServerPort < 1 || ServerPort > 65535)
      ErrorExitVmsStatus (SS$_BADPARAM, "port out of range",
                          __FILE__, __LINE__);
   if (!CliLoggingDisabled && (Config.LoggingEnabled || CliLoggingEnabled))
      LoggingEnabled = true;
   else
      LoggingEnabled = false;
   if (!CliMonitorDisabled && (Config.MonitorEnabled || CliMonitorEnabled))
      MonitorEnabled = true;
   else
      MonitorEnabled = false;
   if (Config.ErrorRecommend)
   {
      strcat (ErrorMemoryAllocation, ErrorRecommendNotify);
      strcat (ErrorRedirectionCount, ErrorRecommendCorrect);
      strcat (ErrorRequestFormat, ErrorRecommendBrowser);
      strcat (ErrorRequestMaxAccept, ErrorRecommendNotify);
      strcat (ErrorStringSize, ErrorRecommendNotify);
      strcat (ErrorTimeFmt, ErrorRecommendCorrect);
   }

   /* set process priority */
   if (VMSnok (status = sys$setpri (0, 0, ProcessPriority, 0, 0, 0)))
      ErrorExitVmsStatus (status, "set process priority", __FILE__, __LINE__);

   /* set the process name */
   if (VMSnok (status =
       sys$fao (&ProcessNameFaoDsc, &Length, &ProcessNameDsc, ServerPort)))
      ErrorExitVmsStatus (status, "process name", __FILE__, __LINE__);
   ProcessName[ProcessNameDsc.dsc$w_length = Length] = '\0';
   if (Debug) fprintf (stdout, "ProcessName |%s|\n", ProcessName);
   if (VMSnok (status = sys$setprn (&ProcessNameDsc)))
      ErrorExitVmsStatus (status, "set process name", __FILE__, __LINE__);

   /* reset logical names provided to the HTTPd monitor utility */
   if (MonitorEnabled)
      if (VMSnok (status = DefineMonitorLogicals (NULL)))
         ErrorExitVmsStatus (status, "monitor logicals", __FILE__, __LINE__);

   /* ensure the GMT logical name is defined ok */
   if (VMSnok (status = TimeSetGmt ()))
      ErrorExitVmsStatus (status, "GMT time string", __FILE__, __LINE__);

   /* load the rule mapping database */
   MapUrl (NULL, NULL, NULL, NULL);

   /* set up to allow the HTTPD process to be controlled */
   if (VMSnok (status = ControlHttpd ()))
      ErrorExitVmsStatus (status, "control", __FILE__, __LINE__);

   /* initialize the timeout function (will exit if a problem) */
   TimeoutSet (NULL, TIMEOUT_INITIALIZE);

   /* initialize the logging */
   if (LoggingEnabled)
      if (VMSnok (status = Logging (NULL, LOGGING_BEGIN)))
         ErrorExitVmsStatus (status, "begin logging", __FILE__, __LINE__);

   /* set up and declare the exit handler */
   ExitHandler.HandlerAddress = &HttpdExit;
   ExitHandler.ArgCount = 1;
   ExitHandler.ExitStatusPtr = &ExitStatus;
   if (VMSnok (status = sys$dclexh (&ExitHandler)))
      ErrorExitVmsStatus (status, "declare exit handler", __FILE__, __LINE__);

   /* disable process swapping */
   if (NoSwapOut)
   {
      if (VMSnok (status = sys$setprv (1, &PswapmMask, 0, 0)))
         ErrorExitVmsStatus (status, "enabling PSWAPM", __FILE__, __LINE__);

      if (VMSnok (status = sys$setswm (1)))
         ErrorExitVmsStatus (status, "setting swap mode", __FILE__, __LINE__);

      if (VMSnok (status = sys$setprv (0, &PswapmMask, 0, 0)))
         ErrorExitVmsStatus (status, "disabling PSWAPM", __FILE__, __LINE__);
   }

   /* effectively an infinite loop */
   exit (HttpdListen ());
}

/*****************************************************************************/
/*
Exit the HTTP server, either via the declared exit handler, or via an explicit 
call before a sys$delprc() in ConcludeProcessing() function or the Control() 
module.
*/

HttpdExit ()

{
   if (Debug) fprintf (stdout, "HttpdExit()\n %%X%08.08X", ExitStatus);

   if (LoggingEnabled)
   {
      /* provide a server exit entry */
      Logging (NULL, LOGGING_END);
   }

   if (MonitorEnabled)
   {
      /* set exit status in logical names used by HTTPd monitor utility */
      Accounting.LastExitStatus = ExitStatus;
      DefineMonitorLogicals (NULL);
   }
}

/*****************************************************************************/
/*
Create a socket for the server to listen at.  Then in a infinite loop accept 
connections from clients.  The connection request is handled at normal 
execution level (not AST delivery level).  User-Mode AST delivery is disabled 
by an AST function called when a connection request is received.  This is done 
to ensure processing of the new connection occurs (relatively) uninterrupted.  
It is performed at normal processing level to allow host name processing to 
occur if necessary (some host database routines will not execute at AST-
delivery level).
*/ 

HttpdListen ()

{
   static unsigned long  SysPrvMask [2] = { PRV$M_SYSPRV, 0 };

   int  status;
   unsigned short  Length;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HttpdListen()\n");

   /*******************************/
   /* set up the listening socket */
   /*******************************/

   /* assign a channel to the internet template device */
   if (VMSnok (status = sys$assign (&InetDeviceDsc, &ServerChannel, 0, 0)))
      ErrorExitVmsStatus (status, "assign server channel", __FILE__, __LINE__);

    /* make the channel a TCP, connection-oriented socket */

   /* need SYSPRV to bind to a well-known port (mark as privileged) */
   if (VMSnok (status = sys$setprv (1, &SysPrvMask, 0, 0)))
      ErrorExitVmsStatus (status, "enabling SYSPRV", __FILE__, __LINE__);

#ifdef IP_UCX
   if (Debug) fprintf (stdout, "sys$qio() IO$_SETMODE\n");
   status = sys$qiow (0, ServerChannel, IO$_SETMODE, &ServerIOsb, 0, 0,
                      &TcpSocket, 0, 0, 0, &ReuseAddressSocketOption, 0);

   if (Debug)
      fprintf (stdout, "sys$qiow() %%X%08.08X IOsb.Status %%X%08.08X\n",
               status, ServerIOsb.Status);
   if (VMSok (status) && VMSnok (ServerIOsb.Status))
      status = ServerIOsb.Status;
   if (VMSnok (status))
      ErrorExitVmsStatus (status, "create server socket", __FILE__, __LINE__);
#endif

#ifdef IP_MULTINET
   if (Debug) fprintf (stdout, "sys$qio() IO$_SOCKET\n");
   status = sys$qiow (0, ServerChannel, IO$_SOCKET, &ServerIOsb, 0, 0,
                      AF_INET, SOCK_STREAM, 0, 0, 0, 0);

   if (Debug)
      fprintf (stdout, "sys$qiow() %%X%08.08X IOsb.Status %%X%08.08X\n",
               status, ServerIOsb.Status);
   if (VMSok (status) && VMSnok (ServerIOsb.Status))
      status = ServerIOsb.Status;
   if (VMSnok (status))
      ErrorExitVmsStatus (status, "create server socket", __FILE__, __LINE__);

   if (Debug) fprintf (stdout, "sys$qio() IO$_SETSOCKOPT\n");
   status = sys$qiow (0, ServerChannel, IO$_SETSOCKOPT, &ServerIOsb, 0, 0,
                      SOL_SOCKET, SO_REUSEADDR,
                      &OptionEnabled, sizeof(OptionEnabled), 0, 0);
   if (Debug)
      fprintf (stdout, "sys$qiow() %%X%08.08X IOsb.Status %%X%08.08X\n",
               status, ServerIOsb.Status);
   if (VMSok (status) && VMSnok (ServerIOsb.Status))
      status = ServerIOsb.Status;
   if (VMSnok (status))
      ErrorExitVmsStatus (status, "set socket option", __FILE__, __LINE__);
#endif

   /* bind the server socket to the host address and port */

   ServerSocketName.sin_family = AF_INET;
   ServerSocketName.sin_port = htons (ServerPort);
   ServerSocketName.sin_addr.s_addr = INADDR_ANY;

#ifdef IP_UCX
   if (Debug) fprintf (stdout, "sys$qiow() IO$_SETMODE\n");
   status = sys$qiow (0, ServerChannel, IO$_SETMODE, &ServerIOsb, 0, 0,
                      0, 0, &ServerSocketNameItem, QueueLength, 0, 0);
   if (Debug)
      fprintf (stdout, "sys$qiow() %%X%08.08X IOsb.Status %%X%08.08X\n",
               status, ServerIOsb.Status);
   if (VMSok (status) && VMSnok (ServerIOsb.Status))
      status = ServerIOsb.Status;
   if (VMSnok (status))
      ErrorExitVmsStatus (status, "bind server socket", __FILE__, __LINE__);
#endif

#ifdef IP_MULTINET
   if (Debug) fprintf (stdout, "sys$qiow() IO$_BIND\n");
   status = sys$qiow (0, ServerChannel, IO$_BIND, &ServerIOsb, 0, 0,
                      &ServerSocketName, sizeof(ServerSocketName), 0, 0, 0, 0);
   if (Debug)
      fprintf (stdout, "sys$qiow() %%X%08.08X IOsb.Status %%X%08.08X\n",
               status, ServerIOsb.Status);
   if (VMSok (status) && VMSnok (ServerIOsb.Status))
      status = ServerIOsb.Status;
   if (VMSnok (status))
      ErrorExitVmsStatus (status, "bind server socket", __FILE__, __LINE__);

   if (Debug) fprintf (stdout, "sys$qio() IO$_LISTEN\n");
   status = sys$qiow (0, ServerChannel, IO$_LISTEN, &ServerIOsb, 0, 0,
                      QueueLength, 0, 0, 0, 0, 0);
   if (Debug)
      fprintf (stdout, "sys$qiow() %%X%08.08X IOsb.Status %%X%08.08X\n",
               status, ServerIOsb.Status);
   if (VMSok (status) && VMSnok (ServerIOsb.Status))
      status = ServerIOsb.Status;
   if (VMSnok (status))
      ErrorExitVmsStatus (status, "listen server socket", __FILE__, __LINE__);
#endif

   /* turn off SYSPRV after binding to well-known port */
   if (VMSnok (status = sys$setprv (0, &SysPrvMask, 0, 0)))
      ErrorExitVmsStatus (status, "disabling SYSPRV", __FILE__, __LINE__);

   /************************************************/
   /* loop listening for and accepting connections */
   /************************************************/

   for (;;)
   {
      /* assign a channel to the internet template device */
      if (VMSnok (status = sys$assign (&InetDeviceDsc, &ClientChannel, 0, 0)))
         ErrorExitVmsStatus (status, "assign client channel",
                             __FILE__, __LINE__);

#ifdef IP_UCX
      if (Debug) fprintf (stdout, "sys$qio() IO$_ACCESS | IO$M_ACCEPT\n");
      status = sys$qio (0, ServerChannel, IO$_ACCESS | IO$M_ACCEPT,
                        &ServerIOsb, &HttpdConnectRequestAst, 0,
                        0, 0, &ClientSocketNameItem, &ClientChannel, 0, 0);
#endif

#ifdef IP_MULTINET
      if (Debug) fprintf (stdout, "sys$qio() IO$_ACCEPT\n");
      status = sys$qio (0, ClientChannel, IO$_ACCEPT,
                        &ServerIOsb, &HttpdConnectRequestAst, 0,
                        &ClientSocketName, sizeof(ClientSocketName),
                        ServerChannel, 0, 0, 0);
#endif

      if (Debug) fprintf (stdout, "sys$qio() %%X%08.08X\n", status);
      if (VMSnok (status)) 
         ErrorExitVmsStatus (status, "accept", __FILE__, __LINE__);

      /* reenable user-mode ASTs (needed on subsequent times through loop) */
      sys$setast (1);

      /* wait for a connection request */
      sys$hiber ();

      /*******************************/
      /* connection request received */
      /*******************************/

      /* NOTE: user-mode ASTs are now disabled (by HttpdConnectRequestAst()) */

      if (Debug)
         fprintf (stdout, "sys$qio() IOsb.Status %%X%08.08X\n",
                  ServerIOsb.Status);
      if (VMSnok (ServerIOsb.Status)) 
         ErrorExitVmsStatus (ServerIOsb.Status, "accept", __FILE__, __LINE__);

      Accounting.ConnectCount++;

      strcpy (ClientInternetAddress, inet_ntoa (ClientSocketName.sin_addr));
      /** ClientPort = ntohs (ClientSocketName.sin_port); **/
      if (Debug) fprintf (stdout, "|%s|\n", ClientInternetAddress);
                                                                        
      if (Config.DnsLookup)
         HttpdHostLookup ();
      else
         strcpy (ClientHostName, ClientInternetAddress);
      if (Debug) fprintf (stdout, "|%s|\n", ClientHostName);

      if (ControlExitRequested || ControlRestartRequested)
      {
         Accounting.ResponseStatusCodeCountArray[5]++;
         ErrorServerShutdown (ClientChannel);
         HttpdShutdownSocket (ClientChannel);
      }
      else
      if (CurrentConnectCount >= MaxConcurrentConnections)
      {
         Accounting.ConnectTooBusyCount++;
         Accounting.ResponseStatusCodeCountArray[5]++;
         ErrorTooBusy (ClientChannel);
         HttpdShutdownSocket (ClientChannel);
      }
      else
      if (ConfigAcceptClientHostName (ClientHostName))
      {
         Accounting.ConnectAcceptedCount++;
         HttpdAcceptRequest ();
      }
      else
      {
         Accounting.ConnectRejectedCount++;
         Accounting.ResponseStatusCodeCountArray[4]++;
         ErrorAccessDenied (ClientChannel);
         HttpdShutdownSocket (ClientChannel);
      }
   }
}

/*****************************************************************************/
/*
Disable user-mode ASTs at the start of connection acceptance processing and 
wake the hibernating process.
*/ 

HttpdConnectRequestAst ()

{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HttpdConnectRequestAst()\n");

   sys$setast (0);
   sys$wake (0, 0);
}

/*****************************************************************************/
/*
Perform the UCX get host name using QIO ACP functionality.  This can execute 
at AST-delivery level.  However, the only way I found of doing this within 
MultiNet was to use the gethostbyaddr() function, which seems to require USER-
mode ASTs to be enabled (not altogether surprising seeing it probably wasn't 
designed to be non-blocking).  Hence this kludge.  Turning on user-mode ASTs 
around gethostbyaddr() can slow down new connection processing significantly 
as AST-delivery is pre-emptive.  HttpdHostLookup() is not AST-driven and hence 
can introduce some granularity, particularly if DNS is involved in resolving 
the host name.
*/ 

HttpdHostLookup () 

{
#ifdef IP_UCX
   static unsigned char ControlSubFunction [4] =
      { INETACP_FUNC$C_GETHOSTBYADDR, INETACP$C_TRANS, 0, 0 };
   static struct dsc$descriptor AddressDsc =
      { 4, DSC$K_CLASS_S, DSC$K_DTYPE_T, &ClientSocketName.sin_addr };
   static struct dsc$descriptor ControlSubFunctionDsc =
      { 4, DSC$K_CLASS_S, DSC$K_DTYPE_T, &ControlSubFunction };
   static struct dsc$descriptor HostNameDsc =
      { sizeof(ClientHostName)-1, DSC$K_CLASS_S, DSC$K_DTYPE_T,
        &ClientHostName };
#endif

   int  status;
   unsigned short  Length;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HttpdHostLookup()\n");

#ifdef IP_UCX
   Length = 0;
   if (Debug) fprintf (stdout, "sys$qio() IO$_ACPCONTROL\n");
   status = sys$qiow (0, ServerChannel, IO$_ACPCONTROL, &ServerIOsb, 0, 0,
                      &ControlSubFunctionDsc,
                      &AddressDsc, &Length, &HostNameDsc, 0, 0);
   if (Debug)
      fprintf (stdout, "sys$qio() %%X%08.08X IOsb.Status %%X%08.08X\n",
               status, ServerIOsb.Status);
   if (Length)
      ClientHostName[Length] = '\0';
   else
      strcpy (ClientHostName, ClientInternetAddress);
#endif

#ifdef IP_MULTINET
   /* reenable user-mode ASTs, to allow gethostbyaddr() to work */
   sys$setast (1);

   if (Debug) fprintf (stdout, "gethostbyaddr()\n");
   HostEntryPtr = gethostbyaddr (&ClientSocketName.sin_addr,
                                 sizeof(ClientSocketName.sin_addr),
                                 AF_INET);

   /* re-disable user-mode ASTs */
   sys$setast (0);

   if (Debug) fprintf (stdout, "HostEntryPtr: %d\n", HostEntryPtr);
   if (HostEntryPtr == NULL)
      strcpy (ClientHostName, ClientInternetAddress);
   else
      strcpy (ClientHostName, HostEntryPtr->h_name);
#endif
}

/*****************************************************************************/
/*
Accepted a connection from a client.  Allocate dynamic memory for request data 
structure and buffers.  Queue a read from the network connection.  Call an AST 
function when the read completes, to process the request.
*/ 

HttpdAcceptRequest ()

{
   int  status;
   struct RequestStruct  *RequestPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HttpdAcceptRequest() %s\n", ClientHostName);

   /* allocate zeroed dynamic memory for the client thread */
   if ((RequestPtr = calloc (1, sizeof(struct RequestStruct))) == NULL)
   {
      status = vaxc$errno;
      if (Debug) fprintf (stdout, "calloc() %%X%08.08X\n", status);
      ErrorRequestCallocFailed (ClientChannel, status, __FILE__, __LINE__);
      HttpdShutdownSocket (ClientChannel);
      Accounting.ConnectCallocFailCount++;
      Accounting.ResponseStatusCodeCountArray[5]++;
      return;
   }

   /* once the thread data structure has been created we have a client! */
   Accounting.ConnectCurrent = ++CurrentConnectCount;
   if (CurrentConnectCount > Accounting.ConnectPeak)
      Accounting.ConnectPeak = CurrentConnectCount;

   /* initialize non-false, non-zero members of the structure */
   RequestPtr->ClientChannel = ClientChannel;
   strcpy (RequestPtr->ClientInternetAddress, ClientInternetAddress);
   strcpy (RequestPtr->ClientHostName, ClientHostName);
   /* a value of one adjusts server counters upon successful operation */
   RequestPtr->AdjustAccounting = 1;
   RequestPtr->AutoScriptEnabled = RequestPtr->SendHttpHeader = true;
   /* initialize the dynamic memory heap */
   RequestPtr->HeapHeadPtr = RequestPtr->HeapTailPtr = NULL;
   RequestPtr->HeapSize = 0;
   /* initialize the output buffer storage */
   RequestPtr->OutputBufferCurrentPtr = RequestPtr->OutputBufferPtr = NULL;
   /* timestamp the transaction */
   sys$gettim (&RequestPtr->BinaryTime);
   sys$numtim (&RequestPtr->NumericTime, &RequestPtr->BinaryTime);
   HttpGmTimeString (RequestPtr->GmDateTime, &RequestPtr->BinaryTime);

   /* initialize the timer for the initial request receipt timeout */
   if (VMSnok (status = TimeoutSet (RequestPtr, TIMEOUT_INPUT)))
   {
      ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
      ConcludeProcessing (RequestPtr);
      return;
   }

   /* 
      Queue an asynchronous read to get the request from the client.
      When the read completes call RequestGet() completion function to
      further process the request.  Allow one byte for terminating null.
   */
   if ((RequestPtr->NetReadBufferPtr =
       HeapAlloc (RequestPtr, NetReadBufferSize+1)) == NULL)
   {
      ErrorHeapAlloc (RequestPtr, __FILE__, __LINE__);
      ConcludeProcessing (RequestPtr);
      return;
   }
   RequestPtr->NetReadBufferSize = NetReadBufferSize;

   if (VMSnok (status = QioNetRead (RequestPtr,
                                    &RequestGet,
                                    RequestPtr->NetReadBufferPtr,
                                    RequestPtr->NetReadBufferSize)))
   {
      /* initial IO failed, abandon the request */
      ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
      ConcludeProcessing (RequestPtr);
      return;
   }
}

/****************************************************************************/
/*
*/

HttpdShutdownSocket (unsigned short Channel)

{
   int  status;
   struct  AnIOsb  IOsb;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HttpdShutdownSocket() %d\n", Channel);

#ifdef IP_UCX
      status = sys$qiow (0, Channel, IO$_DEACCESS | IO$M_SHUTDOWN, &IOsb, 0, 0,
                         0, 0, 0, UCX$C_DSC_RCV, 0, 0);
#endif

#ifdef IP_MULTINET
      status = sys$qiow (0, Channel, IO$_SHUTDOWN, &IOsb, 0, 0,
                         0, 0, 0, 0, 0, 0);
#endif

   if (Debug)
      fprintf (stdout, "sys$qiow() %%X%08.08X IOsb.Status %%X%08.08X\n",
               status, ServerIOsb.Status);

   status = sys$dassgn (Channel);
   if (Debug) fprintf (stdout, "sys$dassgn() %%X%08.08X\n", status);
}

/****************************************************************************/
/*
The 'RequestPtr->LocationPtr' is non-NULL indicating redirection.  This 
can be local, where the local server (this) processes a partial-URL, indicated 
by a leading '/', or it can be non-local, where the URL is passed back to the 
client for processing, indicated by anything other than a leading '/', usually 
a full URL including scheme (e.g. "http:").  Return normal status to indicate 
local-redirection (and that the thread should NOT be disposed of), error 
status to indicate client-handled, non-local-redirection, or a legitimate 
error (and that the thread can go).
*/ 

int HttpdRedirect (struct RequestStruct *RequestPtr)

{
   static $DESCRIPTOR (Http302HeadFaoDsc,
"!AZ 302 Redirection.\r\n\
Server: !AZ\r\n\
Date: !AZ\r\n\
Location: !AZ\r\n\
\r\n");

   static $DESCRIPTOR (Http302FaoDsc,
"!AZ 302 Redirection.\r\n\
Server: !AZ\r\n\
Date: !AZ\r\n\
Location: !AZ\r\n\
\r\n\
<H1>Redirection</H1>\n\
<P>You are only seeing this message \
because your browser cannot auto-<I>magically</I> redirect a request.\n\
<P>The URL you seek is <A HREF=\"!AZ\"><TT>!AZ</TT></A>\n");

   register char  *cptr, *sptr, *zptr;

   int  status;
   unsigned short  Length;
   char  RedirectionUrl [1024];
   $DESCRIPTOR (NetReadBufferDsc, "");

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "HttpdRedirect() |%s|\n", RequestPtr->LocationPtr);

   if (RequestPtr->RedirectionCount++ > 1)
   {
      RequestPtr->ResponseStatusCode = 501;
      ErrorGeneral (RequestPtr, ErrorRedirectionCount, __FILE__, __LINE__);
      return (STS$K_ERROR);
   }

   RequestPtr->ResponseStatusCode = 302;

   MapUrl_VirtualPath (RequestPtr->PathInfoPtr,
                       RequestPtr->LocationPtr,
                       RedirectionUrl, sizeof(RedirectionUrl));

   if (RedirectionUrl[0] == '/')
   {
      /*********************/
      /* local redirection */
      /*********************/

      Accounting.RedirectLocalCount++;

      sptr = RequestPtr->NetReadBufferPtr;
      zptr = RequestPtr->NetReadBufferPtr + RequestPtr->NetReadBufferSize;

      cptr = RequestPtr->Method;
      while (*cptr && sptr < zptr) *sptr++ = *cptr++;
      if (sptr < zptr) *sptr++ = ' ';
      cptr = RedirectionUrl;
      while (*cptr && sptr < zptr) *sptr++ = *cptr++;
      if (sptr < zptr) *sptr++ = ' ';
      cptr = HttpProtocol;
      while (*cptr && sptr < zptr) *sptr++ = *cptr++;
      if (sptr < zptr) *sptr++ = '\r';
      if (sptr < zptr) *sptr++ = '\n';

      if (RequestPtr->HttpUserAgentPtr != NULL)
      {
         cptr = "User-Agent: ";
         while (*cptr && sptr < zptr) *sptr++ = *cptr++;
         cptr = RequestPtr->HttpUserAgentPtr;
         while (*cptr && sptr < zptr) *sptr++ = *cptr++;
         if (sptr < zptr) *sptr++ = '\r';
         if (sptr < zptr) *sptr++ = '\n';
      }
      if (RequestPtr->HttpRefererPtr != NULL)
      {
         cptr = "Referer: ";
         while (*cptr && sptr < zptr) *sptr++ = *cptr++;
         cptr = RequestPtr->HttpRefererPtr;
         while (*cptr && sptr < zptr) *sptr++ = *cptr++;
         if (sptr < zptr) *sptr++ = '\r';
         if (sptr < zptr) *sptr++ = '\n';
      }
      if (RequestPtr->HttpAuthorizationPtr != NULL)
      {
         cptr = "Authorization: ";
         while (*cptr && sptr < zptr) *sptr++ = *cptr++;
         cptr = RequestPtr->HttpAuthorizationPtr;
         while (*cptr && sptr < zptr) *sptr++ = *cptr++;
         if (sptr < zptr) *sptr++ = '\r';
         if (sptr < zptr) *sptr++ = '\n';
      }

      /* now ignore old value of location redirection pointer */
      RequestPtr->LocationPtr = NULL;

      /* blank line ending header */
      if (sptr < zptr) *sptr++ = '\r';
      if (sptr < zptr) *sptr++ = '\n';
      if (sptr < zptr)
      {
         *sptr = '\0';

         /* now point this at the network read buffer */
         RequestPtr->RequestHeaderPtr = RequestPtr->NetReadBufferPtr;
         RequestPtr->RequestHeaderLength =
            sptr - (char*)RequestPtr->NetReadBufferPtr;

         return (SS$_NORMAL);
      }
      else
      {
         RequestPtr->ResponseStatusCode = 500;
         ErrorGeneral (RequestPtr, ErrorStringSize, __FILE__, __LINE__);
         return (STS$K_ERROR);
      }
   }
   else
   {
      /*************************/
      /* non-local redirection */
      /*************************/

      Accounting.RedirectRemoteCount++;

      /* now ignore old value of location redirection pointer */
      RequestPtr->LocationPtr = NULL;

      NetReadBufferDsc.dsc$a_pointer = RequestPtr->NetReadBufferPtr;
      NetReadBufferDsc.dsc$w_length = RequestPtr->NetReadBufferSize;

      if (RequestPtr->MethodHead)
          status = sys$fao (&Http302HeadFaoDsc, &Length, &NetReadBufferDsc,
                            HttpProtocol, SoftwareID,
                            RequestPtr->GmDateTime, RedirectionUrl);
      else
          status = sys$fao (&Http302FaoDsc, &Length, &NetReadBufferDsc,
                            HttpProtocol, SoftwareID, RequestPtr->GmDateTime,
                            RedirectionUrl, RedirectionUrl, RedirectionUrl);
      if (VMSnok (status))
         ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
      else
      if (VMSnok (status =
          QioNetWrite (RequestPtr, &ConcludeProcessing,
                       RequestPtr->NetReadBufferPtr, Length)))
         ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);

      return (STS$K_ERROR);
   }
}

/*****************************************************************************/
/*
Write 'DataLength' bytes located at 'DataPtr' via the network to the client.  
If 'AddressOfCompletionAST' is zero then no I/O completion AST routine is 
called.  If it is non-zero then the function pointed to by the parameter is 
called when the network write completes.
*/ 

QioNetWrite
(
struct RequestStruct *RequestPtr,
void *AddressOfCompletionAST,
char *DataPtr,
int DataLength
)
{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "QioNetWrite() %d bytes\n", DataLength);
   /** if (Debug) fprintf (stdout, "|%s|\n", DataPtr); **/

#ifdef IP_UCX
   status = sys$qio (0, RequestPtr->ClientChannel,
                     IO$_WRITEVBLK, &RequestPtr->NetWriteIOsb,
                     AddressOfCompletionAST, RequestPtr,
                     DataPtr, DataLength, 0, 0, 0, 0);
#endif

#ifdef IP_MULTINET
   status = sys$qio (0, RequestPtr->ClientChannel,
                     IO$_SEND, &RequestPtr->NetWriteIOsb,
                     AddressOfCompletionAST, RequestPtr,
                     DataPtr, DataLength, 0, 0, 0, 0);
#endif

   if (VMSok (status)) RequestPtr->BytesTx += DataLength;
   return (status);
}

/*****************************************************************************/
/*
Unlike NetWriteQio() this function writes to a client channel, not using the 
full request data structure.  This is for writing to the client before any of 
the request data structures are created (e.g. when the server denies access).
This is a blocking function, but shouldn't introduce much granularity as it is
only used for short messages before a client thread is set up.
*/ 

NetWrite
(
unsigned short Channel,
char *DataPtr,
int DataLength
)
{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "QioNetWrite()\n");

#ifdef IP_UCX
   status = sys$qiow (0, Channel, IO$_WRITEVBLK, 0, 0, 0,
                      DataPtr, DataLength, 0, 0, 0, 0);
#endif

#ifdef IP_MULTINET
   status = sys$qiow (0, Channel, IO$_SEND, 0, 0, 0,
                      DataPtr, DataLength, 0, 0, 0, 0);
#endif

   return (status);
}

/*****************************************************************************/
/*
Queue up a read from the client over the network. If 'AddressOfCompletionAST' 
is zero then no I/O completion AST routine is called.  If it is non-zero then 
the function pointed to by the parameter is called when the network write 
completes.
*/ 

QioNetRead
(
struct RequestStruct *RequestPtr,
void *AddressOfCompletionAST,
char *DataBuffer,
int SizeofDataBuffer
)
{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "QioNetRead()\n");

#ifdef IP_UCX
   status = sys$qio (0, RequestPtr->ClientChannel,
                     IO$_READVBLK, &RequestPtr->NetReadIOsb,
                     AddressOfCompletionAST, RequestPtr,
                     DataBuffer, SizeofDataBuffer, 0, 0, 0, 0);
#endif

#ifdef IP_MULTINET
   status = sys$qio (0, RequestPtr->ClientChannel,
                     IO$_RECEIVE, &RequestPtr->NetReadIOsb,
                     AddressOfCompletionAST, RequestPtr,
                     DataBuffer, SizeofDataBuffer, 0, 0, 0, 0);
#endif

   return (status);
}

/*****************************************************************************/
/*
This (moderately complex) function buffers multiple 'records' (usually lines 
of text) before writing them as one, larger 'record' via the network to the 
client.  It will escape HTML-forbidden characters if the appropriate flag is 
set. 

It implements a "double-buffering" scheme that requires an AST to be supplied 
with every call supplying data to be output (i.e. every call EXCEPT an initial 
allocation call or explicit buffer flush), AND that no more output be buffered 
until that AST is delivered (i.e. only one BufferOutput() per AST-called 
function).  This "double-buffering" allows the buffer to "overflow" and be 
written via the network to the client without overwriting that buffered data 
when buffering the new data, and with processing remaining correct for an AST-
driven environment.  This single BufferOut() call per AST routine is very 
critical to the integrity of the functionality! 

Uses the thread structure's 'OutputBufferPtr', 'OutputBufferCurrentPtr' and
'OutputBufferRemaining' storage.  Call with 'OutputPtr' set to NULL to
explicitly flush the buffer at any time.  For uniformity of behaviour, an AST 
function MAY be declared with any call, if it's not employed in an actual 
network write, its explicitly declared.  An AST always sets the network write 
IOsb (it's fudged as SS$_NORMAL normal with declared ASTs to simulate an
actual AST-driven write).
*/ 

int BufferOutput
(
struct RequestStruct *RequestPtr,
void *AstFunctionPtr,
char *OutputPtr,
int OutputLength
)
{
   int  status;
   unsigned char  *TempDoubleBufferPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "BufferOutput()\n");

   if (RequestPtr->OutputBufferPtr == NULL)
   {
      /**********************************************/
      /* allocate heap memory for the output buffer */
      /**********************************************/

      if (Debug) fprintf (stdout, "allocate\n");

      /* allow two bytes for carriage control and null terminator */
      if ((RequestPtr->OutputBufferPtr =
          HeapAlloc (RequestPtr, OutputBufferSize+2)) == NULL)
      {
         ErrorHeapAlloc (RequestPtr, __FILE__, __LINE__);
         return (SS$_INSFMEM);
      }
      RequestPtr->OutputBufferCurrentPtr = RequestPtr->OutputBufferPtr;
      RequestPtr->OutputBufferRemaining =
         RequestPtr->OutputBufferSize = OutputBufferSize;

      /* if nothing to output, must just be initializing the output buffer */
      if (OutputPtr == NULL)
      {
         if (AstFunctionPtr)
         {
            /* bit strange to initialize with an AST, but declare it anyway! */
            RequestPtr->NetWriteIOsb.Status = SS$_NORMAL;
            if (VMSnok (status = sys$dclast (AstFunctionPtr, RequestPtr, 0)))
               ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
            if (Debug) fprintf (stdout, "sys$dclast() %%X%08.08X\n", status);
         }
         else
            status = SS$_NORMAL;

         return (status);
      }
   }

   if (Debug)
      RequestPtr->OutputBufferPtr[RequestPtr->OutputBufferCurrentPtr -
                                  RequestPtr->OutputBufferPtr] = '\0';

   if (OutputPtr == NULL)
   {
      /*************************/
      /* flush buffer contents */
      /*************************/

      if (Debug) fprintf (stdout, "flush\n|%s|\n", RequestPtr->OutputBufferPtr);

      if (RequestPtr->OutputBufferCurrentPtr > RequestPtr->OutputBufferPtr)
      {
         if (VMSnok (status =
             QioNetWrite (RequestPtr, AstFunctionPtr,
                          RequestPtr->OutputBufferPtr,
                          RequestPtr->OutputBufferCurrentPtr -
                             RequestPtr->OutputBufferPtr)))
            ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
      }
      else
      if (AstFunctionPtr)
      {
         /* no network write actually occured, declare an AST to do it */
         RequestPtr->NetWriteIOsb.Status = SS$_NORMAL;
         if (VMSnok (status = sys$dclast (AstFunctionPtr, RequestPtr, 0)))
            ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
         if (Debug) fprintf (stdout, "sys$dclast() %%X%08.08X\n", status);
      }
      else
         status = SS$_NORMAL;

      RequestPtr->OutputBufferCurrentPtr = RequestPtr->OutputBufferPtr;
      RequestPtr->OutputBufferRemaining = OutputBufferSize;

      return (status);
   }

   if (Debug)
      fprintf (stdout, "%d (%d)\n|%s|\n",
               OutputLength, RequestPtr->BufferOutputEscapeHtml, OutputPtr);

   if (RequestPtr->BufferOutputEscapeHtml)
   {
      /************************************/
      /* escape HTML-forbidden characters */
      /************************************/

      /* local storage */
      register int  bcnt, ocnt;
      register unsigned char  *bptr, *optr;

      if (Debug) fprintf (stdout, "escape-HTML\n");

      TempDoubleBufferPtr = NULL;
      optr = OutputPtr;
      ocnt = OutputLength;
      bptr = RequestPtr->OutputBufferCurrentPtr;
      bcnt = RequestPtr->OutputBufferRemaining;

      while (ocnt)
      {
         /* worst-case is having to escape an ampersand */
         if (bcnt < 5)
         {
            if (TempDoubleBufferPtr != NULL)
            {
               /*
                  If there should be so many HTML-escaped characters that
                  the output buffer needs flushing twice ... then don't,
                  its not designed to allow that.  Instead wriggle out of
                  it gracefully (as possible), returning an error message.
                  This should seldom happen, if at all!
               */
               ErrorVmsStatus (RequestPtr, RMS$_RTB, __FILE__, __LINE__);
               return (RMS$_RTB);
            }

            /* only provide the double-buffering when necessary */
            if (RequestPtr->OutputDoubleBufferPtr == NULL)
            {
               if (Debug) fprintf (stdout, "double-buffering\n");
               if ((RequestPtr->OutputDoubleBufferPtr =
                   HeapAlloc (RequestPtr, RequestPtr->OutputBufferSize+2))
                  == NULL)
               {
                  ErrorHeapAlloc (RequestPtr, __FILE__, __LINE__);
                  return (SS$_INSFMEM);
               }
            }

            status = QioNetWrite (RequestPtr, AstFunctionPtr,
                                  RequestPtr->OutputBufferPtr,
                                  bptr - RequestPtr->OutputBufferPtr);

            TempDoubleBufferPtr = RequestPtr->OutputDoubleBufferPtr;
            RequestPtr->OutputDoubleBufferPtr = RequestPtr->OutputBufferPtr;
            RequestPtr->OutputBufferPtr = TempDoubleBufferPtr;

            bptr = RequestPtr->OutputBufferPtr;
            bcnt = OutputBufferSize;

            if (VMSnok (status))
            {
               ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
               return (status);
            }

            /* ensure any AST function is queued only once! */
            AstFunctionPtr = 0;
         }

         switch (*optr)
         {
            case '<' :
               memcpy (bptr, "&lt;", 4); bptr += 4; bcnt -= 4; optr++; break;
            case '>' :
               memcpy (bptr, "&gt;", 4); bptr += 4; bcnt -= 4; optr++; break;
            case '&' :
               memcpy (bptr, "&amp;", 5); bptr += 5; bcnt -= 5; optr++; break;
            default :
               *bptr++ = *optr++; bcnt--;
         }
         ocnt--;
      }

      RequestPtr->OutputBufferCurrentPtr = bptr;
      RequestPtr->OutputBufferRemaining = bcnt;
   }
   else
   {
      /*****************/
      /* buffer output */
      /*****************/

      if (OutputLength > RequestPtr->OutputBufferRemaining)
      {
         /************************************/
         /* insufficient space, flush buffer */
         /************************************/

         if (Debug)
            fprintf (stdout, "full-flush\n|%s|\n", RequestPtr->OutputBufferPtr);

         if (OutputLength > RequestPtr->OutputBufferSize)
         {
            /* record is actually too big to buffer! */
            ErrorVmsStatus (RequestPtr, RMS$_RTB, __FILE__, __LINE__);
            return (RMS$_RTB);
         }

         /* only provide the double-buffering when necessary */
         if (RequestPtr->OutputDoubleBufferPtr == NULL)
         {
            if (Debug) fprintf (stdout, "double-buffering\n");
            if ((RequestPtr->OutputDoubleBufferPtr =
                HeapAlloc (RequestPtr, RequestPtr->OutputBufferSize+2))
               == NULL)
            {
               ErrorHeapAlloc (RequestPtr, __FILE__, __LINE__);
               return (SS$_INSFMEM);
            }
         }

         status = QioNetWrite (RequestPtr, AstFunctionPtr,
                               RequestPtr->OutputBufferPtr,
                               RequestPtr->OutputBufferCurrentPtr -
                                  RequestPtr->OutputBufferPtr);

         TempDoubleBufferPtr = RequestPtr->OutputDoubleBufferPtr;
         RequestPtr->OutputDoubleBufferPtr = RequestPtr->OutputBufferPtr;
         RequestPtr->OutputBufferPtr = TempDoubleBufferPtr;

         RequestPtr->OutputBufferCurrentPtr = RequestPtr->OutputBufferPtr;
         RequestPtr->OutputBufferRemaining = OutputBufferSize;

         if (VMSnok (status))
         {
            ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
            return (status);
         }

         /* ensure any AST function is queued only once! */
         AstFunctionPtr = 0;
      }

      /* buffer new output */
      memcpy (RequestPtr->OutputBufferCurrentPtr, OutputPtr, OutputLength);
      RequestPtr->OutputBufferCurrentPtr += OutputLength;
      RequestPtr->OutputBufferRemaining -= OutputLength;
   }

   if (AstFunctionPtr)
   {
      /******************/
      /* declare an AST */
      /******************/

      /* no network write could have occured, declare an AST to do it */
      RequestPtr->NetWriteIOsb.Status = SS$_NORMAL;
      if (VMSnok (status = sys$dclast (AstFunctionPtr, RequestPtr, 0)))
      {
         ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
         return (status);
      }
      if (Debug) fprintf (stdout, "sys$dclast() %%X%08.08X\n", status);
   }

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
Input and output timeout timers.  Will exit if the initialization encounters 
an error.  Outstanding timers will be cancelled by ConcludeProcessing().
*/

int TimeoutSet
(
struct RequestStruct *RequestPtr,
int Function
)
{
   static unsigned long  InputTimeoutDeltaBinTime [2],
                         OutputTimeoutDeltaBinTime [2];
   static char  InputTimeout [32],
                OutputTimeout [32];
   static $DESCRIPTOR (InputTimeoutDsc, InputTimeout);
   static $DESCRIPTOR (OutputTimeoutDsc, OutputTimeout);
   static $DESCRIPTOR (TimeoutFaoDsc, "0 00:!UL:00.00");

   int  status;
   unsigned short  Length;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "TimeoutSet() %d\n", Function);

   switch (Function)
   {
      case TIMEOUT_INITIALIZE :

         if (!Config.InputTimeoutMinutes)
            Config.InputTimeoutMinutes = DefaultInputTimeoutMinutes;
         if (VMSnok (status =
             sys$fao (&TimeoutFaoDsc, &Length, &InputTimeoutDsc,
                      Config.InputTimeoutMinutes)))
            ErrorExitVmsStatus (status, "input timeout", __FILE__, __LINE__);
         InputTimeout[InputTimeoutDsc.dsc$w_length = Length] = '\0';
         if (Debug) fprintf (stdout, "InputTimeout |%s|\n", InputTimeout);
         if (VMSnok (status =
             sys$bintim (&InputTimeoutDsc, &InputTimeoutDeltaBinTime)))
            ErrorExitVmsStatus (status, "input timeout", __FILE__, __LINE__);

         if (!Config.OutputTimeoutMinutes)
            Config.OutputTimeoutMinutes = DefaultOutputTimeoutMinutes;
         if (VMSnok (status =
             sys$fao (&TimeoutFaoDsc, &Length, &OutputTimeoutDsc,
                      Config.OutputTimeoutMinutes)))
            ErrorExitVmsStatus (status, "output timeout", __FILE__, __LINE__);
         OutputTimeout[OutputTimeoutDsc.dsc$w_length = Length] = '\0';
         if (Debug) fprintf (stdout, "OutputTimeout |%s|\n", OutputTimeout);
         if (VMSnok (status =
             sys$bintim (&OutputTimeoutDsc, &OutputTimeoutDeltaBinTime)))
            ErrorExitVmsStatus (status, "output timeout", __FILE__, __LINE__);

         return (SS$_NORMAL);

      case TIMEOUT_INPUT :

         return (sys$setimr (0, &InputTimeoutDeltaBinTime,
                             &TimeoutAst, RequestPtr, 0));

      case TIMEOUT_OUTPUT :

         /* cancel the input (request) timeout timer */
         sys$cantim (RequestPtr, 0);

         return (sys$setimr (0, &OutputTimeoutDeltaBinTime,
                             &TimeoutAst, RequestPtr, 0));

      default :
         return (SS$_BADPARAM);
   }
}

/*****************************************************************************/
/*
This AST is called when the sys$setimr() expires.  This is done to cleanup 
requests that "hang", often due to a client aborting a data transfer, or a 
script getting into an infinite loop, etc.
*/ 

TimeoutAst (struct RequestStruct *RequestPtr)

{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "TimeoutAst()\n");

   if (RequestPtr->DclHasBeenExecuting)
   {
      /* DCL involved, do any DCL-specific cleanup */
      DclAbortSubprocess (RequestPtr);
   }
   else
   {
      /* cancel network I/O, should force the connection to abort */
      sys$cancel (RequestPtr->ClientChannel);
   }

   /* close these files, should help thread unravel doing this :^) */
   if (RequestPtr->FileFab.fab$w_ifi)
   {
      sys$close (&RequestPtr->FileFab, 0, 0);
      RequestPtr->FileFab.fab$w_ifi = 0;
   }
   if (RequestPtr->ShtmlFileFab.fab$w_ifi)
   {
      sys$close (&RequestPtr->ShtmlFileFab, 0, 0);
      RequestPtr->ShtmlFileFab.fab$w_ifi = 0;
   }
}

/*****************************************************************************/
/*
This function may be AST-delivery executed one or more times, FROM ITSELF, 
before a thread is actually disposed of.  If a next-task function was 
specified, and there has been no error message generated, then execute it.  If 
there are any characters remaining in the output buffer then flush it with AST 
leading straight back to this function!  If an error has been generated then 
send it to the client once again with an AST leading straight back to this 
function!  Check for redirection.  If local redirection do NOT dispose of the 
thread (we'll be re-using the thread context), otherwise close the client 
network socket, release dynamic data structures allocated, etc.
*/

ConcludeProcessing (struct RequestStruct *RequestPtr)

{
   static long  Addx2 = 2;

   int  StatusCodeGroup;
   unsigned long  QuadScratch [2];
   void (*NextTaskFunction)(struct RequestStruct*);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ConcludeProcessing()\n");

   if (RequestPtr->NextTaskFunction != NULL &&
       RequestPtr->ErrorMessagePtr == NULL)
   {
      /**********************************/
      /* execute the next task function */
      /**********************************/

      NextTaskFunction = RequestPtr->NextTaskFunction;
      RequestPtr->NextTaskFunction = NULL;
      (*NextTaskFunction) (RequestPtr);
      return;
   }

   /*
      Flush anything currently remaining in the thread's output buffer.
      AST back to this same function to continue disposing of the thread!
   */
   if (RequestPtr->OutputBufferCurrentPtr > RequestPtr->OutputBufferPtr)
   {
      BufferOutput (RequestPtr, &ConcludeProcessing, NULL, 0);
      return;
   }

   /*
      If an error has been reported send this to the client.
      AST back to this same function to continue disposing of the thread!
   */
   if (RequestPtr->ErrorMessagePtr != NULL)
   {
      ErrorSendToClient (RequestPtr, &ConcludeProcessing);
      /* ensure that no redirection or netx task is done if an error occured */
      RequestPtr->LocationPtr = RequestPtr->NextTaskFunction = NULL;
      return;
   }

   /* ensure that if not otherwise closed, these files get closed! */
   if (RequestPtr->FileFab.fab$w_ifi)
   {
      sys$close (&RequestPtr->FileFab, 0, 0);
      RequestPtr->FileFab.fab$w_ifi = 0;
   }
   if (RequestPtr->ShtmlFileFab.fab$w_ifi)
   {
      sys$close (&RequestPtr->ShtmlFileFab, 0, 0);
      RequestPtr->ShtmlFileFab.fab$w_ifi = 0;
   }

   if (RequestPtr->LocationPtr != NULL)
   {
      /***************/
      /* redirection */
      /***************/

      /* if local do NOT dispose of thread */
      if (VMSok (HttpdRedirect (RequestPtr)))
      {
         /* log the original request, before generating it redirected */
         if (LoggingEnabled) Logging (RequestPtr, LOGGING_ENTRY);

         /* re-call the parsing function to begin re-processing the request */
         RequestParseAndExecute (RequestPtr);
         return;
      }

      /*
         An status of STS$K_ERROR is returned when its external redirect.
         If a genuine error reported when redirecting send this to the client.
         AST back to this same function to continue disposing of the thread!
      */
      if (RequestPtr->ErrorMessagePtr != NULL)
         ErrorSendToClient (RequestPtr, &ConcludeProcessing);

      return;
   }

   /************************************/
   /* completely dispose of the thread */
   /************************************/

   /* cancel any outstanding timeout timer */
   sys$cantim (RequestPtr, 0);

   /* if DCL subprocesses involved, do any DCL-specific cleanup */
   if (RequestPtr->DclHasBeenExecuting) DclDispose (RequestPtr);

   HttpdShutdownSocket (RequestPtr->ClientChannel);

   /* a little more accounting */
   Accounting.ConnectCurrent = CurrentConnectCount-1;
   QuadScratch[0] = RequestPtr->BytesTx;
   QuadScratch[1] = 0;
   lib$addx (&QuadScratch, &Accounting.QuadBytesTx,
             &Accounting.QuadBytesTx, &Addx2);
   QuadScratch[0] = RequestPtr->BytesRx;
   QuadScratch[1] = 0;
   lib$addx (&QuadScratch, &Accounting.QuadBytesRx,
             &Accounting.QuadBytesRx, &Addx2);
   StatusCodeGroup = RequestPtr->ResponseStatusCode / 100;
   if (StatusCodeGroup < 0 || StatusCodeGroup > 5) StatusCodeGroup = 0;
   Accounting.ResponseStatusCodeCountArray[StatusCodeGroup]++;

   /* set logical names used by HTTPd monitor utility */
   if (MonitorEnabled) DefineMonitorLogicals (RequestPtr);

   if (LoggingEnabled) Logging (RequestPtr, LOGGING_ENTRY);

   /* free dynamic memory allocated for the heap and thread structure */
   HeapFree (RequestPtr);
   if (Debug) fprintf (stdout, "free() RequestPtr\n");
   free (RequestPtr);

   /* one less to worry about! */
   if (CurrentConnectCount) CurrentConnectCount--;

   /* don't bother checking for exit or restart if connections still exist */
   if (CurrentConnectCount) return;

   /* if the control module has requested server exit or restart */
   if (ControlExitRequested)
   {
      fprintf (stdout, "%%%s-I-CONTROL, delayed server exit\n", Utility);
      ExitStatus = SS$_NORMAL;
      HttpdExit();
      sys$delprc (0, 0);
   }
   else
   if (ControlRestartRequested)
   {
      fprintf (stdout, "%%%s-I-CONTROL, delayed server restart\n", Utility);
      exit (SS$_NORMAL);
   }
}

/****************************************************************************/

