/*****************************************************************************/
/*
                                 Logging.c

Logging functions for the HFRD VMS HTTPd.  Produces a Web "common" format log 
suitable for analysis by tools capable of processing that format.  Secretes 
four pseudo-requests logging the BEGINing, OPENing, CLOSEing and ENDing of 
logging activities.  These shouldn't introduce problems for analysis tools as 
they look like legitimate POST entries for user "HHTPd"!

Log file name must be supplied via 'LoggingFileName' or the logical name 
HTTPD$LOG (process, job, or system level).  An error is reporting if logging 
is enabled without either of these being supplied. 


VERSION HISTORY
---------------
01-DEC-95  MGD  HTTPd version 3
27-SEP-95  MGD  when possible, use time values in request thread structure
16-JUN-95  MGD  added node name, changed logging fields to be space-separated
07-APR-95  MGD  initial development for addition to multi-threaded daemon
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <errno.h>

/* VMS related header files */
#include <descrip.h>
#include <iodef.h>
#include <lnmdef.h>
#include <prvdef.h>
#include <rms.h>
#include <ssdef.h>
#include <stsdef.h>

#include "httpd.h"

/******************/
/* global storage */
/******************/

boolean  LoggingEnabled = false,
         LoggingFileOpen = false,
         LoggingFlushTimerOutstanding = false;

char  LoggingFileName [256];

/********************/
/* external storage */
/********************/

extern boolean  Debug;
extern int  ExitStatus;
extern char  ServerHostName[];
extern char  SoftwareID[];
extern char  TimeGmtString[];

/****************************/
/* functions in this module */
/****************************/

int Logging (struct RequestStruct*, int);
LoggingFlushFileAST ();

/**********************/
/* external functions */
/**********************/

ErrorExitVmsStatus (int, char*, char*, int);

/*****************************************************************************/
/*
Web "common" format log.

Parameter 'Function' can be:

LOGGING_BEGIN           server startup
LOGGING_OPEN            open server logging, other than server startup
LOGGING_CLOSE           close server logging, other than server shutdown
LOGGING_END             server shutdown
LOGGING_ENTRY           log a request
LOGGING_FLUSH           close the log file, flushing the contents
*/ 

int Logging
(
struct RequestStruct *RequestPtr,
int Function
)
{
   static char  *MonthName [] =
      { "", "Jan", "Feb", "Mar", "Apr", "May", "Jun",
            "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };

   /* flush the log file one minute after the first entry after flushing */
   static $DESCRIPTOR (FlushDeltaTimeDsc, "0 00:01:00.00");

   static $DESCRIPTOR (BytesFaoDsc, "!UL");
   static $DESCRIPTOR (LogFileTimeFaoDsc,
                       "!2ZL/!3AZ/!4ZL:!2ZL:!2ZL:!2ZL !AZ");
   /* host r-ident user [dd/mmm/yyyy:hh:mm:ss gmt] \"request\" status bytes */
   static $DESCRIPTOR (LogFileEntryFaoDsc,
                       "!AZ - !AZ [!AZ] \"!AZ !AZ!AZ!AZ!AZ\" !UL !AZ");
   /* server-host - - [dd/mmm/yyyy:hh:mm:ss +gmt] \"what status\" 200 0 */
   static $DESCRIPTOR (LogFileBogusEntryFaoDsc,
                       "!AZ - HTTPd [!AZ] \"POST !AZ-!8XL\" 200 -");

   static $DESCRIPTOR (LnmFileDevDsc, "LNM$FILE_DEV");
   static $DESCRIPTOR (LoggingFileNameLogicalDsc, "HTTPD$LOG");

   static int  LoggingFileNameLength = 0,
               RecordCount = 0;
   static unsigned short  Length,
                          NumericTime [7];
   static unsigned long  BinaryTime [2],
                         FlushDeltaBinTime [2],
                         SysPrvMask [2] = { PRV$M_SYSPRV, 0 };

   static char  BytesString [32],
                String [1024],
                TimeString [32];
   static $DESCRIPTOR (BytesStringDsc, BytesString);
   static $DESCRIPTOR (StringDsc, String);
   static $DESCRIPTOR (TimeStringDsc, TimeString);

   static struct FAB  LogFileFab;
   static struct RAB  LogFileRab;

   struct {
      short int  buf_len;
      short int  item;
      unsigned char   *buf_addr;
      unsigned short  *ret_len;
   } LnmItems [] =
   {
      { sizeof(LoggingFileName)-1, LNM$_STRING, LoggingFileName, &Length },
      { 0,0,0,0 }
   };

   int  status,
        SetimrStatus,
        SetPrvStatus;
   char  *BytesStringPtr,
         *FunctionNamePtr,
         *MethodPtr,
         *PathInfoPtr,
         *QueryStringPtr,
         *QuestionMarkPtr,
         *RemoteUserPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "Logging() Function: %d\n", Function);

   if (Function == LOGGING_OPEN || Function == LOGGING_BEGIN)
      LoggingEnabled = true;

   if (!LoggingEnabled) return (SS$_NORMAL);

   if (Function == LOGGING_FLUSH && !RecordCount) return (SS$_NORMAL);

   if (Function == LOGGING_FLUSH)
   {
      /******************/
      /* flush log file */
      /******************/

      sys$close (&LogFileFab, 0, 0);
      LoggingFileOpen = false;
      RecordCount = 0;
      return (SS$_NORMAL);
   }

   if (!LoggingFileOpen)
   {
      /*********************/
      /* open the log file */
      /*********************/

      if (Debug) fprintf (stdout, "LoggingFileName |%s|\n", LoggingFileName);

      if (!LoggingFileName[0])
      {
         if (VMSnok (status =
             sys$trnlnm (0, &LnmFileDevDsc,
                         &LoggingFileNameLogicalDsc, 0, &LnmItems)))
            return (status);
         LoggingFileName[Length] = '\0';
         if (Debug) fprintf (stdout, "LoggingFileName |%s|\n", LoggingFileName);
      }

      if (!LoggingFileNameLength)
         LoggingFileNameLength = strlen(LoggingFileName);

      LogFileFab = cc$rms_fab;
      LogFileFab.fab$l_fna = LoggingFileName;  
      LogFileFab.fab$b_fns = LoggingFileNameLength;
      LogFileFab.fab$l_fop = FAB$M_CIF | FAB$M_SQO;
      LogFileFab.fab$b_fac = FAB$M_PUT;
      LogFileFab.fab$b_shr = FAB$M_SHRPUT;
      LogFileFab.fab$b_org = FAB$C_SEQ;
      LogFileFab.fab$b_rat = FAB$M_CR;
   
      /* turn on SYSPRV to allow access to the logging file */
      if (VMSnok (status = sys$setprv (1, &SysPrvMask, 0, 0)))
      {
         if (Debug) fprintf (stdout, "sys$setprv() %%X%08.08X\n", status);
         return (status);
      }

      status = sys$create (&LogFileFab, 0, 0);

      /* turn off SYSPRV */
      if (VMSnok (SetPrvStatus = sys$setprv (0, &SysPrvMask, 0, 0)))
      {
         if (Debug) fprintf (stdout, "sys$setprv() %%X%08.08X\n", SetPrvStatus);
         /* do NOT keep processing if there is a problem turning off SYSPRV! */
         ErrorExitVmsStatus (SetPrvStatus, "reducing privileges",
                             __FILE__, __LINE__);
      }

      if (VMSnok (status))
      {
         if (Debug) fprintf (stdout, "sys$create() %%X%08.08X\n", status);
         return (status);
      }

      LogFileRab = cc$rms_rab;
      LogFileRab.rab$l_fab = &LogFileFab;
      LogFileRab.rab$b_mbf = 4;
      LogFileRab.rab$l_rop = RAB$M_EOF | RAB$M_WBH;

      if (VMSnok (status = sys$connect (&LogFileRab, 0, 0)))
      {
         if (Debug) fprintf (stdout, "sys$connect() %%X%08.08X\n", status);
          sys$close (&LogFileFab, 0, 0);
         return (status);
      }

      LoggingFileOpen = true;
      RecordCount = 0;
   }

   /***********************/
   /* act on the log file */
   /***********************/

   if (RequestPtr == NULL)
   {
      sys$gettim (&BinaryTime);
      sys$numtim (&NumericTime, &BinaryTime);
   }
   else
      sys$numtim (&NumericTime, &RequestPtr->BinaryTime);

   status = sys$fao (&LogFileTimeFaoDsc, &Length, &TimeStringDsc,
                     NumericTime[2],
                     MonthName[NumericTime[1]],
                     NumericTime[0],
                     NumericTime[3],
                     NumericTime[4],
                     NumericTime[5],
                     TimeGmtString);
   /** if (Debug) fprintf (stdout, "sys$fao() %%X%08.08X\n", status); **/
   TimeString[Length] = '\0';

   if (Function == LOGGING_BEGIN || Function == LOGGING_OPEN ||
       Function == LOGGING_END || Function == LOGGING_CLOSE)
   {
      if (Function == LOGGING_BEGIN)
         FunctionNamePtr = "BEGIN";
      else
      if (Function == LOGGING_OPEN)
         FunctionNamePtr = "OPEN";
      else
      if (Function == LOGGING_CLOSE)
         FunctionNamePtr = "CLOSE";
      else
      if (Function == LOGGING_END)
         FunctionNamePtr = "END";
      status = sys$fao (&LogFileBogusEntryFaoDsc, &Length, &StringDsc,
                        ServerHostName,
                        TimeString,
                        FunctionNamePtr,
                        ExitStatus);
      /** if (Debug) fprintf (stdout, "sys$fao() %%X%08.08X\n", status); **/

      LogFileRab.rab$l_rbf = String;
      String[LogFileRab.rab$w_rsz = Length] = '\0';
      /** if (Debug) fprintf (stdout, "String |%s|\n", String); **/
      if (VMSnok (status = sys$put (&LogFileRab, 0, 0)))
         if (Debug) fprintf (stdout, "sys$put() %%X%08.08X\n", status);
      RecordCount++;

      if (Function == LOGGING_END || Function == LOGGING_CLOSE)
      {
         /******************/
         /* close log file */
         /******************/

         sys$close (&LogFileFab, 0, 0);
         LoggingFileOpen = false;
         RecordCount = 0;
         LoggingEnabled = false;
         return (SS$_NORMAL);
      }

      /* BEGIN and OPEN drop through to initialize a flush timer */
   }

   if (Function == LOGGING_ENTRY && RequestPtr != NULL)
   {
      /*************************/
      /* format request record */
      /*************************/

      /* if authentication has not been supplied "place-mark" remote user */
      if (RequestPtr->RemoteUser[0])
         RemoteUserPtr = RequestPtr->RemoteUser;
      else
         RemoteUserPtr = "-";

      if (RequestPtr->Method[0])
         MethodPtr = RequestPtr->Method;
      else
         MethodPtr = "-";

      if (RequestPtr->PathInfoPtr == NULL)
         PathInfoPtr = "-";
      else
         PathInfoPtr = RequestPtr->PathInfoPtr;

      if (RequestPtr->QueryStringPtr == NULL)
         QuestionMarkPtr = QueryStringPtr = "";
      else
      if ((QueryStringPtr = RequestPtr->QueryStringPtr)[0])
         QuestionMarkPtr = "?";
      else
         QuestionMarkPtr = "";

      if (RequestPtr->BytesTx)
      {
         sys$fao (&BytesFaoDsc, &Length, &BytesStringDsc,
                  RequestPtr->BytesTx);
         /** if (Debug) fprintf (stdout, "sys$fao() %%X%08.08X\n", status); **/
         (BytesStringPtr = BytesString)[Length] = '\0';
      }
      else
         BytesStringPtr = "-";

      /* host r-ident user [dd/mmm/yyyy:hh:mm:ss gmt] \"requst\" status bytes */
      status = sys$fao (&LogFileEntryFaoDsc, &Length, &StringDsc,
               RequestPtr->ClientHostName,
               RemoteUserPtr,
               TimeString,
               MethodPtr,
               RequestPtr->ScriptName, PathInfoPtr,
               QuestionMarkPtr, QueryStringPtr,
               RequestPtr->ResponseStatusCode,
               BytesStringPtr);

      /** if (Debug) fprintf (stdout, "sys$fao() %%X%08.08X\n", status); **/

      /********************/
      /* write the record */
      /********************/

      LogFileRab.rab$l_rbf = String;
      String[LogFileRab.rab$w_rsz = Length] = '\0';
      /** if (Debug) fprintf (stdout, "String |%s|\n", String); **/
      if (VMSnok (status = sys$put (&LogFileRab, 0, 0)))
         if (Debug) fprintf (stdout, "sys$put() %%X%08.08X\n", status);
      RecordCount++;
   }

   /*******************************/
   /* flush log file after period */
   /*******************************/

   if (!FlushDeltaBinTime[0])
   {
      if (VMSnok (status = sys$bintim (&FlushDeltaTimeDsc, &FlushDeltaBinTime)))
         ErrorExitVmsStatus (status, "setting log flush period",
                             __FILE__, __LINE__);
   }
   if (!LoggingFlushTimerOutstanding)
   {
      if (VMSnok (SetimrStatus =
          sys$setimr (0, &FlushDeltaBinTime, &LoggingFlushFileAST,
                      &LoggingFlushFileAST, 0)))
         ErrorExitVmsStatus (SetimrStatus, "setting log flush timer",
                             __FILE__, __LINE__);
      LoggingFlushTimerOutstanding = true;
   }

   return (status);
}

/*****************************************************************************/
/*
Timer AST function to "flush" (close) the log file.
*/

LoggingFlushFileAST ()

{
   Logging (NULL, LOGGING_FLUSH);
   LoggingFlushTimerOutstanding = false;
}

/*****************************************************************************/

