/*****************************************************************************/
/*
                                 Menu.c

This module implements a full multi-threaded, AST-driven, asynchronous "menu" 
file send.  The AST-driven nature makes the code a little more difficult to 
follow, but creates a powerful, event-driven, multi-threaded server.  All of 
the necessary functions implementing this module are designed to be non-
blocking. 

This module uses the BufferOutput() function to buffer any output it can into
larger chunks before sending it to the client.

Functions MenuCount() and MenuContents() are event-driven, the 
completion of each record read calling them to process the line just read.  
When end-of-file is encountered the file is closed and the thread disposed of. 

Implements the HFRD menu file.  An HFRD menu comprises a file with 1, 2, 3 or 
more blank-line delimitted sections.  The optional first is the document 
title.  The optional second is a menu description.  The mandatory section 
consists of the menu items.  Any blank-line delimtted sections thereafter 
alternate between descriptive paragraphs and menu sections.  For example: 

|This is the (optional) HTML <TITLE> line.  It must be only one line.|
||
|This is the (optional) descriptive paragraph.|
|It can be multi-line.|
||
|HT_DATA:FILE1.TXT   This is the first text file.|
|HT_DATA:FILE2.TXT   This is the second text file.|
|HT_DATA:FILE1.HTML  This would be an HTML file.|
||
|This is another (optional) descriptive paragraph.|
|It can also be multi-line.|
||
|HT_DATA:FILE1.TXT   This is the first file of the second menu.|
|HT_DATA:FILE2.TXT   This is the second file, etc.|

 
VERSION HISTORY
---------------
01-DEC-95  MGD  HTTPd version 3
27-SEP-95  MGD  added "If-Modified-Since:" functionality
07-AUG-95  MGD  ConfigIncludeCommentedInfo to allow physical file
                specification to be included as commentary within menu output
20-DEC-94  MGD  initial development for multi-threaded daemon
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <ctype.h>

/* VMS related header files */
#include <descrip.h>
#include <libdef.h>
#include <rms.h>
#include <ssdef.h>
#include <stsdef.h>

/* application-related header files */
#include "httpd.h"

/********************/
/* external storage */
/********************/

extern boolean  Debug;
extern int  FileBufferSize;
extern char  SoftwareID[];
extern struct AccountingStruct  Accounting;
extern struct ConfigStruct  Config;

/****************************/
/* functions in this module */
/****************************/

MenuBegin (struct RequestStruct*);
MenuContents (struct RAB*);
MenuCount (struct RAB*);
MenuEnd (struct RequestStruct*);
MenuInterpret (struct RAB*);
MenuNextContents (struct RequestStruct*);
MenuTitle (struct RequestStruct*, void*);
MenuDescription (struct RequestStruct*, void*, int);
MenuItems (struct RequestStruct*, void*);
MenuSearchItem (struct RequestStruct*, void*, char*, char*, char*);

/**********************/
/* external functions */
/**********************/

int BufferOutput (struct RequestStruct*, void*, char*, int);
ConcludeProcessing (struct RequestStruct*);
int CopyTextIntoHtml (char*, char*, int);
ErrorVmsStatus (struct RequestStruct*, int, char*, int);
unsigned char* HeapAlloc (struct RequestStruct*, int);
int IfModifiedSince (struct RequestStruct*, unsigned long*, unsigned long*);
QioNetWrite (struct RequestStruct*, void*, char*, int);

/*****************************************************************************/
/*
Open the specified file.  If there is a problem opening it then just return 
the status, do not report an error or anything.  This is used to determine 
whether the file exists, as when attempting to open one of multiple, possible 
home pages.

As fixed documents (files) can have revision date/times easily checked any
"If-Modified-Since:" request field date/time supplied in the request is 
processed and the file only sent if modified later than specified.
A "Last-Modified:" field is included in any response header. 

When successfully opened generate an HTTP header if required.  Once open and 
connected the menu interpretation becomes I/O event-driven. 

'RequestPtr->FileName' contains the VMS file specification.
'RequestPtr->ContentTypePtr' contains the MIME content type.
'RequestPtr->IfModifiedSinceBinaryTime' contains an optional VMS time.
'RequestPtr->SendHttpHeader' is true if an HTTP header should be generated.
*/

MenuBegin (struct RequestStruct *RequestPtr)

{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MenuBegin() |%s|\n", RequestPtr->FileName);

   /* set up default error report information (just in case!) */
   RequestPtr->ErrorTextPtr = RequestPtr->PathInfoPtr;
   RequestPtr->ErrorHiddenTextPtr = RequestPtr->FileName;

   /*************/
   /* open file */
   /*************/

   RequestPtr->FileFab = cc$rms_fab;
   RequestPtr->FileFab.fab$b_fac = FAB$M_GET;
   RequestPtr->FileFab.fab$l_fna = RequestPtr->FileName;  
   RequestPtr->FileFab.fab$b_fns = strlen(RequestPtr->FileName);
   RequestPtr->FileFab.fab$b_shr = FAB$M_SHRGET;

   /* initialize the NAM block */
   RequestPtr->FileNam = cc$rms_nam;
   RequestPtr->FileFab.fab$l_nam = &RequestPtr->FileNam;
   RequestPtr->FileNam.nam$l_esa = RequestPtr->ExpandedFileName;
   RequestPtr->FileNam.nam$b_ess = sizeof(RequestPtr->ExpandedFileName)-1;

   /* initialize the date extended attribute block */
   RequestPtr->FileXabDat = cc$rms_xabdat;
   RequestPtr->FileFab.fab$l_xab = &RequestPtr->FileXabDat;

   if (VMSnok (status = sys$open (&RequestPtr->FileFab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$open() %%X%08.08X\n", status);

      /* if its a search list treat directory not found as if file not found */
      if ((RequestPtr->FileNam.nam$l_fnb & NAM$M_SEARCH_LIST) &&
          status == RMS$_DNF)
         status = RMS$_FNF;

      /* greater than one adjusts server counters even if there is an error */
      if (RequestPtr->AdjustAccounting > 1)
      {
         Accounting.DoMenuCount++;
         RequestPtr->AdjustAccounting = 0;
      }
      return (status);
   }

   if (RequestPtr->MethodGet &&
       (RequestPtr->IfModifiedSinceBinaryTime[0] ||
        RequestPtr->IfModifiedSinceBinaryTime[1]))
   {
      /*********************/
      /* if modified since */
      /*********************/

      if (VMSnok (status =
          IfModifiedSince (RequestPtr,
                           &RequestPtr->FileXabDat.xab$q_rdt,
                           &RequestPtr->IfModifiedSinceBinaryTime)))
      {
         /* return LIB$_NEGTIM if not modified/sent */
         if (RequestPtr->AdjustAccounting)
         {
            Accounting.DoMenuCount++;
            Accounting.DoMenuNotModifiedCount++;
            RequestPtr->AdjustAccounting = 0;
         }
         MenuEnd (RequestPtr);
         return (SS$_NORMAL);
      }

      /* make sure the "If-Modified-Since:" request header line is ignored */
      RequestPtr->IfModifiedSinceBinaryTime[0] =
         RequestPtr->IfModifiedSinceBinaryTime[1] = 0;
   }

   if (RequestPtr->AdjustAccounting)
   {
      Accounting.DoMenuCount++;
      RequestPtr->AdjustAccounting = 0;
   }

   /*******************************/
   /* connect record access block */
   /*******************************/

   /* allocate heap memory for the file record buffer */
   if (RequestPtr->MenuBufferPtr == NULL)
   {
      /* allow one byte for the terminating null */
      if ((RequestPtr->MenuBufferPtr =
          HeapAlloc (RequestPtr, FileBufferSize+1)) == NULL)
      {
         ErrorHeapAlloc (RequestPtr, __FILE__, __LINE__);
         MenuEnd (RequestPtr);
         return (SS$_NORMAL);
      }
   }

   RequestPtr->FileRab = cc$rms_rab;
   RequestPtr->FileRab.rab$l_fab = &RequestPtr->FileFab;
   /* 2 buffers and read ahead performance option */
   RequestPtr->FileRab.rab$b_mbf = 2;
   RequestPtr->FileRab.rab$l_rop = RAB$M_RAH;
   RequestPtr->FileRab.rab$l_ubf = RequestPtr->MenuBufferPtr;
   RequestPtr->FileRab.rab$w_usz = FileBufferSize;

   if (VMSnok (status = sys$connect (&RequestPtr->FileRab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$connect() %%X%08.08X\n", status);
      ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
      MenuEnd (RequestPtr);
      return (SS$_NORMAL);
   }

   /********************/
   /* begin processing */
   /********************/

   if (RequestPtr->SendHttpHeader)
   {
      if (VMSnok (status = MenuHttpHeader (RequestPtr)))
      {
         MenuEnd (RequestPtr);
         return (SS$_NORMAL);
      }
      if (RequestPtr->MethodHead)
      {
         MenuEnd (RequestPtr);
         return (SS$_NORMAL);
      }
   }

   /* set then RAB user context storage to the client thread pointer */
   RequestPtr->FileRab.rab$l_ctx = RequestPtr;

   RequestPtr->MenuBlankLineCount = RequestPtr->MenuLineCount =
      RequestPtr->MenuSectionCount = RequestPtr->MenuSectionNumber = 0;

   /* queue the first record read, count the number of sections in the menu */
   sys$get (&RequestPtr->FileRab, &MenuCount, &MenuCount);

   /*
      Return to the calling routine now.  Subsequent processing is
      event-driven.  Routine completion ASTs drive the menu generation.
   */
   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
Ensure the menu file is closed.  Flush the output buffer, activating the next 
task (if any).
*/ 

MenuEnd (struct RequestStruct *RequestPtr)

{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MenuEnd()\n");

   if (RequestPtr->FileFab.fab$w_ifi)
   {
      sys$close (&RequestPtr->FileFab, 0, 0);
      RequestPtr->FileFab.fab$w_ifi = 0;
   }

   ConcludeProcessing (RequestPtr);
}

/*****************************************************************************/
/*
Generates an HTTP header for the following menu interpretation.
*/ 
 
int MenuHttpHeader (struct RequestStruct *RequestPtr)

{
   static $DESCRIPTOR (StringDsc, "");

   static $DESCRIPTOR (CommentedInfoFaoDsc,
"<!!-- SoftwareID: !AZ -->\n\
<!!-- !AZ -->\n");

   static $DESCRIPTOR (Http200HeaderFaoDsc,
"!AZ 200 Data follows.\r\n\
Server: !AZ\r\n\
Date: !AZ\r\n\
Last-Modified: !AZ\r\n\
Content-Type: text/html\r\n\
\r\n");

   int  status;
   unsigned short  CommentLength,
                   Length,
                   HeaderLength;
   char  LastModified [32],
         String [1024];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ShtmlHttpHeader()\n");

   RequestPtr->ResponseStatusCode = 200;

   if (VMSnok (status =
       HttpGmTimeString (LastModified, &RequestPtr->FileXabDat.xab$q_rdt)))
      return (status);

   StringDsc.dsc$a_pointer = String;
   StringDsc.dsc$w_length = sizeof(String)-1;

   sys$fao (&Http200HeaderFaoDsc, &HeaderLength, &StringDsc,
            HttpProtocol, SoftwareID, RequestPtr->GmDateTime, LastModified);

   if (Config.IncludeCommentedInfo && !RequestPtr->MethodHead)
   {
      StringDsc.dsc$a_pointer = String + HeaderLength;
      StringDsc.dsc$w_length = sizeof(String) - HeaderLength - 1;

      status = sys$fao (&CommentedInfoFaoDsc, &CommentLength, &StringDsc,
                        SoftwareID, RequestPtr->FileName);
   }
   else
      CommentLength = 0;

   String[Length = HeaderLength+CommentLength] = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", String);

   /* allocate heap memory for the response header */
   if ((RequestPtr->ResponseHeaderPtr =
        HeapAlloc (RequestPtr, Length)) == NULL)
   {
      ErrorHeapAlloc (RequestPtr, __FILE__, __LINE__);
      return (SS$_INSFMEM);
   }
   memcpy (RequestPtr->ResponseHeaderPtr, String, Length);

   if (VMSnok (status =
       QioNetWrite (RequestPtr, 0, String, Length)))
      ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);

   RequestPtr->SendHttpHeader = false;

   return (status);
}

/*****************************************************************************/
/*
Count the number of blank-line delimited sections in the menu file.  This 
function drives itself using AST completion routines.  Quit upon end-of-file, 
or after three sections have been encountered.  When quitting rewind the file 
calling MenuInterpret() as a sys$rewind() completion AST routine.  This begins 
the actual menu interpretation and transfer.
*/ 

MenuCount (struct RAB *RabPtr)

{
   register char  *rptr;
   register struct RequestStruct  *RequestPtr;
   int  status;

   if (Debug)
      fprintf (stdout,
               "MenuCount() sts: %%X%08.08X stv: %%X%08.08X\n",
               RabPtr->rab$l_sts,
               RabPtr->rab$l_stv);

   /* get the pointer to the thread from the RAB user context storage */
   RequestPtr = RabPtr->rab$l_ctx;

   if (VMSnok (RequestPtr->FileRab.rab$l_sts))
   {
      if (RequestPtr->FileRab.rab$l_sts == RMS$_EOF)
      {
         /*************************************************/
         /* end-of-file, rewind ready to process contents */
         /*************************************************/

         if (VMSnok (status =
             sys$rewind (&RequestPtr->FileRab, &MenuInterpret, &MenuInterpret)))
         {
            if (Debug) fprintf (stdout, "sys$rewind() %%X%08.08X\n", status);
            ErrorVmsStatus (RequestPtr, RequestPtr->FileRab.rab$l_sts,
                            __FILE__, __LINE__);
            MenuEnd (RequestPtr);
            return;
         }
         if (RequestPtr->MenuLineCount) RequestPtr->MenuSectionCount++;
         if (Debug)
            fprintf (stdout, "SectionCount: %d\n",
                     RequestPtr->MenuSectionCount);
         return;
      }

      /***************************/
      /* error reading menu file */
      /***************************/

      ErrorVmsStatus (RequestPtr, RequestPtr->FileRab.rab$l_sts,
                      __FILE__, __LINE__);
      MenuEnd (RequestPtr);
      return;
   }

   if (RequestPtr->FileRab.rab$w_rsz)
   {
      RequestPtr->FileRab.rab$l_ubf[RequestPtr->FileRab.rab$w_rsz] = '\0';
      /* terminate on any carriage control that may be in the buffer */
      for (rptr = RequestPtr->FileRab.rab$l_ubf;
           *rptr && *rptr != '\r' && *rptr != '\n';
           rptr++);
      *rptr = '\0';
      /* find non-space characters */
      for (rptr = RequestPtr->FileRab.rab$l_ubf;
           *rptr && isspace(*rptr) && *rptr != '!';
           rptr++);
      /* if its not a blank or comment-only line, count it */
      if (*rptr && *rptr != '!')
         RequestPtr->MenuLineCount++;
   }
   else
   {
      if (RequestPtr->MenuLineCount) RequestPtr->MenuSectionCount++;
      RequestPtr->MenuLineCount = 0;
   }

   /* only need to count as far as 3, we know what to do after that */
   if (RequestPtr->MenuSectionCount == 3)
   {
      /************************************/
      /* rewind ready to process contents */
      /************************************/

      if (VMSnok (status =
          sys$rewind (&RequestPtr->FileRab, &MenuInterpret, &MenuInterpret)))
      {
         if (Debug) fprintf (stdout, "sys$rewind() %%X%08.08X\n", status);
         ErrorVmsStatus (RequestPtr, RequestPtr->FileRab.rab$l_sts,
                         __FILE__, __LINE__);
         MenuEnd (RequestPtr);
         return;
      }
      if (Debug)
         fprintf (stdout, "SectionCount: %d\n", RequestPtr->MenuSectionCount);
      return;
   }

   /* queue another read, completion AST back to this function again */
   sys$get (&RequestPtr->FileRab, &MenuCount, &MenuCount);

   /*
      Return to the AST-interrupted routine now.  Continued processing is
      event-driven.  Routine completion ASTs drive the menu generation.
   */
}

/*****************************************************************************/
/*
Output the HTTP header and queue a read of the first record in the menu file.  
The AST completion routine calls MenuContents() to process the first record, 
thereafter MenuContents() calls itself as an AST completion routine to process 
each record (line) in the file.
*/ 

MenuInterpret (struct RAB *RabPtr)

{
   register struct RequestStruct  *RequestPtr;
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MenuInterpret()\n");

   /* get the pointer to the thread from the RAB user context storage */
   RequestPtr = RabPtr->rab$l_ctx;

   /* reset some counters now that the section count is complete */
   RequestPtr->MenuBlankLineCount =
      RequestPtr->MenuLineCount =
      RequestPtr->MenuSectionNumber = 0;

   /* queue the read of the first record of the rewound file */
   sys$get (&RequestPtr->FileRab, &MenuContents, &MenuContents);

   /*
      Return to the calling routine now.  Subsequent processing is
      event-driven.  I/O completion ASTs drive the menu generation.
   */
}

/*****************************************************************************/
/*
Process the records (lines) of the menu file according to the total count of 
the sections (blank-line delimited text) in the file and the section number 
the current line falls within.  This routine is event-driven, using itself as 
an AST completion routine to sys$get()s.
*/ 

MenuContents (struct RAB *RabPtr)

{
   register unsigned char  *rptr;
   register struct RequestStruct  *RequestPtr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout,
               "MenuContents() sts: %%X%08.08X stv: %%X%08.08X\n",
               RabPtr->rab$l_sts,
               RabPtr->rab$l_stv);

   RequestPtr = RabPtr->rab$l_ctx;

   if (VMSnok (RequestPtr->FileRab.rab$l_sts))
   {
      if (RequestPtr->FileRab.rab$l_sts == RMS$_EOF)
      {
         /********************/
         /* end of menu file */
         /********************/

         MenuEnd (RequestPtr);
         return;
      }

      /***************************/
      /* error reading menu file */
      /***************************/

      ErrorVmsStatus (RequestPtr, RequestPtr->FileRab.rab$l_sts,
                      __FILE__, __LINE__);
      MenuEnd (RequestPtr);
      return;
   }

   /* terminate the line just read */
   (rptr = RequestPtr->FileRab.rab$l_ubf)[RequestPtr->FileRab.rab$w_rsz] = '\0';
   if (*rptr)
   {
      /* terminate on any carriage control that may be in the buffer */
      while (*rptr && *rptr != '\r' && *rptr != '\n') rptr++;
      *rptr = '\0';
   }
   if (Debug)
      fprintf (stdout, "RequestPtr->FileRab.rab$l_ubf |%s|\n",
               RequestPtr->FileRab.rab$l_ubf);

   if (!RequestPtr->FileRab.rab$l_ubf[0])
   {
      /**************/
      /* empty line */
      /**************/

      RequestPtr->MenuBlankLineCount++;
      RequestPtr->MenuLineCount = 0;
      /* queue another read, completion AST back to this function again */
      sys$get (&RequestPtr->FileRab, &MenuContents, &MenuContents);
      return;
   }

   if (RequestPtr->FileRab.rab$l_ubf[0] == '!')
   {
      /*********************/
      /* comment-only line */
      /*********************/

      /* queue another read, completion AST back to this function again */
      sys$get (&RequestPtr->FileRab, &MenuContents, &MenuContents);
      return;
   }

   /******************/
   /* non-blank line */
   /******************/

   if (RequestPtr->MenuBlankLineCount) RequestPtr->MenuSectionNumber++;
   RequestPtr->MenuLineCount++;
   RequestPtr->MenuBlankLineCount = 0;
   if (!RequestPtr->MenuSectionNumber) RequestPtr->MenuSectionNumber++;

   /******************/
   /* interpret line */
   /******************/

   if (RequestPtr->MenuSectionCount >= 2 &&
       RequestPtr->MenuSectionNumber == 1)
      MenuTitle (RequestPtr, &MenuNextContents);
   else
   if (RequestPtr->MenuSectionCount >= 3 &&
       !(RequestPtr->MenuSectionNumber % 2))
   {
      /* ensure the description lines have HTTP carriage-control */
      *rptr++ = '\n';
      *rptr = '\0';
      MenuDescription (RequestPtr, &MenuNextContents,
         rptr - (unsigned char*)RequestPtr->FileRab.rab$l_ubf);
   }
   else
   if (RequestPtr->MenuSectionCount == RequestPtr->MenuSectionNumber)
      MenuItems (RequestPtr, &MenuNextContents);
   else
   if (RequestPtr->MenuSectionCount >= 3 &&
       !((RequestPtr->MenuSectionNumber-1) % 2))
      MenuItems (RequestPtr, &MenuNextContents);
}

/*****************************************************************************/
/*
*/

int MenuNextContents (struct RequestStruct *RequestPtr)

{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MenuNextContents()\n");

   /* queue another read, completion AST back to contents interpretation */
   if (VMSnok (status =
       sys$get (&RequestPtr->FileRab, &MenuContents, &MenuContents)))
      MenuEnd (RequestPtr);
}

/*****************************************************************************/
/*
The line just read is the menu title.
*/

int MenuTitle
(
struct RequestStruct *RequestPtr,
void *AstFunctionPtr
)
{
   static $DESCRIPTOR (TitleFaoDsc, "<TITLE>!AZ</TITLE>\n<H1>!AZ</H1>\n");
   static $DESCRIPTOR (StringDsc, "");

   unsigned short  Length;
   char  String [1024];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MenuTitle()\n");

   /* should only be one line, ignore all but the first */
   if (RequestPtr->MenuLineCount > 1) return;

   StringDsc.dsc$a_pointer = String;
   StringDsc.dsc$w_length = sizeof(String)-1;
   sys$fao (&TitleFaoDsc, &Length, &StringDsc,
            RequestPtr->FileRab.rab$l_ubf, RequestPtr->FileRab.rab$l_ubf);
   String[Length] = '\0';

   /* output the title */
   if (VMSnok (BufferOutput (RequestPtr, AstFunctionPtr, String, Length)))
      MenuEnd (RequestPtr);
}

/*****************************************************************************/
/*
The line just read forms part of a description section.  The calling routine 
has appended HTTP carriage-control, so this line is just output as-is.  If 
this is the first (and possibly only) description section separate using a 
paragraph, or <P>, tag.  If the second (or more) description section first 
terminate the unsigned list, or </UL> begun in the preceding item section 
before separating using the pragraph tag.
*/ 

int MenuDescription
(
struct RequestStruct *RequestPtr,
void *AstFunctionPtr,
int LineLength
)
{
   char  String [256];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MenuDescription()\n");

   if (RequestPtr->MenuLineCount == 1)
   {
      /* first line of the description section */
      if (RequestPtr->MenuSectionNumber == 2)
      {
         /* first description section of the menu */
         if (LineLength > sizeof(String)-5) LineLength -= 5;
         memcpy (String, "<P>\n", 4);
         memcpy (String+4, RequestPtr->FileRab.rab$l_ubf, LineLength);
         String[LineLength += 4] = '\0';
      }
      else
      {
         /* second or subsequent description section of the menu */
         if (LineLength > sizeof(String)-11) LineLength -= 11;
         memcpy (String, "</UL>\n<P>\n", 10);
         memcpy (String+10, RequestPtr->FileRab.rab$l_ubf, LineLength);
         String[LineLength += 10] = '\0';
      }
      if (VMSnok (BufferOutput (RequestPtr, AstFunctionPtr,
                  String, LineLength)))
         MenuEnd (RequestPtr);
   }
   else
   {
      /* second or subsequent line of the description section */
      if (VMSnok (BufferOutput (RequestPtr, AstFunctionPtr,
                  RequestPtr->FileRab.rab$l_ubf, LineLength)))
         MenuEnd (RequestPtr);
   }
}

/*****************************************************************************/
/*
Interpret this line as a menu item comprising a URI, white-space, and a 
description comprising the rest of the line after the first non-space.  Each 
line in the item section is output as an item in an non-ordered list, or <UL>.
*/

int MenuItems
(
struct RequestStruct *RequestPtr,
void *AstFunctionPtr
)
{
   static char  NoDescription [] = "<I>(no description for this line)</I>";
   static $DESCRIPTOR (FileFaoDsc, "!AZ<LI><A HREF=\"!AZ\">!AZ</A>\n");
   static $DESCRIPTOR (StringDsc, "");

   register char  *cptr, *sptr, *zptr;

   unsigned short  Length;
   char  *DescriptionPtr,
         *FormattingPtr;
   char  Description [256],
         String [512],
         Uri [256];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MenuItems()\n");

   if (RequestPtr->MenuLineCount == 1)
      FormattingPtr = "<P>\n<UL>\n";
   else
      FormattingPtr = "";

   cptr = RequestPtr->FileRab.rab$l_ubf;
   /* skip leading white-space */
   while (*cptr && isspace(*cptr)) cptr++;
   zptr = (sptr = Uri) + sizeof(Uri)-2;
   while (*cptr && !isspace(*cptr) && sptr < zptr) *sptr++ = *cptr++;
   *sptr = '\0';
   /* skip intervening white-space */
   while (*cptr && isspace(*cptr)) cptr++;
   zptr = (sptr = Description) + sizeof(Description)-2;
   while (*cptr && sptr < zptr) *sptr++ = *cptr++;
   *sptr = '\0';

   if (Description[0])
      DescriptionPtr = Description;
   else
      DescriptionPtr = NoDescription;

   if (Uri[0] == '\"')
   {
      /* full URI/URL specification, remove quotes and include as-is */
      for (cptr = Uri+1; *cptr && *cptr != '\"'; cptr++);
      if (*cptr == '\"') *cptr = '\0';
      StringDsc.dsc$a_pointer = String;
      StringDsc.dsc$w_length = sizeof(String)-1;
      sys$fao (&FileFaoDsc, &Length, &StringDsc,
               FormattingPtr, Uri+1, DescriptionPtr);
      String[Length] = '\0';
      if (VMSnok (BufferOutput (RequestPtr, AstFunctionPtr, String, Length)))
         MenuEnd (RequestPtr);
   }
   else
   {
      for (cptr = Uri; *cptr && *cptr != '?'; cptr++);
      if (*cptr == '?')
      {
         /* search item */
         *cptr++ = '\0';
         MenuSearchItem (RequestPtr, AstFunctionPtr,
                             Uri, Description, cptr);
      }
      else
      {
         /* file item */
         StringDsc.dsc$a_pointer = String;
         StringDsc.dsc$w_length = sizeof(String)-1;
         sys$fao (&FileFaoDsc, &Length, &StringDsc,
                  FormattingPtr, Uri, DescriptionPtr);
         String[Length] = '\0';
         if (VMSnok (BufferOutput (RequestPtr, AstFunctionPtr, String, Length)))
            MenuEnd (RequestPtr);
      }
   }
}

/*****************************************************************************/
/*
The URI contains a question mark, which is menu-speak for "this is a search
item".  Create and output a search form.  Optional characters following the 
question mark include an "A" for and "about the search" link, "C" for case-
sensitivity buttons, and "O" for output style buttons.
*/

int MenuSearchItem
(
struct RequestStruct *RequestPtr,
void *AstFunctionPtr,
char *Uri,
char *Description,
char *SearchOptions
)
{
   static $DESCRIPTOR (StringDsc, "");

   static $DESCRIPTOR (BeginFormFaoDsc,
"<LI>\n\
<FORM ACTION=\"!AZ\">\n\
<INPUT TYPE=submit VALUE=\"!AZ\">\n\
<INPUT TYPE=text NAME=\"search\">\n\
<INPUT TYPE=reset VALUE=\"Reset\">\n");

   static char  AboutForm[] =
"<BR><I><A HREF=\"?about=search\">About</A> this search.</I>\n";

   static char  CaseForm[] =
"<BR><I>Output By:  \
line<INPUT TYPE=radio NAME=\"hits\" VALUE=\"line\" CHECKED>\
 document<INPUT TYPE=radio NAME=\"hits\" VALUE=\"document\"></I>\n";

   static char  OutputForm[] =
"<BR><I>Case Sensitive:  \
no<INPUT TYPE=radio NAME=\"case\" VALUE=\"no\" CHECKED>\
 yes<INPUT TYPE=radio NAME=\"case\" VALUE=\"yes\"></I>\n";

   static char EndForm[] = "</FORM>\n";

   int  status;
   unsigned short  Length;
   char  HtmlDescription [512],
         String [1024];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MenuSearchItem()\n");

   CopyTextIntoHtml (HtmlDescription, Description, -1);

   StringDsc.dsc$a_pointer = String;
   StringDsc.dsc$w_length = sizeof(String)-1;
   sys$fao (&BeginFormFaoDsc, &Length, &StringDsc, Uri, HtmlDescription);

   while (*SearchOptions)
   {
      if (toupper(*SearchOptions) == 'A')
      {
         if (sizeof(AboutForm)-1 < sizeof(String)-1 - Length)
         {
            memcpy (String+Length, AboutForm, sizeof(AboutForm)-1);
            Length += sizeof(AboutForm)-1;
         }
      }
      else
      if (toupper(*SearchOptions) == 'O')
      {
         if (sizeof(OutputForm)-1 < sizeof(String)-1 - Length)
         {
            memcpy (String+Length, OutputForm, sizeof(OutputForm)-1);
            Length += sizeof(OutputForm)-1;
         }
      }
      else
      if (toupper(*SearchOptions) == 'C')
      {
         if (sizeof(CaseForm)-1 < sizeof(String)-1 - Length)
         {
            memcpy (String+Length, CaseForm, sizeof(CaseForm)-1);
            Length += sizeof(CaseForm)-1;
         }
      }
      SearchOptions++;
   }

   if (sizeof(EndForm)-1 < sizeof(String)-1 - Length)
   {
      memcpy (String+Length, EndForm, sizeof(EndForm)-1);
      Length += sizeof(EndForm)-1;
   }

   String[Length] = '\0';
   if (VMSnok (BufferOutput (RequestPtr, AstFunctionPtr, String, Length)))
      MenuEnd (RequestPtr);
}

/*****************************************************************************/

