/*****************************************************************************/
/*
                                Request.c

Get, process and execute the HTTP request from the client.

Two server directives are detected in this module.  Server directives take the 
form of a special query string containing instructions to the HTTPd server.  
Normally, if a script is not specified the presence of a query string results 
in the default search script being activated.  A server directive alters that.

  httpd=report      generates an HTML report on server performance
  httpd=index       used with a wildcard specification, generates a
                    directory listing with query string controlled format

Error counting within this module attempts to track the reason that any 
connection accepted fails before being passsed to another module for 
processing (i.e. request format error, path forbidden by rule).


VERSION HISTORY
---------------
01-DEC-95  MGD  HTTPd version 3
27-SEP-95  MGD  extensive rework of some functions;
                added 'Referer:', 'User-Agent:', 'If-Modified-Since:'
01-APR-95  MGD  initial development for addition to multi-threaded daemon
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <errno.h>
#include <string.h>

/* VMS related header files */
#include <descrip.h>
#include <iodef.h>
#include <ssdef.h>
#include <stsdef.h>

#include "httpd.h"

/********************/
/* external storage */
/********************/

extern boolean  Debug;
extern int  NetReadBufferSize;
extern char  SoftwareID[];
extern char  ErrorFacilityDisabled[];
extern char  ErrorStringSize[];
extern char  ErrorRequestFormat[];
extern char  ErrorRequestHttpMethod[];
extern char  ErrorRequestMaxAccept[];
extern struct AccountingStruct  Accounting;
extern struct ConfigStruct  Config;

/****************************/
/* functions in this module */
/****************************/

RequestGet (struct RequestStruct*);
RequestExecute (struct RequestStruct*);
RequestFields (struct RequestStruct*);
RequestFile (struct RequestStruct*);
RequestMethodPathQuery (struct RequestStruct*);
RequestParseAndExecute (struct RequestStruct*);

/********************************/
/* external function prototypes */
/********************************/

ConcludeProcessing (struct RequestStruct*);
DclBegin (struct RequestStruct*, char*, char*);
DirBegin (struct RequestStruct*);
ErrorVmsStatus (struct RequestStruct*, int, char*, int);
int FileBegin (struct RequestStruct*);
unsigned char* HeapAlloc (struct RequestStruct*, int);
unsigned char* HeapRealloc (struct RequestStruct*, unsigned char*, int);
int IsMapBegin (struct RequestStruct*);
int MenuBegin (struct RequestStruct*);
ServerReport (struct RequestStruct*);
int TimeoutSet (struct RequestStruct*, int);
int ShtmlBegin (struct RequestStruct*);

/*****************************************************************************/
/*
Process first packet read from the client.  In most cases this will contain 
the complete HTTP header and it can be processed without building up a 
buffered header.  If it does not contain the full header allocate heap memory
to contain the current packet and queue subsequent read(s), expanding the 
request header heap memory, until all is available (or it becomes completely 
rediculous!).
*/ 
 
RequestGet (struct RequestStruct *RequestPtr)

{
   register int  cnlcnt, cnt;
   register char  *cptr, *sptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout,
"RequestGet() Status: %%X%08.08X Count: %d BufferSize: %d HeaderLength: %d\n",
               RequestPtr->NetReadIOsb.Status,
               RequestPtr->NetReadIOsb.Count,
               RequestPtr->NetReadBufferSize,
               RequestPtr->RequestHeaderLength);

   if (VMSnok (RequestPtr->NetReadIOsb.Status))
   {
      Accounting.RequestErrorCount++;
      ErrorVmsStatus (RequestPtr, RequestPtr->NetReadIOsb.Status,
                      __FILE__, __LINE__);
      ConcludeProcessing (RequestPtr);
      return;
   }

   RequestPtr->BytesRx += (cnt = RequestPtr->NetReadIOsb.Count);

   if (RequestPtr->NetReadBufferSize >= 8192)
   {
      /* cannot be a legitimate HTTP request! */
      ConcludeProcessing (RequestPtr);
      return;
   }

   if (RequestPtr->RequestHeaderPtr == NULL)
   {
      RequestPtr->RequestHeaderPtr = RequestPtr->NetReadBufferPtr;
      RequestPtr->RequestHeaderLength = 0;
   }

   RequestPtr->RequestHeaderPtr[cnt] = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", RequestPtr->RequestHeaderPtr); 

   /* look for the end of header blank line */
   cnlcnt = RequestPtr->HeaderConsecutiveNewLineCount;
   cptr = RequestPtr->RequestHeaderPtr;
   while (*cptr && cnlcnt < 2)
   {
      if (*cptr == '\n')
      {
         cptr++;
         cnlcnt++;
         continue;
      }
      if (*cptr == '\r' && cptr[1] == '\n')
      {
         cptr += 2;
         cnlcnt++;
         continue;
      }
      cptr++;
      cnlcnt = 0;
   }

   if (cnlcnt >= 2)
   {
      /* point (possibly back) to the start of the header */
      RequestPtr->RequestHeaderPtr = RequestPtr->NetReadBufferPtr;
      /* make the header length the actual length of the header! */
      RequestPtr->RequestHeaderLength =
         cptr - (char*)RequestPtr->NetReadBufferPtr;

      RequestParseAndExecute (RequestPtr);

      return;
   }

   RequestPtr->HeaderConsecutiveNewLineCount = cnlcnt;

   /**********************************************/
   /* request header not completely received yet */
   /**********************************************/

   RequestPtr->RequestHeaderLength += cnt;

   if (RequestPtr->RequestHeaderLength >= RequestPtr->NetReadBufferSize)
   {
      /* need more buffer space, allow one for termination of the buffer */
      RequestPtr->NetReadBufferSize += NetReadBufferSize;
      if ((RequestPtr->NetReadBufferPtr =
           HeapRealloc (RequestPtr, RequestPtr->NetReadBufferPtr,
                        RequestPtr->NetReadBufferSize + 1))
         == NULL)
      {
         Accounting.RequestErrorCount++;
         ErrorHeapAlloc (RequestPtr, __FILE__, __LINE__);
         ConcludeProcessing (RequestPtr);
         return;
      }
   }

   /* because a realloc() may move the memory recalculate the pointer */
   RequestPtr->RequestHeaderPtr = RequestPtr->NetReadBufferPtr +
                                  RequestPtr->RequestHeaderLength;

   /* 
      Queue an asynchronous read to get more of the header from the client.
      When the read completes call RequestSubsequent() AST completion
      function again to further process the request.
   */
   if (VMSnok (status = QioNetRead (RequestPtr,
                                    &RequestGet,
                                    RequestPtr->RequestHeaderPtr,
                                    RequestPtr->NetReadBufferSize -
                                       RequestPtr->RequestHeaderLength)))
   {
      /* IO failed, abandon the request */
      Accounting.RequestErrorCount++;
      ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
      ConcludeProcessing (RequestPtr);
      return;
   }
   /* further processing of the request will be AST-driven */
}

/****************************************************************************/
/*
Get the method, path and query string, and some header field line values, then
execute GET and POST queries.  This function can be called from two points. 
First, from the function RequestGet(), after it has received a complete HTTP
header from the client.  Second, from the function HttpdRedirect(), where an
HTTP header has been reconstructed in the 'NetReadBufferPtr' in order to effect
a local redirection.
*/

RequestParseAndExecute (struct RequestStruct *RequestPtr)

{
   /*********/
   /* begin */
   /*********/

   if (Debug)
   {
      fprintf (stdout, "RequestParseAndExecute() size: %d\n",
               RequestPtr->RequestHeaderLength);
      fprintf (stdout, "|%s|\n", RequestPtr->RequestHeaderPtr);
   }

   Accounting.RequestParseCount++;

   if (!RequestMethodPathQuery (RequestPtr)) return;
   if (!RequestFields (RequestPtr)) return;

   if (strcmp (RequestPtr->Method, "GET") == 0)
   {
      Accounting.MethodGetCount++;
      RequestPtr->MethodGet = true;
      RequestExecute (RequestPtr);
   }
   else
   if (strcmp (RequestPtr->Method, "HEAD") == 0)
   {
      Accounting.MethodHeadCount++;
      RequestPtr->MethodHead = true;
      RequestExecute (RequestPtr);
   }
   else
   if (strcmp (RequestPtr->Method, "POST") == 0)
   {
      Accounting.MethodPostCount++;
      RequestPtr->MethodPost = true;
      RequestExecute (RequestPtr);
   }
   else
   {
      Accounting.RequestErrorCount++;
      RequestPtr->ResponseStatusCode = 501;
      ErrorGeneral (RequestPtr, ErrorRequestHttpMethod, __FILE__, __LINE__);
      ConcludeProcessing (RequestPtr);
   }
}

/****************************************************************************/
/*
Parse the first <CR><LF> (strict HTTP) or newline (de facto HTTP) terminated
string from the client's request packet into HTTP method, path information and
query string.  Return true to continue processing the request, false to stop
immediately.  If false this function or any it called must have reported the
problem to the client and/or disposed of the thread.
*/ 
 
RequestMethodPathQuery (struct RequestStruct *RequestPtr)

{
   register unsigned char  *cptr, *sptr, *zptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "RequestMethodPathQuery()\n");

   cptr = RequestPtr->RequestHeaderPtr;

   zptr = (sptr = RequestPtr->Method) + sizeof(RequestPtr->Method);
   while (*cptr && !isspace(*cptr) && *cptr != '\r' && *cptr != '\n' &&
          sptr < zptr) *sptr++ = *cptr++;
   if (sptr >= zptr) sptr = RequestPtr->Method;
   *sptr = '\0';

   if (!RequestPtr->Method[0])
   {
      Accounting.RequestErrorCount++;
      RequestPtr->ResponseStatusCode = 400;
      ErrorGeneral (RequestPtr, ErrorRequestFormat, __FILE__, __LINE__);
      ConcludeProcessing (RequestPtr);
      return (false);
   }

   /* skip across white-space between method and URI */
   while (*cptr && isspace(*cptr) && *cptr != '\r' && *cptr != '\n') cptr++;

   /* get the path information */
   zptr = cptr;
   while (*zptr && !isspace(*zptr) && *zptr != '?' &&
          *zptr != '\r' && *zptr != '\n') zptr++;

   if (cptr == zptr)
   {
      /* no path supplied */
      Accounting.RequestErrorCount++;
      RequestPtr->ResponseStatusCode = 400;
      ErrorGeneral (RequestPtr, ErrorRequestFormat, __FILE__, __LINE__);
      ConcludeProcessing (RequestPtr);
      return (false);
   }

   if ((RequestPtr->PathInfoPtr = sptr =
        HeapAlloc (RequestPtr, zptr-cptr+1)) == NULL)
   {
      ErrorHeapAlloc (RequestPtr, __FILE__, __LINE__);
      ConcludeProcessing (RequestPtr);
      return (false);
   }
   memcpy (sptr, cptr, zptr-cptr);
   sptr[zptr-cptr] = '\0';
   cptr = zptr;

   /* get any query string, or create an empty one */
   if (*cptr == '?') cptr++;
   zptr = cptr;
   while (*zptr && !isspace(*zptr) && *zptr != '\r' && *zptr != '\n') zptr++;
   if ((RequestPtr->QueryStringPtr = sptr =
       HeapAlloc (RequestPtr, zptr-cptr+1)) == NULL)
   {
      ErrorHeapAlloc (RequestPtr, __FILE__, __LINE__);
      ConcludeProcessing (RequestPtr);
      return (false);
   }
   memcpy (sptr, cptr, zptr-cptr);
   sptr[zptr-cptr] = '\0';

   return (true);
}

/*****************************************************************************/
/*
Scan the header buffer looking for relevant fields. Return true to continue
processing the request, false to stop immediately.  If false this function or
any it called must have reported the problem to the client and/or disposed of
the thread.
*/ 
 
boolean RequestFields (struct RequestStruct *RequestPtr)

{
   register char  *cptr, *hptr, *sptr, *zptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "RequestFields()\n");

   /* ensure any old field values are ignored (in the case of redirection) */
   while (RequestPtr->HttpAcceptCount)
      RequestPtr->HttpAcceptPtrArray[RequestPtr->HttpAcceptCount--] = NULL;
   RequestPtr->HttpAuthorizationPtr = NULL;
   RequestPtr->ContentLength[0] = '\0';
   RequestPtr->PostContentTypePtr = NULL;
   RequestPtr->HttpIfModifiedSincePtr = NULL;
   RequestPtr->IfModifiedSinceBinaryTime[0] =
      RequestPtr->IfModifiedSinceBinaryTime[1] = 0;
   RequestPtr->HttpRefererPtr = NULL;
   RequestPtr->HttpUserAgentPtr = NULL;

   /* the header is already null-terminated */
   hptr = RequestPtr->RequestHeaderPtr;
   /* look for the request-terminating blank line */
   while (*hptr)
   {
      cptr = hptr;
      while (*hptr && *hptr != '\r' && *hptr != '\n') hptr++;
      if (hptr[0] == '\n' || (hptr[0] == '\r' && hptr[1] == '\n'))
      {
         /* break if blank line (i.e. end-of-header) */
         if (cptr == hptr) break;
      }
      /* step over the carriage control at the end of the line */
      if (*hptr == '\r') hptr++;
      if (*hptr == '\n') hptr++;
/*
      if (Debug)
         fprintf (stdout, "field |%*.*s|\n", hptr-cptr, hptr-cptr, cptr);
*/

#if 0
      /***********/
      /* Accept: */
      /***********/

      /* reduce function call overhead by explicitly checking the first two */
      if (toupper(cptr[0]) == 'A' && toupper(cptr[1]) == 'C' &&
          strsame (cptr, "Accept:", 7))
      {
         if (RequestPtr->HttpAcceptCount >= MaxRequestAccept)
         {
            Accounting.RequestErrorCount++;
            RequestPtr->ResponseStatusCode = 500;
            ErrorGeneral (RequestPtr, ErrorRequestMaxAccept,
                          __FILE__, __LINE__);
            ConcludeProcessing (RequestPtr);
            return (false);
         }
         cptr += 7;
         while (*cptr && isspace(*cptr) && *cptr != '\r' && *cptr != '\n')
             cptr++;
         zptr = cptr;
         while (*zptr && *zptr != '\r' && *zptr != '\n') zptr++;
         if ((sptr = HeapAlloc (RequestPtr, zptr-cptr+1))
             == NULL)
         {
            ErrorHeapAlloc (RequestPtr, __FILE__, __LINE__);
            ConcludeProcessing (RequestPtr);
            return (false);
         }
         memcpy (sptr, cptr, zptr-cptr);
         sptr[zptr-cptr] = '\0';
         RequestPtr->HttpAcceptPtrArray[RequestPtr->HttpAcceptCount++] = sptr;
         if (Debug)
            fprintf (stdout, "Accept[%d] |%s|\n",
                     RequestPtr->HttpAcceptCount, sptr);
         continue;
      }
#endif

      /******************/
      /* Authorization: */
      /******************/

      /* reduce function call overhead by explicitly checking the first two */
      if (toupper(cptr[0]) == 'A' && toupper(cptr[1]) == 'U' &&
          strsame (cptr, "Authorization:", 14))
      {
         cptr += 14;
         while (*cptr && isspace(*cptr) && *cptr != '\r' && *cptr != '\n')
            cptr++;
         zptr = cptr;
         while (*zptr && *zptr != '\r' && *zptr != '\n') zptr++;
         if ((RequestPtr->HttpAuthorizationPtr = sptr =
              HeapAlloc (RequestPtr, zptr-cptr+1)) == NULL)
         {
            ErrorHeapAlloc (RequestPtr, __FILE__, __LINE__);
            ConcludeProcessing (RequestPtr);
            return (false);
         }
         memcpy (sptr, cptr, zptr-cptr);
         sptr[zptr-cptr] = '\0';
         /* overwrite the authorization string in the buffer with spaces */
         memset (cptr, 0x20, zptr-cptr);
         if (Debug) fprintf (stdout, "Authorization |%s|\n", sptr);

         continue;
      }

      /*******************/
      /* Content-Length: */
      /*******************/

      /* reduce function call overhead by explicitly checking the first two */
      if (toupper(cptr[0]) == 'C' && toupper(cptr[1]) == 'O' &&
          strsame (cptr, "Content-Length:", 15))
      {
         cptr += 15;
         while (*cptr && isspace(*cptr) && *cptr != '\r' && *cptr != '\n')
             cptr++;
         zptr = (sptr = RequestPtr->ContentLength) +
                sizeof(RequestPtr->ContentLength);
         while (isdigit(*cptr) && sptr < zptr) *sptr++ = *cptr++;
         if (sptr >= zptr)
         {
            Accounting.RequestErrorCount++;
            RequestPtr->ResponseStatusCode = 400;
            ErrorGeneral (RequestPtr, ErrorRequestFormat, __FILE__, __LINE__);
            ConcludeProcessing (RequestPtr);
            return (false);
         }
         *sptr = '\0';
         if (!isdigit(RequestPtr->ContentLength[0]))
         {
            Accounting.RequestErrorCount++;
            RequestPtr->ResponseStatusCode = 400;
            ErrorGeneral (RequestPtr, ErrorRequestFormat, __FILE__, __LINE__);
            ConcludeProcessing (RequestPtr);
            return (false);
         }
         if (Debug)
            fprintf (stdout, "ContentLength |%s|\n", RequestPtr->ContentLength);
         continue;
      }

      /*****************/
      /* Content-Type: */
      /*****************/

      /* reduce function call overhead by explicitly checking the first two */
      if (toupper(cptr[0]) == 'C' && toupper(cptr[1]) == 'O' &&
          strsame (cptr, "Content-Type:", 13))
      {
         cptr += 13;
         while (*cptr && isspace(*cptr) && *cptr != '\r' && *cptr != '\n')
             cptr++;
         zptr = cptr;
         while (*zptr && *zptr != '\r' && *zptr != '\n') zptr++;
         if ((RequestPtr->PostContentTypePtr = sptr =
              HeapAlloc (RequestPtr, zptr-cptr+1)) == NULL)
         {
            ErrorHeapAlloc (RequestPtr, __FILE__, __LINE__);
            ConcludeProcessing (RequestPtr);
            return (false);
         }
         memcpy (sptr, cptr, zptr-cptr);
         sptr[zptr-cptr] = '\0';
         if (Debug)
            fprintf (stdout, "PostContentType |%s|\n", sptr);
         continue;
      }

      /**********************/
      /* If-Modified-Since: */
      /**********************/

      /* reduce function call overhead by explicitly checking the first two */
      if (toupper(cptr[0]) == 'I' && toupper(cptr[1]) == 'F' &&
          strsame (cptr, "If-Modified-Since:", 18))
      {
         cptr += 18;
         while (*cptr && isspace(*cptr) && *cptr != '\r' && *cptr != '\n')
             cptr++;
         zptr = cptr;
         while (*zptr && *zptr != '\r' && *zptr != '\n') zptr++;
         if ((RequestPtr->HttpIfModifiedSincePtr = sptr =
              HeapAlloc (RequestPtr, zptr-cptr+1)) == NULL)
         {
            ErrorHeapAlloc (RequestPtr, __FILE__, __LINE__);
            ConcludeProcessing (RequestPtr);
            return (false);
         }
         memcpy (sptr, cptr, zptr-cptr);
         sptr[zptr-cptr] = '\0';
         if (Debug) fprintf (stdout, "HttpIfModifiedSince |%s|\n", sptr);

         if (VMSnok (HttpGmTime (sptr, &RequestPtr->IfModifiedSinceBinaryTime)))
         {
            if (Debug) fprintf (stdout, "If-Modified-Since: NBG!\n");
            RequestPtr->HttpIfModifiedSincePtr = NULL;
            RequestPtr->IfModifiedSinceBinaryTime[0] =
            RequestPtr->IfModifiedSinceBinaryTime[1] = 0;
         }

         continue;
      }

      /************/
      /* Referer: */
      /************/

      /* reduce function call overhead by explicitly checking the first two */
      if (toupper(cptr[0]) == 'R' && toupper(cptr[1]) == 'E' &&
          strsame (cptr, "Referer:", 8))
      {
         cptr += 8;
         while (*cptr && isspace(*cptr) && *cptr != '\r' && *cptr != '\n')
             cptr++;
         zptr = cptr;
         while (*zptr && *zptr != '\r' && *zptr != '\n') zptr++;
         if ((RequestPtr->HttpRefererPtr = sptr =
             HeapAlloc (RequestPtr, zptr-cptr+1)) == NULL)
         {
            ErrorHeapAlloc (RequestPtr, __FILE__, __LINE__);
            ConcludeProcessing (RequestPtr);
            return (false);
         }
         memcpy (sptr, cptr, zptr-cptr);
         sptr[zptr-cptr] = '\0';
         if (Debug) fprintf (stdout, "Referer |%s|\n", sptr);
         continue;
      }

      /***************/
      /* User-Agent: */
      /***************/

      /* reduce function call overhead by explicitly checking the first two */
      if (toupper(cptr[0]) == 'U' && toupper(cptr[1]) == 'S' &&
          strsame (cptr, "User-Agent:", 11))
      {
         cptr += 11;
         while (*cptr && isspace(*cptr) && *cptr != '\r' && *cptr != '\n')
             cptr++;
         zptr = cptr;
         while (*zptr && *zptr != '\r' && *zptr != '\n') zptr++;
         if ((RequestPtr->HttpUserAgentPtr = sptr =
             HeapAlloc (RequestPtr, zptr-cptr+1)) == NULL)
         {
            ErrorHeapAlloc (RequestPtr, __FILE__, __LINE__);
            ConcludeProcessing (RequestPtr);
            return (false);
         }
         memcpy (sptr, cptr, zptr-cptr);
         sptr[zptr-cptr] = '\0';
         if (Debug) fprintf (stdout, "UserAgent |%s|\n", sptr);
         continue;
      }
   }

   return (true);
}

/*****************************************************************************/
/*
Execute GET and POST requests.

If a file specification does not contain a file name the function looks for a
home page in the directory.  If no home page found it generates a directory
listing ("Index of" documents).

If the file specification contains a wildcard and there is no query string a
directory listing (index) is generated.  If there is a query string it must
begin with the literal "httpd=index" for a directory listing (index) to be
generated (this allows directives in the query string to be passed to the
directory listing functions).  If it does not begin with this literal the
default query (search) script is invoked.

Detects file types that have an associated script.  If a script is returned by
ConfigContentType(), and the file specification does not include a specific
version number (even such as ";0"), an automatic redirection is generated so
that the specified document is processed by the script, and the output from
that returned to the client.  If it does contain a version number the file is
always directly returned to the client.
*/ 

int RequestExecute (struct RequestStruct *RequestPtr)

{
   register char  *cptr, *sptr;

   int  status,
        Count;
   unsigned short  Length;
   char  FileSpec [1024],
         ScriptSpec [256],
         String [512];
   struct FAB  ParseFab;
   struct NAM  ParseNam;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "RequestExecute()\n");

   /* ensure any old values are ignored, in case its a redirection */
   RequestPtr->LocationPtr = NULL;

   /* request is underway, initialize the output (response) timeout */
   if (VMSnok (status = TimeoutSet (RequestPtr, TIMEOUT_OUTPUT)))
   {
      ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
      ConcludeProcessing (RequestPtr);
      return;
   }

   /*****************************/
   /* internal server directive */
   /*****************************/

   if (tolower(RequestPtr->QueryStringPtr[0]) == 'h')
   {
      if (strsame (RequestPtr->QueryStringPtr, "httpd=report", -1))
      {
         if (RequestPtr->AdjustAccounting)
         {
            Accounting.DoInternalCount++;
            RequestPtr->AdjustAccounting = 0;
         }
         ServerReport (RequestPtr);
         return;
      }
   }

   /****************/
   /* map the path */
   /****************/

   FileSpec[0] = RequestPtr->ScriptName[0] = ScriptSpec[0] = '\0';

   /* map the path, converting it to a VMS file path (or script file name) */
   sptr = MapUrl (RequestPtr->PathInfoPtr, FileSpec,
                  RequestPtr->ScriptName, ScriptSpec);
   if (!sptr[0] && sptr[1])
   {
      Accounting.RequestForbiddenCount++;
      RequestPtr->ResponseStatusCode = 403;
      ErrorGeneral (RequestPtr, sptr+1, __FILE__, __LINE__);
      ConcludeProcessing (RequestPtr);
      return;
   }

   if (Debug)
      fprintf (stdout, "|%s|%s|%s|%s|%s|\n",
               RequestPtr->PathInfoPtr, FileSpec,
               RequestPtr->ScriptName, ScriptSpec, sptr);

   /********************************************/
   /* check for redirection (from the mapping) */
   /********************************************/

   /* find full redirection URL (i.e. the "://" out of "http://host/path") */
   for (cptr = sptr; *cptr && *cptr != '/'; cptr++)
      if ((*(unsigned long*)cptr & 0x00ffffff) == 0x002f2f3a) break;
   if (*cptr == ':')
   {
      if ((RequestPtr->LocationPtr = HeapAlloc (RequestPtr, strlen(sptr)+1))
          == NULL)
      {
         ErrorHeapAlloc (RequestPtr, __FILE__, __LINE__);
         ConcludeProcessing (RequestPtr);
         return;
      }
      strcpy (RequestPtr->LocationPtr, sptr);
      ConcludeProcessing (RequestPtr);
      return;
   }

   /******************************/
   /* check for script execution */
   /******************************/

   if (RequestPtr->ScriptName[0])
   {
      /* get the path derived from the script specification */
      if ((RequestPtr->PathInfoPtr =
          HeapRealloc (RequestPtr, RequestPtr->PathInfoPtr, 
                       (Length=strlen(sptr)+1))) == NULL)
      {
         ErrorHeapAlloc (RequestPtr, __FILE__, __LINE__);
         ConcludeProcessing (RequestPtr);
         return (false);
      }
      memcpy (RequestPtr->PathInfoPtr, sptr, Length);

      RequestScript (RequestPtr, ScriptSpec, FileSpec);
      return;
   }

   if (RequestPtr->MethodPost)
   {
      strcpy (RequestPtr->ScriptName, Config.DefaultPost);
      MapUrl_UrlToVms (RequestPtr->ScriptName, ScriptSpec);
      RequestScript (RequestPtr, ScriptSpec, FileSpec);
      return;
   }

   /********************************/
   /* parse the file specification */
   /********************************/

   ParseFab = cc$rms_fab;
   ParseFab.fab$l_fna = FileSpec;
   ParseFab.fab$b_fns = strlen(FileSpec);
   ParseFab.fab$l_fop = FAB$M_NAM;
   ParseFab.fab$l_nam = &ParseNam;
   ParseNam = cc$rms_nam;
   ParseNam.nam$l_esa = RequestPtr->FileName;
   ParseNam.nam$b_ess = sizeof(RequestPtr->FileName)-1;
   /* parse without disk I/O, syntax check only! */
   ParseNam.nam$b_nop = NAM$M_SYNCHK;

   if (VMSnok (status = sys$parse (&ParseFab, 0, 0)))
   {
      Accounting.RequestRmsErrorCount++;
      RequestPtr->ErrorHiddenTextPtr = FileSpec;
      ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
      ConcludeProcessing (RequestPtr);
      return;
   }

   /* terminate after the version number (if any) */
   ParseNam.nam$l_ver[ParseNam.nam$b_ver] = '\0';
   if (Debug) fprintf (stdout, "FileName |%s|\n", RequestPtr->FileName);

   if ((tolower(RequestPtr->QueryStringPtr[0]) == 'h' &&
        strsame (RequestPtr->QueryStringPtr, "httpd=index", 11)) ||
       ((ParseNam.nam$l_fnb & NAM$M_WILDCARD) &&
        !RequestPtr->QueryStringPtr[0]))
   {
      /******************************/
      /* generate directory listing */
      /******************************/

      if (Config.DirWildcardEnabled)
      {
         strcpy (RequestPtr->DirSpec, FileSpec);
         RequestPtr->DirPathInfoPtr = RequestPtr->PathInfoPtr;
         RequestPtr->DirQueryStringPtr = RequestPtr->QueryStringPtr;
         DirBegin (RequestPtr);
         return;
      }
      else
      {
         Accounting.RequestForbiddenCount++;
         RequestPtr->ResponseStatusCode = 501;
         ErrorGeneral (RequestPtr, ErrorFacilityDisabled, __FILE__, __LINE__);
         ConcludeProcessing (RequestPtr);
         return;
      }
   }

   if (ParseNam.nam$l_name[0] == '.' && ParseNam.nam$l_name[1] == ';' &&
      !RequestPtr->QueryStringPtr[0])
   {
      /*********************************************/
      /* no file name supplied, look for home page */
      /*********************************************/

      /* loop through the possible home page file names */
      for (Count = 0; *(cptr = ConfigHomePage(Count)); Count++)
      {
         /* overwrite any existing file name in the NAM block */
         strcpy (ParseNam.nam$l_name, cptr);
         /* locate the start of the file type */
         for (sptr = ParseNam.nam$l_name; *sptr && *sptr != '.'; sptr++);
         /* set the mime content type for this file type */
         ConfigContentType (RequestPtr, sptr);
         /* if this file is not found then try the next from the array */
         if ((status = RequestFile (RequestPtr)) == RMS$_FNF) continue;
         /* break for any other status (normal or error) */
         break;
      }
      if (VMSnok (status) && status != RMS$_FNF)
      {
         Accounting.RequestRmsErrorCount++;
         strcpy (ParseNam.nam$l_name, "home-page-of-some-sort");
         RequestPtr->ErrorHiddenTextPtr = ParseNam.nam$l_dev;
         ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
         ConcludeProcessing (RequestPtr);
         return;
      }

      /*
         One of the home pages existed.
         Let asynchronous processing handle it from here!
      */
      if (*cptr) return;

      /********************************************/
      /* no home page, generate directory listing */
      /********************************************/

      ParseNam.nam$l_name[0] = '\0';
      strcpy (RequestPtr->DirSpec, ParseNam.nam$l_dev);
      RequestPtr->DirPathInfoPtr = RequestPtr->PathInfoPtr;
      RequestPtr->DirQueryStringPtr = RequestPtr->QueryStringPtr;
      DirBegin (RequestPtr);
      return;
   }

   /********************************/
   /* get the content-type details */
   /********************************/

   ConfigContentType (RequestPtr, ParseNam.nam$l_type);

   if (!isdigit(ParseNam.nam$l_ver[1]))
   {
      /* 
         If a file specification does not include a specific version
         number (which indicates send-me-this-file-regardless) and
         ConfigContentType() has returned a script name, indicating
         an automatic script handling for this file type, then redirect.
      */
      if (RequestPtr->AutoScriptNamePtr[0] == '/')
      {
         /********************************/
         /* automatic script redirection */
         /********************************/

         Accounting.DoAutoScriptCount++;

         Count = strlen(RequestPtr->AutoScriptNamePtr);
         if (RequestPtr->PathInfoPtr != NULL)
            Count += strlen(RequestPtr->PathInfoPtr);
         if (RequestPtr->QueryStringPtr[0])
         {
             /* allow one extra character for the query-string leading '?' */
             Count++;
             Count += strlen(RequestPtr->QueryStringPtr);
         }
         /* allow for the terminating null */
         Count++;

         if ((RequestPtr->LocationPtr = sptr = HeapAlloc (RequestPtr, Count))
             == NULL)
         {
            ErrorHeapAlloc (RequestPtr, __FILE__, __LINE__);
            ConcludeProcessing (RequestPtr);
            return;
         }

         cptr = RequestPtr->AutoScriptNamePtr;
         while (*cptr) *sptr++ = *cptr++;
         if (RequestPtr->PathInfoPtr != NULL)
         {
            cptr = RequestPtr->PathInfoPtr;
            while (*cptr) *sptr++ = *cptr++;
         }
         if (RequestPtr->QueryStringPtr[0])
         {
            *sptr++ = '?';
            cptr = RequestPtr->QueryStringPtr;
            while (*cptr) *sptr++ = *cptr++;
         }
         *sptr = '\0';
         if (Debug)
            fprintf (stdout, "LocationPtr |%s|\n", RequestPtr->LocationPtr);

         /* dispose of thread function does the actual redirection */
         ConcludeProcessing (RequestPtr);
         return;
      }
   }

   /**************************************/
   /* internally processed content-types */
   /**************************************/

   if (strsame (RequestPtr->ContentTypePtr, Config.IsMapContentType, -1))
   {
      /* image mapping configuration file (has a query string!) */
      IsMapBegin (RequestPtr);
      return;
   }

   /******************************************************************/
   /* query string (without script name), an implicit keyword search */
   /******************************************************************/

   if (RequestPtr->QueryStringPtr[0])
   {
      strcpy (RequestPtr->ScriptName, Config.KeywordSearch);
      MapUrl_UrlToVms (RequestPtr->ScriptName, ScriptSpec);
      RequestScript (RequestPtr, ScriptSpec, FileSpec);
      return;
   }

   /****************************************/ 
   /* otherwise ... send the file contents */
   /****************************************/ 

   /* if greater than one, counters are adjusted even if there is an error */
   RequestPtr->AdjustAccounting = 2;
   if (VMSnok (status = RequestFile (RequestPtr)))
   {
      RequestPtr->ErrorHiddenTextPtr = RequestPtr->FileName;
      ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
      ConcludeProcessing (RequestPtr);
      return;
   }
}

/*****************************************************************************/
/*
Send a file to the client.  Special cases are menu and pre-processed HTML.  
*/ 

int RequestFile (struct RequestStruct *RequestPtr)

{
   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "RequestFile() |%s|%s|\n",
               RequestPtr->FileName, RequestPtr->ContentTypePtr);

   if (strsame (RequestPtr->ContentTypePtr, Config.MenuContentType, -1))
      return (MenuBegin (RequestPtr));

   if (strsame (RequestPtr->ContentTypePtr, Config.ShtmlContentType, -1))
   {
      /* pre-processed HTML file (a.k.a. Server Side Includes) */
      strcpy (RequestPtr->ShtmlFileName, RequestPtr->FileName);
      RequestPtr->FileName[0] = '\0';
      return (ShtmlBegin (RequestPtr));
   }

   return (FileBegin (RequestPtr));
}

/*****************************************************************************/
/*
Process request by executing a script.  The 'FileSpec' was the non-script-name 
portion of the path, returned by the original MapUrl().
*/ 

RequestScript
(
struct RequestStruct *RequestPtr,
char *ScriptSpec,
char *FileSpec
)
{
   register char  *sptr;

   int  status;
   struct FAB  ParseFab;
   struct NAM  ParseNam;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "RequestScript() |%s|%s|%s|\n",
               RequestPtr->ScriptName, FileSpec, RequestPtr->PathInfoPtr);

   if (FileSpec[0])
   {
      /* parse the file specification */
      ParseFab = cc$rms_fab;
      ParseFab.fab$l_fna = FileSpec;
      ParseFab.fab$b_fns = strlen(FileSpec);
      ParseFab.fab$l_fop = FAB$M_NAM;
      ParseFab.fab$l_nam = &ParseNam;
      ParseNam = cc$rms_nam;
      ParseNam.nam$l_esa = RequestPtr->FileName;
      ParseNam.nam$b_ess = sizeof(RequestPtr->FileName)-1;
      /* parse without disk I/O, syntax check only! */
      ParseNam.nam$b_nop = NAM$M_SYNCHK;

      if (VMSnok (status = sys$parse (&ParseFab, 0, 0)))
      {
         Accounting.RequestRmsErrorCount++;
         RequestPtr->ErrorHiddenTextPtr = FileSpec;
         ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
         ConcludeProcessing (RequestPtr);
         return;
      }

      ParseNam.nam$l_ver[ParseNam.nam$b_ver] = '\0';

      ConfigContentType (RequestPtr, ParseNam.nam$l_type);
   }
   else
      RequestPtr->FileName[0] = '\0';

   if (Debug)
      fprintf (stdout, "RequestPtr->FileName |%s|\n", RequestPtr->FileName);

   DclBegin (RequestPtr, NULL, ScriptSpec);
}

/*****************************************************************************/

