/*****************************************************************************/
/*
                                 sHTML.c

This module implements a full multi-threaded, AST-driven, asynchronous HTML
pre-processor (also known as Server Side Includes).  The AST-driven nature 
makes the code a little more difficult to follow, but creates a powerful, 
event-driven, multi-threaded server.  All of the necessary functions 
implementing this module are designed to be non-blocking. 

This module uses the BufferOutput() function to buffer any output it can into
larger chunks before sending it to the client.

A pre-processor directive CANNOT BE SPLIT over multiple lines. 

Included files are automatically encapsulated in <PRE></PRE> tags and HTML-
escaped if not "text/html" content-type (i.e. if they are "text/plain" 
content-type).  They can be forced to be directly included by including a 
'par="text/html"' parameter.  Conversly, "text/html" files can be forced to be 
included as plain-text using the 'par="text/plain"' parameter. 

Including another .SHTML file will only have its contents included, it WILL 
NOT be preprocessed. 

Format ('fmt=""') specifications for time values follow those allowed by 
strftime().  If none is specified it defaults to a fairly standard looking 
VMS-style time.

*PRIVILEGED* directives are only allowed in documents owned by SYSTEM ([1,4])
and that are NOT world writable!


DIRECTIVES
----------

<!--#accesses [since=""] [timefmt="[]"] -->   number of accesses of document

<!--#config errmsg="" -->               (not implemented, ignored)
<!--#config timefmt="" -->              set default time format
<!--#config sizefmt="" -->              set file size output format

<!--#dcl dir="" [par=""] -->            DIRECTORY file-spec [qualifiers]
<!--#dcl vdir="" [par=""] -->           DIRECTORY virtual-file-spec [qualifiers]
<!--#dcl show="" -->                    SHOW command
<!--#dcl say="" -->                     WRITE SYS$OUTPUT 'anything'

<!--#dcl exec="" -->                    *PRIVILEGED* execute any DCL command
<!--#dcl file="" [par=""] -->           *PRIVILEGED* execute command procedure
<!--#dcl run="" [par=""] -->            *PRIVILEGED* execute image
<!--#dcl virtual="" [par=""] -->        *PRIVILEGED* execute command procedure
<!--#dcl vrun="" [par=""] -->           *PRIVILEGED* execute virtual image

<!--#dir file="" [par=""] -->           directory (index of) file spec
<!--#dir virtual="" [par=""] -->        directory (index of) virtual spec
<!--#index file="" [par=""] -->         synonym for the above
<!--#index virtual="" [par=""] -->      synonym for the above

<!--#echo created[=fmt] -->             current document creation date/time
<!--#echo date_local[=fmt] -->          current local date/time
<!--#echo date_gmt[=fmt] -->            current GMT date/time
<!--#echo document_name -->             current document URL path
<!--#echo file_name -->                 current document VMS file path
<!--#echo last_modified[=fmt] -->       current document last modified date

<!--#fcreated file="" [fmt=""] -->      specified file creation date/time
<!--#fcreated virtual="" [fmt=""] -->   specified URL document creation date
<!--#flastmod file="" [fmt=""] -->      specified file last modified date/time
<!--#flastmod virtual="" [fmt=""] -->   specified URL document last modified 
<!--#fsize file="" -->                  specified file size (bytes, Kb, Mb)
<!--#fsize virtual="" -->               specified URL document size

<!--#include file="" [type=""] -->      include the file's contents
<!--#include virtual="" [type=""] -->   include the URL document contents


VERSION HISTORY
---------------
01-DEC-95  MGD  new for HTTPd version 3
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <time.h>

/* VMS related header files */
#include <atrdef.h>
#include <descrip.h>
#include <fibdef.h>
#include <iodef.h>
#include <prvdef.h>
#include <rms.h>
#include <ssdef.h>
#include <stsdef.h>

/* application-related header files */
#include "httpd.h"

#define DefaultTimeFormat "%Od-%b-%Y %T"  /* a little VMSish! */
#define DefaultSizeFormat "abbrev"

#define FILE_FCREATED 1
#define FILE_FLASTMOD 2
#define FILE_FSIZE 3

/********************/
/* external storage */
/********************/

extern boolean  Debug;
extern int  FileBufferSize;
extern char  SoftwareID[];
extern struct AccountingStruct  Accounting;
extern struct ConfigStruct  Config;

/****************************/
/* functions in this module */
/****************************/

ShtmlAccesses (struct RequestStruct*, void*, char*);
ShtmlAccessCount (struct RequestStruct*, unsigned long*, unsigned long*);
ShtmlBegin (struct RequestStruct*);
ShtmlDcl (struct RequestStruct*, void*, char*);
ShtmlDir (struct RequestStruct*, void*, char*);
ShtmlDirective (struct RequestStruct*);
ShtmlEcho (struct RequestStruct*, void*, char*);
ShtmlEnd (struct RequestStruct*);
ShtmlIncludeFile (struct RequestStruct*, void*, char*);
ShtmlNextRecord (struct RequestStruct*);
ShtmlNextRecordAST (struct RAB*);
ShtmlParse (struct RequestStruct*);
int ShtmlTimeString (struct RequestStruct*, unsigned long*, char*, char*, int);

/**********************/
/* external functions */
/**********************/

int BufferOutput (struct RequestStruct*, void*, char*, int);
ConcludeProcessing (struct RequestStruct*);
DclBegin (struct RequestStruct*, char*, char*);
ErrorExitVmsStatus (int, char*, char*, int);
ErrorExitVmsStatus (int, char*, char*, int);
ErrorGeneral (struct RequestStruct*, char*, char*, int);
ErrorVmsStatus (struct RequestStruct*, int, char*, int);
unsigned char* HeapAlloc (struct RequestStruct*, int);
int TimeAdjustGMT (boolean, unsigned long*);
int TimeVmsToUnix (unsigned long*, struct tm*);
MapUrl_VirtualPath (char*, char*, char*, int);
QioNetWrite (struct RequestStruct*, void*, char*, int);

/*****************************************************************************/
/*
Open the specified file.  If there is a problem opening it then just return 
the status, do not report an error or anything.  This is used to determine 
whether the file exists, as when attempting to open one of multiple, possible 
home pages.

As pre-processed HTML files are by definition dynamic, no check of any
"If-Modified-Since:" request field date/time is made, and no "Last-Modified:" 
field is included in any response header. 

When successfully opened and connected generate an HTTP header, if required.
Once open and connected the pre-processing becomes AST-driven.

'RequestPtr->ShtmlFileName' contains the VMS file specification.
'RequestPtr->ContentTypePtr' contains the MIME content type.
'RequestPtr->SendHttpHeader' is true if an HTTP header should be generated.
*/

ShtmlBegin (struct RequestStruct *RequestPtr)

{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "ShtmlBegin() |%s|\n", RequestPtr->ShtmlFileName);

   if (!Config.ShtmlEnabled)
   {
      if (RequestPtr->AdjustAccounting)
      {
         Accounting.DoShtmlCount++;
         RequestPtr->AdjustAccounting = 0;
      }
      RequestPtr->ResponseStatusCode = 501;
      ErrorGeneral (RequestPtr, "HTML pre-processing is disabled.",
                    __FILE__, __LINE__);
      ShtmlEnd (RequestPtr);
      return (SS$_NORMAL);
   }

   /* set up default error report information (just in case!) */
   RequestPtr->ErrorTextPtr = RequestPtr->PathInfoPtr;
   RequestPtr->ErrorHiddenTextPtr = RequestPtr->ShtmlFileName;

   RequestPtr->ShtmlFileFab = cc$rms_fab;
   RequestPtr->ShtmlFileFab.fab$b_fac = FAB$M_GET;
   RequestPtr->ShtmlFileFab.fab$l_fna = RequestPtr->ShtmlFileName;  
   RequestPtr->ShtmlFileFab.fab$b_fns = strlen(RequestPtr->ShtmlFileName);
   RequestPtr->ShtmlFileFab.fab$b_shr = FAB$M_SHRGET;

   /* initialize the NAM block */
   RequestPtr->ShtmlFileNam = cc$rms_nam;
   RequestPtr->ShtmlFileFab.fab$l_nam = &RequestPtr->ShtmlFileNam;
   RequestPtr->ShtmlFileNam.nam$l_esa = RequestPtr->ExpandedFileName;
   RequestPtr->ShtmlFileNam.nam$b_ess = sizeof(RequestPtr->ExpandedFileName)-1;

   /* initialize the date extended attribute block */
   RequestPtr->ShtmlFileXabDat = cc$rms_xabdat;
   RequestPtr->ShtmlFileFab.fab$l_xab = &RequestPtr->ShtmlFileXabDat;

   /* initialize and link in the protection extended attribute block */
   RequestPtr->ShtmlFileXabPro = cc$rms_xabpro;
   RequestPtr->ShtmlFileXabDat.xab$l_nxt = &RequestPtr->ShtmlFileXabPro;

   if (VMSnok (status = sys$open (&RequestPtr->ShtmlFileFab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$open() %%X%08.08X\n", status);

      /* if its a search list treat directory not found as if file not found */
      if ((RequestPtr->FileNam.nam$l_fnb & NAM$M_SEARCH_LIST) &&
          status == RMS$_DNF)
         status = RMS$_FNF;

      if (RequestPtr->AdjustAccounting > 1)
      {
         /* greater than one adjusts counters even if there is an error */
         Accounting.DoShtmlCount++;
         RequestPtr->AdjustAccounting = 0;
      }
      return (status);
   }

   if (RequestPtr->AdjustAccounting)
   {
      Accounting.DoShtmlCount++;
      RequestPtr->AdjustAccounting = 0;
   }

   /* allocate heap memory for the file record buffer */
   if (RequestPtr->ShtmlBufferPtr == NULL)
   {
      /* allow two bytes for carriage control and terminating null */
      if ((RequestPtr->ShtmlBufferPtr =
          HeapAlloc (RequestPtr, FileBufferSize+2)) == NULL)
      {
         ErrorHeapAlloc (RequestPtr, __FILE__, __LINE__);
         ShtmlEnd (RequestPtr);
         return (SS$_NORMAL);
      }
   }

   RequestPtr->ShtmlFileRab = cc$rms_rab;
   RequestPtr->ShtmlFileRab.rab$l_fab = &RequestPtr->ShtmlFileFab;
   /* 2 buffers and read ahead performance option */
   RequestPtr->ShtmlFileRab.rab$b_mbf = 2;
   RequestPtr->ShtmlFileRab.rab$l_rop = RAB$M_RAH;
   RequestPtr->ShtmlFileRab.rab$l_ubf = RequestPtr->ShtmlBufferPtr;
   RequestPtr->ShtmlFileRab.rab$w_usz = FileBufferSize;

   if (VMSnok (status = sys$connect (&RequestPtr->ShtmlFileRab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$connect() %%X%08.08X\n", status);
      ShtmlEnd (RequestPtr);
      return (SS$_NORMAL);
   }

   /********************/
   /* begin processing */
   /********************/

   /* make sure any "If-Modified-Since:" request header line is ignored */
   RequestPtr->IfModifiedSinceBinaryTime[0] =
      RequestPtr->IfModifiedSinceBinaryTime[1] = 0;

   if (RequestPtr->SendHttpHeader)
   {
      if (VMSnok (status = ShtmlHttpHeader (RequestPtr)))
      {
         ShtmlEnd (RequestPtr);
         return (SS$_NORMAL);
      }
      if (RequestPtr->MethodHead)
      {
         ShtmlEnd (RequestPtr);
         return (SS$_NORMAL);
      }
   }

   RequestPtr->ShtmlTimeFmtPtr = DefaultTimeFormat;
   RequestPtr->ShtmlSizeFmtPtr = DefaultSizeFormat;

   /* set then RAB user context storage to the client thread pointer */
   RequestPtr->ShtmlFileRab.rab$l_ctx = RequestPtr;

   RequestPtr->ShtmlLineNumber = 0;

   ShtmlNextRecord (RequestPtr);

   /*
      Return to the calling routine now.  Subsequent processing is
      event-driven.  Routine completion ASTs drive the pre-processing.
   */
   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
Ensure the source file is closed.  Flush the output buffer, ensuring the AST-
driven next task function gets executed (if any).
*/ 

ShtmlEnd (struct RequestStruct *RequestPtr)

{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ShtmlEnd()\n");

   if (RequestPtr->ShtmlFileFab.fab$w_ifi)
   {
      sys$close (&RequestPtr->ShtmlFileFab, 0, 0);
      RequestPtr->ShtmlFileFab.fab$w_ifi = 0;
   }

   ConcludeProcessing (RequestPtr);
}

/*****************************************************************************/
/*
Generates an HTTP header for the following data transfer.
*/ 
 
int ShtmlHttpHeader (struct RequestStruct *RequestPtr)

{
   static $DESCRIPTOR (StringDsc, "");

   static $DESCRIPTOR (CommentedInfoFaoDsc,
"<!!-- SoftwareID: !AZ -->\n\
<!!-- !AZ -->\n");

   static $DESCRIPTOR (Http200HeaderFaoDsc,
"!AZ 200 Data follows.\r\n\
Server: !AZ\r\n\
Date: !AZ\r\n\
Content-Type: text/html\r\n\
\r\n");

   int  status;
   unsigned short  CommentLength,
                   Length,
                   HeaderLength;
   char  String [1024];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ShtmlHttpHeader()\n");

   RequestPtr->ResponseStatusCode = 200;

   StringDsc.dsc$a_pointer = String;
   StringDsc.dsc$w_length = sizeof(String)-1;

   sys$fao (&Http200HeaderFaoDsc, &HeaderLength, &StringDsc,
            HttpProtocol, SoftwareID, RequestPtr->GmDateTime);

   if (Config.IncludeCommentedInfo && !RequestPtr->MethodHead)
   {
      StringDsc.dsc$a_pointer = String + HeaderLength;
      StringDsc.dsc$w_length = sizeof(String) - HeaderLength - 1;

      status = sys$fao (&CommentedInfoFaoDsc, &CommentLength, &StringDsc,
                        SoftwareID, RequestPtr->ShtmlFileName);
   }
   else
      CommentLength = 0;

   String[Length = HeaderLength+CommentLength] = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", String);

   /* allocate heap memory for the response header */
   if ((RequestPtr->ResponseHeaderPtr =
        HeapAlloc (RequestPtr, Length)) == NULL)
   {
      ErrorHeapAlloc (RequestPtr, __FILE__, __LINE__);
      return (SS$_INSFMEM);
   }
   memcpy (RequestPtr->ResponseHeaderPtr, String, Length);

   if (VMSnok (status =
       QioNetWrite (RequestPtr, 0, String, Length)))
      ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);

   RequestPtr->SendHttpHeader = false;

   return (status);
}

/*****************************************************************************/
/*
Queue an asynchronous read of the next pre-processed file record.
*/ 
 
int ShtmlNextRecord (struct RequestStruct *RequestPtr)

{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ShtmlNextRecord()\n");

   /* queue the next record read */
   sys$get (&RequestPtr->ShtmlFileRab, &ShtmlNextRecordAST,
            &ShtmlNextRecordAST);

   /*
      Return to the calling routine now.  Subsequent processing is
      event-driven.  Routine completion ASTs drive the pre-processing.
   */
}

/*****************************************************************************/
/*
The asynchronous read of the next pre-processed file record has completed.
*/ 

ShtmlNextRecordAST (struct RAB *RabPtr)

{
   register char  *cptr, *sptr;

   int  status;
   struct RequestStruct  *RequestPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout,
      "ShtmlNextRecordAST() sts: %%X%08.08X stv: %%X%08.08X\n",
      RabPtr->rab$l_sts, RabPtr->rab$l_stv);

   /* get the pointer to the thread from the RAB user context storage */
   RequestPtr = RabPtr->rab$l_ctx;

   if (VMSnok (RequestPtr->ShtmlFileRab.rab$l_sts))
   {
      if (RequestPtr->ShtmlFileRab.rab$l_sts == RMS$_EOF)
      {
         /***************/
         /* end of file */
         /***************/

         ShtmlEnd (RequestPtr);
         return;
      }

      /**********************/
      /* error reading file */
      /**********************/

      ErrorVmsStatus (RequestPtr, RequestPtr->ShtmlFileRab.rab$l_sts,
                      __FILE__, __LINE__);
      ShtmlEnd (RequestPtr);
      return;
   }

   /*****************************/
   /* process the record (line) */
   /*****************************/

   /* first check if any error occured during previous any processing */
   if (RequestPtr->ErrorMessagePtr != NULL)
   {
      ShtmlEnd (RequestPtr);
      return;
   }

   RequestPtr->ShtmlLineNumber++;

   /* terminate the line */
   RequestPtr->ShtmlFileRab.rab$l_ubf
      [RequestPtr->ShtmlFileRab.rab$w_rsz] = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", RequestPtr->ShtmlFileRab.rab$l_ubf);

   /* this pointer keeps track of the current parse position in the line */
   RequestPtr->ShtmlParsePtr = RequestPtr->ShtmlFileRab.rab$l_ubf;

   /* begin parsing the line looking for commented pre-processor directives */
   ShtmlParseLine (RequestPtr);

   /*
      Return to the AST-interrupted routine now.  Continued processing is
      event-driven.  Routine completion ASTs drive the pre-processing.
   */
}

/*****************************************************************************/
/*
'RequestPtr->ShtmlParsePtr' points to the currently parsed-up-to position on 
the line.  Continue parsing that line looking for preprocessor directives.
*/ 

ShtmlParseLine (struct RequestStruct *RequestPtr)

{
   register char  *cptr, *sptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "ShtmlParseLine()\n|%s|\n", RequestPtr->ShtmlParsePtr);

   /* first check if any error occured during any previous processing */
   if (RequestPtr->ErrorMessagePtr != NULL)
   {
      ShtmlEnd (RequestPtr);
      return;
   }

   /* ensure the escape-HTML flag is turned off from any previous processing */
   RequestPtr->BufferOutputEscapeHtml = false;

   /* find the start of any comments/directives */
   for (cptr = sptr = RequestPtr->ShtmlParsePtr; *sptr; sptr++)
   {
      if (*sptr != '<') continue;
      /* numeric "!--#" is a bit more efficient */
      if (*(unsigned long*)(sptr+1) == 0x232d2d21) break;
   }

   if (*sptr == '<')
   {
      /***************************/
      /* pre-processor directive */
      /***************************/

      RequestPtr->ShtmlParsePtr = sptr;
      if (sptr - cptr)
      {
         /* buffer the part line */
         if (VMSnok (BufferOutput (RequestPtr, &ShtmlDirective,
                     cptr, sptr-cptr)))
            ShtmlEnd (RequestPtr);
         return;
      }
      ShtmlDirective (RequestPtr);
      return;
   }
   else
   {
      /************************************/
      /* send line (or last part thereof) */
      /************************************/

      /* end-of-line, ensure it has correct carriage control */
      if (sptr > RequestPtr->ShtmlFileRab.rab$l_ubf)
      {
         if (sptr[-1] != '\n')
         {
            *sptr++ = '\n';
            *sptr = '\0';
         }
      }
      else
      {
         /* blank (empty) line */
         *sptr++ = '\n';
         *sptr = '\0';
      }

      if (VMSnok (BufferOutput (RequestPtr, &ShtmlNextRecord,
                                cptr, sptr-cptr)))
         ShtmlEnd (RequestPtr);
      return;
   }
}

/*****************************************************************************/
/*
'RequestPtr->ShtmlParsePtr' points to the start of a pre-processor directive.  
Parse that directive and execute it, or provide an error message.
*/ 

ShtmlDirective (struct RequestStruct *RequestPtr)

{
   register char  *dptr, *sptr;

   boolean  ok;
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "ShtmlDirective()\n|%s|\n", RequestPtr->ShtmlParsePtr);

   /* find the end of the commented directive and reset parse pointer */
   dptr = sptr = RequestPtr->ShtmlParsePtr + 4;
   /* numeric "-->" is a bit more efficient */
   while (*sptr && (*(unsigned long*)(sptr) & 0x00ffffff) != 0x003e2d2d) sptr++;

   if (*sptr)
      RequestPtr->ShtmlParsePtr = sptr + 3;
   else
   {
      ShtmlExplainError (RequestPtr, "Unterminated directive",
                         dptr, __FILE__, __LINE__);
      ShtmlEnd (RequestPtr);
      return;
   }

   if (strsame (dptr, "#ACCESSES ", 10))
      ShtmlAccesses (RequestPtr, &ShtmlParseLine, dptr);
   else
   if (strsame (dptr, "#CONFIG ", 8))
      ShtmlConfig (RequestPtr, &ShtmlParseLine, dptr);
   else
   if (strsame (dptr, "#DCL ", 5) || strsame (dptr, "#EXEC ", 6))
      ShtmlDcl (RequestPtr, &ShtmlParseLine, dptr);
   else
   if (strsame (dptr, "#DIR ", 5) || strsame (dptr, "#INDEX ", 7))
      ShtmlDir (RequestPtr, &ShtmlParseLine, dptr);
   else
   if (strsame (dptr, "#ECHO ", 6))
      ShtmlEcho (RequestPtr, &ShtmlParseLine, dptr);
   else
   if (strsame (dptr, "#FCREATED ", 10))
      ShtmlFCreated (RequestPtr, &ShtmlParseLine, dptr);
   else
   if (strsame (dptr, "#FLASTMOD ", 10))
      ShtmlFLastMod (RequestPtr, &ShtmlParseLine, dptr);
   else
   if (strsame (dptr, "#FSIZE ", 7))
      ShtmlFSize (RequestPtr, &ShtmlParseLine, dptr);
   else
   if (strsame (dptr, "#INCLUDE ", 9))
      ShtmlIncludeFile (RequestPtr, &ShtmlParseLine, dptr);
   else
   {
      ShtmlExplainError (RequestPtr, "Unknown directive",
                         dptr, __FILE__, __LINE__);
      ShtmlEnd (RequestPtr);
      return;
   }
}

/*****************************************************************************/
/*
Number of times this document has been accessed.

This operation is fairly expensive in terms of I/O, and because it is atomic 
introduces a fair degree of granularity.  Its expense means the "#accesses" 
functionality should be used sparingly.  This functionality can be disabled. 
*/ 

ShtmlAccesses
(
struct RequestStruct *RequestPtr,
void *AstFunctionPtr,
char *DirectivePtr
)

{
   static $DESCRIPTOR (AccessesFaoDsc, "!UL!AZ!AZ");
   static $DESCRIPTOR (StringDsc, "");

   register char  *dptr;

   boolean  SupplySinceTime;
   unsigned short  Length;
   unsigned long  AccessCount;
   unsigned long  SinceBinTime [2];
   char  SinceText [256],
         String [1024],
         TimeFormat [256],
         TimeString [256];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ShtmlAccesses() |%s|\n", DirectivePtr);

   if (!Config.ShtmlAccessesEnabled)
   {
      ShtmlExplainError (RequestPtr, "<I>#accesses</I> directive disabled",
                         dptr, __FILE__, __LINE__);
      ShtmlEnd (RequestPtr);
      return;
   }

   SinceText[0] = TimeFormat[0] = TimeString[0] = '\0';
   SupplySinceTime = false;

   dptr = DirectivePtr;
   while (*dptr && !isspace(*dptr)) dptr++;

   while (*dptr)
   {
      while (*dptr && isspace(*dptr)) dptr++;
      /* numeric "-->" is a bit more efficient */
      if ((*(unsigned long*)(dptr) & 0x00ffffff) == 0x003e2d2d) break;

      if (strsame (dptr, "SINCE=", 6))
      {
         dptr += ShtmlGetTagValue (RequestPtr, DirectivePtr, dptr, SinceText);
         SupplySinceTime = true;
      }
      else
      if (strsame (dptr, "TIMEFMT=", 8))
         dptr += ShtmlGetTagValue (RequestPtr, DirectivePtr, dptr, TimeFormat);
      else
      {
         ShtmlExplainError (RequestPtr, "Unknown tag",
                            DirectivePtr, __FILE__, __LINE__);
         ShtmlEnd (RequestPtr);
         return;
      }
   }

   if (VMSnok (ShtmlAccessCount (RequestPtr, &AccessCount, &SinceBinTime)))
   {
      ShtmlEnd (RequestPtr);
      return;
   }

   if (SupplySinceTime)
      if (!ShtmlTimeString (RequestPtr, &SinceBinTime, TimeFormat,
                            TimeString, sizeof(TimeString)))
      {
         ShtmlEnd (RequestPtr);
         return;
      }

   StringDsc.dsc$a_pointer = String;
   StringDsc.dsc$w_length = sizeof(String)-1;
   sys$fao (&AccessesFaoDsc, &Length, &StringDsc,
            AccessCount, SinceText, TimeString);
   String[Length] = '\0';

   if (VMSnok (BufferOutput (RequestPtr, AstFunctionPtr, String, Length)))
      ShtmlEnd (RequestPtr);
}

/*****************************************************************************/
/*
Config sets the default behaviour for a specific action for the rest of the 
document.
*/ 

ShtmlConfig
(
struct RequestStruct *RequestPtr,
void *AstFunctionPtr,
char *DirectivePtr
)
{
   register char  *cptr, *dptr;

   int  status;
   char  TagValue [256];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ShtmlConfig() |%s|\n", DirectivePtr);

   dptr = DirectivePtr;
   while (*dptr && !isspace(*dptr)) dptr++;
   while (*dptr && isspace(*dptr)) dptr++;

   if (strsame (dptr, "ERRMSG", 6))
   {
      /* this directive is completely ignored :^) */
   }
   else
   if (strsame (dptr, "TIMEFMT", 7))
   {
      dptr += ShtmlGetTagValue (RequestPtr, DirectivePtr, dptr, TagValue);
      if ((cptr = HeapAlloc (RequestPtr, strlen(TagValue)+1)) == NULL)
      {
         ErrorHeapAlloc (RequestPtr, __FILE__, __LINE__);
         ShtmlEnd (RequestPtr);
         return;
      }
      strcpy (RequestPtr->ShtmlTimeFmtPtr = cptr, TagValue);
   }
   else
   if (strsame (dptr, "SIZEFMT", 7))
   {
      dptr += ShtmlGetTagValue (RequestPtr, DirectivePtr, dptr, TagValue);
      if (!(strsame (TagValue, "abbrev", -1) ||
            strsame (TagValue, "bytes", -1) ||
            strsame (TagValue, "blocks", -1)))
      {
         ShtmlExplainError (RequestPtr, "Invalid tag value",
                            DirectivePtr, __FILE__, __LINE__);
         ShtmlEnd (RequestPtr);
         return;
      }
      if ((cptr = HeapAlloc (RequestPtr, strlen(TagValue)+1)) == NULL)
      {
         ErrorHeapAlloc (RequestPtr, __FILE__, __LINE__);
         ShtmlEnd (RequestPtr);
         return;
      }
      strcpy (RequestPtr->ShtmlSizeFmtPtr = cptr, TagValue);
   }
   else
   {
      ShtmlExplainError (RequestPtr, "Unknown tag",
                         DirectivePtr, __FILE__, __LINE__);
      ShtmlEnd (RequestPtr);
      return;
   }

   /* declare an AST to execute the required function */
   if (VMSnok (status = sys$dclast (AstFunctionPtr, RequestPtr, 0)))
   {
      ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
      ShtmlEnd (RequestPtr);
      return;
   }
   if (Debug) fprintf (stdout, "sys$dclast() %%X%08.08X\n", status);
}

/*****************************************************************************/
/*
Process a DCL/EXEC directive.  The array at the beginning of this function
provides  the allowed DCL directives and their DCL command equivalents.  Only
innocuous  DCL commands are allowed (hopefully! e.g. SHOW, WRITE SYS$OUTPUT)
for the  average document.  For documents owned by SYSTEM and NOT WORLD
WRITEABLE (for  obvious reasons) the execution of any DCL command or command
procedure is  allowed allowing maximum flexibility for the privileged document
author.
*/ 

ShtmlDcl
(
struct RequestStruct *RequestPtr,
void *AstFunctionPtr,
char *DirectivePtr
)
{
   /*
      First element is allowed DCL command.
      Second is actual DCL verb/command/syntax.
      Third, if non-empty, indicates it is for privileged documents only!
   */
   static char  *SupportedDcl [] =
   {
      "cmd", "", "", "P",                    /* privileged only! */
      "dir", "directory ", "", "",           /* any document */
      "exec", "", "", "P",                   /* privileged only! */
      "file", "@", "Y", "P",                 /* privileged only! */
      "run", "run ", "Y", "P",               /* privileged only! */
      "say", "write sys$output", "", "",     /* any document */
      "show", "show", "", "",                /* any document */
      "vdir", "directory ", "", "",          /* any document */
      "virtual", "@", "Y", "P",              /* privileged only! */
      "vrun", "run ", "Y", "P",              /* privileged only! */
      "", "", "*"  /* array must be terminated by empty/privileged command! */
   };

   register int  idx;
   register char  *cptr, *dptr, *sptr, *zptr;

   int  status;
   char  DclCommand [1024],
         TagValue [256];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ShtmlDcl() |%s|\n", DirectivePtr);

   dptr = DirectivePtr;
   while (*dptr && !isspace(*dptr)) dptr++;
   while (*dptr && isspace(*dptr)) dptr++;

   for (idx = 0; *SupportedDcl[idx]; idx += 4)
   {
      if (Debug)
         fprintf (stdout, "|%s|%s|\n", SupportedDcl[idx], SupportedDcl[idx+1]);
      /* compare the user-supplied DCL command to the list of allowed */
      cptr = SupportedDcl[idx];
      sptr = dptr;
      while (*cptr && *sptr != '=' && toupper(*cptr) == toupper(*sptr))
         { cptr++; sptr++; }
      if (!*cptr && *sptr == '=') break;
   }

   if (!SupportedDcl[idx][0] && !strsame (DirectivePtr, "#exec ", 6))
   {
      /************************/
      /* unsupported command! */
      /************************/

      ShtmlExplainError (RequestPtr, "Unsupported DCL",
                         DirectivePtr, __FILE__, __LINE__);
      ShtmlEnd (RequestPtr);
      return;
   }

   /*************************/
   /* supported DCL command */
   /*************************/

   if (SupportedDcl[idx+3][0])
   {
      /******************************/
      /* "privileged" DCL requested */
      /******************************/

      if (!Config.ShtmlExecEnabled)
      {
         ShtmlExplainError (RequestPtr, "DCL execution disabled", 
                            DirectivePtr, __FILE__, __LINE__);
         ShtmlEnd (RequestPtr);
         return;
      }
      /* SYSTEM is UIC [1,4] */
      if (RequestPtr->ShtmlFileXabPro.xab$w_grp != 1 ||
          RequestPtr->ShtmlFileXabPro.xab$w_mbm != 4)
      {
         ShtmlExplainError (RequestPtr, "Document must be owned by SYSTEM",
                            DirectivePtr, __FILE__, __LINE__);
         ShtmlEnd (RequestPtr);
         return;
      }
      /* protect word: wwggooss, protect bits: dewr, 1 denies access */
      if (!(RequestPtr->ShtmlFileXabPro.xab$w_pro & 0x2000))
      {
         ShtmlExplainError (RequestPtr, "Document cannot be WORLD writable",
                            DirectivePtr, __FILE__, __LINE__);
         ShtmlEnd (RequestPtr);
         return;
      }
   }

   /****************************************************/
   /* create the DCL command from array and parameters */
   /****************************************************/

   cptr = SupportedDcl[idx+1];
   sptr = DclCommand;
   while (*cptr) *sptr++ = *cptr++;
   *sptr = '\0';

   if (SupportedDcl[idx+2][0])
   {
      /* file specification involved */
      dptr += ShtmlGetFileSpec (RequestPtr, DirectivePtr, dptr, TagValue);
      for (cptr = TagValue; *cptr; *sptr++ = *cptr++);
      *sptr = '\0';
   }
   else
   {
      /* some DCL verb/parameters or another */
      dptr += ShtmlGetTagValue (RequestPtr, DirectivePtr, dptr, TagValue);
      if (TagValue[0] && sptr > DclCommand) *sptr++ = ' ';
      for (cptr = TagValue; *cptr; *sptr++ = *cptr++);
      *sptr = '\0';
   }

   if (Debug) fprintf (stdout, "DclCommand |%s|\n", DclCommand);

   /* by default HTML-forbidden characters in DCL output are escaped */
   RequestPtr->BufferOutputEscapeHtml = true;

   while (*dptr)
   {
      while (*dptr && isspace(*dptr)) dptr++;
      /* numeric "-->" is a bit more efficient */
      if ((*(unsigned long*)(dptr) & 0x00ffffff) == 0x003e2d2d) break;

      if (strsame (dptr, "PAR=", 4))
      {
         /* parameters to the command procedure, directory, etc. */
         dptr += ShtmlGetTagValue (RequestPtr, DirectivePtr, dptr, TagValue);
         if (TagValue[0]) *sptr++ = ' ';
         for (cptr = TagValue; *cptr; *sptr++ = *cptr++);
         *sptr = '\0';
      }
      else
      if (strsame (dptr, "TYPE=", 5))
      {
         dptr += ShtmlGetTagValue (RequestPtr, DirectivePtr, dptr, TagValue);
         if (TagValue[0] && strsame (TagValue, "text/html", -1))
            RequestPtr->BufferOutputEscapeHtml = false;
         else
            RequestPtr->BufferOutputEscapeHtml = true;
      }
      else
      if (*dptr)
      {
         ShtmlExplainError (RequestPtr, "Unknown tag",
                            DirectivePtr, __FILE__, __LINE__);
         ShtmlEnd (RequestPtr);
         return;
      }
   }

   /* ensure the CGI variables contain no bogus information */
   RequestPtr->FileName[0] = '\0';

   RequestPtr->NextTaskFunction = AstFunctionPtr;

   DclBegin (RequestPtr, DclCommand, NULL);
}

/*****************************************************************************/
/*
Generate an "Index of" directory listing by calling DirBegin() task.
*/

ShtmlDir
(
struct RequestStruct *RequestPtr,
void *AstFunctionPtr,
char *DirectivePtr
)
{
   register char  *cptr, *dptr;

   int  status;
   char  TagValue [256];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ShtmlDir() |%s|\n", DirectivePtr);

   TagValue[0] = '\0';

   dptr = DirectivePtr;
   while (*dptr && !isspace(*dptr)) dptr++;

   while (*dptr)
   {
      while (*dptr && isspace(*dptr)) dptr++;
      /* numeric "-->" is a bit more efficient */
      if ((*(unsigned long*)(dptr) & 0x00ffffff) == 0x003e2d2d) break;

      if (strsame (dptr, "FILE=", 5) || strsame (dptr, "VIRTUAL=", 8))
         dptr += ShtmlGetFileSpec (RequestPtr, DirectivePtr, dptr,
                                   RequestPtr->DirSpec);
      else
      if (strsame (dptr, "PAR=", 4))
         dptr += ShtmlGetTagValue (RequestPtr, DirectivePtr, dptr, TagValue);
      else
      {
         ShtmlExplainError (RequestPtr, "Unknown tag",
                            DirectivePtr, __FILE__, __LINE__);
         ShtmlEnd (RequestPtr);
         return;
      }
   }

   if (Debug) fprintf (stdout, "|%s|%s|\n", RequestPtr->DirSpec, TagValue);
   RequestPtr->DirPathInfoPtr = RequestPtr->DirSpec;
   if (TagValue[0])
   {
      if ((cptr = HeapAlloc (RequestPtr, strlen(TagValue)+1)) == NULL)
      {
         ErrorHeapAlloc (RequestPtr, __FILE__, __LINE__);
         ShtmlEnd (RequestPtr);
         return;
      }
      strcpy (RequestPtr->DirQueryStringPtr = cptr, TagValue);
   }
   else
      RequestPtr->DirQueryStringPtr = "";

   RequestPtr->NextTaskFunction = AstFunctionPtr;

   DirBegin (RequestPtr);
}

/*****************************************************************************/
/*
Output the special variable.
*/

boolean ShtmlEcho
(
struct RequestStruct *RequestPtr,
void *AstFunctionPtr,
char *DirectivePtr
)
{
   register char  *cptr, *dptr, *sptr, *zptr;

   int  status,
        Length;
   unsigned long  BinTime [2];
   char  *StringPtr;
   char  TimeFormat [256],
         TimeString [256];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ShtmlEcho() |%s|\n", DirectivePtr);

   dptr = DirectivePtr;
   while (*dptr && !isspace(*dptr)) dptr++;
   while (*dptr && isspace(*dptr)) dptr++;

   if (strsame (dptr, "DATE_LOCAL", 10))
   {
      ShtmlGetTagValue (RequestPtr, DirectivePtr, dptr+10, TimeFormat);
      if (!(Length = ShtmlTimeString (RequestPtr, NULL, TimeFormat,
                                      TimeString, sizeof(TimeString))))
      {
         ShtmlEnd (RequestPtr);
         return;
      }
      StringPtr = TimeString;
   }
   else
   if (strsame (dptr, "DATE_GMT", 8))
   {
      ShtmlGetTagValue (RequestPtr, DirectivePtr, dptr+10, TimeFormat);
      sys$gettim (&BinTime);
      TimeAdjustGMT (true, &BinTime);
      if (!(Length = ShtmlTimeString (RequestPtr, &BinTime, TimeFormat,
                                      TimeString, sizeof(TimeString))))
      {
         ShtmlEnd (RequestPtr);
         return;
      }
      memcpy (TimeString+Length, " GMT", 5);
      Length += 4;
      StringPtr = TimeString;
   }
   else
   if (strsame (dptr, "DOCUMENT_NAME", 13))
      Length = strlen(StringPtr = RequestPtr->PathInfoPtr);
   else
   if (strsame (dptr, "CREATED", 7))
   {
      ShtmlGetTagValue (RequestPtr, DirectivePtr, dptr+7, TimeFormat);
      if (!(Length =
            ShtmlTimeString (RequestPtr,
                             &RequestPtr->ShtmlFileXabDat.xab$q_cdt,
                             TimeFormat, TimeString, sizeof(TimeString))))
      {
         ShtmlEnd (RequestPtr);
         return;
      }
      StringPtr = TimeString;
   }
   else
   if (strsame (dptr, "FILE_NAME", 9))
      Length = strlen(StringPtr = RequestPtr->ShtmlFileName);
   else
   if (strsame (dptr, "LAST_MODIFIED", 13))
   {
      ShtmlGetTagValue (RequestPtr, DirectivePtr, dptr+13, TimeFormat);
      if (!(Length =
            ShtmlTimeString (RequestPtr,
                             &RequestPtr->ShtmlFileXabDat.xab$q_rdt,
                             TimeFormat, TimeString, sizeof(TimeString))))
      {
         ShtmlEnd (RequestPtr);
         return;
      }
      StringPtr = TimeString;
   }
   else
   {
      ShtmlExplainError (RequestPtr, "Unknown tag",
                         DirectivePtr, __FILE__, __LINE__);
      ShtmlEnd (RequestPtr);
      return;
   }

   if (VMSnok (BufferOutput (RequestPtr, AstFunctionPtr, StringPtr, Length)))
      ShtmlEnd (RequestPtr);
}

/*****************************************************************************/
/*
Output the specified file's creation date and time.
*/

boolean ShtmlFCreated
(
struct RequestStruct *RequestPtr,
void *AstFunctionPtr,
char *DirectivePtr
)
{
   register char  *dptr, *sptr, *zptr;

   char  FileName [256],
         TimeFormat [256];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ShtmlFCreated() |%s|\n", DirectivePtr);

   FileName[0] = TimeFormat[0] = '\0';

   dptr = DirectivePtr;
   while (*dptr && !isspace(*dptr)) dptr++;

   while (*dptr)
   {
      while (*dptr && isspace(*dptr)) dptr++;
      /* numeric "-->" is a bit more efficient */
      if ((*(unsigned long*)(dptr) & 0x00ffffff) == 0x003e2d2d) break;

      if (strsame (dptr, "FILE=", 5) || strsame (dptr, "VIRTUAL=", 8))
         dptr += ShtmlGetFileSpec (RequestPtr, DirectivePtr, dptr, FileName);
      else
      if (strsame (dptr, "FMT=", 4))
         dptr += ShtmlGetTagValue (RequestPtr, DirectivePtr, dptr, TimeFormat);
      else
      {
         ShtmlExplainError (RequestPtr, "Unknown tag",
                            DirectivePtr, __FILE__, __LINE__);
         ShtmlEnd (RequestPtr);
         return;
      }
   }

   ShtmlFileDetails (RequestPtr, AstFunctionPtr,
                     FileName, FILE_FCREATED, TimeFormat);
}

/*****************************************************************************/
/*
Output the specified file's last modification date and time.
*/

ShtmlFLastMod
(
struct RequestStruct *RequestPtr,
void *AstFunctionPtr,
char *DirectivePtr
)
{
   register char  *dptr, *sptr, *zptr;

   char  FileName [256],
         TimeFormat [256];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ShtmlFLastMod() |%s|\n", DirectivePtr);

   FileName[0] = TimeFormat[0] = '\0';

   dptr = DirectivePtr;
   while (*dptr && !isspace(*dptr)) dptr++;

   while (*dptr)
   {
      while (*dptr && isspace(*dptr)) dptr++;
      /* numeric "-->" is a bit more efficient */
      if ((*(unsigned long*)(dptr) & 0x00ffffff) == 0x003e2d2d) break;

      if (strsame (dptr, "FILE=", 5) || strsame (dptr, "VIRTUAL=", 8))
         dptr += ShtmlGetFileSpec (RequestPtr, DirectivePtr, dptr, FileName);
      else
      if (strsame (dptr, "FMT=", 4))
         dptr += ShtmlGetTagValue (RequestPtr, DirectivePtr, dptr, TimeFormat);
      else
      {
         ShtmlExplainError (RequestPtr, "Unknown tag",
                            DirectivePtr, __FILE__, __LINE__);
         ShtmlEnd (RequestPtr);
         return;
      }
   }

   ShtmlFileDetails (RequestPtr, AstFunctionPtr,
                     FileName, FILE_FLASTMOD, TimeFormat);
}

/*****************************************************************************/
/*
Output the specified file's size.
*/

ShtmlFSize
(
struct RequestStruct *RequestPtr,
void *AstFunctionPtr,
char *DirectivePtr
)
{
   register char  *dptr;

   char  FileSize [32],
         FileName [256],
         SizeFormat [256];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ShtmlFSize() |%s|\n", DirectivePtr);

   FileName[0] = SizeFormat[0] = '\0';

   dptr = DirectivePtr;
   while (*dptr && !isspace(*dptr)) dptr++;

   while (*dptr)
   {
      while (*dptr && isspace(*dptr)) dptr++;
      /* numeric "-->" is a bit more efficient */
      if ((*(unsigned long*)(dptr) & 0x00ffffff) == 0x003e2d2d) break;

      if (strsame (dptr, "FILE=", 5) || strsame (dptr, "VIRTUAL=", 8))
         dptr += ShtmlGetFileSpec (RequestPtr, DirectivePtr,
                                   dptr, FileName);
      else
      if (strsame (dptr, "FMT=", 4))
         dptr += ShtmlGetTagValue (RequestPtr, DirectivePtr, dptr, SizeFormat);
      else
      {
         ShtmlExplainError (RequestPtr, "Unknown tag",
                            DirectivePtr, __FILE__, __LINE__);
         ShtmlEnd (RequestPtr);
         return;
      }
   }

   ShtmlFileDetails (RequestPtr, AstFunctionPtr,
                     FileName, FILE_FSIZE, SizeFormat);
}

/*****************************************************************************/
/*
Include the contents of the specified file by calling FileBegin() task.
*/

ShtmlIncludeFile
(
struct RequestStruct *RequestPtr,
void *AstFunctionPtr,
char *DirectivePtr
)
{
   register char  *cptr, *dptr;

   int  status;
   char  FileFormat [256];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ShtmlIncludeFile() |%s|\n", DirectivePtr);

   RequestPtr->FileName[0] = FileFormat[0] = '\0';

   dptr = DirectivePtr;
   while (*dptr && !isspace(*dptr)) dptr++;

   while (*dptr)
   {
      while (*dptr && isspace(*dptr)) dptr++;
      /* numeric "-->" is a bit more efficient */
      if ((*(unsigned long*)(dptr) & 0x00ffffff) == 0x003e2d2d) break;

      if (strsame (dptr, "FILE=", 5) || strsame (dptr, "VIRTUAL=", 8))
         dptr += ShtmlGetFileSpec (RequestPtr, DirectivePtr, dptr,
                                   RequestPtr->FileName);
      else
      if (strsame (dptr, "TYPE=", 5))
         dptr += ShtmlGetTagValue (RequestPtr, DirectivePtr, dptr, FileFormat);
      else
      {
         ShtmlExplainError (RequestPtr, "Unknown tag",
                            DirectivePtr, __FILE__, __LINE__);
         ShtmlEnd (RequestPtr);
         return;
      }               
   }

   if (Debug) fprintf (stdout, "|%s|%s|\n", RequestPtr->FileName, FileFormat);

   /* find the file extension and set the content type */
   for (cptr = RequestPtr->FileName; *cptr && *cptr != ']'; cptr++);
   while (*cptr && *cptr != '.') cptr++;
   ConfigContentType (RequestPtr, cptr);
   if (!strsame (RequestPtr->ContentTypePtr, "text/", 5))
   {
      ShtmlExplainError (RequestPtr, "File content type is not text",
                         DirectivePtr, __FILE__, __LINE__);
      ShtmlEnd (RequestPtr);
      return;
   }

   if ((!strsame (RequestPtr->ContentTypePtr, "text/html", -1) &&
        !strsame (FileFormat, "text/html", -1)) ||
       strsame (FileFormat, "text/plain", -1))
      RequestPtr->FileEncapsulateData = true;
   else
      RequestPtr->FileEncapsulateData = false;

   RequestPtr->NextTaskFunction = AstFunctionPtr;

   if (VMSnok (status = FileBegin (RequestPtr)))
   {
      /* numeric " -->" is a bit more efficient */
      while (*dptr && *(unsigned long*)(dptr) != 0x3e2d2d20) dptr++;
      *dptr = '\0';
      RequestPtr->ErrorTextPtr = DirectivePtr;
      ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
      ShtmlEnd (RequestPtr);
      return;
   }

   return (true);
}

/*****************************************************************************/
/*
Using the locale formatting capabilities of function strftime(), output the 
time represented by the specified VMS quadword, binary time.  If 
'BinaryTimePtr' is NULL then default to the current time.  Returns number of 
characters placed into 'TimeString', or zero if an error.
*/ 

int ShtmlTimeString
(
struct RequestStruct *RequestPtr,
unsigned long *BinaryTimePtr,
char *TimeFmtPtr,
char *TimeString,
int SizeOfTimeString
)
{
   int  Length;
   unsigned long  BinaryTime [2];
   struct tm  UnixTime;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ShtmlTimeString() |%s|\n", TimeFmtPtr);

   if (!TimeFmtPtr[0]) TimeFmtPtr = RequestPtr->ShtmlTimeFmtPtr;
   if (BinaryTimePtr == NULL) sys$gettim (BinaryTimePtr = &BinaryTime);
   TimeVmsToUnix (BinaryTimePtr, &UnixTime);

   if (!(Length =
         strftime (TimeString, SizeOfTimeString, TimeFmtPtr, &UnixTime)))
   {
      ShtmlExplainError (RequestPtr, "Problem with date/time format string",
                         TimeFmtPtr, __FILE__, __LINE__);
      ShtmlEnd (RequestPtr);
      return (0);
   }
   return (Length);
}

/*****************************************************************************/
/*
Get the value of a tag parameter (e.g. tag_name="value").  Maximum number of
characters allowed in value is 256.   Returns the number of characters scanned
to get the value.
*/ 

int ShtmlGetTagValue
(
struct RequestStruct *RequestPtr,
char *DirectivePtr,
char *String,
char *Value
)
{
   register char  *sptr, *vptr, *zptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ShtmlGetTagValue() |%s|\n", String);

   sptr = String;

   while (*sptr && *sptr != '=') sptr++;
   if (*sptr) sptr++;
   if (*sptr == '\"') sptr++;
   zptr = (vptr = Value) + 256;
   while (*sptr && *sptr != '\"' && vptr < zptr)
   {
      if (*sptr == '\\') sptr++;
      *vptr++ = *sptr++;
   }
   *vptr = '\0';
   if (*sptr == '\"') sptr++;

   if (Debug) fprintf (stdout, "Value |%s|\n", Value);
   return (sptr - String);
}

/*****************************************************************************/
/*
Get a 'FILE="file_name"' or a 'VIRTUAL="file_name"'.  Maximum number of 
characters allowed in value is 256.   Returns the number of characters scanned 
to get the value. 
*/ 

int ShtmlGetFileSpec
(
struct RequestStruct *RequestPtr,
char *DirectivePtr,
char *String,
char *FileName
)
{
   register char  *fptr, *sptr, *zptr;

   char  VirtualPath [256],
         VirtualFileName [256];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ShtmlGetFileSpec() |%s|\n", String);

   sptr = String;

   if (toupper(*sptr) == 'V')
   {
      /* virtual path to file */
      while (*sptr && *sptr != '=') sptr++;
      if (*sptr) sptr++;
      if (*sptr == '\"') sptr++;
      zptr = (fptr = VirtualPath) + sizeof(VirtualPath);
      while (*sptr && *sptr != ' ' && *sptr != '\"' && fptr < zptr)
      {
         if (*sptr == '\\') sptr++;
         *fptr++ = *sptr++;
      }
      *fptr = '\0';
      if (*sptr == '\"') sptr++;
      if (Debug) fprintf (stdout, "VirtualPath |%s|\n", VirtualPath);

      MapUrl_VirtualPath (RequestPtr->PathInfoPtr, VirtualPath,
                          VirtualFileName, sizeof(VirtualFileName));

      FileName[0] = '\0';
      zptr = MapUrl (VirtualFileName, FileName, NULL, NULL);
      if (!zptr[0])
      {
         ErrorGeneral (RequestPtr, zptr+1, __FILE__, __LINE__);
         return (0);
      }
   }
   else
   {
      while (*sptr && *sptr != '=') sptr++;
      if (*sptr) sptr++;
      if (*sptr == '\"') sptr++;
      zptr = (fptr = FileName) + 256;
      while (*sptr && *sptr != ' ' && *sptr != '\"' && fptr < zptr)
      {
         if (*sptr == '\\') sptr++;
         *fptr++ = *sptr++;
      }
      *fptr = '\0';
      if (*sptr == '\"') sptr++;
   }

   if (Debug) fprintf (stdout, "FileName |%s|\n", FileName);
   return (sptr - String);
}

/*****************************************************************************/
/*
Retrieve and display the specified file's specified attribute (size,
modification time, etc.)  This function completes synchronously, that is it
waits for the single QIO on the file details to complete before assembling the
details for the listing, because of this the function introduces slight
granularity.  It is a candidate for future improvement.

This function uses the ACP-QIO interface detailed in the "OpenVMS I/O User's 
Reference Manual".
*/ 

boolean ShtmlFileDetails
(
struct RequestStruct *RequestPtr,
void *AstFunctionPtr,
char *FileName,
int DisplayThis,
char *FormatPtr
)
{
   static $DESCRIPTOR (AbbrevOneByteFaoDsc, "!UL byte");
   static $DESCRIPTOR (AbbrevBytesFaoDsc, "!UL bytes");
   static $DESCRIPTOR (AbbrevOnekByteFaoDsc, "!UL kbyte");
   static $DESCRIPTOR (AbbrevkBytesFaoDsc, "!UL kbytes");
   static $DESCRIPTOR (AbbrevOneMByteFaoDsc, "!UL Mbyte");
   static $DESCRIPTOR (AbbrevMBytesFaoDsc, "!UL Mbytes");
   static $DESCRIPTOR (OneBlockFaoDsc, "!UL block");
   static $DESCRIPTOR (BlocksFaoDsc, "!UL blocks");
   static $DESCRIPTOR (BytesFaoDsc, "!AZ bytes");
   static $DESCRIPTOR (NumberFaoDsc, "!UL");

   register char  *cptr, *sptr;

   int  status,
        NumBytes;

   unsigned short  AcpChannel,
                   Length;
   unsigned long  AllocatedVbn,
                  Bytes,
                  EndOfFileVbn;
   unsigned long  AtrCdt [2],
                  AtrRdt [2];

   char  ExpandedFileName [256],
         Scratch [256],
         String [256],
         TimeString [256];

   $DESCRIPTOR (DeviceDsc, "");
   $DESCRIPTOR (StringDsc, String);
   $DESCRIPTOR (ScratchDsc, Scratch);


   struct FAB  FileFab;
   struct NAM  FileNam;

   struct {
      unsigned long  OfNoInterest1;
      unsigned short  AllocatedVbnHi;
      unsigned short  AllocatedVbnLo;
      unsigned short  EndOfFileVbnHi;
      unsigned short  EndOfFileVbnLo;
      unsigned short  FirstFreeByte;
      unsigned short  OfNoInterest2;
      unsigned long  OfNoInterestLots [4];
   } AtrRecord;

   struct fibdef  FileFib;
   struct atrdef  FileAtr [] =
   {
      { sizeof(AtrRecord), ATR$C_RECATTR, &AtrRecord },
      { sizeof(AtrCdt), ATR$C_CREDATE, &AtrCdt },
      { sizeof(AtrRdt), ATR$C_REVDATE, &AtrRdt },
      { 0, 0, 0 }
   };

   struct {
      unsigned short  Length;
      unsigned short  Unused;
      unsigned long  Address;
   } FileNameAcpDsc,
     FileFibAcpDsc,
     FileAtrAcpDsc;

   struct {
      unsigned short  Status;
      unsigned short  Unused1;
      unsigned long  Unused2;
   } AcpIOsb;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "ShtmlFileDetails() |%d|%s|\n", DisplayThis, FileName);

   FileFab = cc$rms_fab;
   FileFab.fab$l_fna = FileName;
   FileFab.fab$b_fns = strlen(FileName);
   FileFab.fab$l_fop = FAB$M_NAM;
   FileFab.fab$l_nam = &FileNam;
   FileNam = cc$rms_nam;
   FileNam.nam$l_esa = ExpandedFileName;
   FileNam.nam$b_ess = sizeof(ExpandedFileName)-1;

   if (VMSnok (status = sys$parse (&FileFab, 0, 0)))
   {
      ErrorVmsStatus (RequestPtr, RequestPtr->ShtmlFileRab.rab$l_sts,
                      __FILE__, __LINE__);
      ShtmlEnd (RequestPtr);
      return;
   }

   /* assign a channel to the disk device containing the file */
   DeviceDsc.dsc$w_length = FileNam.nam$b_dev;
   DeviceDsc.dsc$a_pointer = FileNam.nam$l_dev;
   status = sys$assign (&DeviceDsc, &AcpChannel, 0, 0, 0);
   if (Debug) fprintf (stdout, "sys$assign() %%X%08.08X\n", status);
   if (VMSnok (status))
   {
      RequestPtr->ErrorTextPtr = FileNam.nam$l_dev;
      RequestPtr->ErrorHiddenTextPtr = FileName;
      ErrorVmsStatus (RequestPtr, RequestPtr->ShtmlFileRab.rab$l_sts,
                      __FILE__, __LINE__);
      ShtmlEnd (RequestPtr);
      return;
   }

   /*******************************************/
   /* queue an ACP I/O to get file attributes */
   /*******************************************/

   /* set up the File Information Block for the ACP interface */
   memset (&FileFib, 0, sizeof(struct fibdef));
   FileFibAcpDsc.Length = sizeof(FileFib);
   FileFibAcpDsc.Address = &FileFib;
#ifdef __DECC
   FileFib.fib$w_did[0] = FileNam.nam$w_did[0];
   FileFib.fib$w_did[1] = FileNam.nam$w_did[1];
   FileFib.fib$w_did[2] = FileNam.nam$w_did[2];
#else
   FileFib.fib$r_did_overlay.fib$w_did[0] = FileNam.nam$w_did[0];
   FileFib.fib$r_did_overlay.fib$w_did[1] = FileNam.nam$w_did[1];
   FileFib.fib$r_did_overlay.fib$w_did[2] = FileNam.nam$w_did[2];
#endif

   FileNameAcpDsc.Length = FileNam.nam$b_name + FileNam.nam$b_type +
                           FileNam.nam$b_ver;
   FileNameAcpDsc.Address = FileNam.nam$l_name;

   status = sys$qiow (0, AcpChannel, IO$_ACCESS, &AcpIOsb, 0, 0, 
                      &FileFibAcpDsc, &FileNameAcpDsc, 0, 0,
                      &FileAtr, 0);

   sys$dassgn (AcpChannel);

   /* release parse internal data structures */
   FileNam.nam$b_nop = NAM$M_SYNCHK;
   sys$parse (&FileFab, 0, 0);

   if (Debug)
      fprintf (stdout, "sys$qio() %%X%08.08X IOsb: %%X%08.08X\n",
              status, AcpIOsb.Status);

   if (VMSok (status)) status = AcpIOsb.Status;
   if (VMSnok (status)) 
   {
      RequestPtr->ErrorTextPtr = FileNam.nam$l_dev;
      RequestPtr->ErrorHiddenTextPtr = FileName;
      ErrorVmsStatus (RequestPtr, RequestPtr->ShtmlFileRab.rab$l_sts,
                      __FILE__, __LINE__);
      ShtmlEnd (RequestPtr);
      return;
   }

   if (DisplayThis == FILE_FCREATED)
   {
      /*********************/
      /* date/time created */
      /*********************/

      if (!ShtmlTimeString (RequestPtr, &AtrCdt, FormatPtr,
                            String, sizeof(String)))
      {
         ShtmlEnd (RequestPtr);
         return;
      }
      Length = strlen(String);
   }
   else
   if (DisplayThis == FILE_FLASTMOD)
   {
      /*********************/
      /* date/time revised */
      /*********************/

      if (!ShtmlTimeString (RequestPtr, &AtrRdt, FormatPtr,
                            String, sizeof(String)))
      {
         ShtmlEnd (RequestPtr);
         return;
      }
      Length = strlen(String);
   }
   else
   if (DisplayThis == FILE_FSIZE)
   {
      /*************/
      /* file size */
      /*************/

      AllocatedVbn = AtrRecord.AllocatedVbnLo +
                     (AtrRecord.AllocatedVbnHi << 16);
      EndOfFileVbn = AtrRecord.EndOfFileVbnLo +
                     (AtrRecord.EndOfFileVbnHi << 16);

      if (Debug)
         fprintf (stdout,
         "AllocatedVbn: %d EndOfFileVbn: %d FirstFreeByte %d\n",
         AllocatedVbn, EndOfFileVbn, AtrRecord.FirstFreeByte);

      /* the "<< 9" is a bit more efficient than the equivalent "* 512" */
      if (EndOfFileVbn > 1)
         Bytes = (EndOfFileVbn << 9) + AtrRecord.FirstFreeByte;
      else
         Bytes = AtrRecord.FirstFreeByte;
      if (Debug) fprintf (stdout, "Bytes %d\n", Bytes);

      if (!FormatPtr[0]) FormatPtr = RequestPtr->ShtmlSizeFmtPtr;
      if (toupper(FormatPtr[0]) == 'A')  /* "abbrev" */
      {
         if (Bytes < 1024)
         {
            if (Bytes == 1)
               sys$fao (&AbbrevOneByteFaoDsc, &Length, &StringDsc, Bytes);
            else
               sys$fao (&AbbrevBytesFaoDsc, &Length, &StringDsc, Bytes);
         }
         else
         if (Bytes < 1048576)
         {
            if ((NumBytes = Bytes / 1024) == 1)
               sys$fao (&AbbrevOnekByteFaoDsc, &Length, &StringDsc, NumBytes);
            else
               sys$fao (&AbbrevkBytesFaoDsc, &Length, &StringDsc, NumBytes);
         }
         else
         {
            if ((NumBytes = Bytes / 1048576) == 1)
               sys$fao (&AbbrevOneMByteFaoDsc, &Length, &StringDsc, NumBytes);
            else
               sys$fao (&AbbrevMBytesFaoDsc, &Length, &StringDsc, NumBytes);
         }
         String[Length] = '\0';
      }
      else
      if (toupper(FormatPtr[0]) == 'B' &&
          toupper(FormatPtr[1]) == 'Y')  /* "bytes" */
      {
         sys$fao (&NumberFaoDsc, &Length, &ScratchDsc, Bytes);
         Scratch[Length] = '\0';
         sptr = String;
         cptr = Scratch;
         while (Length--)
         {
            *sptr++ = *cptr++;
            if (Length && !(Length % 3)) *sptr++ = ',';
         }
         *sptr = '\0';
         for (cptr = " bytes"; *cptr; *sptr++ = *cptr++);
         Length = sptr - String;
      }
      else
      if (toupper(FormatPtr[0]) == 'B' &&
          toupper(FormatPtr[1]) == 'L')  /* "blocks" */
      {
         if (EndOfFileVbn == 1)
            sys$fao (&OneBlockFaoDsc, &Length, &StringDsc, EndOfFileVbn);
         else
            sys$fao (&BlocksFaoDsc, &Length, &StringDsc, EndOfFileVbn);
         String[Length] = '\0';
      }
   }

   if (VMSnok (BufferOutput (RequestPtr, AstFunctionPtr, String, Length)))
   {
      ShtmlEnd (RequestPtr);
      return (false);
   }
}

/*****************************************************************************/
/*
Keep track of how many times an SHTML file is accessed.  Do this by creating 
another file in the same directory, same name, extension modified by appending 
dollar symbol, containing a single longword record with the binary number of 
times the document has been accessed.  This can be reset by merely deleting 
the access count file.  Provide the access count and creation date of the 
access count file. 

This operation is fairly expensive in terms of I/O, and because it is atomic 
(all occurs during an AST routine, and opens-creates/writes/closes before 
returning to other processing) introduces a fair degree of granularity.  Its 
expense means the "#accesses" functionality should be used sparingly.  This 
functionality can be disabled.
*/ 

ShtmlAccessCount
(
struct RequestStruct *RequestPtr,
unsigned long *AccessCountPtr,
unsigned long *SinceBinTimePtr
)
{
   static unsigned long  SysPrvMask [2] = { PRV$M_SYSPRV, 0 };

   register char  *cptr, *sptr, *zptr;

   int  status,
        CreateStatus,
        SetPrvStatus;
   char  FileName [256];
   struct FAB  FileFab;
   struct RAB  FileRab;
   struct XABDAT  FileXabDat;
   struct XABPRO  FileXabPro;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "ShtmlAccessCount |%s|\n", RequestPtr->ShtmlFileName);

   zptr = (sptr = FileName) + sizeof(FileName)-2;
   for (cptr = RequestPtr->ShtmlFileName;
        *cptr && *cptr != ';' && sptr < zptr;
        *sptr++ = *cptr++);
   if (sptr >= zptr)
   {
      ErrorGeneral (RequestPtr,
         "Document file name too long to create access count file name.",
         __FILE__, __LINE__);
      return (status);
   }
   *sptr++ = '$';
   *sptr = '\0';
   if (Debug) fprintf (stdout, "FileName: %s\n", FileName);

   FileFab = cc$rms_fab;
   FileFab.fab$l_fna = FileName;  
   FileFab.fab$b_fns = sptr-FileName;
   FileFab.fab$l_fop = FAB$M_CIF | FAB$M_SQO;
   FileFab.fab$b_fac = FAB$M_GET | FAB$M_PUT | FAB$M_UPD;
   FileFab.fab$w_mrs = sizeof(unsigned long);
   FileFab.fab$b_shr = FAB$M_SHRGET | FAB$M_SHRPUT | FAB$M_SHRUPD;
   FileFab.fab$b_org = FAB$C_SEQ;
   FileFab.fab$b_rfm = FAB$C_FIX;

   /* initialize the date extended attribute block */
   FileFab.fab$l_xab = &FileXabDat;
   FileXabDat = cc$rms_xabdat;
   FileXabDat.xab$l_nxt = &FileXabPro;

   /* initialize the protection extended attribute block */
   FileXabPro = cc$rms_xabpro;
   FileXabPro.xab$w_pro = 0xaa00;  /* W:RE,G:RE,O:RWED,S:RWED */

   /* turn on SYSPRV to allow access to the counter file */
   if (VMSnok (status = sys$setprv (1, &SysPrvMask, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$setprv() %%X%08.08X\n", status);
      RequestPtr->ErrorTextPtr = "creating access count file";
      RequestPtr->ErrorHiddenTextPtr = FileName;
      ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
      return (status);
   }

   CreateStatus = sys$create (&FileFab, 0, 0);

   /* turn off SYSPRV */
   if (VMSnok (SetPrvStatus = sys$setprv (0, &SysPrvMask, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$setprv() %%X%08.08X\n", SetPrvStatus);
      /* do NOT keep processing if there is a problem turning off SYSPRV! */
      ErrorExitVmsStatus (SetPrvStatus, "reducing privileges",
                          __FILE__, __LINE__);
   }

   if (VMSnok (CreateStatus))
   {
      if (Debug) fprintf (stdout, "sys$create() %%X%08.08X\n", status);
      RequestPtr->ErrorTextPtr = "creating access count file";
      RequestPtr->ErrorHiddenTextPtr = FileName;
      ErrorVmsStatus (RequestPtr, CreateStatus, __FILE__, __LINE__);
      return (CreateStatus);
   }

   FileRab = cc$rms_rab;
   FileRab.rab$l_fab = &FileFab;
   FileRab.rab$l_rbf = AccessCountPtr;
   FileRab.rab$w_rsz = sizeof(*AccessCountPtr);

   if (VMSnok (status = sys$connect (&FileRab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$connect() %%X%08.08X\n", status);
      sys$close (&FileFab, 0, 0);
      RequestPtr->ErrorTextPtr = "connecting access count file";
      RequestPtr->ErrorHiddenTextPtr = FileName;
      ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
      return (status);
   }

   /* copy the creation date (since date) of the access count file */
   memcpy (SinceBinTimePtr, &FileXabDat.xab$q_cdt, 8);

   if (CreateStatus == RMS$_CREATED)
   {
      /***************************************/
      /* count file did not previously exist */
      /***************************************/

      *AccessCountPtr = 1;
      if (Debug) fprintf (stdout, "*AccessCountPtr: %d\n", *AccessCountPtr);

      if (VMSnok (status = sys$put (&FileRab, 0, 0)))
      {
         if (Debug) fprintf (stdout, "sys$put() %%X%08.08X\n", status);
         sys$close (&FileFab, 0, 0);
         RequestPtr->ErrorTextPtr = "updating access count file";
         RequestPtr->ErrorHiddenTextPtr = FileName;
         ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
         return (status);
      }

      sys$close (&FileFab, 0, 0);
      return (SS$_NORMAL);
   }
   else
   {
      /**********************/
      /* count file existed */
      /**********************/

      FileRab.rab$l_ubf = AccessCountPtr;
      FileRab.rab$w_usz = sizeof(*AccessCountPtr);
      if (VMSnok (status = sys$get (&FileRab, 0, 0))) 
      {
         if (Debug) fprintf (stdout, "sys$get() %%X%08.08X\n", status);
         sys$close (&FileFab, 0, 0);
         RequestPtr->ErrorTextPtr = "reading access count file";
         RequestPtr->ErrorHiddenTextPtr = FileName;
         ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
         return (status);
      }

      *AccessCountPtr += 1;
      if (Debug) fprintf (stdout, "*AccessCountPtr: %d\n", *AccessCountPtr);

      if (VMSnok (status = sys$update (&FileRab, 0, 0)))
      {
         if (Debug) fprintf (stdout, "sys$update() %%X%08.08X\n", status);
         sys$close (&FileFab, 0, 0);
         RequestPtr->ErrorTextPtr = "updating access count file";
         RequestPtr->ErrorHiddenTextPtr = FileName;
         ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
         return (status);
      }

      sys$close (&FileFab, 0, 0);
      return (SS$_NORMAL);
   }
}

/*****************************************************************************/
/*
Generate a general error, with explanation about the pre-processor error.
*/ 

int ShtmlExplainError
(
struct RequestStruct *RequestPtr,
char *Explanation,
char *DirectivePtr,
char *SourceFileName,
int SourceLineNumber
)
{
   static $DESCRIPTOR (ErrorMessageFaoDsc,
                       "!AZ (line !UL) ... ``<TT>!AZ</TT>''");

   register char  *dptr;

   unsigned short  Length;
   char  String [1024];
   $DESCRIPTOR (StringDsc, String);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ShtmlExplainError()\n");

   /* numeric " -->" is a bit more efficient */
   dptr = DirectivePtr;
   while (*dptr && *(unsigned long*)(dptr) != 0x3e2d2d20) dptr++;
   *dptr = '\0';

   sys$fao (&ErrorMessageFaoDsc, &Length, &StringDsc,
            Explanation, RequestPtr->ShtmlLineNumber, DirectivePtr);
   String[Length] = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", String);

   ErrorGeneral (RequestPtr, String, SourceFileName, SourceLineNumber);
}

/*****************************************************************************/

