/*****************************************************************************/
/*
                               HyperShelf.c

A CGI-compliant script.

Emulates the 'Bookreader' DECwindows application shelf display and selection 
capability.  HyperShelf however, shows variants of each document if available, 
supported are DECW$BOOK, HTML and PostScript.  The DECW$BOOK documents are 
passed to the 'HyperReader' script, if available.  This script processes 
(many) DECW$BOOK documents for HTML browsers.  The HTML documents are accessed 
directly by the HTTP server via the link URL.  PostScript documents have a 
link for viewing (via the external application) and a link for printing (via 
the 'Print' script, if available). 

Provides the shelf script name, shelf specification and shelf title to 
'HyperReader' to allow it to "close" the book and return immediately to the 
shelf it was opened from (as contrasted to "backing-out" using navigation 
buttons). 

If a shelf specification is not supplied it defaults to "DECW$BOOKSHELF".  If 
a title is not supplied it defaults to "Library".

Requires an HTTPd rule to map "/HyperShelf/internal/*" to whatever directory 
the icons are in (where "HyperShelf" is used as the script name).


FORM FIELDS
-----------
help=           if non-null causes a redirection to the help HTML document
referer=        if non-null provides referer URL for closing library
                (must have any URL-escape percentages URL-escaped!)
shelf=          the name of the shelf to be opened
title=          the title (description) of the shelf being opened


QUALIFIERS
----------
/DBUG           turns on all "if (Debug)" statements
/HELP=          URL of help HTML file (optional)
/HYPERREADER=   name of 'HyperReader' script (optional)
/ICON=          URL location of the icons
/PRINT=         name of 'Print' script (optional)


LOGICAL NAMES
-------------
HTTPD$GMT               timezone offset from Greenwich Mean Time (e.g. "09:30")


BUILD DETAILS
-------------
See BUILD_HYPERSHELF.COM procedure.


VERSION HISTORY (update SoftwareID as well!)
---------------
23-FEB-96  MGD  v2.2.2, bugfix, after modifying the HTTPD$GMT format for the
                        HTTPd server I had forgotten about some scripts
22-NOV-95  MGD  v2.2.1, discovered the TITLE keyword in libraries
12-OCT-95  MGD  v2.2.0, added 'Referer:', 'Last-Modified:'/'If-Modified-Since:'
24-MAY-95  MGD  v2.1.0, minor changes for AXP compatibility
15-APR-95  MGD  v2.0.0, complete rewrite
10-JUN-94  MGD  v1.0.0, initial development
*/
/*****************************************************************************/

#ifdef __ALPHA
   char SoftwareID [] = "HYPERSHELF AXP-2.2.2";
#else
   char SoftwareID [] = "HYPERSHELF VAX-2.2.2";
#endif

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <errno.h>
#include <unixio.h>

/* VMS-related header files */
#include <descrip.h>
#include <libdef.h>
#include <lnmdef.h>
#include <rms.h>
#include <ssdef.h>
#include <stsdef.h>

#define boolean int
#define true 1
#define false 0

#ifdef __ALPHA
#   pragma nomember_alignment
#endif

#define VMSok(x) ((x) & STS$M_SUCCESS)
#define VMSnok(x) (!((x) & STS$M_SUCCESS))
 
#define DefaultShelf "DECW$BOOKSHELF"
#define DecwBook "DECW$BOOK:"

char  Http302Header [] =
"HTTP/1.0 302 Redirection.\r\n\
Location: http://%s%s\r\n\
\r\n";

char  Http404Header [] =
"HTTP/1.0 404 Error report follows.\r\n\
Content-Type: text/html\r\n\
\r\n";

char  Utility [] = "HYPERSHELF";

boolean  Debug,
         HttpHasBeenOutput,
         TimeAheadOfGmt,
         UsingDecwBook;

unsigned long  IfModifiedSinceBinaryTime [2],
               TimeGmtDeltaBinary [2];

char  CgiHttpIfModifiedSince [64],
      CgiHttpReferer [256],
      CgiPathInfo [256],
      CgiPathTranslated [256],
      CgiRequestMethod [16],
      CgiScriptName [256],
      CgiServerName [256],
      FormHelp [16],
      FormTitle [256],
      GmDateTime [32],
      HelpUrl [256],
      HyperReaderScriptName [256],
      HyperShelfIcon [256], 
      IconLocation [256],
      LastModifiedGmDateTime [32],
      OriginalReferer [256],
      PrintScriptName [256], 
      HyperReaderReferer [512],
      TimeGmtString [32],
      TimeGmtVmsString [32],
      UriReferer [512],
      VmsShelfSpec [256];

FILE  *HttpOut;

/* required function prototypes */
char* MapUrl (char*, char*, char*, char*);

/*****************************************************************************/
/*
*/

main (int argc, char *argv[])

{
   register int  acnt;
   register char  *cptr, *sptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   /* open another output stream so that the "\r" and "\n" are not filtered */
#ifdef __DECC
   if ((HttpOut = fopen ("SYS$OUTPUT", "w", "ctx=bin")) == NULL)
      exit (vaxc$errno);
#else
   if ((HttpOut = fopen ("SYS$OUTPUT", "w", "rfm=udf")) == NULL)
      exit (vaxc$errno);
#endif

   /***********************************/
   /* get the command line parameters */
   /***********************************/

   /* using this mechanism parameters must be space-separated! */
   for (acnt = 1; acnt < argc; acnt++)
   {
      if (Debug) fprintf (stdout, "argv[%d] |%s|\n", acnt, argv[acnt]);
      if (strsame (argv[acnt], "/DBUG", -1))
      {
         Debug = true;
         continue;
      }
      if (strsame (argv[acnt], "/HELP=", 5))
      {
         for (cptr = argv[acnt]+5; *cptr && *cptr != '='; cptr++);
         if (*cptr) cptr++;
         sptr = HelpUrl;
         while (*cptr) *sptr++ = *cptr++;
         *sptr = '\0';
         continue;
      }
      if (strsame (argv[acnt], "/HYPERREADER=", 5))
      {
         for (cptr = argv[acnt]+5; *cptr && *cptr != '='; cptr++);
         if (*cptr) cptr++;
         sptr = HyperReaderScriptName;
         while (*cptr) *sptr++ = *cptr++;
         *sptr = '\0';
         continue;
      }
      if (strsame (argv[acnt], "/ICON=", 5))
      {
         for (cptr = argv[acnt]+5; *cptr && *cptr != '='; cptr++);
         if (*cptr) cptr++;
         sptr = IconLocation;
         while (*cptr) *sptr++ = *cptr++;
         if (sptr > IconLocation)
            if (sptr[-1] == '/')
               sptr--;
         *sptr = '\0';
         sprintf (HyperShelfIcon,
            "<IMG ALIGN=top SRC=\"%s/HyperShelf.xbm\" ALT=\"HyperShelf - \"> ",
            IconLocation);
         continue;
      }
      if (strsame (argv[acnt], "/PRINT=", 5))
      {
         for (cptr = argv[acnt]+5; *cptr && *cptr != '='; cptr++);
         if (*cptr) cptr++;
         sptr = PrintScriptName;
         while (*cptr) *sptr++ = *cptr++;
         *sptr = '\0';
         continue;
      }
      fprintf (stdout, "%%%s-E-IVQUAL, unrecognized qualifier\n \\%s\\\n",
               Utility, argv[acnt]+1);
      exit (STS$K_ERROR | STS$M_INHIB_MSG);
   }

   /***********/
   /* process */
   /***********/

   GetCgiVariable ("WWW_REQUEST_METHOD", CgiRequestMethod,
                   sizeof(CgiRequestMethod));

   /* numeric equivalent of "GET\0" (just a bit more efficient, that's all!) */
   if (*(unsigned long*)CgiRequestMethod != 0x00544547) exit (SS$_NORMAL);

   GetCgiVariable ("WWW_FORM_HELP", FormHelp, sizeof(FormHelp));
   if (FormHelp[0])
   {
      GetCgiVariable ("WWW_SERVER_NAME", CgiServerName, sizeof(CgiServerName));
      fprintf (HttpOut, Http302Header, CgiServerName, HelpUrl);
      exit (SS$_NORMAL);
   }

   GetCgiVariable ("WWW_PATH_INFO", CgiPathInfo, sizeof(CgiPathInfo));
   GetCgiVariable ("WWW_PATH_TRANSLATED", CgiPathTranslated,
                   sizeof(CgiPathTranslated));
   GetCgiVariable ("WWW_SCRIPT_NAME", CgiScriptName, sizeof(CgiScriptName));
   GetCgiVariable ("WWW_FORM_SHELF", VmsShelfSpec, sizeof(VmsShelfSpec));
   GetCgiVariable ("WWW_FORM_TITLE", FormTitle, sizeof(FormTitle));

   GetCgiVariable ("WWW_HTTP_IF_MODIFIED_SINCE", CgiHttpIfModifiedSince,
                   sizeof(CgiHttpIfModifiedSince));

   if (VMSnok (status = TimeSetGmt ()))
   {
      if (status != SS$_NOLOGNAM)
      {
         ErrorVmsStatus (status, CgiPathInfo, CgiPathTranslated,
                         __FILE__, __LINE__);
         exit (SS$_NORMAL);
      }
   }
   if (CgiHttpIfModifiedSince[0])
   {
      if (VMSnok (HttpGmTime (CgiHttpIfModifiedSince, 
                              &IfModifiedSinceBinaryTime)))
      {
         if (Debug) fprintf (stdout, "If-Modified-Since: NBG!\n");
         IfModifiedSinceBinaryTime[0] = IfModifiedSinceBinaryTime[0] = 0;
         CgiHttpIfModifiedSince[0] = '\0';
      }
   }

   if (!VmsShelfSpec[0] && CgiPathInfo[0] && CgiPathInfo[1])
      strcpy (VmsShelfSpec, CgiPathTranslated);
   if (!VmsShelfSpec[0]) strcpy (VmsShelfSpec, DefaultShelf);
   if (!FormTitle[0]) strcpy (FormTitle, "Library");

   GetCgiVariable ("WWW_FORM_REFERER", OriginalReferer, sizeof(OriginalReferer));
   if (!OriginalReferer[0])
      GetCgiVariable ("WWW_HTTP_REFERER", OriginalReferer,
                      sizeof(OriginalReferer));
   if (OriginalReferer[0])
   {
      /* re-escape the URL-escaped percentages */
      CopyIntoUri (UriReferer, OriginalReferer, -1);
   }

   HttpGmTimeString (GmDateTime, 0);

   ProcessShelf ();

   exit (SS$_NORMAL);
}

/*****************************************************************************/
/*
Open the supplied shelf file name (in 'VmsShelfSpec').  Read each line in the 
file parsing the entry type (shelf or book), the file specification, and entry 
description (title of shelf or book).  Close the file.  If a file 
specification (shelf or book) does not explictly provide a directory 
component then use the directory of the currently open shelf file.
*/ 
 
ProcessShelf ()

{
   register char  *cptr, *sptr;

   boolean  LibraryTitleOutput;
   int  status,
        Length,
        LineCount,
        LineOutputCount;
   char  FileName [256],
         HtmlFileName [256],
         HtmlTitle [256],
         Line [256],
         LinkFileName [256],
         ParseDefaults [256],
         PostScriptFileName [256],
         Scratch [512],
         ShelfDirectory [256],
         Title [256],
         Type [256],
         UriTitle [256],
         UriFormTitle [256];
   struct FAB  ShelfFab;
   struct RAB  ShelfRab;
   struct XABDAT  ShelfXabDat;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ProcessShelf() |%s|\n", VmsShelfSpec);

   LibraryTitleOutput = false;
   HtmlTitle[0] = UriFormTitle[0] = '\0';

   sptr = HyperReaderReferer;
   sptr += CopyIntoUri (sptr, CgiScriptName, -1);
   sptr += CopyIntoUri (sptr, "?shelf=", -1);
   sptr += CopyIntoUri (sptr, VmsShelfSpec, -1);
   sptr += CopyIntoUri (sptr, "&title=", -1);
   /*
      This escapes the escape percentages leaving the URI still escaped
      when the script (HyperReader) receives it at the other end :^)
   */
   CopyIntoUri (Scratch, FormTitle, -1);
   sptr += CopyIntoUri (sptr, Scratch, -1);
   sptr += CopyIntoUri (sptr, "&referer=", -1);
   sptr += CopyIntoUri (sptr, UriReferer, -1);
   *sptr = '\0';
   if (Debug) fprintf (stdout, "HyperReaderReferer |%s|\n", HyperReaderReferer);

   /* attempt to locate a directory component in the shelf specification */
   sptr = ShelfDirectory;
   for (cptr = VmsShelfSpec; *cptr && *cptr != ':'; *sptr++ = *cptr++);
   if (cptr[0] == ':')
   {
      /* check if its a logical directory or a device with directory */
      while (*cptr && *cptr != ']') *sptr++ = *cptr++;
      /* if no directory was found then roll-back to where we were */
      if (!*cptr) while (*cptr != ':') { cptr--; sptr--; }
   }
   if (*cptr == ':' || *cptr == ']')
      *sptr++ = *cptr;
   else
      sptr = ShelfDirectory;
   *sptr = '\0';
   if (ShelfDirectory[0])
      if (strsame (ShelfDirectory, DecwBook, -1))
         UsingDecwBook = true;
      else;
   else
   {
      strcpy (ShelfDirectory, DecwBook);
      UsingDecwBook = true;
   }
   if (Debug) fprintf (stdout, "ShelfDirectory |%s|\n", ShelfDirectory);

   Length = sprintf (ParseDefaults, "%s.DECW$BOOKSHELF", ShelfDirectory);

   /**************/
   /* open shelf */
   /**************/

   ShelfFab = cc$rms_fab;
   ShelfFab.fab$l_dna = ParseDefaults;  
   ShelfFab.fab$b_dns = Length;
   ShelfFab.fab$b_fac = FAB$M_GET;
   ShelfFab.fab$l_fna = VmsShelfSpec;  
   ShelfFab.fab$b_fns = strlen(VmsShelfSpec);
   ShelfFab.fab$b_shr = FAB$M_SHRGET;
   ShelfFab.fab$l_xab = &ShelfXabDat;

   ShelfXabDat = cc$rms_xabdat;

   if (VMSnok (status = sys$open (&ShelfFab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$open() %%X%08.08X\n", status);
      if (HtmlTitle[0])
         cptr = HtmlTitle;
      else
         cptr = CgiPathInfo;
      ErrorVmsStatus (status, cptr, VmsShelfSpec, __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }

   if (CgiHttpIfModifiedSince[0])
   {
      if (VMSnok (ModifiedSince (&IfModifiedSinceBinaryTime,
                                 &ShelfXabDat.xab$q_rdt)))
      {
         /* book has not been modified since the date/time, don't send */
         sys$close (&ShelfFab, 0, 0);
         exit (SS$_NORMAL);
      }
   }

   HttpGmTimeString (LastModifiedGmDateTime, &ShelfXabDat.xab$q_rdt);

   ShelfRab = cc$rms_rab;
   ShelfRab.rab$l_fab = &ShelfFab;
   /* 2 buffers and read ahead performance option */
   ShelfRab.rab$b_mbf = 2;
   ShelfRab.rab$l_rop = RAB$M_RAH;
   ShelfRab.rab$l_ubf = Line;
   ShelfRab.rab$w_usz = sizeof(Line)-1;

   if (VMSnok (status = sys$connect (&ShelfRab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$connect() %%X%08.08X\n", status);
      sys$close (&ShelfFab, 0, 0);
      if (HtmlTitle[0])
         cptr = HtmlTitle;
      else
         cptr = CgiPathInfo;
      ErrorVmsStatus (status, cptr, VmsShelfSpec, __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }

   /***************/
   /* HTTP header */
   /***************/

   if (!HyperShelfIcon[0]) strcpy (HyperShelfIcon, "<I>HyperShelf</I> ");

   fprintf (HttpOut,
"HTTP/1.0 200 Document follows.\r\n\
Date: %s\r\n\
Last-Modified: %s\r\n\
Content-Type: text/html\r\n\
\r\n\
<!-- SoftwareID: %s -->\n",
   GmDateTime, LastModifiedGmDateTime, SoftwareID);
   fflush (HttpOut);
   HttpHasBeenOutput = true;

   /****************/
   /* file records */
   /****************/

   LineCount = LineOutputCount = 0;

   while (VMSok (status = sys$get (&ShelfRab, 0, 0)))
   {
      LineCount++;
      Line[ShelfRab.rab$w_rsz] = '\0';
      if (Debug) fprintf (stdout, "Line |%s|\n", Line);
      if (!Line[0]) continue;

      /****************************/
      /* process the entry fields */
      /****************************/

      Type[0] = FileName[0] = Title[0] = '\0';
      cptr = Line;
      sptr = Type;
      while (*cptr && isspace(*cptr)) cptr++;
      while (*cptr && *cptr != '\\') *sptr++ = toupper(*cptr++);
      *sptr = '\0';
      if (*cptr == '\\') cptr++;
      sptr = FileName;
      while (*cptr && isspace(*cptr)) cptr++;
      while (*cptr && *cptr != '\\') *sptr++ = *cptr++;
      *sptr = '\0';
      if (*cptr == '\\') cptr++;
      sptr = Title;
      while (*cptr && isspace(*cptr)) cptr++;
      while (*cptr && *cptr != '\\') *sptr++ = *cptr++;
      *sptr = '\0';

      if ((Type[0] != 'T' && Type[0] != 'S' && Type[0] != 'B') ||
          (Type[0] == 'T' && !Title[0]) ||
          ((Type[0] == 'S' || Type[0] == 'B') && (!FileName[0] || !Title[0])))
         sprintf (Title, "*** Unable to parse shelf entry! (line %d) ***",
                  LineCount);

      if (Type[0] == 'T')
      {
         /* title of this shelf */
         if (LibraryTitleOutput) continue;
         if (!HtmlTitle[0]) CopyIntoHtml (HtmlTitle, Title, -1);
         if (!UriFormTitle[0]) CopyIntoUri (UriFormTitle, Title, -1);
      }
      if (!LibraryTitleOutput)
      {
         if (!HtmlTitle[0]) CopyIntoHtml (HtmlTitle, FormTitle, -1);
         if (!UriFormTitle[0]) CopyIntoUri (UriFormTitle, FormTitle, -1);

         fprintf (HttpOut,
"<TITLE>HyperShelf - %s</TITLE>\n\
<H1>%s%s</H1>\n\
<HR><P>\n",
         HtmlTitle, HyperShelfIcon, HtmlTitle);
         LibraryTitleOutput = true;
      }
      if (Type[0] == 'T') continue;

      if (!(Type[0] == 'S' || Type[0] == 'B')) strcpy (Type, "SHELF");
      if (!FileName[0]) strcpy (FileName, "ERROR-IN-SHELF-ENTRY");

      CopyIntoHtml (HtmlTitle, Title, -1);
      CopyIntoUri (UriTitle, Title, -1);

      for (cptr = FileName; *cptr && *cptr != ':'; cptr++);
      if (*cptr || UsingDecwBook)
      {
         /* device/directory was explicitly specified or default directory */
         sptr = LinkFileName;
         for (cptr = FileName; *cptr; cptr++) *sptr++ = tolower(*cptr);
         *sptr = '\0';
      }
      else
      {
         /* device/directory was not explicitly specified */
         sptr = LinkFileName;
         for (cptr = ShelfDirectory; *cptr; cptr++) *sptr++ = tolower(*cptr);
         for (cptr = FileName; *cptr; cptr++) *sptr++ = tolower(*cptr);
         *sptr = '\0';
      }
      if (Debug) fprintf (stdout, "LinkFileName |%s|\n", LinkFileName);

      /*******************/
      /* output the HTML */
      /*******************/

      if (LineOutputCount++) fputs ("<BR>", HttpOut);

      if (Type[0] == 'S')
      {
         /*********/
         /* shelf */
         /*********/

         fprintf (HttpOut,
"<A HREF=\"%s?shelf=%s&title=%s&referer=%s\">\
<IMG SRC=\"%s/shelf.xbm\" ALIGN=top ALT=\"_|\\_\"></A> %s\n",
         CgiScriptName, LinkFileName, UriTitle, UriReferer,
         IconLocation, HtmlTitle);
      }
      else
      {
         /********/
         /* book */
         /********/

         /* this is for version generated by SDM2HTM utility! */
         VariantVersion (LinkFileName, "_0000.HTML", HtmlFileName);
         VariantVersion (LinkFileName, ".PS", PostScriptFileName);

         if (HyperReaderScriptName[0])
         {
            fprintf (HttpOut,
"<A HREF=\"/%s?book=%s&title=%s&referer=%s\">\
<IMG SRC=\"%s/book.xbm\" ALIGN=top ALT=\"[}{]\"></A>\n",
            HyperReaderScriptName, LinkFileName,
            UriFormTitle, HyperReaderReferer, IconLocation);
         }
         else
         {
            fprintf (HttpOut,
"<IMG SRC=\"%s/book.xbm\" ALIGN=top ALT=\"[}{]\">\n",
            IconLocation);
         }

         if (HtmlFileName[0])
         {
            /**************************/
            /* HTML version available */
            /**************************/

            fprintf (HttpOut,
"<A HREF=\"%s\">\
<IMG SRC=\"%s/html.xbm\" ALIGN=top ALT=\"[HT]\"></A>\n",
            MapUrl(NULL,HtmlFileName,NULL,NULL), IconLocation);
         }
         else
         {
            /**********************/
            /* HTML not available */
            /**********************/

            fprintf (HttpOut,
            "<IMG SRC=\"%s/html.xbm\" ALIGN=top ALT=\"[HT]\">\n",
            IconLocation);
         }

         if (PostScriptFileName[0])
         {
            /********************************/
            /* PostScript version available */
            /********************************/

            fprintf (HttpOut,
"<A HREF=\"%s\">\
<IMG SRC=\"%s/ps.xbm\" ALIGN=top ALT=\"[PS]\"></A>\n",
            MapUrl(NULL,PostScriptFileName,NULL,NULL), IconLocation);

            if (PrintScriptName[0])
            {
               fprintf (HttpOut,
"<A HREF=\"/%s%s\">\
<IMG SRC=\"%s/print.xbm\" ALIGN=top ALT=\"[Prn]\"></A>\n",
               PrintScriptName, MapUrl(NULL,PostScriptFileName,NULL,NULL),
               IconLocation);
            }
            else
            {
               fprintf (HttpOut,
"<IMG SRC=\"%s/print.xbm\" ALIGN=top ALT=\"[Prn]\">\n",
               IconLocation);
            }
         }
         else
         {
            /****************************/
            /* PostScript not available */
            /****************************/

            fprintf (HttpOut,
"<IMG SRC=\"%s/ps.xbm\" ALIGN=top ALT=\"[PS]\"></A>\n\
<IMG SRC=\"%s/print.xbm\" ALIGN=top ALT=\"[Prn]\"></A>\n",
            IconLocation,
            IconLocation);
         }

         /**************/
         /* book title */
         /**************/

         fprintf (HttpOut, "%s\n", HtmlTitle);
      }
   }

   /****************/
   /* end of shelf */
   /****************/

   sys$close (&ShelfFab, 0, 0);

   if (status == RMS$_EOF) status = SS$_NORMAL;
   if (VMSnok (status))
   {
      CopyIntoHtml (HtmlTitle, FormTitle, -1);
      if (HtmlTitle[0])
         cptr = HtmlTitle;
      else
         cptr = CgiPathInfo;
      ErrorVmsStatus (status, cptr, VmsShelfSpec, __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }

   if (!LineCount) fputs ("<I>Empty bookshelf!</I>\n", HttpOut);

   if (HelpUrl[0] || OriginalReferer[0])
      fputs ("<P><HR>\n<I>", HttpOut);
   else
      fputs ("<P><HR>\n", HttpOut);
   if (HelpUrl[0])
      fprintf (HttpOut, "<A HREF=\"%s\">Help</A>", HelpUrl);
   if (HelpUrl[0] && OriginalReferer[0])
      fputs ("; ", HttpOut);
   if (OriginalReferer[0])
      fprintf (HttpOut, "<A HREF=\"%s\">Close</A> Library", OriginalReferer);
   if (HelpUrl[0] || OriginalReferer[0])
      fputs ("</I>.\n", HttpOut);
   else
      fputs (".\n", HttpOut);
}

/*****************************************************************************/
/*
Check if the supplied file specification also has a version ending in the 
'VariantType' supplied (e.g. ".PS", ".HTML", etc.) by searching for it.  
Return the variant version's file name in 'FileNameFound'.
*/ 

VariantVersion
(
char *FileSpec,
char *VariantType,
char *FileNameFound
)
{
   register char  *cptr, *sptr;

   int  status;
   char  ExpandedFileName [256],
         FileName [256];
   struct FAB  SearchFab;
   struct NAM  SearchNam;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "VariantVersion() |%s|%s|\n", FileSpec, VariantType);

   sptr = FileName;
   for (cptr = FileSpec; *cptr && *cptr != ';'; *sptr++ = *cptr++);
   *sptr = '\0';
   while (sptr > FileName && *sptr != '.' && *sptr != ']') sptr--;
   if (*sptr == '.')
      strcpy (sptr, VariantType);
   else
      strcat (FileName, VariantType);
   while (*sptr) sptr++;

   SearchFab = cc$rms_fab;
   SearchFab.fab$l_dna = "DECW$BOOK:";
   SearchFab.fab$b_dns = 10;
   SearchFab.fab$l_fna = FileName;
   SearchFab.fab$b_fns = sptr - FileName;
   SearchFab.fab$l_fop = FAB$M_NAM;
   SearchFab.fab$l_nam = &SearchNam;

   SearchNam = cc$rms_nam;
   SearchNam.nam$l_esa = ExpandedFileName;
   SearchNam.nam$b_ess = sizeof(ExpandedFileName)-1;

   if (VMSnok (status = sys$parse (&SearchFab, 0, 0)))
   {
      ErrorVmsStatus (status, FileName, FileName, __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }

   if (Debug)
   {
      *SearchNam.nam$l_ver = '\0';
      fprintf (stdout, "SearchNam.nam$l_dev |%s|\n", SearchNam.nam$l_dev);
      *SearchNam.nam$l_ver = ';';
   }

   SearchNam.nam$l_rsa = FileNameFound;
   SearchNam.nam$b_rss = 255;

   if (VMSok (status = sys$search (&SearchFab, 0, 0)))
   {
      *SearchNam.nam$l_ver = '\0';
      if (Debug) fprintf (stdout, "FileNameFound |%s|\n", FileNameFound);
      for (sptr = FileNameFound; *sptr; sptr++) *sptr = tolower(*sptr);
   }
   else
   {
      if (Debug) fprintf (stdout, "sys$search() %%X%08.08X\n", status);
      FileNameFound[0] = '\0';
      if (status != RMS$_FNF && status != RMS$_NMF)
      {
         ErrorVmsStatus (status, FileName, FileName, __FILE__, __LINE__);
         exit (SS$_NORMAL);
      }
   }

   /* parse without disk I/O to release dynamic structures allocated */
   SearchNam.nam$b_nop = NAM$M_SYNCHK;
   sys$parse (&SearchFab, 0, 0);
}

/*****************************************************************************/
/*
*/

boolean GetCgiVariable
(
char *VariableName,
char *VariableValue,
int VariableValueSize
)
{
   static $DESCRIPTOR (VariableNameDsc, "");
   static $DESCRIPTOR (VariableValueDsc, "");

   register int  status;

   unsigned short  Length;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "GetCgiVariable() |%s|\n", VariableName);

   VariableNameDsc.dsc$w_length = strlen(VariableName);
   VariableNameDsc.dsc$a_pointer = VariableName;
   VariableValueDsc.dsc$w_length = VariableValueSize-1;
   VariableValueDsc.dsc$a_pointer = VariableValue;

   if (VMSok (status =
       lib$get_symbol (&VariableNameDsc, &VariableValueDsc, &Length, 0)))
      VariableValue[Length] = '\0';
   else
      VariableValue[0] = '\0';

   if (status == LIB$_NOSUCHSYM) return (false);
   if (VMSnok (status))
   {
      if (Debug) fprintf (stdout, "lib$get_symbol() %%X%08.08X\n", status);
      exit (status);
   }
   if (Debug) fprintf (stdout, "|%s|\n", VariableValue);
   return (true);
}

/*****************************************************************************/
/*
Copy text from one string to another, converting characters forbidden to 
appear as plain-text in HTML.  For example the '<', '&', etc.  Convert these 
to the corresponding HTML character entities.  Returns number of characters 
copied into HTML string.
*/ 

int CopyIntoHtml
( 
register char *hptr,
register char *sptr,
register int ccnt
)
{
   char  *FirstPtr;

   /*********/
   /* begin */
   /*********/

   FirstPtr = hptr;
   while (*sptr && ccnt--)
   {
      switch (*sptr)
      {
         case '<' : strcpy (hptr, "&lt;"); hptr += 4; sptr++; break;
         case '>' : strcpy (hptr, "&gt;"); hptr += 4; sptr++; break;
         case '&' : strcpy (hptr, "&amp;"); hptr += 5; sptr++; break;
         case '\"' : strcpy (hptr, "&quot;"); hptr += 6; sptr++; break;
         default : if (isprint(*sptr)) *hptr++ = *sptr++; else sptr++;
      }
   }
   *hptr = '\0';
   return (hptr-FirstPtr);
}

/*****************************************************************************/
/*
*/

ErrorGeneral
(
char *Text,
char *SourceFileName,
int SourceLineNumber
)
{
   register char  *cptr;

   /*********/
   /* begin */
   /*********/

   /* 
      The source file format provided by the "__FILE__" macro will be
      "device:[directory]name.type;ver".  Reduce that to "name".
   */
   for (cptr = SourceFileName; *cptr && *cptr != ';'; cptr++);
   if (*cptr)
   {
      while (*cptr != '.') cptr--;
      *cptr-- = '\0';
   }
   while (*cptr != ']') cptr--;
   cptr++;

   if (!HttpHasBeenOutput) fputs (Http404Header, HttpOut);
   fprintf (HttpOut,
"<!-- SoftwareID: %s Module: %s Line: %d -->\n\
<H1>ERROR!</H1>\n\
<P>Reported by server.\n\
<P>%s\n",
   SoftwareID, cptr, SourceLineNumber, Text);
}

/*****************************************************************************/
/*
*/

ErrorVmsStatus
(
int StatusValue,
char *Text,
char *HiddenText,
char *SourceFileName,
int SourceLineNumber
)
{
   static char  Message [256];
   static $DESCRIPTOR (MessageDsc, Message);

   register char  *cptr;
   int  status;
   short int  Length;

   /*********/
   /* begin */
   /*********/

   if (VMSok (status = sys$getmsg (StatusValue, &Length, &MessageDsc, 1, 0))) 
   {
      Message[Length] = '\0';
      Message[0] = toupper(Message[0]);
   }
   else
      exit (status);

   /* 
      The source file format provided by the "__FILE__" macro will be
      "device:[directory]name.type;ver".  Reduce that to "name".
   */
   for (cptr = SourceFileName; *cptr && *cptr != ';'; cptr++);
   if (*cptr)
   {
      while (*cptr != '.') cptr--;
      *cptr-- = '\0';
   }
   while (*cptr != ']') cptr--;
   cptr++;

   if (!HttpHasBeenOutput) fputs (Http404Header, HttpOut);
   fprintf (HttpOut,
"<!-- SoftwareID: %s Module: %s Line: %d -->\n\
<H1>ERROR!</H1>\n\
<P>Reported by server.\n\
<P>%s ... <TT>%s</TT>\n\
<!-- %%X%08.08X \"%s\" -->\n",
   SoftwareID, cptr, SourceLineNumber, Message, Text, StatusValue, HiddenText);
}

/*****************************************************************************/
/*
If the object has been modified since the specified date and time then return 
a normal status indicating that the data transfer is to continue.  If not 
modified then send a "not modified" HTTP header and return an error status to 
indicate the object should not be sent.
*/ 
 
int ModifiedSince
(
unsigned long *SinceBinaryTimePtr,
unsigned long *BinaryTimePtr
)
{
   static unsigned long  OneSecondDelta [2] = { -10000000, -1 };
   static char  Http304Header [] = "HTTP/1.0 304 Not modified.\r\n\r\n";

   int  status;
   unsigned long  AdjustedBinTime [2],
                  ScratchBinTime [2];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ModifiedSince()\n");

   /*
      Add one second to the modified time.  Ensures a negative time
      for VMS where fractional seconds may result in inconclusive
      results when the target time is being specified in whole seconds.
   */ 
   if (VMSnok (status =
       lib$add_times (SinceBinaryTimePtr, &OneSecondDelta, &AdjustedBinTime)))
   {
      ErrorVmsStatus (status, "If-Modified-Since:", "", __FILE__, __LINE__);
      return (status);
   }
   if (Debug) fprintf (stdout, "sys$add_times() %%X%08.08X\n", status);

   /* if a positive time results the file has been modified */
   if (VMSok (status =
       lib$sub_times (BinaryTimePtr, &AdjustedBinTime, &ScratchBinTime)))
      return (status);

   if (Debug) fprintf (stdout, "sys$sub_times() %%X%08.08X\n", status);
   if (status != LIB$_NEGTIM)
   {
      ErrorVmsStatus (status, "If-Modified-Since:", "", __FILE__, __LINE__);
      return (status);
   }

   fputs (Http304Header, HttpOut);

   return (LIB$_NEGTIM);
}

/*****************************************************************************/
/*
Create an HTTP format Greenwich Mean Time (UTC) time string in the storage 
pointed at by 'TimeString', e.g. "Fri, 25 Aug 1995 17:32:40 GMT" (RFC 1123). 
This must be at least 30 characters capacity.  If 'BinTimePtr' is null the 
time string represents the current time.  If it points to a quadword, VMS time 
value the string represents that time.  'TimeString' must point to storage 
large enough for 31 characters.
*/

int HttpGmTimeString
(
char *TimeString,
unsigned long *BinTimePtr
)
{
   static char  *DayNames [] =
      { "", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat", "Sun" };
   static char  *MonthName [] =
      { "", "Jan", "Feb", "Mar", "Apr", "May", "Jun",
            "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };

   static $DESCRIPTOR (HttpTimeFaoDsc, "!AZ, !2ZW !AZ !4ZW !2ZW:!2ZW:!2ZW GMT");
   static $DESCRIPTOR (TimeStringDsc, "");

   int  status;
   unsigned long  BinTime [2],
                  GmTime [2];
   unsigned short  Length;
   unsigned short  NumTime [7];
   unsigned long  DayOfWeek;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HttpGmTimeString()\n");

   if (BinTimePtr == NULL)
      sys$gettim (&GmTime);
   else
   {
      GmTime[0] = BinTimePtr[0];
      GmTime[1] = BinTimePtr[1];
   }
   if (VMSnok (status = TimeAdjustGMT (true, &GmTime)))
      return (status);

   status = sys$numtim (&NumTime, &GmTime);
   if (Debug)
      fprintf (stdout, "sys$numtim() %%X%08.08X %d %d %d %d %d %d %d\n",
               status, NumTime[0], NumTime[1], NumTime[2],
               NumTime[3], NumTime[4], NumTime[5], NumTime[6]);

   if (VMSnok (status = lib$day_of_week (&GmTime, &DayOfWeek)))
      return (status);
   if (Debug)
      fprintf (stdout, "lib$day_of_week() %%X%08.08X is %d\n",
               status, DayOfWeek);

   /* set the descriptor address and size of the resultant time string */
   TimeStringDsc.dsc$w_length = 30;
   TimeStringDsc.dsc$a_pointer = TimeString;

   if (VMSnok (status =
       sys$fao (&HttpTimeFaoDsc, &Length, &TimeStringDsc,
                DayNames[DayOfWeek], NumTime[2], MonthName[NumTime[1]],
                NumTime[0], NumTime[3], NumTime[4], NumTime[5])))
   {
      if (Debug) fprintf (stdout, "sys$fao() %%X%08.08X\n", status);
      TimeString[0] = '\0';
   }
   else
      TimeString[Length] = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", TimeString);

   return (status);
}

/*****************************************************************************/
/*
Given a string such as "Fri, 25 Aug 1995 17:32:40 GMT" (RFC 1123), or "Friday,
25-Aug-1995 17:32:40 GMT" (RFC 1036), create an internal, local, binary time
with the current GMT offset.  See complementary function HttpGmTimeString().
*/ 

int HttpGmTime
(
char *TimeString,
unsigned long *BinTimePtr
)
{
   static char  *MonthName [] =
      { "", "Jan", "Feb", "Mar", "Apr", "May", "Jun",
            "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };

   register char  *tptr;

   int  status;
   unsigned short  Length;
   unsigned short  NumTime [7] = { 0,0,0,0,0,0,0 };
   unsigned long  DayOfWeek;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HttpGmTime() |%s|\n", TimeString);

   tptr = TimeString;
   /* hunt straight for the comma after the weekday name! */
   while (*tptr && *tptr != ',') tptr++;
   if (*tptr) tptr++;
   /* span white space between weekday name and date */
   while (*tptr && isspace(*tptr)) tptr++;
   if (Debug) fprintf (stdout, "tptr |%s|\n", tptr);
   if (!*tptr) return (STS$K_ERROR);

   /* get the date and then skip to month name */
   if (isdigit(*tptr)) NumTime[2] = atoi (tptr);
   while (*tptr && isdigit(*tptr)) tptr++;
   while (*tptr && (*tptr == '-' || isspace(*tptr))) tptr++;
   if (Debug) fprintf (stdout, "tptr |%s|\n", tptr);
   if (!*tptr) return (STS$K_ERROR);

   /* get the month number from the name and skip to the year */
   for (NumTime[1] = 1; NumTime[1] <= 12; NumTime[1]++)
      if (strsame (tptr, MonthName[NumTime[1]], 3)) break;
   if (NumTime[1] > 12) return (STS$K_ERROR);
   while (*tptr && isalpha(*tptr)) tptr++;
   while (*tptr && (*tptr == '-' || isspace(*tptr))) tptr++;
   if (Debug) fprintf (stdout, "tptr |%s|\n", tptr);
   if (!*tptr) return (STS$K_ERROR);

   /* get the year and then skip to the hour */
   if (isdigit(*tptr))
   {
      NumTime[0] = atoi (tptr);
      if (NumTime[0] < 100) NumTime[0] += 1900;
   }
   while (*tptr && isdigit(*tptr)) tptr++;
   while (*tptr && isspace(*tptr)) tptr++;
   if (Debug) fprintf (stdout, "tptr |%s|\n", tptr);
   if (!*tptr) return (STS$K_ERROR);

   /* get the hour, minute and second */
   if (isdigit(*tptr)) NumTime[3] = atoi (tptr);
   while (*tptr && isdigit(*tptr)) tptr++;
   if (*tptr == ':') tptr++;
   if (isdigit(*tptr)) NumTime[4] = atoi (tptr);
   while (*tptr && isdigit(*tptr)) tptr++;
   if (*tptr == ':') tptr++;
   if (isdigit(*tptr)) NumTime[5] = atoi (tptr);
   while (*tptr && isdigit(*tptr)) tptr++;
   if (*tptr == ':') tptr++;
   if (!*tptr) return (STS$K_ERROR);

   /* the only thing remaining should be the "GMT" */
   while (*tptr && isspace(*tptr)) tptr++;
   if (Debug) fprintf (stdout, "tptr |%s|\n", tptr);
   if (!strsame (tptr, "GMT", 3)) return (STS$K_ERROR);

   /*******************************************/
   /* convert what looks like legitimate GMT! */
   /*******************************************/

   if (Debug)
      fprintf (stdout, "NumTime[] %d %d %d %d %d %d %d\n",
               NumTime[0], NumTime[1], NumTime[2], NumTime[3], 
               NumTime[4], NumTime[5], NumTime[6]); 
   status = lib$cvt_vectim (&NumTime, BinTimePtr);
   if (VMSnok (status)) return (status);
   if (Debug) fprintf (stdout, "lib$cvt_vectim() %%X%08.08X\n", status);

   return (TimeAdjustGMT (false, BinTimePtr));
}

/*****************************************************************************/
/*
Translate the logical HTTPD$GMT (defined to be something like "+10:30" or "-
01:15") and convert it into a delta time structure and store in 
'TimeGmtDeltaBinary'.  Store whether it is in advance or behind GMT in boolean 
'TimeAheadOfGmt'.  Store the logical string in 'TimeGmtString'.
*/

int TimeSetGMT ()

{
   register char  *cptr, *sptr;

   int  status;
   unsigned short  Length;
   $DESCRIPTOR (GmtLogicalNameDsc, "HTTPD$GMT");
   $DESCRIPTOR (LnmFileDevDsc, "LNM$FILE_DEV");
   $DESCRIPTOR (TimeGmtVmsStringDsc, "");
   struct {
      short int  buf_len;
      short int  item;
      unsigned char   *buf_addr;
      unsigned short  *ret_len;
   } LnmItems [] =
   {
      { sizeof(TimeGmtString)-1, LNM$_STRING, TimeGmtString, &Length },
      { 0,0,0,0 }
   };

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "TimeSetGMT()\n");

   status = sys$trnlnm (0, &LnmFileDevDsc, &GmtLogicalNameDsc, 0, &LnmItems);
   if (Debug) fprintf (stdout, "sys$trnlnm() %%X%08.08X\n", status);
   if (VMSnok (status)) return (status);

   TimeGmtString[Length] = '\0';
   if (Debug) fprintf (stdout, "TimeGmtString |%s|\n", TimeGmtString);
   sptr = TimeGmtVmsString;
   if (*(cptr = TimeGmtString) == '-')
      TimeAheadOfGmt = false;
   else
      TimeAheadOfGmt = true;
   *sptr++ = *cptr++;
   *sptr++ = '0';
   *sptr++ = ' ';
   while (*cptr && *cptr != ' ') *sptr++ = *cptr++;
   *sptr++ = '\0';
   if (Debug) fprintf (stdout, "TimeGmtVmsString |%s|\n", TimeGmtVmsString);

   /* set the descriptor address and size of the offset time string */
   if (TimeGmtVmsString[0] == '+' || TimeGmtVmsString[0] == '-')
   {
      TimeGmtVmsStringDsc.dsc$w_length = Length-1;
      TimeGmtVmsStringDsc.dsc$a_pointer = TimeGmtVmsString+1;
   }
   else
   {
      TimeGmtVmsStringDsc.dsc$w_length = Length;
      TimeGmtVmsStringDsc.dsc$a_pointer = TimeGmtVmsString;
   }

   return (sys$bintim (&TimeGmtVmsStringDsc, &TimeGmtDeltaBinary));
}

/*****************************************************************************/
/*
The GMT is generated by calculating using an offset using 
'TimeGmtDeltaOffset' and boolean 'TimeAheadOfGmt'.  Adjust either to or from 
GMT.
*/ 

int TimeAdjustGMT
(
boolean ToGmTime,
unsigned long *BinTimePtr
)
{
   int  status;
   unsigned long  AdjustedTime [2];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "TimeAdjustGMT() ToGmTime: %d\n", ToGmTime);

   if ((ToGmTime && TimeAheadOfGmt) || (!ToGmTime && !TimeAheadOfGmt))
   {
      /* to GMT from local and ahead of GMT, or to local from GMT and behind */
      status = lib$sub_times (BinTimePtr, &TimeGmtDeltaBinary, &AdjustedTime);
      if (Debug) fprintf (stdout, "lib$sub_times() %%X%08.08X\n", status);
   }
   else
   {
      /* to GMT from local and behind GMT, or to local from GMT and ahead */
      status = lib$add_times (BinTimePtr, &TimeGmtDeltaBinary, &AdjustedTime);
      if (Debug) fprintf (stdout, "lib$add_times() %%X%08.08X\n", status);
   }

   if (Debug)
   {
      unsigned short  Length;
      char  String [64];
      $DESCRIPTOR (AdjustedTimeFaoDsc, "AdjustedTime: |!%D|\n");
      $DESCRIPTOR (StringDsc, String);

      sys$fao (&AdjustedTimeFaoDsc, &Length, &StringDsc, &AdjustedTime);
      String[Length] = '\0';
      fputs (String, stdout);
   }

   BinTimePtr[0] = AdjustedTime[0];
   BinTimePtr[1] = AdjustedTime[1];

   return (status);
}

/*****************************************************************************/
/*
Copy text from one string to another, converting characters forbidden to 
appear as plain-text text in an HTTP URL.  For example the '?', '+', '&', etc.  
Convert these to "%nn" hexadecimal escaped characters.  Returns number of
characters copied into URI.
*/ 

int CopyIntoUri
( 
register char *uptr,
register char *sptr,
register int  ccnt
)
{
   char  *FirstPtr;

   /*********/
   /* begin */
   /*********/

   FirstPtr = uptr;
   while (*sptr && ccnt--)
   {  
      switch (*sptr)
      {
         case '\t' : strcpy (uptr, "%09"); uptr += 3; sptr++; break;
         case ' ' : strcpy (uptr, "%20"); uptr += 3; sptr++; break;
         case '!' : strcpy (uptr, "%21"); uptr += 3; sptr++; break;
         case '\"' : strcpy (uptr, "%22"); uptr += 3; sptr++; break;
         case '#' : strcpy (uptr, "%23"); uptr += 3; sptr++; break;
         case '$' : strcpy (uptr, "%24"); uptr += 3; sptr++; break;
         case '%' : strcpy (uptr, "%25"); uptr += 3; sptr++; break;
         case '&' : strcpy (uptr, "%26"); uptr += 3; sptr++; break;
         case '\'' : strcpy (uptr, "%27"); uptr += 3; sptr++; break;
         case '(' : strcpy (uptr, "%28"); uptr += 3; sptr++; break;
         case ')' : strcpy (uptr, "%29"); uptr += 3; sptr++; break;
         case '*' : strcpy (uptr, "%2a"); uptr += 3; sptr++; break;
         case '+' : strcpy (uptr, "%2b"); uptr += 3; sptr++; break;
         case ',' : strcpy (uptr, "%2c"); uptr += 3; sptr++; break;
         case '-' : strcpy (uptr, "%2d"); uptr += 3; sptr++; break;
         case '.' : strcpy (uptr, "%2e"); uptr += 3; sptr++; break;
         case '/' : strcpy (uptr, "%2f"); uptr += 3; sptr++; break;
         case ':' : strcpy (uptr, "%3a"); uptr += 3; sptr++; break;
         case ';' : strcpy (uptr, "%3b"); uptr += 3; sptr++; break;
         case '<' : strcpy (uptr, "%3c"); uptr += 3; sptr++; break;
         case '=' : strcpy (uptr, "%3d"); uptr += 3; sptr++; break;
         case '>' : strcpy (uptr, "%3e"); uptr += 3; sptr++; break;
         case '?' : strcpy (uptr, "%3f"); uptr += 3; sptr++; break;
         case '[' : strcpy (uptr, "%5b"); uptr += 3; sptr++; break;
         case '\\' : strcpy (uptr, "%5c"); uptr += 3; sptr++; break;
         case ']' : strcpy (uptr, "%5d"); uptr += 3; sptr++; break;
         case '^' : strcpy (uptr, "%5e"); uptr += 3; sptr++; break;
         case '_' : strcpy (uptr, "%5f"); uptr += 3; sptr++; break;
         case '{' : strcpy (uptr, "%7b"); uptr += 3; sptr++; break;
         case '|' : strcpy (uptr, "%7c"); uptr += 3; sptr++; break;
         case '}' : strcpy (uptr, "%7d"); uptr += 3; sptr++; break;
         case '~' : strcpy (uptr, "%7e"); uptr += 3; sptr++; break;
         default : if (isprint(*sptr)) *uptr++ = *sptr++; else sptr++;
      }
   }
   *uptr = '\0';
   return (uptr-FirstPtr);
}

/****************************************************************************/
/*
Does a case-insensitive, character-by-character string compare and returns 
true if two strings are the same, or false if not.  If a maximum number of 
characters are specified only those will be compared, if the entire strings 
should be compared then specify the number of characters as 0.
*/ 

boolean strsame
(
register char *sptr1,
register char *sptr2,
register int  count
)
{
   while (*sptr1 && *sptr2)
   {
      if (toupper (*sptr1++) != toupper (*sptr2++)) return (false);
      if (count)
         if (!--count) return (true);
   }
   if (*sptr1 || *sptr2)
      return (false);
   else
      return (true);
}

/*****************************************************************************/

