/*****************************************************************************/
/*
                                  IsMap.c


A CGI-compliant image mapping program.

All of the "smarts" for this application have been plagiarized from the NCSA 
imagemap.c application.  Due acknowlegement to Rob McCool, et.al.  For 
reference, the source for imagemap.c should be located in the same directory 
as this application.  The routines directly lifted from NCSA imagemap.c are 
grouped together at the end of this program. 

Up until this application was put together HFRD was reliant on a pre-compiled 
version of the CERN htimage.exe program.  Having IsMap.c gives a certain 
independence, documented code (have a look at imagemap.c!), and better error 
reporting. 

This application allows the image configuration file to be specified in either 
CERN and NCSA format (or a mix of the two for that matter).  It is slightly 
more efficient to use the NCSA format, and this is suggested anyway, but the 
CERN format is provided for backward compatibility.  The region type keywords 
must supply at least 4 characters (3 for "default").  Comments can be prefixed 
by '#' or '!' characters, and blank lines are ignored.

As an extension to both of these mapping applications, IsMap.c can accept
relative paths.   Hence it not only processes full URL's like
"http://host/area/document.html", local absolute URL's like
"/area/document.html", but also local relative URLs such as
"../area/document.html", etc.  This provides for greater ease of mobility for
mapped documents.


NCSA FORMAT
-----------
default url
circle url x,y x,y              # centre-point then edge-point
point url x,y
polygon url x,y x,y [x,y ...]
rectangle url x,y x,y


CERN FORMAT
-----------
default url
circle (x,y) r url              # centre-point then radius
point (x,y) url
polygon (x,y) (x,y) [(x,y) ...] url
rectangle (x,y) (x,y) url


CGI VARIABLE NAMES
------------------
WWW_PATH_INFO           the URL configuration path
WWW_PATH_TRANSLATED     the VMS configuration file specification
WWW_QUERY_STRING        the map coordinate


BUILD
-----
See BUILD_ISMAP.COM


VERSION HISTORY
---------------
16-NOV-95  MGD  v1.0.0, initial development
*/
/*****************************************************************************/

#ifdef __ALPHA
   char SoftwareID [] = "ISMAP AXP-1.0.0";
#else
   char SoftwareID [] = "ISMAP VAX-1.0.0";
#endif

#ifdef __ALPHA
#   pragma nomember_alignment
#endif

/* standard C header files */
#include <ctype.h>
#include <errno.h>
#include <math.h>
#include <stdio.h>
#include <string.h>

/* VMS related header files */
#include <descrip.h>
#include <rms.h>
#include <ssdef.h>
#include <stsdef.h>

#define VMSok(x) ((x) & STS$M_SUCCESS)
#define VMSnok(x) !(((x) & STS$M_SUCCESS))

#define boolean int
#define true 1
#define false 0

/* macro provides NULL pointer if CGI variable does not exist */
#define GetCgiVarIfExists(CharPointer,CgiVariableName) \
   CharPointer = getenv(CgiVariableName)

/* macro provides pointer to empty string even if CGI variable does not exist */
#define GetCgiVar(CharPointer,CgiVariableName) \
   if ((CharPointer = getenv(CgiVariableName)) == NULL) \
       CharPointer = ""; \
   if (Debug) fprintf (stdout, "%s |%s|\n", CgiVariableName, CharPointer);

#define MAXVERTS 100
#define X 0
#define Y 1

char  Utility [] = "ISMAP";

char  Http404Header [] =
"HTTP/1.0 404 Error report follows.\r\n\
Content-Type: text/html\r\n\
\r\n";

boolean  Debug,
         HttpHasBeenOutput;

double  ClickCoord [2];

char  *CgiPathInfoPtr,
      *CgiPathTranslatedPtr,
      *CgiQueryStringPtr;
      
FILE  *HttpOut;

/*****************************************************************************/
/*
*/

main
(
int argc,
char *argv[]
)
{
   static char UnacceptableCoordinate [] = 
               "Client (browser) has not supplied an acceptable coordinate.";

   register char  *cptr;

   int  acnt;

   /*********/
   /* begin */
   /*********/

   /* open another output stream so that the '\r' and '\n' are not filtered */
#ifdef __DECC
   if ((HttpOut = fopen ("SYS$OUTPUT", "w", "ctx=bin")) == NULL)
      exit (vaxc$errno);
#else
   if ((HttpOut = fopen ("SYS$OUTPUT", "w", "rfm=udf")) == NULL)
      exit (vaxc$errno);
#endif

   /***********************************/
   /* get the command line parameters */
   /***********************************/

   /* doing it this way, parameters must be space separated! */
   for (acnt = 1; acnt < argc; acnt++)
   {
      if (Debug) fprintf (stdout, "argv[%d] |%s|\n", acnt, argv[acnt]);
      if (strsame (argv[acnt], "/DBUG", -1))
      {
         Debug = true;
         continue;
      }
      fprintf (stdout, "%%%s-E-IVQUAL, unrecognized qualifier\n \\%s\\\n",
               Utility, argv[acnt]+1);
      exit (STS$K_ERROR | STS$M_INHIB_MSG);
   }

   /*********************/
   /* get CGI variables */
   /*********************/

   GetCgiVar (CgiPathInfoPtr, "WWW_PATH_INFO");
   GetCgiVar (CgiPathTranslatedPtr, "WWW_PATH_TRANSLATED");
   GetCgiVar (CgiQueryStringPtr, "WWW_QUERY_STRING");

   /***********************/
   /* execute the request */
   /***********************/

   if (!CgiPathInfoPtr[0])
      ErrorGeneral (
"Configuration path not specified.\n\
<P><I>(document needs revision)</I>",
         __FILE__, __LINE__);

   if (!CgiQueryStringPtr[0])
      ErrorGeneral (UnacceptableCoordinate, __FILE__, __LINE__);

   /* get the coordinate of the mouse-click */
   cptr = CgiQueryStringPtr;
   while (isspace(*cptr)) cptr++;
   if (Debug) fprintf (stdout, "cptr |%s|\n", cptr);
   if (isdigit(*cptr))
   {
      ClickCoord[X] = (double)atoi(cptr);
      while (*cptr && isdigit(*cptr)) cptr++;
      while (isspace(*cptr) || *cptr == ',') cptr++;
      if (isdigit(*cptr))
         ClickCoord[Y] = (double)atoi(cptr);
      else
         ErrorGeneral (UnacceptableCoordinate, __FILE__, __LINE__);
   }
   else
      ErrorGeneral (UnacceptableCoordinate, __FILE__, __LINE__);
   if (Debug)
      fprintf (stdout, "ClickCoord |%f|%f|\n", ClickCoord[X], ClickCoord[Y]);

   ProcessConfigFile ();

   exit (SS$_NORMAL);
}

/*****************************************************************************/
/*
*/

ProcessConfigFile ()

{
   register char  *cptr, *lptr, *sptr, *zptr;

   boolean  CernCompliant;
   int  status,
        CoordCount,
        LineCount;
   double  ClosestPoint = HUGE_VAL,
           Distance;
   double  CoordArray [MAXVERTS][2];
   char  DefaultPath [256],
         Line [1024],
         Path [256],
         RegionKeyword [256],
         Url [256];
   struct FAB  IsMapFileFab;
   struct RAB  IsMapFileRab;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "ProcessConfigFile() |%s|\n", CgiPathTranslatedPtr);

   /* file access block */
   IsMapFileFab = cc$rms_fab;
   IsMapFileFab.fab$b_fac = FAB$M_GET;
   IsMapFileFab.fab$l_fna = CgiPathTranslatedPtr;
   IsMapFileFab.fab$b_fns = strlen(CgiPathTranslatedPtr);
   IsMapFileFab.fab$b_shr = FAB$M_SHRGET;

   if (VMSnok (status = sys$open (&IsMapFileFab, 0, 0)))
      ErrorVmsStatus (status, CgiPathInfoPtr, CgiPathTranslatedPtr,
                      __FILE__, __LINE__);

   /* record access block */
   IsMapFileRab = cc$rms_rab;
   IsMapFileRab.rab$l_fab = &IsMapFileFab;
   /* 2 buffers, read ahead performance option */
   IsMapFileRab.rab$b_mbf = 2;
   IsMapFileRab.rab$l_rop = RAB$M_RAH;
   IsMapFileRab.rab$l_ubf = Line;
   IsMapFileRab.rab$w_usz = sizeof(Line)-1;

   if (VMSnok (status = sys$connect (&IsMapFileRab, 0, 0)))
      ErrorVmsStatus (status, CgiPathInfoPtr, CgiPathTranslatedPtr,
                      __FILE__, __LINE__);

   /*********************/
   /* loop through file */
   /*********************/

   LineCount = 0;
   DefaultPath[0] = '\0';

   while (VMSok (status = sys$get (&IsMapFileRab, 0, 0)))
   {
      Line[IsMapFileRab.rab$w_rsz] = '\0';
      if (Debug) fprintf (stdout, "Line |%s|\n", Line);

      LineCount++;
      CernCompliant = false;

      /*********************/
      /* loop through line */
      /*********************/

      lptr = Line;
      while (*lptr)
      {
         while (isspace(*lptr)) lptr++;
         if (!*lptr || *lptr == '#' || *lptr == '!') break;

         /**********************/
         /* get region keyword */
         /**********************/

         zptr = (sptr = RegionKeyword) + sizeof(RegionKeyword);
         while (*lptr && !isspace(*lptr) && sptr < zptr)
            *sptr++ = tolower(*lptr++);
         *sptr = '\0';
         if (Debug) fprintf (stdout, "RegionKeyword |%s|\n", RegionKeyword);

         if (!(!memcmp (RegionKeyword, "circ", 4) ||
               !memcmp (RegionKeyword, "poin", 4) ||
               !memcmp (RegionKeyword, "poly", 4) ||
               !memcmp (RegionKeyword, "rect", 4) ||
               !memcmp (RegionKeyword, "def", 3)))
            ErrorConfigFile ("Unknown region keyword",
                             LineCount, 0, __FILE__, __LINE__);

         while (isspace(*lptr)) lptr++;
         if (!*lptr || *lptr == '#' || *lptr == '!')
            ErrorConfigFile ("Insufficient information",
                             LineCount, 0, __FILE__, __LINE__);

         /***********/
         /* get URL */
         /***********/

         if (isalpha(*lptr) || *lptr == '/' || *lptr == '.')
         {
            /**************************/
            /* must be NCSA compliant */
            /**************************/

            if (Debug) fprintf (stdout, "NCSA compliant\n");
            zptr = (sptr = Path) + sizeof(Path);
            while (*lptr && !isspace(*lptr) && sptr < zptr) *sptr++ = *lptr++;
            *sptr = '\0';
            if (Debug) fprintf (stdout, "Path |%s|\n", Path);
         }
         else
         {
            /**************************/
            /* must be CERN compliant */
            /**************************/

            /*
                The CERN configuration file has the URL following the
                coordinates.  Skip over any coordinates here looking
                for the URL, terminate after them, get the URL, resume.
                Turn any coordinate-grouping parentheses into spaces.
            */
            if (Debug) fprintf (stdout, "CERN compliant\n");
            CernCompliant = true;
            cptr = lptr;
            while (*cptr && (isspace(*cptr) || *cptr == '(' ||
                   *cptr == ',' || *cptr == ')' || isdigit(*cptr)))
            {
               if (*cptr == '(' || *cptr == ')')
                  *cptr++ = ' ';
               else
                  cptr++;
            }
            if (cptr > Line && isspace(cptr[-1]))
               cptr[-1] = '\0';
            else
               ErrorConfigFile ("Confused",
                                LineCount, cptr-Line, __FILE__, __LINE__);

            zptr = (sptr = Path) + sizeof(Path);
            while (*cptr && !isspace(*cptr) && sptr < zptr) *sptr++ = *cptr++;
            *sptr = '\0';
            if (Debug) fprintf (stdout, "Path |%s|\n", Path);
         }

         if (!Path[0])
            ErrorConfigFile ("Insufficient information",
                             LineCount, 0, __FILE__, __LINE__);

         /* default URL? */
         if (!memcmp (RegionKeyword, "def", 3))
         {
            strcpy (DefaultPath, Path);
            if (Debug) fprintf (stdout, "DefaultPath |%s|\n", DefaultPath);
            while (*lptr) lptr++;
            continue;
         }

         /*************************/
         /* process coordinate(s) */
         /*************************/

         CoordCount = 0;
         while (*lptr)
         {
            if (Debug) fprintf (stdout, "lptr |%s|\n", lptr);
            while (isspace(*lptr)) lptr++;
            if (!*lptr || *lptr == '#' || *lptr == '!')
            {
               if (CoordCount)
                  break;
               else
                  ErrorConfigFile ("Insufficient information",
                                   LineCount, 0, __FILE__, __LINE__);
            }

            if (CoordCount >= MAXVERTS-1)
               ErrorConfigFile ("Number of coordinates exceeds internal limit",
                                LineCount, 0, __FILE__, __LINE__);

            if (!isdigit(*lptr))
            {
               /********************/
               /* no coordinate(s) */
               /********************/

               ErrorConfigFile ("Coordinate problem",
                                LineCount, lptr-Line+1, __FILE__, __LINE__);
            }

            /********************/
            /* get X coordinate */
            /********************/

            if (Debug) fprintf (stdout, "lptr |%s|\n", lptr);
            CoordArray[CoordCount][X] = (double)atoi(lptr);
            while (*lptr && isdigit(*lptr)) lptr++;

            /*************************************/
            /* find comma-separated Y coordinate */
            /*************************************/

            while (isspace(*lptr) || *lptr == ',') lptr++;
            if (Debug) fprintf (stdout, "lptr |%s|\n", lptr);

            if (isdigit(*lptr))
            {
               /********************/
               /* get Y coordinate */
               /********************/

               CoordArray[CoordCount][Y] = (double)atoi(lptr);
               while (*lptr && isdigit(*lptr)) lptr++;
            }
            else
            {
               /*******************/
               /* no Y coordinate */
               /*******************/

               if (CernCompliant)
               {
                  if (!memcmp (RegionKeyword, "circ", 4) && CoordCount)
                  {
                     /*
                        CERN image mapping "circle" is "(X,Y) radius".
                        Fudge it by creating an "X,Y" out of the radius.
                     */
                     CoordArray[CoordCount][Y] =
                        CoordArray[CoordCount-1][Y] + CoordArray[CoordCount][X];
                     CoordArray[CoordCount][X] = CoordArray[CoordCount-1][X];
                  }
                  else
                     ErrorConfigFile ("Coordinate problem",
                                   LineCount, lptr-Line+1, __FILE__, __LINE__);
               }
               else
                  ErrorConfigFile ("Coordinate problem",
                                   LineCount, lptr-Line+1, __FILE__, __LINE__);
            }
            if (Debug)
               fprintf (stdout, "CoordArray[%d] |%f|%f|\n",
                        CoordCount,
                        CoordArray[CoordCount][X], CoordArray[CoordCount][Y]);

            CoordCount++;
         }

         /********************/
         /* lets try it out! */
         /********************/

         CoordArray[CoordCount][X] = -1;

         if (!memcmp (RegionKeyword, "circ", 4))
         {
            if (CoordCount != 2)
               ErrorConfigFile ("Incorrect number of coordinates",
                                LineCount, 0, __FILE__, __LINE__);
            if (pointincircle (ClickCoord, CoordArray))
            {
               ProcessVirtualPath (CgiPathInfoPtr, Path, Url, sizeof(Url));
               fprintf (HttpOut, "Location: %s\r\n", Url);
               exit (SS$_NORMAL);
            }
            continue;
         }

         if (!memcmp (RegionKeyword, "poly", 4))
         {
            if (CoordCount < 3)
               ErrorConfigFile ("Too few coordinates",
                                LineCount, 0, __FILE__, __LINE__);
            if (pointinpoly (ClickCoord, CoordArray))
            {
               ProcessVirtualPath (CgiPathInfoPtr, Path, Url, sizeof(Url));
               fprintf (HttpOut, "Location: %s\r\n", Url);
               exit (SS$_NORMAL);
            }
            continue;
         }

         if (!memcmp (RegionKeyword, "rect", 4))
         {
            if (CoordCount != 2)
               ErrorConfigFile ("Incorrect number of coordinates",
                                LineCount, 0, __FILE__, __LINE__);
            if (pointinrect (ClickCoord, CoordArray))
            {
               ProcessVirtualPath (CgiPathInfoPtr, Path, Url, sizeof(Url));
               fprintf (HttpOut, "Location: %s\r\n", Url);
               exit (SS$_NORMAL);
            }
            continue;
         }

         if (!memcmp (RegionKeyword, "poin", 4))
         {
            if (CoordCount != 1)
               ErrorConfigFile ("Incorrect number of coordinates",
                                LineCount, 0, __FILE__, __LINE__);
            Distance = (ClickCoord[X] - CoordArray[0][X]) *
                       (ClickCoord[X] - CoordArray[0][X]) +
                       (ClickCoord[Y] - CoordArray[0][Y]) *
                       (ClickCoord[Y] - CoordArray[0][Y]);
            if (Distance < ClosestPoint)
            {
               ClosestPoint = Distance;
               strcpy (DefaultPath, Path);
            }
            continue;
         }
      } 

      /**********************/
      /* end loop thru line */
      /**********************/
   }

   /**********************/
   /* end loop thru file */
   /**********************/

   sys$close (&IsMapFileFab, 0, 0);

   if (status == RMS$_EOF) status = SS$_NORMAL;
   if (VMSnok (status))
      ErrorVmsStatus (status, CgiPathInfoPtr, CgiPathTranslatedPtr,
                      __FILE__, __LINE__);

   if (DefaultPath[0])
   {
      ProcessVirtualPath (CgiPathInfoPtr, DefaultPath, Url, sizeof(Url));
      fprintf (HttpOut, "Location: %s\r\n", Url);
      exit (SS$_NORMAL);
   }
   else
      ErrorConfigFile ("No default specified", 0, 0, __FILE__, __LINE__);
}

/*****************************************************************************/
/*
Given the current document's URL path and the specified document's URL path, 
generate a resultant URL path based on whether the specified document's path 
is absolute or relative the the current document's path.  Insufficient space to
contain the result path will not crash the function and the result will be set
to an empty string.
*/ 

int ProcessVirtualPath
(
char *CurrentPath,
char *DocumentPath,
char *ResultPath,
int SizeOfResultPath
)
{
   register char  *cptr, *dptr, *sptr, *tptr, *zptr;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "ProcessVirtualPath() |%s|%s|\n",
               CurrentPath, DocumentPath);

   if (*(dptr = DocumentPath) == '/' || strstr (DocumentPath, "//") != NULL) 
   {
      /*****************/
      /* absolute path */
      /*****************/

      strcpy (ResultPath, dptr);
      zptr = (sptr = ResultPath) + SizeOfResultPath-1;
      while (*dptr && sptr < zptr) *sptr++ = *dptr++;
      if (sptr == zptr) sptr = ResultPath;
      *sptr = '\0';
      if (Debug) fprintf (stdout, "ResultPath |%s|\n", ResultPath);
      return;
   }

   /* step over any "relative to this directory" syntax ("./") */
   if (dptr[0] == '.' && dptr[1] == '/') dptr += 2;
   if (*dptr != '.')
   {
      /*****************/
      /* inferior path */
      /*****************/

      zptr = (sptr = tptr = ResultPath) + SizeOfResultPath-1;
      cptr = CurrentPath;
      while (*cptr && sptr < zptr)
      {
         if (*cptr == '/') tptr = sptr+1;
         *sptr++ = *cptr++;
      }
      sptr = tptr;
      while (*dptr && sptr < zptr) *sptr++ = *dptr++;
      if (sptr == zptr) sptr = ResultPath;
      *sptr = '\0';
      if (Debug) fprintf (stdout, "ResultPath |%s|\n", ResultPath);
      return;
   }

   /*****************/
   /* relative path */
   /*****************/

   zptr = (sptr = tptr = ResultPath) + SizeOfResultPath-1;
   cptr = CurrentPath;
   while (*cptr && sptr < zptr)
   {
      if (*cptr == '/') tptr = sptr;
      *sptr++ = *cptr++;
   }
   sptr = tptr;
   /* loop, stepping back one level for each "../" in the document path */
   while (dptr[0] == '.' && dptr[1] == '.' && dptr[2] == '/')
   {
      dptr += 3;
      if (sptr > ResultPath) sptr--;
      while (sptr > ResultPath && *sptr != '/') sptr--;
   }
   if (sptr > ResultPath)
      sptr++;
   else
      *sptr++ = '/';
   while (*dptr && sptr < zptr) *sptr++ = *dptr++;
   if (sptr == zptr) sptr = ResultPath;
   *sptr = '\0';
   if (Debug) fprintf (stdout, "ResultPath |%s|\n", ResultPath);
}

/*****************************************************************************/
/*
*/

ErrorConfigFile
(
char *Text,
int LineNumber,
int CharacterPosition,
char *SourceFileName,
int SourceLineNumber
)
{
   register char  *cptr;

   char AtLinePosition [256];

   /*********/
   /* begin */
   /*********/

   /* 
      The source file format provided by the "__FILE__" macro will be
      "device:[directory]name.type;ver".  Reduce that to "name".
   */
   for (cptr = SourceFileName; *cptr && *cptr != ';'; cptr++);
   if (*cptr)
   {
      while (*cptr != '.') cptr--;
      *cptr-- = '\0';
   }
   while (*cptr != ']') cptr--;
   cptr++;

   if (LineNumber)
      if (CharacterPosition)
         sprintf (AtLinePosition, "; at line %d, character %d.",
                  LineNumber, CharacterPosition);
      else
         sprintf (AtLinePosition, "; at line %d.", LineNumber);
   else
   {
      AtLinePosition[0] = '.';
      AtLinePosition[1] = '\0';
   }

   if (!HttpHasBeenOutput) fputs (Http404Header, HttpOut);
   fprintf (HttpOut,
"<!-- SoftwareID: %s Module: %s Line: %d -->\n\
<H1>ERROR!</H1>\n\
<P>Reported by server.\n\
<P>Image configuration file problem.\n\
<!-- %s -->\n\
<BR>%s%s\n\
<P><I>(document needs revision)</I>\n",
   SoftwareID, cptr, SourceLineNumber,
   CgiPathTranslatedPtr,
   Text, AtLinePosition);

   exit (SS$_NORMAL);
}

/*****************************************************************************/
/*
*/

ErrorGeneral
(
char *Text,
char *SourceFileName,
int SourceLineNumber
)
{
   register char  *cptr;

   /*********/
   /* begin */
   /*********/

   /* 
      The source file format provided by the "__FILE__" macro will be
      "device:[directory]name.type;ver".  Reduce that to "name".
   */
   for (cptr = SourceFileName; *cptr && *cptr != ';'; cptr++);
   if (*cptr)
   {
      while (*cptr != '.') cptr--;
      *cptr-- = '\0';
   }
   while (*cptr != ']') cptr--;
   cptr++;

   if (!HttpHasBeenOutput) fputs (Http404Header, HttpOut);
   fprintf (HttpOut,
"<!-- SoftwareID: %s Module: %s Line: %d -->\n\
<H1>ERROR!</H1>\n\
<P>Reported by server.\n\
<P>%s\n",
   SoftwareID, cptr, SourceLineNumber, Text);

   exit (SS$_NORMAL);
}

/*****************************************************************************/
/*
*/

ErrorVmsStatus
(
int StatusValue,
char *Text,
char *HiddenText,
char *SourceFileName,
int SourceLineNumber
)
{
   static char  Message [256];
   static $DESCRIPTOR (MessageDsc, Message);

   register char  *cptr, *sptr;
   int  status;
   short int  Length;

   /*********/
   /* begin */
   /*********/

   if (VMSok (status = sys$getmsg (StatusValue, &Length, &MessageDsc, 1, 0))) 
   {
      Message[Length] = '\0';
      Message[0] = toupper(Message[0]);
   }
   else
      exit (status);

   /* 
      The source file format provided by the "__FILE__" macro will be
      "device:[directory]name.type;ver".  Reduce that to "name".
   */
   for (cptr = SourceFileName; *cptr && *cptr != ';'; cptr++);
   if (*cptr)
   {
      while (*cptr != '.') cptr--;
      *cptr-- = '\0';
   }
   while (*cptr != ']') cptr--;
   cptr++;

   if (StatusValue == RMS$_FNF || StatusValue == RMS$_DNF ||
       StatusValue == RMS$_SYN || StatusValue == RMS$_PRV)
      sptr = "document needs revision";
   else
   if (StatusValue == RMS$_FLK)
      sptr = "try again";
   else
      sptr = "inform system administrator";

   if (!HttpHasBeenOutput) fputs (Http404Header, HttpOut);
   fprintf (HttpOut,
"<!-- SoftwareID: %s Module: %s Line: %d -->\n\
<H1>ERROR!</H1>\n\
<P>Reported by server.\n\
<P>%s ... <TT>%s</TT>\n\
<!-- %%X%08.08X \"%s\" -->\n\
<P><I>(%s)</I>\n",
   SoftwareID, cptr, SourceLineNumber,
   Message, Text, StatusValue, HiddenText, sptr);

   exit (SS$_NORMAL);
}

/****************************************************************************/
/*
Does a case-insensitive, character-by-character string compare and returns 
true if two strings are the same, or false if not.  If a maximum number of 
characters are specified only those will be compared, if the entire strings 
should be compared then specify the number of characters as 0.
*/ 

boolean strsame
(
register char *sptr1,
register char *sptr2,
register int  count
)
{
   while (*sptr1 && *sptr2)
   {
      if (toupper (*sptr1++) != toupper (*sptr2++)) return (false);
      if (count)
         if (!--count) return (true);
   }
   if (*sptr1 || *sptr2)
      return (false);
   else
      return (true);
}

/*****************************************************************************/
/*
NCSA imagemap.c routines.
*/

int pointinrect(double point[2], double coords[MAXVERTS][2])
{
        return ((point[X] >= coords[0][X] && point[X] <= coords[1][X]) &&
        (point[Y] >= coords[0][Y] && point[Y] <= coords[1][Y]));
}

int pointincircle(double point[2], double coords[MAXVERTS][2])
{
        int radius1, radius2;

        radius1 = ((coords[0][Y] - coords[1][Y]) * (coords[0][Y] -
        coords[1][Y])) + ((coords[0][X] - coords[1][X]) * (coords[0][X] -
        coords[1][X]));
        radius2 = ((coords[0][Y] - point[Y]) * (coords[0][Y] - point[Y])) +
        ((coords[0][X] - point[X]) * (coords[0][X] - point[X]));
        return (radius2 <= radius1);
}

int pointinpoly(double point[2], double pgon[MAXVERTS][2])
{
        int i, numverts, inside_flag, xflag0;
        int crossings;
        double *p, *stop;
        double tx, ty, y;

        for (i = 0; pgon[i][X] != -1 && i < MAXVERTS; i++)
                ;
        numverts = i;
        crossings = 0;

        tx = point[X];
        ty = point[Y];
        y = pgon[numverts - 1][Y];

        p = (double *) pgon + 1;
        if ((y >= ty) != (*p >= ty)) {
                if ((xflag0 = (pgon[numverts - 1][X] >= tx)) ==
                (*(double *) pgon >= tx)) {
                        if (xflag0)
                                crossings++;
                }
                else {
                        crossings += (pgon[numverts - 1][X] - (y - ty) *
                        (*(double *) pgon - pgon[numverts - 1][X]) /
                        (*p - y)) >= tx;
                }
        }

        stop = pgon[numverts];

        for (y = *p, p += 2; p < stop; y = *p, p += 2) {
                if (y >= ty) {
                        while ((p < stop) && (*p >= ty))
                                p += 2;
                        if (p >= stop)
                                break;
                        if ((xflag0 = (*(p - 3) >= tx)) == (*(p - 1) >= tx)) {
                                if (xflag0)
                                        crossings++;
                        }
                        else {
                                crossings += (*(p - 3) - (*(p - 2) - ty) *
                                (*(p - 1) - *(p - 3)) / (*p - *(p - 2))) >= tx;
                        }
                }
                else {
                        while ((p < stop) && (*p < ty))
                                p += 2;
                        if (p >= stop)
                                break;
                        if ((xflag0 = (*(p - 3) >= tx)) == (*(p - 1) >= tx)) {
                                if (xflag0)
                                        crossings++;
                        }
                        else {
                                crossings += (*(p - 3) - (*(p - 2) - ty) *
                                (*(p - 1) - *(p - 3)) / (*p - *(p - 2))) >= tx;
                        }
                }
        }
        inside_flag = crossings & 0x01;
        return (inside_flag);
}

/*****************************************************************************/

