/*****************************************************************************/
/*
                                 glist.c

CGI-compliant script to list graphics.  Call with the path to the directory 
and it returns a form allowing selection of what graphics are to be displayed.  
Each directory of graphics is displayed with the a list of subdirectories, 
the path to each graphic in the directory, and the image generated by the 
graphic. 


CGI VARIABLE NAMES
------------------
WWW_FORM_GIF            non-null includes GIF images
WWW_FORM_JPG            non-null includes JPEG images
WWW_FORM_JPG_INLINE     non-null includes JPEG as in-line images
WWW_FORM_LIST           non-null list the directory now
WWW_FORM_XBM            non-null includes X-bitmap images
WWW_PATH_INFO           the URL path component
WWW_PATH_TRANSLATED     the VMS equivalent of the URL path component
WWW_QUERY_STRING        any of the form components
WWW_SCRIPT_NAME         the name of the script being executed


BUILD
-----
See BUILD_GLIST.COM


VERSION HISTORY
---------------
09-FEB-96  MGD  v1.0.0, quick hack
*/
/*****************************************************************************/

#ifdef __ALPHA
   char SoftwareID [] = "GLIST AXP-1.1.0";
#else
   char SoftwareID [] = "GLIST VAX-1.1.0";
#endif

#ifdef __ALPHA
#   pragma nomember_alignment
#endif

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <stat.h>

#include <atrdef.h>
#include <descrip.h>
#include <dvidef.h>
#include <fibdef.h>
#include <iodef.h>
#include <rms.h>
#include <ssdef.h>
#include <stsdef.h>

#define VMSok(x) ((x) & STS$M_SUCCESS)
#define VMSnok(x) !(((x) & STS$M_SUCCESS))

#define boolean int
#define true 1
#define false 0

/* macro provides NULL pointer if CGI variable does not exist */
#define GetCgiVarIfExists(CharPointer,CgiVariableName) \
   CharPointer = getenv(CgiVariableName)

/* macro provides pointer to empty string even if CGI variable does not exist */
#define GetCgiVar(CharPointer,CgiVariableName) \
   if ((CharPointer = getenv(CgiVariableName)) == NULL) \
       CharPointer = ""; \
   if (Debug) fprintf (stdout, "%s |%s|\n", CgiVariableName, CharPointer);

char  ErrorInternal [] = "Internal error.",
      Title [] = "Listing of Graphics",
      Utility [] = "GLIST";

char  Http200Header [] =
"HTTP/1.0 200 Document follows.\r\n\
Content-Type: text/html\r\n\
\r\n";

char  Http404Header [] =
"HTTP/1.0 404 Error report follows.\r\n\
Content-Type: text/html\r\n\
\r\n";

boolean  Debug,
         HttpHasBeenOutput,
         IncludeGif,
         IncludeJpg,
         IncludeXbm,
         JpgInline;

char  *CgiFormGifPtr,
      *CgiFormJpgPtr,
      *CgiFormJpgInlinePtr,
      *CgiFormListPtr,
      *CgiFormXbmPtr,
      *CgiPathInfoPtr,
      *CgiPathTranslatedPtr,
      *CgiQueryStringPtr,
      *CgiScriptNamePtr;
      
FILE  *HttpOut;

/*****************************************************************************/
/*
*/

main
(
int argc,
char *argv[]
)
{
   int  acnt;

   /*********/
   /* begin */
   /*********/

   /* open another output stream so that the '\r' and '\n' are not filtered */
#ifdef __DECC
   if ((HttpOut = fopen ("SYS$OUTPUT", "w", "ctx=bin")) == NULL)
      exit (vaxc$errno);
#else
   if ((HttpOut = fopen ("SYS$OUTPUT", "w", "rfm=udf")) == NULL)
      exit (vaxc$errno);
#endif

   /***********************************/
   /* get the command line parameters */
   /***********************************/

   /* doing it this way, parameters must be space separated! */
   for (acnt = 1; acnt < argc; acnt++)
   {
      if (Debug) fprintf (stdout, "argv[%d] |%s|\n", acnt, argv[acnt]);
      if (strsame (argv[acnt], "/DBUG", -1))
      {
         Debug = true;
         continue;
      }
      fprintf (stdout, "%%%s-E-IVQUAL, unrecognized qualifier\n \\%s\\\n",
               Utility, argv[acnt]+1);
      exit (STS$K_ERROR | STS$M_INHIB_MSG);
   }

   /*********************/
   /* get CGI variables */
   /*********************/

   GetCgiVar (CgiPathInfoPtr, "WWW_PATH_INFO");
   GetCgiVar (CgiPathTranslatedPtr, "WWW_PATH_TRANSLATED");
   GetCgiVar (CgiQueryStringPtr, "WWW_QUERY_STRING");
   GetCgiVar (CgiScriptNamePtr, "WWW_SCRIPT_NAME");
   GetCgiVar (CgiFormListPtr, "WWW_FORM_LIST");
   GetCgiVar (CgiFormGifPtr, "WWW_FORM_GIF");
   GetCgiVar (CgiFormJpgPtr, "WWW_FORM_JPG");
   GetCgiVar (CgiFormJpgInlinePtr, "WWW_FORM_JPG_INLINE");
   GetCgiVar (CgiFormXbmPtr, "WWW_FORM_XBM");

   /***********************/
   /* execute the request */
   /***********************/

   IncludeGif = IncludeJpg = IncludeXbm = false;
   if (CgiFormGifPtr[0]) IncludeGif = true;
   if (CgiFormJpgPtr[0]) IncludeJpg = true;
   if (CgiFormJpgInlinePtr[0]) JpgInline = true;
   if (CgiFormXbmPtr[0]) IncludeXbm = true;
   if (!(IncludeGif || IncludeJpg || IncludeXbm))
      IncludeGif = IncludeXbm = true;

   if (CgiFormListPtr[0])
      ProcessFileSpec ();
   else
      ProvideForm ();

   exit (SS$_NORMAL);
}

/****************************************************************************/
/*
*/ 

ProvideForm ()

{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ProvideForm()\n");

   fprintf (HttpOut,
"%s\
<!-- SoftwareID: %s -->\n\
<TITLE>%s</TITLE>\n\
<H1>%s</H1>\n\
<P><HR>\n\
<FORM ACTION=\"%s%s\">\n\
Listing of graphics in <TT>%s</TT>\n\
<P><INPUT TYPE=checkbox NAME=gif VALUE=yes CHECKED> GIF\n\
<BR><INPUT TYPE=checkbox NAME=jpg VALUE=yes> JPEG \
<I>(<INPUT TYPE=checkbox NAME=jpg_inline VALUE=yes> \
browser supports in-line jpeg)</I>\n\
<BR><INPUT TYPE=checkbox NAME=xbm VALUE=yes CHECKED> X-bitmap\n\
<INPUT TYPE=hidden NAME=list VALUE=now>\n\
<P><INPUT TYPE=submit VALUE=\" list \"> <INPUT TYPE=reset VALUE=\" reset \">\n\
</FORM>\n\
<P><HR>\n",
   Http200Header,
   SoftwareID,
   Title, Title,
   CgiScriptNamePtr, CgiPathInfoPtr, CgiPathInfoPtr);
}

/****************************************************************************/
/*
*/ 

ProcessFileSpec ()

{
   register char  *cptr, *sptr, *zptr;

   boolean  FileOk,
            JpgGraphic;
   int  status,
        Bytes,
        CgiPathTranslatedLength,
        FileCount = 0,
        SubdirectoryCount = 0,
        TotalGif = 0,
        TotalJpg = 0,
        TotalXbm = 0;
   char  *GraphicTypePtr,
         *QuestionMarkPtr;
   char  DirectoryPath [256],
         ExpandedFileSpec [256],
         FileName [256],
         Name [256],
         NameAndType [256];
   struct FAB  FileFab;
   struct NAM  FileNam;
   struct RAB  FileRab;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ProcessFileSpec()\n");

   if (CgiQueryStringPtr[0])
      QuestionMarkPtr = "?";
   else
      QuestionMarkPtr = "";

   /* length is immediately after the closing directory bracket */
   for (cptr = CgiPathTranslatedPtr; *cptr; cptr++);
   while (cptr > CgiPathTranslatedPtr && *cptr != ']') cptr--;
   if (*cptr == ']') cptr++;
   CgiPathTranslatedLength = cptr - CgiPathTranslatedPtr;

   /* find the directory (i.e. the "/dir1/dir2/" from "/dir1/dir2/*.*") */
   zptr = (sptr = DirectoryPath) + sizeof(DirectoryPath)-1;
   for (cptr = CgiPathInfoPtr; *cptr && sptr < zptr; *sptr++ = *cptr++);
   if (sptr >= zptr) exit (ErrorGeneral (ErrorInternal, __FILE__, __LINE__));
   *sptr = '\0';
   while (sptr > DirectoryPath && *sptr != '/') sptr--;
   if (sptr > DirectoryPath && *sptr == '/') *++sptr = '\0';

   /***************************/
   /* list any subdirectories */
   /***************************/

   /* initialize the file access block */
   FileFab = cc$rms_fab;
   FileFab.fab$l_dna = "*.*";
   FileFab.fab$b_dns = 3;
   FileFab.fab$l_fna = CgiPathTranslatedPtr;
   FileFab.fab$b_fns = CgiPathTranslatedLength;
   FileFab.fab$l_fop = FAB$V_NAM;
   FileFab.fab$l_nam = &FileNam;

   /* initialize the file name block */
   FileNam = cc$rms_nam;
   FileNam.nam$l_esa = ExpandedFileSpec;
   FileNam.nam$b_ess = sizeof(ExpandedFileSpec)-1;
   FileNam.nam$l_rsa = FileName;
   FileNam.nam$b_rss = sizeof(FileName)-1;

   if (VMSnok (status = sys$parse (&FileFab, 0, 0)))
      exit (ErrorVmsStatus (status, CgiPathInfoPtr, CgiPathTranslatedPtr,
            __FILE__, __LINE__));

   /* output the HTTP header and page title, etc. */
   FileNam.nam$l_name[-1] = '\0';
   fprintf (HttpOut,
"%s\
<!-- SoftwareID: %s -->\n\
<!-- %s] -->\n\
<TITLE>%s</TITLE>\n\
<H1>%s</H1>\n\
<P><HR>\n",
            Http200Header, SoftwareID,
            (char*)FileNam.nam$l_dev, Title, Title);
   HttpHasBeenOutput = true;
   FileNam.nam$l_name[-1] = ']';

   while (VMSok (status = sys$search (&FileFab, 0, 0)))
   {
      if (Debug)
      {
         FileNam.nam$l_ver[0] = '\0';
         fprintf (stdout, "FileName |%s|\n", FileName);
         FileNam.nam$l_ver[0] = ';';
      }

      /* numeric equivalent of "DIR;" */
      if (*(unsigned long*)(FileNam.nam$l_type+1) != 0x3b524944)
         continue;

      if (!SubdirectoryCount++) fprintf (HttpOut, "<UL>\n");

      *(char*)FileNam.nam$l_type = '\0';
      zptr = (sptr = Name) + sizeof(Name)-1;
      for (cptr = FileNam.nam$l_name;
           *cptr && sptr < zptr;
           *sptr++ = tolower(*cptr++));
      *(char*)FileNam.nam$l_type = '.';
      if (sptr >= zptr) exit (ErrorGeneral (ErrorInternal, __FILE__, __LINE__));
      *sptr = '\0';

      fprintf (HttpOut, "<LI><A HREF=\"%s%s%s/*.*%s%s\">%s%s</A>\n",
               CgiScriptNamePtr, DirectoryPath, Name,
               QuestionMarkPtr, CgiQueryStringPtr,
               DirectoryPath, Name);
   }

   if (SubdirectoryCount) fprintf (HttpOut, "</UL>\n");

   if (status != RMS$_FNF && status != RMS$_NMF)
   {
      *(char*)FileNam.nam$l_ver = '\0';
      exit (ErrorVmsStatus (status, FileName, FileName, __FILE__, __LINE__));
   }

   /*********************/
   /* list any graphics */
   /*********************/

   /* initialize the file access block */
   FileFab = cc$rms_fab;
   FileFab.fab$l_dna = "*.*";
   FileFab.fab$b_dns = 3;
   FileFab.fab$l_fna = CgiPathTranslatedPtr;
   FileFab.fab$b_fns = CgiPathTranslatedLength;
   FileFab.fab$l_fop = FAB$V_NAM;
   FileFab.fab$l_nam = &FileNam;

   /* initialize the file name block */
   FileNam = cc$rms_nam;
   FileNam.nam$l_esa = ExpandedFileSpec;
   FileNam.nam$b_ess = sizeof(ExpandedFileSpec)-1;
   FileNam.nam$l_rsa = FileName;
   FileNam.nam$b_rss = sizeof(FileName)-1;

   if (VMSnok (status = sys$parse (&FileFab, 0, 0)))
      exit (ErrorVmsStatus (status, CgiPathInfoPtr, CgiPathTranslatedPtr,
            __FILE__, __LINE__));

   while (VMSok (status = sys$search (&FileFab, 0, 0)))
   {
      if (Debug)
      {
         FileNam.nam$l_ver[0] = '\0';
         fprintf (stdout, "FileName |%s|\n", FileName);
         FileNam.nam$l_ver[0] = ';';
      }

      /* numeric equivalent of "DIR;" */
      if (*(unsigned long*)(FileNam.nam$l_type+1) == 0x3b524944)
         continue;

      FileOk = JpgGraphic = false;
      /* numeric equivalent of "GIF;" */
      if (IncludeGif && *(unsigned long*)(FileNam.nam$l_type+1) == 0x3b464947)
         { FileOk = true; TotalGif++; GraphicTypePtr = "gif"; }
      if (IncludeJpg)
      {
         /* numeric equivalent of "JPG;" */
         if (*(unsigned long*)(FileNam.nam$l_type+1) == 0x3b47504a)
            { FileOk = JpgGraphic = true; TotalJpg++; GraphicTypePtr = "jpeg"; }
         /* numeric equivalent of "JPEG" */
         if (*(unsigned long*)(FileNam.nam$l_type+1) == 0x4745504a)
            { FileOk = JpgGraphic = true; TotalJpg++; GraphicTypePtr = "jpeg"; }
      }
      /* numeric equivalent of "XBM;" */
      if (IncludeXbm && *(unsigned long*)(FileNam.nam$l_type+1) == 0x3b4d4258)
         { FileOk = true; TotalXbm++; GraphicTypePtr = "x-bitmap"; }

      if (!FileOk) continue;

      if (VMSnok (FileSize (&FileNam, &Bytes)))
         exit (ErrorVmsStatus (status, FileName, FileName, __FILE__, __LINE__));

      if (!FileCount++)
      {
         if (SubdirectoryCount)
            fprintf (HttpOut, "<P><BR>\n<DL>\n");
         else
            fprintf (HttpOut, "<P>\n<DL>\n");
      }

      *(char*)FileNam.nam$l_ver = '\0';
      zptr = (sptr = NameAndType) + sizeof(NameAndType)-1;
      for (cptr = FileNam.nam$l_name;
           *cptr && sptr < zptr;
           *sptr++ = tolower(*cptr++));
      *(char*)FileNam.nam$l_ver = ';';
      if (sptr >= zptr) exit (ErrorGeneral (ErrorInternal, __FILE__, __LINE__));
      *sptr = '\0';

      *(char*)FileNam.nam$l_type = '\0';
      fprintf (HttpOut, "<P><DT><B>%s</B> <I>%s</I>\n<DD>%s%s ",
               FileNam.nam$l_name, GraphicTypePtr,
               DirectoryPath, NameAndType);
      *(char*)FileNam.nam$l_type = '.';
      if (JpgGraphic && !JpgInline)
         fprintf (HttpOut,
"<A HREF=\"%s%s\">[JPG]</A> <I>(%d bytes)</I>\n",
                  DirectoryPath, NameAndType, Bytes);
      else
         fprintf (HttpOut,
"<A HREF=\"%s%s\"><IMG SRC=\"%s%s\" ALIGN=TOP></A> <I>(%d bytes)</I>\n",
                  DirectoryPath, NameAndType,
                  DirectoryPath, NameAndType, Bytes);
   }

   if (FileCount) fprintf (HttpOut, "</DL>\n");

   if (status != RMS$_FNF && status != RMS$_NMF)
   {
      *(char*)FileNam.nam$l_ver = '\0';
      exit (ErrorVmsStatus (status, FileName, FileName, __FILE__, __LINE__));
   }

   if (FileCount)
   {
      if (TotalGif) fprintf (HttpOut, "<BR><I>gif: %d</I>\n", TotalGif);
      if (TotalJpg) fprintf (HttpOut, "<BR><I>jpeg: %d</I>\n", TotalJpg);
      if (TotalXbm) fprintf (HttpOut, "<BR><I>x-bitmap: %d</I>\n", TotalXbm);
   }
   else
      fprintf (HttpOut, "<P><I>No graphics found!</I>\n");
   fprintf (HttpOut, "<P><HR>\n");
}

/*****************************************************************************/
/*
This function uses the ACP-QIO interface detailed in the "OpenVMS I/O User's 
Reference Manual".
*/ 

int FileSize
(
struct NAM *FileNamPtr,
int *BytesPtr
)
{
   static $DESCRIPTOR (DeviceDsc, "");

   static struct {
      unsigned long  OfNoInterest1;
      unsigned short  AllocatedVbnHi;
      unsigned short  AllocatedVbnLo;
      unsigned short  EndOfFileVbnHi;
      unsigned short  EndOfFileVbnLo;
      unsigned short  FirstFreeByte;
      unsigned short  OfNoInterest2;
      unsigned long  OfNoInterestLots [4];
   } AtrRecord;

   static struct fibdef  FileFib;

   static struct atrdef  FileAtr [] =
   {
      { sizeof(AtrRecord), ATR$C_RECATTR, &AtrRecord },
      { 0, 0, 0 }
   };

   static struct {
      unsigned short  Length;
      unsigned short  Unused;
      unsigned long  Address;
   } FileNameAcpDsc,
     FileFibAcpDsc,
     FileAtrAcpDsc;

   static struct {
      unsigned short  Status;
      unsigned short  Unused1;
      unsigned long  Unused2;
   } AcpIOsb;

   int  status;
   unsigned short  AcpChannel;
   unsigned long  AllocatedVbn,
                  EndOfFileVbn;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "FileSize()\n");

   /* assign a channel to the disk device containing the file */
   DeviceDsc.dsc$w_length = FileNamPtr->nam$b_dev;
   DeviceDsc.dsc$a_pointer = FileNamPtr->nam$l_dev;
   status = sys$assign (&DeviceDsc, &AcpChannel, 0, 0, 0);
   if (Debug) fprintf (stdout, "sys$assign() %%X%08.08X\n", status);
   if (VMSnok (status)) return (status);

   /* set up the File Information Block for the ACP interface */
   memset (&FileFib, 0, sizeof(struct fibdef));
   FileFibAcpDsc.Length = sizeof(FileFib);
   FileFibAcpDsc.Address = &FileFib;
   /* quick work around, different syntax for this structure with DEC C? */
#ifdef __DECC
   FileFib.fib$w_did[0] = FileNamPtr->nam$w_did[0];
   FileFib.fib$w_did[1] = FileNamPtr->nam$w_did[1];
   FileFib.fib$w_did[2] = FileNamPtr->nam$w_did[2];
#else
   FileFib.fib$r_did_overlay.fib$w_did[0] = FileNamPtr->nam$w_did[0];
   FileFib.fib$r_did_overlay.fib$w_did[1] = FileNamPtr->nam$w_did[1];
   FileFib.fib$r_did_overlay.fib$w_did[2] = FileNamPtr->nam$w_did[2];
#endif

   FileNameAcpDsc.Address = FileNamPtr->nam$l_name;
   FileNameAcpDsc.Length = FileNamPtr->nam$b_name + FileNamPtr->nam$b_type +
                           FileNamPtr->nam$b_ver;

   status = sys$qiow (0, AcpChannel, IO$_ACCESS, &AcpIOsb, 0, 0, 
                      &FileFibAcpDsc, &FileNameAcpDsc, 0, 0,
                      &FileAtr, 0);

   /* immediately deassign the channel in case we return on an error */
   sys$dassgn (AcpChannel);

   if (Debug)
      fprintf (stdout, "sys$qio() %%X%08.08X IOsb: %%X%08.08X\n",
              status, AcpIOsb.Status);

   if (VMSok (status)) status = AcpIOsb.Status;
   if (VMSnok (status)) return (status);

   AllocatedVbn = AtrRecord.AllocatedVbnLo + (AtrRecord.AllocatedVbnHi << 16);
   EndOfFileVbn = AtrRecord.EndOfFileVbnLo + (AtrRecord.EndOfFileVbnHi << 16);

   if (Debug)
      fprintf (stdout, "AllocatedVbn: %d EndOfFileVbn: %d FirstFreeByte %d\n",
               AllocatedVbn, EndOfFileVbn, AtrRecord.FirstFreeByte);

   if (EndOfFileVbn <= 1)
      *BytesPtr = AtrRecord.FirstFreeByte;
   else
      *BytesPtr = ((EndOfFileVbn - 1) << 9) + AtrRecord.FirstFreeByte;
   if (Debug) fprintf (stdout, "Bytes %d\n", *BytesPtr);

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
*/

ErrorGeneral
(
char *Text,
char *SourceFileName,
int SourceLineNumber
)
{
   register char  *cptr;

   /*********/
   /* begin */
   /*********/

   /* 
      The source file format provided by the "__FILE__" macro will be
      "device:[directory]name.type;ver".  Reduce that to "name.type".
   */
   for (cptr = SourceFileName; *cptr && *cptr != ';'; cptr++);
   if (*cptr)
   {
      while (*cptr != '.') cptr--;
      *cptr-- = '\0';
   }
   while (*cptr != ']') cptr--;
   cptr++;

   if (!HttpHasBeenOutput) fputs (Http404Header, HttpOut);
   fprintf (HttpOut,
"<!-- SoftwareID: %s Module: %s Line: %d -->\n\
<H1>ERROR!</H1>\n\
<P>Reported by server.\n\
<P>%s\n",
   SoftwareID, cptr, SourceLineNumber, Text);

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
*/

ErrorVmsStatus
(
int StatusValue,
char *Text,
char *HiddenText,
char *SourceFileName,
int SourceLineNumber
)
{
   static char  Message [256];
   static $DESCRIPTOR (MessageDsc, Message);

   register char  *cptr;
   int  status;
   short int  Length;

   /*********/
   /* begin */
   /*********/

   if (VMSok (status = sys$getmsg (StatusValue, &Length, &MessageDsc, 1, 0))) 
   {
      Message[Length] = '\0';
      Message[0] = toupper(Message[0]);
   }
   else
      exit (status);

   /* 
      The source file format provided by the "__FILE__" macro will be
      "device:[directory]name.type;ver".  Reduce that to "name.type".
   */
   for (cptr = SourceFileName; *cptr && *cptr != ';'; cptr++);
   if (*cptr)
   {
      while (*cptr != '.') cptr--;
      *cptr-- = '\0';
   }
   while (*cptr != ']') cptr--;
   cptr++;

   if (!HttpHasBeenOutput) fputs (Http404Header, HttpOut);
   fprintf (HttpOut,
"<!-- SoftwareID: %s Module: %s Line: %d -->\n\
<H1>ERROR!</H1>\n\
<P>Reported by server.\n\
<P>%s ... <TT>%s</TT>\n\
<!-- %%X%08.08X \"%s\" -->\n",
   SoftwareID, cptr, SourceLineNumber, Message, Text, StatusValue, HiddenText);

   return (SS$_NORMAL);
}

/****************************************************************************/
/*
Does a case-insensitive, character-by-character string compare and returns 
true if two strings are the same, or false if not.  If a maximum number of 
characters are specified only those will be compared, if the entire strings 
should be compared then specify the number of characters as 0.
*/ 

boolean strsame
(
register char *sptr1,
register char *sptr2,
register int  count
)
{
   while (*sptr1 && *sptr2)
   {
      if (toupper (*sptr1++) != toupper (*sptr2++)) return (false);
      if (count)
         if (!--count) return (true);
   }
   if (*sptr1 || *sptr2)
      return (false);
   else
      return (true);
}

/*****************************************************************************/

