/*****************************************************************************/
/*
                                 Deform.c

A CGI-compliant utility.

This utility is designed for use in script processing.  It will process HTTP 
"GET" and "POST" methods. 

For both methods the utility can be instructed to check for the presence of an 
authenticated remote user name and generate an authentication failure HTTP 
header to force authentication to occur.  A realm must be specified using the 
/AUTHENTICATE= qualifier.

Will generate a redirection header to a document specified by URL using the 
/LOCATION= qualifier.  This can be a full URL ("scheme://host/path") or just a 
full path local to the server ("/path"). 

It processes HTTP "POST" method "application/x-www-form-urlencoded" requests 
into "WWW_FORM_..." symbols names or into a specially formatted file.

If an HTTP reportable error occurs an error header is generated and sent to 
the client and the utility exits with an error status.  It is the 
resposibility of the user to check for errors after image completion and take 
appropriate action.  The most common action within a working script will to be 
to exit the script procedure (e.g. "$ IF .NOT. $STATUS THEN EXIT").


QUALIFIERS
----------
/AUTHENTICATE=  requires a realm, checks for authenticated remote user name
/DBUG           turns on all "if (Debug)" statements
/FILE           parses form fields into temporary file
/HTML           generates an HTTP response header "Content-Type: text/html"
/LOCATION=      URL of document to be redirected to
/SCRATCH        define symbols for the scratch directory and a unique number
/STATUS=        generates an HTTP response header contain the HTTP status
/SYMBOLS        parses form fields into DCL global symbols
/TEXT           generates an HTTP response header "Content-Type: text/plain"
/TYPE=          generates an HTTP response header "Content-Type: [type]"


BUILD DETAILS
-------------
See BUILD_DEFORM.COM procedure.


VERSION HISTORY (update SoftwareID as well!)
---------------
19-SEP-95  MGD  v1.1.1, replace <CR><LF> carriage-control with single newline,
                        still acceptable for HTTP, and slightly more efficient
24-MAY-95  MGD  v1.1.0, minor changes for AXP compatibility
18-APR-95  MGD  v1.0.0, initial development
*/
/*****************************************************************************/

#ifdef __ALPHA
   char SoftwareID [] = "DEFORM AXP-1.1.1";
#else
   char SoftwareID [] = "DEFORM VAX-1.1.1";
#endif

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <errno.h>
#include <unixio.h>

/* VMS-related header files */
#include <descrip.h>
#include <jpidef.h>
#include <libdef.h>
#include <libclidef.h>
#include <rms.h>
#include <ssdef.h>
#include <stsdef.h>

#ifdef __ALPHA
#   pragma nomember_alignment
#endif

#define boolean int
#define true 1
#define false 0

#define VMSok(x) ((x) & STS$M_SUCCESS)
#define VMSnok(x) (!((x) & STS$M_SUCCESS))
 
#define WWWFormSymbolPrefix "WWW_FORM_"
#define ScratchDirectory "HT_ROOT:[SCRATCH]"
#define OutputSymbolName "DEFORM_FILE"
#define ScratchDirectorySymbolName "DEFORM_SCRATCH"
#define UniqueSymbolName "DEFORM_UNIQUE"
#define HttpBufferSize 4096

char  Http200Header [] =
"HTTP/1.0 200 Document follows.\n\
Content-Type: text/html\n\
\n";

char  Http404Header [] =
"HTTP/1.0 404 Error report follows.\n\
Content-Type: text/html\n\
\n";

char  ErrorStringSize [] = "String size too small.";

char  Utility [] = "DEFORM";

boolean  Debug,
         DoIntoSymbols,
         DoIntoFile,
         DoMethodGet,
         DoMethodPost,
         DoScratch,
         HttpHasBeenOutput;

int  ContentCount,
     ContentLength;
     
char  AuthenticateRealm [256],
      CgiContentLength [16],
      CgiContentType [256],
      CgiRequestMethod [16],
      CgiRemoteUser [16],
      CgiScriptName [256],
      CgiServerName [128],
      ContentType [256],
      HttpStatus [256],
      Location [256],
      Output [256],
      Unique [32];

FILE  *HttpIn,
      *HttpOut;

char* CopyToHtml (char*, char*, int);
int GetHttpString (char, char*, int, boolean);

/*****************************************************************************/
/*
*/

main (int argc, char *argv[])

{
   register int  acnt;
   register char  *cptr, *sptr;

   int  status;
   char  String [256];
   unsigned long  BinTime [2];
   unsigned short  NumTime [7];

   /*********/
   /* begin */
   /*********/

   /* open another output stream so that the "\n" are not filtered */
#ifdef __DECC
   if ((HttpOut = fopen ("SYS$OUTPUT", "w", "ctx=bin")) == NULL)
      exit (vaxc$errno);
#else
   if ((HttpOut = fopen ("SYS$OUTPUT", "w", "rfm=udf")) == NULL)
      exit (vaxc$errno);
#endif

   /***********************************/
   /* get the command line parameters */
   /***********************************/

   /* using this mechanism parameters must be space-separated! */
   for (acnt = 1; acnt < argc; acnt++)
   {
      if (Debug) fprintf (stdout, "argv[%d] |%s|\n", acnt, argv[acnt]);
      if (strsame (argv[acnt], "/AUTHENTICATE=", 5))
      {
         for (cptr = argv[acnt]+5; *cptr && *cptr != '='; cptr++);
         if (*cptr) cptr++;
         sptr = AuthenticateRealm;
         while (*cptr) *sptr++ = toupper(*cptr++);
         *sptr = '\0';
         if (!AuthenticateRealm[0])
         {
            fprintf (stdout, "%%%s-E-AUTHENTICATE, realm not specified\n",
                     Utility);
            exit (STS$K_ERROR | STS$M_INHIB_MSG);
         }
         continue;
      }
      if (strsame (argv[acnt], "/DBUG", -1))
      {
         Debug = true;
         continue;
      }
      if (strsame (argv[acnt], "/FILE", 4))
      {
         DoIntoFile = true;
         DoIntoSymbols = false;
         continue;
      }
      if (strsame (argv[acnt], "/GET", 4))
      {
         DoMethodGet = true;
         DoMethodPost = false;
         continue;
      }
      if (strsame (argv[acnt], "/HTML", 4))
      {
         strcpy (ContentType, "text/html");
         continue;
      }
      if (strsame (argv[acnt], "/LOCATION=", 4))
      {
         for (cptr = argv[acnt]+5; *cptr && *cptr != '='; cptr++);
         if (*cptr) cptr++;
         sptr = Location;
         while (*cptr) *sptr++ = *cptr++;
         *sptr = '\0';
         if (!Location[0])
         {
            fprintf (stdout, "%%%s-E-LOCATION, URL not specified\n", Utility);
            exit (STS$K_ERROR | STS$M_INHIB_MSG);
         }
         continue;
      }
      if (strsame (argv[acnt], "/POST", 4))
      {
         DoMethodPost = true;
         DoMethodGet = false;
         continue;
      }
      if (strsame (argv[acnt], "/SCRATCH", 4))
      {
         DoScratch = true;
         continue;
      }
      if (strsame (argv[acnt], "/STATUS=", 4))
      {
         for (cptr = argv[acnt]+4; *cptr && *cptr != '='; cptr++);
         if (*cptr) cptr++;
         sptr = HttpStatus;
         while (*cptr) *sptr++ = toupper(*cptr++);
         *sptr = '\0';
         if (!HttpStatus[0])
         {
            fprintf (stdout,
               "%%%s-E-STATUS, HTTP response status not specified\n",
               Utility);
            exit (STS$K_ERROR | STS$M_INHIB_MSG);
         }
         continue;
      }
      if (strsame (argv[acnt], "/SYMBOLS", 4))
      {
         DoIntoSymbols = true;
         DoIntoFile = false;
         continue;
      }
      if (strsame (argv[acnt], "/TEXT", 4))
      {
         strcpy (ContentType, "text/plain");
         continue;
      }
      if (strsame (argv[acnt], "/TYPE=", 4))
      {
         for (cptr = argv[acnt]+4; *cptr && *cptr != '='; cptr++);
         if (*cptr) cptr++;
         sptr = ContentType;
         while (*cptr) *sptr++ = *cptr++;
         *sptr = '\0';
         if (!ContentType[0])
         {
            fprintf (stdout, "%%%s-E-TYPE, HTTP content type not specified\n",
                     Utility);
            exit (STS$K_ERROR | STS$M_INHIB_MSG);
         }
         continue;
      }
      fprintf (stdout, "%%%s-E-IVQUAL, unrecognized qualifier\n \\%s\\\n",
               Utility, argv[acnt]+1);
      exit (STS$K_ERROR | STS$M_INHIB_MSG);
   }

   /***********/
   /* process */
   /***********/

   GetCgiVariable ("WWW_REQUEST_METHOD", CgiRequestMethod,
                   sizeof(CgiRequestMethod));

   /* numeric equivalents of "GET\0" and "POST" */
   if (!(*(unsigned long*)CgiRequestMethod == 0x00544547 ||
         *(unsigned long*)CgiRequestMethod == 0x54534F50))
      exit (STS$K_ERROR | STS$M_INHIB_MSG);

   /* if the method is "GET\0" and this is meant for "POST" then just exit */
   if (*(unsigned long*)CgiRequestMethod == 0x00544547 && DoMethodPost)
      exit (SS$_NORMAL);
   /* if the method is "POST" and this is meant for "GET" then just exit */
   if (*(unsigned long*)CgiRequestMethod == 0x54534F50 && DoMethodGet)
      exit (SS$_NORMAL);

   if (AuthenticateRealm[0])
   {
      /****************/
      /* authenticate */
      /****************/

      GetCgiVariable ("WWW_REMOTE_USER", CgiRemoteUser, sizeof(CgiRemoteUser));
      if (!CgiRemoteUser[0])
      {
         ErrorAuthentication (__FILE__, __LINE__);
         exit (STS$K_ERROR | STS$M_INHIB_MSG);
      }
   }

   sys$gettim (&BinTime);
   sys$numtim (&NumTime, &BinTime);

   sprintf (Unique,
   "%02.02d%02.02d%02.02d%02.02d%02.02d%02.02d%02.02d",
   NumTime[0]%100, NumTime[1], NumTime[2],
   NumTime[3], NumTime[4], NumTime[5], NumTime[6]);
   if (Debug) fprintf (stdout, "Unique |%s|\n", Unique);

   if (DoScratch)
   {
      SetGlobalSymbol (UniqueSymbolName, Unique);
      SetGlobalSymbol (ScratchDirectorySymbolName, ScratchDirectory);
   }

   /*****************************/
   /* HTTP header functionality */
   /*****************************/

   if (Location[0])
   {
      fputs ("HTTP/1.0 302 Redirection.\n", HttpOut);
      if (Location[0] == '/') 
      {
         GetCgiVariable ("WWW_SERVER_NAME", CgiServerName,
                         sizeof(CgiServerName));
         fprintf (HttpOut, "Location: http://%s%s\n",
                  CgiServerName, Location);
      }
      else
         fprintf (HttpOut, "Location: %s\n", Location);
      fprintf (HttpOut,
"Server: %s\n\
Content-Type: text/html\n\
\n\
<!-- SoftwareID: %s -->\n\
<H1>Redirection!</H1>\n\
<P>Location: <TT>http://%s%s</TT>\n",
      CgiServerName, Location,
      SoftwareID,
      SoftwareID,
      CgiServerName, Location);
      exit (SS$_NORMAL);
   }

   if (HttpStatus[0])
   {
      if (!ContentType[0]) strcpy (ContentType, "text/plain");
      fprintf (HttpOut,
"HTTP/1.0 %s\n\
Content-Type: %s\n\
Server: %s\n\
\n",
      HttpStatus, ContentType, SoftwareID);
      exit (SS$_NORMAL);
   }

   if (ContentType[0])
   {
      fprintf (HttpOut,
"Content-Type: %s\n\
Server: %s\n\
\n",
      ContentType, SoftwareID);
      exit (SS$_NORMAL);
   }

   /* numeric equivalent of "GET\0" (just a bit more efficient, that's all!) */
   if (*(unsigned long*)CgiRequestMethod == 0x00544547)
   {
      /**************/
      /* GET method */
      /**************/

      exit (SS$_NORMAL);
   }

   /* numeric equivalent of "POST" (just a bit more efficient, that's all!) */
   if (*(unsigned long*)CgiRequestMethod == 0x54534F50)
   {
      /***************/
      /* POST method */
      /***************/

      /* get method specific CGI variables */
      GetCgiVariable ("WWW_CONTENT_LENGTH", CgiContentLength,
                      sizeof(CgiContentLength));
      GetCgiVariable ("WWW_CONTENT_TYPE", CgiContentType,
                      sizeof(CgiContentType));

      if (!strsame (CgiContentType, "application/x-www-form-urlencoded", -1))
      {
         sprintf (String, "POST Content-Type: \"<TT>%s</TT>\" not supported.",
                  CgiContentType);
         ErrorGeneral (String, __FILE__, __LINE__);
         exit (STS$K_ERROR | STS$M_INHIB_MSG);
      }

      /* open the HTTP input stream */
      if ((HttpIn = fopen ("HTTP$INPUT", "r")) == NULL)
         exit (vaxc$errno);

      ProcessHeader ();

      ContentLength = atoi (CgiContentLength);

      if (DoIntoSymbols)
         ProcessBodyIntoSymbols ();
      else
      if (DoIntoFile)
         ProcessBodyIntoFile ();
      else
         exit (STS$K_ERROR | STS$M_INHIB_MSG);

      exit (SS$_NORMAL);
   }

   exit (SS$_NORMAL);
}

/*****************************************************************************/
/*
Read (and absorb) the HTTP stream up to the first blank line (end of HTTP 
header).  All required information can be obtained via CGI variables and the 
URL-encoded body.
*/

ProcessHeader ()

{
   char  HeaderLine [1024],
         String [256];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ProcessHeader()\n");

   for (;;)
   {
      if (GetHttpString ('\n', HeaderLine, sizeof(HeaderLine), false)
          == sizeof(HeaderLine))
      {
         sprintf (String,
         "The HTTP header fields can be %d characters maximum.",
         sizeof(HeaderLine)-1);
         ErrorGeneral (String, __FILE__, __LINE__);
         exit (STS$K_ERROR | STS$M_INHIB_MSG);
      }

      if (!HeaderLine[0])
      {
         if (Debug) fprintf (stdout, "end-of-header\n");
         return;
      }
   }
}

/*****************************************************************************/
/*
The process body should be a URL-format string, possibly broken into multiple 
lines.  Read that string getting the URL-encoded form keywords and the 
associated values into global symbols.
*/ 

ProcessBodyIntoSymbols ()

{
   static int  GlobalSymbol = LIB$K_CLI_GLOBAL_SYM;
   static int  MaxFieldNameLength = 255 - sizeof(WWWFormSymbolPrefix) - 1;

   register char  *cptr;

   int  status,
        FieldLength,
        ValueLength;
   char  FieldName [256],
         FieldValue [256],
         String [512],
         SymbolName [256];
   $DESCRIPTOR (SymbolNameDsc, SymbolName);
   $DESCRIPTOR (SymbolValueDsc, FieldValue);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ProcessBody()\n");

   strcpy (SymbolName, WWWFormSymbolPrefix);

   ContentCount = 0;
   while (ContentCount < ContentLength)
   {
      String[0] = '\0';
      FieldLength = GetHttpString ('=', FieldName, sizeof(FieldName), true);
      if (FieldLength < 0)
      {
         strcpy (String, "HTTP stream terminated unexpectedly.");
         ErrorGeneral (String, __FILE__, __LINE__);
         exit (STS$K_ERROR | STS$M_INHIB_MSG);
      }
      if (FieldLength > MaxFieldNameLength)
      {
         FieldName[20] = '\0';
         sprintf (String,
"The field name beginning <TT>%s</TT>... exceeds the %d character maximum.",
         FieldName, MaxFieldNameLength);
         ErrorGeneral (String, __FILE__, __LINE__);
         exit (STS$K_ERROR | STS$M_INHIB_MSG);
      }
      if (Debug) fprintf (stdout, "FieldName |%s|\n", FieldName);

      for (cptr = FieldName; *cptr; cptr++)
         if (!(isalnum(*cptr) || *cptr == '_' || *cptr == '$'))
            *cptr = '_';

      ValueLength = GetHttpString ('&', FieldValue, sizeof(FieldValue), true);
      if (ValueLength < 0)
      {
         strcpy (String, "HTTP stream terminated unexpectedly.");
         ErrorGeneral (String, __FILE__, __LINE__);
         exit (STS$K_ERROR | STS$M_INHIB_MSG);
      }
      if (ValueLength > 255)
      {
         sprintf (String,
"The value of field <TT>%s</TT> exceeds the 255 character maximum.",
         FieldName);
         ErrorGeneral (String, __FILE__, __LINE__);
         exit (STS$K_ERROR | STS$M_INHIB_MSG);
      }
      if (Debug) fprintf (stdout, "FieldValue |%s|\n", FieldValue);

      strcpy (SymbolName+sizeof(WWWFormSymbolPrefix)-1, FieldName);
      SymbolNameDsc.dsc$w_length = FieldLength + sizeof(WWWFormSymbolPrefix)-1;
      SymbolValueDsc.dsc$w_length = ValueLength;

      status = lib$set_symbol (&SymbolNameDsc, &SymbolValueDsc, &GlobalSymbol);
      if (VMSnok (status))
      {
         ErrorVmsStatus (status, FieldName, FieldName, __FILE__, __LINE__);
         exit (STS$K_ERROR | STS$M_INHIB_MSG);
      }
   }
}

/*****************************************************************************/
/*
*/ 

ProcessBodyIntoFile ()

{
   register int  len;
   register char  *cptr, *sptr;

   int  status,
        FieldCount,
        FieldLength,
        ValueLength;
   char  FieldName [256],
         FieldValue [16384],
         FileName [256],
         String [512];
   FILE  *OutFile;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ProcessBodyIntoFile()\n");

   sprintf (FileName, "%s%s.TMP", ScratchDirectory, Unique);
   if (Debug) fprintf (stdout, "FileName |%s|\n", FileName);

   FieldCount = 0;

   if ((OutFile = fopen (FileName, "w")) == NULL)
   {
      status = vaxc$errno;
      ErrorVmsStatus (status, "temporary file", FileName, __FILE__, __LINE__);
      exit (STS$K_ERROR | STS$M_INHIB_MSG);
   }

   SetGlobalSymbol (OutputSymbolName, FileName);

   while (ContentCount < ContentLength)
   {
      /* get the next field name from HTTP stream */
      FieldLength = GetHttpString ('=', FieldName, sizeof(FieldName), true);
      if (FieldLength == sizeof(FieldName))
      {
         FieldName[20] = '\0';
         sprintf (String,
"The field name beginning <TT>%s</TT>... exceeds the %d character maximum.",
         FieldName, sizeof(FieldName)-1);
         fclose (OutFile);
         remove (FileName);
         ErrorGeneral (String, __FILE__, __LINE__);
         exit (STS$K_ERROR | STS$M_INHIB_MSG);
      }
      if (Debug) fprintf (stdout, "FieldName |%s|\n", FieldName);

      /* get the corresponding field value from HTTP stream */
      ValueLength = GetHttpString ('&', FieldValue, sizeof(FieldValue), true);
      if (ValueLength == sizeof(FieldValue))
      {
         sprintf (String,
"The value of field <TT>%s</TT> exceeds the %d character maximum.",
         FieldName, sizeof(FieldValue)-1);
         fclose (OutFile);
         remove (FileName);
         ErrorGeneral (String, __FILE__, __LINE__);
         exit (STS$K_ERROR | STS$M_INHIB_MSG);
      }
      if (Debug) fprintf (stdout, "FieldValue |%s|\n", FieldValue);

      /* if not the first field separate by a blank line */
      if (FieldCount++) fputs ("\n", OutFile);

      /* place the field name into the file */
      fprintf (OutFile, "%s\n", FieldName);

      /* place the field value into the file */
      cptr = FieldValue;
      while (*cptr)
      {
         sptr = cptr;
         while (*cptr && *cptr != '\n') cptr++;
         if (*cptr) *cptr++ = '\0';
         fprintf (OutFile, "=%s\n", sptr);
      }
   }

   fclose (OutFile);
}

/*****************************************************************************/
/*
This function fills the 'String' storage with up to 'SizeOfString'-1 
characters from the HTTP input stream until the storage fills or the delimiter 
is encountered.  If the delimitter is specified as the null character ('\0') 
the all of the reset of the data stream is placed into the 'String' storage.  
It returns 0 or greater as the length of the returned string.  If the 'String' 
storage is too small to completely receive the characters required the 
function returns 'SizeOfString' to indicate the overflow.  Uses the global 
storage 'ContentLength' (if non-zero) to check when the stream should 
terminate, keeping track of the number of characters processed in global 
storage 'ContentCount'.  Not designed to retrieve from binary data streams!
*/ 

int GetHttpString
(
char Delimiter,
char *String,
int SizeOfString,
boolean UrlEscaped
)
{
   static char  *HttpBufferPtr = NULL;
   static char  HttpBuffer [HttpBufferSize];

   register int  sleft, clen, ccnt;
   register char  c, del, fc;
   register char  *sptr, *hbptr;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout,
               "GetHttpString() ContentLength/Count: %d/%d\n",
               ContentLength, ContentCount);

   if (HttpBufferPtr == NULL)
      *(hbptr = HttpBuffer) = '\0';
   else
      hbptr = HttpBufferPtr;

   /* put these into register storage for efficiency */
   clen = ContentLength;
   ccnt = ContentCount;
   del = Delimiter;
   sptr = String;
   sleft = SizeOfString - 1;

   while ((!del || *hbptr != del) && sleft && (!clen || ccnt < clen))
   {
      if (!*hbptr)
      {
         /* buffer empty (or first call), read from HTTP source */
         if (Debug) fprintf (stdout, "fgets()\n");
         if (fgets (HttpBuffer, sizeof(HttpBuffer), HttpIn) == NULL)
         {
            ErrorGeneral ("HTTP stream terminated unexpectedly.",
                          __FILE__, __LINE__);
            exit (STS$K_ERROR | STS$M_INHIB_MSG);
         }
         hbptr = HttpBuffer;
         if (Debug) fprintf (stdout, "HttpBuffer (%d) |%s|\n", __LINE__, hbptr);
         continue;
      }
      if (*hbptr == '\r' || *hbptr == '\n')
      {
         hbptr++;
         continue;
      }
      if (!UrlEscaped)
      {
         *sptr++ = *hbptr++;
         ccnt++;
         sleft--;
         continue;
      }
      if (*hbptr != '%')
      {
         /* not an escaped character */
         if (*hbptr == '+')
         {
            *sptr++ = ' ';
            hbptr++;
         }
         else
            *sptr++ = *hbptr++;
         ccnt++;
         sleft--;
         continue;
      }

      /* an escaped character ("%xx" where xx is a hex number) */
      hbptr++;
      if (!*hbptr)
      {
         /* buffer empty read from HTTP source */
         if (Debug) fprintf (stdout, "fgets()\n");
         if (fgets (HttpBuffer, sizeof(HttpBuffer), HttpIn) == NULL)
         {
            ErrorGeneral ("HTTP stream terminated unexpectedly.",
                          __FILE__, __LINE__);
            exit (STS$K_ERROR | STS$M_INHIB_MSG);
         }
         hbptr = HttpBuffer;
         if (Debug) fprintf (stdout, "HttpBuffer (%d) |%s|\n", __LINE__, hbptr);
      }
      ccnt++;
      c = 0;
      /* first hex digit */
      fc = *hbptr;
      if (fc >= '0' && fc <= '9')
         c = (fc - (int)'0') << 4;
      else
      if (toupper(fc) >= 'A' && toupper(fc) <= 'F')
         c = (toupper(fc) - (int)'A' + 10) << 4;
      else
      {
         /* not a valid hex digit, assume its a mistake! */
         *sptr++ = '%';
         continue;
      }
      if (fc) hbptr++;
      if (!*hbptr)
      {
         /* buffer empty read from HTTP source */
         if (Debug) fprintf (stdout, "fgets()\n");
         if (fgets (HttpBuffer, sizeof(HttpBuffer), HttpIn) == NULL)
         {
            ErrorGeneral ("HTTP stream terminated unexpectedly.",
                          __FILE__, __LINE__);
            exit (STS$K_ERROR | STS$M_INHIB_MSG);
         }
         hbptr = HttpBuffer;
         if (Debug) fprintf (stdout, "HttpBuffer (%d) |%s|\n", __LINE__, hbptr);
      }
      ccnt++;
      /* second hex digit */
      if (*hbptr >= '0' && *hbptr <= '9')
         c += (*hbptr - (int)'0');
      else
      if (toupper(*hbptr) >= 'A' && toupper(*hbptr) <= 'F')
         c += (toupper(*hbptr) - (int)'A' + 10);
      else
      {
         /* not a valid hex digit, assume its a bigger mistake! */
         *sptr++ = '%';
         *sptr++ = fc;
         continue;
      }
      if (*hbptr) hbptr++;
      *sptr++ = c;
      ccnt++;
      sleft--;
   }

   if (*hbptr == del)
   {
      /* if the loop exited due to a delimiter encountered, step over it */
      hbptr++;
      ccnt++;
   }

   *sptr = '\0';
   if (Debug) fprintf (stdout, "String |%s|\n", String);

   /* if we're using a 'ContentLength' then keep track of 'ContentCount' */
   if (clen)
      ContentCount = ccnt;
   else
      ContentCount = 0;

   HttpBufferPtr = hbptr;

   if (!sleft && *hbptr != del)
   {
      /* must have run out of string space, return overflow indication */
      return (SizeOfString);
   }
   else
      return (sptr - String);
}

/*****************************************************************************/
/*
*/

boolean GetCgiVariable
(
char *VariableName,
char *VariableValue,
int VariableValueSize
)
{
   static $DESCRIPTOR (VariableNameDsc, "");
   static $DESCRIPTOR (VariableValueDsc, "");

   register int  status;

   unsigned short  Length;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "GetCgiVariable() |%s|\n", VariableName);

   VariableNameDsc.dsc$w_length = strlen(VariableName);
   VariableNameDsc.dsc$a_pointer = VariableName;
   VariableValueDsc.dsc$w_length = VariableValueSize-1;
   VariableValueDsc.dsc$a_pointer = VariableValue;

   if (VMSok (status =
       lib$get_symbol (&VariableNameDsc, &VariableValueDsc, &Length, 0)))
      VariableValue[Length] = '\0';
   else
      VariableValue[0] = '\0';

   if (status == LIB$_NOSUCHSYM) return (false);
   if (VMSnok (status))
   {
      if (Debug) fprintf (stdout, "lib$get_symbol() %%X%08.08X\n", status);
      exit (status);
   }
   if (Debug) fprintf (stdout, "|%s|\n", VariableValue);
   return (true);
}

/*****************************************************************************/
/*
*/ 

SetGlobalSymbol
(
char *SymbolName,
char *SymbolValue
)
{
   static int  GlobalSymbol = LIB$K_CLI_GLOBAL_SYM;
   static $DESCRIPTOR (SymbolNameDsc, "");
   static $DESCRIPTOR (SymbolValueDsc, "");

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "SetGlobalSymbol() |%s|%s|\n",
               SymbolName, SymbolValue);

   SymbolNameDsc.dsc$w_length = strlen(SymbolName);
   SymbolNameDsc.dsc$a_pointer = SymbolName;
   SymbolValueDsc.dsc$w_length = strlen(SymbolValue);
   SymbolValueDsc.dsc$a_pointer = SymbolValue;

   status = lib$set_symbol (&SymbolNameDsc, &SymbolValueDsc, &GlobalSymbol);
   if (VMSnok (status))
   {
      ErrorVmsStatus (status, SymbolName, SymbolName, __FILE__, __LINE__);
      exit (STS$K_ERROR | STS$M_INHIB_MSG);
   }
}

/*****************************************************************************/
/*
*/

ErrorAuthentication
(
char *SourceFileName,
int SourceLineNumber
)
{
   register char  *cptr;

   /*********/
   /* begin */
   /*********/

   /* 
      The source file format provided by the "__FILE__" macro will be
      "device:[directory]name.type;ver".  Reduce that to "name.type".
   */
   for (cptr = SourceFileName; *cptr && *cptr != ';'; cptr++);
   if (*cptr)
   {
      while (*cptr != '.') cptr--;
      *cptr-- = '\0';
   }
   while (*cptr != ']') cptr--;
   cptr++;

   fprintf (HttpOut,
"HTTP/1.0 401 Authentication failure.\n\
WWW-Authenticate: Basic realm=\"%s\"\n\
Content-Type: text/html\n\
\n\
<!-- SoftwareID: %s Module: %s Line: %d -->\n\
<H1>ERROR!</H1>\n\
<P>Reported by server.\n\
<P>Authentication failure.\n",
   AuthenticateRealm, SoftwareID, cptr, SourceLineNumber);
}

/*****************************************************************************/
/*
*/

ErrorGeneral
(
char *Text,
char *SourceFileName,
int SourceLineNumber
)
{
   register char  *cptr;

   /*********/
   /* begin */
   /*********/

   /* 
      The source file format provided by the "__FILE__" macro will be
      "device:[directory]name.type;ver".  Reduce that to "name.type".
   */
   for (cptr = SourceFileName; *cptr && *cptr != ';'; cptr++);
   if (*cptr)
   {
      while (*cptr != '.') cptr--;
      *cptr-- = '\0';
   }
   while (*cptr != ']') cptr--;
   cptr++;

   if (!HttpHasBeenOutput) fputs (Http404Header, HttpOut);
   fprintf (HttpOut,
"<!-- SoftwareID: %s Module: %s Line: %d -->\n\
<H1>ERROR!</H1>\n\
<P>Reported by server.\n\
<P>%s\n",
   SoftwareID, cptr, SourceLineNumber, Text);
}

/*****************************************************************************/
/*
*/

ErrorVmsStatus
(
int StatusValue,
char *Text,
char *HiddenText,
char *SourceFileName,
int SourceLineNumber
)
{
   static char  Message [256];
   static $DESCRIPTOR (MessageDsc, Message);

   register char  *cptr;
   int  status;
   short int  Length;

   /*********/
   /* begin */
   /*********/

   if (VMSok (status = sys$getmsg (StatusValue, &Length, &MessageDsc, 1, 0))) 
   {
      Message[Length] = '\0';
      Message[0] = toupper(Message[0]);
   }
   else
      exit (status);

   /* 
      The source file format provided by the "__FILE__" macro will be
      "device:[directory]name.type;ver".  Reduce that to "name.type".
   */
   for (cptr = SourceFileName; *cptr && *cptr != ';'; cptr++);
   if (*cptr)
   {
      while (*cptr != '.') cptr--;
      *cptr-- = '\0';
   }
   while (*cptr != ']') cptr--;
   cptr++;

   if (!HttpHasBeenOutput) fputs (Http404Header, HttpOut);
   fprintf (HttpOut,
"<!-- SoftwareID: %s Module: %s Line: %d -->\n\
<H1>ERROR!</H1>\n\
<P>Reported by server.\n\
<P>%s ... <TT>%s</TT>\n\
<!-- %%X%08.08X \"%s\" -->\n",
   SoftwareID, cptr, SourceLineNumber, Message, Text, StatusValue, HiddenText);
}

/****************************************************************************/
/*
Does a case-insensitive, character-by-character string compare and returns 
true if two strings are the same, or false if not.  If a maximum number of 
characters are specified only those will be compared, if the entire strings 
should be compared then specify the number of characters as 0.
*/ 

boolean strsame
(
register char *sptr1,
register char *sptr2,
register int  count
)
{
   while (*sptr1 && *sptr2)
   {
      if (toupper (*sptr1++) != toupper (*sptr2++)) return (false);
      if (count)
         if (!--count) return (true);
   }
   if (*sptr1 || *sptr2)
      return (false);
   else
      return (true);
}

/*****************************************************************************/

