/*****************************************************************************/
/*
                              PostUrl.c

A module to decode and store MIME content-type "x-url-encoded" HTTP POST 
method requests.  Designed to be a self-contained object module that can 
either reside centrally in the software directory or locally to a program that 
uses it.  Uses VMS run-time library routines to store URL form names and 
corresponding values in a binary tree for quick lookup. 

Relevant, callable functions: 

   int PostParseHttpStream (int);

   Call once with parameter set to the numeric equivalent of the CGI
   'Content-Length' to parse the HTTP stream.

   char* PostGetFormValue (char*, int, char*);

   Call for each URL-encoded form element name to be returned.  The first
   parameter is the name of the form element (not case-sensitive).  Second
   is the length of that name.  Third is a pointer to the string to be
   returned if the name is not found (usually a pointer to "").  A pointer
   to the name's value is returned.

This module expects the functions ErrorVmsStatus() and ErrorGeneral() be 
callable.  Examples of these may be found in many HFRD VMS CGI scripts.


BUILD_PROCEDURE
---------------
See BUILD_POSTURL.COM


VERSION HISTORY 
---------------
19-FEB-96  MGD  initial development
*/ 
/*****************************************************************************/

#include <stdio.h>
#include <errno.h>
#include <file.h>

#include <libdef.h>
#include <ssdef.h>
#include <stsdef.h>

#define boolean int
#define true 1
#define false 0

#define VMSok(x) ((x) & STS$M_SUCCESS)
#define VMSnok(x) (!((x) & STS$M_SUCCESS))

/**************************/
/* structure declarations */
/**************************/

struct PostRecordStruct
{
   struct PostRecordStruct  *LeftLink;
   struct PostRecordStruct  *RightLink;
   unsigned short  Reserved;
   char *NamePtr;
   int NameLength;
   char *ValuePtr;
   int ValueLength;
   /* name and value strings are stored here! */
};

/******************/
/* global storage */
/******************/

boolean  IgnoreDuplicates = false;
char  ErrorUrlDuplicate [] = "Duplicate form element name.",
      ErrorUrlEncoding [] = "Problem with client's URL encoding.";
PostRecordStructOverhead = sizeof(struct PostRecordStruct);
struct PostRecordStruct  *PostTreeHead = NULL;

/********************/
/* external storage */
/********************/

extern boolean  Debug;
extern char  SoftwareID[];
extern char  Utility[];

/***********************/
/* function prototypes */
/***********************/

int PostTreeAddNode (char*, int, char*, int);
char* PostGetFormValue (char*, int, char*);
int PostParseHttpStream (int);
int PostTreeAllocateRecord (struct PostRecordStruct*,
                            struct PostRecordStruct**,
                            unsigned long);
int PostTreeCompareNodes (struct PostRecordStruct*,
                          struct PostRecordStruct*,
                          unsigned long);

/**********************/
/* external functions */
/**********************/

int ErrorGeneral (int, char*, char*);
int ErrorVmsStatus (int, char*, char*, char*, char*);

/*****************************************************************************/
/*
Open the raw HTTP stream.  Read that stream skipping over the HTTP header to 
the body.  Parse the URL-encoded body into form element names and values, 
storing them in a binary tree using VMS LIB binary tree routines.  Close the 
stream after "Content-Length:" bytes.
*/ 

boolean PostParseHttpStream (register int ContentLength)

{
   register int  ccnt, cblcnt, vcnt;
   register char  *cptr, *sptr, *zptr;

   boolean  InName;
   int  FormElementCount,
        HexEncodeCount,
        NameLength,
        ValueSize;
   char  *ValuePtr;
   char  HttpRecord [1024],
         Name [256]; 
   int  status,
        HttpInFd;
   
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "PostParseHttpStream()\n");

   /* open the HTTP input stream */
#ifdef __DECC
   if ((HttpInFd = open ("HTTP$INPUT", O_RDONLY, 0, "ctx=bin")) == -1)
      exit (vaxc$errno);
#else
   if ((HttpInFd = open ("HTTP$INPUT", O_RDONLY, 0, "rfm=udf")) == -1)
      exit (vaxc$errno);
#endif

   InName = true;
   zptr = (sptr = Name) + sizeof(Name) - 1;
   FormElementCount = HexEncodeCount = ValueSize = cblcnt = vcnt = 0;
   ValuePtr = NULL;

   while (ContentLength)
   {
      if (Debug) fprintf (stdout, "ContentLength: %d\n", ContentLength);

      if ((ccnt = read (HttpInFd, HttpRecord, sizeof(HttpRecord)-1)) <= 0)
         break;
      if (Debug)
      {
         HttpRecord[ccnt] = '\0';
         fprintf (stdout, "%d|%d|%s|\n", ccnt, ContentLength, HttpRecord);
      }

      cptr = HttpRecord;
      if (cblcnt < 2)
      {
         /******************/
         /* in HTTP header */
         /******************/

         while (ccnt && cblcnt < 2)
         {
            /** if (Debug) fprintf (stdout, "cptr |%s|\n", cptr); **/
            if (*cptr == '\n')
               cblcnt++;
            else
            if (*cptr != '\r')
               cblcnt = 0;
            cptr++;
            ccnt--;
         }
         if (!ccnt) continue;
      }

      /****************/
      /* in HTTP body */
      /****************/

      while (ccnt && ContentLength)
      {
         if (InName)
         {
            /*********************/
            /* form element name */
            /*********************/

            while (ccnt && ContentLength && *cptr != '=')
            {
               /** if (Debug) fprintf (stdout, "cptr |%s|\n", cptr); **/
               if (*cptr == '\n' || *cptr == '\r')
               {
                  cptr++;
                  ccnt--;
                  ContentLength--;
                  continue;
               }
               if (sptr >= zptr)
                  exit (ErrorVmsStatus (SS$_BUFFEROVF, "URL form name", "",
                        __FILE__, __LINE__));
               *sptr++ = *cptr++;
               ccnt--;
               ContentLength--;
            }
            if (ccnt && ContentLength && *cptr == '=')
            {
               cptr++;
               ccnt--;
               ContentLength--;
               *sptr = '\0';
               NameLength = sptr - Name;
               InName = false;
               /* initialize the value storage */
               if ((sptr = ValuePtr) == NULL)
                  zptr = sptr;
               else
                  zptr = sptr + ValueSize - 1;
               vcnt = 0;
            }
            if (!ccnt) continue;
         }

         while (ccnt && ContentLength && *cptr != '&')
         {
            /*********************/
            /* form element body */
            /*********************/

            /** if (Debug) fprintf (stdout, "cptr |%s|\n", cptr); **/

            if (HexEncodeCount)
            {
               if (HexEncodeCount == 2)
               {
                  if (*cptr >= '0' && *cptr <= '9')
                     *sptr = ((*(unsigned char*)cptr - '0') << 4);
                  else
                  if (toupper(*cptr) >= 'A' && toupper(*cptr) <= 'F')
                     *sptr = ((*(unsigned char*)cptr - '7') << 4);
                  else
                     exit (ErrorGeneral (ErrorUrlEncoding, __FILE__, __LINE__));

                  HexEncodeCount--;
                  cptr++;
                  ccnt--;
                  ContentLength--;
                  continue;
               }
               if (HexEncodeCount == 1)
               {
                  if (*cptr >= '0' && *cptr <= '9')
                     *sptr++ |= (*(unsigned char*)cptr - '0');
                  else
                  if (toupper(*cptr) >= 'A' && toupper(*cptr) <= 'F')
                     *sptr++ |= (*(unsigned char*)cptr - '7');
                  else
                     exit (ErrorGeneral (ErrorUrlEncoding, __FILE__, __LINE__));

                  HexEncodeCount--;
                  cptr++;
                  ccnt--;
                  ContentLength--;
                  continue;
               }
            }

            if (*cptr == '\n' || *cptr == '\r')
            {
               cptr++;
               ccnt--;
               ContentLength--;
               continue;
            }

            if (sptr >= zptr)
            {
               ValueSize += 1024;
               if ((ValuePtr = realloc (ValuePtr, ValueSize)) == NULL)
                  exit (ErrorVmsStatus (vaxc$errno, "realloc()", "",
                        __FILE__, __LINE__));
               /* recalculate the pointer in case realloc() moved storage */
               sptr = ValuePtr + vcnt;
               /* allow room for a null without needing to realloc() */
               zptr = ValuePtr + ValueSize - 1;
            }

            if (*cptr == '+')
            {
               *sptr++ = ' ';
               cptr++;
            }
            else
            if (*cptr == '%')
            {
               HexEncodeCount = 2;
               cptr++;
            }
            else
               *sptr++ = *cptr++;

            ccnt--;
            ContentLength--;
         }

         if (ccnt && ContentLength && *cptr == '&')
         {
            cptr++;
            ccnt--;
            ContentLength--;
            *sptr = '\0';
            PostTreeAddNode (Name, NameLength, ValuePtr, sptr - ValuePtr);
            FormElementCount++;
            InName = true;
            /* initialize the name storage */
            zptr = (sptr = Name) + sizeof(Name) - 1;
         }
      }
   }

   if (InName)
      exit (ErrorGeneral (ErrorUrlEncoding, __FILE__, __LINE__));

   if (!ContentLength && sptr > ValuePtr)
   {
      *sptr = '\0';
      PostTreeAddNode (Name, NameLength, ValuePtr, sptr - ValuePtr);
      FormElementCount++;
   }

   close (HttpInFd);
   if (ValuePtr != NULL) free (ValuePtr);

   if (!FormElementCount)
      exit (ErrorGeneral (ErrorUrlEncoding, __FILE__, __LINE__));
}

/*****************************************************************************/
/*
Find a node in the VMS run-time library binary tree.  Return a pointer to a 
string which is either the value corresponding to the name or the address of 
the 'NotFoundPtr' (usually "" or NULL).
*/ 

char* PostGetFormValue
(
char *Name,
int NameLength,
char *NotFoundPtr
)
{
   int  status;
   struct PostRecordStruct  PostRecord;
   struct PostRecordStruct  *TreeNodePtr;
   
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "PostGetFormValue() |%s|\n", Name);

   PostRecord.NamePtr = Name;
   if (NameLength > 0)
      PostRecord.NameLength = NameLength;
   else
      PostRecord.NameLength = strlen(Name);

   /*********************/
   /* lookup the record */
   /*********************/

   status = lib$lookup_tree (&PostTreeHead, &PostRecord,
                             &PostTreeCompareNodes, &TreeNodePtr);
   if (Debug) fprintf (stdout, "lib$lookup_tree() %%X%08.08X\n", status);

   if (VMSok (status))
      return (TreeNodePtr->ValuePtr);
   else
   if (status == LIB$_KEYNOTFOU)
      return (NotFoundPtr);

   exit (ErrorVmsStatus (status, "", "", __FILE__, __LINE__));
}

/*****************************************************************************/
/*
Add a node into the VMS run-time library binary tree.
*/ 

boolean PostTreeAddNode
(
char *Name,
int NameLength,
char *Value,
int ValueLength
)
{
   static int  NoDuplicatesFlag = 0;

   register char  *cptr, *sptr, *zptr;

   int  status;
   struct PostRecordStruct  PostRecord;
   struct PostRecordStruct  *PostRecordPtr,
                            *TreeNodePtr;
   
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "PostTreeAddNode() |%s|%s|\n", Name, Value);

   if ((PostRecordPtr =
        malloc (PostRecordStructOverhead + NameLength + ValueLength + 2))
       == NULL)
     exit (ErrorVmsStatus (vaxc$errno, "malloc()", "", __FILE__, __LINE__));
   
   PostRecordPtr->NamePtr = (char*)PostRecordPtr + PostRecordStructOverhead;
   memcpy (PostRecordPtr->NamePtr, Name,
           (PostRecordPtr->NameLength = NameLength) + 1);

   PostRecordPtr->ValuePtr = (char*)PostRecordPtr + PostRecordStructOverhead +
                             NameLength + 1;
   memcpy (PostRecordPtr->ValuePtr, Value,
           (PostRecordPtr->ValueLength = ValueLength) + 1);

  if (VMSnok (status =
      lib$insert_tree (&PostTreeHead, PostRecordPtr,
                       &NoDuplicatesFlag, &PostTreeCompareNodes,
                       &PostTreeAllocateRecord, &TreeNodePtr, 0)))
     exit (ErrorVmsStatus (status, "inserting into binary tree", "",
                           __FILE__, __LINE__));
  if (Debug) fprintf (stdout, "lib$insert_tree() %%X%08.08X\n", status);

  if (IgnoreDuplicates) return;

  if (status == LIB$_KEYALRINS)
     exit (ErrorGeneral (ErrorUrlDuplicate, __FILE__, __LINE__));
}

/*****************************************************************************/
/*
Called by lib$insert_tree() and lib$lookup_tree().  Returns negative number if
user node is lexicographically less than the tree node, positive if greater,
or zero if exactly the same.
*/ 

PostTreeCompareNodes
(
struct PostRecordStruct *UserNodePtr,
struct PostRecordStruct *TreeNodePtr,
unsigned long Unused
)
{
   
   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "PostTreeCompareNodes() |%s|%s|\n",
               UserNodePtr->NamePtr, TreeNodePtr->NamePtr);

   if (UserNodePtr->NameLength != TreeNodePtr->NameLength)
      return (UserNodePtr->NameLength - TreeNodePtr->NameLength);

   {
      /* local storage */
      register char  *sptr1, *sptr2;

      sptr1 = UserNodePtr->NamePtr;
      sptr2 = TreeNodePtr->NamePtr;
      while (*sptr1 && *sptr2)
      {
         if (toupper (*sptr1) != toupper (*sptr2))
            return (toupper (*sptr1) - toupper (*sptr2));
         sptr1++;
         sptr2++;
      }
      if (*sptr1)
         return (1);
      else
      if (*sptr2)
         return (-1);
      else
         return (0);
   }
}

/*****************************************************************************/
/*
Called by lib$insert_tree().  Insert the address in 'UserNodePtr' into the
location pointed at by 'NewTreeNodePtrAddress' (inside the binary tree
structure presumably!)
*/ 

PostTreeAllocateRecord
(
struct PostRecordStruct *UserNodePtr,
struct PostRecordStruct **NewTreeNodePtrAddress,
unsigned long Unused
)
{
   if (Debug) fprintf (stdout, "PostTreeAllocateRecord()\n");

   *NewTreeNodePtrAddress = UserNodePtr;

   return (SS$_NORMAL);
}

/****************************************************************************/

