/*****************************************************************************/
/*
                                PostMail.c

A CGI-compliant script.

The HTTP GET method is used to obtain the fill-out form for submitting a 
message. 

The HTTP POST method is used to submit that form's fields for mailing.  All 
information is retrieved form the "application/x-www-form-urlencoded" contents 
of the request body via the HTTP$INPUT stream available to the POST script 
(this program). 

The AUTHENTICATED originating user name is obtained from the "WWW_REMOTE_USER" 
CGI variable.

The message is posted via the VMS callable mail interface.  One kludge is 
necessary to make the message appear to originate from the authenticated 
remote user.  The callable mail interface allows alteration of the "From:" 
header using SYSPRV.  Unfortunately (and annoyingly) this does not seem to 
propagate via mail transports (e.g. "MX%"), these seem to only use the 
username of the originating process.  This utility sets the originating 
process' username to that of the authenticated remote username using a short 
piece of kernel-mode macro (plagiarized from the "BECOME" utility) called 
before the mail interface is invoked.  It is necessary to reset the username 
to the original after completion as this (certainly unsupported) mechanism 
seems to change the process header of the parent, not script subprocess. 

The form fields have a role in both the GET of the mail form and the POST of
the message mailing.  In the former the field contents specifiy the defaults of
the form fields.  Setting the 'fixed' field to "yes" causes all supplied fields
to be fixed (i.e. unchangeable).  In the latter (POST) the form fields of
course provide the details for mailing the message.


FORM FIELDS
-----------
body=           body of message (or via 'message')
deform=         like "mailto", but data stream is escaped and de-"<form>"ed
fixed=          if non-null makes supplied text in mail form unalterable
help=           if non-null cause a redirection header to the help URL
list=           URL format specification for distribution list file
mailto=         the rest of the POST data stream is the unescaped message body
message=        URL format specification for the message file (or via 'body')
sc=             if non-null prevents message delivery (i.e. safety catch)
subject=        synopsis of message
textarea=       specifies the colums and rows of the body field (CCxRR)
to=             destination address(es) (if '*' and 'list' then all of list)


QUALIFIERS
----------
/DBUG           turns on all "if (Debug)" statements
/ICON=          URL of utility icon (optional)
/SMTP=          SMTP transport (e.g. "MX", "IN")


INSTALL DETAILS
---------------
INSTALL ADD directory:POSTMAIL.EXE /PRIVILEGE=CMKRNL


BUILD DETAILS
-------------
See BUILD_POSTMAIL.COM procedure.


VERSION HISTORY (update SoftwareID as well!)
---------------
21-NOV-95  MGD  v1.2.0, changed reporting in PostAddAddress() to incorporate
                        MAIL signalled messages, for more extensive information
24-MAY-95  MGD  v1.1.0, minor changes for AXP compatibility
03-APR-95  MGD  v1.0.0, initial development
*/
/*****************************************************************************/

#ifdef __ALPHA
   char SoftwareID [] = "POSTMAIL AXP-1.2.0";
#else
   char SoftwareID [] = "POSTMAIL VAX-1.2.0";
#endif

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <errno.h>
#include <unixio.h>

/* VMS-related header files */
#include <descrip.h>
#include <jpidef.h>
#include <libdef.h>
#include <maildef.h>
#include <rms.h>
#include <ssdef.h>
#include <stsdef.h>

#ifdef __ALPHA
#   pragma nomember_alignment
#endif

#define boolean int
#define true 1
#define false 0

#define VMSok(x) ((x) & STS$M_SUCCESS)
#define VMSnok(x) (!((x) & STS$M_SUCCESS))
 
#define HttpBufferSize 4096
#define FormBodySize 16384

char  Http200Header [] =
"HTTP/1.0 200 Document follows.\r\n\
Content-Type: text/html\r\n\
\r\n";

char  Http302Header [] =
"HTTP/1.0 302 Redirection.\r\n\
Location: http://%s:%s%s\r\n\
\r\n";

char  Http404Header [] =
"HTTP/1.0 404 Error report follows.\r\n\
Content-Type: text/html\r\n\
\r\n";

char  ErrorStringSize [] = "String size too small.";

char  Utility [] = "POSTMAIL";

struct AnExitHandler
{
   unsigned long  Flink;
   unsigned long  HandlerAddress;
   unsigned long  ArgCount;
   unsigned long  ExitStatus;
};

boolean  Debug,
         HttpHasBeenOutput;

int  ContentCount,
     ContentLength,
     ExitStatus,
     SendFailedCount,
     SendOkCount;
     
char  CgiContentLength [16],
      CgiContentType [256],
      CgiRequestMethod [16],
      CgiRemoteUser [16],
      CgiScriptName [256],
      CgiServerName [128],
      CgiServerPort [128],
      FormBody [FormBodySize],
      FormFixed [16],
      FormHelp [16],
      FormList [256],
      FormMessage [256],
      FormSubject [256],
      FormSafetyCatch [16],
      FormTo [4096],
      FormTextArea [16],
      HelpUrl [256],
      Icon [256],
      IconImage [256],
      JpiUserName [13],
      SmtpTransport [256];

FILE  *HttpIn,
      *HttpOut;

char* CopyToHtml (char*, char*, int);

struct AnExitHandler  ExitHandler;

/* required function prototypes */
int GetHttpString (char, char*, int, boolean);
extern int SetUserName (char*);
ExitViaTelephoneBooth ();
char* MapUrl (char*, char*, char*, char*);

/*****************************************************************************/
/*
*/

main (int argc, char *argv[])

{
   register int  acnt;
   register char  *cptr, *sptr;

   int  status;
   char  String [256];

   /*********/
   /* begin */
   /*********/

   /* open another output stream so that the "\n" are not filtered */
#ifdef __DECC
   if ((HttpOut = fopen ("SYS$OUTPUT", "w", "ctx=bin")) == NULL)
      exit (vaxc$errno);
#else
   if ((HttpOut = fopen ("SYS$OUTPUT", "w", "rfm=udf")) == NULL)
      exit (vaxc$errno);
#endif

   /***********************************/
   /* get the command line parameters */
   /***********************************/

   /* using this mechanism parameters must be space-separated! */
   for (acnt = 1; acnt < argc; acnt++)
   {
      if (Debug) fprintf (stdout, "argv[%d] |%s|\n", acnt, argv[acnt]);
      if (strsame (argv[acnt], "/DBUG", -1))
      {
         Debug = true;
         continue;
      }
      if (strsame (argv[acnt], "/HELP=", 5))
      {
         for (cptr = argv[acnt]+5; *cptr && *cptr != '='; cptr++);
         if (*cptr) cptr++;
         sptr = HelpUrl;
         while (*cptr) *sptr++ = *cptr++;
         *sptr = '\0';
         continue;
      }
      if (strsame (argv[acnt], "/ICON=", 5))
      {
         for (cptr = argv[acnt]+5; *cptr && *cptr != '='; cptr++);
         if (*cptr) cptr++;
         sptr = Icon;
         while (*cptr) *sptr++ = *cptr++;
         *sptr = '\0';
         sprintf (IconImage, "<IMG ALIGN=bottom SRC=\"%s\" ALT=\"\"> ", Icon);
         continue;
      }
      if (strsame (argv[acnt], "/SMTP=", 5))
      {
         for (cptr = argv[acnt]+5; *cptr && *cptr != '='; cptr++);
         if (*cptr) cptr++;
         sptr = SmtpTransport;
         while (*cptr) *sptr++ = toupper(*cptr++);
         *sptr = '\0';
         continue;
      }
      fprintf (stdout, "%%%s-E-IVQUAL, unrecognized qualifier\n \\%s\\\n",
               Utility, argv[acnt]+1);
      exit (STS$K_ERROR | STS$M_INHIB_MSG);
   }

   /*****************************/
   /* get general CGI variables */
   /*****************************/

   GetCgiVariable ("WWW_REMOTE_USER", CgiRemoteUser, sizeof(CgiRemoteUser));
   GetCgiVariable ("WWW_REQUEST_METHOD", CgiRequestMethod,
                   sizeof(CgiRequestMethod));

   /* numeric equivalent of "GET\0" (just a bit more efficient, that's all!) */
   if (*(unsigned long*)CgiRequestMethod == 0x00544547)
   {
      /**************/
      /* GET method */
      /**************/

      /* get method specific CGI variables */
      GetCgiVariable ("WWW_SCRIPT_NAME", CgiScriptName, sizeof(CgiScriptName));
      GetCgiVariable ("WWW_FORM_BODY", FormBody, sizeof(FormBody));
      GetCgiVariable ("WWW_FORM_FIXED", FormFixed, sizeof(FormFixed));
      GetCgiVariable ("WWW_FORM_HELP", FormHelp, sizeof(FormHelp));
      GetCgiVariable ("WWW_FORM_LIST", FormList, sizeof(FormList));
      GetCgiVariable ("WWW_FORM_MESSAGE", FormMessage, sizeof(FormMessage));
      GetCgiVariable ("WWW_FORM_TEXTAREA", FormTextArea, sizeof(FormTextArea));
      GetCgiVariable ("WWW_FORM_SC", FormSafetyCatch, sizeof(FormSafetyCatch));
      GetCgiVariable ("WWW_FORM_TO", FormTo, sizeof(FormTo));
      GetCgiVariable ("WWW_FORM_SUBJECT", FormSubject, sizeof(FormSubject));

      if (FormHelp[0])
      {
         GetCgiVariable ("WWW_SERVER_NAME", CgiServerName,
                         sizeof(CgiServerName));
         GetCgiVariable ("WWW_SERVER_PORT", CgiServerPort,
                         sizeof(CgiServerPort));
         fprintf (HttpOut, Http302Header,
                  CgiServerName, CgiServerPort, HelpUrl);
         exit (SS$_NORMAL);
      }

      MessageForm ();

      exit (SS$_NORMAL);
   }

   /***************/
   /* POST method */
   /***************/

   /* numeric equivalent of "POST" (just a bit more efficient, that's all!) */
   if (*(unsigned long*)CgiRequestMethod != 0x54534F50) exit (SS$_NORMAL);

   /* get method specific CGI variables */
   GetCgiVariable ("WWW_CONTENT_LENGTH", CgiContentLength,
                   sizeof(CgiContentLength));
   GetCgiVariable ("WWW_CONTENT_TYPE", CgiContentType, sizeof(CgiContentType));

   if (!strsame (CgiContentType, "application/x-www-form-urlencoded", -1) &&
       !strsame (CgiContentType, "text/plain", -1))
   {
      sprintf (String, "POST Content-Type: \"<TT>%s</TT>\" not supported.",
               CgiContentType);
      ErrorGeneral (String, __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }

   /* open the HTTP input stream */
   if ((HttpIn = fopen ("HTTP$INPUT", "r")) == NULL)
      exit (vaxc$errno);

   PostProcessHeader ();

   ContentLength = atoi (CgiContentLength);
   PostProcessBody ();

   PostCheckParameters ();

   PostMessage (CgiRemoteUser, FormTo, FormSubject, FormBody,
                FormList, FormMessage);

   exit (SS$_NORMAL);
}

/*****************************************************************************/
/*
Generate an HTML form to accept a mail message.  This function is called when 
using the HTTP GET method.
*/ 

MessageForm ()

{
   register int  cnt;
   register char  *cptr, *sptr;

   int  status,
        CharCount,
        Cols,
        Rows,
        SelectorSize;
   char  HtmlSubject [256],
         HtmlBody [4096],
         HtmlString [1024],
         HtmlTo [256],
         ListFileName [256],
         MessageFileName [256],
         String [1024];
   FILE  *ListFile,
         *MessageFile;
         
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MessageForm()\n");

   /* if the remote user name is not present (not authenticated) then fail */
   if (!CgiRemoteUser[0])
   {
      ErrorAuthentication (__FILE__, __LINE__);
      exit (SS$_NORMAL);
   }

   if (FormMessage[0])
   {
      /****************/
      /* message file */
      /****************/

      sptr = MapUrl (FormMessage, MessageFileName, NULL, NULL);
      if (!*sptr)
      {
         ErrorGeneral (sptr+1, __FILE__, __LINE__);
         exit (SS$_NORMAL);
      }

      if ((MessageFile = fopen (MessageFileName, "r")) == NULL)
      {
         status = vaxc$errno;
         ErrorVmsStatus (status, FormMessage, MessageFileName,
                         __FILE__, __LINE__);
         exit (SS$_NORMAL);
      }
   }

   if (FormList[0])
   {
      /*********************/
      /* distribution list */
      /*********************/

      sptr = MapUrl (FormList, ListFileName, NULL, NULL);
      if (!*sptr)
      {
         ErrorGeneral (sptr+1, __FILE__, __LINE__);
         exit (SS$_NORMAL);
      }

      if ((ListFile = fopen (ListFileName, "r")) == NULL)
      {
         status = vaxc$errno;
         ErrorVmsStatus (status, FormList, ListFileName,
                         __FILE__, __LINE__);
         exit (SS$_NORMAL);
      }

      /* count the number of lines containing legitimate strings */
      SelectorSize = 0;
      while (fgets (String, sizeof(String), ListFile) != NULL)
      {
         for (cptr = String; isspace(*cptr); cptr++);
         if (!isprint(*cptr)) continue;
         sptr = cptr;
         while (*cptr && !isspace(*cptr) && isprint(*cptr)) cptr++;
         if (cptr == sptr) continue;
         if (++SelectorSize >= 5) break;
      }
      if (!SelectorSize)
      {
         fclose (ListFile);
         sprintf (String, "Empty distribution list: <TT>%s</TT>. <!-- %s -->",
                  FormList, ListFileName);
         ErrorGeneral (String, __FILE__, __LINE__);
         exit (SS$_NORMAL);
      }
      rewind (ListFile);
   }

   /********************************/
   /* html title, from field, etc. */
   /********************************/

   fprintf (HttpOut,
"%s\
<HTML>\n\
<!-- SoftwareID: %s -->\n\
<TITLE>Post Mail Message</TITLE>\n\
<H1>%sPost Mail Message</H1>\n\
<P><HR>\n\
<FORM METHOD=POST ACTION=\"%s\">\n\
<P>From: <TT>%s</TT>\n",
   Http200Header, SoftwareID, IconImage, CgiScriptName,
   CgiRemoteUser);
   HttpHasBeenOutput = true;

   if (FormList[0])
      fprintf (HttpOut, "<INPUT NAME=list TYPE=hidden VALUE=\"%s\">\n",
               FormList);

   /************/
   /* to field */
   /************/

   if (FormFixed[0] && FormList[0] && FormTo[0] == '*')
   {
      /* unalterable, all members of a distribution list */
      fputs ("<P>To: <TT>", HttpOut);
      CharCount = cnt = 0;
      while (fgets (String, sizeof(String), ListFile) != NULL)
      {
         for (cptr = String; isspace(*cptr); cptr++);
         if (!isprint(*cptr)) continue;
         sptr = cptr;
         while (*cptr && !isspace(*cptr) && isprint(*cptr)) cptr++;
         if (cptr == sptr) continue;
         *cptr = '\0';
         CopyToHtml (HtmlTo, sptr, sizeof(HtmlTo));
         if (CharCount >= 55)
         {
            fputs (",\n", HttpOut);
            CharCount = 0;
         }
         if (!cnt++ || !CharCount)
            CharCount += fprintf (HttpOut, "%s", HtmlTo);
         else
            CharCount += fprintf (HttpOut, ",%s", HtmlTo);
      }
      fclose (ListFile);
      fprintf (HttpOut, "</TT>\n<INPUT NAME=to TYPE=hidden VALUE=\"%s\">\n",
               FormTo);
   }
   else
   if (FormList[0] && FormTo[0] == '*')
   {
      /* all members of a distribution list in a field */
      fputs ("<P>To: <INPUT NAME=to TYPE=text SIZE=48 MAXIMUM=4096 VALUE=\"",
             HttpOut);
      cnt = 0;
      while (fgets (String, sizeof(String), ListFile) != NULL)
      {
         for (cptr = String; isspace(*cptr); cptr++);
         if (!isprint(*cptr)) continue;
         sptr = cptr;
         while (*cptr && !isspace(*cptr) && isprint(*cptr)) cptr++;
         if (cptr == sptr) continue;
         *cptr = '\0';
         CopyToHtml (HtmlTo, sptr, sizeof(HtmlTo));
         if (!cnt++)
            fputs (HtmlTo, HttpOut);
         else
            fprintf (HttpOut, ",%s", HtmlTo);
      }
      fclose (ListFile);
      fputs ("\">\n", HttpOut);
   }
   else
   if (FormFixed[0] && FormTo[0])
   {
      /* unalterable destination */
      CopyToHtml (HtmlTo, FormTo, sizeof(HtmlTo));
      fprintf (HttpOut,
"<P>To: <TT>%s</TT>\n\
<INPUT NAME=to TYPE=hidden VALUE=\"%s\">\n",
      HtmlTo, FormTo);
   }
   else
   if (FormList[0])
   {
      /* select one (or all) members of a distribution list */
      fprintf (HttpOut,
"<P>To:\n<SELECT NAME=to SIZE=%d>\n\
<OPTION VALUE=\"*\" SELECTED>\"all on list\"\n",
      SelectorSize);
      while (fgets (String, sizeof(String), ListFile) != NULL)
      {
         for (cptr = String; isspace(*cptr); cptr++);
         if (!isprint(*cptr)) continue;
         sptr = cptr;
         while (*cptr && !isspace(*cptr) && isprint(*cptr)) cptr++;
         if (cptr == sptr) continue;
         *cptr = '\0';
         CopyToHtml (HtmlTo, sptr, sizeof(HtmlTo));
         fprintf (HttpOut, "<OPTION VALUE=\"%s\">%s\n", sptr, HtmlTo);
      }
      fclose (ListFile);
      fputs ("</SELECT>\n", HttpOut);
   }
   else
   {
      /* field to allow entry of destination address(es) */
      fprintf (HttpOut,
"<P>To: <INPUT NAME=to TYPE=text SIZE=48 MAXIMUM=255 VALUE=\"%s\">\n",
      FormTo);
   }

   /*****************/
   /* subject field */
   /*****************/

   if (FormFixed[0] && FormSubject[0])
   {
      /* unalterable subject */
      CopyToHtml (HtmlSubject, FormSubject, sizeof(HtmlSubject));
      /* "hidden" fields don't seem capable of handling quotes in any form! */
      for (cptr = FormSubject; *cptr; cptr++)
         if (*cptr == '\"') *cptr = '\'';
      fprintf (HttpOut,
"<P>Subject: <TT>%s</TT>\n\
<INPUT NAME=subject TYPE=hidden VALUE=\"%s\">\n",
      HtmlSubject, FormSubject);
   }
   else
   {
      /* field to allow entry of subject */
      fprintf (HttpOut,
"<P>Subject: <INPUT NAME=subject TYPE=text SIZE=48 MAXIMUM=255 VALUE=\"%s\">\n",
      FormSubject);
   }

   /*************/
   /* text area */
   /*************/

   if (FormTextArea[0])
   {
      Cols = atoi(FormTextArea);
      Rows = atoi(FormTextArea+3);
   }
   else
      Cols = Rows = 0;
   if (!Cols) Cols = 64;
   if (!Rows) Rows = 8;
   if (Cols > 80)
      Cols = 80;
   else
   if (Cols < 40)
      Cols = 40;
   if (Rows > 64)
      Cols = 64;
   else
   if (Rows < 4)
      Cols = 4;

   if (FormFixed[0] && FormMessage[0]) 
   {
      /* message contents from file as fixed message body  */
      fputs ("<P>Message:\n<P><BLOCKQUOTE><PRE>", HttpOut);
      while (fgets (String, sizeof(String)-3, MessageFile) != NULL)
      {
         /* change newline to cr/lf carriage control */
         for (cptr = String; *cptr && *cptr != '\n'; cptr++);
         *cptr++ = '\r';
         *cptr++ = '\n';
         *cptr = '\0';
         CopyToHtml (HtmlString, String, sizeof(HtmlString));
         fputs (HtmlString, HttpOut);
      }
      fclose (MessageFile);
      fprintf (HttpOut,
"</PRE></BLOCKQUOTE>\n\
<INPUT NAME=message TYPE=hidden VALUE=\"%s\">\n",
      FormMessage);
   }
   else
   if (FormMessage[0]) 
   {
      /* message contents from file into text area */
      fprintf (HttpOut, "<P><TEXTAREA NAME=body ROWS=%d COLS=%d>",
               Rows, Cols);
      while (fgets (String, sizeof(String)-3, MessageFile) != NULL)
      {
         /* change newline to cr/lf carriage control */
         for (cptr = String; *cptr && *cptr != '\n'; cptr++);
         *cptr++ = '\r';
         *cptr++ = '\n';
         *cptr = '\0';
         CopyToHtml (HtmlString, String, sizeof(HtmlString));
         fputs (HtmlString, HttpOut);
      }
      fclose (MessageFile);
      fputs ("</TEXTAREA>\n", HttpOut);
   }
   else
   if (FormFixed[0] && FormBody[0]) 
   {
      /* unalterable message text */
      CopyToHtml (HtmlBody, FormBody, sizeof(HtmlBody));
      /* "hidden" fields don't seem capable of handling quotes in any form! */
      for (cptr = FormBody; *cptr; cptr++)
         if (*cptr == '\"') *cptr = '\'';
      fprintf (HttpOut,
"<P>Message:\n<P><BLOCKQUOTE><PRE>%s</PRE></BLOCKQUOTE>\n\
<INPUT NAME=body TYPE=hidden VALUE=\"%s\">\n",
      HtmlBody, FormBody);
   }
   else
   {
      /* field to allow entry of message text */
      fprintf (HttpOut,
"<P><TEXTAREA NAME=body ROWS=%d COLS=%d>%s</TEXTAREA>\n",
      Rows, Cols, FormBody);
   }

   /***********************/
   /* submit button, etc. */
   /***********************/

   fputs (
"<P><INPUT TYPE=submit VALUE=\"send\">\n\
(<B>to cancel</B> this message use your browser to <B>navigate backwards</B>)\n\
<INPUT TYPE=reset VALUE=\"reset\">\n",
   HttpOut);

   if (FormSafetyCatch[0])
      fputs (
"<BR><I>( safety-catch <INPUT TYPE=checkbox NAME=sc VALUE=\"on\" CHECKED>)</I>\n",
      HttpOut);

   fputs ("</FORM>\n<P><HR>\n", HttpOut);

   if (!(FormTextArea[0] || (FormFixed[0] && (FormBody[0] || FormMessage[0]))))
   {
      /*********************************/
      /* form to change text area size */
      /*********************************/

      fprintf (HttpOut, "<FORM ACTION=\"%s\">\n", CgiScriptName);

      if (FormList[0])
         fprintf (HttpOut, "<INPUT TYPE=hidden NAME=list VALUE=\"%s\">\n",
                  FormList);

      if (FormMessage[0])
         fprintf (HttpOut, "<INPUT TYPE=hidden NAME=message VALUE=\"%s\">\n",
                  FormMessage);

      if (FormTo[0])
         fprintf (HttpOut, "<INPUT TYPE=hidden NAME=to VALUE=\"%s\">\n",
                  FormTo);

      if (FormSubject[0])
         fprintf (HttpOut, "<INPUT TYPE=hidden NAME=subject VALUE=\"%s\">\n",
                  FormSubject); 

      if (FormBody[0])
         fprintf (HttpOut, "<INPUT TYPE=hidden NAME=body VALUE=\"%s\">\n",
                  FormBody);

      if (FormFixed[0])
         fprintf (HttpOut,
                  "<INPUT TYPE=hidden NAME=fixed VALUE=\"%s\">\n",
                  FormFixed);

      fputs (
"<INPUT TYPE=submit VALUE=\"change\">\n\
<I>Changing text area size does not propagate any altered fields!</I>\n\
<SELECT NAME=TextArea>\n\
<OPTION VALUE=\"80x24\" SELECTED>80 x 24\n\
<OPTION VALUE=\"80x12\">80 x 12\n\
<OPTION VALUE=\"80x6\">80 x 6\n\
<OPTION VALUE=\"64x24\">64 x 24\n\
<OPTION VALUE=\"64x12\">64 x 12\n\
<OPTION VALUE=\"64x6\">64 x 6\n\
<OPTION VALUE=\"40x24\">40 x 24\n\
<OPTION VALUE=\"40x12\">40 x 12\n\
<OPTION VALUE=\"40x6\">40 x 6\n\
</SELECT>\n\
</FORM>\n",
      HttpOut);
   }

   if (HelpUrl[0])
   {
      /*************/
      /* help form */
      /*************/

      fprintf (HttpOut,
"<FORM ACTION=\"%s\">\n\
<INPUT TYPE=submit VALUE=\"help\">\n\
</FORM>\n",
      HelpUrl);
   }

   fputs ("<HTML>\n", HttpOut);
}

/*****************************************************************************/
/*
Read (and absorb) the HTTP stream up to the first blank line (end of HTTP 
header).  All required information can be obtained via CGI variables and the 
URL-encoded body.
*/

PostProcessHeader ()

{
   char  HeaderLine [1024],
         String [256];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "PostProcessHeader()\n");

   for (;;)
   {
      if (GetHttpString ('\n', HeaderLine, sizeof(HeaderLine), false)
          == sizeof(HeaderLine))
      {
         sprintf (String,
         "The HTTP header fields can be %d characters maximum.",
         sizeof(HeaderLine)-1);
         ErrorGeneral (String, __FILE__, __LINE__);
         exit (SS$_NORMAL);
      }

      if (!HeaderLine[0])
      {
         if (Debug) fprintf (stdout, "end-of-header\n");
         return;
      }
   }
}

/*****************************************************************************/
/*
The process body should be a URL-format string, possibly broken into multiple 
lines.  Read that string getting the URL-encoded form keywords and the 
associated values.
*/ 

PostProcessBody (char *HttpBufferPtr)

{
   register int  len;

   int  status;
   char  Field [256],
         String [256];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "PostProcessBody()\n");

   ContentCount = 0;
   while (ContentCount < ContentLength)
   {
      String[0] = '\0';
      if (GetHttpString ('=', Field, sizeof(Field), true) == sizeof(Field))
      {
         sprintf (String, "The field names can be %d characters maximum.",
                  sizeof(Field)-1);
         ErrorGeneral (String, __FILE__, __LINE__);
         exit (SS$_NORMAL);
      }

      if (strsame (Field, "body", -1))
      {
         len = GetHttpString ('&', FormBody, sizeof(FormBody), true);
         if (len == sizeof(FormBody))
         {
            sprintf (String,
            "The body of a message can be %d characters maximum.",
            sizeof(FormBody)-1);
         }
      }
      else
      if (strsame (Field, "deform", -1))
      {
         DeformBody ();
         return;
      }
      else
      if (strsame (Field, "list", -1))
         len = GetHttpString ('&', FormList, sizeof(FormList), true);
      else
      if (strsame (Field, "mailto", -1))
      {
         /* do not URL-escape this data stream */
         len = GetHttpString ('\0', FormBody, sizeof(FormBody), false);
         if (len == sizeof(FormBody))
         {
            sprintf (String,
            "The body of a message can be %d characters maximum.",
            sizeof(FormBody)-1);
         }
      }
      else
      if (strsame (Field, "message", -1))
         len = GetHttpString ('&', FormMessage, sizeof(FormMessage), true);
      else
      if (strsame (Field, "subject", -1))
      {
         len = GetHttpString ('&', FormSubject, sizeof(FormSubject), true);
         if (len == sizeof(FormSubject))
         {
            sprintf (String,
            "The subject can be %d characters maximum.",
            sizeof(FormSubject)-1);
         }
      }
      else
      if (strsame (Field, "sc", -1))
         len = GetHttpString ('&', FormSafetyCatch, sizeof(FormSafetyCatch),
                              true);
      else
      if (strsame (Field, "to", -1))
      {
         len = GetHttpString ('&', FormTo, sizeof(FormTo), true);
         if (len == sizeof(FormTo))
         {
            sprintf (String,
            "The destination address(es) can be %d characters maximum.",
            sizeof(FormTo)-1);
         }
      }
      else
         sprintf (String, "Unknown form field: \"<TT>%128.128s</TT>\".", Field);

      if (len < 0) strcpy (String, "HTTP stream terminated unexpectedly.");

      if (String[0])
      {
         ErrorGeneral (String, __FILE__, __LINE__);
         exit (SS$_NORMAL);
      }
   }
}

/*****************************************************************************/
/*
Gets the rest of the HTTP POST data stream, creating a plain text version of 
URL-enconded form data.  The plain text version comprises the field name 
followed by a newline, and form field value followed by a newline.  Muliply 
form name/value pairs are separated by blank (empty) lines.  If the value is 
empty then the field name is immediately followed by a blank line.  A non-
empty value may be a single string, etc., or a series of lines as might be 
returned by a <TEXTAREA> form element.  All value lines are prefixed by a 
single equal symbol (=), allowing blank form lines to be differentiated from 
the blank lines separating multiple form field name/values.
*/ 

DeformBody ()

{
   register int  len;
   register char  *cptr, *iptr, *sptr;

   int  status;
   char  FieldName [256],
         FieldValue [4096],
         String [256],
         ValueIndent [256];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DeformBody()\n");

   /* get the value of the "deform=" field, used to indent value */
   if (GetHttpString ('&', ValueIndent, sizeof(ValueIndent), true)
       == sizeof(ValueIndent))
   {
      sprintf (String, "The field values can be %d characters maximum.",
               sizeof(ValueIndent)-1);
      ErrorGeneral (String, __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }
   if (!ValueIndent[0]) { ValueIndent[0] = '='; ValueIndent[1] = '\0'; }

   sptr = FormBody;
   len = sizeof(FormBody)-1;
   while (ContentCount < ContentLength)
   {
      /* get the next field name from HTTP stream */
      if (GetHttpString ('=', FieldName, sizeof(FieldName), true)
         == sizeof(FieldName))
      {
         sprintf (String, "The field names can be %d characters maximum.",
                  sizeof(FieldName)-1);
         ErrorGeneral (String, __FILE__, __LINE__);
         exit (SS$_NORMAL);
      }

      /* get the corresponding field value from HTTP stream */
      if (GetHttpString ('&', FieldValue, sizeof(FieldValue), true)
          == sizeof(FieldValue))
      {
         sprintf (String, "The field values can be %d characters maximum.",
                  sizeof(FieldValue)-1);
         ErrorGeneral (String, __FILE__, __LINE__);
         exit (SS$_NORMAL);
      }

      if (sptr > FormBody)
      {
         /* i.e. not the first form name to be placed in the body */
         if (!len--) break;
         /* if the previous field value ended in a blank line */
         if (sptr[-1] == '\n')
         {
            /* remove it, no use for trailing blank lines! */
            sptr--;
            len++;
         }
         /* newline after previous, separate next field name by a blank line */
         if (!len--) break;
         *sptr++ = '\n';
         if (!len--) break;
         *sptr++ = '\n';
      }
      /* place the field name into the message body */
      for (cptr = FieldName; *cptr; cptr++)
      {
         if (!len--) break;
         *sptr++ = *cptr;
      }
      if (len < 0) break;
      if (!len--) break;
      *sptr++ = '\n';

      /* place the field value into the message body */
      cptr = FieldValue;
      while (*cptr)
      {
         if (!len--) break;
         iptr = ValueIndent;
         while (*iptr)
         {
            if (!len--) break;
            *sptr++ = *iptr++;
         }
         if (len < 0) break;
         while (*cptr && *cptr != '\n')
         {
            if (!len--) break;
            *sptr++ = *cptr++;
         }
         if (*cptr)
         {
            if (!len--) break;
            *sptr++ = *cptr++;
         }
         else
         {
            if (!len--) break;
            *sptr++ = '\n';
         }
      }
      if (len < 0) break;
   }

   if (len < 0)
   {
      sprintf (String,
      "The <I>deformed</I> message body can be %d characters maximum.",
      sizeof(FormBody)-1);
      ErrorGeneral (String, __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }
   *sptr = '\0';
}

/*****************************************************************************/
/*
Check that all the REQUIRED form fields have been supplied.  Report to client 
if not.
*/ 

PostCheckParameters ()

{
   static char  Problems [] =
     "<B>The following prevented processing ...</B>\n<BR>\n";

   register char  c;
   register char  *sptr;

   char  String [1024];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "PostCheckParameters()\n");

   *(sptr = String) = '\0';
   if (!FormSubject[0])
   {
      if (!String[0])
      { 
         strcpy (sptr, Problems);
         sptr += sizeof(Problems)-1;
      }
      strcpy (sptr, "<BR><I>Subject</I> not supplied.\n");
      while (*sptr) sptr++;
   }
   if (!FormBody[0] && !FormMessage[0])
   {
      if (!String[0])
      { 
         strcpy (sptr, Problems);
         sptr += sizeof(Problems)-1;
      }
      strcpy (sptr, "<BR><I>Body</I> not supplied.\n");
      while (*sptr) sptr++;
   }
   if (!FormTo[0])
   {
      if (!String[0])
      { 
         strcpy (sptr, Problems);
         sptr += sizeof(Problems)-1;
      }
      strcpy (sptr, "<BR><I>To</I> not supplied.\n");
      while (*sptr) sptr++;
   }
   if (String[0])
   {
      ErrorGeneral (String, __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }
}

/*****************************************************************************/
/*
Use the VMS callable mail interface to create and send a VMS mail message.  
This is the main function called when using the HTTP POST method.  'To' can be 
a list of comma-separated addresses, or if an asterisk then the supplied list 
file is opened and all addresses available therein are used.  'Text' is a 
null-terminated string of '\n'-separated lines of plain text.
*/ 

int PostMessage
(
char *FromUserName,
char *To,
char *Subject,
char *Body,
char *ListPath,
char *MessagePath
)
{
   register char  *cptr, *sptr;

   int  status,
        MailStatus;
   unsigned long  SendContext = 0;
   char  ListFileName [256],
         MessageFileName [256],
         String [1024];
   FILE  *MessageFile,
         *ListFile;

   struct {
      short int  buf_len;
      short int  item;
      unsigned char   *buf_addr;
      unsigned short  *ret_len;
   }
   BodyPartItem [] =
   {
      { 0, MAIL$_SEND_RECORD, 0, 0 },
      { 0, MAIL$_NOSIGNAL, 0, 0 },
      {0,0,0,0}
   },
   SubjectItem [] =
   {
      { 0, MAIL$_SEND_SUBJECT, Subject, 0 },
      { 0, MAIL$_NOSIGNAL, 0, 0 },
      {0,0,0,0}
   },
   NoSignalItem [] =
   {
      { 0, MAIL$_NOSIGNAL, 0, 0 },
      {0,0,0,0}
   },
   JpiUserNameItem [] =
   {
      { 12, JPI$_USERNAME, JpiUserName, 0 },
      {0,0,0,0}
   },
   NullItem = {0,0,0,0};

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "PostMessage()\n");

   /* if the remote user name is not present (not authenticated) then fail */
   if (!CgiRemoteUser[0])
   {
      ErrorAuthentication (__FILE__, __LINE__);
      exit (SS$_NORMAL);
   }

   if (MessagePath[0])
   {
      /****************/
      /* message file */
      /****************/

      sptr = MapUrl (MessagePath, MessageFileName, NULL, NULL);
      if (!*sptr)
      {
         ErrorGeneral (sptr+1, __FILE__, __LINE__);
         exit (SS$_NORMAL);
      }

      if ((ListFile = fopen (MessageFileName, "r")) == NULL)
      {
         status = vaxc$errno;
         ErrorVmsStatus (status, MessagePath, MessageFileName,
                         __FILE__, __LINE__);
         exit (SS$_NORMAL);
      }
   }

   if (To[0] == '*')
   {
      /*********************/
      /* distribution list */
      /*********************/

      sptr = MapUrl (ListPath, ListFileName, NULL, NULL);
      if (!*sptr)
      {
         ErrorGeneral (sptr+1, __FILE__, __LINE__);
         exit (SS$_NORMAL);
      }

      if ((ListFile = fopen (ListFileName, "r")) == NULL)
      {
         status = vaxc$errno;
         ErrorVmsStatus (status, ListPath, ListFileName,
                         __FILE__, __LINE__);
         exit (SS$_NORMAL);
      }
   }

   /****************/
   /* start report */
   /****************/

   fprintf (HttpOut,
"Content-Type: text/html\r\n\
\r\n\
<!-- SoftwareID: %s -->\n\
<TITLE>Mailing Report</TITLE>\n\
<H1>%sMailing Report</H1>\n\
<P><HR>\n\
<BLOCKQUOTE>\n",
   SoftwareID, IconImage);
   fflush (HttpOut);
   HttpHasBeenOutput = true;

   /*****************/
   /* begin to mail */
   /*****************/

   TelephoneBooth (FromUserName);

   MailStatus = mail$send_begin (&SendContext, &NullItem, &NullItem);

   if (To[0] == '*')
   {
      /*********************/
      /* distribution list */
      /*********************/

      while (fgets (String, sizeof(String), ListFile) != NULL)
      {
         for (cptr = String; isspace(*cptr); cptr++);
         if (!isprint(*cptr)) continue;
         sptr = cptr;
         while (*cptr && !isspace(*cptr) && isprint(*cptr)) cptr++;
         if (cptr == sptr) continue;
         while (*cptr && *cptr != '\n') cptr++;
         *cptr = '\0';
         PostAddAddress (&SendContext, sptr);
      }
      fclose (ListFile);
   }
   else
   {
      /* a single, or multiple comma-separated addresses */
      cptr = To;
      while (*cptr)
      {
         sptr = cptr;
         while (*cptr && *cptr != ',') cptr++;
         if (*cptr) *cptr++ = '\0';
         PostAddAddress (&SendContext, sptr);
      }
   }

   /***********/
   /* subject */
   /***********/

   SubjectItem[0].buf_len = strlen(Subject);
   if (VMSok (MailStatus))
      MailStatus = mail$send_add_attribute (&SendContext, &SubjectItem,
                                            &NullItem);

   if (VMSok (MailStatus))
   {
      /*******************/
      /* body of message */
      /*******************/

      if (MessagePath[0])
      {
         /* from file */
         while (fgets (String, sizeof(String), MessageFile) != NULL)
         {
            /* trim trailing newline */
            for (cptr = String; *cptr != '\n'; cptr++);
            *cptr = '\0';
            BodyPartItem[0].buf_addr = String;
            BodyPartItem[0].buf_len = cptr - String;
            MailStatus = mail$send_add_bodypart (&SendContext, &BodyPartItem,
                                                 &NullItem);
         }
         fclose (MessageFile);
      }
      else
      {
         /* from form field */
         cptr = Body;
         while (*cptr && VMSok (MailStatus))
         {
            BodyPartItem[0].buf_addr = cptr;
            while (*cptr && *cptr != '\n') cptr++;
            BodyPartItem[0].buf_len = cptr - (char*)BodyPartItem[0].buf_addr;
            if (*cptr) cptr++;
            MailStatus = mail$send_add_bodypart (&SendContext, &BodyPartItem,
                                                 &NullItem);
         }
      }
   }

   /****************/
   /* send message */
   /****************/

   if (!FormSafetyCatch[0])
      if (VMSok (MailStatus))
         MailStatus = mail$send_message (&SendContext, &NoSignalItem,
                                         &NoSignalItem);

   mail$send_end (&SendContext, &NullItem, &NullItem);

   if (VMSnok (MailStatus))
   {
      ErrorVmsStatus (status, "sending message", "" , __FILE__, __LINE__);
      return;
   }

   if (Debug) fprintf (stdout, "MailStatus: %%X%08.08X\n", MailStatus);

   /*******************/
   /* complete report */
   /*******************/

   fputs ("</PRE></BLOCKQUOTE>\n<P><HR>\n", HttpOut);

   if (SendOkCount == 1 && !SendFailedCount)
      fprintf (HttpOut, "<I>Successfully sent.</I>\n");
   else
   if (SendOkCount && !SendFailedCount)
      fprintf (HttpOut, "<I>%d (all) successfully sent.</I>\n", SendOkCount);
   else
   if (SendOkCount)
      fprintf (HttpOut, "<I>%d successfully sent.</I>\n", SendOkCount);

   if (!SendOkCount && SendFailedCount == 1)
      fprintf (HttpOut, "<I>Send failed.</I>");
   else
   if (!SendOkCount && SendFailedCount)
      fprintf (HttpOut, "<I>%d (all) failed.</I>\n", SendFailedCount);
   else
   if (SendFailedCount)
      fprintf (HttpOut, "<I>%d failed.</I>\n", SendFailedCount);

   if (FormSafetyCatch[0])
      fputs (
"<H1>Message not actually sent !</H1>\n\
<B><I>Safety-Catch</I> is on.</B>\n",
      HttpOut);

   fputs ("</HTML>\n", HttpOut);
}

/*****************************************************************************/
/*
Using the VMS callable mail interface add a destination address to the mail 
message.  Called from PostMessage().
*/ 

int PostAddAddress
(
unsigned long  *SendContextPtr,
char *Address
)
{
   register char  *cptr, *sptr;

   int  status;
   short int  Length;
   static char  String [512];

   struct {
      short int  buf_len;
      short int  item;
      unsigned char   *buf_addr;
      unsigned short  *ret_len;
   }  SendUserNameItem [] =
   {
      { 0, MAIL$_SEND_USERNAME, 0, 0 },
      {0,0,0,0}
   },
   NullItem = {0,0,0,0};

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "PostAddAddress() |%s|\n", Address);

   if (SendOkCount || SendFailedCount)
      fprintf (HttpOut, "</PRE>%s<PRE>", Address);
   else
      fprintf (HttpOut, "%s<PRE>", Address);
   fflush (HttpOut);

   for (cptr = Address; *cptr && *cptr != '@' && *cptr != '%'; cptr++);
   if (*cptr == '@' && cptr > Address)
   {
      /*******************************/
      /* RFC821-style address (SMTP) */
      /*******************************/

      if (!SmtpTransport[0])
      {
         SendFailedCount++;
         fprintf (HttpOut, "%%%s-E-SMTP, not available.\n", Utility);
         return (SS$_NORMAL);
      }

      sptr = String;
      strcpy (sptr, SmtpTransport);
      while (*sptr) sptr++;
      *sptr++ = '%';
      *sptr++ = '\"';
      /* skip leading spaces */
      for (cptr = Address; isspace(*cptr); cptr++);
      while (*cptr) *sptr++ = *cptr++;
      /* trim trailing spaces */
      while (isspace(sptr[-1])) sptr--;
      *sptr++ = '\"';
      *sptr = '\0';
      SendUserNameItem[0].buf_addr = String;
      SendUserNameItem[0].buf_len = sptr - String;
   }
   else
   if (*cptr == '%')
   {
      /***************************/
      /* mail transport involved */
      /***************************/

      SendFailedCount++;
      fprintf (HttpOut, "%%%s-E-TRANS, mail transports not supported\n",
               Utility);
      return (SS$_NORMAL);
   }
   else
   {
      /*******************/
      /* vanilla address */
      /*******************/

      SendUserNameItem[0].buf_addr = Address;
      SendUserNameItem[0].buf_len = strlen(Address);
   }

   if (Debug)
      fprintf (stdout, "address |%s|\n", SendUserNameItem[0].buf_addr);
   status = mail$send_add_address (SendContextPtr, &SendUserNameItem,
                                   &NullItem);
   if (Debug) fprintf (stdout, "mail$send_add_address() %%X%08.08X\n", status);

   if (VMSok (status))
   {
      fprintf (HttpOut, "<I>ok</I>");
      SendOkCount++;
   }
   else
      SendFailedCount++;

   return (status);
}

/*****************************************************************************/
/*
Assume the identity of the mild-mannered user.  Creates a space-padded 12 
character string containing the username.  Calls a macro routine to copy this 
name into the process header (requires CMKRNL).  Effectively the process 
becomes identified by the string.  Used by PostMessage() to masquerade as the
sender of the message.  SYSPRV with MAIL$_SEND_FROM_LINE is fine until a
foreign mail transport is invoked (e.g. MX) and then ceases to work!  This
kludge becomes necessary!
*/ 

TelephoneBooth (char *UserName)

{
   register int  len;
   register char  *cptr, *sptr;

   int  status;
   unsigned long  PID = 0;
   char  PaddedUserName [13];
   struct {
      short int  buf_len;
      short int  item;
      unsigned char   *buf_addr;
      unsigned short  *ret_len;
   }
   JpiUserNameItem [] =
   {
      { 12, JPI$_USERNAME, JpiUserName, 0 },
      {0,0,0,0}
   };

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "TelephoneBooth() |%s|\n", UserName);

   if (VMSnok (status = sys$getjpiw (0, &PID, 0, &JpiUserNameItem, 0, 0, 0)))
   {
      fprintf (stdout, "%%%s-I-WHERE, %s %d\n", Utility, __FILE__, __LINE__);
      exit (status);
   }
   JpiUserName[12] = '\0';
   if (Debug) fprintf (stdout, "JpiUserName |%s|\n", JpiUserName);

   sptr = PaddedUserName;
   len = sizeof(PaddedUserName)-1;
   for (cptr = UserName; *cptr && len--; *sptr++ = *cptr++);
   if (len < 0)
   {
      ErrorGeneral (ErrorStringSize, __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }
   while (len-- > 0) *sptr++ = ' ';
   *sptr = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", PaddedUserName);

   /* ensure the username is changed back upon exit */
   ExitHandler.HandlerAddress = &ExitViaTelephoneBooth;
   ExitHandler.ArgCount = 1;
   ExitHandler.ExitStatus = &ExitStatus;
   if (VMSnok (status = sys$dclexh (&ExitHandler)))
   {
      fprintf (stdout, "%%%s-I-WHERE, %s %d\n", Utility, __FILE__, __LINE__);
      exit (status);
   }

   if (VMSnok (status = SetUserName (PaddedUserName)))
   {
      fprintf (stdout, "%%%s-I-WHERE, %s %d\n", Utility, __FILE__, __LINE__);
      exit (status);
   }
}

/*****************************************************************************/
/*
A VMS exit handler.  When the images exit (normal or error) ensure the assumed
username is changed back.
*/ 

ExitViaTelephoneBooth ()

{
   if (Debug)
      fprintf (stdout, "ExitViaTelephoneBooth()\n %%X%08.08X", ExitStatus);

   if (JpiUserName[0]) SetUserName (JpiUserName);
   JpiUserName[0] = '\0';
}

/*****************************************************************************/
/*
This function fills the 'String' storage with up to 'SizeOfString'-1 
characters from the HTTP input stream until the storage fills or the delimiter 
is encountered.  If the delimitter is specified as the null character ('\0') 
the all of the reset of the data stream is placed into the 'String' storage.  
It returns 0 or greater as the length of the returned string.  If the 'String' 
storage is too small to completely receive the characters required the 
function returns 'SizeOfString' to indicate the overflow.  Uses the global 
storage 'ContentLength' (if non-zero) to check when the stream should 
terminate, keeping track of the number of characters processed in global 
storage 'ContentCount'.  Not designed to retrieve from binary data streams!
*/ 

int GetHttpString
(
char Delimiter,
char *String,
int SizeOfString,
boolean UrlEscaped
)
{
   static char  *HttpBufferPtr = NULL;
   static char  HttpBuffer [HttpBufferSize];

   register int  sleft, clen, ccnt;
   register char  c, del, fc;
   register char  *sptr, *hbptr;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout,
               "GetHttpString() ContentLength/Count: %d/%d\n",
               ContentLength, ContentCount);

   if (HttpBufferPtr == NULL)
      *(hbptr = HttpBuffer) = '\0';
   else
      hbptr = HttpBufferPtr;

   /* put these into register storage for efficiency */
   clen = ContentLength;
   ccnt = ContentCount;
   del = Delimiter;
   sptr = String;
   sleft = SizeOfString - 1;

   while ((!del || *hbptr != del) && sleft && (!clen || ccnt < clen))
   {
      if (!*hbptr)
      {
         /* buffer empty (or first call), read from HTTP source */
         if (Debug) fprintf (stdout, "fgets()\n");
         if (fgets (HttpBuffer, sizeof(HttpBuffer), HttpIn) == NULL)
         {
            ErrorGeneral ("HTTP stream terminated unexpectedly.",
                          __FILE__, __LINE__);
            exit (SS$_NORMAL);
         }
         hbptr = HttpBuffer;
         if (Debug) fprintf (stdout, "HttpBuffer (%d) |%s|\n", __LINE__, hbptr);
         continue;
      }
      if (*hbptr == '\r' || *hbptr == '\n')
      {
         hbptr++;
         continue;
      }
      if (!UrlEscaped)
      {
         *sptr++ = *hbptr++;
         ccnt++;
         sleft--;
         continue;
      }
      if (*hbptr != '%')
      {
         /* not an escaped character */
         if (*hbptr == '+')
         {
            *sptr++ = ' ';
            hbptr++;
         }
         else
            *sptr++ = *hbptr++;
         ccnt++;
         sleft--;
         continue;
      }

      /* an escaped character ("%xx" where xx is a hex number) */
      hbptr++;
      if (!*hbptr)
      {
         /* buffer empty read from HTTP source */
         if (Debug) fprintf (stdout, "fgets()\n");
         if (fgets (HttpBuffer, sizeof(HttpBuffer), HttpIn) == NULL)
         {
            ErrorGeneral ("HTTP stream terminated unexpectedly.",
                          __FILE__, __LINE__);
            exit (SS$_NORMAL);
         }
         hbptr = HttpBuffer;
         if (Debug) fprintf (stdout, "HttpBuffer (%d) |%s|\n", __LINE__, hbptr);
      }
      ccnt++;
      c = 0;
      /* first hex digit */
      fc = *hbptr;
      if (fc >= '0' && fc <= '9')
         c = (fc - (int)'0') << 4;
      else
      if (toupper(fc) >= 'A' && toupper(fc) <= 'F')
         c = (toupper(fc) - (int)'A' + 10) << 4;
      else
      {
         /* not a valid hex digit, assume its a mistake! */
         *sptr++ = '%';
         continue;
      }
      if (fc) hbptr++;
      if (!*hbptr)
      {
         /* buffer empty read from HTTP source */
         if (Debug) fprintf (stdout, "fgets()\n");
         if (fgets (HttpBuffer, sizeof(HttpBuffer), HttpIn) == NULL)
         {
            ErrorGeneral ("HTTP stream terminated unexpectedly.",
                          __FILE__, __LINE__);
            exit (SS$_NORMAL);
         }
         hbptr = HttpBuffer;
         if (Debug) fprintf (stdout, "HttpBuffer (%d) |%s|\n", __LINE__, hbptr);
      }
      ccnt++;
      /* second hex digit */
      if (*hbptr >= '0' && *hbptr <= '9')
         c += (*hbptr - (int)'0');
      else
      if (toupper(*hbptr) >= 'A' && toupper(*hbptr) <= 'F')
         c += (toupper(*hbptr) - (int)'A' + 10);
      else
      {
         /* not a valid hex digit, assume its a bigger mistake! */
         *sptr++ = '%';
         *sptr++ = fc;
         continue;
      }
      if (*hbptr) hbptr++;
      *sptr++ = c;
      ccnt++;
      sleft--;
   }

   if (*hbptr == del)
   {
      /* if the loop exited due to a delimiter encountered, step over it */
      hbptr++;
      ccnt++;
   }

   *sptr = '\0';
   if (Debug) fprintf (stdout, "String |%s|\n", String);

   /* if we're using a 'ContentLength' then keep track of 'ContentCount' */
   if (clen)
      ContentCount = ccnt;
   else
      ContentCount = 0;

   HttpBufferPtr = hbptr;

   if (!sleft && *hbptr != del)
   {
      /* must have run out of string space, return overflow indication */
      return (SizeOfString);
   }
   else
      return (sptr - String);
}

/*****************************************************************************/
/*
*/

boolean GetCgiVariable
(
char *VariableName,
char *VariableValue,
int VariableValueSize
)
{
   static $DESCRIPTOR (VariableNameDsc, "");
   static $DESCRIPTOR (VariableValueDsc, "");

   register int  status;

   unsigned short  Length;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "GetCgiVariable() |%s|\n", VariableName);

   VariableNameDsc.dsc$w_length = strlen(VariableName);
   VariableNameDsc.dsc$a_pointer = VariableName;
   VariableValueDsc.dsc$w_length = VariableValueSize-1;
   VariableValueDsc.dsc$a_pointer = VariableValue;

   if (VMSok (status =
       lib$get_symbol (&VariableNameDsc, &VariableValueDsc, &Length, 0)))
      VariableValue[Length] = '\0';
   else
      VariableValue[0] = '\0';

   if (status == LIB$_NOSUCHSYM) return (false);
   if (VMSnok (status))
   {
      if (Debug) fprintf (stdout, "lib$get_symbol() %%X%08.08X\n", status);
      fprintf (stdout, "%%%s-I-WHERE, %s %d\n", Utility, __FILE__, __LINE__);
      exit (status);
   }
   if (Debug) fprintf (stdout, "|%s|\n", VariableValue);
   return (true);
}

/*****************************************************************************/
/*
Copy string from 'cptr' to 'sptr', escaping forbidden HTML characters.  A 
maximum of 'scnt'-1 characters are copied, with the resulting string always 
terminated.  The count of the number of characters copied is returned 
(excluding the terminating null).
*/ 

char* CopyToHtml
(
register char *sptr,
register char *cptr,
register int scnt
)
{
   register char  *StartPtr;

   if (scnt-- <= 0) return;
   StartPtr = sptr;
   while (*cptr && scnt--)
   {
      switch (*cptr)
      {
         case '\"' :
            if (scnt >= 6)
            {
               *sptr++ = '&'; *sptr++ = 'q'; *sptr++ = 'u'; *sptr++ = 'o'; 
               *sptr++ = 't'; *sptr++ = ';';
               scnt -= 6;
               cptr++;
            }
            break;
         case '<' :
            if (scnt >= 4)
            {
               *sptr++ = '&'; *sptr++ = 'l'; *sptr++ = 't'; *sptr++ = ';'; 
               scnt -= 4;
               cptr++;
            }
            break;
         case '>' :
            if (scnt >= 4)
            {
               *sptr++ = '&'; *sptr++ = 'g'; *sptr++ = 't'; *sptr++ = ';'; 
               scnt -= 4;
               cptr++;
            }
            break;
         case '&' :
            if (scnt >= 5)
            {
               *sptr++ = '&'; *sptr++ = 'a'; *sptr++ = 'm'; *sptr++ = 'p'; 
               *sptr++ = ';';
               scnt -= 5;
               cptr++;
            }
            break;
         default :
            *sptr++ = *cptr++;
            scnt--;
      }
   }
   *sptr = '\0';
   return (sptr - StartPtr);
}

/*****************************************************************************/
/*
*/

ErrorAuthentication
(
char *SourceFileName,
int SourceLineNumber
)
{
   register char  *cptr;

   /*********/
   /* begin */
   /*********/

   /* 
      The source file format provided by the "__FILE__" macro will be
      "device:[directory]name.type;ver".  Reduce that to "name.type".
   */
   for (cptr = SourceFileName; *cptr && *cptr != ';'; cptr++);
   if (*cptr)
   {
      while (*cptr != '.') cptr--;
      *cptr-- = '\0';
   }
   while (*cptr != ']') cptr--;
   cptr++;

   fprintf (HttpOut,
"HTTP/1.0 401 Authentication failure.\n\
WWW-Authenticate: Basic realm=\"HFRD\"\n\
Content-Type: text/html\n\
\n\
<!-- SoftwareID: %s Module: %s Line: %d -->\n\
<H1>ERROR!</H1>\n\
<P>Reported by server.\n\
<P>Authentication failure.\n",
   SoftwareID, cptr, SourceLineNumber);
}

/*****************************************************************************/
/*
*/

ErrorGeneral
(
char *Text,
char *SourceFileName,
int SourceLineNumber
)
{
   register char  *cptr;

   /*********/
   /* begin */
   /*********/

   /* 
      The source file format provided by the "__FILE__" macro will be
      "device:[directory]name.type;ver".  Reduce that to "name.type".
   */
   for (cptr = SourceFileName; *cptr && *cptr != ';'; cptr++);
   if (*cptr)
   {
      while (*cptr != '.') cptr--;
      *cptr-- = '\0';
   }
   while (*cptr != ']') cptr--;
   cptr++;

   if (!HttpHasBeenOutput) fputs (Http404Header, HttpOut);
   fprintf (HttpOut,
"<!-- SoftwareID: %s Module: %s Line: %d -->\n\
<H1>ERROR!</H1>\n\
<P>Reported by server.\n\
<P>%s\n",
   SoftwareID, cptr, SourceLineNumber, Text);
}

/*****************************************************************************/
/*
*/

ErrorVmsStatus
(
int StatusValue,
char *Text,
char *HiddenText,
char *SourceFileName,
int SourceLineNumber
)
{
   static char  Message [256];
   static $DESCRIPTOR (MessageDsc, Message);

   register char  *cptr;
   int  status;
   short int  Length;

   /*********/
   /* begin */
   /*********/

   if (VMSok (status = sys$getmsg (StatusValue, &Length, &MessageDsc, 1, 0))) 
   {
      Message[Length] = '\0';
      Message[0] = toupper(Message[0]);
   }
   else
      exit (status);

   /* 
      The source file format provided by the "__FILE__" macro will be
      "device:[directory]name.type;ver".  Reduce that to "name.type".
   */
   for (cptr = SourceFileName; *cptr && *cptr != ';'; cptr++);
   if (*cptr)
   {
      while (*cptr != '.') cptr--;
      *cptr-- = '\0';
   }
   while (*cptr != ']') cptr--;
   cptr++;

   if (!HttpHasBeenOutput) fputs (Http404Header, HttpOut);
   fprintf (HttpOut,
"<!-- SoftwareID: %s Module: %s Line: %d -->\n\
<H1>ERROR!</H1>\n\
<P>Reported by server.\n\
<P>%s ... <TT>%s</TT>\n\
<!-- %%X%08.08X \"%s\" -->\n",
   SoftwareID, cptr, SourceLineNumber, Message, Text, StatusValue, HiddenText);
}

/****************************************************************************/
/*
Does a case-insensitive, character-by-character string compare and returns 
true if two strings are the same, or false if not.  If a maximum number of 
characters are specified only those will be compared, if the entire strings 
should be compared then specify the number of characters as 0.
*/ 

boolean strsame
(
register char *sptr1,
register char *sptr2,
register int  count
)
{
   while (*sptr1 && *sptr2)
   {
      if (toupper (*sptr1++) != toupper (*sptr2++)) return (false);
      if (count)
         if (!--count) return (true);
   }
   if (*sptr1 || *sptr2)
      return (false);
   else
      return (true);
}

/*****************************************************************************/

