/*****************************************************************************/
/*
                             SearchTextFile.c

VERSION HISTORY
---------------
28-MAR-95  MGD  minor revision
24-NOV-94  MGD  minor revision
10-JUN-94  MGD  initial development
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <ctype.h>

/* VMS related header files */
#include <rmsdef.h>
#include <rms.h>
#include <ssdef.h>
#include <stsdef.h>

#ifdef __ALPHA
#   pragma nomember_alignment
#endif

#define boolean int
#define true 1
#define false 0
 
#define VMSok(x) ((x) & STS$M_SUCCESS)
#define VMSnok(x) !(((x) & STS$M_SUCCESS))

/* external declarations */
extern boolean  Debug;
extern char  ExtractScriptName[];
extern FILE  *HttpOut;

/* external functions */
extern char* SearchTextString (char*, char*, boolean);
extern char* MapUrl (char*, char*, char*, char*);

/*****************************************************************************/
/*
Search each record of what is presumed to be a plain-text file for the 
supplied string.  When the first hit occurs output an HTML anchor for 
retrieving the entire file.  For each hit output an HTML anchor to extract a 
specified range of record (lines) from the file.  This search only checks for 
at least one match in a line before considering it a hit.
*/

SearchTextFile
(
char *FileName,
int FileNameLength,
char *DocumentName,
char *UriSearchString,
char *SearchString,
int SearchStringLength,
int *RecordCountPtr,
int *FileHitCountPtr,
int *TotalHitCountPtr,
boolean CaseSensitive,
boolean DocumentOnly,
int ExtractNumberOfRecords,
boolean ExactNumberOfRecords
)
{
   register char  *rptr, *sptr;
   register int  idx,
                 len,
                 LastSectionRecordNumber = 1,
                 RecordNumber = 0;
   int  status,
        HitCount = 0;
   char  *CaseSensitivePtr = "";
   char  String [2048],
         Record [1024+1];
   struct FAB  FileFAB;
   struct RAB  FileRAB;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "SearchTextFile() |%s|%s|\n", FileName, SearchString);

   FileFAB = cc$rms_fab;
   FileFAB.fab$b_fac = FAB$M_GET;
   FileFAB.fab$l_fna = FileName;  
   FileFAB.fab$b_fns = FileNameLength;
   FileFAB.fab$b_shr = FAB$M_SHRGET;

   if (VMSnok (status = sys$open (&FileFAB, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$open() %%X%08.08X\n", status);
      return (status);
   }

   FileRAB = cc$rms_rab;
   FileRAB.rab$l_fab = &FileFAB;
   /* 2 buffers and read ahead performance option */
   FileRAB.rab$b_mbf = 2;
   FileRAB.rab$l_rop = RAB$M_RAH;
   FileRAB.rab$l_ubf = Record;
   FileRAB.rab$w_usz = sizeof(Record)-1;

   if (VMSnok (status = sys$connect (&FileRAB, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$connect() %%X%08.08X\n", status);
      sys$close (&FileFAB, 0, 0);
      return (status);
   }

   if (CaseSensitive) CaseSensitivePtr = "&case=yes";

   /**********************/
   /* search all records */
   /**********************/

   while (VMSok (status = sys$get (&FileRAB, 0, 0)))
   {
      RecordNumber++;
      Record[FileRAB.rab$w_rsz] = '\0';

      /* terminate on any carriage control that may be in the record */
      for (rptr = Record; *rptr && *rptr != '\r' && *rptr != '\n'; rptr++);
      *rptr = '\0';

      /* if necessary generate document name from first non-blank line */
      if (!DocumentName[0]) GenerateDocumentName (Record, DocumentName);

      if (*(rptr = Record))
         while (*rptr && isspace(*rptr)) rptr++;
      if (!*rptr)
      {
         /* the line contained none, or only white-space characters */
         LastSectionRecordNumber = RecordNumber + 1;
         continue;
      }

      rptr = SearchTextString (Record, SearchString, CaseSensitive);
      if (*rptr)
      {
         /********/
         /* hit! */
         /********/

         if (Debug) fprintf (stdout, "Hit |%s|\n", rptr);
         if (!HitCount++)
         {
            /*************************************/
            /* first hit, entire document anchor */
            /*************************************/

            if (!*TotalHitCountPtr) fputs ("<P>\n<OL>\n", HttpOut);

            if (DocumentOnly)
            {
               fprintf (HttpOut,
                  "<LI><A HREF=\"/%s%s?highlight=%s%s\">\"%s\"</A>\n",
                  ExtractScriptName, MapUrl(NULL,FileName,NULL,NULL),
                  UriSearchString, CaseSensitivePtr, DocumentName);

               /**********************************************/
               /* document only, break from search loop now! */
               /**********************************************/

               break;
            }
            else
            {
               fprintf (HttpOut,
"<LI>TEXT document: <A HREF=\"/%s%s?highlight=%s%s\">\"%s\"</A>\n\
<OL>\n",
               ExtractScriptName, MapUrl(NULL,FileName,NULL,NULL),
               UriSearchString, CaseSensitivePtr, DocumentName);
            }
         }
         *TotalHitCountPtr += 1;

         /***********************/
         /* file extract anchor */
         /***********************/

         sptr = String;
         strcpy (sptr, "<LI>");
         sptr += 4;
         /* copy the record up to the first character of the search string */
         sptr = CopyTextIntoHtml (sptr, Record, rptr-Record);

         /* add the HTML anchor */
         len = sprintf (sptr,
            "<A HREF=\"/%s%s?highlight=%s",
            ExtractScriptName, MapUrl(NULL,FileName,NULL,NULL),
            UriSearchString);
         sptr += len;

         if (ExactNumberOfRecords)
            len = sprintf (sptr, "&start=%d&end=%d&exact=true\">",
                  RecordNumber, RecordNumber+ExtractNumberOfRecords-1);
         else
            len = sprintf (sptr, "&start=%d&end=%d\">",
                  LastSectionRecordNumber,
                  RecordNumber+ExtractNumberOfRecords-1);
         sptr += len;

         /* matched string, highlighted within the anchor */
         sptr = CopyTextIntoHtml (sptr, rptr, SearchStringLength);
         strcpy (sptr, "</A>");
         sptr += 4;

         /* rest of record after the matched search string */
         sptr = CopyTextIntoHtml (sptr, rptr+SearchStringLength, -1);
         *sptr++ = '\n'; *sptr = '\0';

         if (Debug) fprintf (stdout, "String |%s|\n", String);
         fputs (String, HttpOut);
      }
   }

   /***************/
   /* end of file */
   /***************/

   if (status == RMS$_EOF) status = SS$_NORMAL;
   sys$close (&FileFAB, 0, 0);

   if (status == RMS$_RTB)
   {
      fprintf (HttpOut,
"<P><B>ERROR!</B> <I>Record (line) %d too big for query search buffer!</I>\
<BR><TT>%s</TT> <!-- %s --><P>\n",
      RecordNumber, MapUrl(NULL,FileName,NULL,NULL), FileName);
      status = SS$_NORMAL;
   }

   if (HitCount)
   {
      if (!DocumentOnly) fputs ("</OL>\n", HttpOut);
      *FileHitCountPtr += 1;
   }

   *RecordCountPtr += RecordNumber;

   return (status);
}

/*****************************************************************************/
/*
This function serves to generate a document name string (for use in the HTML
<TITLE> tag) for a plain-text file from the first line containing alpha-
numeric characters.  Copy the string pointed to by 'sptr' into the string 
pointed to by 'NamePtr', compressing white-space.
*/ 

GenerateDocumentName
(
register char *sptr,
register char *nptr
)
{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "GenerateDocumentName() |%s|\n", sptr);

   /* skip leading non-alphanumerics */
   while (*sptr && !isalnum(*sptr)) sptr++;
   while (*sptr)
   {
      /* copy alphanumeric element */
      while (*sptr && !isspace(*sptr) && isalnum(*sptr)) *nptr++ = *sptr++;
      if (!*sptr) break;
      /* skip intervening/trailing non-alphanumerics */
      while (*sptr && !isalnum(*sptr)) sptr++;
      if (!*sptr) break;
      /* add a single space between alphanumeric elements */
      *nptr++ = ' ';
   }
   *nptr = '\0';
}

/*****************************************************************************/
