/*****************************************************************************/
/*
                                chkmap.c

Allows command-line checking of the HFRD VMS HTTPd rule mapping.


QUALIFIERS
----------
/DBUG                   turns on all "if (Debug)" statements
/HELP                   display brief usage information
/MAP=                   specifies the location of the mapping file
/SWID                   display software ID, image name, system architecture


BUILD DETAILS
-------------
See BUILD_CHKMAP.COM


VERSION HISTORY (update SoftwareID as well!)
---------------
01-DEC-95  MGD  v1.0.0, initial development
*/
/*****************************************************************************/

#ifdef __ALPHA
   char SoftwareID [] = "CHKMAP v1.0.0 AXP";
#else
   char SoftwareID [] = "CHKMAP v1.0.0 VAX";
#endif

/* standard C header files */
#include <stdio.h>
#include <ctype.h>

/* VMS related header files */
#include <jpidef.h>
#include <lnmdef.h>
#include <psldef.h>
#include <ssdef.h>
#include <syidef.h>
#include <stsdef.h>

#include "[-.httpd]httpd.h"

#define boolean int
#define true 1
#define false 0
 
#define VMSok(x) ((x) & STS$M_SUCCESS)
#define VMSnok(x) !(((x) & STS$M_SUCCESS))

char  Utility [] = "CHKMAP";

boolean  Debug,
         DoShowSWID,
         DoShowHelp;

char  CommandLine [256],
      HttpdMapFileName [256],
      Path [256];

/* required prototypes */
char* MapUrl (char*, char*, char*, char*);
 
/*****************************************************************************/
/*
*/

int main ()

{
   register char  *sptr;

   int  status;
   char  Script [256],
         ScriptVms [256],
         Vms [256];

   /*********/
   /* begin */
   /*********/

   if (VMSnok (status = ParseCommandLine ()))
      exit (status);

   if (DoShowSWID) exit (ShowSWID ());

   if (DoShowHelp) exit (ShowHelp ());

   if (HttpdMapFileName[0])
      if (VMSnok (status = DefineHttpMap ()))
         exit (status);

   Script[0] = ScriptVms[0] = Vms[0] = '\0';

   if (Path[0] == '/')
   {
      sptr = MapUrl (Path, Vms, Script, ScriptVms);
      if (!*sptr)
      {
         fprintf (stdout, "%%%s-W-ERROR, %s\n", Utility, sptr+1);
         exit (STS$K_ERROR | STS$M_INHIB_MSG);
      }
      if (Script[0])
      {
         fprintf (stdout, "%%%s-I-SCRIPT, \"%s\" %s\n",
                  Utility, Script, ScriptVms);
         strcpy (Path, sptr);
         if (Vms[0])
            fprintf (stdout, "%%%s-I-PATH, \"%s\" %s\n", Utility, Path, Vms);
         else
            fprintf (stdout, "%%%s-I-PATH, \"%s\"\n", Utility, Path);
      }
      else
         fprintf (stdout, "%%%s-I-VMS, %s\n", Utility, Vms);
   }
   else
   if (Path[0])
   {
      strcpy (Vms, Path);
      Path[0] = '\0';
      sptr = MapUrl (Path, Vms, NULL, NULL);
      if (!*sptr)
      {
         fprintf (stdout, "%%%s-W-ERROR, %s\n", Utility, sptr+1);
         exit (STS$K_ERROR | STS$M_INHIB_MSG);
      }
      fprintf (stdout, "%%%s-I-PATH, \"%s\"\n", Utility, Path);
   }
   else
      sptr = MapUrl (NULL, NULL, NULL, NULL);

   exit (SS$_NORMAL);
}

/*****************************************************************************/
/*
Define a process-private logical name for the rule mapping file.  It will 
automatically be deleted upon image exit.
*/
 
int DefineHttpMap ()
 
{
   unsigned char  UserAccessMode = PSL$C_USER;
   $DESCRIPTOR (LnmProcessDsc, "LNM$PROCESS");
   $DESCRIPTOR (HttpdMapLogicalNameDsc, "HTTPD$MAP");
   struct {
      unsigned short  buf_len;
      unsigned short  item;
      unsigned char   *buf_addr;
      unsigned short  *ret_len;
   }
      LnmItem [] =
   {
       { strlen(HttpdMapFileName), LNM$_STRING, HttpdMapFileName, 0 },
       { 0,0,0,0 }
   };

   /*********/
   /* begin */
   /*********/

   return (sys$crelnm (0, &LnmProcessDsc, &HttpdMapLogicalNameDsc, 
                       &UserAccessMode, &LnmItem));
}

/*****************************************************************************/
/*
*/
 
int ShowHelp ()
 
{
   fprintf (stdout,
"%%%s-I-HELP, usage for the CHKMAP utility (%s)\n\
\n\
Allows command-line checking of rules within the HFRD VMS HTTPd mapping\n\
file.  The file is located via the logical name HTTPD$MAP or by specifying\n\
an alternate location using the /MAP qualifier.\n\
\n\
Checks URL format paths, converting these into VMS file system paths,\n\
URL format CGI scripts, converting into script and path specifications,\n\
VMS file system format paths, converting these into URL format paths.\n\
\n\
If no URL or VMS file path is specified the mapping file is loaded to\n\
check for any rule format problems, but no conversion takes place.\n\
\n\
$ CHKMAP [\"url_path\"|vms_file_path] [qualifiers ...]\n\
\n\
/HELP /MAP=[file] /SWID\n\
\n\
Usage examples:\n\
\n\
$ CHKMAP\n\
$ CHKMAP \"/hyperdata/home.html\"\n\
$ CHKMAP \"/script/path\" /MAP=test_map.conf\n\
$ CHKMAP user$disk:[bloggsm.www]home.html\n\
\n",
   Utility, SoftwareID);
 
   return (SS$_NORMAL);
}
 
/****************************************************************************/
/*
Display software ID, system architecture and image name.
*/
 
/* htis is currently not in header files for VAX C */
#define SYI$_ARCH_NAME 4454
 
int ShowSWID ()
 
{
   register char  *cptr;
   register int  status;
   char  JpiImageName [256],
         SyiArchName [16];
   struct {
      short BufferLength;
      short ItemCode;
      void  *BufferPtr;
      void  *LengthPtr;
   } JPIitems [] =
   {
      { sizeof(JpiImageName), JPI$_IMAGNAME, JpiImageName, 0 },
      {0,0,0,0}
   },
     SYIitems [] =
   {
      { 15, SYI$_ARCH_NAME, SyiArchName, 0 },
      {0,0,0,0}
   };
 
   if (VMSnok (status = sys$getjpiw (0, 0, 0, &JPIitems, 0, 0, 0)))
      return (status);
 
   if (VMSnok (status = sys$getsyiw (0, 0, 0, &SYIitems, 0, 0, 0)))
      return (status);
 
   /* remove the the version number included */
   for (cptr = JpiImageName; *cptr && *cptr != ';'; cptr++);
   *cptr = '\0';
 
   fprintf (stdout,
   "%%%s-I-SOFTWAREID, %s\n%%%s-I-IMAGE, %s\n%%%s-I-ARCHITECTURE, %s\n",
   Utility, SoftwareID, Utility, JpiImageName, Utility, SyiArchName);
 
   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
*/

char* SysGetMsg (int StatusValue)

{
   static char  Message [256];
   register int  status;
   short int  Length;
   $DESCRIPTOR (MessageDsc, Message);

   if (VMSok (status = sys$getmsg (StatusValue, &Length, &MessageDsc, 0, 0))) 
      Message[Length] = '\0';
   else
      sprintf (Message, "%%sys$getmsg() %%X%08.08X", status);

   return (Message);
}

/****************************************************************************/
/*
Does a case-insensitive, character-by-character string compare and returns 
true if two strings are the same, or false if not.  If a maximum number of 
characters are specified only those will be compared, if the entire strings 
should be compared then specify the number of characters as 0.
*/ 
 
boolean strsame
(
register char *sptr1,
register char *sptr2,
register int  count
)
{
   while (*sptr1 && *sptr2)
   {
      if (toupper (*sptr1++) != toupper (*sptr2++)) return (false);
      if (count)
         if (!--count) return (true);
   }
   if (*sptr1 || *sptr2)
      return (false);
   else
      return (true);
}
 
/****************************************************************************/
/*
This function allows images activated by a "foreign verb" to behave in a way 
that approximates the CLI$ (Command Line Interpreter) utility calls.  Get the 
entire command line following the verb that activated the image.  The command 
line is returned in uppercase, space compressed (i.e. maximum of one space 
between text elements, trimmed of leading and trailing spaces).  Returns a 
warning status if there were no parameters/qualifiers on the command line.
The variable CommandLine is global.
*/ 
 
int ParseCommandLine ()
 
{
   int  status;
   unsigned short  Length;
   unsigned long  Flags = 0;
   struct dsc$descriptor_s 
          CommandLineDsc = { sizeof(CommandLine)-1, DSC$K_DTYPE_T,
                             DSC$K_CLASS_S, CommandLine };
 
   /* get the entire command line following the verb */
   if (VMSnok (status = lib$get_foreign (&CommandLineDsc, 0, &Length, &Flags)))
      return (status);
   CommandLine[Length] = '\0';
 
   if (ParseCommand (CommandLine))
      return (SS$_NORMAL);
   else
      return (STS$K_ERROR | STS$M_INHIB_MSG);
}
 
/****************************************************************************/
/*
This function allows images activated by a "foreign verb" to behave in a way 
that approximates the CLI$ (Command Line Interpreter) utility calls.  Quoted 
strings are always indicated by being parsed to include a single leading 
quote.
*/ 
 
boolean ParseCommand (char *CommandLine)
 
{
   register int  QuoteCount = 0;
   register char  *cptr, *eptr;
   boolean  CommandLineOK = true;
   char  Entity [256] = "";
 
   /* set up any argument defaults */
   ParseCommandEntity (NULL);
 
   cptr = CommandLine;
   eptr = Entity;
 
   for (;;)
   {
      if (*cptr == '\"')
      {
         QuoteCount++;
         *eptr++ = *cptr++;
         continue;
      }
 
      if (QuoteCount & 1 && *cptr)
      {
         /* inside quoted text, copy all characters as literals */
         *eptr++ = *cptr++;
         continue;
      }
 
      if (*cptr == '/' || isspace (*cptr) || !*cptr)
      {
         if (isspace (*cptr))
         {
            /* span the white space */
            while (*cptr && isspace (*cptr)) cptr++;
            if (*cptr == '=')
            {
               /* part of a qualifier, continue to get the value */
               *eptr++ = *cptr++;
               /* span any intervening white space */
               while (*cptr && isspace (*cptr)) cptr++;
               continue;
            }
         }
 
         if (Entity[0])
         {
            *eptr = '\0';
            if (!ParseCommandEntity (Entity)) CommandLineOK = false;
         }
 
         /* if end of command line then break from loop */
         if (!*cptr) break;
 
         /* start of new entity */
         eptr = Entity;
         /* if start of qualifier ensure slash is copied */
         if (*cptr == '/') *eptr++ = *cptr++;
 
         continue;
      }
 
      /* any other character, just copy, ensure upper case */
      *eptr++ = toupper(*cptr++);
   }
 
   return (CommandLineOK);
}
 
/*****************************************************************************/
/*
Get a string value from a qualifier, e.g. '/EXAMPLE=TEST'.
*/
 
boolean ParseCommandString
(
char *Entity,
char *String,
boolean Qualifier,
boolean ReportErrors,
boolean EnsureUpperCase
)
{
   register int  QuoteCount = 0;
   register char  *eptr, *sptr;
 
   if (Debug) fprintf (stdout, "ParseCommandString()\nEntity: '%s'\n", Entity);
 
   eptr = Entity;
 
   if (Qualifier)
   {
      /* scan down to equate symbol */
      while (*eptr && *eptr != '=') eptr++;
      if (*eptr) eptr++;
      if (!*eptr)
      {
         if (ReportErrors)
         {
            fprintf (stdout,
            "%%%s-E-VALREQ, missing qualifier or keyword value\n \\%s\\\n",
            Utility, Entity+1);
         }
         return (false);
      }
   }
 
   sptr = String;
   while (*eptr)
   {
      if (*eptr == '\"')
      {
         if (QuoteCount & 1)
         {
            /* are inside quotes, check for escaped quotes ("") */
            if (*++eptr != '\"')
            {
               /* now outside quotes */
               QuoteCount++;
            }
            /* drop thru to character copy */
         }
         else
         {
            /* now inside quotes */
            QuoteCount++;
            eptr++;
            continue;
         }
      }
 
      if (EnsureUpperCase)
         *sptr++ = toupper(*eptr++);
      else
         *sptr++ = *eptr++;
   }
   *sptr = '\0';
 
   if (Debug) fprintf (stdout, "String: '%s'\n", String);
 
   return (true);
}
 
/*****************************************************************************/
/*
Get an integer value from a qualifier, e.g. '/EXAMPLE=99'.
*/
 
boolean ParseCommandInteger
(
char *Entity,
int *IntegerPtr,
int Base,
boolean ReportErrors
)
{
   register char  *eptr;
   char  *sptr;
 
   if (Debug)
      fprintf (stdout, "ParseCommandInteger() '%s' Base: %d\n", Entity, Base);
 
   for (eptr = Entity; *eptr && *eptr != '='; eptr++);
   if (*eptr) eptr++;
   if (*eptr)
   {
      *IntegerPtr = strtol (eptr, &sptr, Base);
      if (sptr > eptr && !*sptr)
         return (true);
      else
      {
         if (ReportErrors)
         {
            fprintf (stdout,
            "%%%s-E-BADVALUE, '%s' is an invalid keyword value\n",
            Utility, eptr);
         }
         return (false);
      }
   }
   else
   {
      if (ReportErrors)
      {
         fprintf (stdout,
         "%%%s-E-VALREQ, missing qualifier or keyword value\n \\%s\\\n",
         Utility, Entity+1);
      }
      return (false);
   }
}
 
/*****************************************************************************/
/*
A single command line "entity" has been parsed, check if its recognised.  This 
function is the one modified for the individual requirements of each program.
*/
 
boolean ParseCommandEntity (char *Entity)
 
{
   if (Entity == NULL)
   {
      /* set up any argument defaults */
      Debug = DoShowSWID = DoShowHelp = false;
      HttpdMapFileName[0] = '\0';
      return (true);
   }
 
   if (Debug) fprintf (stdout, "ParseCommandEntity() Entity: '%s'\n", Entity);
 
   if (Entity[0] == '/')
   {
      /* turns on all "if (Debug)" statements */
      if (strsame (Entity, "/DBUG", -1))
         return (Debug = true);

      if (strsame (Entity, "/HELP", 4))
         return (DoShowHelp = true);

      if (strsame (Entity, "/MAP=", 5))
         return (ParseCommandString (Entity, HttpdMapFileName,
                                     true, true, true));

      if (strsame (Entity, "/SWID", 4))
         return (DoShowSWID = true);

      fprintf (stdout,
      "%%%s-E-IVQUAL, unrecognised qualifier\n \\%s\\\n", Utility, Entity+1);
      return (false);
   }
 
   if (!Path[0])
   {
      ParseCommandString (Entity, Path, false, false, false);
      return (true);
   }

   fprintf (stdout,
   "%%%s-E-MAXPARM, too many parameters\n \\%s\\\n", Utility, Entity);
   return (false);
}
   
/*****************************************************************************/

