/*****************************************************************************/
/*
                               HTTPdMon.c

This utility displays information about an HFRD HTTPd server.  The information 
is derived from JPI data about the process, and interpreted from three 
logicals defined and refreshed by the HTTPd server. 

   HTTPDnn$COUNT      (binary) server accumulators for various events, etc.
   HTTPDnn$PID        (binary) process ID of the HTTPd server
   HTTPDnn$REQUEST    summary of the most recent completed request


QUALIFIERS
----------
/DBUG                   turns on all "if (Debug)" statements
/HELP                   display brief usage information
/PORT=                  the IP port number of the server to monitor
/REFRESH=               number of seconds between display refresh
/SWID                   display software ID, image name, system architecture


BUILD DETAILS
-------------
See BUILD_HTTPDMON.COM


VERSION HISTORY (update SoftwareID as well!)
---------------
01-DEC-95  MGD  v1.0.0, initial development
*/
/*****************************************************************************/

#ifdef __ALPHA
   char SoftwareID [] = "HTTPDMON v1.0.0 AXP";
#else
   char SoftwareID [] = "HTTPDMON v1.0.0 VAX";
#endif

/* standard C header files */
#include <stdio.h>
#include <ctype.h>

/* VMS related header files */
#include <ssdef.h>
#include <stsdef.h>
#include <descrip.h>
#include <iodef.h>
#include <jpidef.h>
#include <libclidef.h>
#include <libdtdef.h>
#include <lnmdef.h>
#include <psldef.h>
#include <prvdef.h>

/* this header file contains the accounting structure definition */
#include "../httpd/httpd.h"

#define boolean int
#define true 1
#define false 0
 
#define VMSok(x) ((x) & STS$M_SUCCESS)
#define VMSnok(x) !(((x) & STS$M_SUCCESS))

char  *Utility = "HTTPDMON";

#define ControlZ '\x1a'

boolean  Debug,
         DoShowHelp,
         ControlY;

int  RefreshSeconds,
     ServerPort = 80;

char  CommandLine [256],
      ServerProcessName [16],
      Screen [4096];
char  *ScrPtr;

struct AccountingStruct  Accounting;

/* ANSI terminal control sequences */
char ANSIcls [] = "\x1b[0;0H\x1b[J",
     ANSIhome [] = "\x1b[0;0H",
     ANSIceol [] = "\x1b[K",
     ANSIceos [] = "\x1b[J",
     ANSInormal [] = "\x1b[0m",
     ANSIbold [] = "\x1b[1m",
     ANSIblink [] = "\x1b[5m",
     ANSIreverse [] = "\x1b[7m";

/* required prototypes */
ControlY_AST ();
char* TimeString (); 
char* SysGetMsg (int);
 
/*****************************************************************************/
/*
*/

int main ()

{
   int  status,
        MonitorStatus;

   /*********/
   /* begin */
   /*********/

   if (VMSnok (status = ParseCommandLine ()))
      exit (status);

   if (DoShowHelp) exit (ShowHelp ());

   if (VMSnok (status = OnControlY (&ControlY_AST)))
      exit (status);

   MonitorStatus = MonitorHttpd ();

   if (VMSnok (status = OnControlY (0)))
      exit (status);

   exit (MonitorStatus);
}

/*****************************************************************************/
/*
Assign a channel to the terminal device.  Create a buffered screenful of 
information about the HTTPd server and output it in one IO.
*/

int MonitorHttpd ()

{
   int  status,
        PortLength;
   unsigned short  TTChannel;
   char  ReadBuffer;
   char  Line [128],
         Port [16];
   char  *TimeStringPtr;
   
   $DESCRIPTOR (TTDsc, "TT:");
   struct {
      unsigned short  Status;
      unsigned short  Offset;
      unsigned short  Terminator;
      unsigned short  TerminatorSize;
   } IOsb;

   /*********/
   /* begin */
   /*********/

   if (VMSnok (status = sys$assign (&TTDsc, &TTChannel, 0, 0, 0)))
      return (status);

   if (VMSnok (status =
       sys$qiow (0, TTChannel, IO$_WRITELBLK, 0, 0, 0,
                 ANSIcls, sizeof(ANSIcls)-1, 0, 0, 0, 0)))
      return (status);

   sprintf (ServerProcessName, "HTTPd:%d", ServerPort);
   PortLength = sprintf (Port, " Port: %d", ServerPort);

   for (;;)
   {
      if (ControlY) break;

      TimeStringPtr = TimeString ();

      ScrPtr = Screen;
      ScrPtr += sprintf (ScrPtr,
         "%s%s%s%s%*s%s         %s%s%s\r\n",
         ANSIhome,
         ANSIbold, Port, ANSInormal,
         72 - PortLength - sizeof(SoftwareID)-1 - strlen(TimeStringPtr),
         "", SoftwareID, ANSIbold, TimeStringPtr, ANSInormal);

      /* we need the startup count and last exit status from count date */
      if (VMSnok (status = GetAccounting ()))
         return (status);
      if (VMSnok (status = AddProcess ()))
         return (status);
      if (VMSnok (status = AddAccounting ()))
         return (status);
      if (VMSnok (status = AddRequest ()))
         return (status);

      strcpy (ScrPtr, ANSIceos);
      ScrPtr += sizeof(ANSIceos)-1;

      if (VMSnok (status =
          sys$qiow (0, TTChannel, IO$_WRITELBLK, 0, 0, 0,
                    Screen, ScrPtr-Screen, 0, 0, 0, 0)))
         return (status);

      status = sys$qiow (0, TTChannel, IO$_READLBLK | IO$M_TIMED,
                         &IOsb, 0, 0,
                         &ReadBuffer, 1, RefreshSeconds, 0, 0, 0);

      if (status == SS$_TIMEOUT) continue;
      if (VMSnok (status)) return (status);
      if (IOsb.Terminator == ControlZ) return (SS$_NORMAL);
   }

   if (VMSnok (status =
       sys$qiow (0, TTChannel, IO$_WRITELBLK, 0, 0, 0,
                 Screen, ScrPtr-Screen, 0, 0, 0, 0)))
      return (status);

   return (status);
}

/*****************************************************************************/
/*
Add the HTTPd server process information to the screen buffer using the 
process ID from the PID logical and a a sys$getjpi() call.  The PID logical 
contains binary data in the form of a longword process ID.
*/ 

int AddProcess ()

{
   static char  JpiPrcNam [16],
                JpiUserName [13],
                LogicalName [32],
                UpTime [32];
   static unsigned long  JpiBufIO,
                         JpiCpuTime,
                         JpiDirIO,
                         JpiFilCnt,
                         JpiFilLm,
                         JpiPageFlts,
                         JpiPagFilCnt,
                         JpiPgFlQuota,
                         JpiPrcCnt,
                         JpiPrcLm,
                         JpiVirtPeak,
                         JpiWsPeak,
                         JpiWsSize,
                         Pid;
   static unsigned long  ConnectTime [2],
                         CurrentTime [2],
                         JpiLoginTime [2];
   static char  LastExitStatus [16] = "n/a";
   static $DESCRIPTOR (LnmSystemDsc, "LNM$SYSTEM");
   static $DESCRIPTOR (LogicalNameDsc, LogicalName);
   static $DESCRIPTOR (UpTimeFaoDsc, "!%D");
   static $DESCRIPTOR (UpTimeDsc, UpTime);
   static struct
   {
      unsigned short  buf_len;
      unsigned short  item;
      unsigned char   *buf_addr;
      unsigned short  *ret_len;
   }
      JpiItems [] =
   {
      { sizeof(JpiCpuTime), JPI$_CPUTIM, &JpiCpuTime, 0 },
      { sizeof(JpiBufIO), JPI$_BUFIO, &JpiBufIO, 0 },
      { sizeof(JpiDirIO), JPI$_DIRIO, &JpiDirIO, 0 },
      { sizeof(JpiFilCnt), JPI$_FILCNT, &JpiFilCnt, 0 },
      { sizeof(JpiFilLm), JPI$_FILLM, &JpiFilLm, 0 },
      { sizeof(JpiLoginTime), JPI$_LOGINTIM, &JpiLoginTime, 0 },
      { sizeof(JpiPageFlts), JPI$_PAGEFLTS, &JpiPageFlts, 0 },
      { sizeof(JpiPagFilCnt), JPI$_PAGFILCNT, &JpiPagFilCnt, 0 },
      { sizeof(JpiPgFlQuota), JPI$_PGFLQUOTA, &JpiPgFlQuota, 0 },
      { sizeof(JpiPrcCnt), JPI$_PRCCNT, &JpiPrcCnt, 0 },
      { sizeof(JpiPrcLm), JPI$_PRCLM, &JpiPrcLm, 0 },
      { sizeof(JpiPrcNam), JPI$_PRCNAM, &JpiPrcNam, 0 },
      { sizeof(JpiUserName), JPI$_USERNAME, &JpiUserName, 0 },
      { sizeof(JpiVirtPeak), JPI$_VIRTPEAK, &JpiVirtPeak, 0 },
      { sizeof(JpiWsSize), JPI$_WSPEAK, &JpiWsPeak, 0 },
      { sizeof(JpiWsSize), JPI$_WSSIZE, &JpiWsSize, 0 },
      {0,0,0,0}
   },
      LnmItem [] =
   {
      { sizeof(Pid), LNM$_STRING, &Pid, 0 },
      {0,0,0,0}
   };

   register char  *cptr;

   int  status;
   unsigned short  Length;
   char  *StatePtr,
         *UpTimePtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "AddProcess()\n");

   if (!LogicalName[0])
      LogicalNameDsc.dsc$w_length =
         sprintf (LogicalName, "HTTPD%d$PID", ServerPort);

   Pid = 0;
   status = sys$trnlnm (0, &LnmSystemDsc, &LogicalNameDsc, 0, &LnmItem);
   if (Debug) fprintf (stdout, "sys$trnlnm() %%X%08.08X\n", status);
   if (VMSnok (status) && status != SS$_NOLOGNAM) return (status);
   if (Debug) fprintf (stdout, "Pid: %08.08X\n", Pid);

   if (Pid)
      status = sys$getjpiw (0, &Pid, 0, &JpiItems, 0, 0, 0);
   else
      status = SS$_NONEXPR;
   if (Debug) fprintf (stdout, "sys$getjpi() %%X%08.08X\n", status);
   if (VMSnok (status))
   {
      if (status == SS$_NONEXPR)
         StatePtr = "-NOT EXECUTING-";
      else
      if (status == SS$_SUSPENDED)
         StatePtr = "-MWAIT-";
      else
         return (status);
   }
   else
      StatePtr = "";

   JpiUserName[12] = '\0';
   for (cptr = JpiUserName; *cptr && *cptr != ' '; cptr++);
   *cptr = '\0';
   if (Debug) fprintf (stdout, "JpiUserName |%s|\n", JpiUserName);

   JpiPrcNam[15] = '\0';
   for (cptr = JpiPrcNam; *cptr && *cptr != ' '; cptr++);
   *cptr = '\0';
   if (Debug) fprintf (stdout, "JpiPrcNam |%s|\n", JpiPrcNam);

   sys$gettim (&CurrentTime);
   lib$sub_times (&CurrentTime, &JpiLoginTime, &ConnectTime);
   sys$fao (&UpTimeFaoDsc, &Length, &UpTimeDsc, &ConnectTime);
   UpTime[Length] = '\0';
   for (UpTimePtr = UpTime; isspace(*UpTimePtr); UpTimePtr++);

   if (Accounting.StartupCount > 1)
      sprintf (LastExitStatus, "%%X%08.08X", Accounting.LastExitStatus);

   ScrPtr += sprintf (ScrPtr,
"%s\r\n\
 %sProcess:%s %s  %sPID:%s %08.08X  %sUser:%s  %s  %s%s%s\
%s\r\n\
 %sStartup:%s %d  %sExit Status:%s %s  (Zeroed: %d)\
%s\r\n\
      %sUp:%s %s  %sCPU:%s %d %02.02d:%02.02d:%02.02d.%02.02d\
%s\r\n\
     %sBIO:%s %d  %sDIO:%s %d  %sPg.Flts.:%s %d  %sPg.Used:%s %d%%\
%s\r\n\
  %sWsSize:%s %d (%dkB)  %sWsPeak:%s %d (%dkB)  %sVirtPeak:%s %d (%dkB)\
%s\r\n\
   %sFiles:%s %d/%d  %sSubprocesses:%s %d/%d\
%s\r\n",
      ANSIceol,
      ANSIbold, ANSInormal,
      JpiPrcNam,
      ANSIbold, ANSInormal,
      Pid,
      ANSIbold, ANSInormal,
      JpiUserName,
      ANSIbold, StatePtr, ANSInormal,
      ANSIceol,
      ANSIbold, ANSInormal,
      Accounting.StartupCount,
      ANSIbold, ANSInormal,
      LastExitStatus,
      Accounting.ZeroedCount,
      ANSIceol,
      ANSIbold, ANSInormal,
      UpTimePtr,
      ANSIbold, ANSInormal,
      JpiCpuTime / 8640000,                     /* CPU day */
      (JpiCpuTime % 8640000) / 360000,          /* CPU hour */
      (JpiCpuTime % 360000) / 6000,             /* CPU minute */
      (JpiCpuTime % 6000 ) / 100,               /* CPU second */
      JpiCpuTime % 100,                         /* CPU 10mS */
      ANSIceol,
      ANSIbold, ANSInormal,
      JpiBufIO,
      ANSIbold, ANSInormal,
      JpiDirIO,
      ANSIbold, ANSInormal,
      JpiPageFlts,
      ANSIbold, ANSInormal,
      100 - (JpiPagFilCnt * 100) / JpiPgFlQuota,
      ANSIceol,
      ANSIbold, ANSInormal,
      JpiWsSize, JpiWsSize / 2,
      ANSIbold, ANSInormal,
      JpiWsPeak, JpiWsPeak / 2,
      ANSIbold, ANSInormal,
      JpiVirtPeak, JpiVirtPeak / 2,
      ANSIceol,
      ANSIbold, ANSInormal,
      JpiFilLm - JpiFilCnt, JpiFilLm,
      ANSIbold, ANSInormal,
      JpiPrcCnt, JpiPrcLm,
      ANSIceol);

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
Get the HTTPd server counter information from its logical definition.
*/

int GetAccounting ()

{
   static unsigned short  Length;
   static char  LogicalName [32];
   static $DESCRIPTOR (LnmSystemDsc, "LNM$SYSTEM");
   static $DESCRIPTOR (LogicalNameDsc, LogicalName);
   static struct {
      short BufferLength;
      short ItemCode;
      void  *BufferPtr;
      void  *LengthPtr;
   } LnmItem [] =
   {
      { sizeof(Accounting), LNM$_STRING, &Accounting, &Length },
      {0,0,0,0}
   };

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "GetAccounting()\n");

   if (!LogicalName[0])
      LogicalNameDsc.dsc$w_length =
         sprintf (LogicalName, "HTTPD%d$COUNT", ServerPort);

   status = sys$trnlnm (0, &LnmSystemDsc, &LogicalNameDsc, 0, &LnmItem);
   if (Debug) fprintf (stdout, "sys$trnlnm() %%X%08.08X\n", status);
   if (VMSnok (status) && status != SS$_NOLOGNAM) return (status);
   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
Add the HTTPd server counter information to the screen buffer from the count 
logical.  The information in this logical is binary data in the form of the 
HTTPd count data structure.  This is used to extract each field member from 
the data.
*/

int AddAccounting ()

{
   static unsigned short  Length;
   static char  QuadBytesRx [32],
                QuadBytesTx [32];
   /*
      VAX VMS does not seem to like "!@UQ" and must be made "!@UJ"!!!
      The VAX version not supporting this doesn't seem to be documented,
      but at run-time returns a %X00000014 status (%SYSTEM-F-BADPARAM).
      Implication: VAX version can only report a maximum of 2^32-1 bytes, or
      4,294,967,295 before overflow, AXP 2^64-1 (calculator gave an ERROR :^)
   */
   static $DESCRIPTOR (QuadWordFaoDsc, "!@UJ");
   static $DESCRIPTOR (QuadBytesRxDsc, QuadBytesRx);
   static $DESCRIPTOR (QuadBytesTxDsc, QuadBytesTx);

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "AddAccounting()\n");

   if (VMSnok (status =
       sys$fao (&QuadWordFaoDsc, &Length, &QuadBytesRxDsc,
                &Accounting.QuadBytesRx[0])))
      return (status);
   QuadBytesRx[Length] = '\0';
   if (VMSnok (status =
       sys$fao (&QuadWordFaoDsc, &Length, &QuadBytesTxDsc,
                &Accounting.QuadBytesTx[0])))
      return (status);
   QuadBytesTx[Length] = '\0';

   ScrPtr += sprintf (ScrPtr,
"%s\r\n\
 %sConnect:%s %d  %sAccept:%s %d  %sReject:%s %d\
  %sBusy:%s %d  %sCurrent:%s %d  %sPeak:%s %d\
%s\r\n\
   %sError:%s %d  %sParse:%s %d (%d redirect)  %sForbidden:%s %d  %sRMS:%s %d\
%s\r\n\
     %sGET:%s %d  %sHEAD:%s %d  %sPOST:%s %d\
%s\r\n\
    %sFile:%s %d (%d)  %sMenu:%s %d (%d)  %sDirectory:%s %d\
%s\r\n\
   %ssHTML:%s %d  %sIsMap:%s %d  %sInternal:%s %d\
%s\r\n\
  %sScript:%s %d  %sAuto.Script:%s %d  %sSubprocess:%s %d (%d)\
%s\r\n\
     %s1xx:%s %d  %s2xx:%s %d  %s3xx:%s %d  %s4xx:%s %d  %s5xx:%s %d\
  (%d errors)  %sRedirect:%s %d\
%s\r\n\
      %sRx:%s %s  %sTx:%s %s  (bytes)\
%s\r\n",
      ANSIceol,
      ANSIbold, ANSInormal,
      Accounting.ConnectCount,
      ANSIbold, ANSInormal,
      Accounting.ConnectAcceptedCount,
      ANSIbold, ANSInormal,
      Accounting.ConnectRejectedCount,
      ANSIbold, ANSInormal,
      Accounting.ConnectTooBusyCount,
      ANSIbold, ANSInormal,
      Accounting.ConnectCurrent,
      ANSIbold, ANSInormal,
      Accounting.ConnectPeak,
      ANSIceol,
      ANSIbold, ANSInormal,
      Accounting.RequestErrorCount,
      ANSIbold, ANSInormal,
      Accounting.RequestParseCount,
      Accounting.RedirectLocalCount,
      ANSIbold, ANSInormal,
      Accounting.RequestForbiddenCount,
      ANSIbold, ANSInormal,
      Accounting.RequestRmsErrorCount,
      ANSIceol,
      ANSIbold, ANSInormal,
      Accounting.MethodGetCount,
      ANSIbold, ANSInormal,
      Accounting.MethodHeadCount,
      ANSIbold, ANSInormal,
      Accounting.MethodPostCount,
      ANSIceol,
      ANSIbold, ANSInormal,
      Accounting.DoFileCount,
      Accounting.DoFileNotModifiedCount,
      ANSIbold, ANSInormal,
      Accounting.DoMenuCount,
      Accounting.DoMenuNotModifiedCount,
      ANSIbold, ANSInormal,
      Accounting.DoDirectoryCount,
      ANSIceol,
      ANSIbold, ANSInormal,
      Accounting.DoShtmlCount,
      ANSIbold, ANSInormal,
      Accounting.DoIsMapCount,
      ANSIbold, ANSInormal,
      Accounting.DoInternalCount,
      ANSIceol,
      ANSIbold, ANSInormal,
      Accounting.DoScriptCount,
      ANSIbold, ANSInormal,
      Accounting.DoAutoScriptCount,
      ANSIbold, ANSInormal,
      Accounting.DclExecutedCount,
      Accounting.DclExecutedMoreThanOnceCount,
      ANSIceol,
      ANSIbold, ANSInormal,
      Accounting.ResponseStatusCodeCountArray[1],
      ANSIbold, ANSInormal,
      Accounting.ResponseStatusCodeCountArray[2],
      ANSIbold, ANSInormal,
      Accounting.ResponseStatusCodeCountArray[3],
      ANSIbold, ANSInormal,
      Accounting.ResponseStatusCodeCountArray[4],
      ANSIbold, ANSInormal,
      Accounting.ResponseStatusCodeCountArray[5],
      Accounting.ResponseStatusCodeCountArray[0],
      ANSIbold, ANSInormal,
      Accounting.RedirectRemoteCount,
      ANSIceol,
      ANSIbold, ANSInormal,
      QuadBytesRx,
      ANSIbold, ANSInormal,
      QuadBytesTx,
      ANSIceol);

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
Add the information about the latest request to the screen buffer from the 
request logical.  Each plain-text string in this logical is terminated by a 
null character.
*/ 

int AddRequest ()

{
   static unsigned short  Length;
   static char  LogicalName [32],
                LogicalValue [256];
   static $DESCRIPTOR (LnmSystemDsc, "LNM$SYSTEM");
   static $DESCRIPTOR (LogicalNameDsc, LogicalName);
   struct {
      short BufferLength;
      short ItemCode;
      void  *BufferPtr;
      void  *LengthPtr;
   } LnmItem [] =
   {
      { sizeof(LogicalValue)-1, LNM$_STRING, LogicalValue, &Length },
      {0,0,0,0}
   };

   register int  cnt;
   register char  *cptr, *sptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "AddRequest()\n");

   if (!LogicalName[0])
      LogicalNameDsc.dsc$w_length =
         sprintf (LogicalName, "HTTPD%d$REQUEST", ServerPort);

   Length = 0;
   status = sys$trnlnm (0, &LnmSystemDsc, &LogicalNameDsc, 0, &LnmItem);
   if (Debug) fprintf (stdout, "sys$trnlnm() %%X%08.08X\n", status);
   if (VMSnok (status) && status != SS$_NOLOGNAM) return (status);
   LogicalValue[Length] = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", LogicalValue);

   sptr = ScrPtr;
   cptr = LogicalValue;
   cnt = Length;

   sptr += sprintf (sptr, "%s\r\n", ANSIceol);

   sptr += sprintf (sptr, "    %sTime:%s ", ANSIbold, ANSInormal);
   while (*cptr && cnt--) *sptr++ = *cptr++;
   if (cnt) { cnt--; cptr++; }

   sptr += sprintf (sptr, "  %sHost:%s ",
                    ANSIbold, ANSInormal);
   while (*cptr && cnt--) *sptr++ = *cptr++;
   if (cnt) { cnt--; cptr++; }

   sptr += sprintf (sptr, "%s\r\n  %sStatus:%s ",
                    ANSIceol, ANSIbold, ANSInormal);
   while (*cptr && cnt--) *sptr++ = *cptr++;
   if (cnt) { cnt--; cptr++; }

   sptr += sprintf (sptr, "  %sRx:%s ", ANSIbold, ANSInormal);
   while (*cptr && cnt--) *sptr++ = *cptr++;
   if (cnt) { cnt--; cptr++; }

   sptr += sprintf (sptr, "  %sTx:%s ", ANSIbold, ANSInormal);
   while (*cptr && cnt--) *sptr++ = *cptr++;
   if (cnt) { cnt--; cptr++; }

   sptr += sprintf (sptr, " %s\r\n %sRequest:%s ",
                    ANSIceol, ANSIbold, ANSInormal);
   while (*cptr && cnt--) *sptr++ = *cptr++;
   if (cnt) { cnt--; cptr++; }

   sptr += sprintf (sptr, "%s", ANSIceol);

   ScrPtr = sptr;

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
*/

int OnControlY (void *FunctionAddress)

{
   static boolean  Disabled = false;
   static unsigned long  Mask = LIB$M_CLI_CTRLY,
                         OldMask;
   static unsigned short  TTChannel = 0;

   int  status;
   $DESCRIPTOR (TTDsc, "TT:");

   /*********/
   /* begin */
   /*********/

   if (FunctionAddress)
   {
      if (!TTChannel)
         if (VMSnok (status = sys$assign (&TTDsc, &TTChannel, 0, 0, 0)))
            return (status);

      if (VMSnok (status =
          sys$qiow (0, TTChannel, IO$_SETMODE | IO$M_CTRLYAST, 0, 0, 0,
                    FunctionAddress, 0, PSL$C_USER, 0, 0, 0)))
         return (status);

      if (VMSnok (status =
          sys$qiow (0, TTChannel, IO$_SETMODE | IO$M_CTRLCAST, 0, 0, 0,
                    FunctionAddress, 0, PSL$C_USER, 0, 0, 0)))
         return (status);

      if (!Disabled)
      {
         Disabled = true;
         return (lib$disable_ctrl (&Mask, &OldMask));
      }
      else
         return (status);
   }
   else
   {
      if (VMSnok (status = sys$cancel (TTChannel)))
         return (status);

      if (Disabled)
      {
         Disabled = false;
         return (lib$enable_ctrl (&OldMask, 0));
      }
      else
         return (status);
   }
}

/*****************************************************************************/
/*
*/

ControlY_AST ()

{
   ControlY = true;
   sys$wake (0, 0);
   OnControlY (&ControlY_AST);
}

/*****************************************************************************/
/*
*/

char* TimeString ()

{
   static int  LibDayOfWeek = LIB$K_DAY_OF_WEEK;
   static char  *WeekDays [] =
   {"","Monday","Tuesday","Wednesday","Thursday","Friday","Saturday","Sunday"};
   static char  TimeString [35];
   static $DESCRIPTOR (DayDateTimeDsc, "!AZ, !%D");
   static $DESCRIPTOR (TimeStringDsc, TimeString);

   unsigned long  BinTime [2];
   int  status,
        DayOfWeek;
   unsigned short  Length;

   /*********/
   /* begin */
   /*********/

   sys$gettim (&BinTime);
   lib$cvt_from_internal_time (&LibDayOfWeek, &DayOfWeek, &BinTime);
   sys$fao (&DayDateTimeDsc, &Length, &TimeStringDsc,
            WeekDays[DayOfWeek], &BinTime);
   TimeString[Length-3] = '\0';
   return (TimeString);
}

/****************************************************************************/
/*
*/
 
int ShowHelp ()
 
{
   fprintf (stdout,
"%%%s-I-HELP, usage for the HTTPd Monitor utility (%s)\n\
\n\
Provides a display of the current status of the HyperText Tranfer Protocol\n\
Daemon (HTTPd) process.  Must be executed on a system that has the process\n\
running on it.  Defaults to server associated with IP port number 80.\n\
\n\
$ HTTPDMON [qualifiers ...]\n\
\n\
/HELP /PORT=integer /REFRESH=integer /SWID\n\
\n\
Usage examples:\n\
\n\
$ HTTPDMON\n\
$ HTTPDMON /PORT=8080\n\
$ HTTPDMON /REFRESH=5\n\
\n",
   Utility, SoftwareID);
 
   return (SS$_NORMAL);
}
 
/*****************************************************************************/
/*
*/

char* SysGetMsg (int StatusValue)

{
   static char  Message [256];
   register int  status;
   short int  Length;
   $DESCRIPTOR (MessageDsc, Message);

   if (VMSok (status = sys$getmsg (StatusValue, &Length, &MessageDsc, 0, 0))) 
      Message[Length] = '\0';
   else
      sprintf (Message, "%%sys$getmsg() %%X%08.08X", status);

   return (Message);
}

/****************************************************************************/
/*
Does a case-insensitive, character-by-character string compare and returns 
true if two strings are the same, or false if not.  If a maximum number of 
characters are specified only those will be compared, if the entire strings 
should be compared then specify the number of characters as 0.
*/ 
 
boolean strsame
(
register char *sptr1,
register char *sptr2,
register int  count
)
{
   while (*sptr1 && *sptr2)
   {
      if (toupper (*sptr1++) != toupper (*sptr2++)) return (false);
      if (count)
         if (!--count) return (true);
   }
   if (*sptr1 || *sptr2)
      return (false);
   else
      return (true);
}
 
/****************************************************************************/
/*
This function allows images activated by a "foreign verb" to behave in a way 
that approximates the CLI$ (Command Line Interpreter) utility calls.  Get the 
entire command line following the verb that activated the image.  The command 
line is returned in uppercase, space compressed (i.e. maximum of one space 
between text elements, trimmed of leading and trailing spaces).  Returns a 
warning status if there were no parameters/qualifiers on the command line.
The variable CommandLine is global.
*/ 
 
int ParseCommandLine ()
 
{
   int  status;
   unsigned short  Length;
   unsigned long  Flags = 0;
   struct dsc$descriptor_s 
          CommandLineDsc = { sizeof(CommandLine)-1, DSC$K_DTYPE_T,
                             DSC$K_CLASS_S, CommandLine };
 
   /* get the entire command line following the verb */
   if (VMSnok (status = lib$get_foreign (&CommandLineDsc, 0, &Length, &Flags)))
      return (status);
   CommandLine[Length] = '\0';
 
   if (ParseCommand (CommandLine))
      return (SS$_NORMAL);
   else
      return (STS$K_ERROR | STS$M_INHIB_MSG);
}
 
/****************************************************************************/
/*
This function allows images activated by a "foreign verb" to behave in a way 
that approximates the CLI$ (Command Line Interpreter) utility calls.  Quoted 
strings are always indicated by being parsed to include a single leading 
quote.
*/ 
 
boolean ParseCommand (char *CommandLine)
 
{
   register int  QuoteCount = 0;
   register char  *cptr, *eptr;
   boolean  CommandLineOK = true;
   char  Entity [256] = "";
 
   /* set up any argument defaults */
   ParseCommandEntity (NULL);
 
   cptr = CommandLine;
   eptr = Entity;
 
   for (;;)
   {
      if (*cptr == '\"')
      {
         QuoteCount++;
         *eptr++ = *cptr++;
         continue;
      }
 
      if (QuoteCount & 1 && *cptr)
      {
         /* inside quoted text, copy all characters as literals */
         *eptr++ = *cptr++;
         continue;
      }
 
      if (*cptr == '/' || isspace (*cptr) || !*cptr)
      {
         if (isspace (*cptr))
         {
            /* span the white space */
            while (*cptr && isspace (*cptr)) cptr++;
            if (*cptr == '=')
            {
               /* part of a qualifier, continue to get the value */
               *eptr++ = *cptr++;
               /* span any intervening white space */
               while (*cptr && isspace (*cptr)) cptr++;
               continue;
            }
         }
 
         if (Entity[0])
         {
            *eptr = '\0';
            if (!ParseCommandEntity (Entity)) CommandLineOK = false;
         }
 
         /* if end of command line then break from loop */
         if (!*cptr) break;
 
         /* start of new entity */
         eptr = Entity;
         /* if start of qualifier ensure slash is copied */
         if (*cptr == '/') *eptr++ = *cptr++;
 
         continue;
      }
 
      /* any other character, just copy, ensure upper case */
      *eptr++ = toupper(*cptr++);
   }
 
   return (CommandLineOK);
}
 
/*****************************************************************************/
/*
Get a string value from a qualifier, e.g. '/EXAMPLE=TEST'.
*/
 
boolean ParseCommandString
(
char *Entity,
char *String,
boolean Qualifier,
boolean ReportErrors,
boolean EnsureUpperCase
)
{
   register int  QuoteCount = 0;
   register char  *eptr, *sptr;
 
   if (Debug) fprintf (stdout, "ParseCommandString()\nEntity: '%s'\n", Entity);
 
   eptr = Entity;
 
   if (Qualifier)
   {
      /* scan down to equate symbol */
      while (*eptr && *eptr != '=') eptr++;
      if (*eptr) eptr++;
      if (!*eptr)
      {
         if (ReportErrors)
         {
            fprintf (stdout,
            "%%%s-E-VALREQ, missing qualifier or keyword value\n \\%s\\\n",
            Utility, Entity+1);
         }
         return (false);
      }
   }
 
   sptr = String;
   while (*eptr)
   {
      if (*eptr == '\"')
      {
         if (QuoteCount & 1)
         {
            /* are inside quotes, check for escaped quotes ("") */
            if (*++eptr != '\"')
            {
               /* now outside quotes */
               QuoteCount++;
            }
            /* drop thru to character copy */
         }
         else
         {
            /* now inside quotes */
            QuoteCount++;
            eptr++;
            continue;
         }
      }
 
      if (EnsureUpperCase)
         *sptr++ = toupper(*eptr++);
      else
         *sptr++ = *eptr++;
   }
   *sptr = '\0';
 
   if (Debug) fprintf (stdout, "String: '%s'\n", String);
 
   return (true);
}
 
/*****************************************************************************/
/*
Get an integer value from a qualifier, e.g. '/EXAMPLE=99'.
*/
 
boolean ParseCommandInteger
(
char *Entity,
int *IntegerPtr,
int Base,
boolean ReportErrors
)
{
   register char  *eptr;
   char  *sptr;
 
   if (Debug)
      fprintf (stdout, "ParseCommandInteger() '%s' Base: %d\n", Entity, Base);
 
   for (eptr = Entity; *eptr && *eptr != '='; eptr++);
   if (*eptr) eptr++;
   if (*eptr)
   {
      *IntegerPtr = strtol (eptr, &sptr, Base);
      if (sptr > eptr && !*sptr)
         return (true);
      else
      {
         if (ReportErrors)
         {
            fprintf (stdout,
            "%%%s-E-BADVALUE, '%s' is an invalid keyword value\n",
            Utility, eptr);
         }
         return (false);
      }
   }
   else
   {
      if (ReportErrors)
      {
         fprintf (stdout,
         "%%%s-E-VALREQ, missing qualifier or keyword value\n \\%s\\\n",
         Utility, Entity+1);
      }
      return (false);
   }
}
 
/*****************************************************************************/
/*
A single command line "entity" has been parsed, check if its recognised.  This 
function is the one modified for the individual requirements of each program.
*/
 
boolean ParseCommandEntity (char *Entity)
 
{
   if (Entity == NULL)
   {
      /* set up any argument defaults */
      Debug = DoShowHelp = false;
      RefreshSeconds = 2;
      return (true);
   }
 
   if (Debug) fprintf (stdout, "ParseCommandEntity() Entity: '%s'\n", Entity);
 
   if (Entity[0] == '/')
   {
      /* turns on all "if (Debug)" statements */
      if (strsame (Entity, "/DBUG", -1))
         return (Debug = true);

      if (strsame (Entity, "/HELP", 4))
         return (DoShowHelp = true);

      if (strsame (Entity, "/PORT=", 4))
         return (ParseCommandInteger (Entity, &ServerPort, 10, true));

      if (strsame (Entity, "/REFRESH=", 4))
         return (ParseCommandInteger (Entity, &RefreshSeconds, 10, true));

      fprintf (stdout,
      "%%%s-E-IVQUAL, unrecognised qualifier\n \\%s\\\n", Utility, Entity+1);
      return (false);
   }
 
   fprintf (stdout,
   "%%%s-E-MAXPARM, too many parameters\n \\%s\\\n", Utility, Entity);
   return (false);
}
   
/*****************************************************************************/

