 /*   *   MATRIX DECOMPOSITION MODULE  *
  *   Author:    *       Kenneth S. Kundert   *       UC Berkeley  *  *   Advising Professor:(  *       Alberto Sangiovanni-Vincentelli  *  *L  *   This module contains the routines to decompose the matrix into LU form.  *<  *   >>> User accessable functions contained in this module:  *   OrderAndDecomposeMatrix  *   DecomposeMatrix  *,  *   >>> Functions contained in this module:  *   OrderAndDecomposeMatrix  *   DecomposeMatrix  *   EstimateLargestElement   *   CreateMarkowitzVectors   *   CountMarkowitz   *   MarkowitzProducts  *   SearchForPivot   *   SearchForSingleton   *   QuicklySearchDiagonal  *   SearchDiagonal   *   SearchEntireMatrix   *   FindLargestInCol   *   FindBiggestInColExclude  *   ExchangeRowsAndCols  *   ExchangeRows   *   ExchangeCols   *   ExchangeColElements  *   ExchangeRowElements  *   RealRowColElimination  *   ComplexRowColElimination   *   UpdateMarkowitzNumbers   *   CreateFillIn   *   WriteStatus  */                    /*  *  IMPORTS   *  *  >>> Import decriptions:   *  stdio.h   *     Standard C IO library.   *  MtrxPers.hH  *     Macros that customize the sparse matrix routines. This file comes6  *     to us indirectly, it is imported by MtrxDefs.h.  *  MtrxExpt.h:  *     Macros and declarations to be imported by the user.  *  MtrxDefs.hH  *     Matrix type and macro definitions for the sparse matrix routines.  *  MtrxError.h &  *     Matrix error macro definitions.  */     #include <stdio.h> #include "MtrxExpt.h"  #include "MtrxDefs.h"  #include "MtrxError.h"             /*  *   GLOBAL VARIABLES   *  *   >>> Global variables:  *   gMatrix  (MatrixPointer) 2  *       Global pointer to matrix data structures.$  *   LargestAfterDecompose  (double)G  *       Contains the value of the lergest element in the matrix after    *       decomposition. %  *   LargestBeforeDecompose  (double) J  *       Contains an estimate of the lergest element in the matrix before   *       decomposition.   *   LargestInCol  (double) K  *       Contains the magnitude of the largest element in the column of the 3  *       current pivot, excluding the pivot itself.   *   LargestPivot  (double) )  *       Magnitude of the largest pivot.    *   PivotsOriginalCol  (int) '  *       Column pivot was chosen from.    *   PivotsOriginalRow  (int) $  *       Row pivot was chosen from. !  *   PivotSelectionMethod  (char) L  *       Character that indicates which pivot search method was successful.   *   Singletons  (int)N  *       The number of singletons available for pivoting.  Note that if row I L  *       and column I both contain singletons, only one of them is counted.   *   Size  (int)  *       Size of matrix.    *   SmallestPivot  (double)*  *       Magnitude of the smallest pivot.   *   Step  (int)G  *       Indicates which row and column elimination is currently being    *       performed.   *   Threshold  (double)N  *       The minimum magniude an element must have relative to other elements 3  *       in its column to be considered as a pivot.   *   TrashCan  (MatrixElement)I  *       This is a dummy double that is used to by the user to stuff data J  *       related to the zero row or column.  In other words, when the userL  *       adds an element in row zero or column zero, then the matrix returnsL  *       a pointer to TrashCan.  In this way the user can have a uniform wayM  *       data into the matrix independent of whether a component is connected   *       to ground.   */    /* Begin global definitions. */  MatrixPointer  gMatrix; 8 static double  LargestPivot, SmallestPivot, Threshold;  M static double  LargestInCol, LargestBeforeDecompose, LargestAfterDecompose;   J static int  Size, Step, Singletons, PivotsOriginalRow, PivotsOriginalCol; " static char  PivotSelectionMethod;  struct  MatrixElement  TrashCan;                 /*  *   ORDER AND DECOMPOSE MATRIX   *H  *   This routine chooses a pivot order for the matrix and decomposes itH  *   into LU form.  It handles both the initial decompose and subsequentJ  *   decomposes when a reordering is desired.  This is handled in a manner%  *   that is transparent to the user.   *  *   >>> Returned:C  *   The error code is returned.  Possible errors are listed below.   *  *   >>> Arguments:   *   Matrix  <input>  (char *)  *       Pointer to matrix. !  *   Source  <input>  (double []) N  *       Representitive source vector that is used to determine pivoting orderP  *       when the right hand side vector is sparse.  If Source is a NULL pointerH  *       then the Source vector is assumed to be full and it is not used/  *       when determining the pivoting order.   "  *   lThreshold  <input>  (double)M  *       This number determines what the pivot threshold will be.  It should  P  *       be between zero and one.  If it is one then the pivoting method becomesO  *       complete pivoting, which is very slow and tends to fill up the matrix. P  *       If it is set close to zero the pivoting method becomes strict MarkowitzL  *       with no threshold.  The pivot threshold is used to eliminate pivot P  *       candidates that would cause excessive element growth if they were used.O  *       Element growth is the cause of roundoff error.  Element growth occurs  M  *       even in well-conditioned matrices.  Setting the Threshold large will L  *       reduce element growth and roundoff error, but setting it too large N  *       will cause execution time to be excessive and will result in a large G  *       number of fill-ins.  If this occurs, accuracy can actually be  L  *       degraded because of the large number of operations required on the L  *       matrix due to the large number of fill-ins.  A good value seems to N  *       be 0.001.  The default is chosen by giving a value larger than one orM  *       less than or equal to zero.  This value should be increased and the  K  *       matrix resolved if growth is found to be excessive.  Changing the  O  *       pivot threshold does not improve performance on matrices where growth  F  *       is low, as is often the case with ill-conditioned matrices.  !  *   Growth  <output>  (double *) L  *       Local version of gMatrix->Growth.  Ratio of the largest element in M  *       the matrix after decomposition to the largest element in the matrix  N  *       before decomposition.  Growth is a measure of the amount of roundoff H  *       error incurred during the decomposition. If the growth becomes L  *       large, the pivot threshold should be lowered.  Calculation of this M  *       number requires a significant amount of time, if this number is not  @  *       wanted, the macro MONITOR_GROWTH should be set false.  *  *   PseudoCondition  <output>  (double *)K  *       A very rough and easily fooled estimation of the condition of the  N  *       matrix.  If this number is large, then the matrix is ill-conditioned G  *       and the accuracy of the answer is suspect.  Reducing the pivot P  *       threshold has little effect on this problem.  Ill-conditioning suggestsK  *       that something is wrong with the problem formulation. This number  I  *       is not calculated unless CALCULATE_PSEUDOCONDITION is set true.  *  *   LargestElement  <output>  (double *) M  *       An estimate of the largest element in either the original matrix or  J  *       the decomposed matrix.  Not computed unless MONITOR_GROWTH is set  *       true.  *    *   >>> Local variables:   *   pPivot  (ElementPointer) 6  *       Pointer to the element being used as a pivot."  *   ReorderingRequired  (BOOLEAN)<  *       Flag that indicates whether reordering is required.  *  *   >>> Global variables:  *   gMatrix  <set> "  *   LargestAfterDecompose  <used>#  *   LargestBeforeDecompose  <used>   *   LargestInCol  <set>  *   Size  <set>  *   Step  <set>  *   Threshold  <set>   *  *   >>> Possible errors:   *   NO_MEMORY
  *   RANGE
  *   SINGULAR   *   ILL_CONDITIONED'  *   Error is cleared in this function.   */    int = OrderAndDecomposeMatrix (Matrix, Source, lThreshold, Growth,  <                             PseudoCondition, LargestElement)    register  MatrixPointer  Matrix;I double  Source[], lThreshold, *Growth, *PseudoCondition, *LargestElement;                       { ! register  ElementPointer  pPivot;  int   ReorderingRequired;   ElementPointer SearchForPivot(); double FindLargestInCol(); ELEMENT_MAG_DECLARATIONS;    /* Begin. */     gMatrix = Matrix;      Matrix->Error = NO_ERROR;      if (Matrix == NULL)      {   Matrix->Error = RANGE;         goto  FatalError;      }      Size = Matrix->Size;/     Matrix->Threshold = Threshold = lThreshold;      ReorderingRequired = FALSE; 1     if (Growth == NULL) Growth = &Matrix->Growth; I     if (LargestElement == NULL) LargestElement = &Matrix->LargestElement; L     if (PseudoCondition == NULL) PseudoCondition = &Matrix->PseudoCondition;   #if (MONITOR_GROWTH)     EstimateLargestElement(); "     if (Matrix->Error == SINGULAR)         goto FatalError; #endif  G /* Initialize LargestPivot and SmallestPivot for use in PseudoCondition     calculation.  */      LargestPivot = 0.0; !     SmallestPivot = LARGEST_REAL;          if (NOT Matrix->FirstTime)       { G /* Matrix has been decomposed before and reordering is not required. */ ,         for (Step = 1; Step <= Size; Step++)(         {   pPivot = Matrix->Diag[Step];?             LargestInCol = FindLargestInCol(pPivot->NextInCol); A             if ((LargestInCol * Threshold < ELEMENT_MAG(pPivot))) $             {   if (Matrix->Complex)5                     ComplexRowColElimination(pPivot);                  else2                     RealRowColElimination(pPivot);
             }              else  *             {   ReorderingRequired = TRUE;%                 break; /* for loop */ 
             } 	         } #         if (NOT ReorderingRequired)  	    goto Done;          else	         { M /* A pivot was not large enough to maintain accuracy, so a partial reordering     is required. */  % #if (ANNOTATE >= ON_STRANGE_BEHAVIOR) 6             printf("Reordering,  Step = %1d\n", Step);- #endif                                        	         } ,     } /* End of if(NOT Matrix->FirstTime) */     else     { O /* This is the first time the matrix has been decomposed.  These few statements L    indicate to the rest of the code that a full reodering is required ratherF    than a partial reordering, which occurs during a failure of a fast     decompose. */           Step = 1; #         if (NOT Matrix->RowsLinked)              LinkRows(); !         CreateMarkowitzVectors(); #         if (Matrix->Error >= FATAL)              goto FatalError;     }   & /* Form initial Markowitz products. */     CountMarkowitz(Source);      MarkowitzProducts();  + /* Perform reordering and decomposition. */ +     for (Step = Step; Step <= Size; Step++) "     {   pPivot = SearchForPivot();&         if (Matrix->Error == SINGULAR)             goto FatalError;$         ExchangeRowsAndCols(pPivot);           if (Matrix->Complex)-             ComplexRowColElimination(pPivot);          else*             RealRowColElimination(pPivot);  #         if (Matrix->Error >= FATAL)              goto FatalError;'         UpdateMarkowitzNumbers(pPivot);    #if (ANNOTATE == FULL)           WriteStatus(); #endif     }    Done:      Matrix->FirstTime = FALSE;     Matrix->Decomposed = TRUE;   #if (MONITOR_GROWTH)L     Matrix->Growth = *Growth = LargestAfterDecompose/LargestBeforeDecompose;E     Matrix->LargestElement = *LargestElement = LargestAfterDecompose;  #else #     Matrix->Growth = *Growth = 0.0; 3     Matrix->LargestElement = *LargestElement = 0.0;  #endif #if (CALCULATE_PSEUDOCONDITION) N     Matrix->PseudoCondition = *PseudoCondition = LargestPivot / SmallestPivot; #else 5     Matrix->PseudoCondition = *PseudoCondition = 0.0;  #endif       return(Matrix->Error);   FatalError: ,     Matrix->Growth = *Growth = LARGEST_REAL;>     Matrix->PseudoCondition = *PseudoCondition = LARGEST_REAL;     return (Matrix->Error);  }                          /*  *   DECOMPOSE MATRIX   *N  *   This routine is the companion routine to OrderAndDecomposeMatrix.  UnlikeM  *   OrderAndDecomposeMatrix, DecomposeMatrix cannot change the ordering.  It L  *   is slightly faster, but its main utility is stems from the fact that itH  *   will not change the ordering if a small pivot is encountered.  The <  *   standard way to use these two routines is to first use H  *   OrderAndDecomposeMatrix for the initial decompose.  For subsequent K  *   decomposes, DecomposeMatrix is used.  The value of Growth returned be  B  *   MatrixDecompose is monitored, and if it becomes significantly;  *   larger than the value of Growth initially returned by  L  *   OrderAndDecomposeMatrix, then the matrix should be reordered by callingL  *   OrderAndDecomposeMatrix.  If DecomposeMatrix is called for the initial >  *   decompose of the matrix, then it will automatically call 8  *   OrderAndDecomposeMatrix with the default threshold.  *  *   >>> Returned:C  *   The error code is returned.  Possible errors are listed below.   *  *   >>> Arguements:  *   Matrix  <input>  (char *)  *       Pointer to matrix. !  *   Growth  <output>  (double *) L  *       Local version of gMatrix->Growth.  Ratio of the largest element in M  *       the matrix after decomposition to the largest element in the matrix  N  *       before decomposition.  Growth is a measure of the amount of roundoff O  *       error incurred during the decomposition. If the growth becomes large,  L  *       the pivot threshold should be lowered.  Calculation of this number N  *       requires a significant amount of time, if this number is not wanted, 8  *       the macro MONITOR_GROWTH should be set false.  *  *   PseudoCondition  <output>  (double *)K  *       A very rough and easily fooled estimation of the condition of the  N  *       matrix.  If this number is large, then the matrix is ill-conditioned G  *       and the accuracy of the answer is suspect.  Reducing the pivot P  *       threshold has little effect on this problem.  Ill-conditioning suggestsK  *       that something is wrong with the problem formulation. This number  I  *       is not calculated unless CALCULATE_PSEUDOCONDITION is set true.  *  *   LargestElement  <output>  (double *) M  *       An estimate of the largest element in either the original matrix or  J  *       the decomposed matrix.  Not computed unless MONITOR_GROWTH is set  *       true.  *    *   >>> Local variables:   *   pPivot  (ElementPointer) 6  *       Pointer to the element being used as a pivot.  *  *   >>> Global variables:  *   gMatrix  <set> "  *   LargestAfterDecompose  <used>#  *   LargestBeforeDecompose  <used>   *   LargestInCol  <set>  *   Size  <set>  *   Step  <set>  *  *   >>> Possible errors: 
  *   RANGE
  *   SINGULAR '  *   Error is cleared in this function.   */    int A DecomposeMatrix (Matrix, Growth, PseudoCondition, LargestElement)     register  MatrixPointer  Matrix;3 double  *Growth, *PseudoCondition, *LargestElement;                       { ! register  ElementPointer  pPivot;  double FindLargestInCol(); ELEMENT_MAG_DECLARATIONS;    /* Begin. */     gMatrix = Matrix;      Matrix->Error = NO_ERROR;      if (Matrix == NULL)      {   Matrix->Error = RANGE;         goto FatalError;     }      if (Matrix->FirstTime)P         return(OrderAndDecomposeMatrix(Matrix, NULL, DEFAULT_THRESHOLD, Growth, O                                              PseudoCondition, LargestElement)); 1     if (Growth == NULL) Growth = &Matrix->Growth; I     if (LargestElement == NULL) LargestElement = &Matrix->LargestElement; L     if (PseudoCondition == NULL) PseudoCondition = &Matrix->PseudoCondition;   #if (MONITOR_GROWTH)     EstimateLargestElement();  #endif   #if (CALCULATE_PSEUDOCONDITION) G /* Initialize LargestPivot and SmallestPivot for use in PseudoCondition   * calculation.  */      LargestPivot = 0.0; !     SmallestPivot = LARGEST_REAL;  #endif       Size = Matrix->Size;     if (Matrix->Complex),     {   for (Step = 1; Step <= Size; Step++)(         {   pPivot = Matrix->Diag[Step]; #if (MONITOR_GROWTH)?             LargestInCol = FindLargestInCol(pPivot->NextInCol);  #endif-             ComplexRowColElimination(pPivot); 	         }      }      else,     {   for (Step = 1; Step <= Size; Step++)(         {   pPivot = Matrix->Diag[Step]; #if (MONITOR_GROWTH)?             LargestInCol = FindLargestInCol(pPivot->NextInCol);  #endif*             RealRowColElimination(pPivot);	         }      }        Matrix->Decomposed = TRUE;   #if (MONITOR_GROWTH)L     Matrix->Growth = *Growth = LargestAfterDecompose/LargestBeforeDecompose;E     Matrix->LargestElement = *LargestElement = LargestAfterDecompose;  #else #     Matrix->Growth = *Growth = 0.0; 3     Matrix->LargestElement = *LargestElement = 0.0;  #endif #if (CALCULATE_PSEUDOCONDITION) N     Matrix->PseudoCondition = *PseudoCondition = LargestPivot / SmallestPivot; #else 5     Matrix->PseudoCondition = *PseudoCondition = 0.0;  #endif       if (Matrix->Error < FATAL)         return(Matrix->Error);   FatalError: ,     Matrix->Growth = *Growth = LARGEST_REAL;>     Matrix->PseudoCondition = *PseudoCondition = LARGEST_REAL;     return(Matrix->Error); }                      /*8  *   ESTIMATE THE LARGEST ELEMENT IN THE ORIGINAL MATRIX  *N  *   Scans Matrix to determine the approximate magnitude of the largest of theO  *   previously chosen pivots.  If this is the first time the decomposition is  J  *   run then the diagonal elements are assumed to be the pivots.  If all L  *   previously chosen pivots are zero then the whole matrix is scanned and 9  *   the magnitude of the largest element is determined.    *  *   >>> Local variables:   *   Largest  (double)2  *       Absolute value of largest element found.   *   Magnitude  (double)-  *       Absolute value of diagonal element.    *   pElement  (ElementPointer) L  *       Points to elements in the matrix and is used to scan matrix to find  *       largest element.     *   ppDiag  (ElementPointer *) K  *       Pointer into the array of pointers to the Diagonal elements.  This @  *       variable is used to quickly scan the diagonal elements.  *  *   >>> Global Varaibles:  *   gMatrix  <used>!  *   LargestAfterDecompose  <set> "  *   LargestBeforeDecompose  <set>  *   LargestPivot  <set>  *   Size  <used>   *   SmallestPivot  <set>   *  *   >>> Possible errors:   *   SINGULAR   */    static EstimateLargestElement()   { . register  ElementPointer  pElement, *ppDiag;   register  int  I;  double  Magnitude, Largest;  ELEMENT_MAG_DECLARATIONS;    /* Begin. */! /* Find largest pivot element. */      Largest = 0.0;  . /* Use ppDiag to scan through the diagonals.*/     ppDiag = gMatrix->Diag;      for (I = 1; I <= Size; I++)       {   if (*(++ppDiag) != NULL)-         {   Magnitude = ELEMENT_MAG(*ppDiag); $             if (Largest < Magnitude)$                 Largest = Magnitude;	         }      }         if (Largest != 0.0)      { O /* Just to avoid trouble on some degenerate matrices with very small diagonals, @  * scan first column in search for largest element in matrix. */*         pElement = gMatrix->FirstInCol[1];          while (pElement != NULL).         {   Magnitude = ELEMENT_MAG(pElement);+             pElement = pElement->NextInCol; $             if (Largest < Magnitude)$                 Largest = Magnitude;	         }   J /* Assume that matrix is diagonally dominant so that largest element is on    diagonal.  */)         LargestBeforeDecompose = Largest;      }      else     {   B /* All previously chosen pivots are zero; search entire matrix for    largest element. */#         for (I = 1; I <= Size; I++) .         {   pElement = gMatrix->FirstInCol[I];$             while (pElement != NULL)2             {   Magnitude = ELEMENT_MAG(pElement);/                 pElement = pElement->NextInCol; (                 if (Largest < Magnitude)(                     Largest = Magnitude;
             } 	         } )         LargestBeforeDecompose = Largest;      }   N /* Set LargestAfterDecompose equal to LargestBeforeDecompose so Growth >= 1 */3     LargestAfterDecompose = LargestBeforeDecompose;                   if (Largest == 0.0) #     {   gMatrix->Error = SINGULAR;           return;      }   	                return;  }                                 /*1  *   INITIALIZE COLUMN AND ROW ELIMINATED VECTORS   *O  *   Initalizes the vectors that keep track of which columns and rows have been "  *   eliminated and in what order.  *  *   >>> Local variables:   *   SizePlusOne  (unsigned),  *       Size of the arrays to be allocated.  *   SizePlusTwo  (unsigned)H  *       This is the size of the MarkowitzProduct array.  The two extra &  *       elements are needed because: E  *       1) arrays in C start with index 0 but first index used is 1. M  *       2) the last element in the array is set to zero to act as an end of  +  *           array flag for a while loop.     *  *   >>> Global variables:  *   gMatrix  <used>  *   Size  <used>   *  *   >>> Possible errors:   *   NO_MEMORY    */    static CreateMarkowitzVectors()   { # unsigned  SizePlusOne, SizePlusTwo;    /* Begin. */ /* Create Markowitz arrays. */'     SizePlusOne = (unsigned)(Size + 1); '     SizePlusTwo = (unsigned)(Size + 2); E     if (( gMatrix->MarkowitzRow = MALLOC(int, SizePlusOne))  == NULL) #         gMatrix->Error = NO_MEMORY; E     if (( gMatrix->MarkowitzCol = MALLOC(int, SizePlusOne))  == NULL) #         gMatrix->Error = NO_MEMORY; G     if (( gMatrix->MarkowitzProd = MALLOC(long, SizePlusTwo))  == NULL) #         gMatrix->Error = NO_MEMORY;   9 /* Create Intermediate vectors for use in MatrixSolve. */ J     if (( gMatrix->IntermediateReal = MALLOC(double,SizePlusOne)) == NULL)#         gMatrix->Error = NO_MEMORY;        if (gMatrix->Complex) N     {   if (( gMatrix->IntermediateImag = MALLOC(double,SizePlusOne)) == NULL)'             gMatrix->Error = NO_MEMORY;      }   )     gMatrix->InitialDecompStarted = TRUE;      return;  }          /*  *   COUNT MARKOWITZ    *L  *   Scans Matrix to determine the Markowitz counts for each row and column.  *  *   >>> Argument:!  *   Source  <input>  (double []) N  *       Representitive source vector that is used to determine pivoting orderP  *       when the right hand side vector is sparse.  If Source is a NULL pointerH  *       then the Source vector is assumed to be full and it is not used.  *       when determining the pivoting order.   *  *   >>> Local variables:   *   Count  (int) &  *       Temporary counting variable.   *   pElement  (ElementPointer) %  *       Pointer to matrix elements.    *  *   >>> Global variables:  *   gMatrix - (used)   *   Size - (used)  *   Step - (used)  */    static CountMarkowitz( Source ) register  double  Source[];                       {  register  int Count, I; # register  ElementPointer  pElement;    /* Begin. */  - /* Correct array pointer for ARRAY_OFFSET. */  #if (NOT ARRAY_OFFSET). #if (SEPARATED_COMPLEX_VECTORS OR NOT COMPLEX)         if (Source != NULL)              --Source;  #else          if (Source != NULL) !             if (gMatrix->Complex)                  Source -= 2;	 	    else                  --Source;  #endif #endif  / /* Generate MarkowitzRow Count for each row. */ $     for (I = Step; I <= Size; I++)       { E /* Set Count to -1 initially to remove count due to pivot element. */          Count = -1; *         pElement = gMatrix->FirstInRow[I];          while (pElement != NULL)         {   Count++;+             pElement = pElement->NextInRow; 	         }   4 /* Include nonzero elements in the Source vector. */. #if (SEPARATED_COMPLEX_VECTORS OR NOT COMPLEX)         if (Source != NULL) !         {   if (Source[I] != 0.0)                  Count++;	         }  #else          if (Source != NULL) !         {   if (gMatrix->Complex) D             {   if ((Source[2*I] != 0.0) OR (Source[2*I+1] != 0.0))  		    Count++; 	    }	 	    else %             {   if (Source[I] != 0.0)  		    Count++; 	    }	         }  #endif)         gMatrix->MarkowitzRow[I] = Count;      }   6 /* Generate the MarkowitzCol count for each column. */"     for (I = Step; I <= Size; I++)     { E /* Set Count to -1 initially to remove count due to pivot element. */          Count = -1; *         pElement = gMatrix->FirstInCol[I];          while (pElement != NULL)         {   Count++;+             pElement = pElement->NextInCol; 	         } )         gMatrix->MarkowitzCol[I] = Count;      }      return;  }                      /*  *   MARKOWITZ PRODUCTS   *M  *   Calculates MarkowitzProduct for each diagonal element from the Markowitz   *   counts.  *  *   Local Varaibles:    *   pMarkowtizProduct  (long *)E  *       Pointer that points into MarkowitzProdoct array. Is used to  -  *       sequentially access entries quickly.   *   pMarkowtizRow  (int *) N  *       Pointer that points into MarkowitzRow array. Is used to sequentially    *       access entries quickly.  *   pMarkowtizCol  (int *) M  *       Pointer that points into MarkowitzCol array. Is used to sequentially    *       access entries quickly.  *   Product  (long)2  *       Temporary storage for Markowitz product./  *  *   >>> Global variables:  *   gMatrix  <used>  *   Singletons  <set>  *   Size  <used>   *   Step  <used>   */    static MarkowitzProducts()       { 1 register  int  I, *pMarkowitzRow, *pMarkowitzCol; , register  long  Product, *pMarkowitzProduct;   /* Begin. */     Singletons = 0;   8     pMarkowitzProduct = &(gMatrix->MarkowitzProd[Step]);3     pMarkowitzRow = &(gMatrix->MarkowitzRow[Step]); 3     pMarkowitzCol = &(gMatrix->MarkowitzCol[Step]);   "     for (I = Step; I <= Size; I++)6     {   Product = *pMarkowitzRow++ * *pMarkowitzCol++;2         if ((*pMarkowitzProduct++ = Product) == 0)             Singletons++;      }      return;  }                        /*  *   SEARCH FOR BEST PIVOT  *J  *   Performs a search to determine the element with the lowest Markowitz K  *   Product that is also acceptable.  An acceptable element is one that is O  *   larger than the AbsoluteThreshold and at least as large as Threshold times L  *   the largest element in the same column.  The first step is to look for M  *   singletons if any exist.  If none are found, then all the diagonals are  N  *   searched. The diagonal is searched once quickly using the assumption thatK  *   elements on the diagonal are large compared to other elements in their O  *   column, and so the pivot can be chosen soley on the basis of the Markowitz L  *   criterion.  After a element has been chosen to be pivot on the basis ofI  *   its Markowitz product, it is checked to see if it is large enough.   L  *   Waiting to the end of the Markowitz search to check the size of a pivotI  *   candidate saves considerable time, but is not guatanteed to find an  M  *   acceptable pivot.  Thus if unsucessful a second pass of the diagonal is  P  *   made.  This second pass checks to see if an element is large enough during K  *   the search, not after it.  If still no acceptable pivot candidate has  ?  *   been found, the search expands to cover the entire matrix.   *  *   >>> Returned:K  *   A pointer to the element chosen to be pivot.  If every element in the  +  *   matrix is zero, then NULL is returned.   *  *   >>> Local variables: "  *   ChosenPivot  (ElementPointer)A  *       Pointer to element that has been chosen to be the pivot.   *  *   >>> Global variables:  *   gMatrix  <used>!  *   PivotSelectionMethod  <set>    *   Singletons <used>  *  *   >>> Possible errors: 
  *   SINGULAR   *   ILL_CONDITIONED    */    static ElementPointer    SearchForPivot()     { % register ElementPointer  ChosenPivot; % ElementPointer  SearchForSingleton(); ( ElementPointer  QuicklySearchDiagonal();! ElementPointer  SearchDiagonal(); % ElementPointer  SearchEntireMatrix();    /* Begin. */  G /*  If singletons exist, look for an acceptable one to use as pivot. */      if (Singletons) +     {   ChosenPivot = SearchForSingleton();           if (ChosenPivot != NULL)'         {   PivotSelectionMethod = 's';               return(ChosenPivot);	         }      }   L /*  Either no singletons exist or they weren't acceptable.  Take quick first"  *  pass at searching diagonal. */*     ChosenPivot = QuicklySearchDiagonal();     if (ChosenPivot != NULL)#     {   PivotSelectionMethod = 'q';          return(ChosenPivot);     }   B /*  Quick search of diagonal failed, carefully search diagonal. */#     ChosenPivot = SearchDiagonal();      if (ChosenPivot != NULL)#     {   PivotSelectionMethod = 'd';          return(ChosenPivot);     }   ; /*  No acceptable pivot found yet, search entire matrix. */ '     ChosenPivot = SearchEntireMatrix();      PivotSelectionMethod = 'e';        return(ChosenPivot);   }                            /*)  *   SEARCH FOR SINGLETON TO USE AS PIVOT   *K  *   Performs a search to find a singleton to use as the pivot.  The first  N  *   acceptable singleton is used.  A singleton is acceptable if it is larger O  *   in magnitude than the AbsoluteThreshold.  A singleton does not need to be  $  *   on the diagonal to be selected.  *  *   >>> Returned:H  *   A pointer to the singleton chosen to be pivot.  In no singleton is   *   acceptable, return NULL.   *  *   >>> Local variables: $  *   ChosenPivot  (ElementPointer)  B  *       Pointer to element that has been chosen to be the pivot.   *   lSingletons  (int) J  *       A local version of Singletons, which is a count of the number of /  *       singletons that can be used as pivots.    *   pMarkowitzProduct  (long *)P  *       Pointer that points into MarkowitzProduct array. It is used to quickly .  *       access succesive Markowitz products.   *  *   >>> Global variables:  *   gMatrix  <used>  *   Size  <used>   *   Singletons  <set>  *   Step  <used>   */    static ElementPointer    SearchForSingleton()     { ( register  ElementPointer  ChosenPivot;   register  int  I; % register  long  *pMarkowitzProduct;    int  lSingletons;  ELEMENT_MAG_DECLARATIONS;    /* Begin. */I /* Initialize pointer that is to scan through MarkowitzProduct vector. */ 8     pMarkowitzProduct = &(gMatrix->MarkowitzProd[Step]);  I /* Decrement the count of available singletons, on the assumption that an #    acceptable one will be found. */      lSingletons = Singletons--;   G /* Assure that following while loop will always terminate, this is just G    preventative medicine, if things are working right this should never     be needed. */+         gMatrix->MarkowitzProd[Size+1] = 0;        while (lSingletons-- > 0)      { " /* Singletons exist, find them. */  H /* This is tricky.  Am using a pointer to sequentially step through the J  * MarkowitzProduct array.  Search terminates when singleton (Product = 0)>  * is found.  Note that the conditional in the while statementH  * ( *pMarkowitzProduct ) is true as long as the MarkowitzProduct is notF  * equal to zero.  The row (and column) index on the diagonal is then O  * calculated by subtracting the pointer to the Markowitz product of the first  N  * diagonal from the pointer to the Markowitz product of the desired element,   * the singleton. */  &         while ( *pMarkowitzProduct++ )'         {   /* Just passing through. */ 	         } ;         I = pMarkowitzProduct - gMatrix->MarkowitzProd - 1;    /* Assure that I is valid. */          if (I > Size) 3             break;  /* while (lSingletons-- > 0) */   B /* Singleton has been found in either/both row or/and column I. */5         if ((ChosenPivot = gMatrix->Diag[I]) != NULL) 	         { % /* Singleton lies on the diagonal. */ F             if (ELEMENT_MAG(ChosenPivot) > gMatrix->AbsoluteThreshold)$                 return(ChosenPivot);	         }          else	         { 2 /* Singleton does not lie on diagonal, find it. */.             if (gMatrix->MarkowitzCol[I] == 0)5             {   ChosenPivot = gMatrix->FirstInCol[I]; K                 while ((ChosenPivot != NULL) AND (ChosenPivot->Row < Step)) 9                     ChosenPivot = ChosenPivot->NextInCol; J                 if (ELEMENT_MAG(ChosenPivot) > gMatrix->AbsoluteThreshold)(                     return(ChosenPivot);                 else6                 {   if (gMatrix->MarkowitzRow[I] == 0)=                     {   ChosenPivot = gMatrix->FirstInRow[I]; P                         while((ChosenPivot != NULL) AND (ChosenPivot->Col<Step))A                             ChosenPivot = ChosenPivot->NextInRow; R                         if (ELEMENT_MAG(ChosenPivot) > gMatrix->AbsoluteThreshold)0                             return(ChosenPivot);                     }                  } 
             }              else5             {   ChosenPivot = gMatrix->FirstInRow[I]; K                 while ((ChosenPivot != NULL) AND (ChosenPivot->Col < Step)) 9                     ChosenPivot = ChosenPivot->NextInRow; J                 if (ELEMENT_MAG(ChosenPivot) > gMatrix->AbsoluteThreshold)(                     return(ChosenPivot);
             } 	         } 8 /* Singleton not acceptable (too small), try another. */'     } /* end of while(lSinglatons>0) */   ) /* All singletons were not acceptable. */ L /* Restore Singletons count. Initial assumption that an acceptable singleton    would be found was wrong. */      Singletons++;      return(NULL);  }                                   #if (MODIFIED_MARKOWITZ) /*I  *   QUICK SEARCH OF DIAGONAL FOR PIVOT WITH MODIFIED MARKOWITZ CRITERION   *J  *   Searches the diagonal looking for the best pivot.  For a pivot to be L  *   acceptable it must be larger than the pivot Threshold times the largestJ  *   element in its reduced column.  Amoung the acceptable diagonals, the J  *   one with the smallest MarkowitzProduct is sought.  Search terminates I  *   early if a diagonal is found with a MarkowitzProduct of one and its  I  *   magnitude is larger than the other elements in its row and column.   K  *   Since its MarkowitzProduct is one, there is only one other element in  I  *   both its row and column, and, as a condition for early termination,  I  *   these elements must be located symmetricly in the matrix.  If a tie  M  *   occurs between elements of equal MarkowitzProduct, then the element with L  *   the largest ratio between its magnitude and the largest element in its L  *   column is used.  The search will be terminated after a given number of M  *   ties have occured and the best (largest ratio) of the tied element will  J  *   be used as the pivot.  The number of ties that will trigger an early :  *   termination is MinMarkowitzProduct * TIES_MULTIPLIER.  *  *   >>> Returned:M  *   A pointer to the diagonal element chosen to be pivot.  If no diagonal is $  *   acceptable, a NULL is returned.  *  *   >>> Local variables: "  *   ChosenPivot  (ElementPointer)E  *       Pointer to the element that has been chosen to be the pivot. %  *   LargestInColOfAccepted  (double) K  *       The magnitude of the largest element in the column (excluding the  L  *       diagonal) for the column that contains the favored pivot candidate.!  *   LargestOffDiagonal  (double) L  *       Magnitude of the largest of the off-diagonal terms associated with 8  *       a diagonal with MarkowitzProduct equal to one.   *   Magnitude  (double),  *       Absolute value of diagonal element.  *   MaxRatio  (double) N  *       Among the elements tied with the smallest Markowitz product, MaxRatioO  *       is the best (smallest) ratio of LargestInCol to the diagonal Magnitude J  *       found so far.  The smaller the ratio, the better numerically the "  *       element will be as pivot.   *   MinMarkowitzProduct  (long)L  *       Smallest Markowitz product found of pivot candidates that lie along  *       diagonal.  *   NumberOfTies  (int)M  *       A count of the number of Markowitz ties that have occured at current   *       MarkowitzProduct.  *   pDiag  (ElementPointer)  .  *       Pointer to current diagonal element.    *   pMarkowitzProduct  (long *)P  *       Pointer that points into MarkowitzProduct array. It is used to quickly .  *       access succesive Markowitz products.   *   Ratio  (double)L  *       For the current pivot candidate, Ratio is the ratio of the largest D  *       element in its column (excluding itself) to its magnitude. %  *   TiedElements  (ElementPointer[]) E  *       Array of pointers to the elements with the minimun Markowitz   *       product. "  *   pOtherInCol  (ElementPointer)I  *       When there is only one other element in a column other than the  L  *       diagonal, pOtherInCol is used to point to it.  Used when Markowitz I  *       product is to determine if off diagonals are placed symmetricly. "  *   pOtherInRow  (ElementPointer)O  *       When there is only one other element in a row other than the diagonal, L  *       pOtherInRow is used to point to it.  Used when Markowitz product is>  *       to determine if off diagonals are placed symmetricly.  *  *   >>> Global variables:  *   gMatrix  <used>  *   LargestInCol  <set>  *   Size  <used>   *   Step  <used>   *   Threshold  <used>  */    static ElementPointer    QuicklySearchDiagonal()    { 7 register long  MinMarkowitzProduct, *pMarkowitzProduct; ; register  ElementPointer  pDiag, pOtherInRow, pOtherInCol;   int  I, NumberOfTies; B ElementPointer  ChosenPivot, TiedElements[MAX_MARKOWITZ_TIES + 1];O double  Magnitude, Ratio, MaxRatio, LargestOffDiagonal, LargestInColOfAccepted; " double  FindBiggestInColExclude(); ELEMENT_MAG_DECLARATIONS;    /* Begin. */     NumberOfTies = -1;/     MinMarkowitzProduct = LARGEST_LONG_INTEGER; :     pMarkowitzProduct = &(gMatrix->MarkowitzProd[Step-1]);  = /* Assure that following while loop will always terminate. */ (     gMatrix->MarkowitzProd[Size+1] = -1;  O /* This is tricky.  Am using a pointer in the inner while loop to sequentially  H  * step through the MarkowitzProduct array.  Search terminates when the O  * Markowitz product of zero placed at location Size+1 is found.  The row (and  O  * column) index on the diagonal is then calculated by subtracting the pointer  J  * to the Markowitz product of the first diagonal from the pointer to the L  * Markowitz product of the desired element. The outer for loop is infinite #  * and is broken by using break. */   $     for(;;)  /* Endless for loop. */<     {   while (MinMarkowitzProduct < *(++pMarkowitzProduct))'         {   /* Just passing through. */ 	         }   7         I = pMarkowitzProduct - gMatrix->MarkowitzProd;   < /* Assure that I is valid; if I > Size, terminate search. */         if (I > Size) )             break; /* Endless for loop */   /         if ((pDiag = gMatrix->Diag[I]) == NULL) ,             continue; /* Endless for loop */K         if ((Magnitude = ELEMENT_MAG(pDiag)) <= gMatrix->AbsoluteThreshold) ,             continue; /* Endless for loop */  $         if (*pMarkowitzProduct == 1)	         { O /* Case where only one element exists in row and column other than diagonal. */   ! /* Find off diagonal elements. */ +             pOtherInRow = pDiag->NextInRow; +             pOtherInCol = pDiag->NextInCol; <             if (pOtherInRow == NULL AND pOtherInCol == NULL)6             {    pOtherInRow = gMatrix->FirstInRow[I];+                  while(pOtherInRow != NULL) L                  {   if (pOtherInRow->Col >= Step AND pOtherInRow->Col != I)                          break; :                      pOtherInRow = pOtherInRow->NextInRow;                  }6                  pOtherInCol = gMatrix->FirstInCol[I];+                  while(pOtherInCol != NULL) L                  {   if (pOtherInCol->Row >= Step AND pOtherInCol->Row != I)                          break; :                      pOtherInCol = pOtherInCol->NextInCol;                  }
             }   L /* Accept diagonal as pivot if diagonal is larger than off diagonals and the+  * off diagonals are placed symmetricly. */ >             if (pOtherInRow != NULL  AND  pOtherInCol != NULL)9             {   if (pOtherInRow->Col == pOtherInCol->Row) F                 {   LargestOffDiagonal = MAX(ELEMENT_MAG(pOtherInRow),P                                                       ELEMENT_MAG(pOtherInCol));8                     if (Magnitude >= LargestOffDiagonal)                     { > /* Accept pivot, it is unlikely to contribute excess error. */@                         LargestInCol = ELEMENT_MAG(pOtherInCol);&                         return(pDiag);                     }                  } 
             } 	         }   5         if (*pMarkowitzProduct < MinMarkowitzProduct) 	         { P /* Notice strict inequality in test. This is a new smallest MarkowitzProduct. */$             TiedElements[0] = pDiag;5             MinMarkowitzProduct = *pMarkowitzProduct;              NumberOfTies = 0; 	         }          else	         { ' /* This case handles Markowitz ties. */ 2             if (NumberOfTies < MAX_MARKOWITZ_TIES)5             {   TiedElements[++NumberOfTies] = pDiag; J                 if (NumberOfTies >= MinMarkowitzProduct * TIES_MULTIPLIER)1                     break; /* Endless for loop */ 
             } 	         } $     } /* End of endless for loop. */  A /* Test to see if any element was chosen as a pivot candidate. */      if (NumberOfTies < 0)          return(NULL);   ; /* Determine which of tied elements is best numerically. */      ChosenPivot = NULL;      MaxRatio = 1.0 / Threshold;   '     for (I = 0; I <= NumberOfTies; I++)       {   pDiag = TiedElements[I];'         Magnitude = ELEMENT_MAG(pDiag); 6         LargestInCol = FindBiggestInColExclude(pDiag);)         Ratio = LargestInCol / Magnitude;          if (Ratio < MaxRatio)           {   ChosenPivot = pDiag;             MaxRatio = Ratio; 2             LargestInColOfAccepted = LargestInCol;	         }      } *     LargestInCol = LargestInColOfAccepted;     return(ChosenPivot); }                     # #else  /* Not MODIFIED_MARKOWITZ */  /*D  *   QUICK SEARCH OF DIAGONAL FOR PIVOT WITH CONVENTIONAL MARKOWITZ   *   CRITERION  *J  *   Searches the diagonal looking for the best pivot.  For a pivot to be L  *   acceptable it must be larger than the pivot Threshold times the largestJ  *   element in its reduced column.  Amoung the acceptable diagonals, the J  *   one with the smallest MarkowitzProduct is sought.  Search terminates I  *   early if a diagonal is found with a MarkowitzProduct of one and its  I  *   magnitude is larger than the other elements in its row and column.   K  *   Since its MarkowitzProduct is one, there is only one other element in  I  *   both its row and column, and, as a condition for early termination,  @  *   these elements must be located symmetricly in the matrix.    *  *   >>> Returned:M  *   A pointer to the diagonal element chosen to be pivot.  If no diagonal is $  *   acceptable, a NULL is returned.  *  *   >>> Local variables: "  *   ChosenPivot  (ElementPointer)E  *       Pointer to the element that has been chosen to be the pivot. !  *   LargestOffDiagonal  (double) L  *       Magnitude of the largest of the off-diagonal terms associated with 8  *       a diagonal with MarkowitzProduct equal to one.   *   Magnitude  (double),  *       Absolute value of diagonal element.   *   MinMarkowitzProduct  (long)H  *       Smallest Markowitz product found of pivot candidates which are   *       acceptable.  *   pDiag  (ElementPointer)  .  *       Pointer to current diagonal element.    *   pMarkowitzProduct  (long *)P  *       Pointer that points into MarkowitzProduct array. It is used to quickly .  *       access succesive Markowitz products. "  *   pOtherInCol  (ElementPointer)I  *       When there is only one other element in a column other than the  L  *       diagonal, pOtherInCol is used to point to it.  Used when Markowitz I  *       product is to determine if off diagonals are placed symmetricly. "  *   pOtherInRow  (ElementPointer)O  *       When there is only one other element in a row other than the diagonal, L  *       pOtherInRow is used to point to it.  Used when Markowitz product is>  *       to determine if off diagonals are placed symmetricly.  *  *   >>> Global variables:  *   gMatrix  <used>  *   LargestInCol  <set>  *   Size  <used>   *   Step  <used>   *   Threshold  <used>  */    static ElementPointer    QuicklySearchDiagonal()    { 7 register long  MinMarkowitzProduct, *pMarkowitzProduct;   register  ElementPointer  pDiag; int  I; 6 ElementPointer  ChosenPivot, pOtherInRow, pOtherInCol;& double  Magnitude, LargestOffDiagonal;" double  FindBiggestInColExclude(); ELEMENT_MAG_DECLARATIONS;    /* Begin. */     ChosenPivot = NULL; /     MinMarkowitzProduct = LARGEST_LONG_INTEGER; :     pMarkowitzProduct = &(gMatrix->MarkowitzProd[Step-1]);  = /* Assure that following while loop will always terminate. */ (     gMatrix->MarkowitzProd[Size+1] = -1;  O /* This is tricky.  Am using a pointer in the inner while loop to sequentially  H  * step through the MarkowitzProduct array.  Search terminates when the O  * Markowitz product of zero placed at location Size+1 is found.  The row (and  O  * column) index on the diagonal is then calculated by subtracting the pointer  J  * to the Markowitz product of the first diagonal from the pointer to the M  * Markowitz product of the desired element. The outer for loop is infinite,    * broken by using break. */  %     for (;;)  /* Endless for loop. */ =     {   while (*(++pMarkowitzProduct) >= MinMarkowitzProduct) '         {   /* Just passing through. */ 	         }   7         I = pMarkowitzProduct - gMatrix->MarkowitzProd;   < /* Assure that I is valid; if I > Size, terminate search. */         if (I > Size) )             break; /* Endless for loop */   /         if ((pDiag = gMatrix->Diag[I]) == NULL) ,             continue; /* Endless for loop */K         if ((Magnitude = ELEMENT_MAG(pDiag)) <= gMatrix->AbsoluteThreshold) ,             continue; /* Endless for loop */  $         if (*pMarkowitzProduct == 1)	         { O /* Case where only one element exists in row and column other than diagonal. */   ! /* Find off diagonal elements. */ +             pOtherInRow = pDiag->NextInRow; +             pOtherInCol = pDiag->NextInCol; <             if (pOtherInRow == NULL AND pOtherInCol == NULL)6             {    pOtherInRow = gMatrix->FirstInRow[I];+                  while(pOtherInRow != NULL) L                  {   if (pOtherInRow->Col >= Step AND pOtherInRow->Col != I)                          break; :                      pOtherInRow = pOtherInRow->NextInRow;                  }6                  pOtherInCol = gMatrix->FirstInCol[I];+                  while(pOtherInCol != NULL) L                  {   if (pOtherInCol->Row >= Step AND pOtherInCol->Row != I)                          break; :                      pOtherInCol = pOtherInCol->NextInCol;                  }
             }   L /* Accept diagonal as pivot if diagonal is larger than off diagonals and the+  * off diagonals are placed symmetricly. */ >             if (pOtherInRow != NULL  AND  pOtherInCol != NULL)9             {   if (pOtherInRow->Col == pOtherInCol->Row) F                 {   LargestOffDiagonal = MAX(ELEMENT_MAG(pOtherInRow),P                                                       ELEMENT_MAG(pOtherInCol));8                     if (Magnitude >= LargestOffDiagonal)                     { > /* Accept pivot, it is unlikely to contribute excess error. */@                         LargestInCol = ELEMENT_MAG(pOtherInCol);&                         return(pDiag);                     }                  } 
             } 	         }   1         MinMarkowitzProduct = *pMarkowitzProduct;          ChosenPivot = pDiag;%     }  /* End of endless for loop. */        if (ChosenPivot != NULL)<     {   LargestInCol = FindBiggestInColExclude(ChosenPivot);B         if( ELEMENT_MAG(ChosenPivot) <= Threshold * LargestInCol )             ChosenPivot = NULL;      }      return(ChosenPivot); }  #endif                   /*@  *   SEARCH DIAGONAL FOR PIVOT WITH MODIFIED MARKOWITZ CRITERION  *J  *   Searches the diagonal looking for the best pivot.  For a pivot to be L  *   acceptable it must be larger than the pivot Threshold times the largestJ  *   element in its reduced column.  Amoung the acceptable diagonals, the H  *   one with the smallest MarkowitzProduct is sought.  If a tie occurs F  *   between elements of equal MarkowitzProduct, then the element withL  *   the largest ratio between its magnitude and the largest element in its L  *   column is used.  The search will be terminated after a given number of N  *   ties have occured and the best (smallest ratio) of the tied element will J  *   be used as the pivot.  The number of ties that will trigger an early 9  *   termination is MinMarkowtzProduct * TIES_MULTIPLIER.   *  *   >>> Returned:M  *   A pointer to the diagonal element chosen to be pivot.  If no diagonal is $  *   acceptable, a NULL is returned.  *  *   >>> Local variables: "  *   ChosenPivot  (ElementPointer)E  *       Pointer to the element that has been chosen to be the pivot. %  *   LargestInColOfAccepted  (double) P  *       Magnitude of the largest element in the column of the diagonal element 9  *       that is currently the pivot candidate of choice.   *   lSize  (int) O  *       Local version of size which is placed in a register to increase speed.   *   Magnitude  (double),  *       Absolute value of diagonal element.   *   MinMarkowitzProduct  (long)N  *       Smallest Markowitz product found of those pivot candidates which are   *       acceptable.  *   NumberOfTies  (int)M  *       A count of the number of Markowitz ties that have occured at current   *       MarkowitzProduct.  *   pDiag  (ElementPointer)  .  *       Pointer to current diagonal element.   *   pMarkowitzProduct  (long*) P  *       Pointer that points into MarkowitzProduct array. It is used to quickly .  *       access succesive Markowitz products.   *   Ratio  (double)6  *       For the current pivot candidate, Ratio is theF  *       Ratio of the largest element in its column to its magnitude.   *   RatioOfAccepted  (double)J  *       For the best pivot candidate found so far, RatioOfAccepted is theF  *       Ratio of the largest element in its column to its magnitude.   *  *   >>> Global variables:  *   gMatrix  <used>  *   LargestInCol  <set>  *   Size  <used>   *   Step  <used>   *   Threshold  <used>  */    static ElementPointer    SearchDiagonal()     {  register  int  I, lSize;7 register long  MinMarkowitzProduct, *pMarkowitzProduct;   register  ElementPointer  pDiag; int  NumberOfTies; ElementPointer  ChosenPivot;* double  Magnitude, Ratio, RatioOfAccepted; double  LargestInColOfAccepted; " double  FindBiggestInColExclude(); ELEMENT_MAG_DECLARATIONS;    /* Begin. */     ChosenPivot = NULL;      lSize = Size; /     MinMarkowitzProduct = LARGEST_LONG_INTEGER; :     pMarkowitzProduct = &(gMatrix->MarkowitzProd[Step-1]);   /* Begin search of diagonal. */ #     for (I = Step; I <= lSize; I++)      { 9         if (*(++pMarkowitzProduct) > MinMarkowitzProduct) $             continue; /* for loop *//         if ((pDiag = gMatrix->Diag[I]) == NULL) $             continue; /* for loop */K         if ((Magnitude = ELEMENT_MAG(pDiag)) <= gMatrix->AbsoluteThreshold) $             continue; /* for loop */  8 /* Test to see if diagonal's magnitude is acceptable. */6         LargestInCol = FindBiggestInColExclude(pDiag);1         if (Magnitude < Threshold * LargestInCol) $             continue; /* for loop */  5         if (*pMarkowitzProduct < MinMarkowitzProduct) 	         { P /* Notice strict inequality in test. This is a new smallest MarkowitzProduct. */              ChosenPivot = pDiag;5             MinMarkowitzProduct = *pMarkowitzProduct; 7             RatioOfAccepted = LargestInCol / Magnitude; 2             LargestInColOfAccepted = LargestInCol;             NumberOfTies = 0; 	         }          else	         { ' /* This case handles Markowitz ties. */              NumberOfTies++; -             Ratio = LargestInCol / Magnitude; (             if (Ratio < RatioOfAccepted)$             {   ChosenPivot = pDiag;(                 RatioOfAccepted = Ratio;6                 LargestInColOfAccepted = LargestInCol;             }   F             if (NumberOfTies >= MinMarkowitzProduct * TIES_MULTIPLIER)6             {   LargestInCol = LargestInColOfAccepted;$                 return(ChosenPivot);
             } 	         }      } /* End of for(Step) */*     LargestInCol = LargestInColOfAccepted;     return(ChosenPivot); }                      /*(  *   SEARCH ENTIRE MATRIX FOR BEST PIVOT  *I  *   Performs a search over the entire matrix looking for the acceptable  J  *   element with the lowest MarkowitzProduct.  If there are several that L  *   are tied for the smallest MarkowitzProduct, the tie is broken by using N  *   the ratio of the magnitude of the element being considered to the largestO  *   element in the same column.  If no element is acceptable then the largest  C  *   element in the reduced submatrix is used as the pivot and the  J  *   matrix is declared to be ILL_CONDITIONED.  If the largest element is 1  *   zero, the matrix is declared to be SINGULAR.   *  *   >>> Returned:L  *   A pointer to the diagonal element chosen to be pivot.  If no element is=  *   found, then NULL is returned and the matrix is SINGULAR.   *  *   >>> Local variables: "  *   ChosenPivot  (ElementPointer)E  *       Pointer to the element that has been chosen to be the pivot. %  *   LargestInColOfAccepted  (double) I  *       Magnitude of the largest element in the column of the currently  D  *       favored pivot candidate.  The pivot candidate is excluded.    *   LargestElementMag  (double)M  *       Magnitude of the largest element yet found in the reduced submatrix.   *   lSize  (int) ?  *       Local version of Size; placed in a register for speed.   *   Magnitude  (double),  *       Absolute value of diagonal element.   *   MinMarkowitzProduct  (long)H  *       Smallest Markowitz product found of pivot candidates which are   *       acceptable.  *   NumberOfTies  (int)M  *       A count of the number of Markowitz ties that have occured at current   *       MarkowitzProduct.   *   pElement  (ElementPointer) %  *       Pointer to current element.  &  *   pLargestElement  (ElementPointer)K  *       Pointer to the largest element yet found in the reduced submatrix.   *   Product  (long):  *       Markowitz product for the current row and column.  *   Ratio  (double)6  *       For the current pivot candidate, Ratio is theF  *       Ratio of the largest element in its column to its magnitude.   *   RatioOfAccepted  (double)J  *       For the best pivot candidate found so far, RatioOfAccepted is theF  *       Ratio of the largest element in its column to its magnitude.   *  *   >>> Global variables:  *   gMatrix  <used>  *   LargestInCol  <set>  *   Size  <used>   *   Step  <used>   *   Threshold  <used>  *  *   >>> Possible errors: 
  *   SINGULAR   *   ILL_CONDITIONED    */    static ElementPointer    SearchEntireMatrix()     {  register  int  I, lSize;# register  ElementPointer  pElement;  int  NumberOfTies;# long  Product, MinMarkowitzProduct; - ElementPointer  ChosenPivot, pLargestElement; = double  Magnitude, LargestElementMag, Ratio, RatioOfAccepted;   double  LargestInColOfAccepted;  double  FindLargestInCol(); " double  FindBiggestInColExclude(); ELEMENT_MAG_DECLARATIONS;    /* Begin. */     ChosenPivot = NULL;      lSize = Size;      LargestElementMag = 0.0;/     MinMarkowitzProduct = LARGEST_LONG_INTEGER;   7 /* Begin search of matrix on column by column basis. */ #     for (I = Step; I <= lSize; I++) *     {   pElement = gMatrix->FirstInCol[I];  9         while (pElement != NULL AND pElement->Row < Step) +             pElement = pElement->NextInCol;   >         if((LargestInCol = FindLargestInCol(pElement)) == 0.0)$             continue; /* for loop */            while (pElement != NULL)	         { L /* Check to see if element is the largest encountered so far.  If so, record     its magnitude and address. */H             if ((Magnitude = ELEMENT_MAG(pElement)) > LargestElementMag).             {   LargestElementMag = Magnitude;+                 pLargestElement = pElement; 
             } + /* Calculate element's MarkowitzProduct. */ =             Product = gMatrix->MarkowitzRow[pElement->Row] *  ;                       gMatrix->MarkowitzCol[pElement->Col];   @ /* Test to see if element is acceptable as a pivot candidate. */4             if ((Product <= MinMarkowitzProduct) AND:                 (Magnitude > Threshold * LargestInCol) AND9                 (Magnitude > gMatrix->AbsoluteThreshold)) 
             { N /* Test to see if element has lowest MarkowitzProduct yet found, or whether it,    is tied with an element found earlier. */2                 if (Product < MinMarkowitzProduct)                 { P /* Notice strict inequality in test. This is a new smallest MarkowitzProduct. */+                     ChosenPivot = pElement; 2                     MinMarkowitzProduct = Product;?                     RatioOfAccepted = LargestInCol / Magnitude; :                     LargestInColOfAccepted = LargestInCol;%                     NumberOfTies = 0;                  }                  else                 { ' /* This case handles Markowitz ties. */ #                     NumberOfTies++; 5                     Ratio = LargestInCol / Magnitude; 0                     if (Ratio < RatioOfAccepted)/                     {   ChosenPivot = pElement; 0                         RatioOfAccepted = Ratio;>                         LargestInColOfAccepted = LargestInCol;                     }   N                     if (NumberOfTies >= MinMarkowitzProduct * TIES_MULTIPLIER)>                     {   LargestInCol = LargestInColOfAccepted;-                         return(ChosenPivot);                       }                  } 
             } +             pElement = pElement->NextInCol; /         }  /* End of while(pElement != NULL) */      } /* End of for(Step) */     if (ChosenPivot != NULL)         return(ChosenPivot);  !     if (LargestElementMag == 0.0) "     {   gMatrix->Error = SINGULAR;         return(NULL);      }   <     LargestInCol = FindBiggestInColExclude(pLargestElement);%     gMatrix->Error = ILL_CONDITIONED;      return(pLargestElement); }                          /*@  *   DETERMINE THE MAGNITUDE OF THE LARGEST ELEMENT IN A COLUMN   *L  *   This routine searches a column and returns the magnitude of the largestJ  *   element.  This routine begins the search at the element pointed to by  *   pElement, the parameter.   *O  *   The search is conducted by starting at the element specified by a pointer, M  *   which should be one below the diagonal, and moving down the column.  On  O  *   the way down the column, the magnitudes of the elements are tested to see  '  *   if they are the largest yet found.   *  *   >>> Returned:L  *   The magnitude of the largest element in the column below and including /  *   the one pointed to by the input parameter.   *  *   >>> Arguements:(  *   pElement  <input>  (ElementPointer)K  *       The pointer to the first element to be tested.  Also, used by the  <  *       routine to access all lower elements in the column.  *  *   >>> Local variables:   *   Largest  (double).  *       The magnitude of the largest element.  *   Magnitude  (double)7  *       The magnitude of the currently active element.   */   
 static double  FindLargestInCol( pElement )  # register  ElementPointer  pElement;  { ! double  Magnitude, Largest = 0.0;  ELEMENT_MAG_DECLARATIONS;    /* Begin. */= /* Search column for largest element beginning at Element. */      while (pElement != NULL):     {   if ((Magnitude = ELEMENT_MAG(pElement)) > Largest)              Largest = Magnitude;'         pElement = pElement->NextInCol;      }        return(Largest); }                      /*@  *   DETERMINE THE MAGNITUDE OF THE LARGEST ELEMENT IN A COLUMN   *   EXCLUDING AN ELEMENT   *L  *   This routine searches a column and returns the magnitude of the largestJ  *   element.  One given element is specifically excluded from the search.  *K  *   The search is conducted by starting at the first element in the column O  *   and moving down the column until the active part of the matrix is entered, K  *   i.e. the reduced submatrix.  The rest of the column is then traversed  %  *   looking for the largest element.   *  *   >>> Returned:N  *   The magnitude of the largest element in the active portion of the column,2  *   excluding the specified element, is returned.  *  *   >>> Arguements:(  *   pElement  <input>  (ElementPointer)N  *       The pointer to the element that is to be excluded from search. ColumnI  *       to be searched is one that contains this element.  Also used to  +  *       access the elements in the column.   *  *   >>> Local variables:   *   Col  (int) L  *       The number of the column to be searched.  Also the column number of1  *       the element to be avoided in the search.   *   Largest  (double).  *       The magnitude of the largest element.  *   lStep  (int) N  *       A local version of Step, which indicates where the active part of the.  *       matrix begins.  Made local for speed.  *   Magnitude  (double)7  *       The magnitude of the currently active element.   *   Row  (int) B  *       The row number of element to be excluded from the search.  *  *   >>> Global variables:        *   gMatrix  <used>  *   Step  <used>   */   
 static double # FindBiggestInColExclude( pElement )   # register  ElementPointer  pElement;  {  register  int  Row, lStep;	 int  Col;  double  Largest, Magnitude;  ELEMENT_MAG_DECLARATIONS;    /* Begin. */     lStep = Step;      Row = pElement->Row;     Col = pElement->Col;(     pElement = gMatrix->FirstInCol[Col];  < /* Travel down column until reduced submatrix is entered. */:     while ((pElement != NULL) AND (pElement->Row < lStep))'         pElement = pElement->NextInCol;   & /* Initialize the variable Largest. */     if (pElement->Row != Row) (         Largest = ELEMENT_MAG(pElement);     else         Largest = 0.0;  C /* Search rest of column for largest element, avoiding diagonal. */ 4     while ((pElement = pElement->NextInCol) != NULL):     {   if ((Magnitude = ELEMENT_MAG(pElement)) > Largest)%         {   if (pElement->Row != Row) $                 Largest = Magnitude;	         }      }        return(Largest); }                      /*  *   EXCHANGE ROWS AND COLUMNS  *N  *   Exchanges two rows and two columns so that the selected pivot is moved to6  *   the upper left corner of the remaining submatrix.  *  *   >>> Argument:&  *   pPivot  <input>  (ElementPointer)&  *       Pointer to the current pivot.  *  *   >>> Local variables:   *   Col  (int) 7  *       Column that the pivot was originally found in.   *   Row  (int) 4  *       Row that the pivot was originally found in.!  *   OldMarkowitzProd_Col  (long) J  *       Markowitz product associated with the diagonal element in the row   *       the pivot was found in.!  *   OldMarkowitzProd_Row  (long) M  *       Markowitz product associated with the diagonal element in the column    *       the pivot was found in."  *   OldMarkowitzProd_Step  (long)M  *       Markowitz product associated with the diagonal element that is being L  *       moved so that the pivot can be placed in the upper left-hand corner"  *       of the reduced submatrix.  *  *   >>> Global variables:  *   gMatrix  <used>  *   PivotsOriginalCol  <set>   *   PivotsOriginalRow  <set>   *   Singletons  <set>  *   Step  <used>   */    static ExchangeRowsAndCols(pPivot)     ElementPointer  pPivot;  {  register  int   Row, Col; H long  OldMarkowitzProd_Step, OldMarkowitzProd_Row, OldMarkowitzProd_Col;" ElementPointer FindElementInCol();   /* Begin. */     Row = pPivot->Row;     Col = pPivot->Col;     PivotsOriginalRow = Row;     PivotsOriginalCol = Col;  (     if ((Row == Step) AND (Col == Step))         return;     /* Exchange rows and columns. */     if (Row == Col) !     {   ExchangeRows (Step, Row); !         ExchangeCols (Step, Col); N         SWAP (long, gMatrix->MarkowitzProd[Step],gMatrix->MarkowitzProd[Row]);E         SWAP (ElementPointer,gMatrix->Diag[Row],gMatrix->Diag[Step]);      }      else     {   < /* Initialize variables that hold old Markowitz products. */=         OldMarkowitzProd_Step = gMatrix->MarkowitzProd[Step]; ;         OldMarkowitzProd_Row = gMatrix->MarkowitzProd[Row]; ;         OldMarkowitzProd_Col = gMatrix->MarkowitzProd[Col];    /* Exchange rows. */         if (Row != Step)%         {   ExchangeRows (Step, Row); 1             gMatrix->NumberOfInterchangesIsOdd =  N                                        NOT gMatrix->NumberOfInterchangesIsOdd;G             gMatrix->MarkowitzProd[Row] = gMatrix->MarkowitzRow[Row] * PN                                                    gMatrix->MarkowitzCol[Row];   /* Update singleton count. */oN             if ((gMatrix->MarkowitzProd[Row]==0) != (OldMarkowitzProd_Row==0)).             {   if (OldMarkowitzProd_Row == 0)!                     Singletons--;n                 else!                     Singletons++;a
             }m	         }    /* Exchange columns. */a         if (Col != Step)%         {   ExchangeCols (Step, Col);e1             gMatrix->NumberOfInterchangesIsOdd = rN                                        NOT gMatrix->NumberOfInterchangesIsOdd;G             gMatrix->MarkowitzProd[Col] = gMatrix->MarkowitzCol[Col] * oN                                                    gMatrix->MarkowitzRow[Col];   /* Update singleton count. */*N             if ((gMatrix->MarkowitzProd[Col]==0) != (OldMarkowitzProd_Col==0)).             {   if (OldMarkowitzProd_Col == 0)!                     Singletons--;r                 else!                     Singletons++; 
             } 5                                                       N             gMatrix->Diag[Col] = FindElementInCol(&(gMatrix->FirstInCol[Col]),P                                                                  Col,Col,FALSE);	         }r         if (Row != Step)O             gMatrix->Diag[Row] = FindElementInCol(&(gMatrix->FirstInCol[Row]), sP                                                                  Row,Row,FALSE);M         gMatrix->Diag[Step] = FindElementInCol(&(gMatrix->FirstInCol[Step]), .P                                                                Step,Step,FALSE);   /* Update singleton count. */ E         gMatrix->MarkowitzProd[Step] = gMatrix->MarkowitzCol[Step] *  P                                                     gMatrix->MarkowitzRow[Step];L         if ((gMatrix->MarkowitzProd[Step]==0) != (OldMarkowitzProd_Step==0))+         {   if (OldMarkowitzProd_Step == 0)o                 Singletons--;a             else                 Singletons++;r	         }        }d }m                 r /*  *   EXCHANGE ROWS  *M  *   Performs all required operations to exchange two rows. Those operations  J  *   include: swap FirstInRow pointers, fixing up the NextInCol pointers, L  *   swapping row indexes in MatrixElements, and swapping Markowitz row and   *   column counts.a  *  *   >>> Arguements:  *   Row1  <input>  (int) C  *       Row index of one of the rows, becomes the smallest index.    *   Row2  <input>  (int)i?  *       Row index of the other row, becomes the largest index.   *  *   Local variables:   *   Column  (int)E  *       Column in which row elements are currently being exchanged. a  *   Row1Ptr  (ElementPointer)(  *       Pointer to an element in Row1.   *   Row2Ptr  (ElementPointer)(  *       Pointer to an element in Row2.   *   Element1  (ElementPointer))9  *       Pointer to the element in Row1 to be exchanged. r  *   Element2  (ElementPointer)d9  *       Pointer to the element in Row2 to be exchanged.    *  *   >>> Global variables:  *   gMatrix  <used>  */i   static ExchangeRows( Row1, Row2 )    int  Row1, Row2; {r+ register  ElementPointer  Row1Ptr, Row2Ptr;h int  Column;# ElementPointer  Element1, Element2;r   /* Begin. */,     if (Row1 > Row2)  SWAP(int, Row1, Row2);  (     Row1Ptr = gMatrix->FirstInRow[Row1];(     Row2Ptr = gMatrix->FirstInRow[Row2];.     while (Row1Ptr != NULL OR Row2Ptr != NULL)     {aC /* Exchange elements in rows while traveling from left to right. */r         if (Row1Ptr == NULL)"         {   Column = Row2Ptr->Col;             Element1 = NULL;             Element2 = Row2Ptr;l)             Row2Ptr = Row2Ptr->NextInRow; 	         }u!         else if (Row2Ptr == NULL)o"         {   Column = Row1Ptr->Col;             Element1 = Row1Ptr;              Element2 = NULL;)             Row1Ptr = Row1Ptr->NextInRow;;	         }a-         else if (Row1Ptr->Col < Row2Ptr->Col)*"         {   Column = Row1Ptr->Col;             Element1 = Row1Ptr;              Element2 = NULL;)             Row1Ptr = Row1Ptr->NextInRow;t	         }a-         else if (Row1Ptr->Col > Row2Ptr->Col)n"         {   Column = Row2Ptr->Col;             Element1 = NULL;             Element2 = Row2Ptr; )             Row2Ptr = Row2Ptr->NextInRow;r	         }o1         else   /* Row1Ptr->Col == Row2Ptr->Col */m"         {   Column = Row1Ptr->Col;             Element1 = Row1Ptr;              Element2 = Row2Ptr; )             Row1Ptr = Row1Ptr->NextInRow;t)             Row2Ptr = Row2Ptr->NextInRow; 	         }   E         ExchangeColElements( Row1, Element1, Row2, Element2, Column);t=     }  /* end of while(Row1Ptr != NULL OR Row2Ptr != NULL) */   I     SWAP( int, gMatrix->MarkowitzRow[Row1], gMatrix->MarkowitzRow[Row2]);dP     SWAP( ElementPointer, gMatrix->FirstInRow[Row1], gMatrix->FirstInRow[Row2]);W     SWAP( int, gMatrix->RowEliminationOrder[Row1], gMatrix->RowEliminationOrder[Row2]);*       return;  }o                   /*  *   EXCHANGE COLUMNSh  *P  *   Performs all required operations to exchange two columns. Those operations J  *   include: swap FirstInCol pointers, fixing up the NextInRow pointers, O  *   swapping column indexes in MatrixElements, and swapping Markowitz row and e  *   column counts.e  *  *   >>> Arguements:  *   Col1  <input>  (int)gI  *       Column index of one of the columns, becomes the smallest index. r  *   Col2  <input>  (int) E  *       Column index of the other column, becomes the largest index *  *  *   Local variables:   *   Row  (int)iE  *       Row in which column elements are currently being exchanged. c  *   Col1Ptr  (ElementPointer)(  *       Pointer to an element in Col1.   *   Col2Ptr  (ElementPointer)(  *       Pointer to an element in Col2.   *   Element1  (ElementPointer)e9  *       Pointer to the element in Col1 to be exchanged.    *   Element2  (ElementPointer) 9  *       Pointer to the element in Col2 to be exchanged. s  *  *   >>> Global variables:  *   gMatrix  <used>  */o   static ExchangeCols( Col1, Col2 )  p int  Col1, Col2; {o+ register  ElementPointer  Col1Ptr, Col2Ptr;h	 int  Row; # ElementPointer  Element1, Element2;l   /* Begin. */,     if (Col1 > Col2)  SWAP(int, Col1, Col2);  (     Col1Ptr = gMatrix->FirstInCol[Col1];(     Col2Ptr = gMatrix->FirstInCol[Col2];.     while (Col1Ptr != NULL OR Col2Ptr != NULL)     {iC /* Exchange elements in rows while traveling from top to bottom. */f         if (Col1Ptr == NULL)         {   Row = Col2Ptr->Row;c             Element1 = NULL;             Element2 = Col2Ptr;h)             Col2Ptr = Col2Ptr->NextInCol;t	         } !         else if (Col2Ptr == NULL)o         {   Row = Col1Ptr->Row;t             Element1 = Col1Ptr;N             Element2 = NULL;)             Col1Ptr = Col1Ptr->NextInCol;l	         } -         else if (Col1Ptr->Row < Col2Ptr->Row)          {   Row = Col1Ptr->Row;m             Element1 = Col1Ptr;t             Element2 = NULL;)             Col1Ptr = Col1Ptr->NextInCol;e	         }t-         else if (Col1Ptr->Row > Col2Ptr->Row)l         {   Row = Col2Ptr->Row;l             Element1 = NULL;             Element2 = Col2Ptr;h)             Col2Ptr = Col2Ptr->NextInCol; 	         }a1         else   /* Col1Ptr->Row == Col2Ptr->Row */u         {   Row = Col1Ptr->Row;t             Element1 = Col1Ptr;a             Element2 = Col2Ptr;e)             Col1Ptr = Col1Ptr->NextInCol;c)             Col2Ptr = Col2Ptr->NextInCol;R	         }s  B         ExchangeRowElements( Col1, Element1, Col2, Element2, Row);=     }  /* end of while(Col1Ptr != NULL OR Col2Ptr != NULL) */   I     SWAP( int, gMatrix->MarkowitzCol[Col1], gMatrix->MarkowitzCol[Col2]);hP     SWAP( ElementPointer, gMatrix->FirstInCol[Col1], gMatrix->FirstInCol[Col2]);W     SWAP( int, gMatrix->ColEliminationOrder[Col1], gMatrix->ColEliminationOrder[Col2]);o       return;S }                  < /*&  *   EXCHANGE TWO ELEMENTS IN A COLUMN  *K  *   Performs all required operations to exchange two elements in a column.DO  *   Those operations are: restring NextInCol pointers and swapping row indexes   *   in the MatrixElements.G  *  *   >>> Arguements:  *   Row1  <input>  (int),-  *       Row of top element to be exchanged. t(  *   Element1  <input>  (ElementPointer)1  *       Pointer to top element to be exchanged.    *   Row2  <input>  (int)m0  *       Row of bottom element to be exchanged. (  *   Element2  <input>  (ElementPointer)4  *       Pointer to bottom element to be exchanged.   *   Column <input>  (int)3  *       Column that exchange is to take place in. =  *  *   >>> Local variables: )  *   ElementAboveRow1  (ElementPointer *) N  *       Location of pointer which points to the element above Element1. This K  *       pointer is modified so that it points to correct element on exit. w)  *   ElementAboveRow2  (ElementPointer *)sN  *       Location of pointer which points to the element above Element2. This K  *       pointer is modified so that it points to correct element on exit. s'  *   ElementBelowRow1  (ElementPointer)N,  *       Pointer to element below Element1. '  *   ElementBelowRow2  (ElementPointer)t,  *       Pointer to element below Element2.   *   pElement  (ElementPointer) .  *       Pointer used to traverse the column.   *  *   >>> Global variables:  *   gMatrix  <used>  */e   static= ExchangeColElements( Row1, Element1, Row2, Element2, Column )   S- register  ElementPointer  Element1, Element2;[ int  Row1, Row2, Column; {C5 ElementPointer  *ElementAboveRow1, *ElementAboveRow2; 3 ElementPointer  ElementBelowRow1, ElementBelowRow2; # register  ElementPointer  pElement;)   /* Begin. */* /* Search to find the ElementAboveRow1. */6     ElementAboveRow1 = &(gMatrix->FirstInCol[Column]);!     pElement = *ElementAboveRow1;       while (pElement->Row < Row1)2     {   ElementAboveRow1 = &(pElement->NextInCol);%         pElement = *ElementAboveRow1;      }      if (Element1 != NULL)d/     {   ElementBelowRow1 = Element1->NextInCol;*         if (Element2 == NULL)o	         {c: /* Element2 does not exist, move Element1 down to Row2. */L             if ( ElementBelowRow1 != NULL AND ElementBelowRow1->Row < Row2 )
             {e: /* Element1 must be removed from linked list and moved. */5                 *ElementAboveRow1 = ElementBelowRow1;    /* Search column for Row2. */h,                 pElement = ElementBelowRow1;                 do>                 {   ElementAboveRow2 = &(pElement->NextInCol);1                     pElement = *ElementAboveRow2; F                 }   while (pElement != NULL AND pElement->Row < Row2);   /* Place Element1 in Row2. */ -                 *ElementAboveRow2 = Element1;o/                 Element1->NextInCol = pElement; 4                 *ElementAboveRow1 =ElementBelowRow1;
             }s!             Element1->Row = Row2; 	         }t         else	         {mB /* Element2 does exist, and the two elements must be exchanged. *//             if ( ElementBelowRow1->Row == Row2) 
             {r5 /* Element2 is just below Element1, exchange them. */ :                 Element1->NextInCol = Element2->NextInCol;/                 Element2->NextInCol = Element1;o-                 *ElementAboveRow1 = Element2;n
             }              else
             {TC /* Element2 is not just below Element1 and must be searched for. */i,                 pElement = ElementBelowRow1;                 do>                 {   ElementAboveRow2 = &(pElement->NextInCol);1                     pElement = *ElementAboveRow2;r1                 }   while (pElement->Row < Row2);t  o7                 ElementBelowRow2 = Element2->NextInCol;t  # /* Switch Element1 and Element2. */ -                 *ElementAboveRow1 = Element2;a7                 Element2->NextInCol = ElementBelowRow1;U-                 *ElementAboveRow2 = Element1;t7                 Element1->NextInCol = ElementBelowRow2;e
             }x!             Element1->Row = Row2;n!             Element2->Row = Row1;i	         }      }E     else     {x /* Element1 does not exist. */$         ElementBelowRow1 = pElement;   /* Find Element2. */*         if (ElementBelowRow1->Row != Row2)         {   do:             {   ElementAboveRow2 = &(pElement->NextInCol);-                 pElement = *ElementAboveRow2;.-             }   while (pElement->Row < Row2);s  /         ElementBelowRow2 = Element2->NextInCol;l   /* Move Element2 to Row1. */4             *ElementAboveRow2 = Element2->NextInCol;)             *ElementAboveRow1 = Element2;e3             Element2->NextInCol = ElementBelowRow1;e	         }s         Element2->Row = Row1;a     }r     return;d }m                       /*#  *   EXCHANGE TWO ELEMENTS IN A ROW   *H  *   Performs all required operations to exchange two elements in a row.K  *   Those operations are: restring NextInRow pointers and swapping column r#  *   indexes in the MatrixElements.x  *  *   >>> Arguements:  *   Col1  <input>  (int) 3  *       Col of left-most element to be exchanged. e(  *   Element1  <input>  (ElementPointer)7  *       Pointer to left-most element to be exchanged. c  *   Col2  <input>  (int)l4  *       Col of right-most element to be exchanged. (  *   Element2  <input>  (ElementPointer)8  *       Pointer to right-most element to be exchanged.   *   Row <input>  (int)t0  *       Row that exchange is to take place in.   *  *   >>> Local variables:t*  *   ElementLeftOfCol1  (ElementPointer *)H  *       Location of pointer which points to the element to the left of I  *       Element1. This pointer is modified so that it points to correct    *       element on exit. *  *   ElementLeftOfCol2  (ElementPointer *)H  *       Location of pointer which points to the element to the left of I  *       Element2. This pointer is modified so that it points to correct    *       element on exit. )  *   ElementRightOfCol1  (ElementPointer)d/  *       Pointer to element right of Element1.  )  *   ElementRightOfCol2  (ElementPointer)y/  *       Pointer to element right of Element2.    *   pElement  (ElementPointer)a+  *       Pointer used to traverse the row.    *  *   >>> Global variables:  *   gMatrix  <used>  */h   static: ExchangeRowElements( Col1, Element1, Col2, Element2, Row )  i int  Col1, Col2, Row; , register ElementPointer  Element1, Element2;   {t7 ElementPointer  *ElementLeftOfCol1, *ElementLeftOfCol2;C7 ElementPointer  ElementRightOfCol1, ElementRightOfCol2; $ register   ElementPointer  pElement;   /* Begin. */+ /* Search to find the ElementLeftOfCol1. */r4     ElementLeftOfCol1 = &(gMatrix->FirstInRow[Row]);"     pElement = *ElementLeftOfCol1;      while (pElement->Col < Col1)3     {   ElementLeftOfCol1 = &(pElement->NextInRow);r&         pElement = *ElementLeftOfCol1;     }      if (Element1 != NULL) 1     {   ElementRightOfCol1 = Element1->NextInRow;>         if (Element2 == NULL)<	         { > /* Element2 does not exist, move Element1 to right to Col2. */P             if ( ElementRightOfCol1 != NULL AND ElementRightOfCol1->Col < Col2 )
             {*: /* Element1 must be removed from linked list and moved. */8                 *ElementLeftOfCol1 = ElementRightOfCol1;   /* Search Row for Col2. */.                 pElement = ElementRightOfCol1;                 do?                 {   ElementLeftOfCol2 = &(pElement->NextInRow);O2                     pElement = *ElementLeftOfCol2;F                 }   while (pElement != NULL AND pElement->Col < Col2);   /* Place Element1 in Col2. */o.                 *ElementLeftOfCol2 = Element1;/                 Element1->NextInRow = pElement;U7                 *ElementLeftOfCol1 =ElementRightOfCol1; 
             }d!             Element1->Col = Col2; 	         }=         else	         {>B /* Element2 does exist, and the two elements must be exchanged. */1             if ( ElementRightOfCol1->Col == Col2)P
             {=8 /* Element2 is just right of Element1, exchange them. */:                 Element1->NextInRow = Element2->NextInRow;/                 Element2->NextInRow = Element1;P.                 *ElementLeftOfCol1 = Element2;
             }g             else
             {oF /* Element2 is not just right of Element1 and must be searched for. */.                 pElement = ElementRightOfCol1;                 do?                 {   ElementLeftOfCol2 = &(pElement->NextInRow);a2                     pElement = *ElementLeftOfCol2;1                 }   while (pElement->Col < Col2);)  9                 ElementRightOfCol2 = Element2->NextInRow;   # /* Switch Element1 and Element2. */M.                 *ElementLeftOfCol1 = Element2;9                 Element2->NextInRow = ElementRightOfCol1; .                 *ElementLeftOfCol2 = Element1;9                 Element1->NextInRow = ElementRightOfCol2; 
             }O!             Element1->Col = Col2;=!             Element2->Col = Col1;e	         }      }r     else     {t /* Element1 does not exist. */&         ElementRightOfCol1 = pElement;   /* Find Element2. */,         if (ElementRightOfCol1->Col != Col2)         {   do;             {   ElementLeftOfCol2 = &(pElement->NextInRow);P.                 pElement = *ElementLeftOfCol2;-             }   while (pElement->Col < Col2);n  5             ElementRightOfCol2 = Element2->NextInRow;    /* Move Element2 to Col1. */5             *ElementLeftOfCol2 = Element2->NextInRow;R*             *ElementLeftOfCol1 = Element2;5             Element2->NextInRow = ElementRightOfCol1; 	         }          Element2->Col = Col1;H     }S     return;T }O                       /*6  *   PERFORM ROW AND COLUMN ELIMINATION ON REAL MATRIX  *N  *   Eliminates a single row and column of the matrix and leaves single row ofL  *   the upper triangular matrix and a single column of the lower triangular/  *   matrix in its wake.  Uses Gauss's method. o  *  *   >>> Argument:&  *   pPivot  <input>  (ElementPointer)&  *       Pointer to the current pivot.  *  *   >>> Local variables:   *   ColPtr  (ElementPointer)s>  *       Points to matrix element in lower triangular column. +  *   LargestInEliminatedRowAndCol  (double).M  *       Contains the value of the largest element in the row or column being H  *       eliminated.  Used in calculation of element growth for roundoff  *       error estimation.  *   pElement (ElementPointer)'  *       Points to elements in matrix.    *   PivotMagnitude  (double)q6  *       Magnitude of the element being used as pivot.  *   Row  (int)g  *       Row index.   *   RowPtr  (ElementPointer)*;  *       Points to matrix element in upper triangular row. *  *   UpperTri  (double)a2  *       Current upper triangular matrix element.   *  *   >>> Global variable:   *   gMatrix  <used>!  *   LargestAfterDecompose  <set>   *   Step  <used>   *  *   >>> Possible errors:a  *   NO_MEMORY    */M   static RealRowColElimination(pPivot)*     ! register  ElementPointer  pPivot;g { 
 #if (REAL)  3 register  ElementPointer  pElement, ColPtr, RowPtr;g register  int  Row; @ double  UpperTri, PivotMagnitude,  LargestInEliminatedRowAndCol; ELEMENT_MAG_DECLARATIONS;M ElementPointer  CreateFillIn();f   /* Begin. */ /* Perform elimination. */ s!     pPivot = gMatrix->Diag[Step]; (     PivotMagnitude = ABS(pPivot->Real);    /* Test for zero pivot. */     if (PivotMagnitude == 0.0)"     {   gMatrix->Error = SINGULAR;         return;r     }e   #if (MONITOR_GROWTH)E     LargestInEliminatedRowAndCol = MAX(PivotMagnitude, LargestInCol);) #endif   #if (CALCULATE_PSEUDOCONDITION)eG     if (PivotMagnitude < SmallestPivot) SmallestPivot = PivotMagnitude;sE     if (PivotMagnitude > LargestPivot) LargestPivot = PivotMagnitude;A #endif       RowPtr = pPivot;0     while ((RowPtr = RowPtr->NextInRow) != NULL)     {   ) /* Calculate upper triangular element. */ 2         UpperTri = (RowPtr->Real /= pPivot->Real);   #if (MONITOR_GROWTH)9         if (ABS(UpperTri) > LargestInEliminatedRowAndCol)19             LargestInEliminatedRowAndCol = ABS(UpperTri);C #endif                          pElement = RowPtr;         ColPtr = pPivot;4         while ((ColPtr = ColPtr->NextInCol) != NULL)         {   Row = ColPtr->Row;  I /* Find element in row being eliminated that lines up with current lower      triangular element. */g<             while ((pElement = pElement->NextInCol) != NULL)(             {   if (pElement->Row < Row)/                     continue;  /* while loop */g)                 if (pElement->Row == Row)(+                     goto RealElementExists;N                 else n,                     break;  /* while loop */
             }   4 /* Desired element was not found, create fill-in. */6             pElement = CreateFillIn(Row, RowPtr->Col);,             if (gMatrix->Error == NO_MEMORY)                 return;a   RealElementExists:6             pElement->Real -= UpperTri * ColPtr->Real;	         }t?     }  /* end of while((RowPtr = RowPtr->NextInRow) != NULL) */    #if (MONITOR_GROWTH)F /* Update the value of the largest element in the matrices L and U. */=     if (LargestInEliminatedRowAndCol > LargestAfterDecompose)i=         LargestAfterDecompose = LargestInEliminatedRowAndCol;s #endif       return;    #endif  /* REAL */ }r                   /*9  *   PERFORM ROW AND COLUMN ELIMINATION ON COMPLEX MATRIX   *N  *   Eliminates a single row and column of the matrix and leaves single row ofL  *   the upper triangular matrix and a single column of the lower triangular/  *   matrix in its wake.  Uses Gauss's method. e  * n  *   >>> Argument:&  *   pPivot  <input>  (ElementPointer)&  *       Pointer to the current pivot.  *  *   >>> Local variables:=  *   ColPtr  (ElementPointer)r>  *       Points to matrix element in lower triangular column. +  *   LargestInEliminatedRowAndCol  (double)rM  *       Contains the value of the largest element in the row or column being=H  *       eliminated.  Used in calculation of element growth for roundoff  *       error estimation.  *   pElement (ElementPointer)'  *       Points to elements in matrix.    *   PivotMagnitude  (double)r6  *       Magnitude of the element being used as pivot.  *   Row  (int)r  *       Row index.   *   RowPtr  (ElementPointer) ;  *       Points to matrix element in upper triangular row. -  *   UpperTriImag  (double) G  *       Imaginary portion of current upper triangular matrix element. M  *   UpperTriReal  (double)iB  *       Real portion of current upper triangular matrix element.   *  *   >>> Global variables:  *   gMatrix  <used>!  *   LargestAfterDecompose  <set>n  *   Step  <used>*  *  *   Possible errors:s  *   NO_MEMORY  */    static  ComplexRowColElimination(pPivot)  ! register  ElementPointer  pPivot;l {n
 #if (COMPLEX)d  3 register  ElementPointer  pElement, ColPtr, RowPtr;* register  int  Row;  double  Magnitude;  A double  UpperTriReal, UpperTriImag, LargestInEliminatedRowAndCol;i ELEMENT_MAG_DECLARATIONS;  CMPLX_DIVIDE_DECLARATIONS; ElementPointer  CreateFillIn();r   /* Begin. */ /* Perform elimination. */  %     Magnitude = ELEMENT_MAG(pPivot);     /* Test for zero pivot. */     if (Magnitude == 0.0) "     {   gMatrix->Error = SINGULAR;         return;i     }E   #if (CALCULATE_PSEUDOCONDITION)e"     if (Magnitude < SmallestPivot)"         SmallestPivot = Magnitude;!     if (Magnitude > LargestPivot)E!         LargestPivot = Magnitude;  #endif   #if (MONITOR_GROWTH)?     LargestInEliminatedRowAndCol = MAX(Magnitude,LargestInCol);  #endif       RowPtr = pPivot;0     while ((RowPtr = RowPtr->NextInRow) != NULL)     {G  ) /* Calculate upper triangular element. */ 1 /* Complex divide : UpperTri = RowPtr / pPivot */t?         CMPLX_DIVIDE(UpperTriReal, UpperTriImag, RowPtr->Real,  @                       RowPtr->Imag, pPivot->Real, pPivot->Imag);   #if (MONITOR_GROWTH):         Magnitude = ABS(UpperTriReal) + ABS(UpperTriImag);5         if (Magnitude > LargestInEliminatedRowAndCol)e5             LargestInEliminatedRowAndCol = Magnitude;C #endif                 % /* Store upper triangular element. */f$         RowPtr->Real = UpperTriReal;$         RowPtr->Imag = UpperTriImag;                L         pElement = RowPtr;         ColPtr = pPivot;4         while ((ColPtr = ColPtr->NextInCol) != NULL)         {   Row = ColPtr->Row;  I /* Find element in row being eliminated that lines up with current lower  3    triangular element and subtract off LowerTri. */ <             while ((pElement = pElement->NextInCol) != NULL)(             {   if (pElement->Row < Row)/                     continue;  /* while loop */u)                 if (pElement->Row == Row) .                     goto ComplexElementExists;                 else m,                     break;  /* while loop */
             }l  4 /* Desired element was not found, create fill-in. */6             pElement = CreateFillIn(Row, RowPtr->Col);,             if (gMatrix->Error == NO_MEMORY)                 return;P   ComplexElementExists:e8 /* Complex expression : pElement -= UpperTri * ColPtr */P             CMPLX_MULT_SUBT_ASSIGN(pElement->Real, pElement->Imag, UpperTriReal,H                               UpperTriImag, ColPtr->Real, ColPtr->Imag);  	         } ?     }  /* end of while((RowPtr = RowPtr->NextInRow) != NULL) */w   #if (MONITOR_GROWTH)F /* Update the value of the largest element in the matrices L and U. */=     if (LargestInEliminatedRowAndCol > LargestAfterDecompose)s=         LargestAfterDecompose = LargestInEliminatedRowAndCol;P #endif       return;    #endif  /* COMPLEX */r }i         / /*  *   UPDATE MARKOWITZ NUMBERS*  *O  *   Updates the Markowitz numbers after a row and column have been eliminated.*"  *   Also updates singleton count.  *  *   >>> Argument:&  *   pPivot  <input>  (ElementPointer)&  *       Pointer to the current pivot.  *  *   >>> Local variables:o  *   Row  (int)M  *       Row index.   *   Col  (int)r  *       Column index. z  *   ColPtr  (ElementPointer)t>  *       Points to matrix element in upper triangular column.   *   RowPtr  (ElementPointer)+;  *       Points to matrix element in lower triangular row.    *  *   >>> Global variables:  *   gMatrix  <used>  *   Singletons  <set>  */    static UpdateMarkowitzNumbers(pPivot)   ElementPointer  pPivot;P {T register  int  Row, Col;) register  ElementPointer  ColPtr, RowPtr;    /* Begin. */ /* Update Markowitz numbers. */      ColPtr = pPivot;0     while ((ColPtr = ColPtr->NextInCol) != NULL)     {   Row = ColPtr->Row;E         gMatrix->MarkowitzProd[Row] = --gMatrix->MarkowitzRow[Row] *  O                                                     gMatrix->MarkowitzCol[Row];n,         if (gMatrix->MarkowitzRow[Row] == 0)             Singletons++;      }i       RowPtr = pPivot;0     while ((RowPtr = RowPtr->NextInRow) != NULL)     {   Col = RowPtr->Col;`         gMatrix->MarkowitzProd[Col] = gMatrix->MarkowitzRow[Col] * --gMatrix->MarkowitzCol[Col];T         if ((gMatrix->MarkowitzCol[Col] == 0) AND (gMatrix->MarkowitzRow[Col] != 0))             Singletons++;      }a     return;  }*                     /*  *   CREATE FILL-INo  *E  *   This routine is used to create fill-ins and splice them into then  *   matrix.  *  *   >>> Returns:p  *   Pointer to fill-in.  *  *   >>> Arguements:  *   Col  <input>  (int)#  *       Column index for element.    *   Row  <input>  (int)   *       Row index for element.   *  *   >>> Local variables:p  *   pElement  (ElementPointer)o/  *       Pointer to an element in the matrix.  x'  *   ppElementAbove  (ElementPointer *)rP  *       This contains the address of the pointer to the element just above the M  *       one being created. It is used to speed the search and it is updated E/  *       with address of the created element.  h  *  *   >>> Global variables:  *   gMatrix - (used)a  *   Singletons - (set)u  *  *   >>> Possible errors:   *   NO_MEMORY  */t   static ElementPointer  CreateFillIn (Row, Col)    register int  Row;	 int  Col;*    {t4 register  ElementPointer  pElement, *ppElementAbove;  ElementPointer  CreateElement();   /* Begin. */! /* Find Element above fill-in. */i/     ppElementAbove = &gMatrix->FirstInCol[Col];e     pElement = *ppElementAbove;      while (pElement != NULL)      {   if (pElement->Row < Row)2         {   ppElementAbove = &pElement->NextInCol;'             pElement = *ppElementAbove; 	         }S$         else break; /* while loop */     })  ( /* End of search, create the element. */7     pElement = CreateElement(Row, Col, ppElementAbove);*  + /* Update Markowitz counts and products. */c\     gMatrix->MarkowitzProd[Row] = ++gMatrix->MarkowitzRow[Row] * gMatrix->MarkowitzCol[Row];P     if ((gMatrix->MarkowitzRow[Row] == 1) AND (gMatrix->MarkowitzCol[Row] != 0))         Singletons--;v\     gMatrix->MarkowitzProd[Col] = gMatrix->MarkowitzRow[Col] * ++gMatrix->MarkowitzCol[Col];P     if ((gMatrix->MarkowitzRow[Col] != 0) AND (gMatrix->MarkowitzCol[Col] == 1))         Singletons--;e	          P     return (pElement); }c                   /*  *  *   WRITE STATUS*  *?  *   Write a summary of important variables to standard output.d  *  *   >>> Global variables:  *   gMatrix  <used>!  *   PivotSelectionMethod  <used>O  *   PivotsOriginalCol  <used>  *   PivotsOriginalRow  <used>  *   Size  <used>   *   Singletons  <used>   *   Step  <used>i  */o   static
 WriteStatus()s   {p #if (ANNOTATE == FULL) *	 int  I;  t   /* Begin. */"     printf("Step = %1d   ", Step);\     printf("Pivot Originally found at %1d,%1d using ", PivotsOriginalRow,PivotsOriginalCol);      switch(PivotSelectionMethod)9     {   case 's': printf("SearchForSingleton\n");  break; <         case 'q': printf("QuicklySearchDiagonal\n");  break;5         case 'd': printf("SearchDiagonal\n");  break; 9         case 'e': printf("SearchEntireMatrix\n");  break;v     }o&                                       "     printf("MarkowitzRow     = ");     for (I = 1; I <= Size; I++)k1         printf("%2d  ",gMatrix->MarkowitzRow[I]);s     printf("\n");u  "     printf("MarkowitzCol     = ");     for (I = 1; I <= Size; I++)r1         printf("%2d  ",gMatrix->MarkowitzCol[I]);r     printf("\n");z  "     printf("MarkowitzProduct = ");     for (I = 1; I <= Size; I++)E2         printf("%2d  ",gMatrix->MarkowitzProd[I]);     printf("\n");r  ,     printf("Singletons = %2d\n",Singletons);  "     printf("RowElimOrder     = ");     for (I = 1; I <= Size; I++)O8         printf("%2d  ",gMatrix->RowEliminationOrder[I]);     printf("\n");P  "     printf("ColElimOrder     = ");     for (I = 1; I <= Size; I++)e8         printf("%2d  ",gMatrix->ColEliminationOrder[I]);     printf("\n\n");a   /*  PrintMatrix(FALSE); */ o       return;S    #endif   /* ANNOTATE == FULL */  }                   