/*                               /Net/dxcern/userd/timbl/hypertext/WWW/Library/src/HTTCP.html
                              GENERIC NETWORK COMMUNICATION
                                             
   This module has the common code for handling TCP/IP and DECnet connections etc. The
   module is a part of the CERN Common WWW Library.
   
 */

#ifndef HTTCP_H
#define HTTCP_H

#include "tcp.h"

#include "HTUtils.h"
#include "HTAccess.h"

#ifdef SHORT_NAMES
#define HTInetStatus            HTInStat
#define HTInetString            HTInStri
#define HTParseInet             HTPaInet
#endif
/*

Connection Management

   All connections are established through theese two functions.
   
  ACTIVE CONNECTION ESTABLISHMENT
  
   This makes an active connect to the specified host. The HTNetInfo structure is parsed
   in order to handle errors. Default port might be overwritten by any port indication in
   the URL specified as <host>:<port> If it is a multihomed host then HTDoConnect measures
   the time to do the connection and updates the calculated weights in the cache of
   visited hosts.
   
 */

extern int HTDoConnect PARAMS(( HTNetInfo       *net,
                                char            *url,
                                u_short         default_port,
                                u_long          *addr));
/*

  PASSIVE CONNECTION ESTABLISHMENT
  
   This function makes a non-blocking accept on a port and polls every second until
   MAX_ACCEPT_POLL is reached.
   
 */

extern int HTDoAccept PARAMS((HTNetInfo *net));
/*

Caching Hosts Names

   This part of the HTTCP module maintains a cache of all visited hosts so that subsequent
   connects to the same host doesn't imply a new request to the DNS every time.
   
   Multihomed hosts are treated specially in that the time spend on every connect is
   measured and kept in the cache. On the next request to the same host, the IP-address
   with the lowest average connect time is chosen. If one IP-address fails completely,
   e.g. connection refused then it disabled and HTDoConnect tries one of the other
   IP-addresses to the same host.
   
   If the connect fails in the case of at single-homed host then the entry is removed from
   the cache and HTDoConnect tries again asking the DNS.
   
  RECALCULATING THE TIME-WEIGHTS ON MULTIHOMED HOSTS
  
   On every connect to a multihomed host, the average connect time is updated
   exponentially for all the entries.
   
 */

extern void HTTCPAddrWeights PARAMS((char * host, time_t deltatime));
/*

  CONTROL VARIABLES
  
   This parameter determines the maximum number of hosts in the cache. The default number
   is 500.
   
 */

extern unsigned int     HTConCacheSize;
/*

Errors and status indications

   Theese functions return an explanation if an error has occured.
   
  ERRNO MESSAGE
  
   Return error message corresponding to current errno, just like strerror().
   
 */

extern CONST char * HTErrnoString NOPARAMS;
/*

  DESCRIPTION OF WHAT CAUSED THE ERROR
  
   The parameter `where' gives a description of what caused the error, often the name of a
   system call.
   
   This function should only rarely be called directly. Instead the common error function
   HTErrorAdd() should be used as then the error is parsed all the way to the user. The
   function returns a negative status in the unix way.
   
 */

extern int HTInetStatus PARAMS((char * where));
/*

  PARSE A CARDINAL VALUE
  
 */

/*      Parse a cardinal value                                 parse_cardinal()
**      ----------------------
**
** On entry:
**      *pp points to first character to be interpreted, terminated by
**      non 0..9 character.
**      *pstatus points to status already valid,
**      maxvalue gives the largest allowable value.
**
** On exit:
**      *pp points to first unread character,
**      *pstatus points to status updated iff bad
*/

PUBLIC unsigned int HTCardinal PARAMS((int *            pstatus,
                                       char **          pp,
                                       unsigned int     max_value));
/*

Internet Name Server Functions

   The following functions are available to get information about a specified host.
   
  PRODUCE A STRING FOR AN INTERNET ADDRESS
  
   This function is equivalent to the BSD system call inet_ntoa in that it converts a
   numeric 32-bit IP-address to a dotted-notation decimal string. The pointer returned
   points to static memory which must be copied if it is to be kept.
   
 */

extern CONST char * HTInetString PARAMS((struct sockaddr_in * sin));
/*

  PARSE AN INTERNET NODE ADDRESS AND PORT
  
   This function finds the address of a specified host and fills out the sockaddr
   structure. str points to a string with a node name or number, with optional trailing
   colon and port number. sin points to the binary internet or decnet address field.
   
   On exit *sin is filled in. If no port is specified in str, that field is left unchanged
   in *sin. If it is a multihomed host then multihomed is returned as YES. This is used
   for knowing when to measure the setup time for a connection in order to use the fastest
   IP-address on the multihomed host.
   
 */

extern int HTParseInet PARAMS((struct sockaddr_in *     sin,
                               CONST char *             str,
                               BOOL *                   multihomed));
/*

  NAME OF A MACHINE ON THE OTHER SIDE OF A SOCKET
  
   This function should have been called HTGetHostByAddr but for historical reasons this
   is not the case.
   
   Note:This function used to be called HTGetHostName but this is now used to find you own
   host name, see HTGetHostName()
   
 */

extern char * HTGetHostBySock PARAMS((int soc));
/*

  HOST ADDRESS RETUNED FOR SPECIFIED HOST NAME
  
   This function gets the address of the host and puts it in to the socket structure. It
   maintains its own cache of connections so that the communication to the Domain Name
   Server is minimized. If it is a multihomed host, then  `multi' is set to YES.
   
 */

extern char * HTGetHostByName PARAMS((char *host, SockA *sin, BOOL *multi));
/*

  GET NAME OF THIS MACHINE
  
   This functions a char pointer to a static location containing the name of this host.
   
 */

extern CONST char * HTGetHostName NOPARAMS;
/*

  SET NAME OF THIS MACHINE
  
   This function overwrites any other value of current host name. This might be set by the
   user to change the value in the ID value parsed to a news host when posting.
   
 */

extern void HTSetHostName PARAMS((char * host));
/*

  GET USER MAIL ADDRESS
  
   This functions returns a char pointer to a static location containing the mail address
   of the current user. The static location is different from the one of the current host
   name so different values can be assigned. The default value is <USER>@hostname where
   hostname is as returned by HTGetHostName().
   
 */

extern CONST char * HTGetMailAddress NOPARAMS;
/*

  SET USER MAIL ADDRESS
  
   This function overwrites any other value of current mail address. This might be set by
   the user to change the value in the  From field in the HTTP Protocol.
   
 */

extern void HTSetMailAddress PARAMS((char * address));
/*

 */

#endif   /* HTTCP_H */
/*

   End of file  */
