/* FINGER.C  16-May-1995
 * ====================
 * This is a gateway script for handling FINGER requests via an http server.
 *				Foteos Macrides	(macrides@sci.wfeb.edu)
 *
 * Usage:  http://myhost[:port]/htbin/finger/[path_info_host[/port]][?query]
 *  E.g.:  http://sci.wfeb.edu:8002/htbin/finger/
 *
 *	If just a '/' and no host (and optional port) is used for the
 *	path_info_host field, the SERVER_NAME and port 79 are assumed.
 *	If there is no path_info_host at all, an error message is
 *	returned.  If there is no query, an ISINDEX cover page for
 *	submitting a FINGER query is returned.
 *
 * Socket routines are used to communicate with the FINGER server.
 * Make_FINGER.com builds the executable for either MULTINET or UCX.
 * FINGER.COM is used as an interface for invoking FINGER.EXE and uses
 *  Set_DCL_Env.EXE (from the VMSIndex distribution) to set up the
 *  CERN-like CGI environment for the OSU http server.
 * FINGER.COM, FINGER.EXE (and Set_DCL_Env.EXE for the OSU server)
 *  should all be located in the same Exec (htbin) directory.
 */
/****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unixio.h>
#include <unixlib.h>
#include <string.h>
#include <types.h>
#include <errno.h>
#include <ctype.h>
#include <socket.h>
#include <in.h>
#include <netdb.h>

#define FINGER_PORT 79
#define LINEBUF_SIZE 4096

static void error_msg();
static int open_remote ( char *, int );

int main ( int argc, char **argv )
{
    int status, length, query_length, reply_length, port, soc, count, i;
    char *cp, *cp1, *host, key[80], rawq[1024], query[1024], *request;
    char linebuf[LINEBUF_SIZE+2];
    unsigned short include_ISINDEX = TRUE;

    /*
     * If it's a HEAD request, just send a header and exit.
     */
    if ( !strcmp ( getenv ( "WWW_REQUEST_METHOD" ), "HEAD" ) ) {
        printf ( "Content-type: text/html\n\n" );
	return 1;
    }

    /*
     * Get the host and port.  Use SERVER_HOST if only '/' was used as
     * PATH_INFO.  Use FINGER_PORT if no "/port" was included.
     */
    if ( ( host = getenv ( "WWW_PATH_INFO" ) ) == NULL || *host == '\0' ) {
        printf ( "Content-type: text/html\n\n" );
	printf ( "<HTML>\n<HEAD>\n<TITLE>Error</TITLE>\n</HEAD>\n<BODY>" );
	printf (
	    "You must specify a FINGER server as PATH_INFO in the URL.<p>\n" );
	printf ( "You can use simply '/' as the PATH_INFO if the FINGER\n" );
	printf ( "server has the same address as the http server that is\n" );
	printf ( "executing this gateway, and the FINGER server is on\n" );
	printf ( "port %d.<p>\n", FINGER_PORT );
	printf ( "You need not specify the FINGER server port if it's %d,\n",
		 FINGER_PORT );
	printf ( "i.e., use <em>/host</em> as PATH_INFO.<p>\n" );
	printf ( "Otherwise, use <em>/host/%d</em> as PATH_INFO.\n",
		 FINGER_PORT );
	printf ( "</BODY>\n</HTML>\n" );
	return 1;
    }
    if ( *(++host) == '\0') {
        host = getenv ( "WWW_SERVER_NAME" );
	length = strlen ( host );
	for ( i = 0; i < length; i++ )
	    host[i] = tolower ( host[i] );
    }
    if ( ( cp = strchr ( host, '/' ) ) != NULL ) {
	*cp++ = '\0';
	port = atoi ( cp );
    } else {
	port = (int)FINGER_PORT;
    }

    /*
     * If no query, return a cover page.
     */
    if ( ( count = atoi ( getenv ( "WWW_KEY_COUNT" ) ) ) == 0 ) {
        printf ( "Content-type: text/html\n\n" );
	printf ( "<HTML>\n<HEAD>\n" );
	printf ( "<TITLE>FINGER server on %s</TITLE>\n", host );
	printf ( "<ISINDEX>\n<HEAD>\n<BODY>\n" );
        printf (
 "<H2>Query the FINGER Server on <em>%s</em> port %d or a remote host</H2>\n",
	    host, port );
	printf ( "The FINGER utility displays information about the\n" );
	printf ( "users currently logged into the local system or at a\n" );
	printf ( "remote host.  To display information about users at a\n" );
	printf ( "remote host, the remote host must have a FINGER server\n" );
	printf ( "enabled.\n" );
	printf ( "<H1>LOCAL</H1>\n" );
	printf ( "Enter a username as the query to finger that user\n" );
	printf ( "at %s<p>\n", host );
	printf ( "Enter '@' as the query to finger all users currently\n" );
	printf ( "logged on at %s<p>\n", host );
	printf ( "<H1>REMOTE</H1>\n" );
	printf ( "Use <em>username@host</em> to finger a user at another\n" );
	printf ( "host.<p>\n" );
	printf ( "Use <em>@host</em> to finger all users currently\n" );
	printf ( "logged on at another host.<p>\n" );
	printf ( "</BODY>\n</HTML>\n" );
	return 1;
    }

    /*
     * Get the query, save it as rawq, then process it for the submission.
     */
    strcpy ( key, "WWW_KEY_1" ); 
    strcpy ( query, getenv ( key ) );
    query_length = strlen ( query );
    for ( i = 2; i <= count; i++ ) {
	sprintf ( key, "WWW_KEY_%d", i );
	cp = getenv ( key );
	if ( ( strlen ( cp ) + query_length ) > 1022 ) {
	    cp[1022 - query_length] = '\0';
	    i = count;
	}
	strcat ( query, " " );
	strcat ( query, cp );
	query_length = strlen ( query );
    }
    strcpy ( rawq, query );
    for ( i = 0; query[i] != '\0'; i++ )
	query[i] = tolower(query[i]);
    if ( strncmp(query, "finger ", 6) == 0 ) {
	for ( i = 0; query[i+6] != '\0'; i++ )
	    query[i] = query[i+6];
	query[i] = '\0';
    }
    /*
     * If there's an '@' in the query, trim the query there,
     * and if anything follows it, make that the host and
     * check for a :port or /port field.  If the host or
     * port change from those when we started, don't include
     * an ISINDEX tag in the return document because a query
     * on that will go to the wrong host and/or port.
     */
    if ( ( cp = strchr ( query, '@' ) ) != NULL ) {
    	*(cp++) = '\0';
	if ( *cp != '\0' ) {
	    if ( ( ( cp1 = strchr ( cp, ':' ) ) != NULL ) ||
	    	   ( cp1 = strchr ( cp, '/' ) ) != NULL ) {
	        *cp1++ = '\0';
		i = atoi ( cp1 );
		if ( i != port ) {
		    include_ISINDEX = FALSE;
		    port = i;
		}
	    }
	    if ( strcmp ( host, cp ) )
	        include_ISINDEX = FALSE;
	    host = cp;
	}
    }

    /*
     * Connect to the FINGER server.
     */
    soc = open_remote ( host, port );
    if ( soc < 0 )
        return 1;

    /*
     *  Create the request and send it to the FINGER server.
     */
    request = malloc ( strlen ( query ) + 2 );
    sprintf ( request, "%s\r\n", query);
#ifdef MULTINET
    length = socket_write ( soc, request, strlen ( request ) );
#else
    length = write ( soc, request, strlen ( request ) );
#endif /* MULTINET */
    if ( length != strlen ( request ) ) {
        free ( request );
	error_msg ( "500 server error", "Write error to FINGER server\n" );
	return 1;
    }
    free ( request );

    /*
     * Read FINGER server's reply and output to client as PRE-formatted
     * text.  Some day we'll add parsing to create links with the unique
     * ID fields as queries if such a list was returned, but at present
     * what's returned across servers is too inconsistent to be parsed
     * reliably.  So, for now, the user must enter the unique ID (if one
     * is present) or full name (always present) manually and resubmit.
     */
    printf ( "Content-type: text/html\n\n" );
    printf ( "<HTML>\n<HEAD>\n" );
    printf ( "<TITLE>FINGER server on %s</TITLE>\n", host );
    if ( include_ISINDEX == TRUE )
        printf ( "<ISINDEX>\n" );
    printf ( "<HEAD>\n<BODY>\n" );
    printf ( "<H1>%s</H1>\n<PRE>", rawq );
    reply_length = 0;
    while ( length > 0 ) {
#ifdef MULTINET
        length = socket_read ( soc, linebuf, LINEBUF_SIZE );
#else
        length = read ( soc, linebuf, LINEBUF_SIZE );
#endif /* MULTINET */
	if ( length <= 0 ) {
	    linebuf[0] = '\0';
	} else {
	    reply_length += length;
	    linebuf[length] = '\0';
	    /*
	     * Check for any angle brackets and replace them with
	     * square brackets.  We should use HTML entities, but
	     * it's not worth the overhead.
	     */
	    for ( i = 0; i < length; i++ ) {
	        if ( linebuf[i] == '<' )
		    linebuf[i] = '[';
		else if ( linebuf[i] == '>' )
		    linebuf[i] = ']';
	    }
	}
	if ( length > 0 )
	    printf ( "%s", linebuf );
    }
    if ( reply_length < 1 )
        printf ( "No reply from server!  Perhaps access is restricted.\n" );
    printf ( "</PRE>\n</BODY>\n</HTML>\n" );

    return 1;
}

/****************************************************************************/
static void error_msg ( char *status_line, char *message )
{
    printf ( "Content-type: text/plain\nstatus: %s\n\n%s\n",
	status_line, message );
}

/****************************************************************************/
/* Create socket and connect to remote host.
 */
static int open_remote ( char *remhost, int port )
{
    struct sockaddr local, remote;
    struct hostent *hostinfo;
    int i, j, rem_port, sd, status;
    char *alt_port;
    rem_port = port;
#ifdef DEBUG
printf("remote host: %s, port: %d\n", remhost, port );
#endif /* DEBUG */

    alt_port = strchr ( remhost, ':' );
    if ( alt_port ) {
	*alt_port++ = '\0';
	rem_port = atoi ( alt_port );
    }
    hostinfo = gethostbyname ( remhost );
    if ( !hostinfo ) {
	error_msg ( "503 Connect error", "Could not find host address" );
	return -1;
    }
    /*
     * Attempt connect to remote host.
     */
    sd = socket ( AF_INET, SOCK_STREAM, 0 );
    if ( sd < 0 ) {
	    fprintf(stderr,"Error creating socket: %s\n", strerror(errno) );
	return sd;
    }
    local.sa_family = AF_INET;
    for ( j = 0; j < sizeof(local.sa_data); j++ ) local.sa_data[j] = '\0';
    local.sa_data[0] = local.sa_data[1] = 0;
    status = bind ( sd, &local, sizeof ( local ) );
    if ( status < 0 ) {
	error_msg ( "503 Connect error", "Could not bind socket" );
	return -1;
    }

    remote.sa_family = hostinfo->h_addrtype;
    for ( j = 0; j < hostinfo->h_length; j++ ) {
	    remote.sa_data[j+2] = hostinfo->h_addr_list[0][j];
    }
    remote.sa_data[0] = rem_port >> 8;
    remote.sa_data[1] = (rem_port&255);
    status = connect ( sd, &remote, sizeof(remote) );
    if ( status != 0 ) {
	error_msg ( "503 Connect error", "Could not connect to host" );
	return -1;
    }
    return sd;
}

