/* IDENT.C  03-Jun-1995		For the VMS CERN and OSU DECthreads servers.
 * ====================
 * This is a script for sending IDENT requests and setting the
 * REMOTE_IDENT CGI symbol to the client's username, if successful,
 * or to "Unidentified" if the request fails.
 *				Foteos Macrides	(macrides@sci.wfeb.edu)
 *
 * Usage:  http://myhost[:port]/htbin/DCL_script_interface
 *  E.g.:  http://www.wfeb.edu/htbin/ident
 *
 *	A DCL command file interface calls ident.exe to query the
 *	the ident server on the client's host, and set WWW_REMOTE_IDENT
 *	if successful.  That symbol can then be used to regulate the
 *	command file's behavior.
 *
 * Socket routines are used to communicate with the IDENT server.
 * Make_IDENT.com builds the executable for either MULTINET or UCX.
 * IDENT.COM is an example interface for invoking IDENT.EXE.  It
 *  simply shows the CGI symbol environment on return.
 * Set_DCL_Env.EXE (from the VMSIndex distribution) is used to set up
 *  the CERN-like CGI environment for the OSU http server.
 * IDENT.COM, IDENT.EXE (and Set_DCL_Env.EXE for the OSU server)
 *  should all be located in the same Exec (htbin) directory.
 */
/****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unixio.h>
#include <unixlib.h>
#include <string.h>
#include <setjmp.h>
#include <signal.h>
#include <descrip.h>
#include <libclidef.h>
#include <types.h>
#include <errno.h>
#include <ctype.h>
#include <socket.h>
#include <in.h>
#include <netdb.h>

#define IDENT_PORT 113		/* RFC-specified ident server port	    */
#define LINEBUF_SIZE 512	/* Size of request and reply buffers	    */
#define MAX_PREFIX_LEN 100	/* Maximum prefix length, default is "WWW_" */
#define IDENT_TIMEOUT	10	/* Maximum secs to allow for ident request  */

static jmp_buf timebuf;

static int open_remote ( char *, int );
static void timeout ( int sig );

int main ( int argc, char **argv )
{
    int status, length, port, soc, rmt_port, our_port, r_port, o_port, i;
    int table = LIB$K_CLI_LOCAL_SYM, LIB$SET_SYMBOL();
    unsigned int ident_timeout = IDENT_TIMEOUT;
    char *cp, *cp1, *host, request[LINEBUF_SIZE], reply[LINEBUF_SIZE];
    char prefix[256], symname[256], username[256];
    $DESCRIPTOR(symbol,"");
    $DESCRIPTOR(value,"");
    symbol.dsc$a_pointer = symname;
    value.dsc$a_pointer = username;

    /*
     * Get the CGI symbol prefix.
     */
    strncpy( prefix, ((argc > 1) ? argv[1] : "WWW_"), MAX_PREFIX_LEN );
    prefix[MAX_PREFIX_LEN] = '\0';

    /*
     * If the CGI symbol environment wasn't set,
     * or if it's a HEAD request, just exit.
     */
    sprintf( symname, "%sREQUEST_METHOD", prefix );
    if ( ( cp = ( getenv ( symname ) ) ) == NULL ||
    	 !strcmp ( cp, "HEAD" ) ) {
	return 1;
    }

    /*
     * If the server already set the symbol, just exit.
     */
    sprintf( symname, "%sREMOTE_IDENT", prefix );
    if ( ( cp = ( getenv ( symname ) ) ) != NULL && *cp != '\0' ) {
	return 1;
    }

    /*
     * Get the client's host and set the IDENT port.
     */
    sprintf( symname, "%sREMOTE_HOST", prefix );
    host = getenv ( symname );
    length = strlen ( host );
    for ( i = 0; i < length; i++ )
        host[i] = tolower ( host[i] );
    port = (int)IDENT_PORT;

    /*
     *  Create the request we're sending to the IDENT server.
     */
    sprintf( symname, "%sREMOTE_PORT", prefix );
    rmt_port = atoi ( getenv ( symname ) );
    sprintf( symname, "%sSERVER_PORT", prefix );
    our_port = atoi ( getenv ( symname ) );
    sprintf ( request, "%d,%d\r\n", rmt_port, our_port );

    /*
     * Load a dummy username to return on failures.
     */
    sprintf ( username, "Unidentified" );

    /*
     * Set the maximum time for the ident request, and attempt to do it.
     */ 
    if ( setjmp ( timebuf ) == 0 ) {
        signal(SIGALRM, timeout);
	alarm(ident_timeout);

	/*
	 * Connect to the IDENT server.
	 */
	soc = open_remote ( host, port );
	if ( soc < 0 ) {
	    alarm ( 0 );
            goto set_symbol;
	}

	/*
	 * Send the request to the IDENT server.
	 */
#ifdef MULTINET
	length = socket_write ( soc, request, strlen ( request ) );
#else
	length = write ( soc, request, strlen ( request ) );
#endif /* MULTINET */
	if ( length != strlen ( request ) ) {
	    alarm ( 0 );
	    goto set_symbol;
	}

	/*
	 * Read the IDENT server's reply and set the REMOTE_IDENT symbol value.
	 */
#ifdef MULTINET
	length = socket_read ( soc, reply, LINEBUF_SIZE );
	socket_close ( soc );
#else
	length = read ( soc, reply, LINEBUF_SIZE );
	close ( soc );
#endif /* MULTINET */
	alarm(0);
	if ( length < 0 )
	    goto set_symbol;

	/*
	 * Parse the reply for the username.
	 */
	reply[length] = '\0';
        if ( sscanf ( reply, "%u , %u : USERID :%*[^:]:%255s",
			     &r_port, &o_port, username ) == 3 &&
		      r_port == rmt_port && o_port == our_port ) {
	    if ( ( cp = strchr ( username, '\r' ) ) )
	        *cp = 0;
        } else {
	    sprintf ( username, "Unidentified" );
	}
    }

set_symbol:
    sprintf( symname, "%sREMOTE_IDENT", prefix );
    symbol.dsc$w_length = strlen( symname );
    value.dsc$w_length = strlen( username );
    status = LIB$SET_SYMBOL ( &symbol, &value, &table );

    return 1;
}

/***************************************************************************/
/* Create socket and connect to remote host.
 */
static int open_remote ( char *remhost, int port )
{
    struct sockaddr local, remote;
    struct hostent *hostinfo;
    int i, j, rem_port, sd, status;
    rem_port = port;

    hostinfo = gethostbyname ( remhost );
    if ( !hostinfo ) {
	return -1;
    }
    /*
     * Attempt connect to remote host.
     */
    sd = socket ( AF_INET, SOCK_STREAM, 0 );
    if ( sd < 0 ) {
	return sd;
    }
    local.sa_family = AF_INET;
    for ( j = 0; j < sizeof(local.sa_data); j++ ) local.sa_data[j] = '\0';
    local.sa_data[0] = local.sa_data[1] = 0;
    status = bind ( sd, &local, sizeof ( local ) );
    if ( status < 0 ) {
	return -1;
    }

    remote.sa_family = hostinfo->h_addrtype;
    for ( j = 0; j < hostinfo->h_length; j++ ) {
	    remote.sa_data[j+2] = hostinfo->h_addr_list[0][j];
    }
    remote.sa_data[0] = rem_port >> 8;
    remote.sa_data[1] = (rem_port&255);
    status = connect ( sd, &remote, sizeof(remote) );
    if ( status != 0 ) {
	return -1;
    }
    return sd;
}

/***************************************************************************/
/* Handle timeouts
*/
static void timeout ( int sig )
{
    longjmp ( timebuf, sig );
}

