/* WHOIS.C  16-May-1995
 * ====================
 * This is a gateway script for handling WHOIS requests via an http server.
 *				Foteos Macrides	(macrides@sci.wfeb.edu)
 *
 * Usage:  http://myhost[:port]/htbin/whois/[path_info_host[/port]][?query]
 *  E.g.:  http://sci.wfeb.edu:8002/htbin/whois/whois.cc.rochester.edu?.inglis
 *
 *	If just a '/' and no host (and optional port) is used for the
 *	path_info_host field, the SERVER_NAME and port 43 are assumed.
 *	If there is no path_info_host at all, an error message is
 *	returned.  If there is no query, an ISINDEX cover page for
 *	submitting a WHOIS query is returned.
 *
 * Socket routines are used to communicate with the WHOIS server.
 * Make_WHOIS.com builds the executable for either MULTINET or UCX.
 * WHOIS.COM is used as an interface for invoking WHOIS.EXE and uses
 *  Set_DCL_Env.EXE (from the VMSIndex distribution) to set up the
 *  CERN-like CGI environment for the OSU http server.
 * WHOIS.COM, WHOIS.EXE (and Set_DCL_Env.EXE for the OSU server)
 *  should all be located in the same Exec (htbin) directory.
 */
/****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unixio.h>
#include <unixlib.h>
#include <string.h>
#include <types.h>
#include <errno.h>
#include <ctype.h>
#include <socket.h>
#include <in.h>
#include <netdb.h>

#define WHOIS_PORT 43
#define LINEBUF_SIZE 4096

static void error_msg();
static int open_remote ( char *, int );

int main ( int argc, char **argv )
{
    int status, length, port, soc, count, i;
    char *cp, *host, key[80], query[1024], *request;
    char linebuf[LINEBUF_SIZE+2];

    /*
     * If it's a HEAD request, just send a header and exit.
     */
    if ( !strcmp ( getenv ( "WWW_REQUEST_METHOD" ), "HEAD" ) ) {
        printf ( "Content-type: text/html\n\n" );
	return 1;
    }

    /*
     * Get the host and port.  Use SERVER_HOST if only '/' was used as
     * PATH_INFO.  Use WHOIS_PORT if no "/port" was included.
     */
    if ( ( host = getenv ( "WWW_PATH_INFO" ) ) == NULL || *host == '\0' ) {
        printf ( "Content-type: text/html\n\n" );
	printf ( "<HTML>\n<HEAD>\n<TITLE>Error</TITLE>\n</HEAD>\n<BODY>" );
	printf (
	    "You must specify a WHOIS server as PATH_INFO in the URL.<p>\n" );
	printf ( "You can use simply '/' as the PATH_INFO if the WHOIS\n" );
	printf ( "server has the same address as the http server that is\n" );
	printf ( "executing this gateway, and the WHOIS server is on\n" );
	printf ( "port %d.<p>\n", WHOIS_PORT );
	printf ( "You need not specify the WHOIS server port if it's %d,\n",
		 WHOIS_PORT );
	printf ( "i.e., use <em>/host</em> as PATH_INFO.<p>\n" );
	printf ( "Otherwise, use <em>/host/%d</em> as PATH_INFO.\n",
		 WHOIS_PORT );
	printf ( "</BODY>\n</HTML>\n" );
	return 1;
    }
    if ( *(++host) == '\0') {
        host = getenv ( "WWW_SERVER_NAME" );
	length = strlen ( host );
	for ( i = 0; i < length; i++ )
	    host[i] = tolower ( host[i] );
    }
    if ( ( cp = strchr ( host, '/' ) ) != NULL ) {
	*cp++ = '\0';
	port = atoi ( cp );
    } else {
	port = (int)WHOIS_PORT;
    }

    /*
     * If no query, return a cover page.
     */
    if ( ( count = atoi ( getenv ( "WWW_KEY_COUNT" ) ) ) == 0 ) {
        printf ( "Content-type: text/html\n\n" );
	printf ( "<HTML>\n<HEAD>\n" );
	printf ( "<TITLE>WHOIS server on %s</TITLE>\n", host );
	printf ( "<ISINDEX>\n<HEAD>\n<BODY>\n" );
        printf (
	    "<H2>Send query to WHOIS Server on <em>%s</em> port %d</H2>\n",
	    host, port );
	printf ( "If your query does not return complete information\n" );
	printf ( "about the person you are seeking, submit the person's\n" );
	printf ( "unique ID (handle) or full name exactly as reported in\n" );
	printf ( "the initial hit list.<p>\n" );
	printf ( "Some WHOIS servers respond to the query <em>help</em>\n" );
	printf ( "or <em>?</em> with a description of their full command\n" );
	printf ( "syntax and services.<p>\n" );
	printf ( "</BODY>\n</HTML>\n" );
	return 1;
    }

    /*
     * Get the query.
     */
    strcpy ( key, "WWW_KEY_1" ); 
    strcpy ( query, getenv ( key ) );
    length = strlen ( query );
    for ( i = 2; i <= count; i++ ) {
	sprintf ( key, "WWW_KEY_%d", i );
	cp = getenv ( key );
	if ( ( strlen ( cp ) + length ) > 1022 ) {
	    cp[1022 - length] = '\0';
	    i = count;
	}
	strcat ( query, " " );
	strcat ( query, cp );
	length = strlen ( query );
    }
    for ( i = 0; query[i] != '\0'; i++ )
	query[i] = tolower(query[i]);
    if ( strncmp(query, "whois ", 6) == 0 ) {
	for ( i = 0; query[i+6] != '\0'; i++ )
	    query[i] = query[i+6];
	query[i] = '\0';
    }

    /*
     * Connect to the WHOIS server.
     */
    soc = open_remote ( host, port );
    if ( soc < 0 )
        return 1;

    /*
     *  Create the request and send it to the WHOIS server.
     */
    request = malloc ( strlen ( query ) + 2 );
    sprintf ( request, "%s\r\n", query);
#ifdef MULTINET
    length = socket_write ( soc, request, strlen ( request ) );
#else
    length = write ( soc, request, strlen ( request ) );
#endif /* MULTINET */
    if ( length != strlen ( request ) ) {
        free ( request );
	error_msg ( "500 server error", "Write error to WHOIS server\n" );
	return 1;
    }
    free ( request );

    /*
     * Read WHOIS server's reply and output to client as PRE-formatted
     * text.  Some day we'll add parsing to create links with the unique
     * ID fields as queries if such a list was returned, but at present
     * what's returned across servers is too inconsistent to be parsed
     * reliably.  So, for now, the user must enter the unique ID (if one
     * is present) or full name (always present) manually and resubmit.
     */
    printf ( "Content-type: text/html\n\n" );
    printf ( "<HTML>\n<HEAD>\n" );
    printf ( "<TITLE>WHOIS server on %s</TITLE>\n<ISINDEX>\n", host );
    printf ( "<HEAD>\n<BODY>\n" );
    printf ( "<H1>%s</H1>\n<PRE>", query );
    while ( length > 0 ) {
#ifdef MULTINET
        length = socket_read ( soc, linebuf, LINEBUF_SIZE );
#else
        length = read ( soc, linebuf, LINEBUF_SIZE );
#endif /* MULTINET */
	if ( length <= 0 ) {
	    linebuf[0] = '\0';
	} else {
	    linebuf[length] = '\0';
	    /*
	     * Check for any angle brackets and replace them with
	     * square brackets.  We should use HTML entities, but
	     * it's not worth the overhead.
	     */
	    for ( i = 0; i < length; i++ ) {
	        if ( linebuf[i] == '<' )
		    linebuf[i] = '[';
		else if ( linebuf[i] == '>' )
		    linebuf[i] = ']';
	    }
	}
	if ( length > 0 )
	    printf ( "%s", linebuf );
    }
    printf ( "</PRE>\n</BODY>\n</HTML>\n" );

    return 1;
}

/****************************************************************************/
static void error_msg ( char *status_line, char *message )
{
    printf ( "Content-type: text/plain\nstatus: %s\n\n%s\n",
	status_line, message );
}

/****************************************************************************/
/* Create socket and connect to remote host.
 */
static int open_remote ( char *remhost, int port )
{
    struct sockaddr local, remote;
    struct hostent *hostinfo;
    int i, j, rem_port, sd, status;
    char *alt_port;
    rem_port = port;
#ifdef DEBUG
printf("remote host: %s, port: %d\n", remhost, port );
#endif /* DEBUG */

    alt_port = strchr ( remhost, ':' );
    if ( alt_port ) {
	*alt_port++ = '\0';
	rem_port = atoi ( alt_port );
    }
    hostinfo = gethostbyname ( remhost );
    if ( !hostinfo ) {
	error_msg ( "503 Connect error", "Could not find host address" );
	return -1;
    }
    /*
     * Attempt connect to remote host.
     */
    sd = socket ( AF_INET, SOCK_STREAM, 0 );
    if ( sd < 0 ) {
	    fprintf(stderr,"Error creating socket: %s\n", strerror(errno) );
	return sd;
    }
    local.sa_family = AF_INET;
    for ( j = 0; j < sizeof(local.sa_data); j++ ) local.sa_data[j] = '\0';
    local.sa_data[0] = local.sa_data[1] = 0;
    status = bind ( sd, &local, sizeof ( local ) );
    if ( status < 0 ) {
	error_msg ( "503 Connect error", "Could not bind socket" );
	return -1;
    }

    remote.sa_family = hostinfo->h_addrtype;
    for ( j = 0; j < hostinfo->h_length; j++ ) {
	    remote.sa_data[j+2] = hostinfo->h_addr_list[0][j];
    }
    remote.sa_data[0] = rem_port >> 8;
    remote.sa_data[1] = (rem_port&255);
    status = connect ( sd, &remote, sizeof(remote) );
    if ( status != 0 ) {
	error_msg ( "503 Connect error", "Could not connect to host" );
	return -1;
    }
    return sd;
}

