/* Support routines for multi-threaded SSL protocol engine.
 * 
 * The service thread communicates with the client over a single network
 * connection (MST link) using a simple protocol to multiplex 2 logical data 
 * streams (application and ssl data) over this link.
 *
 * Each message sent over the network link has the following structure:
 *    struct rpc_msg { 
 *      char channel;
 *      char function;
 *      short length;
 *      char data[MAX_DATA];
 *    } msg;
 *
 * The channel field designates the virtual data stream this message applies
 * to and is one of:
 *   A - Application data (payload).
 *   R - Remote client connection that initiated the SSL connection.  Encrypted
 *       data is sent over this connection.
 *   G - General data, reserved for future use.
 *
 * The data streams are half-duplex read/write and have following functions:
 *   G - Get, requests that up to msg.length bytes of data be returned.  The
 *       data is returned in the next 'C' function response that matches the
 *       requesting channel.
 *   P - Put, requests that the first msg.length bytes of msg.data be appended
 *       to the designated stream.
 *   C - Confirms a get or put.  Every get and put will get a confirm response,
 *       you cannot initiate another function on a channel until the previous
 *       operation has been confirmed.
 *
 *  The 2 channels may interleave their operations, for example:
 *        Server msg           Client msg
 *         A, Get, 4092          ---->
 *                               <----  R, get, 4092
 *         R, Confirm, {hello}   ---->
 *                               <----  R, put, {srv hello}
 *         R, Confirm, 0         ---->
 *                               .		(SSL handshake completed)
 *                               .              (read first app data).
 *                               <----  A, confirm, {http data}
 *         A, Put, {http data}   ---->
 *                               <----  A, confirm, 0
 *
 *  The length field is not permitted to be larger that 4092 bytes.
 *
 * Author: Dave Jones
 * Date:   24-AUG-1996
 * Revised: 25-SEP-1997		UPdate for 0.8.1
 * Revised: 22-APR-1999		relocate to base_code directory.
 */
#include <stdlib.h>
#include <stdio.h>

#include <string.h>		/* from ssltest.c */
#include <errno.h>
/*
 * Include headers for SSL library calls.  We must assume following
 * compiler qualifiers are used:
 *     /STANDARD=ANSI89/PREFIX=ALL
 *     /INCLUDE=(SOFT:[{ssl include directory}]
 *     /DEFINE=(FLAT_INC,VMS)
 *     /WARNING=DISABLE=DOLLARID
 */
#include "buffer.h"
#include "x509.h"
#include "ssl.h"
#include "err.h"
/*
 * Include header files from http server source tree.
 */
#include "message_service.h"			/* mst_* prototypes */

BIO *bio_err=NULL;
BIO *bio_stdout=NULL;
BIO_METHOD *BIO_s_mst();

/*************************************************************************/
/* Define structure for messages exchanged over network link.
 */
struct rpc_msg {		/* Should have member alignment inhibited */
   char channel;		/* 'A'-app data. 'R'-remote client 'G'-global */
   char function;		/* 'G'-get, 'P'-put, 'C'-confirm, 'X'-close */
   unsigned short int length;	/* Amount of data returned or max to return */
   char data[4092];		/* variable data */
};
#define RPC_HDR_SIZE (sizeof(struct rpc_msg) - 4092)

static SSL_CTX *s_ctx=NULL;
/***************************************************************************/
/* Initialize routine, called once at program load.
 */
int ssl_threads_init ( char * certificate, char *version )
{
    int status, length, pt_initialize(), n;
    /*
     * Set up locking.
     */
    pt_initialize ( );
    /*
     * Initialize standard out and error files, init error messages.
     */
	if (bio_err == NULL)
		if ((bio_err=BIO_new(BIO_s_file())) != NULL)
			BIO_set_fp(bio_err,stderr,BIO_NOCLOSE);
	if (bio_stdout == NULL)
		if ((bio_stdout=BIO_new(BIO_s_file())) != NULL)
			BIO_set_fp(bio_stdout,stdout,BIO_NOCLOSE);
	SSL_load_error_strings();
	SSLeay_add_all_algorithms();
    /*
     * Create a common SSL_CTX for all threads to use and initialize
     * with caller specified certificate.  SSL version is selectable by caller.
     */
        status = 0;
	for ( n = 0; version[n] && version[n] != ':'; n++ );
	if ( strncmp ( version, "23", n ) == 0 ) {
	    s_ctx=SSL_CTX_new(SSLv23_server_method());
	} else if ( strncmp ( version, "3", n ) == 0 ) {
	    s_ctx=SSL_CTX_new(SSLv3_server_method());
	} else {	/* default */
	    s_ctx=SSL_CTX_new(SSLv2_server_method());
	}
	if (s_ctx == NULL) goto end;

	if ( version[n] ) {
	    /* Take portion after first colon as cipher list */
	    status = SSL_CTX_set_cipher_list ( s_ctx, &version[n+1] );
	}

	status = SSL_CTX_use_certificate_file
		(s_ctx,certificate,SSL_FILETYPE_PEM);
	if (status == 1) status = SSL_CTX_use_RSAPrivateKey_file
		(s_ctx,certificate,SSL_FILETYPE_PEM);

end:
    return status;
}
/***************************************************************************/
/*  Handle an SSL session, called concurrently by multiple threads.
 *  Each new network connection creates a new thread which then calls
 *  ssl_do_thread.
 */
int ssl_do_thread ( void *link, 	/* handle for network/mst connection */
	int link_type, 			/* handle type: *ignored* */
	int *id )			/* Thread Id */
{
    int status, length, link_state, written, pt_set_identity();
     struct rpc_msg msg;
	SSL *s_ssl=NULL;
	BIO *c_to_s=NULL;
	BIO *s_to_c=NULL;
	BIO *s_bio=NULL;
	int i;
	/*
	 * Create an SSL server context using our saved certificate info,
	 */
	pt_set_identity ( id );
	s_ssl=SSL_new(s_ctx);
	if (s_ssl == NULL) goto err;
	/*
	 * Create 2 I/O streams for moving data into and out of the
	 * the SSL interpreter via the communication link with our partner.
	 */
	c_to_s=BIO_new(BIO_s_mst());
	s_to_c=BIO_new(BIO_s_mst());
	if ((s_to_c == NULL) || (c_to_s == NULL)) goto err;
	BIO_set_fd ( c_to_s, link_type, (long) link );
	BIO_set_fd ( s_to_c, link_type, (long) link );
	/*
	 * Create ssl stream.
	 */
	s_bio=BIO_new(BIO_f_ssl());
	if (s_bio == NULL) goto err;

	SSL_set_accept_state(s_ssl);
	SSL_set_bio(s_ssl,c_to_s,s_to_c);
	BIO_set_ssl(s_bio,s_ssl,BIO_CLOSE);
	/*
	 * Let partner process/thread drive connection.
	 * Link states: 0-idle, 1-read pending, 2-write pending, 3-closed.
	 */
	for (link_state = 0; link_state < 3; ) {
	    /*
	     * Wait for remote end to request data action on A channel.
	     */
	    while ( link_state == 0 ) {
		status = mst_read ( (mst_link_t) link,
			(char *) &msg, sizeof(msg), &length );
		if ( (status&1) == 0 ) {
		    if ( status != MST_DISCONNECT ) 
			printf("Error in main loop get: %d\n", status );
		    link_state = 3;
		    break;
		}
	   	if ( length < RPC_HDR_SIZE ) {
		    printf("Error in main loop get size: %d\n", length );
		    break;
		    link_state = 3;
		}
	   	if ( msg.channel != 'A' ) {
		    printf("Error in main loop, unexpected channel: %c\n", 
			msg.channel );
		    break;
		    link_state = 3;
		}
		if ( msg.function == 'G' ) {
		    link_state = 1;
		} else if ( msg.function == 'P' ) {
		    link_state = 2;	/* write pending */
		} else if ( msg.function == 'X' ) {
		    link_state = 3;
		} else {
		    link_state = 3;
		}
	    }
	    if ( link_state == 1 ) {
		/*
		 * Remote end want applciation data.
		 */
		i = BIO_read ( s_bio, msg.data, msg.length );
		if ( i < 0 ) link_state = 3;
		else {
		    msg.channel = 'A';
		    msg.function = 'C';		/* confirm */
		    msg.length = i;
		    status = mst_write ( (mst_link_t) link,
				(char *) &msg, i+RPC_HDR_SIZE, &written );
		    if ( (status&1) == 0 ) break;
		    link_state = 0;
		}
	    } else if ( link_state == 2 ) {
		i = BIO_write ( s_bio, msg.data, msg.length );
		if ( i < 0 ) link_state = 3;
		else {
		    msg.channel = 'A';
		    msg.function = 'C';		/* confirm */
		    msg.length = 0;
		    status = mst_write ( (mst_link_t) link,
				(char *) &msg, RPC_HDR_SIZE, &written );
		    if ( (status&1) == 0 ) break;
		    link_state = 0;
		}
	    }
	}
err:
	/* We have to set the BIO's to NULL otherwise they will be
	 * free()ed twice.  Once when th s_ssl is SSL_free()ed and
	 * again when c_ssl is SSL_free()ed.
	 * This is a hack required because s_ssl and c_ssl are sharing the same
	 * BIO structure and SSL_set_bio() and SSL_free() automatically
	 * BIO_free non NULL entries.
	 * You should not normally do this or be required to do this */
	s_ssl->rbio=NULL;
	s_ssl->wbio=NULL;

	if (c_to_s != NULL) BIO_free(c_to_s);
	if (s_to_c != NULL) BIO_free(s_to_c);
	if (s_bio != NULL) BIO_free(s_bio);
	return(0);
}

/* Copyright (C) 1995-1996 Eric Young (eay@mincom.oz.au)
 * All rights reserved.
 * 
 * This file is part of an SSL implementation written
 * by Eric Young (eay@mincom.oz.au).
 * The implementation was written so as to conform with Netscapes SSL
 * specification.  This library and applications are
 * FREE FOR COMMERCIAL AND NON-COMMERCIAL USE
 * as long as the following conditions are aheared to.
 * 
 * Copyright remains Eric Young's, and as such any Copyright notices in
 * the code are not to be removed.  If this code is used in a product,
 * Eric Young should be given attribution as the author of the parts used.
 * This can be in the form of a textual message at program startup or
 * in documentation (online or textual) provided with the package.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *    This product includes software developed by Eric Young (eay@mincom.oz.au)
 * 
 * THIS SOFTWARE IS PROVIDED BY ERIC YOUNG ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * 
 * The licence and distribution terms for any publically available version or
 * derivative of this code cannot be changed.  i.e. this code cannot simply be
 * copied and put under another distribution licence
 * [including the GNU Public Licence.]
 */
