/* Program to implement trivial case of secret sharing */
/* A secret is split n ways */

/* Copyright 1999 Damien Miller */
/* This code is licensed under the GNU GPL. Please see the file */
/* COPYING for details */

/* Optionally prints SHA1 hashes of secrets */

/* $Id: nway.c,v 1.3 1999/08/24 02:12:56 dmiller Exp $ */

#include <stdlib.h>
#include <stdio.h>
#include <getopt.h>
#include <unistd.h>
#include <syslog.h>
#include <errno.h>
#include <string.h>
#include <fcntl.h>
#include <sys/types.h>

#ifdef USE_SHA1
#include <openssl/sha.h>
#endif /* USE_SHA1 */

void join(char *secret, char *shadow_base, int n_shadows);
void split(char *secret, char *shadow_base, int n_shadows);
void usage(void);

int main(int argc, char **argv)
{
	extern char *optarg;
	struct option long_options[] =
	{
		{	"secret", 		1,	NULL,	's'},
		{	"shadow-base", 1,	NULL,	'b'},
		{	"num-shadows", 1,	NULL,	'n'},
		{	"split", 		0,	NULL,	'S'},
		{	"join", 			0,	NULL,	'J'},
		{	"help", 			0,	NULL,	'h'},
		{	NULL, 			0,	NULL,	0},
	};
	enum { NONE, SPLIT, JOIN} mode = NONE;
	char *secret_file = NULL;
	char *shadow_base = NULL;
	int num_shadows = 0;
	int c;
		
	while (1)
	{
		c = getopt_long(argc, argv, "s:b:n:SJh", long_options, NULL);
		if (c == -1)
			break;

		switch (c)
		{
			case 's':
				secret_file = strdup(optarg);
				break;
			case 'b':
				shadow_base = strdup(optarg);
				break;
			case 'n':
				num_shadows = atoi(optarg);
				break;
			case 'S':
				mode = SPLIT;
				break;
			case 'J':
				mode = JOIN;
				break;
			case 'h':
				usage();
				exit(0);
			default:
				fprintf(stderr, "Invalid commandline arguments.\n");
				usage();
				exit(1);
		}
	}
	
	/* Make sure that we have been given enough options */
	if (mode == NONE)
	{	
		fprintf(stderr, "You must specify a mode (--split or --join).\n");
		usage();
		exit(1);
	}
	if (secret_file == NULL)
	{	
		fprintf(stderr, "You must specify a secret file (--secret).\n");
		usage();
		exit(1);
	}
	if (shadow_base == NULL)
	{	
		fprintf(stderr, "You must specify a base filename for the shadows (--shadow-base).\n");
		usage();
		exit(1);
	}
	if (num_shadows < 2)
	{	
		fprintf(stderr, "You must specify at least two shadows (--num-shadows).\n");
		usage();
		exit(1);
	}

	if (mode == SPLIT)
		split(secret_file, shadow_base, num_shadows);
	
	if (mode == JOIN)
		join(secret_file, shadow_base, num_shadows);

	return(0);	
}

void usage(void)
{
	fprintf(stderr, "Usage: nway [OPTIONS]\n");
	fprintf(stderr, "\n");
	fprintf(stderr, "Cryptographically split and join files.\n");
	fprintf(stderr, "\n");
	fprintf(stderr, "Options:\n");
	fprintf(stderr, "\n");
	fprintf(stderr, "--split, -S              Splits secret file into multiple shadows\n");
	fprintf(stderr, "--join, -J               Rejoins split secrets\n");
	fprintf(stderr, "--secret, -s    [file]   Specify filename of secret information\n");
	fprintf(stderr, "--fragment, -f [file]    Specify base filename of shadows\n");
	fprintf(stderr, "--num-shadows, -n [file] Specify number of shadows\n");
	fprintf(stderr, "--help                   Display this help\n");
}

void split(char *secret, char *shadow_base, int n_shadows)
{
	int in, rand, *out;
	int bytes_read_in, bytes_read_rand, bytes_written;
	char secret_buffer[8192], random_buffer[8192];
	int filename_len;
	char *out_filename;
	int c;
	int d;

#ifdef USE_SHA1
	/* Set up SHA1 has structures */
	SHA_CTX sha1_secret;

	SHA1_Init(&sha1_secret);
#endif /* USE_SHA1 */

	/* Allocate space for file descriptors */
	out = malloc(sizeof(*out) * n_shadows);
	
	/* Allocate space for output filenames */
	filename_len = strlen(shadow_base);
	out_filename = malloc(filename_len + 21);
	strcpy(out_filename, shadow_base);

	for(c = 0; c < n_shadows; c++)
	{
		/* Build output filename */
		sprintf(out_filename + filename_len, ".s%04i", c);

		/* Open output file */
		out[c] = open(out_filename, O_WRONLY|O_CREAT|O_TRUNC|O_SYNC, 0600);
		if (out[c] == -1)
		{
			fprintf(stderr, "Couldn't open %s for writing: %s\n", out_filename, strerror(errno));
			exit(2);
		}
	}
	free(out_filename);

	/* Open secret input file */
	in = open(secret, O_RDONLY);
	if (in == -1)
	{
		fprintf(stderr, "Couldn't open %s for reading: %s\n", secret, strerror(errno));
		exit(2);
	}
	
	/* Open random device */
	rand = open("/dev/urandom", O_RDONLY);
	if (in == -1)
	{
		fprintf(stderr, "Couldn't open /dev/urandom for reading: %s\n", strerror(errno));
		exit(2);
	}
	
	while (1)
	{
		/* Read bytes from secret file */
		bytes_read_in = read(in, secret_buffer, sizeof(secret_buffer));
		if (bytes_read_in == -1)
		{
			perror("Input file read failed");
			exit(3);
		} else if (bytes_read_in == 0)
		{
			break;
		}
		
#ifdef USE_SHA1
		/* Update hash of secret */		
		SHA1_Update(&sha1_secret, secret_buffer, bytes_read_in);
#endif /* USE_SHA1 */

		/* XOR each buffer of secret data with an equal sized buffer */
		/* of strong random data */
		for(c = 1; c < n_shadows; c++)
		{
			/* Read some random bytes */
			bytes_read_rand = read(rand, random_buffer, bytes_read_in);
			if (bytes_read_rand != bytes_read_in)
			{
				perror("Random device read failed");
				exit(3);
			}

			/* XOR each byte read by random byte */
			for(d = 0; d < bytes_read_in; d++)
				secret_buffer[d] ^= random_buffer[d];
	
			/* Write each random secret shadow */
			bytes_written = write(out[c], random_buffer, bytes_read_in);
			if (bytes_written != bytes_read_in)
			{
				perror("Output file write failed");
				exit(3);
			}
		}
			
		/* Write out the XOR'd shadow */		
		bytes_written = write(out[0], secret_buffer, bytes_read_in);
		if (bytes_written != bytes_read_in)
		{
			perror("Output file write failed");
			exit(3);
		}
	}
	
#ifdef USE_SHA1

	SHA1_Final(secret_buffer, &sha1_secret);
	printf("    Secret SHA1 hash: ");
	for(d = 0; d < SHA_DIGEST_LENGTH; d++)
		printf("%02x", secret_buffer[d] & 0xFF);
	printf("\n");
#endif /* USE_SHA1 */
	
	close(in);
	close(rand);

	for(c = 0; c < n_shadows; c++)
		close(out[c]);

	exit(0);	
}

void join(char *secret, char *shadow_base, int n_shadows)
{
	int *in, out;
	int bytes_read_in, bytes_written;
	int c;
	int d;
	int finished;
	char share_buffer[8192], secret_buffer[8192];
	int filename_len;
	char *in_filename;

#ifdef USE_SHA1
	SHA_CTX sha1_secret;

	SHA1_Init(&sha1_secret);
#endif /* USE_SHA1 */
	
	/* Allocate space for file descriptors */
	in = malloc(sizeof(*in) * n_shadows);
	
	/* Allocate space for output filenames */
	filename_len = strlen(shadow_base);
	in_filename = malloc(filename_len + 21);
	strcpy(in_filename, shadow_base);

	for(c = 0; c < n_shadows; c++)
	{
		/* Build input filename */
		sprintf(in_filename + filename_len, ".s%04i", c);

		/* Open input file */
		in[c] = open(in_filename, O_RDONLY);
		if (in[c] == -1)
		{
			fprintf(stderr, "Couldn't open %s for reading: %s\n", in_filename, strerror(errno));
			exit(2);
		}
	}
	free(in_filename);

	out = open(secret, O_WRONLY|O_CREAT|O_TRUNC|O_SYNC, 0600);
	if (out == -1)
	{
		fprintf(stderr, "Couldn't open %s for writing: %s\n", secret, strerror(errno));
		exit(2);
	}

	bytes_read_in = 0;
	finished = 0;
	
	while (!finished)
	{
		memset(secret_buffer, '\0', sizeof(secret_buffer));
		memset(share_buffer, '\0', sizeof(share_buffer));
		
		for(c = 0; c < n_shadows; c++)
		{
			bytes_read_in = read(in[c], share_buffer, sizeof(share_buffer));
			if (bytes_read_in == -1)
			{
				perror("Input file read failed");
				exit(3);
			} else if (bytes_read_in == 0)
			{
				finished = 1;
				break;
			}
		
			/* XOR each byte read by random byte */
			for(d = 0; d < bytes_read_in; d++)
				secret_buffer[d] ^= share_buffer[d];
		}
			
#ifdef USE_SHA1
		/* Update hash of secret */		
		SHA1_Update(&sha1_secret, secret_buffer, bytes_read_in);
#endif /* USE_SHA1 */

		/* Write each secret */
		bytes_written = write(out, secret_buffer, bytes_read_in);
		if (bytes_written != bytes_read_in)
		{
			perror("Output file write failed");
			exit(3);
		}
	}
	
#ifdef USE_SHA1
	SHA1_Final(secret_buffer, &sha1_secret);
	printf("    Secret SHA1 hash: ");
	for(c = 0; c < SHA_DIGEST_LENGTH; c++)
		printf("%02x", secret_buffer[c] & 0xFF);
	printf("\n");
#endif /* USE_SHA1 */
	
	for(c = 0; c < n_shadows; c++)
		close(in[c]);
		
	close(out);

	exit(0);	
}
