#include "util.h"
#include "encoder.h"
#include "tables.h"


/************************************************************************
*
* window_subband()
*
* PURPOSE:  Overlapping window on PCM samples
*
* SEMANTICS:
* 32 16-bit pcm samples are scaled to fractional 2's complement and
* concatenated to the end of the window buffer #x#. The updated window
* buffer #x# is then windowed by the analysis window #c# to produce the
* windowed sample #z#
*
************************************************************************/

void window_subband(short **buffer, double z[64], int k)
{
    static double x[2][HAN_SIZE];
    double *xk;
    static int off[2] = {0,0};
    double *ep0,*ep1, *ep2, *ep3, *ep4, *ep5, *ep6, *ep7;

    int i,q;
    double t;

    xk=x[k];

    /* replace 32 oldest samples with 32 new samples */
    for (i=0;i<32;i++) 
      xk[31-i+off[k]] = (double) *(*buffer)++/32768;

    ep0 = &enwindow[0];
    ep1 = &enwindow[64];
    ep2 = &enwindow[128];
    ep3 = &enwindow[192];
    ep4 = &enwindow[256];
    ep5 = &enwindow[320];
    ep6 = &enwindow[384];
    ep7 = &enwindow[448];
 
    q = off[k];

    /* shift samples into proper window positions */
    for( i = 0 ; i<64 ; i++, q++ ) {        
        t =  xk[(q)&(512-1)] * *ep0++;
        t += xk[(q+64)&(512-1)] * *ep1++;
        t += xk[(q+128)&(512-1)] * *ep2++;
        t += xk[(q+192)&(512-1)] * *ep3++;
        t += xk[(q+256)&(512-1)] * *ep4++;
        t += xk[(q+320)&(512-1)] * *ep5++;
        t += xk[(q+384)&(512-1)] * *ep6++;
        t += xk[(q+448)&(512-1)] * *ep7++;
        z[i] = t;
        }
    
    off[k] += 480;              /*offset is modulo (HAN_SIZE-1)*/
    off[k] &= HAN_SIZE-1;

}
 
/************************************************************************
*
* filter_subband()
*
* PURPOSE:  Calculates the analysis filter bank coefficients
*
* SEMANTICS:
*      The windowed samples #z# is filtered by the digital filter matrix #m#
* to produce the subband samples #s#. This done by first selectively
* picking out values from the windowed samples, and then multiplying
* them by the filter matrix, producing 32 subband samples.
*
************************************************************************/


void filter_subband(double z[64], double s[SBLIMIT]) {

   double yprime[32];
   int i,j;
   static int init = 0;
   static double filter[16][32];
   
   double s0,s1;

   if(!init) {
      init=1;
      for (i=0; i<16; i++)  for (j=0; j<32; j++)
         filter[i][j] = cos((double)((2*i+1)*j*PI64));
      }

   yprime[0] = z[16];

   for( i=1; i<=16; i++ ) yprime[i] = z[i+16]+z[16-i];
   for( i=17; i<=31; i++ ) yprime[i] = z[i+16]-z[80-i];
 
   for( i=0; i<16; i++ ) {
      s0 = s1 = 0.0;

      for( j=0; j<32; j+=2 ) {
         s0 += filter[i][j]*yprime[j];
         s1 += filter[i][j+1]*yprime[j+1];
         }

     s[i] = s0+s1;
     s[31-i] = s0-s1;

     }

   }


