/*************************************************************************************************************
 *
 * CHECK_NRPE.C
 *
 * Program: Remote monitoring daemon plugin for NetSaint
 * License: GPL
 * Copyright (c) 1999 Ethan Galstad (netsaint@linuxbox.com)
 *
 * Last Modified: 06-21-1999
 *
 * Command line: CHECK_NRPE <host_address> [-p port] [-c command] [-wt warn_time] [-ct crit_time] [-to to_sec]
 *
 * Description:
 *
 * This plugin will attempt to connect to the NetSaint remote plugin executor daemon on the
 * specified server and port.  The daemon will attempt to run the command defined as
 * [command].  Program output and return code are sent back from the daemon and displayed
 * as this plugin's own output and return code.
 *
 ************************************************************************************************************/

#include <stdio.h>		/* obligatory includes */
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <string.h>
#include <time.h>
#include <signal.h>

#include "nrpe.h"

#define OK		0
#define ERROR		-1

#define TRUE		1
#define FALSE		0

#define	STATE_CRITICAL 2	/* service state return codes */
#define STATE_WARNING  1
#define STATE_OK       0
#define STATE_UNKNOWN  -1

#define DEFAULT_SERVER_PORT	5666
#define DEFAULT_TIMEOUT		10	/* timeout after 10 seconds */
#define DEFAULT_NRPE_COMMAND	"_NRPE_CHECK"

#define NRPE_PROTO	"tcp"

#define MAX_INPUT_BUFFER	256

time_t start_time,end_time;
int server_port=DEFAULT_SERVER_PORT;
char server_name[256];
char query_string[256]=DEFAULT_NRPE_COMMAND;;
int timeout=DEFAULT_TIMEOUT;
int warning_time=0;
int check_warning_time=FALSE;
int critical_time=0;
int check_critical_time=FALSE;

int my_connect(char *address,int port,char *protocol,int *sd);
int process_arguments(int,char **);
void alarm_handler(int);


int main(int argc, char **argv){
	int sd;
	int rc;
	int result;
	char buffer[MAX_INPUT_BUFFER];
	packet send_packet;
	packet receive_packet;

	result=process_arguments(argc,argv);

	if(result!=OK){

		printf("Incorrect command line arguments supplied\n");
		printf("\n");
		printf("NRPE Plugin for NetSaint\n");
		printf("Copyright (c) 1999 Ethan Galstad (netsaint@linuxbox.com)\n");
		printf("Last Modified: 06-21-1999\n");
		printf("License: GPL\n");
		printf("\n");
		printf("Usage: %s <host_address> [-p port] [-c command] [-wt warn_time] [-ct crit_time] [-to to_sec]\n",argv[0]);
		printf("\n");
		printf("Options:\n");
		printf(" <host_address> = The IP address of the host running the monitoring daemon\n");
		printf(" [port]         = The port on which the daemon is running - default is %d\n",DEFAULT_SERVER_PORT);
		printf(" [command]      = The name of the command that the remote daemon should run\n");
		printf(" [warn_time]    = Response time in seconds necessary to result in a warning status\n");
		printf(" [crit_time]    = Response time in seconds necessary to result in a critical status\n");
		printf(" [to_sec]       = Number of seconds before connection attempt times out - default is %d seconds\n",DEFAULT_TIMEOUT);
		printf("\n");

		return STATE_UNKNOWN;
	        }

	/* initialize alarm signal handling */
	signal(SIGALRM,alarm_handler);

	/* set socket timeout */
	alarm(timeout);

	time(&start_time);

	/* try to connect to the host at the given port number */
	result=my_connect(server_name,server_port,NRPE_PROTO,&sd);

	/* we connected, so close connection before exiting */
	if(result==STATE_OK){

		/* send the query packet */
		bzero(&send_packet,sizeof(send_packet));
		send_packet.packet_type=QUERY_PACKET;
		send_packet.packet_version=NRPE_PACKET_VERSION_1;
		send_packet.buffer_length=strlen(query_string);
		strcpy(&send_packet.buffer[0],query_string);

		rc=send(sd,&send_packet,sizeof(send_packet),0);

		if(rc==-1){
			printf("check_nrpe: Error sending query to host\n");
			close(sd);
			return STATE_UNKNOWN;
		        }

		/* wait for the response packet */
		bzero(&receive_packet,sizeof(receive_packet));
		rc=recv(sd,&receive_packet,sizeof(receive_packet),0);

		if(rc==-1){
			printf("check_nrpe: Error receiving data from host\n");
			close(sd);
			return STATE_UNKNOWN;
		        }


		time(&end_time);

		result=STATE_OK;

		if(check_critical_time==TRUE && (end_time-start_time)>critical_time)
			result=STATE_CRITICAL;
		else if(check_warning_time==TRUE && (end_time-start_time)>warning_time)
			result=STATE_WARNING;

		if(result!=STATE_OK)
			printf("NRPE problem - %d second response time\n",(int)(end_time-start_time));
		else{
			result=receive_packet.result_code;
			printf("%s\n",receive_packet.buffer);
		        }

		/* close the connection */
		close(sd);
	        }

	/* reset the alarm */
	alarm(0);

	return result;
        }



int process_arguments(int argc, char **argv){
	int x;


	/* no options were supplied */
	if(argc<2)
		return ERROR;

	/* first option is always the server name/address */
	strcpy(server_name,argv[1]);

	for(x=3;x<=argc;x++){

		if(!strcmp(argv[x-1],"-c")){
			if(x<argc){
				strcpy(query_string,argv[x]);
				x++;
			        }
			else
				return ERROR;
		        }
		else if(!strcmp(argv[x-1],"-p")){
			if(x<argc){
				server_port=atoi(argv[x]);
				x++;
			        }
			else
				return ERROR;
		        }
		else if(!strcmp(argv[x-1],"-to")){
			if(x<argc){
				timeout=atoi(argv[x]);
				if(timeout<=0)
					return ERROR;
				x++;
			        }
			else
				return ERROR;
		        }
		else if(!strcmp(argv[x-1],"-wt")){
			if(x<argc){
				warning_time=atoi(argv[x]);
				check_warning_time=TRUE;
				x++;
			        }
			else
				return ERROR;
		        }
		else if(!strcmp(argv[x-1],"-ct")){
			if(x<argc){
				critical_time=atoi(argv[x]);
				check_critical_time=TRUE;
				x++;
			        }
			else
				return ERROR;
		        }
		else
			return ERROR;
	        }

	return OK;
        }



int my_connect(char *host_name,int port,char *protocol,int *sd){
	struct sockaddr_in servaddr;
	struct hostent *hp;
	struct protoent *ptrp;
	int result;

	bzero((char *)&servaddr,sizeof(servaddr));
	servaddr.sin_family=AF_INET;
	servaddr.sin_port=htons(port);

	/* try to bypass using a DNS lookup if this is just an IP address */
	if(!inet_aton(host_name,&servaddr.sin_addr)){

		/* else do a DNS lookup */
		hp=gethostbyname((const char *)host_name);
		if(hp==NULL){
			printf("check_nrpe: Invalid host name '%s'\n",host_name);
			return STATE_UNKNOWN;
		        }
    
		memcpy(&servaddr.sin_addr,hp->h_addr,hp->h_length);
	        }

	/* map TCP transport protocol name to protocol number */
	if(((int)(ptrp=getprotobyname(protocol)))==0){
		printf("check_nrpe: Cannot map \"%s\" to protocol number\n",protocol);
		return STATE_UNKNOWN;
	        }

	/* create a socket */
	*sd=socket(PF_INET,SOCK_STREAM,ptrp->p_proto);
	if(*sd<0){
		printf("check_nrpe: Socket creation failed\n");
		return STATE_UNKNOWN;
	        }

	/* open a connection */
	time(&start_time);
	result=connect(*sd,(struct sockaddr *)&servaddr,sizeof(servaddr));
	if(result<0){
		switch(errno){  
		case ECONNREFUSED:
			printf("check_nrpe: Connection refused at port %d by host\n",port);
			break;
		case ETIMEDOUT:
			printf("check_nrpe: Timeout while attempting connection\n");
			break;
		case ENETUNREACH:
			printf("check_nrpe: Network is unreachable\n");
			break;
		default:
			printf("check_nrpe: Connection refused or timed out\n");
		        }

		return STATE_CRITICAL;
	        }

	return STATE_OK;
        }



void alarm_handler(int sig){

	printf("Socket timeout after %d seconds\n",timeout);

	exit(STATE_CRITICAL);
        }
