/****************************************************************
 *
 * NRPE.C
 *
 * Program: NetSaint Remote Plugin Executor
 * Version: 0.1b3
 * License: GPL
 * Copyright (c) 1999 Ethan Galstad (netsaint@linuxbox.com)
 *
 * Last Modified: 06-21-1999
 *
 * Command line: NRPE <config_file>
 *
 * Description:
 *
 * This program is designed to run as a background process and
 * handle incoming requests (from the host running NetSaint) for
 * plugin execution.  It is useful for running "local" plugins
 * such as check_users, check_load, check_disk, etc. without
 * having to use rsh or ssh.
 ***************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <errno.h>

#include "nrpe.h"

#define TEMP_FILE	"./nrpe.tmp"

#define	OK		0
#define ERROR		1

#define	STATE_UNKNOWN	-1
#define STATE_OK	0
#define STATE_WARNING	1
#define STATE_CRITICAL	2

#define MAX_INPUT_BUFFER	1024

#define DEFAULT_SERVER_PORT	5666

#define NRPE_HELLO_COMMAND	"_NRPE_CHECK"

void wait_for_connections(void);
void handle_connection(int);
int read_config_file(char *);
int my_system(char *);
int add_command(char *,char *);
command *find_command(char *);
void sighandler(int);
void free_memory(void);

int server_port=DEFAULT_SERVER_PORT;
struct sockaddr_in allowed_ip;
command *command_list=NULL;


int main(int argc, char **argv){
	int result;

	if(argc!=2){

		printf("NRPE - NetSaint Remote Plugin Executor 0.1b3\n");
		printf("Copyright (c) 1999 Ethan Galstad (netsaint@linuxbox.com)\n");
		printf("Last Modified: 06-21-1999\n");
		printf("License: GPL\n");
		printf("\n");

		printf("Usage: %s <config_file>\n",argv[0]);
		printf("\n");
		exit(STATE_UNKNOWN);
		}

	/* read the config file */
	result=read_config_file(argv[1]);	

	/* exit if there are errors... */
	if(result==ERROR){
		printf("Bailing out due to errors...\n");
		return STATE_UNKNOWN;
		}

	
	/* daemonize... */
	if(fork()==0){

		/* else wait for connections */
		wait_for_connections();
	        }

	return STATE_UNKNOWN;
	}


int read_config_file(char *filename){
	FILE *fp;
	char input_buffer[MAX_INPUT_BUFFER];
	char *temp_buffer;
	char *varname;
	char *varvalue;
	int line;


	/* open the config file for reading */
	fp=fopen(filename,"r");

	/* exit if we couldn't open the config file */
	if(fp==NULL)
		return ERROR;

	line=0;
	while(fgets(input_buffer,MAX_INPUT_BUFFER-1,fp)){

		line++;

		/* skip comments and blank lines */
		if(input_buffer[0]=='#')
			continue;
		if(input_buffer[0]=='\x0')
			continue;
		if(input_buffer[0]=='\n')
			continue;

		/* get the variable name */
		varname=strtok(input_buffer,"=");
		if(varname==NULL){
			printf("Error: NULL variable name - Line %d\n",line);
			return ERROR;
		        }

		/* get the variable value */
		varvalue=strtok(NULL,"\n");
		if(varvalue==NULL){
			printf("Error: NULL variable value - Line %d\n",line);
			return ERROR;
		        }

		if(!strcmp(varname,"server_port")){
			server_port=atoi(varvalue);
			if(server_port<1024){
				printf("Error: Invalid port number - Line %d\n",line);
				return ERROR;
			        }
		        }

		else if(!strcmp(varname,"allowed_ip")){
			if(strlen(input_buffer)>15){
				printf("Error: Invalid IP address length - Line %d\n",line);
				return ERROR;
			        }
			if(!inet_aton(varvalue,&allowed_ip.sin_addr)){
				printf("Error: Invalid IP address - Line %d\n",line);
				return ERROR;
			        }
		        }

		else if(strstr(input_buffer,"command[")){
			temp_buffer=strtok(varname,"[");
			temp_buffer=strtok(NULL,"]");
			if(temp_buffer==NULL){
				printf("Error: Invalid command - Line %d\n",line);
				return ERROR;
			        }
			add_command(temp_buffer,varvalue);
		        }

		else{
			printf("Error: Unknown configuration file option - Line %d\n",line);
			return ERROR;
		        }

	        }


	/* close the config file */
	fclose(fp);

	return OK;
	}


int add_command(char *command_name, char *command_line){
	command *new_command;

	/* allocate memory for the new command */
	new_command=(command *)malloc(sizeof(command));
	if(new_command==NULL)
		return ERROR;

	strcpy(new_command->command_name,command_name);
	strcpy(new_command->command_line,command_line);

	/* add new command to head of list in memory */
	new_command->next=command_list;
	command_list=new_command;

	printf("Added command[%s]=%s\n",command_name,command_line);

	return OK;
        }


command *find_command(char *command_name){
	command *temp_command;

	for(temp_command=command_list;temp_command!=NULL;temp_command=temp_command->next)
		if(!strcmp(command_name,temp_command->command_name))
			return temp_command;

	return NULL;
        }



void wait_for_connections(void){
	struct sockaddr_in myname;
	struct sockaddr_in *nptr;
	struct sockaddr addr;
	char buf[80];
	int rc;
	int sock, new_sd, addrlen, cnt;
	char allowed_host[16];
	char connecting_host[16];

	/* create a socket for listening */
	sock=socket(AF_INET, SOCK_STREAM, 0);

	/* exit if we couldn't create the socket */
	if(sock<0){
    		printf("Error: Network server socket failure %d\n", errno);
		return;
		}

	myname.sin_family=AF_INET;
	myname.sin_port=htons(server_port);
	myname.sin_addr.s_addr=INADDR_ANY;
	bzero(&myname.sin_zero,8);

	/* bind the address to the Internet socket */
	if(bind(sock,(struct sockaddr *)&myname,sizeof(myname))<0){
		printf("Error: Network server bind failure %d\n",errno);
		return;
	        }

	/* open the socket for listening */
	if(listen(sock,5)<0){
	    	printf("Error: Network server listen failure %d\n", errno);
		return;
		}

	/* Ignore child process termination */
	signal(SIGCHLD,SIG_IGN);

	/* Trap other signals */
	signal(SIGPIPE,SIG_IGN);
	signal(SIGINT,sighandler);
	signal(SIGQUIT,sighandler);
	signal(SIGTERM,sighandler);
	signal(SIGBUS,sighandler);
	signal(SIGFPE,sighandler);
	signal(SIGILL,sighandler);
	signal(SIGHUP,sighandler);
	signal(SIGABRT,sighandler);	
	signal(SIGSEGV,sighandler);
	signal(SIGKILL,sighandler);

	printf("\n");
	printf("Listening for connections on port %d\n",htons(myname.sin_port));
	printf("Allowed IP address: %s\n",(char *)inet_ntoa(allowed_ip.sin_addr));
	printf("\n");


	/* listen for connection requests - fork() if we get one */
	while(1){

    		new_sd=accept(sock,0,0);

		if(new_sd<0){
			printf("Error: Network server accept failure %d\n", errno);
			close(sock);
			return;
			}

		/* child process should handle the connection */
    		if(fork()==0){

		  /* child does not need to listen for connections */
      			close(sock);  


			/* find out who just connected... */
			addrlen=sizeof(addr);
			rc=getpeername(new_sd,&addr,&addrlen);
			if(rc<0){
				printf("Error: Network server getpeername() failure %d\n",errno);
				close(new_sd);
				return;
		                }

			nptr=(struct sockaddr_in *)&addr;
			printf("Connection from %s port %d\n",inet_ntoa(nptr->sin_addr),nptr->sin_port);

			/* is this is blessed machine? */
			sprintf(allowed_host,"%s",inet_ntoa(allowed_ip.sin_addr));
			sprintf(connecting_host,"%s",inet_ntoa(nptr->sin_addr));
			if(strcmp(allowed_host,connecting_host))
				printf("Host is not allowed to connect!\n");
			else{
				printf("Host address checks out ok\n");
				handle_connection(new_sd);
			        }

			printf("Connection from %s closed.\n",inet_ntoa(nptr->sin_addr));
      			close (new_sd); /* close prior to exiting  */
      			return;
    			}

		/* parent doesn't need the new connection */
		else
			close(new_sd);
  		}

	/* we shouldn't ever get here... */
	return;
	}



void handle_connection(int sock){
	command *temp_command;
	packet receive_packet;
	packet send_packet;
	char buffer[MAX_INPUT_BUFFER];
	char command[MAX_INPUT_BUFFER];
	int result;
	int rc;
	FILE *fp;


	printf("Handling the connection...\n");

	/* get the request packet */
	bzero(&receive_packet,sizeof(receive_packet));
	rc=recv(sock,&receive_packet,sizeof(receive_packet),0);

	if(rc==-1){
		printf("Error reading request from client, bailing out...\n");
		return;
	        }

	/* make sure this is the right type of packet */
	if(receive_packet.packet_type!=QUERY_PACKET || receive_packet.packet_version!=NRPE_PACKET_VERSION_1){
		printf("Received invalid packet, bailing out...\n");
		return;
	        }

	printf("Host is asking for command '%s' to be run...\n",receive_packet.buffer);

	/* if this is the version check command, just spew it out */
	if(!strcmp(&receive_packet.buffer[0],NRPE_HELLO_COMMAND)){
		sprintf(buffer,"NRPE v0.1b3");
		printf("Response: %s\n",buffer);
		result=STATE_OK;
	        }

	/* find the command we're supposed to run */
	else{
		temp_command=find_command(receive_packet.buffer);
		if(temp_command==NULL){
			sprintf(buffer,"NRPE: Command '%s' not defined",receive_packet.buffer);
			printf("%s\n",buffer);
			result=STATE_UNKNOWN;
	                }

		else{

			/* get the command line to run */
			sprintf(command,"%s > %s 2> /dev/null",temp_command->command_line,TEMP_FILE);

			printf("Running command: %s\n",command);

			/* run the command */
			result=my_system(command);
	
			/* get the command output */
			fp=fopen(TEMP_FILE,"r");
			if(fp==NULL)
				strcpy(buffer,"Unable to open temp file");
			else{
				if(!fgets(buffer,MAX_INPUT_BUFFER-1,fp))
					strcpy(buffer,"Unable to read output");
				fclose(fp);
	                        }
			
			/* remove the temp file */
			remove(TEMP_FILE);
		        }
	        }

	/* strip newline character from end of output buffer */
	if(buffer[strlen(buffer)-1]=='\n')
		buffer[strlen(buffer)-1]='\x0';

	/* make sure the buffer isn't too long - if it is, chop it off */
	if(strlen(buffer)>MAX_PACKETBUFFER_LENGTH-1)
		buffer[MAX_PACKETBUFFER_LENGTH-1]='\x0';

	/* fill the response packet with data */
	bzero(&send_packet,sizeof(send_packet));
	send_packet.packet_type=RESPONSE_PACKET;
	send_packet.packet_version=NRPE_PACKET_VERSION_1;
	send_packet.result_code=result;
	send_packet.buffer_length=strlen(buffer);
	strcpy(&send_packet.buffer[0],buffer);
	
	/* send the response back to the client */
	send(sock,&send_packet,sizeof(send_packet),0);

	printf("Return Code: %d, Output: %s\n",result,buffer);

	return;
        }



int my_system(char *command){
        int pid;
	int status;
	int result;
	extern int errno;
	extern char **environ;


	/* if no command was passed, return with no error */
	if(command==NULL)
	        return STATE_OK;

	/* fork */
	pid=fork();

	/* return an error if we couldn't fork */
	if(pid==-1)
	        return STATE_UNKNOWN;  

	/* execute the command in the child process */
        if (pid==0){
		char *argv[4];
		argv[0]="sh";
		argv[1]="-c";
		argv[2]=command;
		argv[3]=0;
		execve("/bin/sh",argv,environ);
		exit(STATE_UNKNOWN);
	        }

	/* wait for the command to finish */
        do{
                if (waitpid(pid, &status, 0) == -1) {
		        if (errno != EINTR)
                                return STATE_UNKNOWN;
                        } 

		/* get the exit code returned from the program */
		result=WEXITSTATUS(status);

		/* check the bounds for the return code */
		if(result<0||result>2)
			return STATE_UNKNOWN;

		return result;
           } while(1);

        return STATE_OK;
        }



/* handle signals */
void sighandler(int sig){

	free_memory();
	exit(0);

	return;
        }


void free_memory(void){
	command *this_command;
	command *next_command;
	
	/* free memory for the command list */
	this_command=command_list;
	while(this_command!=NULL){
		next_command=this_command->next;
		free(this_command);
		this_command=next_command;
	        }

	return;
        }
