/*
Gnusniff - Network packet sniffer
Copyright (C) 1998 Peter Hawkins

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <stdio.h>
#include <string.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netipx/ipx.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <glib.h>
#include "hosts.h"
#include "interface.h"
#include "utils.h"
#include "p_ipx.h"

static int lookup_func(host_t *host)
{
 struct hostent *he;

 he = gethostbyaddr((char *)&((struct sockaddr_in *)&host->address)->sin_addr, 4, AF_INET);

 pt_mutex_enter(host->mutex);
 host->status = 1;
 if ((he!=NULL) && (he->h_name!=NULL)) strncpy(host->hostname, he->h_name, 256);
 pt_mutex_exit(host->mutex);
 
 pthread_exit(NULL);
}

void lookuphost(host_t *host)
{
 pthread_attr_t attr;

 if (host->status!=0) return; /* If a DNS query has already occurred on this host. */
 
 pthread_attr_init(&attr);
 pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);
   
 if (pthread_create(&host->thread, &attr, (void *)lookup_func, host) != 0)
 {
    fprintf(stderr, "Error creating a reverse DNS lookup thread!\n");
    exit(1);
 }
}

host_t *addaddrtohost(host_t *node, struct sockaddr *addr)
{
 memcpy(&node->address, addr, sizeof(struct sockaddr));
 switch (addr->sa_family) {
   case AF_INET: strncpy(node->hostname, inet_ntoa(((struct sockaddr_in *)&node->address)->sin_addr), 256); break;
   case AF_IPX: strncpy(node->hostname, ipxaddr_string(ntohl(((struct sockaddr_ipx *)&node->address)->sipx_network), ((struct sockaddr_ipx *)&node->address)->sipx_node), 256); break;
 }
 strncpy(node->swaddress, node->hostname, ADDRESS_STRING_LENGTH);
 return node;
}

/* Compare two socket addresses. Returns TRUE if equal. */
gboolean sockcmp(struct sockaddr *a, struct sockaddr *b)
{
/* If the addresses are from different address families, they can't possibly be
  equal. */
 if (a->sa_family != b->sa_family) return FALSE;
 switch (a->sa_family) {
   case AF_INET: return (( *((int *) &((struct sockaddr_in *)a)->sin_addr) )==( *((int *) &((struct sockaddr_in *)b)->sin_addr) ));
   case AF_IPX: {
     struct sockaddr_ipx *aipx = (struct sockaddr_ipx *)a, *bipx = (struct sockaddr_ipx *)b;
     if (aipx->sipx_network != bipx->sipx_network) return FALSE;
     return (!memcmp(aipx->sipx_node, bipx->sipx_node, IPX_NODE_LEN));
   }
 }
 return FALSE;
}

host_t *gethostforaddr(struct packet_t *pkt, struct sockaddr *addr)
{
 host_t *node;
 if (pkt->parent->hostlist) {
   node = pkt->parent->hostlist;
   if (sockcmp(&node->address, addr)) return node;

   while (node->next!=NULL) {
     node = node->next;
     if (sockcmp(&node->address, addr)) return node;
   }
   node->next = g_malloc0(sizeof(host_t));
   node = node->next;
 } else {
   node = pkt->parent->hostlist = g_malloc0(sizeof(host_t));
 }
 addaddrtohost(node, addr);

 return node;
}

host_t *gethostforhwaddr(struct packet_t *pkt, char *hwaddr, u_int len)
{
 host_t *node;

 len = MIN(len, 64);

 if (pkt->parent->hostlist) {
   node = pkt->parent->hostlist;
   if (!strncmp(hwaddr, node->hwaddress, len)) return node;

   while (node->next!=NULL) {
     node = node->next;
     if (!strncmp(hwaddr, node->hwaddress, len)) return node;
   }
   node->next = g_malloc0(sizeof(host_t));
   node = node->next;
 } else {
   node = pkt->parent->hostlist = g_malloc0(sizeof(host_t));
 }
 strncpy(node->hwaddress, hwaddr, len);

 return node;
}


