/*
Gnusniff - Network packet sniffer
Copyright (C) 1998 Peter Hawkins

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <net/if_arp.h>
#include <net/if.h>
#include <sys/time.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <gnome.h>
#include "main.h"
#include "net-support.h"
#include "mainwnd.h"
//#include "pktview.h"
#include "gtkhex.h"
#include "sniff.h"
#include "utils.h"
#include "pictures.h"


static const gchar *authors[] = {"dph-man", NULL };

void cb_about()
{
 GtkWidget *About = gnome_about_new ( _(APPNAME), "0.0.5",
                        _("Copyright Peter Hawkins (C) 1998"),
                        authors,
                        "All bugs are my cat's fault\nI don't have a cat :-)",
                        "/usr/local/share/pixmaps/gnoapp-logo.xpm");
 gtk_widget_show(About);
}

void cb_destroy()
{
 gtk_main_quit();
 return;
}

/* This callback is called if either the Start Capturing or Stop Capturing
   buttons are pressed. */
void cb_btnclicked(GtkWidget *widget, gpointer data)
{
 if (main_wnd.current_if!=NULL) {
  if (data) {
    sniff_begin(main_wnd.current_if);
    gtk_widget_set_sensitive(main_wnd.startcapturebtn, FALSE);
    gtk_widget_set_sensitive(main_wnd.stopcapturebtn, TRUE);
  } else {
    sniff_end(main_wnd.current_if);
    gtk_widget_set_sensitive(main_wnd.startcapturebtn, TRUE);
    gtk_widget_set_sensitive(main_wnd.stopcapturebtn, FALSE);
  }
 }
}

static const char *if_port_text[][4] = {
  /* Keep in step with <linux/netdevice.h> */
  { "unknown", NULL , NULL, NULL },
  { "10base2", "bnc", "coax", NULL },
  { "10baseT", "utp", "tpe", NULL },
  { "AUI", "thick", "db15", NULL },
  { "100baseT", NULL, NULL, NULL },
  { "100baseTX", NULL, NULL, NULL },
  { "100baseFX", NULL, NULL, NULL },
  { NULL, NULL, NULL, NULL },
};

static void update_if_list(interface *ptr, GtkWidget *iflist)
{
  char buffer[1024];
  char *temp[2];
  
  struct aftype *ap;
  struct hwtype *hw;
  int hf;
  int can_compress = 0;
#if HAVE_AFIPX
  static struct aftype *ipxtype=NULL;
#endif
#if HAVE_AFECONET
  static struct aftype *ectype = NULL;
#endif
#if HAVE_AFATALK
   static struct aftype *ddptype = NULL;
#endif
#if HAVE_AFINET6
  FILE *f;
  char addr6[40], devname[20];
  struct sockaddr_in6 sap;
  int plen, scope, dad_status, if_idx;
  extern struct aftype inet6_aftype;
  char addr6p[8][5];
  
  if (!strncmp(ptr->name, "sit", 3)) 
    ptr->addr.sa_family = AF_INET6;      /* fix this up properly one day */
#endif

  gtk_clist_freeze(GTK_CLIST(iflist));
  gtk_clist_clear(GTK_CLIST(iflist));

  ap = get_afntype(ptr->addr.sa_family);
  if (ap == NULL) ap = get_afntype(0);

  hf=ptr->type;

  if (strncmp(ptr->name, "lo", 2) == 0)
  	hf=255;
  	
  if (hf==ARPHRD_CSLIP || hf==ARPHRD_CSLIP6)
    can_compress = 1;
  
  hw = get_hwntype(hf);
  if (hw == NULL) hw = get_hwntype(-1);

  temp[0]="Link encapsulation"; temp[1]=hw->title;
  gtk_clist_append(GTK_CLIST(iflist), temp);
  
  /* Don't print the hardware address for ATM if it's null. */
  /* Don't print the hardware address for ATM or Ash if it's null. */
  if (hw->sprint != NULL && ((strncmp(ptr->name, "atm", 3) &&
			      strncmp(ptr->name, "ash", 3)) || 
   (ptr->hwaddr[0] || ptr->hwaddr[1] || ptr->hwaddr[2] || ptr->hwaddr[3] || 
   ptr->hwaddr[4] || ptr->hwaddr[5]))) {
    temp[0] = "Hardware address"; temp[1] = hw->print(ptr->hwaddr);
    gtk_clist_append(GTK_CLIST(iflist), temp);
  }
#ifdef IFF_PORTSEL
  if (ptr->flags & IFF_PORTSEL) {
    temp[0] = "Media";
    snprintf(buffer, 1024, "%s%s", if_port_text[ptr->map.port][0],
	   (ptr->flags & IFF_AUTOMEDIA)?"(auto)":"");
    temp[1] = (char *)buffer;
    gtk_clist_append(GTK_CLIST(iflist), temp);
  }
#endif
#if HAVE_AFINET6  
  if (ap->af != AF_INET6) {
#endif
    snprintf(buffer, 1024, "%s address", ap->name);
    temp[0]=buffer; temp[1] = ap->sprint(&ptr->addr, 1);
    buffer[0] = toupper(buffer[0]); /* Capitalise the first letter */
    gtk_clist_append(GTK_CLIST(iflist), temp);
    
    if (ptr->flags & IFF_POINTOPOINT) {
      temp[0] = "Point-to-Point"; temp[1] = ap->sprint(&ptr->dstaddr, 1);
      gtk_clist_append(GTK_CLIST(iflist), temp);
    } else {
      temp[0] = "Broadcast"; temp[1] = ap->sprint(&ptr->broadaddr, 1);
      gtk_clist_append(GTK_CLIST(iflist), temp);
    }
    temp[0] = "Netmask"; temp[1] = ap->sprint(&ptr->netmask, 1);
    gtk_clist_append(GTK_CLIST(iflist), temp);
#if HAVE_AFINET6
  }
  /* FIXME: should be integrated into interface.c.   */ 

  if ((f = fopen(_PATH_PROCNET_IFINET6, "r")) != NULL) {
    while(fscanf(f, "%4s%4s%4s%4s%4s%4s%4s%4s %02x %02x %02x %02x %s\n",
		 addr6p[0], addr6p[1], addr6p[2], addr6p[3],
		 addr6p[4], addr6p[5], addr6p[6], addr6p[7],
		 &if_idx, &plen, &scope, &dad_status, devname) != EOF) {
      if (!strcmp(devname, ife->name)) {
	sprintf(addr6, "%s:%s:%s:%s:%s:%s:%s:%s",
		addr6p[0], addr6p[1], addr6p[2], addr6p[3],
		addr6p[4], addr6p[5], addr6p[6], addr6p[7]);
	inet6_aftype.input(1, addr6, (struct sockaddr *)&sap);
        
        temp[0]="Inet6 address";
        snprintf(buffer, 1024, "%s/%d", inet6_aftype.sprint((struct sockaddr *)&sap, 1), plen);
        temp[1]=buffer;
        gtk_clist_append(GTK_CLIST(iflist), temp);

        temp[0]="Scope";
	switch (scope) {
	case 0: temp[1]="Global"; break;
	case IPV6_ADDR_LINKLOCAL: temp[1]="Link"; break;
	case IPV6_ADDR_SITELOCAL: temp[1]="Site"; break;
	case IPV6_ADDR_COMPATv4: temp[1]="Compat"; break;
	case IPV6_ADDR_LOOPBACK: temp[1]="Host"; break;
	default: temp[1]="Unknown";
	}
        gtk_clist_append(GTK_CLIST(iflist), temp);
      }
    }
    fclose(f);
  }
#endif

#ifdef HAVE_AFIPX
  if (ipxtype==NULL)
    ipxtype=get_afntype(AF_IPX);

  if (ipxtype!=NULL) {
    if(ptr->has_ipx_bb) {
      temp[0] = "IPX/Ethernet II address"; temp[1] = ipxtype->sprint(&ptr->ipxaddr_bb,1);
      gtk_clist_append(GTK_CLIST(iflist), temp);
    }
    if(ptr->has_ipx_sn) {
      temp[0] = "IPX/Ethernet SNAP address"; temp[1] = ipxtype->sprint(&ptr->ipxaddr_sn,1);
      gtk_clist_append(GTK_CLIST(iflist), temp);
    }
    if(ptr->has_ipx_e2) {
      temp[0] = "IPX/Ethernet 802.2 address"; temp[1] = ipxtype->sprint(&ptr->ipxaddr_e2,1);
      gtk_clist_append(GTK_CLIST(iflist), temp);
    }
    if(ptr->has_ipx_e3) {
      temp[0] = "IPX/Ethernet 802.3 address"; temp[1] = ipxtype->sprint(&ptr->ipxaddr_e3,1);
      gtk_clist_append(GTK_CLIST(iflist), temp);
    }
  }
#endif

#ifdef HAVE_AFATALK
  if (ddptype==NULL)
    ddptype=get_afntype(AF_APPLETALK);
  if (ddptype!=NULL) {
    if (ptr->has_ddp) {
      temp[0] = "EtherTalk Phase 2 address"; temp[1] = ddptype->sprint(&ptr->ddpaddr,1);
      gtk_clist_append(GTK_CLIST(iflist), temp);
    }
  }
#endif

#ifdef HAVE_AFECONET
  if (ectype == NULL)
    ectype = get_afntype(AF_ECONET);
  if (ectype != NULL) {
    if (ptr->has_econet) {
      temp[0]=_("Econet address"); temp[1] = ectype->sprint(&ptr->ecaddr,1);
      gtk_clist_append(GTK_CLIST(iflist), temp);
    }
  }
#endif

  temp[0]="Flags"; temp[1]=buffer;

  buffer[0]=0; // Empty the buffer;
  
  if (ptr->flags == 0) strncat(buffer, "[NO FLAGS] ", 1024);
  if (ptr->flags & IFF_UP) strncat(buffer, "UP ", 1024);
  if (ptr->flags & IFF_BROADCAST) strncat(buffer, "BROADCAST ", 1024);
  if (ptr->flags & IFF_DEBUG) strncat(buffer, "DEBUG ", 1024);
  if (ptr->flags & IFF_LOOPBACK) strncat(buffer, "LOOPBACK ", 1024);
  if (ptr->flags & IFF_POINTOPOINT) strncat(buffer, "POINTOPOINT ", 1024);
  if (ptr->flags & IFF_NOTRAILERS) strncat(buffer, "NOTRAILERS ", 1024);
  if (ptr->flags & IFF_RUNNING) strncat(buffer, "RUNNING ", 1024);
  if (ptr->flags & IFF_NOARP) strncat(buffer, "NOARP ", 1024);
  if (ptr->flags & IFF_PROMISC) strncat(buffer, "PROMISC ", 1024);
  if (ptr->flags & IFF_ALLMULTI) strncat(buffer, "ALLMULTI ", 1024);
  if (ptr->flags & IFF_SLAVE) strncat(buffer, "SLAVE ", 1024);
  if (ptr->flags & IFF_MASTER) strncat(buffer, "MASTER ", 1024);
  if (ptr->flags & IFF_MULTICAST) strncat(buffer, "MULTICAST ", 1024);
  #ifdef HAVE_DYNAMIC
  if (ptr->flags & IFF_DYNAMIC) strncat(buffer,_("DYNAMIC "), 1024);
  #endif
  gtk_clist_append(GTK_CLIST(iflist), temp);

  temp[0]="MTU"; temp[1]=buffer;
  snprintf(buffer, 1024, "%d", ptr->mtu);
  gtk_clist_append(GTK_CLIST(iflist), temp);

  temp[0]="Metric"; temp[1]=buffer;
  snprintf(buffer, 1024, "%d", ptr->metric?ptr->metric:1);
  gtk_clist_append(GTK_CLIST(iflist), temp);

#ifdef SIOCSKEEPALIVE
  if (ptr->outfill || ptr->keepalive) {
    temp[0]=_("Outfill"); temp[1]=buffer;
    snprintf(buffer, 1024, "%d", ptr->outfill);
    gtk_clist_append(GTK_CLIST(iflist), temp);

    temp[0]=_("Keepalive"); temp[1]=buffer;
    snprintf(buffer, 1024, "%d", ptr->keepalive);
    gtk_clist_append(GTK_CLIST(iflist), temp);
  }
#endif

  if (ptr->statistics_valid) { 
	  /* XXX: statistics are currently only printed for the original address,
	   *	  not for the aliases, although strictly speaking they're shared
	   *	  by all addresses.
	   */

  /* If needed, display the interface statistics. */
  temp[0]="RX packets"; // Temp[1] already = buffer.
  snprintf(buffer, 1024, "%lu", ptr->stats.rx_packets);
  gtk_clist_append(GTK_CLIST(iflist), temp);

  temp[0]="RX errors"; // Temp[1] already = buffer.
  snprintf(buffer, 1024, "%lu", ptr->stats.rx_errors);
  gtk_clist_append(GTK_CLIST(iflist), temp);

  temp[0]="RX dropped"; // Temp[1] already = buffer.
  snprintf(buffer, 1024, "%lu", ptr->stats.rx_dropped);
  gtk_clist_append(GTK_CLIST(iflist), temp);

  temp[0] = "RX overruns";
  snprintf(buffer, 1024, "%lu", ptr->stats.rx_fifo_errors);
  gtk_clist_append(GTK_CLIST(iflist), temp);

  temp[0] = "RX frame";
  snprintf(buffer, 1024, "%lu", ptr->stats.rx_frame_errors);
  gtk_clist_append(GTK_CLIST(iflist), temp);

  if (can_compress) {
    temp[0] = "RX compressed";
    snprintf(buffer, 1024, "%lu", ptr->stats.rx_compressed);
    gtk_clist_append(GTK_CLIST(iflist), temp);
  }



  temp[0]="TX packets"; // Temp[1] already = buffer.
  snprintf(buffer, 1024, "%lu", ptr->stats.tx_packets);
  gtk_clist_append(GTK_CLIST(iflist), temp);

  temp[0]="TX errors";
  snprintf(buffer, 1024, "%lu", ptr->stats.tx_errors);
  gtk_clist_append(GTK_CLIST(iflist), temp);

  temp[0]="TX dropped";
  snprintf(buffer, 1024, "%lu", ptr->stats.tx_dropped);
  gtk_clist_append(GTK_CLIST(iflist), temp);

  temp[0] = "TX overruns";
  snprintf(buffer, 1024, "%lu", ptr->stats.tx_fifo_errors);
  gtk_clist_append(GTK_CLIST(iflist), temp);

  temp[0] = "TX carrier";
  snprintf(buffer, 1024, "%lu", ptr->stats.tx_carrier_errors);
  gtk_clist_append(GTK_CLIST(iflist), temp);

  temp[0] = "TX collisions";
  snprintf(buffer, 1024, "%lu", ptr->stats.collisions);
  gtk_clist_append(GTK_CLIST(iflist), temp);

  if (can_compress) {
    temp[0] = "TX compressed";
    snprintf(buffer, 1024, "%lu", ptr->stats.tx_compressed);
    gtk_clist_append(GTK_CLIST(iflist), temp);
  }
  if (ptr->tx_queue_len != -1) {
    temp[0] = "TX queue length";
    snprintf(buffer, 1024, "%lu", ptr->tx_queue_len);
    gtk_clist_append(GTK_CLIST(iflist), temp);
  }
  }


  if ((ptr->map.irq || ptr->map.mem_start || ptr->map.dma ||
		ptr->map.base_addr)) {
    if (ptr->map.irq) {
      temp[0] = "Interrupt"; temp[1] = buffer;
      snprintf(buffer, 1024, "%d", ptr->map.irq);
      gtk_clist_append(GTK_CLIST(iflist), temp);
    }
    if (ptr->map.base_addr>=0x100) {    /* Only print devices using it for
					  I/O maps */
      temp[0] = "Base address"; temp[1] = buffer;
      snprintf(buffer, 1024, "0x%x", ptr->map.base_addr);
      gtk_clist_append(GTK_CLIST(iflist), temp);
    }
    if (ptr->map.mem_start) {
      temp[0] = "Memory"; temp[1] = buffer;
      snprintf(buffer, 1024, "%lx-%lx", ptr->map.mem_start,ptr->map.mem_end);
      gtk_clist_append(GTK_CLIST(iflist), temp);
    }
    if (ptr->map.dma) {
      temp[0] = "DMA channel"; temp[1] = buffer;
      snprintf(buffer, 1024, "%x", ptr->map.dma);
      gtk_clist_append(GTK_CLIST(iflist), temp);
    }
  }

 gtk_clist_thaw(GTK_CLIST(iflist));

}

static gint cb_update_packets(gpointer data)
{
  GtkWidget *packetlist;
  struct packet_t *pkt;
  int i=0;
  gint b;
  char *temp[4], buffer[64];

  packetlist = main_wnd.packetlist;
  gtk_clist_freeze(GTK_CLIST(packetlist));
  if (data) gtk_clist_clear(GTK_CLIST(packetlist));

  if ((main_wnd.current_if!=NULL) && (main_wnd.current_if->sniff.packets!=NULL)) {
    pt_mutex_enter(main_wnd.current_if->sniff.mutex);
    pkt = main_wnd.current_if->sniff.packets;
    
    while (pkt!=NULL) {
     i++;
     if (i > GTK_CLIST(packetlist)->rows) {
       packet_decode(pkt, 0, NULL, NULL);
       snprintf(buffer,64,"%d",pkt->len);
       temp[0] = buffer; temp[1] = pkt->description;
       temp[2] = pkt->dstaddrstr; temp[3]=pkt->srcaddrstr;
       b = gtk_clist_append(GTK_CLIST(packetlist),temp);
       gtk_clist_set_row_data(GTK_CLIST(packetlist), b, (gpointer) pkt);
     }
     pkt = pkt->next;
    }
    pt_mutex_exit(main_wnd.current_if->sniff.mutex);
  }

  gtk_clist_thaw(GTK_CLIST(packetlist));
  return TRUE;
}

/* Note that this is called every time the user clicks on an interface tree item,
   whether it is already selected or not. */
void cb_select_child (GtkWidget *root_tree, GtkWidget *child,
                             GtkWidget *subtree)
{
  interface *ife;

  ife = (interface *) gtk_object_get_user_data(GTK_OBJECT(child));
  main_wnd.current_if = ife;
  gtk_frame_set_label(GTK_FRAME(main_wnd.frame1), ife->name);
  gtk_frame_set_label(GTK_FRAME(main_wnd.frame2), ife->name);
  gtk_frame_set_label(GTK_FRAME(main_wnd.frame3), ife->name);

  update_if_list(ife, main_wnd.iflist);

  /* Depending on whether the interface attached to this tree node is active
     or not, add/remove a data display refresh timer, and enable/disable the
     start/stop capturing buttons. */
  if (main_wnd.current_if->sniff.active) {
    mainwnd_add_refresh_timeout();
    gtk_widget_set_sensitive(main_wnd.startcapturebtn, FALSE);
    gtk_widget_set_sensitive(main_wnd.stopcapturebtn, TRUE);
  } else {
    mainwnd_remove_refresh_timeout();
    gtk_widget_set_sensitive(main_wnd.startcapturebtn, TRUE);
    gtk_widget_set_sensitive(main_wnd.stopcapturebtn, FALSE);
  }

  cb_update_packets((void *)1);
  GTK_HEX(main_wnd.phex)->buffer = NULL;
  GTK_HEX(main_wnd.phex)->buffer_size = 0;
  gtk_hex_data_changed(GTK_HEX(main_wnd.phex), 0, 0);
  gtk_widget_set_usize(main_wnd.phex,main_wnd.phex->allocation.width,main_wnd.phex->allocation.height);
  gtk_clist_clear(GTK_CLIST(main_wnd.pctree));
}

void mainwnd_add_refresh_timeout(void)
{
 if (main_wnd.timeout==-1) {
  main_wnd.timeout = gtk_timeout_add(1000, cb_update_packets, (void *)0);
 }
}

void mainwnd_remove_refresh_timeout(void)
{
 if (main_wnd.timeout!=-1) {
   gtk_timeout_remove(main_wnd.timeout);
   main_wnd.timeout = -1;
 }
}

int cb_addtolist(char *name,char *description, u_int pos, u_int len, char *value, GtkWidget *data);

static int num_titles;
static char nodetitles[20][1024];

void cb_select_packet (GtkWidget *list,gint row, gint column,
            GdkEventButton *event, gpointer data)
{
// GdkRectangle rect;
 struct packet_t *pkt = (struct packet_t *) gtk_clist_get_row_data(GTK_CLIST(list), row);;
 g_assert(pkt!=NULL);

 gtk_clist_freeze(GTK_CLIST(main_wnd.pctree));
 gtk_clist_clear(GTK_CLIST(main_wnd.pctree));
 num_titles = 0;
 packet_decode(pkt, 0, (decoded_field_func *) &cb_addtolist, main_wnd.pctree);
 gtk_clist_thaw(GTK_CLIST(main_wnd.pctree));
 GTK_HEX(main_wnd.phex)->buffer = pkt->data;
 GTK_HEX(main_wnd.phex)->buffer_size = pkt->len;
 gtk_hex_data_changed(GTK_HEX(main_wnd.phex), 0, pkt->len);
 gtk_widget_set_usize(main_wnd.phex,main_wnd.phex->allocation.width,main_wnd.phex->allocation.height);
/* rect.x = rect.y = 0;
 rect.width = main_wnd.phex->allocation.width;
 rect.height = main_wnd.phex->allocation.height;
 gtk_widget_draw(main_wnd.phex, &rect);*/

 // If someone double left clicked on a packet, open a packet viewer window.
/* if (event && event->button == 1 && event->type == GDK_2BUTTON_PRESS)
 {
  view_packet(pkt);
 }*/

 
}

int cb_addtolist(char *name,char *description, u_int pos, u_int len, char *value, GtkWidget *data)
{
  char *titles[3] = {NULL, NULL, NULL};
  static GtkCTreeNode *currentparent=NULL;

  titles[0] = name; titles[1]=value;
  if (len==0) { // If this is to be a new protocol
    strncpy(nodetitles[num_titles], value, 1024);
    currentparent = gtk_ctree_insert_node(GTK_CTREE(data), NULL, NULL, titles,
                      5, img_hlprotocol, mask_hlprotocol, img_hlprotocol, mask_hlprotocol,
                      FALSE, FALSE);
    gtk_ctree_node_set_row_data(GTK_CTREE(data), currentparent, (void *)num_titles++);
  } else {
    GtkCTreeNode *node;
    node = gtk_ctree_insert_node(GTK_CTREE(data), currentparent, NULL, titles, 5, NULL,
                     NULL, NULL, NULL, TRUE, FALSE);
    gtk_ctree_node_set_row_data(GTK_CTREE(data), node, (void *)pos);
  }
  return 0;
}

void cb_select_field (GtkCTree *tree, GtkCTreeNode *row, gint column, GtkWidget *hex)
{
  gboolean isleaf;
  u_int l;

  gtk_ctree_get_node_info(tree, row, NULL, NULL, NULL, NULL, NULL, NULL, &isleaf, NULL);
  if (isleaf) {
   l = (u_int) gtk_ctree_node_get_row_data(GTK_CTREE(tree), row);
   gtk_hex_set_cursor(GTK_HEX(hex), (l / 8));
  }
}

void mw_pkt_expand(GtkCTree *tree, GtkCTreeNode *node, gpointer data)
{
 gtk_ctree_node_set_text(tree, node, 1, "");
}

void mw_pkt_collapse(GtkCTree *tree, GtkCTreeNode *node, gpointer data)
{
 gtk_ctree_node_set_text(tree, node, 1, nodetitles[(int)gtk_ctree_node_get_row_data(GTK_CTREE(tree),node)]);
}
