/*
Gnusniff - Network packet sniffer
Copyright (C) 1998 Peter Hawkins

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <netinet/in.h>
#include <arpa/nameser.h>
#include <arpa/inet.h>
#include <glib.h>
#include "interface.h"
#include "protocol.h"

struct tok {
 int v;
 char *s;
};

/* Compatibility */
#ifndef T_TXT
#define T_TXT		16		/* text strings */
#endif
#ifndef T_RP
#define T_RP		17		/* responsible person */
#endif
#ifndef T_AFSDB
#define T_AFSDB		18		/* AFS cell database */
#endif
#ifndef T_X25
#define T_X25		19		/* X_25 calling address */
#endif
#ifndef T_ISDN
#define T_ISDN		20		/* ISDN calling address */
#endif
#ifndef T_RT
#define T_RT		21		/* router */
#endif
#ifndef T_NSAP
#define T_NSAP		22		/* NSAP address */
#endif
#ifndef T_NSAP_PTR
#define T_NSAP_PTR	23		/* reverse NSAP lookup (deprecated) */
#endif
#ifndef T_SIG
#define T_SIG		24		/* security signature */
#endif
#ifndef T_KEY
#define T_KEY		25		/* security key */
#endif
#ifndef T_PX
#define T_PX		26		/* X.400 mail mapping */
#endif
#ifndef T_GPOS
#define T_GPOS		27		/* geographical position (withdrawn) */
#endif
#ifndef T_AAAA
#define T_AAAA		28		/* IP6 Address */
#endif
#ifndef T_LOC
#define T_LOC		29		/* Location Information */
#endif

#ifndef T_UNSPEC
#define T_UNSPEC	103		/* Unspecified format (binary data) */
#endif
#ifndef T_UNSPECA
#define T_UNSPECA	104		/* "unspecified ascii". Ugly MIT hack */
#endif

#ifndef C_CHAOS
#define C_CHAOS		3		/* for chaos net (MIT) */
#endif
#ifndef C_HS
#define C_HS		4		/* for Hesiod name server (MIT) (XXX) */
#endif

static char dns_type[] = "DNS";

static char *ns_ops[] = {
	"Standard query", "Inverse query", "Status request", "Unknown opcode - 3",
        "Notify", "Dynamic update request", "Unknown opcode - 6", "Unknown opcode - 7",
	"Unknown opcode - 8", "UpdateA", "UpdateD", "UpdateDA",
	"UpdateM", "UpdateMA", "Zone Init", "Zone Ref",
};

static char *ns_resp[] = {
	"(No error)", "Format error - The name server was unable to interpret the query.",
        "Server failure - The name server was unable to process this query due to a problem with the name server.",
        "Non-existant domain - The domain name referenced in the query does not exist.",
	"Not implemented - The name server does not support the requested type of query.",
        "Refused - The name server refuses to perform the specified operation.",
        "Unknown response - 6", "Unknown response - 7",
	"Unknown response - 8", "Unknown response - 9", "Unknown response - 10", "Unknown response - 11",
	"Unknown response - 12", "Unknown response - 13", "Unknown response - 14", "No change",
};

static char *ns_short_resp[] = {
	"", " FormErr", " ServFail", " NXDomain",
	" NotImp", " Refused", " Resp6", " Resp7",
	" Resp8", " Resp9", " Resp10", " Resp11",
	" Resp12", " Resp13", " Resp14", " NoChange",
};

static int dns_nprint(struct packet_t *pkt, int offset, const u_char *input, u_char *output, int outputlen)
{
 u_char i;
 int outputpos=0;
 const u_char *originalinput = input;
 int flag=0;

 i = *input++;
 while (i>0) {
   if ((i & INDIR_MASK) == INDIR_MASK) {
     input = pkt->data + offset + (((i & ~INDIR_MASK) << 8) | *input);
     flag = 1;
     i = *input++;
   }

   if ((outputpos + i) > outputlen) {
    memcpy(output + outputpos, input, outputlen - outputpos);
    output[outputlen] = 0;
    return outputlen;
   }
   memcpy(output + outputpos, input, i);
   outputpos += i; input += i;

   i = *input++;
   if (i) output[outputpos++] = '.';
 }
 output[outputpos++] = 0;
 return flag?2:input - originalinput;
}

static struct tok type2str[] = {
	{ T_A,		"Host address" },
	{ T_NS,		"Authoritative name server" },
	{ T_MD,		"Mail destination (obsolete)" },
	{ T_MF,		"Mail forwarder (obsolete)" },
	{ T_CNAME,	"Canonical name for an alias" },
	{ T_SOA,	"Start of a zone of authority" },
	{ T_MB,		"Mail box domain name" },
	{ T_MG,		"Mail group member" },
	{ T_MR,		"Mail rename domain name" },
	{ T_NULL,	"NULL" },
	{ T_WKS,	"Well known service description" },
	{ T_PTR,	"Domain name pointer" },
	{ T_HINFO,	"Host information" },
	{ T_MINFO,	"Mailbox or mail list information" },
	{ T_MX,		"Mail exchange" },
	{ T_TXT,	"Text strings" },
	{ T_RP,		"Responsible person" },
	{ T_AFSDB,	"AFS cell database" },
	{ T_X25,	"X.25 calling address" },
	{ T_ISDN,	"ISDN calling address" },
	{ T_RT,		"Router" },
	{ T_NSAP,	"NSAP address" },
	{ T_NSAP_PTR,	"Reverse NSAP lookup (deprecated)" },
	{ T_SIG,	"Security signature" },
	{ T_KEY,	"Security key" },
	{ T_PX,		"X.400 mail mapping" },
	{ T_GPOS,	"Geographical position (withdrawn)" },
	{ T_AAAA,	"IPv6 address" },
	{ T_LOC ,	"Location information" },
#ifndef T_UINFO
#define T_UINFO 100
#endif
	{ T_UINFO,	"User (finger) information" },
#ifndef T_UID
#define T_UID 101
#endif
	{ T_UID,	"User ID" },
#ifndef T_GID
#define T_GID 102
#endif
	{ T_GID,	"Group ID" },
	{ T_UNSPEC,	"Unspecified format (binary data)" },
	{ T_UNSPECA,	"UNSPECA" },
        { T_IXFR,       "Incremental zone transfer" },
	{ T_AXFR,	"Transfer zone of authority" },
	{ T_MAILB,	"Transfer mailbox records" },
	{ T_MAILA,	"Transfer mail agent records" },
	{ T_ANY,	"Any" },
	{ 0,		NULL }
};

static struct tok class2str[] = {
	{ C_IN,		"Internet" },		/* Not used (DPH: that's tcpdump crap, it's used all the time)*/
	{ C_CHAOS,	"CHAOS net (MIT)" },
	{ C_HS,		"Hesoid name server (MIT)" },
	{ C_ANY,	"Any" },
	{ 0,		NULL }
};

/* Print a query field of a DNS query */
static int dns_qprint(struct packet_t *pkt, const u_char *input, int questionnum, int offset, decoded_field_func *ff, gpointer data)
{
 char buffer[1024], buffer2[256], *tok;
 int i,j, pos;

 
 // From tcpdump and RFC 1035...
 i = dns_nprint(pkt, offset, input, buffer, 1024);
 
 if (ff) {
   pos = ((int)input - (int)(pkt->data)) * 8;
 
   snprintf(buffer2, 256, "Q%d - Hostname", questionnum);
   ff(buffer2, NULL, pos, i, buffer, data);

   tok = NULL;
   for (j=0; type2str[j].s!=NULL; j++) {
    if (type2str[j].v == ntohs(*((u_short *)(input + i)))) {
      tok = type2str[j].s;
      break;
    }
   }
   if (tok == NULL) tok = "UNKNOWN";

   pos+= i*8;
   snprintf(buffer2, 256, "Q%d - Type", questionnum);
   ff(buffer2, NULL, pos, 16, tok, data);
   i+=2;

   tok = NULL;
   for (j=0; class2str[j].s!=NULL; j++) {
    if (class2str[j].v == ntohs(*((u_short *)(input + i)))) {
      tok = class2str[j].s;
      break;
    }
   }
   if (tok == NULL) tok = "UNKNOWN";

   pos+=16;
   snprintf(buffer2, 256, "Q%d - Class", questionnum);
   ff(buffer2, NULL, pos, 16, tok, data);
   i+=2;
 }
 return i;
}

/* Print a Resource Record field of a DNS query. */
static int dns_rprint(struct packet_t *pkt, const u_char *input, int questionnum, int offset, decoded_field_func *ff, gpointer data)
{
 char buffer[1024], buffer2[256], *tok;
 int i,j, pos, len, typ;
 
 // From tcpdump and RFC 1035...
 i = dns_nprint(pkt, offset, input, buffer, 1024);

 if (ff) {
   pos = ((int)input - (int)pkt->data) * 8;

   snprintf(buffer2, 256, "R%d - Hostname", questionnum);
   ff(buffer2, NULL, pos, i, buffer, data);
   pos += i*8;


   typ = ntohs(*((u_short *)(input + i)));
   tok = NULL;
   for (j=0; type2str[j].s!=NULL; j++) {
    if (type2str[j].v == typ) {
      tok = type2str[j].s;
      break;
    }
   }
   if (tok == NULL) tok = "UNKNOWN";

   snprintf(buffer2, 256, "R%d - Type", questionnum);
   ff(buffer2, NULL, pos, 16, tok, data);
   i+=2; pos+=16;

   tok = NULL;
   for (j=0; class2str[j].s!=NULL; j++) {
    if (class2str[j].v == ntohs(*((u_short *)(input + i)))) {
      tok = class2str[j].s;
      break;
    }
   }
   if (tok == NULL) tok = "UNKNOWN";

   snprintf(buffer2, 256, "R%d - Class", questionnum);
   ff(buffer2, NULL, pos, 16, tok, data);
   i+=2; pos+=16;

   snprintf(buffer, 1024, "%d seconds", ntohl(*((u_long *)(input + i))));
   snprintf(buffer2, 256, "R%d - TTL", questionnum);
   ff(buffer2, NULL, pos, 32, buffer, data);
   i+=4; pos+=32;

   len = ntohs(*((u_short *)(input + i)));
   snprintf(buffer, 1024, "%d bytes", len);
   snprintf(buffer2, 256, "R%d - RDATA length", questionnum);
   ff(buffer2, NULL, pos, 16, buffer, data);
   i+=2; pos+=16;

   buffer[0] = 0;
   switch (typ) {
    case T_A:
      strncpy(buffer, inet_ntoa(*((struct in_addr *)(input + i))), 1024); break;
    case T_NS: case T_CNAME: case T_PTR:
      dns_nprint(pkt, offset, input + i, buffer, 1024); break;
    case T_MX:
      pos+=16;
      dns_nprint(pkt, offset, input + i + 2, buffer, 1024); break;
    case T_TXT:
      {
       int x;
       x = *(input + i);
       snprintf(buffer, 1024, "%.*s", x, input + i + 1);
      }
      break;
    case T_UNSPECA:
      snprintf(buffer, 1024, "%.*s", len, input + i); break;
   }

   if (strlen(buffer) > 0) {
    snprintf(buffer2, 256, "R%d - RDATA", questionnum);
    ff(buffer2, NULL, pos, len*8, buffer, data);
   }
   i+=len;
 } else { // We're just skipping, not displaying...
  i += 10 + ntohs(*((u_short *)(input + i + 8)));
 }
 return i;
}

int dns_decode(struct packet_t *pkt, u_int offset, decoded_field_func *ff, gpointer data)
{
 	register const HEADER *np;
	register int qdcount, ancount, nscount, arcount;
	register const u_char *cp;
        char buffer[1024];
        int i, j;

	np = (const HEADER *)(pkt->data + offset);
	/* get the byte-order right */
	qdcount = ntohs(np->qdcount);
	ancount = ntohs(np->ancount);
	nscount = ntohs(np->nscount);
	arcount = ntohs(np->arcount);

        strncpy(pkt->description, dns_type,PACKET_DESCRIPTION_LENGTH);

        if (ff) {
          snprintf(buffer, 1024, "%s ID=0x%.4X%s", np->qr?"Response":"Query", ntohs(np->id), np->qr?ns_short_resp[np->rcode]:"");
          ff("DNS", "Domain name protocol", offset*8, 0, buffer, data);
          snprintf(buffer, 1024, "0x%.4X", ntohs(np->id));
          ff("ID", NULL, offset*8, 16, buffer, data);
          ff("Type (QR)", NULL, 16 + offset*8, 1, np->qr?"Response":"Query", data);
          ff("Opcode", NULL, 17 + offset*8, 4, ns_ops[np->opcode], data);
        }
        j=0;
        
	if (np->qr) {
		/* this is a response */

                if (ff) {
                  ff("Response", NULL, 28 + offset*8, 4, ns_resp[np->rcode], data);
                  ff("Answer", NULL, 21 + offset*8, 1, np->aa?"Authoritative":"Unauthoritative", data);
                  if (np->tc) ff("Truncated", NULL, 22+ offset*8, 1,
                     "The message was truncated as it was longer than was permitted on this data channel.", data);
                  ff("Recursion", NULL, 24 + offset*8, 1, np->ra?"Available":"Unavailable", data);
                  
                  if (qdcount!=1) {                  
                    snprintf(buffer, 1024, "%d", qdcount);
                    ff("No. of questions", NULL, 32 + offset*8, 16, buffer, NULL);
                  }
                  for (i=1; i<=qdcount; i++)
                    j += dns_qprint(pkt, ((const u_char *)(np + 1)) + j, i, offset, ff, data);

                  for (i=1; i<=ancount; i++)
                    j += dns_rprint(pkt, ((const u_char *)(np + 1)) + j, i, offset, ff, data);
                }
	}
	else {
		/* this is a request */
                if (ff) {

// DPH: Can anyone tell me WTF this is supposed to do? This comes from tcpdump.
		/* any weirdness? */
//		if (*(((u_short *)np)+1) & htons(0x6ff))
//			printf(" [b2&3=0x%x]", ntohs(*(((u_short *)np)+1)));

                  if (np->opcode == IQUERY) {

                    if (qdcount) {
                      snprintf(buffer, 1024, "%d", qdcount);
                      ff("No. of questions", NULL, 32 + offset*8, 16, buffer, NULL);
                    }
                    if (ancount != 1) {
                      snprintf(buffer, 1024, "%d", ancount);
                      ff("No. of answers", NULL, 48 + offset*8, 16, buffer, NULL);
                    }
                    
                  } else {
                    if (ancount) {
                      snprintf(buffer, 1024, "%d", ancount);
                      ff("No. of answers", NULL, 48 + offset*8, 16, buffer, NULL);
                    }
                    if (qdcount != 1) {
                      snprintf(buffer, 1024, "%d", qdcount);
                      ff("No. of questions", NULL, 32 + offset*8, 16, buffer, NULL);
                    }
                  }
                  if (nscount) {
                      snprintf(buffer, 1024, "%d", nscount);
                      ff("No. of authority entries", NULL, 64 + offset*8, 16, buffer, NULL);
                  }
                  if (arcount) {
                      snprintf(buffer, 1024, "%d", arcount);
                      ff("No. of resource records", NULL, 80 + offset*8, 16, buffer, NULL);
                  }
                  j=0;
                  for (i=1; i<=qdcount; i++)
                    j += dns_qprint(pkt, ((const u_char *)(np + 1)) + j, i, offset, ff, data);
                }
	}
}

