/*
Gnomesniff - Network packet sniffer
Copyright (C) 1998 Peter Hawkins

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/
/*
p_ether.c
Contains the implementation for the Ethernet 802.2, 802.3, Ethernet II and
Ethernet Snap.
*/

#include <stdio.h>
#include <string.h>
#include <net/if_arp.h>
#include <netinet/in.h>
#include "protocol.h"
#include "interface.h"
#include "net-support.h"
#include "p_ipv4.h"
#include "p_ipx.h"

/* Pinched from karpski */
/* 
The definition of an ethernet 802.3/802.2 frame
Can also be used for Ether-II, Ethernet SNAP and Ethernet-raw packets
*/
struct frame		/* Structure of an 802.3/802.2 frame */
{
   u_char daddr[6];	/* Dest mac */
   u_char saddr[6];	/* Src mac */
   u_short type;	/* Ethernet-II type (rfc 1700) */
   u_char dsap;		/* Dest. SAP */
   u_char ssap;		/* Src SAP */
   u_char cntrl;	/* Control number */
   u_char vendorid[3];	/* Vendor ID - usually same as src mac's vendor */
   u_short type2;	/* 802.3/802.2 type location */
};

protocol_t ether8022, ether8023, etherii, ethersnap;

char eiidescription[] = "Ethernet II frame",
     esnapdescription[] = "Ethernet SNAP frame";


/* Print an ethernet address we splurged from somewhere. Code nicked from
   karpski. */
static char *print_addr(char s[], int places, char tick)
{
   int i;
   int j;
   int adv;
   unsigned char rval;
   static char scratchbuf[255];

   j= 0;
   scratchbuf[j] = '\0';
   for (i=0 ; i<places; i++)
   {
       rval = s[i];
       if ((places == 6) || (places == 3))
       {
          sprintf(&scratchbuf[j], "%02X%n", rval, &adv);
          j+=adv;
       }
       else
       {
          sprintf(&scratchbuf[j], "%hd%n", rval, &adv);
          j+=adv;
       }
       if (i!=(places-1))
          scratchbuf[j++] = tick;
   }
   scratchbuf[j] = '\0';
   return(scratchbuf);
}



/* Decode an ethernet packet. */
int ether_decode(struct packet_t *pkt, u_int offset, decoded_field_func *ff, gpointer data)
{
 struct frame *frm = (struct frame *) (pkt->data + offset);
 ushort frmtype = htons(frm->type);
 char *temp;

 g_assert(offset<pkt->len);

 temp = print_addr(frm->daddr, 6, ':');
 pkt->dest = gethostforhwaddr(pkt, temp, strlen(temp));
 strncpy(pkt->dstaddrstr, temp, ADDRESS_STRING_LENGTH);
 
 temp = print_addr(frm->saddr, 6, ':');
 pkt->src = gethostforhwaddr(pkt, temp, strlen(temp));
 strncpy(pkt->srcaddrstr, temp, ADDRESS_STRING_LENGTH);
 
   if (frmtype < 1501)
   {
      
/* Raw ether / Raw 802.3 */
      if ((frm->dsap == 0xff) && (frm->ssap == 0xff))
      {
         if (ff) ff("Raw ethernet frame", NULL, offset*8, 0, NULL, data);
         strncpy(pkt->description, "Raw IPX packet", PACKET_DESCRIPTION_LENGTH);
         pkt->valid = TRUE; // Who knows?
         ipx_decode(pkt, offset + 16, ff, data);
         return 16;
/*         packet_info->dptr=(char *)framepkt+16;
         packet_info->dlen=frmtype;
         packet_info->frame_enc_type = 1;
         add_protolist(packet_info->frametype, frmtype ,packet_info->packetlen);
         strcpy(packet_info->protostr, packet_info->frametype);
         return;*/
      }
      

/* Ether SNAP */
      if ((frm->dsap == 0xaa) && (frm->ssap == 0xaa) && (frm->cntrl == 0x03))
      {
        if (ff) ff("Ethernet SNAP frame", NULL, offset*8, 0, NULL, data);
        frmtype = htons(frm->type2);
        snprintf(pkt->description,1024,"Ether/SNAP type %4X", frmtype);
        return sizeof(struct frame);
/*         packet_info->dlen=frmtype-8;
         frmtype = htons(frm->type2);
         sprintf(packet_info->frametype, "\nEther/SNAP type %4X\n", frmtype);
         packet_info->frame_enc_type = 3;
         payload = (char *) (framepkt + sizeof(struct frame));*/
      }
      else
      {
        if (ff) ff("Ethernet 802.2/802.3 frame", NULL, offset*8, 0, NULL, data);
/* 802.3/802.2 */
//         packet_info->dlen=frmtype;
//         packet_info->dptr=framepkt + 17;
//         packet_info->frame_enc_type = 2;
         switch (frm->dsap)
         {
            case 0: strcpy(pkt->description, "XID (802.3/802.2)"); break;
            case 2: strcpy(pkt->description, "ISM (802.3/802.2)"); break;
            case 3: strcpy(pkt->description, "GSM (802.3/802.2)"); break;
            case 4 : case  8 : case 0x0c :strcpy(pkt->description, "SNA (802.3/802.2)"); break; 
            case 0x06: strcpy(pkt->description, "IP (802.3/802.2)");
            ipv4_decode(pkt, offset + 17, ff, data); break;
            case 0x0e: strcpy(pkt->description, "(802.3/802.2)"); break;
            case 0x10: strcpy(pkt->description, "Novell/SDLC (802.3/802.2)"); break;
            case 0x20 : case 0x34 : case 0xec : strcpy(pkt->description, "CLNP ISO OSI (802.3/802.2)"); break;
            case 0x42: strcpy(pkt->description, "BPDU (802.3/802.2)"); break;
            case 0x7e: strcpy(pkt->description, "X.25 PLP (802.3/802.2)"); break;
            case 0x80: strcpy(pkt->description, "XNS (802.3/802.2)"); break;
            case 0x86: strcpy(pkt->description, "Nestar (802.3/802.2)"); break;
            case 0x8e: strcpy(pkt->description, "Active station (802.3/802.2)"); break;
            case 0x98: strcpy(pkt->description, "ARP (802.3/802.2)"); break;
            case 0xbc: strcpy(pkt->description, "Vines (802.3/802.2)"); break;
            case 0xe0: strcpy(pkt->description, "IPX (802.3/802.2)");
            ipx_decode(pkt, offset + 17, ff, data); break;
            case 0xF0: strcpy(pkt->description, "IBM Netbios (802.3/802.2)"); break;
            case 0xf4 : case 0xf5: strcpy(pkt->description, "LAN Manager (802.3/802.2)"); break;
            case 0xf8 : strcpy(pkt->description, "Rem. prog load (802.3/802.2)"); break;
            case 0xfa : strcpy(pkt->description, "UB (802.3/802.2)"); break;
            case 0xfc : strcpy(pkt->description, "IBM RPL (802.3/802.2)"); break;
            case 0xfe : strcpy(pkt->description, "ISO Netlayer (802.3/802.2)"); break;
            case 0xff : strcpy(pkt->description, "LLC broadcast (802.3/802.2)"); break;
            default : sprintf(pkt->description, "Unknown DSAP (%2X)", frm->dsap); /*packet_info->outfd = defaultfd;*/ break;
         }
//         add_protolist(packet_info->frametype, frmtype, packet_info->packetlen);
//         strcpy(packet_info->protostr, packet_info->frametype);
         return 17;
      }
   }
   
/* Ethernet II packet (standard ETHER) */
   else
   {
      if (ff) ff("Ethernet II frame", NULL, offset*8, 0, NULL, data);
//      packet_info->frame_enc_type = 4;
//      packet_info->dlen = packetlen - 14;
//      payload = (char *) (framepkt + 14);
      strncpy(pkt->description, "Ethernet II packet", PACKET_DESCRIPTION_LENGTH);
// Can anyone tell me where to find an ethernet II spec? an RFC or something?
      switch (ntohs(frm->type)) {
        case 0x0800: // IPv4
          ipv4_decode(pkt, offset + 14, ff, data); break;
        case 0x86dd: // IPv6
          break;
        case 0x0806: // ARP
          break;
        case 0x8137: // IPX
          ipx_decode(pkt, offset + 14, ff, data);
          break;
      }
      return 14;
   }

/* Here we should only have SNAP or ETHER2 packets */

//   get_packet_info(frmtype, payload, packet_info);
//   add_protolist(packet_info->frametype, frmtype, packet_info->packetlen);

//   if (packet_info->protocol_ptr)
//   {
//      add_protolist(packet_info->protocol_ptr->protocol_short_name, frmtype, packet_info->packetlen);
//      sprintf(packet_info->protostr, "%s/%s",packet_info->protocol_ptr->protocol_short_name, packet_info->frametype);
//   }
//   else
//      strcpy(packet_info->protostr, packet_info->frametype);









//  ff(ether8022.name, ether8022.description, NULL, "", data);


/* hw = get_hwntype(pkt->parent->parent->type);
 if ((hw!=NULL) && (hw->print!=NULL)) {
   temp = hw->print(frm->daddr);
   strncat(pkt->destination.as_string,temp,16);
   temp = hw->print(frm->saddr);
   strncat(pkt->source.as_string,temp,16);
   

 } else {
   snprintf(buffer, 1024, "Invalid address");
   strncat(pkt->destination.as_string,buffer,16);
   strncat(pkt->source.as_string,buffer,16);
 }
 pkt->description = e8022description;*/
 return 0;
}

gboolean ether_isproto(struct packet_t *pkt, u_int offset)
{
//  struct frame *frm = (struct frame *) (pkt->data + offset);

 if (pkt->parent->parent->type == ARPHRD_ETHER) {
   // FIXME: Add more tests here.
   return (ether_decode(pkt, offset, NULL, NULL)>0);
 }
 return FALSE;
}

protocol_t ether8022 =
{
 "Ethernet 802.2",
 "Ethernet 802.2 link encapsulation protocol. FIXME! More description here!",
 ether_isproto,
 ether_decode
};

protocol_t ether8023 =
{
 "Ethernet 802.3",
 "Ethernet 802.3 link encapsulation protocol. FIXME! More description here!",
 ether_isproto,
 ether_decode
};

protocol_t etherii =
{
 "Ethernet II",
 "Ethernet II link encapsulation protocol. FIXME! More description here!",
 ether_isproto,
 ether_decode
};

protocol_t ethersnap =
{
 "Ethernet SNAP",
 "Ethernet SNAP link encapsulation protocol. FIXME! More description here!",
 ether_isproto,
 ether_decode
};
