/*
Gnusniff - Network packet sniffer
Copyright (C) 1998 Peter Hawkins

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/
/*
IPv4.c
Contains the implementation for the IPv4 and the related ICMPv4 protocols
*/
#include <string.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <netinet/ip_icmp.h>
#include <arpa/inet.h>
#include "protocol.h"
#include "p_tcp.h"
#include "p_udp.h"
#include "hosts.h"
#include "utils.h"

/***************************************************************************
 IPv4 protocol implementation
 ***************************************************************************/

extern protocol_t ipv4, icmpv4;
int icmpv4_decode(struct packet_t *pkt, u_int offset, decoded_field_func *ff, gpointer data);

// This all comes straight from RFC 791.
static field_storage_t ipv4_fields[13]=
{
{
  "Version",
  "The version field indicates the format of the internet header. This is version 4.",
  0,4 // Offset 0, length 4 (bits)
},
{
  "IHL",
  "Internet Header Length is the length of the internet header in 32
  bit words, and thus points to the beginning of the data.  Note that
  the minimum value for a correct header is 5.",
  4,4
},
{
  "Service Type",
  "The Type of Service provides an indication of the abstract
   parameters of the quality of service desired.  These parameters are
   to be used to guide the selection of the actual service parameters
   when transmitting a datagram through a particular network.  Several
   networks offer service precedence, which somehow treats high
   precedence traffic as more important than other traffic (generally
   by accepting only traffic above a certain precedence at time of high
   load).  The major choice is a three way tradeoff between low-delay,
   high-reliability, and high-throughput.

   Bits 0-2:  Precedence.
   Bit    3:  0 = Normal Delay,       1 = Low Delay.
   Bits   4:  0 = Normal Throughput,  1 = High Throughput.
   Bits   5:  0 = Normal Reliability, 1 = High Reliability.
   Bit  6-7:  Reserved for Future Use.

      0     1     2     3     4     5     6     7
   +-----+-----+-----+-----+-----+-----+-----+-----+
   |                 |     |     |     |     |     |
   |   PRECEDENCE    |  D  |  T  |  R  |  0  |  0  |
   |                 |     |     |     |     |     |
   +-----+-----+-----+-----+-----+-----+-----+-----+
        Precedence

       111 - Network Control
       110 - Internetwork Control
       101 - CRITIC/ECP
       100 - Flash Override
       011 - Flash
       010 - Immediate
       001 - Priority
       000 - Routine

   The use of the Delay, Throughput, and Reliability indications may
   increase the cost (in some sense) of the service.  In many networks
   better performance for one of these parameters is coupled with worse
   performance on another.  Except for very unusual cases at most two
   of these three indications should be set.

   The type of service is used to specify the treatment of the datagram
   during its transmission through the internet system.

   The Network Control precedence designation is intended to be used
   within a network only.  The actual use and control of that
   designation is up to each network. The Internetwork Control
   designation is intended for use by gateway control originators only.
   If the actual use of these precedence designations is of concern to
   a particular network, it is the responsibility of that network to
   control the access to, and use of, those precedence designations.",

  8,8
},
{
  "Length",
  "Total Length is the length of the datagram, measured in octets,
   including internet header and data.  This field allows the length of
   a datagram to be up to 65,535 octets.  Such long datagrams are
   impractical for most hosts and networks.  All hosts must be prepared
   to accept datagrams of up to 576 octets (whether they arrive whole
   or in fragments).  It is recommended that hosts only send datagrams
   larger than 576 octets if they have assurance that the destination
   is prepared to accept the larger datagrams.

   The number 576 is selected to allow a reasonable sized data block to
   be transmitted in addition to the required header information.  For
   example, this size allows a data block of 512 octets plus 64 header
   octets to fit in a datagram.  The maximal internet header is 60
   octets, and a typical internet header is 20 octets, allowing a
   margin for headers of higher level protocols.",
  16,16
},
{
  "Identification",
  "An identifying value assigned by the sender to aid in assembling the
   fragments of a datagram.",
  32,16
},
{
  "Flags",
  "Various Control Flags.

   Bit 0: reserved, must be zero
   Bit 1: (DF) 0 = May Fragment,  1 = Don't Fragment.
   Bit 2: (MF) 0 = Last Fragment, 1 = More Fragments.

       0   1   2
     +---+---+---+
     |   | D | M |
     | 0 | F | F |
     +---+---+---+",
  48,3
},
{
  "Fragment Offset",
  "This field indicates where in the datagram this fragment belongs.
   The fragment offset is measured in units of 8 octets (64 bits).  The
   first fragment has offset zero.",
  51,13
},
{
  "Time to Live",
  "This field indicates the maximum time the datagram is allowed to
   remain in the internet system.  If this field contains the value
   zero, then the datagram must be destroyed.  This field is modified
   in internet header processing.  The time is measured in units of
   seconds, but since every module that processes a datagram must
   decrease the TTL by at least one even if it process the datagram in
   less than a second, the TTL must be thought of only as an upper
   bound on the time a datagram may exist.  The intention is to cause
   undeliverable datagrams to be discarded, and to bound the maximum
   datagram lifetime.",
   64,8
},
{
  "Protocol",
  "This field indicates the next level protocol used in the data
   portion of the internet datagram.",
   72,8
},
{
  "Header Checksum",
  "A checksum on the header only.  Since some header fields change
   (e.g., time to live), this is recomputed and verified at each point
   that the internet header is processed.

   The checksum algorithm is:

     The checksum field is the 16 bit one's complement of the one's
     complement sum of all 16 bit words in the header.  For purposes of
     computing the checksum, the value of the checksum field is zero.

   This is a simple to compute checksum and experimental evidence
   indicates it is adequate, but it is provisional and may be replaced
   by a CRC procedure, depending on further experience.",
  80,16
},
{
  "Source Address",
  "The source address.",
  96,32
},
{
  "Destination Address",
  "The destination address.",
  128,32
},
{
  "Options",
  "Variable length... See RFC 971... it's too big to include here...",
  160,0
}
};

/* Pinched from tcpdump */
unsigned short
in_cksum(register const u_short *sp, u_int len)
{
	register u_int32_t sum = 0;
	register int count;

	/*
	 * No need for endian conversions.
	 */
	for (count = len / 2; --count >= 0; )
		sum += *sp++;
	while (sum > 0xffff)
		sum = (sum & 0xffff) + (sum >> 16);
	sum = ~sum & 0xffff;

	return (sum);
}

#define IPOFF_MASK      0x1fff
#define IPOFF_OFFSET(x) ((x) & IPOFF_MASK)
#define IPOFF_FLAGS(x)  ((x) & ~IPOFF_MASK)
#define IPOFF_RF        0x8000  // Reserved fragment flag
#define IPOFF_DF        0x4000  // Don't fragment flag
#define IPOFF_MF        0x2000  // More fragments flag

char ipv4_type[]="IP packet";

gboolean ipv4_isproto(struct packet_t *pkt, u_int offset)
{
 struct iphdr *iphdr;
 /* Check:
    1) The packet is long enough to contain an IPv4 header.
    2) The ip header's internal length is sufficient.
    3) That the ip header version is indeed IPv4. */
 iphdr=(struct iphdr *)((pkt->data) + offset);

 if ((pkt->len<sizeof(struct iphdr)) || (iphdr->ihl<5) ||
    (iphdr->version!=4)) return FALSE;

 return TRUE;
}

/* Decode an IPv4 packet
   Takes a pointer to a packet_t struct and the offset of the header.
*/
#define IPADDFIELD(a,b) ff(ipv4_fields[a].name, ipv4_fields[a].description, ipv4_fields[a].pos + (offset * 8), ipv4_fields[a].len, b, data)
int ipv4_decode(struct packet_t *pkt, u_int offset, decoded_field_func *ff, gpointer data)
{
 struct iphdr *iphdr, *temphdr;
 size_t hdrlen;
 char buffer[1024];
 u_short checksum;
 struct sockaddr_in tempaddr;
 

 iphdr = (struct iphdr *) ((pkt->data) + offset );

 // Check for valid packet...
 if ((pkt->len<sizeof(struct iphdr)) || (iphdr->ihl<5) ||
    (iphdr->version!=4)) return -2;

 // Fill in the packet structure with details
 tempaddr.sin_family = AF_INET;
 tempaddr.sin_addr.s_addr = iphdr->saddr;
 if (pkt->src == NULL)
   pkt->src = gethostforaddr(pkt, (struct sockaddr *)&tempaddr);
 else
   addaddrtohost(pkt->src, (struct sockaddr *)&tempaddr);
/* The DNS kludge code stops a reverse DNS lookup from occurring on a packet
   which is itself a DNS lookup. This caused problems where the DNS packets you
   send out to resolve a packet's domain name are themselves sniffed, as this
   results in you doing a trawl of the internet DNS servers, continually working
   up in the DNS server chain of authorisation.
   It checks if this packet is either TCP or UDP, and if so, if the source or
   destination port is 53, in which case it is assumed to be DNS itself, and is
   ignored for the purposes of DNS lookups. */
#define DNS_KLUDGE 1

#ifdef DNS_KLUDGE
   if (!(((iphdr->protocol == IPPROTO_TCP) || (iphdr->protocol == IPPROTO_UDP)) && ( (ntohs(*(u_short *)(pkt->data + offset + (iphdr->ihl * 4))) == 53) || (ntohs(*(u_short *)(pkt->data + offset + 2 + (iphdr->ihl * 4))) == 53) )))
#endif
 lookuphost(pkt->src);

 tempaddr.sin_addr.s_addr = iphdr->daddr;
 if (pkt->dest == NULL)
   pkt->dest = gethostforaddr(pkt, (struct sockaddr *)&tempaddr);
 else
   addaddrtohost(pkt->dest, (struct sockaddr *)&tempaddr);
#ifdef DNS_KLUDGE
   if (!(((iphdr->protocol == IPPROTO_TCP) || (iphdr->protocol == IPPROTO_UDP)) && ( (ntohs(*(u_short *)(pkt->data + offset + (iphdr->ihl * 4))) == 53) || (ntohs(*(u_short *)(pkt->data + offset + 2 + (iphdr->ihl * 4))) == 53) )))
#endif DNS_KLUDGE
 lookuphost(pkt->dest);

 strncpy(pkt->description, ipv4_type, PACKET_DESCRIPTION_LENGTH);

 // Do a checksum on a copy of the packet
 hdrlen=iphdr->ihl * 4;
 temphdr=g_malloc(hdrlen);
 memcpy(temphdr,iphdr,hdrlen);
 temphdr->check=0;

 checksum = in_cksum((unsigned short *) temphdr, hdrlen);
 pkt->valid = pkt->valid && (checksum==iphdr->check);

 g_free(temphdr);

 pt_mutex_enter(pkt->src->mutex);
 strncpy(pkt->srcaddrstr, pkt->src->hostname, ADDRESS_STRING_LENGTH);
 pt_mutex_exit(pkt->src->mutex);

 pt_mutex_enter(pkt->dest->mutex);
 strncpy(pkt->dstaddrstr, pkt->dest->hostname, ADDRESS_STRING_LENGTH);
 pt_mutex_exit(pkt->dest->mutex);

 if (ff!=NULL) {
   int flags,mf,df,rf;
   flags=IPOFF_FLAGS(ntohs(iphdr->frag_off));
   mf=flags & IPOFF_MF;
   df=flags & IPOFF_DF;
   rf=flags & IPOFF_RF;
   snprintf(buffer, 1024, "SRC=%s DST=%s ID=%.4X LEN=%u %s%s%sFRAG=%.4X",
            pkt->src->swaddress, pkt->dest->swaddress, iphdr->id,
            ntohs(iphdr->tot_len), mf?"MF ":"", df?"DF ":"", rf?"RF ":"",
            IPOFF_OFFSET(ntohs(iphdr->frag_off)));
   ff(ipv4.name, ipv4.description, offset*8, 0, buffer, data);
   snprintf(buffer, 1024, "%u", iphdr->version);
   IPADDFIELD(0,buffer);
   snprintf(buffer, 1024, "%u dwords", iphdr->ihl);
   IPADDFIELD(1,buffer);
   snprintf(buffer, 1024, "0x%.2X", iphdr->tos);
   IPADDFIELD(2,buffer);
   snprintf(buffer, 1024, "%u bytes", ntohs(iphdr->tot_len));
   IPADDFIELD(3,buffer);
   snprintf(buffer, 1024, "0x%.4X", ntohs(iphdr->id));
   IPADDFIELD(4,buffer);
   snprintf(buffer, 1024, "%s %s %s",(mf?"MF":"(MF)"),(df?"DF":"(DF)"),(rf?"RF":"(RF)"));
   IPADDFIELD(5,buffer);
   snprintf(buffer, 1024, "0x%.4X bytes",IPOFF_OFFSET(ntohs(iphdr->frag_off)));
   IPADDFIELD(6,buffer);
   snprintf(buffer, 1024, "%u seconds",iphdr->ttl);
   IPADDFIELD(7,buffer);
   snprintf(buffer, 1024, "0x%.02X (",iphdr->protocol);

   switch (iphdr->protocol)
   {
    case IPPROTO_IP: case IPPROTO_TCP: strncat(buffer, "TCP)", 1024); break;
    case IPPROTO_ICMP: strncat(buffer, "ICMPv4)", 1024); break;
    case IPPROTO_IGMP: strncat(buffer, "IGMP)", 1024); break;
    case IPPROTO_IPIP: strncat(buffer, "IP)", 1024); break;
    case IPPROTO_EGP: strncat(buffer, "EGP)", 1024); break;
    case IPPROTO_PUP: strncat(buffer, "PUP)", 1024); break;
    case IPPROTO_UDP: strncat(buffer, "UDP)", 1024); break;
    case IPPROTO_IDP: strncat(buffer, "IDP)", 1024); break;
    case IPPROTO_IPV6: strncat(buffer, "IPv6)", 1024); break;
    case IPPROTO_ICMPV6: strncat(buffer, "ICMPv6)", 1024); break;
    case IPPROTO_RAW: strncat(buffer, "Raw data)", 1024); break;
    default:
    strncat(buffer, "Unknown type)", 1024); break;
   }
   IPADDFIELD(8,buffer);
   snprintf(buffer, 1024, "0x%.4X",iphdr->check);
   if (iphdr->check != checksum) {
    char buffer2[256];
    snprintf(buffer2, 256, " (should be 0x%.4X)", checksum);
    strncat(buffer, buffer2, 1024);
   }
   IPADDFIELD(9,buffer);
   IPADDFIELD(10,pkt->src->swaddress);
   IPADDFIELD(11,pkt->dest->swaddress);

 }

 /* If this is fragment 0, pawn this packet off onto the next level protocol. */
 if (IPOFF_OFFSET(ntohs(iphdr->frag_off)) == 0) {
  switch (iphdr->protocol)
  {
   case IPPROTO_ICMP:
     icmpv4_decode(pkt, offset + (iphdr->ihl * 4), ff, data);
     break;
   case IPPROTO_TCP: case IPPROTO_IP:
     tcp_decode(pkt, offset + (iphdr->ihl * 4), ff, data);
     break;
   case IPPROTO_UDP:
     udp_decode(pkt, offset + (iphdr->ihl * 4), ff, data);
     break;
   default:
   ; // Do nothing. We don't understand this subprotocol type.
  }
 }
 // All done

 return hdrlen;
}
/*
GList      *ipv4_get_display_fields( struct packet_t *pkt, u_int offset )
{
 GList *l=NULL,*n=NULL;
 field_t *data;
 struct iphdr *iphdr;
 int i;

 iphdr = (struct iphdr *) ((pkt->data) + offset );

 for (i=0;i<13;i++)
 {
   data=g_malloc0(sizeof(field_t));
   data->name=ipv4_fields[i].name;
   data->description=ipv4_fields[i].description;
   data->location=&ipv4_fields[i].location;
   n=g_list_append(n,data);
   if (l==NULL) l=n;
 }
 sprintf(((field_t *)(l->data))->value ,"%u",iphdr->version);
 n=l->next;
 sprintf(((field_t *)(n->data))->value,"%u dwords)",iphdr->ihl);
 n=n->next;
 sprintf(((field_t *)(n->data))->value,"0x%X",iphdr->tos);
 n=n->next;
 sprintf(((field_t *)(n->data))->value,"%u bytes",ntohs(iphdr->tot_len));
 n=n->next;
 sprintf(((field_t *)(n->data))->value,"0x%.4X",ntohs(iphdr->id));
 n=n->next;
 {
  int mf,df,rf,flags;
  flags=IPOFF_FLAGS(ntohs(iphdr->frag_off));
  mf=flags & IPOFF_MF;
  df=flags & IPOFF_DF;
  rf=flags & IPOFF_RF;
  sprintf(((field_t *)(n->data))->value,"%s %s %s",(mf?"MF":"(MF)"),(df?"DF":"(DF)"),(rf?"RF":"(RF)"));
 }
 n=n->next;
 sprintf(((field_t *)(n->data))->value,"0x%.4X bytes",IPOFF_OFFSET(ntohs(iphdr->frag_off)));
 n=n->next;
 sprintf(((field_t *)(n->data))->value,"%u seconds",iphdr->ttl);
 n=n->next;

 {
  protocol_t *p;
  p=ipv4_get_contained_protocol(pkt,offset);
  if (p!=NULL)
    sprintf(((field_t *)(n->data))->value,"%s",p->name);
  else
    sprintf(((field_t *)(n->data))->value,"%u (unrecognised protocol)",iphdr->protocol);
 }
 n=n->next;
 sprintf(((field_t *)(n->data))->value,"0x%.4X",iphdr->check);
 n=n->next;
 strncpy(((field_t *)(n->data))->value,pkt->source.as_string,16);
 n=n->next;
 strncpy(((field_t *)(n->data))->value,pkt->destination.as_string,16);
 n=n->next;
 strcpy(((field_t *)(n->data))->value,"Unknown");

 return l;
}
*/
/***************************************************************************
 ICMPv4 protocol implementation
 ***************************************************************************/

static char icmpv4_types[19][32] = {
"ICMP Echo reply", "Invalid ICMP", "Invalid ICMP", "ICMP Destination unreachable",
"ICMP Source quench", "ICMP Redirect (change route)", "Invalid ICMP", "Invalid ICMP",
"ICMP Echo request", "Invalid ICMP", "Invalid ICMP", "ICMP Time exceeded",
"ICMP Parameter problem", "ICMP Timestamp request", "ICMP Timestamp reply",
"ICMP Information request", "ICMP Information reply", "ICMP Address mask request",
"ICMP Address mask Reply" };

static field_storage_t icmpv4_fields[3] =
{
 {
  "Type",
  "This field specifies the type of ICMP packet. Valid range is 0-18.",
  0,8
 },
 {
  "Code",
  "This field means different things depending on the type of the packet.",
  8,8
 },
 {
  "Checksum",
  "Checksum of the ICMP header.",
  16,16
 }
};

/* Decode an ICMPv4 packet
   Takes a pointer to a packet_t struct and the offset of the header.
*/
int icmpv4_decode(struct packet_t *pkt, u_int offset, decoded_field_func *ff, gpointer data)
{
 struct icmphdr *icmphdr;

 icmphdr = (struct icmphdr *) ((pkt->data) + offset );
 // Check for valid ICMP message type
 if (icmphdr->type > NR_ICMP_TYPES) return -2;

 strncpy(pkt->description, icmpv4_types[icmphdr->type], PACKET_DESCRIPTION_LENGTH);

 if (ff!=NULL) {
   char buffer[1024];
   ff(icmpv4.name, icmpv4.description, offset*8, 0, "", data);
   ff(icmpv4_fields[0].name, icmpv4_fields[0].description, icmpv4_fields[0].pos + (offset * 8), icmpv4_fields[0].len, icmpv4_types[icmphdr->type], data);
   snprintf(buffer,1024,"0x%.2X", icmphdr->code);
   ff(icmpv4_fields[1].name, icmpv4_fields[1].description, icmpv4_fields[1].pos + (offset * 8), icmpv4_fields[1].len, buffer, data);
 }
 // FIXME: Should check checksum.
 return 0; // FIXME: should return length of header
}


/***************************************************************************
 Protocol registration
 ***************************************************************************/

protocol_t ipv4 =
{
 "IPv4",
 "The Internet Protocol version 4, as defined by RFC 791.",
 ipv4_isproto,
 ipv4_decode
};

protocol_t icmpv4 =
{
 "ICMPv4",
 "The Internet Control Message Protocol version 4, as defined by RFC 792.",
 NULL, // isproto isn't needed as the IP packet always tells us if this is ICMPv4
 icmpv4_decode
};

