/*
Gnusniff - Network packet sniffer
Copyright (C) 1998 Peter Hawkins

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <stdlib.h>
#include <string.h>
#include <netinet/in.h>
#include <netipx/ipx.h>
#include "interface.h"
#include "protocol.h"


char ipx_type[] = "IPX packet";
/* Splurged from tcpdump... */
/* well-known sockets */
#define	IPX_SKT_NCP		0x0451
#define	IPX_SKT_SAP		0x0452
#define	IPX_SKT_RIP		0x0453
#define	IPX_SKT_NETBIOS		0x0455
#define	IPX_SKT_DIAGNOSTICS	0x0456

/* IPX transport header */
struct ipxHdr {
    u_short	cksum;		/* Checksum */
    u_short	length;		/* Length, in bytes, including header */
    u_char	tCtl;		/* Transport Control (i.e. hop count) */
    u_char	pType;		/* Packet Type (i.e. level 2 protocol) */
    u_long	dstNet;	        /* destination net */
    u_char	dstNode[6];	/* destination node */
    u_short	dstSkt;		/* destination socket */
    u_long	srcNet;	        /* source net */
    u_char	srcNode[6];	/* source node */
    u_short	srcSkt;		/* source socket */
} ipx_hdr_t;

const char *
ipxaddr_string(u_int32_t net, const u_char *node)
{
    static char line[256];

    sprintf(line, "%x.%02x:%02x:%02x:%02x:%02x:%02x",
	    net, node[0], node[1], node[2], node[3], node[4], node[5]);

    return line;
}

/*gboolean ipx_isproto(struct packet_t *pkt, u_int offset)
{
 if (pkt->len < sizeof(struct ipxHdr)) return FALSE;
 // Any other tests?
}*/

int ipx_decode(struct packet_t *pkt, u_int offset, decoded_field_func *ff, gpointer data)
{
 const struct ipxHdr *ipx = (const struct ipxHdr *) ((pkt->data) + offset);
 struct sockaddr_ipx tempaddr;

 // If the packet is too short to contain an IPX packet, bail out.
 if (pkt->len < sizeof(struct ipxHdr)) return -2;

 tempaddr.sipx_family = AF_IPX;
 tempaddr.sipx_port = 0;
 tempaddr.sipx_type = ipx->pType;

 tempaddr.sipx_network = ipx->dstNet;
 memcpy(tempaddr.sipx_node, ipx->dstNode, IPX_NODE_LEN);
 if (pkt->dest == NULL)
   pkt->dest = gethostforaddr(pkt, (struct sockaddr *)&tempaddr);
 else
   addaddrtohost(pkt->dest, (struct sockaddr *)&tempaddr);

 tempaddr.sipx_network = ipx->srcNet;
 memcpy(tempaddr.sipx_node, ipx->srcNode, IPX_NODE_LEN);
 if (pkt->src == NULL)
   pkt->src = gethostforaddr(pkt, (struct sockaddr *)&tempaddr);
 else
   addaddrtohost(pkt->src, (struct sockaddr *)&tempaddr);

 strncpy(pkt->srcaddrstr,ipxaddr_string(ntohl(ipx->srcNet), ipx->srcNode),ADDRESS_STRING_LENGTH);
 strncpy(pkt->dstaddrstr,ipxaddr_string(ntohl(ipx->dstNet), ipx->dstNode),ADDRESS_STRING_LENGTH);
 strncpy(pkt->description, ipx_type, PACKET_DESCRIPTION_LENGTH);

 pkt->valid = (pkt->len > ipx->length);

 if (ff!=NULL) {
  char buffer[1024];

  ff("IPX", "Description of IPX", offset*8, 0, NULL, data);

  snprintf(buffer, 1024, "0x%.4X", ntohs(ipx->cksum));
  ff("Checksum",NULL, offset*8, 16, buffer, data);
  snprintf(buffer, 1024, "%d bytes", ntohs(ipx->length));
  ff("Packet length",NULL, 16+offset*8, 16, buffer, data);
  snprintf(buffer, 1024, "0x%.2X", ipx->tCtl);
  ff("Transport control (hop count)",NULL, 32+offset*8, 8, buffer, data);
  snprintf(buffer, 1024, "0x%.2X", ipx->pType);
  ff("Packet type",NULL, 40+offset*8, 8, buffer, data);
  snprintf(buffer, 1024, "0x%.8X", ntohl(ipx->dstNet));
  ff("Destination net",NULL, 48+offset*8, 32, buffer, data);
  snprintf(buffer, 1024, "%02X:%02X:%02X:%02X:%02X:%02X", ipx->dstNode[0], ipx->dstNode[1], ipx->dstNode[2], ipx->dstNode[3], ipx->dstNode[4], ipx->dstNode[5]);
  ff("Destination node",NULL, 80+offset*8, 48, buffer, data);
  snprintf(buffer, 1024, "0x%.4X", ntohs(ipx->dstSkt));
  ff("Destination socket",NULL, 128+offset*8, 16, buffer, data);
  snprintf(buffer, 1024, "0x%.4X", ntohl(ipx->srcNet));
  ff("Source net",NULL, 144+offset*8, 32, buffer, data);
  snprintf(buffer, 1024, "%02X:%02X:%02X:%02X:%02X:%02X", ipx->srcNode[0], ipx->srcNode[1], ipx->srcNode[2], ipx->srcNode[3], ipx->srcNode[4], ipx->srcNode[5]);
  ff("Source node",NULL, 176+offset*8, 48, buffer, data);
  snprintf(buffer, 1024, "0x%.4X", ntohs(ipx->srcSkt));
  ff("Source socket",NULL, 224+offset*8, 16, buffer, data);
 }
 return sizeof(struct ipxHdr) + offset;
}



protocol_t ipx =
{
 "IPX",
 "IPX. Anyone got a better description?",
 NULL,
 ipx_decode
};
