/*
Gnusniff - Network packet sniffer
Copyright (C) 1998 Peter Hawkins

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <stdlib.h>
#include <string.h>
#include <netinet/in.h>
#include <netinet/udp.h>
#include <arpa/nameser.h>
#include <glib.h>
#include "interface.h"
#include "protocol.h"
#include "p_ipv4.h"
#include "p_dns.h"

char udp_type[] = "UDP";

static const char udptype_unknown[] = "Unknown";

#define TFTP_PORT 69
#define KERBEROS_PORT 88
#define SUNRPC_PORT 111
#define SNMP_PORT 161
#define NTP_PORT 123
#define SNMPTRAP_PORT 162
#define RIP_PORT 520
#define KERBEROS_SEC_PORT 750


static const char *portname(u_short port)
{
 switch(port)
 {
  case NAMESERVER_PORT:// DNS
    return "DNS";
  case TFTP_PORT: // Trivial FTP
    return "TFTP";
  case SUNRPC_PORT:
    return "SUNRPC";
  case SNMP_PORT:
    return "SNMP";
  case NTP_PORT:
    return "NTP";
  case SNMPTRAP_PORT:
    return "SNMPTRAP";
  case RIP_PORT:
    return "RIP";
  case KERBEROS_SEC_PORT:
    return "KERBEROS";
  default:
    return udptype_unknown;
 }
}

int udp_decode(struct packet_t *pkt, u_int offset, decoded_field_func *ff, gpointer data)
{
 const struct udphdr *udp = (const struct udphdr *)((pkt->data) + offset);
 char buffer[1024];
 const char *temp;

 pkt->srcport = udp->source;
 snprintf(buffer,1024,":%d", ntohs(udp->source));
 strncat(pkt->srcaddrstr, buffer, ADDRESS_STRING_LENGTH);

 pkt->destport = udp->dest;
 snprintf(buffer,1024,":%d", ntohs(udp->dest));
 strncat(pkt->dstaddrstr, buffer, ADDRESS_STRING_LENGTH);

 strncpy(pkt->description, udp_type,PACKET_DESCRIPTION_LENGTH);

 pkt->valid = (udp->dest!=0) && (udp->len > sizeof(struct udphdr));
 // FIXME: Check the checksum

 temp = portname( ((udp->source==0) || (ntohs(udp->source) > IPPORT_RESERVED))?ntohs(udp->dest):ntohs(udp->source) );
 if (temp!=udptype_unknown) {
   snprintf(buffer, 1024, "/%s", temp);
   strncat(pkt->description, buffer, PACKET_DESCRIPTION_LENGTH);
 }

 if (ff!=NULL) {
   snprintf(buffer, 1024, "SRC=%d DEST=%d LEN=%d",ntohs(udp->source), ntohs(udp->dest), ntohs(udp->len));
   ff("UDP","User Datagram Protocol", offset*8, 0, buffer, data);
   
   snprintf(buffer, 1024, "%d (%s)", ntohs(udp->source), portname(ntohs(udp->source)));
   ff("Source port", NULL, offset*8, 16, buffer, data);

   snprintf(buffer, 1024, "%d (%s)", ntohs(udp->dest), portname(ntohs(udp->dest)));
   ff("Destination port", NULL, 16+offset*8, 16, buffer, data);

   snprintf(buffer, 1024, "%d bytes", ntohs(udp->len));
   ff("Length", NULL, 32 + offset*8, 16, buffer, data);
   
   snprintf(buffer, 1024, "0x%.4X", ntohs(udp->check));
   ff("Header checksum", NULL, 48+offset*8, 16, buffer, data);
 }

 switch (((udp->source==0) || (ntohs(udp->source) > IPPORT_RESERVED))?ntohs(udp->dest):ntohs(udp->source)) {
  case NAMESERVER_PORT:
    dns_decode(pkt, offset + sizeof(struct udphdr), ff, data);
    break;
  default:
   ;
 }

 return offset + sizeof(struct udphdr);
}




