/*
Gnusniff - Network packet sniffer
Copyright (C) 1998 Peter Hawkins

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

/* This file handles a few of the actual sniffing aspects. */

#include "config.h"
#include <sys/types.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <glib.h>
#include <gnome.h>
#include "sniff.h"
#include "mainwnd.h"
#include "utils.h"

struct packet_t *packet_add(struct packet_t **list)
{
 struct packet_t *n;
 g_assert(list!=NULL);

 n = *list;
 if (*list==NULL) {
   n = *list = g_malloc0(sizeof(struct packet_t));
 } else {
   g_assert(n!=NULL);
   while (n->next!=NULL) {
     n=n->next;
   }
   n->next = g_malloc0(sizeof(struct packet_t));
   n=n->next;
 }
 return n;
}

void packet_list_free(struct packet_t **list)
{
 struct packet_t *n, *t;
 
 g_assert(list!=NULL);
 n=*list;

 while (n!=NULL) {
  t = n;
  n = n->next;
  g_free(t);
 }
}


static void sniff_callback(struct interface *ife, struct pcap_pkthdr *hdr, u_char *data)
{
 struct packet_t *pkt;
 pt_block_thread();
 pt_mutex_enter(ife->sniff.mutex);

 ife->sniff.numpackets++;
 pkt = packet_add(&ife->sniff.packets);

 pkt->parent = &ife->sniff;
 pkt->len = hdr->caplen;
 pkt->data = g_malloc(pkt->len);
 memcpy(pkt->data, data, pkt->len);

 pt_mutex_exit(ife->sniff.mutex);
 pt_unblock_thread();
}

static int sniff_func(struct interface *ife)
{
 do
 {
    pcap_loop(ife->sniff.pcap, -1, (pcap_handler) sniff_callback, (void *)ife);
 } while (1);

 return 1;
}

int sniff_begin(struct interface *ife)
{
   pthread_attr_t attr;
   char ebuf[1024];

   if (geteuid()!=0) {
     GtkWidget *mbox = gnome_message_box_new("You must be root to capture packet data!\n",
                                              GNOME_MESSAGE_BOX_ERROR, "Ok", NULL);
     gtk_window_set_modal(GTK_WINDOW(mbox), TRUE);
     gtk_widget_show(mbox);
     return -1; /* If we are not root, we cannot capture */
   }
   packet_list_free(&ife->sniff.packets);
   memset(&ife->sniff, 0, sizeof(struct sniff_t));

   ife->sniff.parent = ife;
   
   ife->sniff.active = 1;
   
   ife->sniff.pcap = pcap_open_live(ife->name, 1500, 1, 0, ebuf);
   
   if (!ife->sniff.pcap)
   {
      fprintf(stderr,"ERROR: %s\n", ebuf);
      exit(1);
   }
   
   pthread_attr_init(&attr);
   pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);
   
   if (pthread_create(&ife->sniff.capture_thread, &attr, (void *)sniff_func, ife) != 0)
   {
      fprintf(stderr, "Error creating the capture thread!\n");
      exit(1);
   }

   mainwnd_add_refresh_timeout();

   return 0;
}

int sniff_end(struct interface *ife)
{
 if (ife->sniff.active) {
  pthread_cancel(ife->sniff.capture_thread);
  pcap_close(ife->sniff.pcap);
  ife->sniff.active = 0;
  mainwnd_remove_refresh_timeout();
 }
 return 0;
}
