#ifndef CD_H
#define CD_H 1

/* cd.h: declarations file for the cgmdraw module.

        Written by G. Edward Johnson <mailto:lorax@nist.gov>
        Date: April 1996
        Copyright: cd software produced by NIST, an agency of the 
	U.S. government, is by statute not subject to copyright
	in the United States. Recipients of this software assume all 
	responsibilities associated with its operation, modification
	and maintenance.
 
        Portions of this package are from the gd package written by
	Thomas Boutell and are copyright 1994, 1995, Quest Protein 
	Database Center, Cold Spring Harbor Labs.  They are marked in the
	source code.

*/

/* #include <prof.h> */

/* stdio is needed for file I/O. */
#include <stdio.h>

/* This can not be changed to a value larger than 256, though smaller
 * values can be used.
 */

#define cdMaxColors 256

/* If you know you will be working with large pictures, increase the values
 * of the next two constants.
 */

/* The initial size of the element list.  When it fills up, we will just
 * make it bigger.  Starting  with a larger number reduces the frequency of
 * the list growing, but increases the memory needed for small pictures 
 */

#define CDSTARTLISTSIZE 4096

/* How much the element list grows by.  When the list fills up, we allocate
 * a new larger list.  This number is how much larger.  using a larger number
 * decreases the frequency of the list growing, but if only a small amount 
 * more is needed, it could waste memory 
 */

#define CDGROWLISTSIZE 2048

/* Image type. See functions below; you will not need to change
	the elements directly. Use the provided macros to
	access sx, sy, the color table, and colorsTotal for 
	read-only purposes. */

#if defined(__cplusplus) || defined(c_plusplus)
extern "C" {
#endif

typedef struct cdImageStruct {
	unsigned char * elemlist;
	int sx;
	int sy;
	int colorsTotal;
	int red[cdMaxColors];
	int green[cdMaxColors];
	int blue[cdMaxColors]; 
	int open[cdMaxColors];
	/* You can have multiple pictures in the file,  this keeps track
	 * of which one you are on */
	int picnum;
	/* Linetype, line width, line color have a broader scope in CGM */
	int ltype;
	int lwidth;
	int lcolor;
	/* interior style [of filled objects] (for me) can be empty, hollow, 
	 * solid, hatch [don't do pattern, geometric pattern, interpolated*/
	int shapestyle;
	/* fill color, color used on inside of closed objects, significant
	 * if interior style is hollow, solid, hatch, or geometric pattern */
	int shapecolor;
	/* hatch index, which hatch style to use, 1=horizontal, 2=vertical,
	 * 3=pos.slope, 4=neg.slope, 5=hor/vert.crosshatch,
	 * 6=pos/neg.crosshatch*/
	int shapehatch;
	/* The edges of filled shapes can have line styles too.  They 
	 * correspond to the ones for lines.  These next few set them. */
	int edgetype;
	int edgewidth;
	int edgecolor;
	int edgevis; /* is the edge visible or invisible */
	/* now for the TEXT related attributes,  Text Color, Text Height,
	 * and Text font index */
	int textcolor;
	int textheight;
	int textfont;
	int textpath;
	/* the next three are used for maintaining the element list */
        long int bytestoend; /* number of bytes to end of the element list */
	long int listlen; /* the total length of the element list */
        unsigned char * curelemlist; /* where we curently are in the list */
} cdImage;

typedef cdImage * cdImagePtr;


/* Point type for use in polygon drawing. */

typedef struct cdPointStruct{
	int x, y;
} cdPoint, *cdPointPtr;



/* Functions to manipulate images. */

cdImagePtr cdImageCreate(int sx, int sy);
int cdCgmNewPic(cdImagePtr im, int sticky);
int cdImageCgm(cdImagePtr im, FILE *);
int cdImageDestroy(cdImagePtr im);

/* Use cdLine, not cdImageLine */
int cdLine(cdImagePtr im, int x1, int y1, int x2, int y2);
/* Corners specified (not width and height). Upper left first, lower right
 	second. */
int cdRectangle(cdImagePtr im, int x1, int y1, int x2, int y2);
/* center x, then center y, then radius of circle */
int cdCircle(cdImagePtr im, int cx, int cy, int r);
/* start, middle and end of arc */
int cdArc3Pt(cdImagePtr im, int sx,int sy, int ix,int iy, int ex,int ey);
/* cl is 0 for pie closure, 1 for cord closure */
int cdArc3PtClose(cdImagePtr im, int sx,int sy, int ix,int iy, int ex,int ey, int cl);
int cdEllipse(cdImagePtr im, int cx,int cy, int d1x,int d1y, int d2x,int d2y );
/* polyshapes */
int cdPolygon(cdImagePtr im, cdPointPtr p, int n);
int cdPolyLine(cdImagePtr im, cdPointPtr p, int n);

/* Functions for Compatibility with gd */
int cdImageLine(cdImagePtr im, int x1, int y1, int x2, int y2, int color);
int cdImageRectangle(cdImagePtr im, int x1, int y1, int x2, int y2, int color);


int cdImageBoundsSafe(cdImagePtr im, int x, int y);
/* These put characters in the picture.  CGM can handle fonts*/
/* (x,y) is the lower left corner of where the text goes */
int cdText(cdImagePtr im, int x, int y, const char *);


/* Functions for allocating colors */
int cdImageColorAllocate(cdImagePtr im, int r, int g, int b);
int cdImageColorClosest(cdImagePtr im, int r, int g, int b);
int cdImageColorExact(cdImagePtr im, int r, int g, int b);
int cdImageColorDeallocate(cdImagePtr im, int color);
int cdImageColor16(cdImagePtr im);

/* gej: functions that set style attributes */
int cdSetLineAttrib(cdImagePtr im, int lntype, int lnwidth, int lncolor);
int cdSetShapeFillAttrib(cdImagePtr im, int instyle, int incolor, int inhatch);
int cdSetShapeEdgeAttrib(cdImagePtr im, int edtype, int edwidth, int edcolor, int edvis);
int cdSetTextAttrib(cdImagePtr im, int font, int color, int height);
/* gej: or if you prefer, set the attributes individually */
int cdSetLineType(cdImagePtr im, int lntype);
int cdSetLineWidth(cdImagePtr im, int lnwidth);
int cdSetLineColor(cdImagePtr im, int lncolor);
int cdSetFillStyle(cdImagePtr im, int instyle);
int cdSetFillColor(cdImagePtr im, int incolor);
int cdSetFillHatch(cdImagePtr im, int inhatch);
int cdSetEdgeType(cdImagePtr im, int edtype);
int cdSetEdgeWidth(cdImagePtr im, int edwidth);
int cdSetEdgeColor(cdImagePtr im, int edcolor);
int cdSetEdgeVis(cdImagePtr im, int edvis);
int cdSetTextFont(cdImagePtr im, int font);
int cdSetTextColor(cdImagePtr im, int color);
int cdSetTextHeight(cdImagePtr im, int height);
/* geJ: these individual attributes can't be set with a group function */
int cdSetTextPath(cdImagePtr im, int tpath);
int cdSetTextOrient(cdImagePtr im, int xup, int yup, int xbase, int ybase);

/* Macros to access information about images. READ ONLY. Changing
	these values will NOT have the desired result. */
#define cdImageSX(im) ((im)->sx)
#define cdImageSY(im) ((im)->sy)
#define cdImageColorsTotal(im) ((im)->colorsTotal)
#define cdImageRed(im, c) ((im)->red[(c)])
#define cdImageGreen(im, c) ((im)->green[(c)])
#define cdImageBlue(im, c) ((im)->blue[(c)])

/* Source: Independent JPEG Group
 * In ANSI C, and indeed any rational implementation, size_t is also the
 * type returned by sizeof().  However, it seems there are some irrational
 * implementations out there, in which sizeof() returns an int even though
 * size_t is defined as long or unsigned long.  To ensure consistent results
 * we always use this SIZEOF() macro in place of using sizeof() directly.
 */

#define SIZEOF(object)  ((size_t) sizeof(object))
 
/* GeJ: these are helper functions I use in cd.  That means DON'T call
 * them from your program.  Yes, that means you.  */
int cdCgmHeader(cdImagePtr);
int cdCgmPic(cdImagePtr, int);
int cdImageColorClear(cdImagePtr im);

#if defined(__cplusplus) || defined(c_plusplus)
}
#endif


#endif
