#!perl
#script: upload.pl

use strict "refs";
use strict "subs";
use CGI::Carp 'fatalsToBrowser';
use Date::Format;
use File::Copy;
use File::stat;
use VMS::Filespec;
use URI::Escape;
$|=1;

# use CGI.PM library
use CGI qw(:standard :ssl);
use_named_parameters(1);

# set to 1 to print debugging messages, 0 to disable debugging messages
$DEBUG = 0;

## UPLOAD.PL
## by Evan Antworth, Summer Institute of Linguistics, <evan.antworth@sil.org>
## Upload remote file(s) to Web server directory tree
## Requires CGI.PM library, available from:
##   http://www-genome.wi.mit.edu/ftp/distribution/software/WWW/cgi_docs.html
## See "local configuration" section below before using
## User must have an account on server
## User must have write access to destination directory on server
## User's Web browser must support file upload and JavaScript
## Supports cookies (optional)
## Keeps a log file of attempted uploads (optional)
## Use can be restricted to accepted IPs
## Runs on both VMS and UNIX (at least, but only tested on VMS)
## Web server (i.e. the account under which it runs) must have read access to
##   password file (on VMS, the Web server account must have SYSPRV)
## To port the script to other OSes, see these subroutines:
##   save_file, translate_path, validate_filename
## Revision history
##   last modified 31-Aug-1998

## Known bugs:
## none

# set various globals
$server_name = server_name();
$server_port = server_port();
$script_name = script_name();
$path_info = path_info();
$path_translated = path_translated();
$remote_host = remote_host();
$remote_user = remote_user();
$ssl = https();  # this doesn't seem to work!
&define_javascript;

########## local configuration ########################################

# customize these settings for your site

# manually set your operating system
# this script tries to automatically determine your OS
# don't use this unless the script doesn't correctly recognize your OS
# accepted values: UNIX, VMS, WINDOWS, MACINTOSH, OS2
# $OS = 'VMS';

# 1 if you are running a secure (SSL) server, 0 otherwise
$secure_server = 1;
$secure_port = '443';

# Web servers that support this script
# leave list empty if only one server
@servers = (qw/
yourhost.edu
yourhost2.edu
/);

# only accept connections from these nets
# specify IP number of nets to accept as xxx.xxx.xxx.0  e.g. 208.145.80.0
# leave list empty to accept connections from all nets
# "localhost" included for debugging purposes
@accept_nets = (qw/
/);

# 1 to use cookies, 0 not to use cookies
$use_cookies = 1;

# 1 to use cookies only on secure connection, 0 otherwise
$secure_cookies = 0;

# email address of Webmaster/page maintainer
$email = 'Webmaster@yourhost.edu';

# single line in header (use HTML tags)
# $header_text = '<img src=/icons/logo.gif>';

# single line in footer (use HTML tags)
$footer_text = '<strong>[<a href="/">Home</a>]</strong>';

# number of upload filefields to print in form
$filefields = 5;

# keep a log file of all upload attempts
# give physical file specification
# leave empty to turn off logging
$logfile = '/WWW_ROOT/WEBMASTER/UPLOAD.LOG';

# required version of CGI.PM (VMS needs 2.39 or later)
$required_cgi_pm_version = '2.39';

# limit total size of uploaded files
# comment out to allow unlimited size
# $post_max = 1024 * 2000;   # max 2000K

# message to display if browser doesn't support JavaScript (use HTML tags)
$nojavascript = <<EOF;
<p><b><big>JavaScript is required to use this form!
Either your browser doesn't support JavaScript, 
or you have disabled it.</big></b>
EOF
   ;

# instructions for use of the form (use HTML tags)
$help_message = <<EOF;
<p>You must have a user account on host $server_name to upload files.
Your Web browser must support file uploading and JavaScript. Netscape 2+
and Explorer 4+ will work, other browsers may (not) work.
To upload files, do the following:
<ol>
<li>Determine the URL of the directory where you want to upload the file(s),
e.g. intranet.sil.org/softwaredev/projects/
<li>If necessary, use the pop-up menu to select a different server, 
then click GO.
<li>Fill in your username and password.
<li>Specify the directory path of the file(s) to upload, 
e.g. /softwaredev/projects/
<li>Use the Browse buttons to select files to upload from your local disk.
<br>You can upload ascii files (text, html) and binary files 
(gif, jpeg, zip, etc.).
<li>Click UPLOAD.
<li>Below the form (in this same space) you will see a message describing
either success or failure for each file upload attempt.
</ol>
EOF
   ;

## changes to CGI.PM ##

# For VMS, CGI.PM version 2.39 (or later) needs these changes.

# This change is optional, but recommended.
# Find this line near the top of the file (about line 26):
# $TempFile::TMPDIRECTORY = '/usr/tmp';
# change it to
# $TempFile::TMPDIRECTORY = '/SYS$SCRATCH';
# and uncomment it (remove the #).

# This change may be necessary in versions of the OSU Web server prior to 3.3
#  in order for the redirect function to work properly
# In the "header" subroutine, locate these lines (line numbers will differ
#  in later versions of CGI.PM):
# 
#  1116       push(@header,@other);
#  1117       push(@header,"Content-Type: $type");
# 
# and change them to:
# 
#  1116       push(@header,"Content-Type: $type");
#  1117       push(@header,@other);


# If you are using Perl version 5.005 (or later?), you must make these changes:
# (1) Change the definition of $CRLF from \n to \r\n; i.e. find this line:
#  137    $CRLF = "\n";
#  and change it to:
#  137    $CRLF = "\r\n";
# (2) In the readHeader subroutine, comment out these lines:
# 2976    if ($CGI::OS eq 'VMS') {  # tssk, tssk: inconsistency alert!
# 2977      local($CRLF) = "\015\012";
# 2978   }

########## end local configuration ########################################


# set/override CGI globals
$CGI::DISABLE_UPLOADS = 0;  # enable uploads
$CGI::POST_MAX = $post_max if $post_max;

# borrowed from CGI.PM
# FIGURE OUT THE OS WE'RE RUNNING UNDER
# Some systems support the $^O variable.  If not
# available then require() the Config library
unless ($OS) {
    unless ($OS = $^O) {
        require Config;
        $OS = $Config::Config{'osname'};
    }
}
if ($OS=~/Win/i) {
    $OS = 'WINDOWS';
} elsif ($OS=~/vms/i) {
    $OS = 'VMS';
} elsif ($OS=~/^MacOS$/i) {
    $OS = 'MACINTOSH';
} elsif ($OS=~/os2/i) {
    $OS = 'OS2';
} else {
    $OS = 'UNIX';
}

# see if we are changing servers
if ($newserver = param('server')) {
   $protocol = 'http';
   $port = '';
   if ($server_port eq $secure_port) {
       $protocol = 'https';
       $port = ":$secure_port";
   } else {
       $port = ":$server_port" if $server_port ne '80';
   }
   print redirect(-location=>"$protocol://$newserver$port$script_name");
   exit 0;
}

# check for cookies
if ($use_cookies) {
   if (param('username')) {
      $remote_user = uc(param('username'));
   } else {
      $remote_user = (cookie(-name=>'upload_username') or $remote_user);
   }
   if (param('path')) {
      $default_path = param('path');
   } else {
      $default_path = (cookie(-name=>'upload_path') or '/');
   }
   $cookie_username = cookie(-name=>'upload_username',
                             -value=>$remote_user,
                             -secure=>$secure_cookies,
                             -expires=>'+1y');
   $cookie_path   = cookie(-name=>'upload_path',
                             -value=>$default_path,
                             -secure=>$secure_cookies,
                             -expires=>'+1y');
# use header with cookies
   print header(-expires=>'now',-cookie=>[$cookie_username,$cookie_path]);
} else {
# use header without cookies
   print header(-expires=>'now');
} #if

#if form is resubmitted, collect filenames
@filenames = param('filename');

# start printing page
print start_html(-title=>"Upload files to $server_name",
                 -script=>$JSCRIPT,
                 -noscript=>$nojavascript,
                 -onLoad=>'checkBrowser()');

# print various globals for debugging
print p,"DEBUG: ",
  "Perl: $], OS: $OS, post_max: $post_max, secure_server: $secure_server, secure_port: $secure_port, ",
  "use_cookies: $use_cookies, secure_cookies: $secure_cookies, ",
  "remote_host: $remote_host, remote_user: $remote_user, ssl: $ssl, ",
  "server_port: $server_port, server_name: $server_name,  script_name: $script_name, ",
  "path_info: $path_info, path_translated: $path_translated, ",
  "newserver: $newserver\n" if $DEBUG;

print $header_text, h1({align=>"CENTER"},"Upload files to $server_name"),"\n";

# see if remote user's IP matches an accepted net
if (!&validate_addr) {
   &handle_error(4,"You are not allowed to use this form.",
     "This form cannot accept connections from your remote address: $remote_host.");
}

# suggest using a secure connection to the form
if (($secure_server) && ($server_port ne $secure_port)) {
   print p,'If you can, please use this ',
           a({HREF=>"https://$server_name:$secure_port$script_name"},
           'secure connection'),".\n";
}

# point user to messages after the form
if (@filenames) { # i.e. if form is being resubmitted with params
   print p,"See ",a({href=>'#messages'},"below"),
         " for results of upload attempts.\n";
} else { # first invocation of form
   print p,"See ",a({href=>'#messages'},"below")," for instructions.\n";
}

if (scalar(@servers) < 2) {
   print p,"Server: $server_name\n";
} else {
#simple form to change server
print start_form,
  p,"Server: ",
  popup_menu(-name=>'server',-values=>\@servers,-default=>$server_name),
  submit(-value=>'GO'),
  endform,"\n";
}

# multipart form for uploading files
print start_multipart_form(
            -name=>'UploadForm',
            -onSubmit=>"return scriptAction('$script_name')"),
  p,"Username: ",
  textfield(-name=>'username',-default=>$remote_user,-size=>'20',-maxlength=>'30'),"\n",
  br,"Password: ",
  password_field(-name=>'password',-size=>'20',-maxlength=>'30'),"\n",
  p,"Directory path: ",
  textfield(-name=>'path',-default=>$default_path,-size=>'50',-maxlength=>'80'),
    "e.g., /softwaredev/projects/\n",
  p,"File(s) to upload:\n";
  $filefields = 1 if $filefields < 1;
  for ($i=1; $i<=$filefields; $i++) {
      print br,filefield(-name=>'filename',-size=>'50'),"\n";
  }
  print p,reset,submit(-name=>'upload',-value=>'U P L O A D'),
  endform,"\n",hr(),a({NAME=>'messages'},''),"\n";

# process the multipart form if a file name was submitted
if ((@filenames) && ($filenames[0])) {
#resubmitted form with filenames
   $CGI::VERSION ge $required_cgi_pm_version 
       or &handle_error(4,"You need <A HREF=\"http://www-genome.wi.mit.edu/ftp/distribution/software/WWW/cgi_docs.html\">CGI.PM</A> version 2.39 or higher!");
   $user = param('username') or &handle_error(4,"Username missing!");
   ($login,$pwd,$gid,$uid) = getpwnam($user)
       or &handle_error(4,"$user is not a valid user.",
             "Do you have an account on $server_name\?");
   $password = param('password');
###   print p,"DEBUG: user: $user, password: $password, pwd: ($pwd)\n" if $DEBUG;
   &validate_password($user,$password,$pwd) 
       or &handle_error(4,"Incorrect password.");
   $path = param('path')
       or &handle_error(4,"Missing directory path.");
   $path .= '/' unless $path =~ /\/$/;
   $path_info 
       or &handle_error(4,"Bad directory path.",
          "Make sure that your browser is using JavaScript.");
   $destdir = path_translated();
   $destdir .= '/' unless $destdir =~ /\/$/;
   print p,"DEBUG: path_info: $path_info, path: $path, destdir: $destdir\n" if $DEBUG;
   (-d $destdir) or &handle_error(4,"No such directory: $server_name$path",
          "Physical path: $destdir",
          "Use your Web browser to verify the URL that you are trying to access.");
   # loop for saving uploaded files
   foreach $file (@filenames) {
      last if !$file;
      print p,"DEBUG: file: $file" if $DEBUG;
      $tmpfile = tmpFileName($file);
      $filename = uri_unescape("$file"); # should use CGI::unescape??
      print p,"DEBUG: file: $file<br>tmpfile: $tmpfile<br>filename: $filename\n" if $DEBUG;
      $mimetype = uploadInfo($file)->{'Content-Type'} or $mimetype = 'unknown';
      # just in case the browser send the entire remote path
      $filename =~ s/^.*\\//;  # delete DOS path
      $filename =~ s/^.*\///;  # delete Unix path
      $filename =~ s/^.*\://;  # delete Macintosh path
      print p,"DEBUG: file: $file<br>tmpfile: $tmpfile<br>filename: $filename\n" if $DEBUG;
      unless (&validate_filename($filename)) { print hr(); next; }
      &save_file;
      print hr();
      next;
   } #foreach
} elsif ((@filenames) && (!$filenames[0])) {
#submitted form with no filenames
   &handle_error(3,"No files to upload!","Use the Browse buttons to choose files to upload.");
   print hr();
} else {
#first invocation of form, print help message
   &print_help;
} #if
#print dump() if $DEBUG;
&print_footer;

####### subroutines #######

# return 1 if remote host address matches an accepted net, else return 0
# globals: $remote_host, @accept_nets
sub validate_addr {
  my($remote_addr,$net);
  return 1 if $remote_host eq 'localhost';
  $remote_addr = substr($remote_host,0,rindex($remote_host,'.')+1);
  if (@accept_nets) {
     foreach $net (@accept_nets) {
       $net = substr($net,0,rindex($net,'.')+1);
       return 1 if $remote_addr eq $net;
     }
     return 0;
  } else {
     return 1;
  } #if
}

# return 1 if submitted password matches user's real password, else return 0
sub validate_password {
  my($name,$password,$pwd) = @_;
  ($pwd ne crypt("\U$password","\U$name")) ? return 0 : return 1;
}

# save the uploaded file
# globals: $OS
sub save_file {
  if ($OS eq 'VMS') {
     &save_file_VMS;
  } elsif ($OS eq 'UNIX') {
     &save_file_UNIX;
  } else {
     &handle_error(4,"Can\'t handle your operating system: $OS");
  }
}


# globals: $tmpfile, $filename, $file, $destdir, $user, $password, 
#          $server_name, $path, $mimetype, $DEBUG
sub save_file_VMS {
  local($filetype,$outputfile,$modtime,$stats,$errno,$msg);
  my($tmpfile) = &translate_path($tmpfile);
  close $file;
  $filetype = 'ascii';
  if (-B $tmpfile) {  # binary file
     $filetype = 'binary';
     if (system("SET FILE/ATTR=(RFM:FIX,LRL:512,MRS:512,RAT:NONE) $tmpfile")) {
        &handle_error(2,"Couldn\'t change file attributes to binary.");
     }
  }
  $outputfile = &translate_path("$destdir$filename");
  print p,"DEBUG: tmpfile: $tmpfile, ",(-s $tmpfile),' bytes' if $DEBUG;
  print p,"DEBUG: copy $tmpfile 0\"$user password\"::$outputfile\n" if $DEBUG;
  if (File::Copy::syscopy($tmpfile, "0\"$user $password\"::$outputfile", 0)) {
     #copy succeeded
     $stats = stat($outputfile);
     $modtime = &format_date($stats->mtime); # use modification time of current version
     my($size) = $stats->size;
     print p,"File saved: $filename",br,"Verify the uploaded file\'s name, type, MIME-type, date-time, and size:\n",br,
       "$server_name$path$filename, $filetype, $mimetype, $modtime, $size bytes\n";
     &log_upload(1);
  } else {
     #copy failed
     print p,"DEBUG: error code: ", int $!, ", error msg: $!\n" if $DEBUG;
     &handle_error(3,"File not saved: $filename","$!");
     &log_upload(0,$msg);
     unlink $tmpfile;
  } #if
} #save_file_VMS


# globals: $tmpfile, $filename, $file, $destdir, $user, $password, $gid, $uid,
#          $server_name, $path, $mimetype, $DEBUG
sub save_file_UNIX {
  local($filetype,$outputfile,$modtime,$stats,$destdir_gid,$destdir_uid);
  my($bytesread,$buffer);
  $outputfile = &translate_path("$destdir$filename");
  print p,"DEBUG: tmpfile: $tmpfile, ",(-s $tmpfile),' bytes' if $DEBUG;
  my($destdir_stats) = stat($destdir);
  $destdir_uid = $destdir_stats->uid;
  $destdir_gid = $destdir_stats->gid;
  (($uid eq $destdir_uid) && ($gid eq $destdir_gid))
      or &handle_error(4,"You do not have write access to $destdir");
  if (File::Copy::copy($tmpfile, $outputfile)) {
     #copy succeeded
     $stats = stat($outputfile);
     $modtime = &format_date($stats->mtime);
     my($size) = $stats->size;
     print p,"File saved: $filename",br,"Verify the uploaded file\'s name, type, MIME-type, date-time, and size:\n",br,
       "$server_name$path$filename, $filetype, $mimetype, $modtime, $size bytes\n";
     &log_upload(1);
  } else {
     #copy failed
     print p,"DEBUG: error code: ", int $!, ", error msg: $!\n" if $DEBUG;
     &handle_error(3,"File not saved: $filename","$!");
     &log_upload(0,$msg);
     unlink $tmpfile;
  } #if
  close $file;
  #save succeeded
}

# severity: 1 = informational message
#           2 = warning or alert
#           3 = nonfatal error
#           4 = fatal error (abort rest of script)
#globals: @filenames
sub handle_error {
  my($severity,$message,@help) = @_;
  if ($severity eq 1) {
     &print_message('',$message,@help);
  } elsif ($severity eq 2) {
     &print_message('Warning',$message,@help);
  } elsif ($severity eq 3) {
     &print_message('Error',$message,@help);
  } elsif ($severity eq 4) {
     &print_message('Error',$message,@help);
     print hr();
     &print_footer;
     &log_upload(0,$message);
     foreach $file (@filenames) { close $file if $file; }
     exit 0;
  } else {
     # shouldn't happen
  } #if
}

sub print_message {
  my($banner,$message,@help) = @_;
  my($line);
  print h3("$banner") if $banner;
  print p,strong("$message\n") if $message;
  foreach $line (@help) { print  br,"$line\n"; }
}

# globals: $help_message
sub print_help {
  print $help_message,p,hr() if $help_message;
}

# globals: $footer_text, $email, $CGI::VERSION
sub print_footer {
  print p,"$footer_text\n" if $footer_text;
  print p,'Questions? Problems? Contact ',
    a({HREF=>"mailto:$email"},"$email\n") if $email;
  print p,'This script uses ',
    a({HREF=>'http://www-genome.wi.mit.edu/ftp/distribution/software/WWW/cgi_docs.html'},'CGI.PM')," Version $CGI::VERSION\n",
  end_html;
}

# globals: $logfile, $user, $remote_host, $server_name, $path, $filename,
#          $filetype, $mimetype,
# locals from calling subr: $modtime, $stats
sub log_upload {
  return 0 unless $logfile;
  my($status,$message) = @_;
  my $ctime = &format_date(time);
  open(LOG,">>$logfile") or return 0;
  if ($status) { #upload succeeded
     my($size) = $stats->size;
     print LOG "[$ctime] $user ($remote_host): $server_name$path$filename, $filetype, $mimetype, $modtime, $size bytes\n";
  } else { #upload failed
     print LOG "[$ctime] $user ($remote_host): $server_name$path$filename, $message\n";
  }
  close LOG;
#  print p,"Transaction recorded in log file\n"
  return 1;
}

# globals: $OS
sub translate_path {
  if ($OS eq 'VMS') {
     return vmsify($_[0]);
  } elsif ($OS eq 'UNIX') {
     return @_;
  } else {
     return @_;
  }
}

# uses Date::Format
sub format_date {
  my $time = shift;
  return time2str("%a %b %e %T %Y", $time);
}

# globals: $OS
sub validate_filename {
  if ($OS eq 'VMS') {
     &validate_filename_VMS;
  } elsif ($OS eq 'UNIX') {
     return 1;
  } else {
     return 1;
  }
}

#globals: $file
sub validate_filename_VMS {
   my($name) = $_[0];
   my(@x,$x,$msg);
   @x = ($name =~ /\./g);
   $x = scalar @x;
   if ( ($name=~/^[\w\.\-\_]+$/) && ($x <= 1) ) {
      return 1;
   } else {
      $msg = "File name syntax specification error: $name";
      &handle_error(3,"File not saved!",
        $msg,
        "Only alphanumerics, hyphen and underscore are allowed.",
        "Only one dot is allowed.");
      &log_upload(0,$msg);
      close $file;
      return 0;
   } #if
}

## JavaScript ##
sub define_javascript {
$JSCRIPT=<<EOF;
var browser=navigator.appName;
var version=parseInt(navigator.appVersion);
function checkBrowser() {
//alert(browser + ' ' + version);
   if (browser == "Microsoft Internet Explorer" && version < 4) {
      alert('You must have version 4+ of Internet Exlorer to use this form.')
      return false;
   }
   if (browser == "Netscape" && version < 2) {
      alert('You must have version 2+ of Netscape to use this form.')
      return false;
   }
}

function scriptAction(script) {
   var path = document.UploadForm.path.value;
   checkBrowser();
   if (browser == "Netscape" && version < 4) {
      var valid = true;
   } else {
      var valid = validateForm();
   }
   if (valid) {
   document.UploadForm.action = script + document.UploadForm.path.value;
//alert(document.UploadForm.action);
   return true;
   } else {
     return false;
   }
}

function validateForm() {
   var username = document.UploadForm.username.value;
   var password = document.UploadForm.password.value;
   var path = document.UploadForm.path.value;
//   var filename = document.UploadForm.filename.value;
   if (username == '') {
      alert("Missing username.");
      document.UploadForm.username.focus();
      document.UploadForm.username.select();
      return false;
   } 
   if (password == '') {
      alert("Missing password.");
      document.UploadForm.password.focus();
      document.UploadForm.password.select();
      return false;
   }
   if (path == '') {
      alert("Missing URL directory.");
      document.UploadForm.path.focus();
      document.UploadForm.path.select();
      return false;
   } else {
     if (path.indexOf(unescape("%5C")) >= 0) {
        alert("Backslash not allowed in URL directory. Use / as separator.");
        document.UploadForm.path.focus();
        document.UploadForm.path.select();
        return false;
     }
     if (path.indexOf(' ') >= 0) {
        alert("Spaces not allowed in URL directory.");
        document.UploadForm.path.focus();
        document.UploadForm.path.select();
        return false;
     }
     var c = path.charAt(0);
     if (c != '/') {
        alert("URL directory must begin with a /.");
        document.UploadForm.path.focus();
        document.UploadForm.path.select();
        return false;
     }
   }
//can't do this because filename fields have same name
//   if (filename.length == 0) {
//      alert("Missing upload file.");
//      return false;
//   }
   return true;
}
EOF
   ;
} #define_javascript
