/*************************************************************************
 *                                                                       *
 * DS_STRIP - turns a DSREPAIR.DIB file back into its original           *
 *            components. Works with Netware 4.x and 5.x.                *
 *            Simple Nomad [thegnome@nmrc.org] http://www.nmrc.org/      *
 *                                                                       *
 * 18Mar99 -                                                             *
 * Tested on Linux, AIX, and NT.                                         *
 * Linux compile:   gcc -O2 -o ds_strip ds_strip.c                       *
 *    NT (DJGPP):   gcc -O2 -o ds_strip.exe ds_strip.c                   *
 *   AIX compile:   cc -O2 -DENDIAN -o ds_strip ds_strip.c               *
 *                                                                       *
 * 21May99 -                                                             *
 * Added the -p switch and updated the help routine.                     *
 * Fixed obscure bug if last file in dsrepair.dib is less than 4 bytes.  *
 *                                                                       *
 *************************************************************************/

#include <stdio.h>
#include <stdlib.h>

typedef unsigned char uint8;
typedef unsigned long uint32;
typedef unsigned int byte;

typedef struct just4bytes {
	uint32 offset;
} OFFSET;

union
{
  uint32 longData;
  uint8 shortData[4];
} output;

union
{
  uint32 longData;
  uint8 shortData[4];
} input;

typedef struct just20bytes {
  uint8 filename[16];
  uint32 length;
} NW4REC;

typedef struct just2words {
  uint32 filenum;
  uint32 length;
} NW5REC;

/*
 * Global variables
 */
FILE *fBackup;
FILE *fTemp;

#define FALSE 0
#define TRUE 1

/*
 * Used for flipping byte orders in an unsigned long. Handy for
 * endian issues, and used to read certain offsets properly.
 */
uint32 make_conversion(uint32 k) {
  int i;
  uint32 j;

  input.longData=k;
  for (i=0; i<4; i++)
    output.shortData[i]=input.shortData[3-i];
  return(output.longData);
}

/*
 * Checks to see if it's a Netware 5.x DSREPAIR.DIB file. Return TRUE
 * if it is, otherwise it's Netware 4.x.
 */
int isNW5(void)
{
  uint32 k;
  fseek(fBackup,sizeof(k)*3,SEEK_SET);
  fread(&k,sizeof(k),1,fBackup);
#ifndef ENDIAN
  k=make_conversion(k); /* Reverse the bytes if Intel */
#endif
  if (k==0x454e5452) return (FALSE); /* 45 4e 54 52 == E N T R, so NW4 */
  return (TRUE); /* NW5*/
}

/*
 * Rebuilds the files from Netware 4.x DSREPAIR.DIB.
 */
void makeNW4file(int verbose,int justNDS)
{
  uint32 l,end;
  NW4REC nw4rec;
  int i,j=1;

  /* seek to end to get length of file */
  fseek(fBackup,0,SEEK_END);
  end=ftell(fBackup);

  /* back to beginning to start stripping */
  fseek(fBackup,sizeof(l)*3,SEEK_SET);
  while(!(feof(fBackup)))
  {
    fread(&nw4rec,sizeof(nw4rec),1,fBackup);
    fTemp=fopen(nw4rec.filename,"wb");
    if(fTemp==NULL)
    {
      printf("Unable to open %s\n",nw4rec.filename);
      exit(-1);
    }
#ifdef ENDIAN
    nw4rec.length=make_conversion(nw4rec.length);
#endif
    if (verbose==TRUE) printf("%d: Filename - %s  Length - %08x Location %08x\n",j,nw4rec.filename,nw4rec.length,ftell(fBackup));
    for(l=0;l<nw4rec.length;l++) fputc(fgetc(fBackup),fTemp);
    fclose(fTemp);

    /* The next line helps check for a condition with a 1 byte file
       located at the end of the dsrepair.dib file. This prevents
       an infinite loop due to some funky pointer condition during
       the strip where we never reach eof. Not pretty but it fixes it. */
    if(end-(ftell(fBackup))<sizeof(nw4rec)) break;
    j++;
    if(justNDS==TRUE) if(j==5) break; /* The NDS files are the first four... */
  }
}

/*
 * Rebuilds the files from Netware 5.x DSREPAIR.DIB.
 */
void makeNW5file(int filetype, int verbose)
{
  uint32 i,k;
  NW5REC nw5rec;
  char filename[12];

  switch(filetype)
  {
    case 1:
      strcpy(filename,"0.dsd");
      break;
    case 2:
      strcpy(filename,"1.dsd");
      break;
    case 3:
      strcpy(filename,"2.dsd");
      break;
    case 4:
      strcpy(filename,"3.dsd");
      break;
    default:
      strcpy(filename,"0.dsb");
      break;
  }

  fTemp=fopen(filename,"wb");
  if(fTemp==NULL)
  {
    printf("Unable to open %s\n",filename);
    exit(-1);
  }
  fseek(fBackup,(sizeof(k)*filetype)+28,SEEK_SET);
  fread(&k,sizeof(k),1,fBackup);
#ifdef ENDIAN
  k=make_conversion(k);
#endif
  fseek(fBackup,k,SEEK_SET);
  fread(&nw5rec,sizeof(nw5rec),1,fBackup);
#ifdef ENDIAN
  nw5rec.length=make_conversion(nw5rec.length);
#endif
  if (verbose==TRUE) printf("Filename - %s  Length - %08x Location - %08x\n",filename,nw5rec.length,ftell(fBackup));
  for(i=0;i<nw5rec.length;i++) fputc(fgetc(fBackup),fTemp);
  fclose(fTemp);
}

/*
 * Help/usage.
 */
void printHelp(int i)
{
  printf("USAGE: ds_strip <options>\n\n");
  printf("  -n                  Just NDS files (Netware 4.x only)\n");
  printf("  -h                  This HELP screen\n");
  printf("  -v                  Verbose screen output\n");
  printf("  -p <path\\filename>  Use alternate filename or path\n");
  printf("                       (default is DSREPAIR.DIB in\n");
  printf("                       current directory)\n");
  printf("\nWill autoconvert Netware 4.x and Netware 5.x files.\n\n");
  exit(i);
}

/*
 * Main program....
 */
int main(int argc, char *argv[])
{
  int i;
  int verbose=FALSE;
  int justNDS=FALSE;
  int pathset=FALSE;
  char *pathtofile;

/* Say hello... */
  printf("DS_STRIP - extracts NDS and other files from DSREPAIR.DIB\n");
  printf("Comments/bugs: pandora@nmrc.org\n");
  printf("http://www.nmrc.org/pandora\n");
  printf("1999 (c) Nomad Mobile Research Centre\n\n");

  /* process the command line */
  for (i=1 ; i<argc ; i++)
  {
    if (argv[i][0]=='-')
      switch(argv[i][1])
      {
        case 'p':
        case 'P':
          if ((argv[i+1][0]=='-') || (i+1==argc))
          {
            printf("Error with path and filename for DSREPAIR file.\n\n");
            printHelp(-1);
            break;
          }
          strcat(pathtofile,argv[i+1]);
          pathset=TRUE;
          break;
        case 'h':
        case 'H':
        case '?':
          printHelp(0);
          break;
        case 'v':
        case 'V':
          verbose=TRUE;
          break;
        case 'n':
        case 'N':
          justNDS=TRUE;
          break;
        default:
         break;
      }
  }

  if (pathset==FALSE) strcat(pathtofile,"dsrepair.dib");

  fBackup=fopen(pathtofile,"rb");
  if (fBackup==NULL)
  {
    printf("Unable to open %s.\n",pathtofile);
    exit(1);
  }

  printf("Opening %s repair file....\n",pathtofile);

  i=isNW5();
  if(i==FALSE)
  {
    printf("Netware 4.x DSREPAIR.DIB detected....\n");
    makeNW4file(verbose,justNDS);
  }
  else
  {
    printf("Netware 5.x DSREPAIR.DIB detected....\n");
    for(i=0;i<5;i++) makeNW5file(i,verbose);
  }
  fclose(fBackup);
  printf("Stripping complete.\n\n");
}

