//----------------------------------------------------------------------
//
// NewSID
//
// Copyright (c) 1997, 1998 Mark Russinovich and Bryce Cogswell
//
// Changes the computer SID. 
//
// This code is protected under copyright law. You do not have 
// permission to use this code in a commercial SID-changing product.
//
//----------------------------------------------------------------------
#define UNICODE 1
#include <windows.h>
#include <lm.h>
#include <commctrl.h>
#include <stdio.h>
#include <tchar.h>
#include <process.h>
#include "resource.h"  

//
// Defines for security buffer sizes
//
#define SZ_REL_SD_BUF		8192
#define SZ_ABS_SD_BUF		8192

//
// Max length of path name displayed in
// progress dialogs
//
#define MAX_DLG_PATHLEN		55

//
// ProductType name - this isn't exact, but its
// good enough
//
#define PRODUCTTYPEPATH		L"System\\C"
#define PRODUCTTYPENAME		L"ProductOptions"

//
// Profile list path
//
#define PROFILELISTPATH L"Software\\Microsoft\\Windows NT\\CurrentVersion\\ProfileList"

//
// General buffer for storing temporary strings
//
static TCHAR		msgbuf[ 1024 ];

//
// Are we synchronizing with another computer?
//
BOOLEAN					SyncSID = FALSE;
TCHAR					SyncComputerName[MAX_COMPUTERNAME_LENGTH+1];
   
//
// The binary representations of the old and the
// new subauthority fields of the computer SID
//
const DWORD				SubAuthLength = 5 * sizeof(DWORD);
CHAR					NewSubAuth[ 5 * sizeof(DWORD) ];
CHAR					OldSubAuth[ 5 * sizeof(DWORD) ];

//
// Application instance handle
//
HINSTANCE			hInst;

//
// Values the main dialog can return, which indicate
// whether to proceed or not.
//
enum {
	DIALOG_OK,
	DIALOG_CANCEL,
};

//
// Forwards
//
LONG APIENTRY MainDialog( HWND hDlg, UINT message, UINT wParam, LONG lParam );


//===================================================================
//
// SECONDARY DIALOG PROCEDURES
//
//===================================================================


//----------------------------------------------------------------------
//
// DisplayError
//
// Formats an error message for the last error and pops a message box.
//
//----------------------------------------------------------------------
void DisplayError( TCHAR *ErrorTitle )
{
	TCHAR	*errMsg;
	TCHAR	message[256];
	
	FormatMessage( FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,
			NULL, GetLastError(), 
			MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), 
			(LPTSTR) &errMsg, 0, NULL );
	swprintf( message, L"%s:\n\n%s", ErrorTitle, errMsg );

	MessageBox( NULL, message, L"NewSID",
				MB_ICONERROR|MB_OK );
	
	LocalFree( errMsg );
}


//----------------------------------------------------------------------
//
// UpdateProgressDialog
//
// Make the pathname fit into the dialog box
//
//----------------------------------------------------------------------
void UpdateProgressDialog( HWND hDlg, DWORD Control, PTCHAR PathName )
{
	TCHAR		shortName[MAX_DLG_PATHLEN];
	DWORD		i, j, k;

	if( (i = wcslen( PathName )) > MAX_DLG_PATHLEN ) {

		j = MAX_DLG_PATHLEN-2;
		shortName[MAX_DLG_PATHLEN-1] = 0;
		while( j > (MAX_DLG_PATHLEN/2+3) ) {

			shortName[j] = PathName[i];
			j--;
			i--;
		}

		for( k = 0; k < MAX_DLG_PATHLEN/2; k++ ) {

		   shortName[k] = PathName[k];
		}

		while( k <= j ) shortName[k++] = L'.';


		SetDlgItemText( hDlg, Control, shortName );

	} else {

		SetDlgItemText( hDlg, Control, PathName );
	}
}


//----------------------------------------------------------------------
//  
// UpdateDlgProc
//
// Dialog box shown while file systems or the Registry is
//  being scanned for SID update.
//
//----------------------------------------------------------------------
BOOL CALLBACK UpdateDlgProc (HWND hDlg, UINT message, UINT wParam,
                       		LONG lParam) {
	switch (message) {
	case WM_INITDIALOG :

		return TRUE ;
 	}
	return FALSE ;
}


//===================================================================
//
// COMPUTER NAME CHANGING FUNCTIONS
//
//===================================================================

//----------------------------------------------------------------------
//
// SetCurrentPrivilege
//
// Used to set the privilege necessary for adding computers to domains.
//
//----------------------------------------------------------------------
BOOL SetCurrentPrivilege( LPCTSTR Privilege,  BOOL bEnablePrivilege )
{
    HANDLE hToken;
    TOKEN_PRIVILEGES tp;
    LUID luid;
    TOKEN_PRIVILEGES tpPrevious;
    DWORD cbPrevious=sizeof(TOKEN_PRIVILEGES);
    BOOL bSuccess=FALSE;

    if(!LookupPrivilegeValue(NULL, Privilege, &luid)) return FALSE;

    if(!OpenProcessToken(
            GetCurrentProcess(),
            TOKEN_QUERY | TOKEN_ADJUST_PRIVILEGES,
            &hToken
            )) return FALSE;

    //
    // first pass.  get current privilege setting
    //
    tp.PrivilegeCount           = 1;
    tp.Privileges[0].Luid       = luid;
    tp.Privileges[0].Attributes = 0;

    AdjustTokenPrivileges(
            hToken,
            FALSE,
            &tp,
            sizeof(TOKEN_PRIVILEGES),
            &tpPrevious,
            &cbPrevious
            );

    if(GetLastError() == ERROR_SUCCESS) {
        //
        // second pass.  set privilege based on previous setting
        //
        tpPrevious.PrivilegeCount     = 1;
        tpPrevious.Privileges[0].Luid = luid;

        if(bEnablePrivilege) {
            tpPrevious.Privileges[0].Attributes |= (SE_PRIVILEGE_ENABLED);
        }
        else {
            tpPrevious.Privileges[0].Attributes ^= (SE_PRIVILEGE_ENABLED &
                tpPrevious.Privileges[0].Attributes);
        }

        AdjustTokenPrivileges(
                hToken,
                FALSE,
                &tpPrevious,
                cbPrevious,
                NULL,
                NULL
                );

        if(GetLastError() == ERROR_SUCCESS) bSuccess=TRUE;
    }

    CloseHandle(hToken);

    return bSuccess;
}


//----------------------------------------------------------------------
//  
// NameDlgProc
//
// Pops up the standard About dialog box.
//
//----------------------------------------------------------------------
BOOL CALLBACK NameDlgProc (HWND hDlg, UINT message, UINT wParam,
                       		LONG lParam) {

	TCHAR	computerName[MAX_COMPUTERNAME_LENGTH+2];
	DWORD	len;

	switch (message) {
	case WM_INITDIALOG :

		//
		// Initialize name information
		//
		len = MAX_COMPUTERNAME_LENGTH+2;
		GetComputerName( computerName, &len );
		SetDlgItemText( hDlg, IDC_NAMESELECT, computerName ); 
		return TRUE ;
 
	case WM_COMMAND :
		switch (wParam) {

		case IDOK :

			GetDlgItemText( hDlg, IDC_NAMESELECT, computerName, MAX_COMPUTERNAME_LENGTH+2 );
			if( !SetComputerName( computerName ) ) {

				DisplayError( L"Error setting new computer name");
				return TRUE;
			} 

			MessageBox( hDlg, L"New computer name applied.", 
				L"NewSID", MB_ICONINFORMATION|MB_OK );			
			EndDialog (hDlg, 0) ;
			return TRUE ;

		case IDCANCEL:
			EndDialog( hDlg, 0 );
			return TRUE;
		}
		break; 

	case WM_CLOSE:	
		EndDialog (hDlg, 0);
		return TRUE;
	}
	return FALSE ;
}


//----------------------------------------------------------------------
//  
// SidChoiceDlgProc
//
// Determines if the SID should be random or retrieved from another
// computer.
//
//----------------------------------------------------------------------
BOOL CALLBACK SidChoiceDlgProc (HWND hDlg, UINT message, UINT wParam,
                       		LONG lParam) {

	HKEY syncHandle;

	switch (message) {
	case WM_INITDIALOG :

		//
		// Initialize name information
		//
		SetDlgItemText( hDlg, IDC_DCNAME, L"" ); 
		return TRUE ;
 
	case WM_COMMAND :
		switch (wParam) {

		case IDOK :

			GetDlgItemText( hDlg, IDC_DCNAME, SyncComputerName, MAX_COMPUTERNAME_LENGTH+2 );
			if( wcslen( SyncComputerName )) {
				//
				// Open the registry of the target computer
				//
				if( RegConnectRegistry( SyncComputerName, HKEY_LOCAL_MACHINE, &syncHandle ) ) {

					MessageBox( NULL, L"NewSID was unable to connect to the specified computer's Registry.",
									L"NewSID", MB_ICONERROR|MB_OK );
					return TRUE;
				}	
				RegCloseKey( syncHandle );
				SyncSID = TRUE;	
				EndDialog (hDlg, 0) ;
	
			} else {

				MessageBox( NULL, L"No computer name was specified.",
								L"NewSID", MB_ICONWARNING|MB_OK );	
			}
			return TRUE ;

		case IDCANCEL:
			EndDialog( hDlg, DIALOG_CANCEL );
			return TRUE;
		}
		break; 

	case WM_CLOSE:	
		EndDialog (hDlg, DIALOG_CANCEL);
		return TRUE;
	}
	return FALSE ;
}
//===================================================================
//
// RANDOM NUMBER GENERATION FOR MAKING A NEW SID
//
//===================================================================

//----------------------------------------------------------------------
//
// Random
//
// 32-bit pseudo-random numbers
//
//----------------------------------------------------------------------
volatile BOOL RandomExpired;

void __cdecl RandomTrigger( void * arg )
{
	// Wait until main thread is ready to go
	while ( RandomExpired )
		Sleep( 1 );
	// Now sleep while main thread counts
	Sleep( 1 );
	// Notify main thread that time is up
	RandomExpired = TRUE;
	// Exit
	_endthread();
}

DWORD RandomBits( void )
{
	register DWORD cnt = 0;

	// Tell trigger we're not ready yet
	RandomExpired = TRUE;
	// Start trigger
	_beginthread( RandomTrigger, 0, NULL );
	// Okay, tell it to start sleeping
	RandomExpired = FALSE;
	// Count as fast as we can
	while ( !RandomExpired )  {
		++cnt;
	}
	// Only low bits are accurate!
	return cnt;
}


DWORD Random( void )
{
	return	RandomBits() << 28	^
			RandomBits() << 24	^
			RandomBits() << 20	^
			RandomBits() << 16	^
			RandomBits() << 12	^
			RandomBits() << 8	^
			RandomBits() << 4	^
			RandomBits() << 0;
}


//===================================================================
//
// REGISTRY SECURITY CONVENIENCE ROUTNES 
//
//===================================================================


//----------------------------------------------------------------------
//
// GetRegAccess
//
// Returns the original security descriptor of a key.
//
//----------------------------------------------------------------------
PSECURITY_DESCRIPTOR GetRegAccess (HKEY hKey)
{
	DWORD		nb = 0;
	PSECURITY_DESCRIPTOR SecDesc;
	//
	// Get access
	//
	if (RegGetKeySecurity (hKey, DACL_SECURITY_INFORMATION, NULL, &nb) != ERROR_INSUFFICIENT_BUFFER)
		return NULL;
	SecDesc = (PSECURITY_DESCRIPTOR) malloc (nb);
	if (RegGetKeySecurity (hKey, DACL_SECURITY_INFORMATION, SecDesc, &nb) != ERROR_SUCCESS)
		free (SecDesc);
	return (SecDesc);
}


//----------------------------------------------------------------------
//
// SetRegAccess
//
// Sets the key with the specified security descriptor. If the key's
// name is passed (Rooot != NULL), then the key is re-opened to give
// the caller updated access rights that reflect the changed security.
//
//----------------------------------------------------------------------
LONG SetRegAccess (HKEY hKey, LPCTSTR lpSubKey,
					PSECURITY_DESCRIPTOR SecDesc, PHKEY phKey)
{
	//
	// Grant requested access
	//
	if (RegSetKeySecurity (*phKey, DACL_SECURITY_INFORMATION, SecDesc) != ERROR_SUCCESS)
		return FALSE;

	//
	// Re-open the key if requested
	//
	if (! hKey)
		return TRUE;

	RegCloseKey (*phKey);
	return (RegOpenKey (hKey, lpSubKey, phKey) == ERROR_SUCCESS);
}


//----------------------------------------------------------------------
//
// GetRegSecDesc
//
// Gets the security descriptor from the specified key.
//
//----------------------------------------------------------------------
PSECURITY_DESCRIPTOR GetRegSecDesc (HKEY Root, TCHAR *Path, SECURITY_INFORMATION Information)
{
	HKEY					hKey;
	LONG					Status;
	DWORD					nb = 0;
	PSECURITY_DESCRIPTOR	SecDesc;

	//
	// Open the key with no access requests, since we don't need
	// any.
	// SECURITY_DESCRIPTOR
	if (RegOpenKeyEx (Root, Path, 0, KEY_READ, &hKey) != ERROR_SUCCESS)
		return NULL;

	//
	// Grab a copy of the security for key
	//
	if (RegGetKeySecurity (hKey, Information, NULL, &nb) != ERROR_INSUFFICIENT_BUFFER)
		return NULL;
	SecDesc = malloc (nb);
	Status = RegGetKeySecurity (hKey, Information, SecDesc, &nb);

	//
	// Close the key anyway
	//
	RegCloseKey (hKey);

	if ((Status) != ERROR_SUCCESS)
		free (SecDesc);
	return SecDesc;
}


//----------------------------------------------------------------------
//
// SecurityReplaceSID
//
// This convenience function scans the passed in SID for an instance
// of the old computer security SID, and if it finds it, it updates
// the instance with the new computer SID.
//
//----------------------------------------------------------------------
BOOLEAN SecurityReplaceSID( PSID pSid )
{
	BOOLEAN		foundSid;
	DWORD		numSubAuths;
	PDWORD		firstSubAuth;
	DWORD		i;

	foundSid = FALSE;
	numSubAuths = *GetSidSubAuthorityCount(pSid);
	if( numSubAuths >= SubAuthLength/sizeof(DWORD) ) {

		firstSubAuth = GetSidSubAuthority(pSid, 0); 
		for( i = 0; i < numSubAuths - SubAuthLength/sizeof(DWORD) + 1; i++ ) {

			if( !memcmp( &firstSubAuth[i], &OldSubAuth[sizeof(DWORD)], 
							SubAuthLength-sizeof(DWORD) )) {

				//
				// Found it - change it and continue
				//
				memcpy( &firstSubAuth[i], &NewSubAuth[sizeof(DWORD)], 
								SubAuthLength-sizeof(DWORD) );
				foundSid = TRUE;
				break;
			}
		}
	}
	return foundSid;
}

//===================================================================
//
// REGISTRY SECURITY DESCRIPTOR UPDATING 
//
//===================================================================

//----------------------------------------------------------------------
//
// CheckKeyOwnershipForSID
//
// Sees if the file ownership SID contains the old computer SID, and
// if so, replaces it.
//
//----------------------------------------------------------------------
void CheckKeyOwnershipForSID(HKEY hKey, PSECURITY_DESCRIPTOR psdRegSD )
{
	PSID		psidRegOwnerSID;
	PSID		psidRegGroupSID;
	BOOL		bOwnerDefaulted;

	//
	// Get the owner SID
	//
	if (!GetSecurityDescriptorOwner(psdRegSD,
		(PSID *)&psidRegOwnerSID, (LPBOOL)&bOwnerDefaulted) ||
		!psidRegOwnerSID) { 

		return;
	}

	//
	// If the old SID is in the owner, we have to write
	// the updated owner to the key
	//
    if( SecurityReplaceSID( psidRegOwnerSID ) ) {
		if (!RegSetKeySecurity(hKey,
				(SECURITY_INFORMATION)OWNER_SECURITY_INFORMATION,
				psdRegSD)){
			
			return;
		}
    }

	//
	// Now get the Group SID and do the same thing
	//
	if (!GetSecurityDescriptorGroup(psdRegSD,
		(PSID *)&psidRegGroupSID, (LPBOOL)&bOwnerDefaulted) ||
		!psidRegOwnerSID) { 

		return;
	}

	//
	// If the old SID is in the owner, we have to write
	// the updated owner to the key
	//
    if( SecurityReplaceSID( psidRegGroupSID ) ) {
		if (!RegSetKeySecurity(hKey,
				(SECURITY_INFORMATION)GROUP_SECURITY_INFORMATION,
				psdRegSD)){
			
			return;
		}
    }
	return;
}


//----------------------------------------------------------------------
//
// CheckKeyACLForSID
//
// Scan's the security descriptor's ACEs, looking for instances
// of the old computer SID.
//
//----------------------------------------------------------------------
void CheckKeyACLForSID( HKEY hKey, BOOLEAN Dacl, 
					   PSECURITY_DESCRIPTOR psdRegSD )
{
	PACL       paclReg;
	BOOL       bHasACL;
	BOOL       bOwnerDefaulted;
	DWORD      dwAcl_i;
	BOOLEAN    descriptorModified;
	DWORD      dwLastError   = NO_ERROR;
	ACL_SIZE_INFORMATION asiAclSize;
	DWORD      dwBufLength = sizeof(asiAclSize);
	ACCESS_ALLOWED_ACE   *paaAllowedAce;

	if( Dacl ) {
		if (!GetSecurityDescriptorDacl(psdRegSD,
							 (LPBOOL)&bHasACL, (PACL *)&paclReg,
							 (LPBOOL)&bOwnerDefaulted)) {
			return;
		}
	} else {
		if (!GetSecurityDescriptorSacl(psdRegSD,
							 (LPBOOL)&bHasACL, (PACL *)&paclReg,
							 (LPBOOL)&bOwnerDefaulted)) {
			return;
		}
	}

	//
	// If no ACL to process, so OK, return
	//
	if (!bHasACL || !paclReg )
		return;

	if (!GetAclInformation(paclReg, (LPVOID)&asiAclSize,
				 (DWORD)dwBufLength, (ACL_INFORMATION_CLASS)AclSizeInformation)){
		return;
	}

	// 
	// Look through the ACEs
	//
	descriptorModified = FALSE;
	for (dwAcl_i = asiAclSize.AceCount-1;  ((int)dwAcl_i) >= 0;  dwAcl_i--) {

		//
		// If we can't get an ACE, bail
		//
		if (!GetAce(paclReg, dwAcl_i, (LPVOID *)&paaAllowedAce)) {

			return;
		}

		//
		// Make sure we're dealing with an ACE we know
		//
		if (!( (paaAllowedAce->Header.AceType == ACCESS_ALLOWED_ACE_TYPE)
				||(paaAllowedAce->Header.AceType == ACCESS_DENIED_ACE_TYPE )
				||(paaAllowedAce->Header.AceType == SYSTEM_AUDIT_ACE_TYPE  )
				||(paaAllowedAce->Header.AceType == SYSTEM_ALARM_ACE_TYPE  ))) {
			continue;
		}

		//
		// Look at the SID's subauthorities to see if there's a match
		// with the old computer SID
		//
		descriptorModified |= SecurityReplaceSID((PSID)&(paaAllowedAce->SidStart)); 
	}

	//
	// If the security descriptor was modified because an
	// old computer SID was converted to the new one, write
	// the new descriptor to disk.
	//
	if( descriptorModified ) {

		//
		// Modify the SD on the hard disk
		//
		if( Dacl ) {

			//
			// Modify the SD in virtual memory
			//
			if (!SetSecurityDescriptorDacl(psdRegSD,
								TRUE, paclReg, FALSE)) {

				return;
			}

			if (!RegSetKeySecurity(hKey,
							(SECURITY_INFORMATION)DACL_SECURITY_INFORMATION,
							psdRegSD)){

				return;
			}
		} else {

			//
			// Modify the SD in virtual memory
			//
			if (!SetSecurityDescriptorSacl(psdRegSD,
								TRUE, paclReg, FALSE)) {

				return;
			}

			if (!RegSetKeySecurity(hKey,
							(SECURITY_INFORMATION)SACL_SECURITY_INFORMATION,
							psdRegSD)){

				return;
			}
		}
	}
}


//----------------------------------------------------------------------
//
// CheckKeySID
//
// Reads the key's security descriptor and converts it to absolute
// format. Then it calls a function to check the ownership SIDs and
// finally one to check the DACL SIDs, to see if they need to be
// updated.
//
//----------------------------------------------------------------------
BOOL CheckKeySID( HKEY hKey, HKEY Root, LPTSTR lpszFullName,
				 PSECURITY_DESCRIPTOR psdSrelRegSD )
{
	UCHAR       ucBufAbs[SZ_ABS_SD_BUF];
	UCHAR       ucBufCtrl[sizeof(PSECURITY_DESCRIPTOR_CONTROL)];
	DWORD       dwSDLength = SZ_REL_SD_BUF;
	PSECURITY_DESCRIPTOR psdAbsRegSD = (PSECURITY_DESCRIPTOR)&ucBufAbs;
	PSECURITY_DESCRIPTOR_CONTROL psdcCtrl = (PSECURITY_DESCRIPTOR_CONTROL)&ucBufCtrl;
	PACL        paclDacl;
	PACL        paclSacl;
	PSID        psidSidOwn;
	PSID        psidSidPG;
	BOOL        bDaclPresent;
	BOOL        bDaclDefaulted;
	BOOL        bSaclPresent;
	BOOL        bSaclDefaulted;
	BOOL        bOwnerDefaulted;
	BOOL        bGroupDefaulted;
	BOOL        bSDSelfRelative;
	DWORD       dwRevision;

	//
	// Build File SD in absolute format for potential later modification
	//
	if (!InitializeSecurityDescriptor(psdAbsRegSD,
		 SECURITY_DESCRIPTOR_REVISION)) { 
		return FALSE;
	}

	//
	// Get Control from relative format File SD
	//
	if (!GetSecurityDescriptorControl(psdSrelRegSD,
			psdcCtrl,
			&dwRevision)) { 
		return FALSE;
	} 
	bSDSelfRelative = (SE_SELF_RELATIVE & *psdcCtrl);

	//
	// Set DACL into absolute format File SD
	//
	if (bDaclPresent = (SE_DACL_PRESENT   & *psdcCtrl)) {

		bDaclDefaulted = (SE_DACL_DEFAULTED & *psdcCtrl);
	}

	if (!GetSecurityDescriptorDacl(psdSrelRegSD,
							&bDaclPresent,      // fDaclPresent flag
							&paclDacl,
							&bDaclDefaulted)) {

		return FALSE;
	}
	if (!SetSecurityDescriptorDacl(psdAbsRegSD,
							bDaclPresent,       // fDaclPresent flag
							paclDacl,
							bDaclDefaulted)) {

		return FALSE;
	}

	//  
	// Set SACL into absolute format File SD
	//
	if (bSaclPresent = (SE_SACL_PRESENT   & *psdcCtrl)){

		bSaclDefaulted = (SE_SACL_DEFAULTED & *psdcCtrl);
	}

	if (!GetSecurityDescriptorSacl(psdSrelRegSD,
						&bSaclPresent,      // fSaclPresent flag
						&paclSacl,
						&bSaclDefaulted)) {

		return FALSE;
	}
	if (!SetSecurityDescriptorSacl(psdAbsRegSD,
						bSaclPresent,       // fSaclPresent flag
						paclSacl,
						bSaclDefaulted)) {

		return FALSE;
	} 

	//
	// Set Owner into absolute format File SD
	//
	bOwnerDefaulted = (SE_OWNER_DEFAULTED & *psdcCtrl);
	if (!GetSecurityDescriptorOwner(psdSrelRegSD,
					&psidSidOwn,
					&bOwnerDefaulted)) {

		return FALSE;
	}
	if (!SetSecurityDescriptorOwner(psdAbsRegSD,
					psidSidOwn,
					bOwnerDefaulted)) {

		return FALSE;
	}

	//
	// Set Group into absolute format File SD
	//
	bGroupDefaulted = (SE_GROUP_DEFAULTED & *psdcCtrl);
	if (!GetSecurityDescriptorGroup(psdSrelRegSD,
					&psidSidPG,
					&bGroupDefaulted)) {

		return FALSE;
	}
	if (!SetSecurityDescriptorGroup(psdAbsRegSD,
					psidSidPG,
					bGroupDefaulted)) {

		return FALSE;
	}

	//
	// Now we can see if the old computer SID is embedded
	// in the security descriptor, and change it if necessary
	//
	CheckKeyOwnershipForSID(hKey, psdAbsRegSD );
	
	//
	// Check to see if SID is embedded in the DACL and then
	// SACL
	//
	CheckKeyACLForSID( hKey, TRUE, psdAbsRegSD );
	CheckKeyACLForSID( hKey, FALSE, psdAbsRegSD );

	//
	// Always reset the DACL and close the key
	//
	SetRegAccess( NULL, NULL, psdAbsRegSD, &hKey );
	RegCloseKey( hKey );
	free( psdSrelRegSD );

	return(TRUE);
}


//----------------------------------------------------------------------
//
// UpdateKeySID
//
// Updates the SIDs for the specified registry key and recurses
// into subkeys.
//
//----------------------------------------------------------------------
void UpdateKeySID( HWND hDlg, HKEY Root, PTCHAR RootName, PTCHAR PathName,
				  PSECURITY_DESCRIPTOR newSecDesc )
{
	TCHAR			subKeyName[ 1024 ];
	TCHAR			subRootName[ 1024 ];
	TCHAR			subPathName[ 1024 ];
	HKEY			hKey;
	FILETIME		Time;
	DWORD			subKeys;
	int				idx;
	DWORD			status, nb;
	PSECURITY_DESCRIPTOR	origDesc;

	//
	// Update the progress dialog
	//
	UpdateProgressDialog( hDlg, IDC_PATHNAME, RootName );

	//
	// Change the security on the key so that we can enumerate it
	//
	status = RegOpenKeyEx( Root, PathName, 0, 
						WRITE_DAC|READ_CONTROL|
						ACCESS_SYSTEM_SECURITY,
						&hKey );
	if( status != ERROR_SUCCESS ) {
		return;
	}

	//
	// Get the source security and set the new descriptor
	//
	origDesc = GetRegAccess( hKey );
	SetRegAccess( Root, PathName, newSecDesc, &hKey );

	//
	// See if there are subkeys to enumerate
	//
	status = RegQueryInfoKey( hKey, NULL, NULL, NULL,
							&subKeys, NULL,
							NULL, NULL, NULL, NULL,
							NULL, NULL );
	if( status == ERROR_SUCCESS && subKeys ) {

		//
		// Allocate buffers 
		//
		idx = 0;
		while(1)  {

			nb = sizeof subKeyName;
			status = RegEnumKeyEx( hKey, idx, subKeyName, &nb, NULL, NULL, NULL, &Time );
			if ( status == ERROR_NO_MORE_ITEMS )
				break;
			if ( status != ERROR_SUCCESS )
				break;

			//
			// Construct the names
			//
			wsprintf( subRootName, L"%s\\%s", RootName, subKeyName );
			if( wcscmp( PathName, L"" )) 
				wsprintf( subPathName, L"%s\\%s", PathName, subKeyName );
			else
				wcscpy( subPathName, subKeyName );

			//
			// Process the subkey - leave the producttype alone, since
			// we would otherwise getting an annoying popup about
			// messing with our license.
			//
			if( !( !wcsnicmp( subPathName, PRODUCTTYPEPATH, wcslen(PRODUCTTYPEPATH)) &&
					!wcsicmp( subKeyName, PRODUCTTYPENAME))) 
			{
				UpdateKeySID( hDlg, Root,  subRootName, subPathName,
							  newSecDesc );
			}
			idx++;
		}
	}

	//
	// Now process the security on this key. This
	// will restore the security either to updated
	// (SID converted) settings, or the original
	// settings of the key. It will also close
	// the key.
	//
	CheckKeySID( hKey, Root, PathName, origDesc );
}

//----------------------------------------------------------------------
//
// UnloadProfileHives
//
// Unloads the user hives that were loaded into the Registry.
//
//----------------------------------------------------------------------
void UnloadProfileHives( DWORD NumLoadedHives )
{
	TCHAR		hiveName[256];
	DWORD		i;

	for( i = 0; i < NumLoadedHives; i++ ) {

		wsprintf( hiveName, L"User%d", i );
		RegUnLoadKey( HKEY_LOCAL_MACHINE, hiveName );
	}
}

//----------------------------------------------------------------------
//
// LoadProfileHives
//
// Loads all the user profile hives into the Registry, so that
// their security descriptors can be updated.
//
//----------------------------------------------------------------------
void LoadProfileHives( DWORD *NumLoadedHives )
{
	int				idx;
	DWORD			status, nb, type;
	HKEY			hKey, hProfileKey;
	TCHAR			profileKeyPath[1024], hiveName[256];
	TCHAR			subKeyName[1024], imagePath[1024], hivePath[1024];

	//
	// First, locate the user hives and load them
	//
	*NumLoadedHives = 0;
	status = RegOpenKey( HKEY_LOCAL_MACHINE, PROFILELISTPATH, &hKey );
	if( status == ERROR_SUCCESS ) {

		idx = 0;
		while(1)  {

			nb = 1024;
			status = RegEnumKeyEx( hKey, idx, subKeyName, &nb, NULL, NULL, NULL, NULL );
			if ( status == ERROR_NO_MORE_ITEMS )
				break;
			if ( status != ERROR_SUCCESS )
				break;

			//
			// Open the subkey and get the location of the registry hive
			//
			wsprintf( profileKeyPath, L"%s\\%s", PROFILELISTPATH, subKeyName );
			status = RegOpenKey( HKEY_LOCAL_MACHINE, profileKeyPath, &hProfileKey );
			if( status == ERROR_SUCCESS ) {
				
				nb = 1024;
				status = RegQueryValueEx( hProfileKey, L"ProfileImagePath",
										NULL, &type, (PCHAR) imagePath, &nb );
				if( status == ERROR_SUCCESS ) {

					//
					// Expand the system root
					//
					ExpandEnvironmentStrings( imagePath, hivePath, 1024 );

					//
					// Append the hive file name if the path isn't a file
					//
					if( GetFileAttributes( hivePath ) & FILE_ATTRIBUTE_DIRECTORY ) {
  					
						wcscat( hivePath, L"\\ntuser.dat" );
					}

					//
					// Now load the hive
					//
					wsprintf( hiveName, L"User%d", *NumLoadedHives );
					status = RegLoadKey( HKEY_LOCAL_MACHINE, hiveName, hivePath );
					if( status == ERROR_SUCCESS ) {

						(*NumLoadedHives)++;
					}	
				}
			}
			idx++;
		}
		RegCloseKey( hKey );
	}
}


//----------------------------------------------------------------------
//
// UpdateRegistrySID
//
// Scans all the Registry hives looking for SIDs
// that need to be changed in the security descriptors.
//
//----------------------------------------------------------------------
void UpdateRegistrySID( PSECURITY_DESCRIPTOR newSecDesc )
{
	TCHAR			startPath[16];
	HWND			regDialog;
	DWORD			NumLoadedHives;

	//
	// First, load all the user profile hives
	//
	LoadProfileHives( &NumLoadedHives );
	
	//
	// Create a dialog to show the file traversal
	//
	regDialog = CreateDialog( hInst, L"REGISTRY", NULL,
							(DLGPROC) UpdateDlgProc );

	//
	// Update all the loaded root keys
	//
	wcscpy( startPath, L"");
	UpdateKeySID( regDialog, HKEY_LOCAL_MACHINE, L"HKLM", startPath, newSecDesc ); 	
	wcscpy( startPath, L"");
	UpdateKeySID( regDialog, HKEY_CURRENT_USER, L"HKCU", startPath, newSecDesc );

	//
	// Delete the file dialog
	//
	DestroyWindow( regDialog );

	//
	// Unload user profile hives
	//
	UnloadProfileHives( NumLoadedHives );
}


//===================================================================
//
// FILE SECURITY DESCRIPTOR UPDATING 
//
//===================================================================


//----------------------------------------------------------------------
//
// CheckFileOwnershipForSID
//
// Sees if the file ownership SID contains the old computer SID, and
// if so, replaces it.
//
//----------------------------------------------------------------------
void CheckFileOwnershipForSID(PSECURITY_DESCRIPTOR psdFileSD, 
							  LPTSTR  lpszFullName)
{
	PSID		psidFileOwnerSID;
	PSID		psidFileGroupSID;
	BOOL		bOwnerDefaulted;

	//
	// Get the owner SID
	//
	if (!GetSecurityDescriptorOwner(psdFileSD,
		(PSID *)&psidFileOwnerSID, (LPBOOL)&bOwnerDefaulted)) { 

		return;
	}

	//
	// If the old SID is in the owner, we have to write
	// the updated owner to disk
	//
    if( SecurityReplaceSID( psidFileOwnerSID) ) {
		if (!SetFileSecurity(lpszFullName,
            (SECURITY_INFORMATION)OWNER_SECURITY_INFORMATION,
            psdFileSD)){
			
			return;
		}
    }

	//
	// Now do the same with the Group SID
	//
	if (!GetSecurityDescriptorOwner(psdFileSD,
		(PSID *)&psidFileGroupSID, (LPBOOL)&bOwnerDefaulted)) { 

		return;
	}

	//
	// If the old SID is in the owner, we have to write
	// the updated owner to disk
	//
    if( SecurityReplaceSID( psidFileGroupSID) ) {
		if (!SetFileSecurity(lpszFullName,
            (SECURITY_INFORMATION)GROUP_SECURITY_INFORMATION,
            psdFileSD)){
			
			return;
		}
    }
	return;
}


//----------------------------------------------------------------------
//
// CheckACLForSID
//
// Scan's the security descriptor's ACEs, looking for instances
// of the old computer SID.
//
//----------------------------------------------------------------------
void CheckFileACLForSID( BOOLEAN Dacl, PSECURITY_DESCRIPTOR psdFileSD,
                             LPTSTR  lpszFullName)
{
	PACL       paclFile;
	BOOL       bHasACL;
	BOOL       bOwnerDefaulted;
	DWORD      dwAcl_i;
	BOOLEAN    descriptorModified;
	DWORD      dwLastError   = NO_ERROR;
	ACL_SIZE_INFORMATION asiAclSize;
	DWORD      dwBufLength = sizeof(asiAclSize);
	ACCESS_ALLOWED_ACE   *paaAllowedAce;

	if( Dacl ) {
		if (!GetSecurityDescriptorDacl(psdFileSD,
							 (LPBOOL)&bHasACL, (PACL *)&paclFile,
							 (LPBOOL)&bOwnerDefaulted)) {
			return;
		}
	} else {
		if (!GetSecurityDescriptorSacl(psdFileSD,
							 (LPBOOL)&bHasACL, (PACL *)&paclFile,
							 (LPBOOL)&bOwnerDefaulted)) {
			return;
		}
	}

	//
	// If no ACL to process, so OK, return
	//
	if (!bHasACL || !paclFile ) 
		return;

	if (!GetAclInformation(paclFile, (LPVOID)&asiAclSize,
				 (DWORD)dwBufLength, (ACL_INFORMATION_CLASS)AclSizeInformation)){
		return;
	}
	descriptorModified = FALSE;

	// 
	// Look through the ACEs
	//
	for (dwAcl_i = asiAclSize.AceCount-1;  ((int)dwAcl_i) >= 0;  dwAcl_i--) {

		//
		// If we can't get an ACE, bail
		//
		if (!GetAce(paclFile, dwAcl_i, (LPVOID *)&paaAllowedAce)) {

			return;
		}

		//
		// Make sure we're dealing with an ACE we know
		//
		if (!( (paaAllowedAce->Header.AceType == ACCESS_ALLOWED_ACE_TYPE)
				||(paaAllowedAce->Header.AceType == ACCESS_DENIED_ACE_TYPE )
				||(paaAllowedAce->Header.AceType == SYSTEM_AUDIT_ACE_TYPE  )
				||(paaAllowedAce->Header.AceType == SYSTEM_ALARM_ACE_TYPE  ))) {
			continue;
		}

		//
		// Look at the SID's subauthorities to see if there's a match
		// with the old computer SID
		//
		descriptorModified |= SecurityReplaceSID((PSID)&(paaAllowedAce->SidStart));
	}

	//
	// If the security descriptor was modified because an
	// old computer SID was converted to the new one, write
	// the new descriptor to disk.
	//
	if( descriptorModified ) {

		//
		// Modify the SD on the hard disk
		//
		if( Dacl ) {

			//
			// Modify the SD in virtual memory
			//
			if (!SetSecurityDescriptorDacl(psdFileSD,
								TRUE, paclFile, FALSE)) {

				return;
			}

			if (!SetFileSecurity(lpszFullName,
							(SECURITY_INFORMATION)DACL_SECURITY_INFORMATION,
							psdFileSD)){

				return;
			}
		} else {

			//
			// Modify the SD in virtual memory
			//
			if (!SetSecurityDescriptorSacl(psdFileSD,
								TRUE, paclFile, FALSE)) {

				return;
			}

			if (!SetFileSecurity(lpszFullName,
							(SECURITY_INFORMATION)SACL_SECURITY_INFORMATION,
							psdFileSD)){

				return;
			}
		}
	}
}


//----------------------------------------------------------------------
//
// CheckFileSID
//
// Reads the file's security descriptor and converts it to absolute
// format. Then it calls a function to check the ownership SIDs and
// finally one to check the DACL SIDs, to see if they need to be
// updated.
//
//----------------------------------------------------------------------
BOOL CheckFileSID(LPTSTR lpszFullName)
{
	UCHAR		ucBuf[SZ_REL_SD_BUF];
	UCHAR       ucBufAbs[SZ_ABS_SD_BUF];
	UCHAR       ucBufCtrl[sizeof(PSECURITY_DESCRIPTOR_CONTROL)];
	DWORD       dwSDLength = SZ_REL_SD_BUF;
	DWORD       dwSDLengthNeeded;
	PSECURITY_DESCRIPTOR psdSrelFileSD = (PSECURITY_DESCRIPTOR)&ucBuf;
	PSECURITY_DESCRIPTOR psdAbsFileSD = (PSECURITY_DESCRIPTOR)&ucBufAbs;
	PSECURITY_DESCRIPTOR_CONTROL psdcCtrl = (PSECURITY_DESCRIPTOR_CONTROL)&ucBufCtrl;
	PACL        paclDacl;
	PACL        paclSacl;
	PSID        psidSidOwn;
	PSID        psidSidPG;
	BOOL        bDaclPresent;
	BOOL        bDaclDefaulted;
	BOOL        bSaclPresent;
	BOOL        bSaclDefaulted;
	BOOL        bOwnerDefaulted;
	BOOL        bGroupDefaulted;
	BOOL        bSDSelfRelative;
	DWORD       dwRevision;

	//
	// Get the file's security descriptor
	//
	if (!GetFileSecurity(lpszFullName,
						(SECURITY_INFORMATION)( OWNER_SECURITY_INFORMATION
					   | GROUP_SECURITY_INFORMATION
					   | DACL_SECURITY_INFORMATION
					   | SACL_SECURITY_INFORMATION),
						psdSrelFileSD,
						dwSDLength,
						(LPDWORD)&dwSDLengthNeeded)) {

		return(FALSE);
	}

	//
	// Build File SD in absolute format for potential later modification
	//
	if (!InitializeSecurityDescriptor(psdAbsFileSD,
		 SECURITY_DESCRIPTOR_REVISION)) { 
		return FALSE;
	}

	//
	// Get Control from relative format File SD
	//
	if (!GetSecurityDescriptorControl(psdSrelFileSD,
			psdcCtrl,
			&dwRevision)) { 
		return FALSE;
	} 
	bSDSelfRelative = (SE_SELF_RELATIVE & *psdcCtrl);

	//
	// Set DACL into absolute format File SD
	//
	if (bDaclPresent = (SE_DACL_PRESENT   & *psdcCtrl)) {

		bDaclDefaulted = (SE_DACL_DEFAULTED & *psdcCtrl);
	}

	if (!GetSecurityDescriptorDacl(psdSrelFileSD,
							&bDaclPresent,      // fDaclPresent flag
							&paclDacl,
							&bDaclDefaulted)) {

		return FALSE;
	}
	if (!SetSecurityDescriptorDacl(psdAbsFileSD,
							bDaclPresent,       // fDaclPresent flag
							paclDacl,
							bDaclDefaulted)) {

		return FALSE;
	}

	//  
	// Set SACL into absolute format File SD
	//
	if (bSaclPresent = (SE_SACL_PRESENT   & *psdcCtrl)){

		bSaclDefaulted = (SE_SACL_DEFAULTED & *psdcCtrl);
	}

	if (!GetSecurityDescriptorSacl(psdSrelFileSD,
						&bSaclPresent,      // fSaclPresent flag
						&paclSacl,
						&bSaclDefaulted)) {

		return FALSE;
	}
	if (!SetSecurityDescriptorSacl(psdAbsFileSD,
						bSaclPresent,       // fSaclPresent flag
						paclSacl,
						bSaclDefaulted)) {

		return FALSE;
	} 

	//
	// Set Owner into absolute format File SD
	//
	bOwnerDefaulted = (SE_OWNER_DEFAULTED & *psdcCtrl);
	if (!GetSecurityDescriptorOwner(psdSrelFileSD,
					&psidSidOwn,
					&bOwnerDefaulted)) {

		return FALSE;
	}
	if (!SetSecurityDescriptorOwner(psdAbsFileSD,
					psidSidOwn,
					bOwnerDefaulted)) {

		return FALSE;
	}

	//
	// Set Group into absolute format File SD
	//
	bGroupDefaulted = (SE_GROUP_DEFAULTED & *psdcCtrl);
	if (!GetSecurityDescriptorGroup(psdSrelFileSD,
					&psidSidPG,
					&bGroupDefaulted)) {

		return FALSE;
	}
	if (!SetSecurityDescriptorGroup(psdAbsFileSD,
					psidSidPG,
					bGroupDefaulted)) {

		return FALSE;
	}

	//
	// Now we can see if the old computer SID is embedded
	// in the security descriptor, and change it if necessary
	//
	CheckFileOwnershipForSID(psdAbsFileSD,lpszFullName);
	
	//
	// Check to see if SID is embedded in the DACL and then
	// SACL
	//
	CheckFileACLForSID( TRUE, psdAbsFileSD,lpszFullName);
	CheckFileACLForSID( FALSE, psdAbsFileSD, lpszFullName );
	return(TRUE);
}


//----------------------------------------------------------------------
//
// UpdateFileSID
//
// Updates the SIDs for the specified file or directory and recurses
// on subdirectories and files.
//
//----------------------------------------------------------------------
void UpdateFileSID( HWND hDlg, PTCHAR PathName )
{
	TCHAR			subName[MAX_PATH], searchName[MAX_PATH];
	HANDLE			dirHandle;
	WIN32_FIND_DATA foundFile;

	//
	// Process this file or directory
	//
	UpdateProgressDialog( hDlg, IDC_PATHNAME, PathName );
	CheckFileSID( PathName );

	//
	// Scan the files and/or directories if this is a directory
	//
	wsprintf( searchName, L"%s\\*.*", PathName );
	if( (dirHandle = FindFirstFile( searchName, &foundFile )) == 
		INVALID_HANDLE_VALUE  ) {

		//
		// Nothing to process
		//
		return;
	}

	do {

		if( wcscmp( foundFile.cFileName, L"." ) &&
			wcscmp( foundFile.cFileName, L".." )) {

			//
			// Process the subfile or directory
			//
			wsprintf( subName, L"%s\\%s", PathName, foundFile.cFileName );
			UpdateFileSID( hDlg, subName );
		}
	} while( FindNextFile( dirHandle, &foundFile ));

	FindClose( dirHandle );
}


//----------------------------------------------------------------------
//
// UpdateFileSystemSID
//
// Scans all the non-removable NTFS drives looking for SIDs
// that need to be changed in the security descriptors.
//
//----------------------------------------------------------------------
void UpdateFileSystemSID( void )
{
	DWORD	drives, curDrive;
	TCHAR	drivePath[16], startPath[16];
	TCHAR	fileSystem[256];
	HWND	fileDialog;

	//
	// Create a dialog to show the file traversal
	//
	fileDialog = CreateDialog( hInst, L"FILE", NULL,
							(DLGPROC) UpdateDlgProc );

	drives = GetLogicalDrives();
	curDrive = 2;  // start with 'C:'
	while( curDrive < 26 ) { 

		wsprintf( drivePath, L"%c:\\", curDrive + L'A');

		GetVolumeInformation( drivePath, NULL, 0, NULL, NULL,
								NULL, fileSystem, 256 );

		//
		// Only process local NTFS drives
		//
		if( ((1 << curDrive) & drives ) && 
			(GetDriveType( drivePath ) == DRIVE_FIXED) &&
			!wcsicmp( fileSystem, L"NTFS")) {

			wsprintf( startPath, L"%c:", curDrive + L'A');
			UpdateFileSID( fileDialog, startPath ); 	
		}
		curDrive++;
	}

	//
	// Delete the file dialog
	//
	DestroyWindow( fileDialog );
}


//===================================================================
//
// SAM AND SECURITY SID CHANGING ROUTINES
//
//===================================================================

//----------------------------------------------------------------------
//
// CopyKey2
//
// Copies a subtree - this is the core of our recursive subkey copy
// functions.
//
//----------------------------------------------------------------------
VOID CopyKey2( HKEY Root, TCHAR *Source, TCHAR *Destination, PSECURITY_DESCRIPTOR newSecDesc )
{
	HKEY					srcKey, dstKey;
	DWORD					status;
	PSECURITY_DESCRIPTOR	origDesc;
	TCHAR					*SourceTail;
	TCHAR					*DestTail;
	FILETIME				Time;
	int						idx;
	TCHAR					valName[1024];
	DWORD					valNameLen, valDataLen, valType, valDataBufsize = 1024;
	BYTE					*valData = malloc( valDataBufsize );
	BOOLEAN					firstPass = TRUE;
	DWORD					nb;

	//
	// Open the source
	//
	status = RegOpenKey( Root, Source, &srcKey );
	if( status != ERROR_SUCCESS ) {
		return;
	}

	//
	// Get the source security and set the new descriptor
	//
	origDesc = GetRegAccess( srcKey );
	SetRegAccess( Root, Source, newSecDesc, &srcKey );

	//
	// Create a copy
	//
	status = RegCreateKey( Root, Destination, &dstKey );
	if( status != ERROR_SUCCESS ) {
		return;
	}

	//
	// Enumerate source values and create copies in the destination
	//
	wcscpy( valName, L"");
	idx = 0;
	valDataLen = 0;

	do {

		//
		// Copy to the destination 
		//
		if( !firstPass ) {
			if( valType == (DWORD) -1 ) valDataLen = 0;
			(void) RegSetValueEx( dstKey, valName, 0, valType, valData, valDataLen );
		} else {
			firstPass = FALSE;
		}
		valNameLen = 1024;
		valDataLen = valDataBufsize;
		status = RegEnumValue( srcKey, idx++, valName, &valNameLen,
					NULL, &valType,	valData, &valDataLen  );
		while ( status == ERROR_MORE_DATA )  {
			valDataBufsize = valDataLen += 1024;
			valData = realloc( valData, valDataBufsize );
			status = RegEnumValue( srcKey, idx-1, valName, &valNameLen,
									NULL, &valType,	valData, &valDataLen  );
		}

	} while ( status == ERROR_SUCCESS );

	//
	// Enumerate source subkeys and create copies in the destination
	//	
	SourceTail = Source + wcslen(Source);
	*SourceTail++ = L'\\';
	DestTail = Destination + wcslen(Destination);
	*DestTail++ = L'\\';

	idx = 0;
	while(1)  {

		nb = 1024;
		status = RegEnumKeyEx( srcKey, idx, SourceTail, &nb, NULL, NULL, NULL, &Time );
		if ( status == ERROR_NO_MORE_ITEMS )
			break;
		if ( status != ERROR_SUCCESS )
			break;

		// Copy recursively
		wcscpy( DestTail, SourceTail );
		CopyKey2( Root, Source, Destination, newSecDesc );

		//
		// Restart the emumeration
		//
		RegCloseKey( srcKey );
		SourceTail[-1] = L'\0';
		status = RegOpenKey( Root, Source, &srcKey );
		if( status != ERROR_SUCCESS ) {
			break;
		}
		wcscpy( SourceTail-1, L"\\" );
		idx = 0;
	}	

	//
	// Set copy of access on destination(which closes the keys)
	//
	SetRegAccess( NULL, NULL, newSecDesc, &dstKey );
	RegDeleteKey( srcKey, NULL );
	RegCloseKey( srcKey );
	RegCloseKey( dstKey );
	free( origDesc );
}


//----------------------------------------------------------------------
//
// CopyKey
//
// Top level function for recursive key copy routine.
//
//----------------------------------------------------------------------
VOID CopyKey( HKEY Root, TCHAR *Source, TCHAR *Destination, PSECURITY_DESCRIPTOR newSecDesc )
{
	TCHAR	fullSource[ 1024 ];
	TCHAR	fullDest[ 1024 ];

	wcscpy( fullSource, Source );
	wcscpy( fullDest, Destination );
	CopyKey2( Root, fullSource, fullDest, newSecDesc );
}



//----------------------------------------------------------------------
//
// ChangeComputerSID
//
// Recursively dive into the key, changing all occurrences of oldSubAuth
// to newSubAuth, whether its in a value, or in a key name.
//
//----------------------------------------------------------------------
BOOL ChangeComputerSID( HKEY Root, TCHAR *path, PSECURITY_DESCRIPTOR newSecDesc )
{
	LONG		status;
	HKEY		hKey;
	int         idx;
	TCHAR       Name[ 1024 ];
	TCHAR       *Tail, *newSubPtr;
	FILETIME	Time;
	TCHAR		newSubName[1024], textNewSubAuth[256];
	TCHAR		textSubAuth[256], textSubAuthTmp[256];
	TCHAR		valName[1024];
	DWORD		valNameLen, valDataLen, valType, valDataBufsize = 1024;
	BYTE	*	valData = malloc( valDataBufsize );
	DWORD		nb, j;	
	BOOLEAN		valChanged;
	PBYTE		sidloc;
	PSECURITY_DESCRIPTOR oldSecDesc;

	//
	// Open key 
	//
	status = RegOpenKey( Root, path, &hKey );
	if ( status != ERROR_SUCCESS ) {

		//
		// Work around a bug in MTS where it creates an unopenable
		// key
		//
		if( status == ERROR_FILE_NOT_FOUND ) 
			return TRUE;
		else
			return FALSE;
	}

	//
	// Get the key's security descriptor and set the new one
	//
	if( !(oldSecDesc = GetRegAccess( hKey ))) 
		return FALSE;
	if( !SetRegAccess( Root, path, newSecDesc, &hKey ) ) 
		return FALSE;

	//
	// Scan the values looking for the old SID and change it
	// to the new SID.
	//
	wcscpy( valName, L"");
	idx = 0;
	valNameLen = 1024;
	valDataLen = valDataBufsize;
	status = RegQueryValue( hKey, path, (PTCHAR) valData, &valDataLen );
	while ( status == ERROR_MORE_DATA )  {
		valDataBufsize = valDataLen += 1024;
		valData = realloc( valData, valDataBufsize );
		status = RegQueryValue( hKey, path, (PTCHAR) valData, &valDataLen );
	}
	do {
		BYTE * tail = valData + valDataLen - SubAuthLength + 1;

		//
		// If its large enough, scan for the old SID
		//
		valChanged = FALSE;

		for ( sidloc = valData;
			  (sidloc < tail)  &&  (sidloc = memchr( sidloc, OldSubAuth[0], tail - sidloc ));
			  sidloc++ )
		{
			if ( memcmp( sidloc, OldSubAuth, SubAuthLength ) == 0 ) {
				memcpy( sidloc, NewSubAuth, SubAuthLength );
				sidloc += SubAuthLength - 1;
				valChanged = TRUE;
			}
		}

		//
		// If we performed a replacement, update the value
		//
		if( valChanged ) {
			if( RegSetValueEx( hKey, valName, 0, valType, valData, valDataLen )	!= ERROR_SUCCESS ) 
				return FALSE;
		}

		valNameLen = 1024;
		valDataLen = valDataBufsize;
		status = RegEnumValue( hKey, idx++, valName, &valNameLen,
								NULL, &valType,	valData, &valDataLen  );
		while ( status == ERROR_MORE_DATA )  {
			valDataBufsize = valDataLen += 1024;
			valData = realloc( valData, valDataBufsize );
			status = RegEnumValue( hKey, idx-1, valName, &valNameLen,
									NULL, &valType,	valData, &valDataLen  );
		}
	
	} while ( status == ERROR_SUCCESS );


	//
	// Now we enumerate subkeys, because if the tail has 
	// the SID embedded in it, we have to copy the subtree
	//
	wcscpy( Name, path );
	Tail = Name + wcslen(Name);
	*Tail++ = L'\\';

	wsprintf( textSubAuth, L"" );
	for( j = 2; j < SubAuthLength / sizeof(DWORD); j++ ) {
		wsprintf( textSubAuthTmp, L"-%u", ((PDWORD) OldSubAuth)[j] );
		wcscat( textSubAuth, textSubAuthTmp );
	}

	wsprintf( textNewSubAuth, L"" );
	for( j = 2; j < SubAuthLength / sizeof(DWORD); j++ ) {
		wsprintf( textSubAuthTmp, L"-%u", ((PDWORD) NewSubAuth)[j] );
		wcscat( textNewSubAuth, textSubAuthTmp );
	}

	//
	// Get subkey names
	//
	for ( idx = 0;; idx ++ )  {

		nb = sizeof Name;
		status = RegEnumKeyEx( hKey, idx, Tail, &nb, NULL, NULL, NULL, &Time );
		if ( status == ERROR_NO_MORE_ITEMS )
			break;
		if ( status != ERROR_SUCCESS )
			break;		

		if( !ChangeComputerSID( Root, Name, newSecDesc ) ) {

			SetRegAccess( NULL, NULL, oldSecDesc, &hKey );
			RegCloseKey( hKey );
			free( oldSecDesc);
			return FALSE;
		}

		for( j = 0; j < wcslen( Tail ); j++ ) {
			if( !wcsncmp( textSubAuth, Tail+j, wcslen( textSubAuth ) ) ) {

				//
				// Got a match - copy the tree to the new name
				//
				wsprintf( newSubName, L"%s", Name );
				newSubPtr = (TCHAR *) ((DWORD) newSubName + (DWORD) Tail - (DWORD) Name + j*2); 
				wcscpy( newSubPtr, textNewSubAuth );
				wcscat( newSubPtr, Tail + j + wcslen( textSubAuth ));

				CopyKey( Root, Name, newSubName, newSecDesc );

				//
				// Restart the scan of the key
				//
				idx = -1;
				break;
			}
		}
	}
	
	//
	// Change scurity back and close key
	//
	SetRegAccess( NULL, NULL, oldSecDesc, &hKey );
	RegCloseKey( hKey );
	free( oldSecDesc);
	free( valData );
	return TRUE;
}


//===================================================================
//
// MAIN DIALOG AND WINMAN
//
//===================================================================

//----------------------------------------------------------------------
//
// WelcomeDialog
//
// Splash copyright, and choice.
//
//----------------------------------------------------------------------
LONG CALLBACK WelcomeDialog( HWND hDlg, UINT message, UINT wParam, LONG lParam )
{
	switch (message) {
		case WM_COMMAND:
			switch (wParam) {
				case IDOK:
					EndDialog( hDlg, 0 );
					return TRUE ;
				case IDC_SYNC:
					if( DialogBox( hInst, L"SIDCHOICE", NULL, (DLGPROC)SidChoiceDlgProc ) !=
						DIALOG_CANCEL ) {
						EndDialog( hDlg, 0 );
					}
					return TRUE;
				case IDC_NAME:
					DialogBox( hInst, L"NAME", NULL, (DLGPROC)NameDlgProc );
					return TRUE;
				case IDCANCEL:
					EndDialog( hDlg, DIALOG_CANCEL );
					return TRUE;
			}
			break; 

		case WM_CLOSE:	
			EndDialog( hDlg, DIALOG_CANCEL );
			return TRUE;
	}

	return DefWindowProc( hDlg, message, wParam, lParam );
}

//----------------------------------------------------------------------
//  
// ShutDown
//
// Reboot the machine
//
//----------------------------------------------------------------------
BOOL ShutDown()
{
	HANDLE				hToken;    
	TOKEN_PRIVILEGES	tkp;      
	BOOL				fResult;     
 
	// 
	// Get the current process token handle 
	// so we can get shutdown privilege. 
	// 
	if (!OpenProcessToken(GetCurrentProcess(), 
	        TOKEN_ADJUST_PRIVILEGES | TOKEN_QUERY, &hToken)) 
	    return FALSE;
 
	// Get the LUID for shutdown privilege. 
	LookupPrivilegeValue(NULL, SE_SHUTDOWN_NAME, 
	        &tkp.Privileges[0].Luid); 
 
	tkp.PrivilegeCount = 1;  
	tkp.Privileges[0].Attributes = SE_PRIVILEGE_ENABLED; 
	// Get shutdown privilege for this process. 
 	AdjustTokenPrivileges(hToken, FALSE, &tkp, 0, 
	    (PTOKEN_PRIVILEGES) NULL, 0); 

  	// Cannot test the return value of AdjustTokenPrivileges. 
	if (GetLastError() != ERROR_SUCCESS) 
	    return FALSE; 
 
	// Actually shutdown
	fResult = InitiateSystemShutdown(  NULL, NULL, 0, TRUE, TRUE );
 
	if (!fResult)  
	    return FALSE;
 
	// Disable shutdown privilege.
	tkp.Privileges[0].Attributes = 0; 
	AdjustTokenPrivileges(hToken, FALSE, &tkp, 0, 
	        (PTOKEN_PRIVILEGES) NULL, 0); 
 
	if (GetLastError() != ERROR_SUCCESS) 
	    return FALSE;

	return TRUE;
}


//----------------------------------------------------------------------
//
// DoSIDChange
//
// Actually directs the change of the computer's sid
//
//----------------------------------------------------------------------
void DoSIDChange( BOOLEAN Auto )
{
	BYTE				*	SidPtr;
	DWORD					Step = 0;
	DWORD					nb;
	DWORD					Status;
	HKEY					hKey;
	WNDCLASSEX				wndclass ;
	HANDLE					syncHandle;
	DWORD					valType;
	PBYTE					vData;
	HWND					generateDialog;
	PSECURITY_DESCRIPTOR	newSecDesc, oldSecDesc;

	//
	// create a generic window class
	//
	wndclass.cbSize			= sizeof( WNDCLASSEX );
	wndclass.style          = CS_HREDRAW | CS_VREDRAW ;
	wndclass.cbClsExtra     = 0 ;
	wndclass.cbWndExtra     = DLGWINDOWEXTRA ;
	wndclass.hInstance      = hInst;
	wndclass.hIcon          = LoadIcon (hInst, L"APPICON") ;
	wndclass.hIconSm		= LoadIcon (hInst, L"APPICON");
	wndclass.hCursor        = LoadCursor (NULL, IDC_ARROW) ;
	wndclass.hbrBackground  = (HBRUSH) (COLOR_BTNFACE+1);
	wndclass.lpszMenuName   = NULL ;

	wndclass.lpfnWndProc    = (WNDPROC) WelcomeDialog ;
	wndclass.lpszClassName  = L"Welcome";
	RegisterClassEx( &wndclass );

	//
	// If not automatic mode, ask the user
	//
	if( !Auto ) {

		Status = DialogBox( hInst, L"Welcome", NULL, 
						(DLGPROC)WelcomeDialog );

		if( Status == DIALOG_CANCEL ) return;
	}

	//
	// Now, get the descriptor of HKLM\SOFTWARE and apply this to SECURITY
	//
	newSecDesc = GetRegSecDesc( HKEY_LOCAL_MACHINE, L"SOFTWARE",
					DACL_SECURITY_INFORMATION );

	//
	// Read the last subauthority of the current computer SID
	//
	if( RegOpenKey( HKEY_LOCAL_MACHINE, L"SECURITY\\SAM\\Domains\\Account", 
		&hKey) != ERROR_SUCCESS ) {

		MessageBox( NULL, L"NewSID was unable to access the\nComputer's SID.",
									L"NewSID", MB_ICONERROR|MB_OK );	
		return;
	}
	oldSecDesc = GetRegAccess( hKey );
	SetRegAccess( HKEY_LOCAL_MACHINE, L"SECURITY\\SAM\\Domains\\Account", 
				newSecDesc, &hKey );

	nb = 0;
	vData = NULL;
	RegQueryValueEx( hKey, L"V", NULL, &valType, vData, &nb );
	vData = (PBYTE) malloc( nb );
	Status = RegQueryValueEx( hKey, L"V", NULL, &valType, vData, &nb );
	if( Status != ERROR_SUCCESS ) {
		
		SetRegAccess( HKEY_LOCAL_MACHINE, L"SECURITY\\SAM\\Domains\\Account", 
				oldSecDesc, &hKey );
		MessageBox( NULL, L"NewSID was unable to access the\nComputer's SID.",
									L"NewSID", MB_ICONERROR|MB_OK );	
		return;
	}
	SetRegAccess( NULL, NULL, oldSecDesc, &hKey );
	RegCloseKey( hKey );

	//
	// Make sure that we're dealing with a SID we understand
	//
	SidPtr = NULL;
	if ( nb % sizeof(DWORD) == 0 )  {
		for ( SidPtr = vData + nb - 5*sizeof(DWORD); SidPtr >= vData; SidPtr -= sizeof(DWORD) )
			if ( ((PDWORD)SidPtr)[0] == 0x05000000  &&  ((PDWORD)SidPtr)[1] == 0x00000015 )
				break;
		if ( SidPtr < vData )
			SidPtr = NULL;
	}
	if ( SidPtr == NULL )  {
		MessageBox( NULL, L"NewSID does not undetstand the format of\nthe Computer's SID.",
						L"NewSID", MB_ICONERROR|MB_OK );	
		return;
	}
	memcpy( OldSubAuth, SidPtr, SubAuthLength );

	//
	// Either generate a new random SID, or obtain the SID from a specified
	// domain controller
	//
	if( SyncSID ) {

		generateDialog = CreateDialog( hInst, L"READSYNCSID", NULL,
								(DLGPROC) UpdateDlgProc );

		//
		// Open the registry of the target computer
		//
		if( RegConnectRegistry( SyncComputerName, HKEY_LOCAL_MACHINE, &syncHandle ) ) {

			DestroyWindow( generateDialog );
			MessageBox( NULL, L"NewSID was unable to access the\ntarget computer's SID.",
										L"NewSID", MB_ICONERROR|MB_OK );	
			return;
		}

		//
		// Read the SID of the target computer
		//
		if( RegOpenKey( syncHandle, L"SECURITY\\SAM\\Domains\\Account", 
			&hKey) != ERROR_SUCCESS ) {

			DestroyWindow( generateDialog );
			MessageBox( NULL, L"NewSID was unable to access the\ntarget computer's SID.",
										L"NewSID", MB_ICONERROR|MB_OK );	
			return;
		}
		oldSecDesc = GetRegAccess( hKey );
		SetRegAccess( syncHandle, L"SECURITY\\SAM\\Domains\\Account", 
					newSecDesc, &hKey );

		nb = 0;
		vData = NULL;
		RegQueryValueEx( hKey, L"V", NULL, &valType, vData, &nb );
		vData = (PBYTE) malloc( nb );
		Status = RegQueryValueEx( hKey, L"V", NULL, &valType, vData, &nb );
		if( Status != ERROR_SUCCESS ) {
			
			SetRegAccess( syncHandle, L"SECURITY\\SAM\\Domains\\Account", 
					oldSecDesc, &hKey );
			DestroyWindow( generateDialog );
			MessageBox( NULL, L"NewSID was unable to access the\ntarget computer's SID.",
										L"NewSID", MB_ICONERROR|MB_OK );
			return;
		}
		SetRegAccess( NULL, NULL, oldSecDesc, &hKey );
		RegCloseKey( hKey );

		//
		// Make sure that we're dealing with a SID we understand
		//
		SidPtr = NULL;
		if ( nb % sizeof(DWORD) == 0 )  {
			for ( SidPtr = vData + nb - 5*sizeof(DWORD); SidPtr >= vData; SidPtr -= sizeof(DWORD) )
				if ( ((PDWORD)SidPtr)[0] == 0x05000000  &&  ((PDWORD)SidPtr)[1] == 0x00000015 )
					break;
			if ( SidPtr < vData )
				SidPtr = NULL;
		}
		if ( SidPtr == NULL )  {
			DestroyWindow( generateDialog );
			MessageBox( NULL, L"NewSID does not undetstand the format of\nthe target computer's SID.",
							L"NewSID", MB_ICONERROR|MB_OK );	
			return;
		}
		memcpy( NewSubAuth, SidPtr, SubAuthLength );


	} else {

		//
		// Generating a truly random SID takes awhile, so put up a dialog
		//
		generateDialog = CreateDialog( hInst, L"GENERATE", NULL,
								(DLGPROC) UpdateDlgProc );

		//
		// Generate new components for the 4 subauthorities
		//
		((PDWORD)NewSubAuth)[0]	= ((PDWORD)OldSubAuth)[0];
		((PDWORD)NewSubAuth)[1]	= ((PDWORD)OldSubAuth)[1];
		((PDWORD)NewSubAuth)[2]	= Random();
		((PDWORD)NewSubAuth)[3]	= Random();
		((PDWORD)NewSubAuth)[4]	= Random();
	}

	DestroyWindow( generateDialog );

	// 
	// Replace the original computer SID with the one we generated
	//
	if( ChangeComputerSID( HKEY_LOCAL_MACHINE, L"SECURITY",
								newSecDesc )) {

		//
		// Update SIDs embedded in all service security descriptors, which includes
		// file, port and printer shares
		//
		ChangeComputerSID( HKEY_LOCAL_MACHINE, L"SYSTEM\\CurrentControlSet\\Services",
								newSecDesc );

		//
		// Update SIDs embedded in NTFS security descriptors
		//
		UpdateFileSystemSID();

		//
		// Update SIDs embedded in Registry keys
		//
		UpdateRegistrySID( newSecDesc );

		//
		// Change the profile information 
		//
		if( ChangeComputerSID( HKEY_LOCAL_MACHINE, 
					L"Software\\Microsoft\\Windows NT\\CurrentVersion\\ProfileList",
					newSecDesc ) ) {

			if(!Auto)
				MessageBox( NULL, L"NewSID has successfully changed the\nComputer's SID.\n\nPress OK to reboot the computer.",
											L"NewSID", MB_ICONQUESTION|MB_OK );	

		} else {

			MessageBox( NULL, L"NewSID has successfully changed the\nComputer's SID but was unable\n"
							  L"to update profile information.\n\nPress OK to reboot the computer.",
										L"NewSID", MB_ICONQUESTION|MB_OK );	
		}

		//
		// After changing security descriptors, we have to reboot or users may run into 
		// situations where they can't access files they own!
		//
		ShutDown();

	} else {

		//
		// Something went wrong
		//
		MessageBox( NULL, L"NewSID was unable to change the\nComputer's SID.\n\nIt is suggested you restore your SAM and\nSECURITY keys from backup.",
									L"NewSID", MB_ICONERROR|MB_OK );	
	}
}


//----------------------------------------------------------------------
//  
// WinMain
//
//----------------------------------------------------------------------
int CALLBACK WinMain( HINSTANCE hInstance, HINSTANCE hPrevInstance,
        				LPSTR lpCmdLine, int nCmdShow )
{
	HANDLE				Client;
	BOOLEAN				Auto = FALSE;
	TOKEN_PRIVILEGES  *	Priv;
	TCHAR				buf[ 1024 ];
	int					j, i;
	TCHAR				computerName[256];
	static TCHAR	*	P[] = {
		SE_TAKE_OWNERSHIP_NAME,
		SE_SECURITY_NAME,
		SE_BACKUP_NAME,
		SE_RESTORE_NAME,
		SE_MACHINE_ACCOUNT_NAME,
		SE_CHANGE_NOTIFY_NAME 
	};

	// 
	// Give ourselves privileges for manipulating security
	// 
	if ( ! OpenProcessToken( GetCurrentProcess(), TOKEN_ADJUST_PRIVILEGES, &Client ) ) {
		DisplayError( L"NewSID was unable to obtain necessary privilege to change the computer SID");
		return FALSE;
	}
	Priv = (void *)buf;
	Priv->PrivilegeCount = 0;
	for ( j = 0; j < sizeof P/sizeof P[0]; ++j )  {
		Priv->PrivilegeCount += 1;
		if ( ! LookupPrivilegeValue( NULL, P[j], &Priv->Privileges[j].Luid ) ) {
			DisplayError( L"NewSID was unable to obtain necessary privilege to change the computer SID");
			return FALSE;
		}
		Priv->Privileges[j].Attributes	= SE_PRIVILEGE_ENABLED;
	}
	if ( ! AdjustTokenPrivileges( Client, FALSE, Priv, 0, NULL, NULL ) ) {
		DisplayError(  L"NewSID was unable to obtain necessary privilege to change the computer SID" );
		return FALSE;
	}

	// 
	// do init stuff
	// 
	InitCommonControls();
	hInst = hInstance;

	//
	// See if the /a automatic switch was passed
	//
	if( !strnicmp( "/a", lpCmdLine, 2 )) {
		
		Auto = TRUE;
	}

	// 
	// Ask the user if they *really* want to change the SID
	// 
	DoSIDChange( Auto );

	//
	// If this is an auto change, see if they want to change
	// the computer name 
	//
	if( Auto ) {

		//
		// See if there's a computer name as well
		//
		i = 2;
		while( lpCmdLine[i] && lpCmdLine[i] == L' ' ) i++;
		if( lpCmdLine[i] ) {

			//
			// We should realy log an error if necessary
			//
			j = 0;
			while( lpCmdLine[i] && lpCmdLine[i] != L' ') {
				
				computerName[j++] = (WCHAR) lpCmdLine[i++];
			}
			computerName[j] = L'';


			if( !SetComputerName(computerName ) ) {
				
				DisplayError( L"NewSID was unable to change the computer name" );
				return -1;
			}
		}
	}

	return 0;
}










