/*
 * Video Capture Driver ( Video for Linux 1/2 )
 * for the Matrox Marvel G200 and Rainbow Runner-G series
 *
 * This module is an interface to the G100 and G200 video extension
 * registers.  
 *
 * Copyright (C) 1999  Ryan Drake <stiletto@mediaone.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#ifndef MGAVIDEO_H
#define MGAVIDEO_H

//#include <linux/i2c.h>
/* video registers */
#define VINCTL0         0x3e00
#define VINCTL1         0x3e04
#define VBIADDR0        0x3e08
#define VBIADDR1        0x3e0c
#define VINADDR0        0x3e10
#define VINADDR1        0x3e14
#define VINNEXTWIN      0x3e18
#define VINCTL          0x3e1c
#define VSTATUS         0x3e30
#define VICLEAR         0x3e34
#define VIEN            0x3e38

/* CODEC registers */
#define CODECCTL        0x3e40
#define CODECADDR       0x3e44
#define CODECHOSTPTR    0x3e48
#define CODECHARDPTR    0x3e4c
#define CODECLCODE      0x3e50

/* Backend scaler registers */
#define BESA1ORG        0x3d00
#define BESA2ORG        0x3d04
#define BESB1ORG        0x3d08
#define BESB2ORG        0x3d0c
#define BESA1CORG       0x3d10
#define BESA2CORG       0x3d14
#define BESB1CORG       0x3d18
#define BESB2CORG       0x3d1c
#define BESCTL          0x3d20
#define BESPITCH        0x3d24
#define BESHCOORD       0x3d28
#define BESVCOORD       0x3d2c
#define BESHISCAL       0x3d30
#define BESVISCAL       0x3d34
#define BESHSRCST       0x3d38
#define BESHSRCEND      0x3d3c
#define BESV1WGHT       0x3d48
#define BESV2WGHT       0x3d4c
#define BESHSRCLST      0x3d50
#define BESV1SRCLST     0x3d54
#define BESV2SRCLST     0x3d58
#define BESGLOBCTL      0x3dc0
#define BESSTATUS       0x3dc4

#define VINCTLx_MASK    0x00000fff
#define VBIADDRx_MASK   0x00fffE00
#define VINADDRx_MASK   0x00fffE00


/* mga video status */
#define VI_IVSYNC    0x0001 /* input vsync interrupt */
#define VI_CMDCMPL   0x0002 /* command complete interrupt */
#define VI_BLVL      0x0004 /* buffer level interrupt */
#define VI_EOI       0x0008 /* codec end of image interrupt */
#define VS_FIELD     0x0100 /* field 0 or 1 detected */
#define VS_VIDEO     0x0200 /* indicates video was captured */
#define VS_RAWVBI    0x0400 /* indicates raw vbi was captured */
#define VS_SLICEDVBI 0x0800 /* indicates sliced vbi was captured */
#define VI_IVSYNC    0x0001 /* input vsync interrupt */
#define VI_CMDCMPL   0x0002 /* command complete interrupt */
#define VI_BLVL      0x0004 /* buffer level interrupt */
#define VI_EOI       0x0008 /* codec end of image interrupt */
#define VS_STALLED   0x1000 /* codec stalled after transfer stopped */

#define	VC_CODECEN	0x0001	/* enable codec interface */
#define	VC_MODE		0x0002	/* codec mode VMI/I33 */
#define	VC_TRIG		0x0004	/* codec command trigger */
#define	VC_DATAIN	0x0008	/* Decompress/Compress */
#define	VC_VMIMODE	0x0010	/* VMI mode A/B */
#define	VC_STOP		0x0020	/* Stop after current field */
#define	VC_TRANSEN	0x0040	/* enable transfer to/from frame buffer */


#define VIN_OFF      0x0
#define VIN_ON       0x1
#define VBI_OFF      0x0
#define VBI_RAW      0x2
#define VBI_SLICED   0x6


/* video attributes - these should probably be configurable on the fly
 *                    so users with different desktop sizes can keep
 *                    captured data off the desktop
 */
#define _VINWID   704
#define _VINHGT   _VINHGT_NTSC
#define _VINHGT_NTSC   240
#define _VINHGT_PAL    290
#define	_VIN_WINDOW	(704 * 291 * 2)
#define	_VBI_WINDOW	(704 * 64 * 2)

#define	_VIN_FIELD_EVEN	1
#define	_VIN_FIELD_ODD	2
#define	_VIN_FIELD_BOTH	4

/****************************************************************************
* mga_dev : represents one mga board.  since we can be dealing with multiple
*           boards, all entry points to this module must specify a board
****************************************************************************/

struct codec {
	int	codec_id;
	int	memory_size;
	void	*private;
	int	state;

	int	(*codec_read_register)(struct codec *codec, void *buffer,int location, int size);
	int	(*codec_write_register)(struct codec *codec, void *buffer,int location, int size);
	int	(*codec_read_memory)(struct codec *codec, void *buffer,int location, int size);
	int	(*codec_write_memory)(struct codec *codec, void *buffer,int location, int size);
	int	(*codec_reset)(struct codec *codec);
	int	(*codec_start)(struct codec *codec);
};

#define	ZR36060_ID	1

struct mga_dev
{
        struct i2c_bus i2c;

        u8* fb;
        u8* ctrl;

        int inuse;
        char name[16];

        /* Device info from pci */
        u32 device;
        int irq;  

        /* caps */
        int hasaudio;
        int hastuner;
        int hasdecoder;

        /* status of the video grabber */
        u32 status;
        u32 locked;

	u32 field_wanted; /* even, odd, interlaced */

        /* registered bottom half of isr */
        struct tq_struct tqnode_dpc;
        int req_even;
        int req_odd;
        int irq_status;

	u32 vin_field;

        /* address for even and odd frames */
        u32 vin_addr0;
        u32 vbi_addr0;

        u32 vin_addr1;
        u32 vbi_addr1;

	u32 vinpitch;


	u8 *interface_base;
	u8 *command_base;
	u8 *read_base;
	u32 host_interface_base;
	u32 codec_offset;

	u32 codecctl_shadow;
	u32 codecmisc_shadow;

        /* statistics */
        int grabbed;

#ifdef	NEVER
	struct	zr36060	zr;
#endif
	struct	codec	zr36060;
	struct	codec	zr36700;
};



struct mga_dev;

/* component device control */
void mgavideo_tuner( struct mga_dev* mga, unsigned int cmd, void* arg );
void mgavideo_audio( struct mga_dev* mga, unsigned int cmd, void* arg );
void mgavideo_decoder( struct mga_dev* mga, unsigned int cmd, void* arg );
void mgavideo_zr36060_reset(struct mga_dev* mga);

#define MGAVIDEO_HAS_TUNER   1
#define MGAVIDEO_HAS_AUDIO   2
#define MGAVIDEO_HAS_DECODER 4
int mgavideo_querycaps( struct mga_dev* mga );

/* get pointer to currently grabbed video */
u8* mgavideo_lock_video( struct mga_dev* mga );
void mgavideo_unlock_video( struct mga_dev* mga );

/* video interrupt control */
void mgavideo_ivsync_enable( struct mga_dev* mga, int en );
void mgavideo_register_bh( struct mga_dev* mga, int req_even, int req_odd,
                           void(*routine)(void*), void* data );

void  mgavideo_set_dims(struct mga_dev *mga, int x, int y);

/* video preview */
void* mgavideo_get_base( struct mga_dev* mga );
void mgavideo_set_window( struct mga_dev* mga, int x, int y, int w, int h );
void mgavideo_set_colorkey( struct mga_dev* mga, int r, int g, int b );
void mgavideo_set_overlay( struct mga_dev* mga );
void mgavideo_preview_enable( struct mga_dev* mga, int en );
                                
/* find next mga adapter */
struct mga_dev* mgavideo_get( void );

/* release a mga adapter that is not needed anymore */
void mgavideo_release( struct mga_dev* mga );

#endif /* MGAVIDEO_H */
