/*
 * dvddecoder: interface to the DVD/MPEG decoder for the DVD Video Navigator
 * Copyright (C) 1999 Christian Wolff for convergence integrated media GmbH
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * Or, point your browser to http://www.gnu.org/copyleft/gpl.html
 * 
 * The author can be reached at scarabaeus@convergence.de, 
 * the project's page is at http://linuxtv.org/dvd/
 */

#include <stdio.h>

#include "dvd_decoder.h"
#include "cvdvext.h"

#define DEFAULTDEVICE      "/dev/msc0"  // LSI DVD card, major 162, minor 0

FILE* device=NULL;  // our character device for the card

struct drawcmd dc;
struct decodercmd decmd;

// Set decoding stream number for video data
// dec_stream: 0 thru 15 (should be 0 for DVD)
// returns 0 on success, 1 on error
int DecoderSetVideoStream(unsigned int dec_stream) {
  if (device==NULL) return 1;
  decmd.param1=dec_stream;
  decmd.cmd=Decoder_Set_VideoStreamID;
  return DecoderCommand(device,decmd);
}

// Set attributes of video data
//   bits: descr.
//   15-14 Video compression mode (0=MPEG-1, 1=MPEG-2)
//   13-12 TV system (0=525/60, 1=625/50)
//   11-10 Aspect ratio (0=4:3, 3=16:9)
//    9- 8 permitted display mode on 4:3 monitor (0=both, 1=only pan-scan, 2=only letterbox)
//    7    line 21-1 data present in GOP (1=yes, 0=no)
//    6    line 21-2 data present in GOP (1=yes, 0=no)
//    5- 3 source resolution (0=720x480/576, 1=704x480/576, 2=352x480/576, 3=352x240/288)
//    2    source letterboxed (1=yes, 0=no)
//    0    film/camera mode (0=camera, 1=film (625/50 only))
// returns 0 on success, 1 on error
int DecoderSetVideoAttr(unsigned int atr) {
  if (device==NULL) return 1;
  decmd.param1=atr;
  decmd.cmd=Decoder_Set_Videoattribute;
  return DecoderCommand(device,decmd);
}

// Set decoding stream number for audio data
// dec_stream: 0 thru 7, or 15 for audio disable
// extension: 
//   0: MPEG1 Audio or MPEG2 Audio without extension stream
//   1: MPEG2 Audio with extension stream
// returns 0 on success, 1 on error
int DecoderSetAudioStream(unsigned int dec_stream, int extension) {
  if (device==NULL) return 1;
  if (dec_stream<=7) {
    decmd.param1=dec_stream&0x07;
    decmd.param2=((extension)?0x10|(dec_stream&0x07):-1);
  } else {
    decmd.param1=-1;
    decmd.param2=-1;
  }
  decmd.cmd=Decoder_Set_AudioStreamID;
  return DecoderCommand(device,decmd);
}

// Set attributes of audio data
//   bits: descr.
//   15-13 audio coding mode (0=ac3, 2=mpeg1, 3=mpeg2ext, 4=LPCM, 6=DTS, 7=SDDS)
//   12    multichannel extension
//   11-10 audio type (0=not spec, 1=language included)
//    9- 8 audio application mode (0=not spec, 1=karaoke, 2=surround)
//    7- 6 Quantization / DRC (mpeg audio: 1=DRC exists)(lpcm: 0=16bit, 1=20bit, 2=24bit)
//    5- 4 Sample frequency fs (0=48kHz, 1=96kHz)
//    2- 0 number of audio channels (n+1 channels)
// returns 0 on success, 1 on error
int DecoderSetAudioAttr(unsigned int atr) {
  if (device==NULL) return 1;
  decmd.param1=atr;
  decmd.cmd=Decoder_Set_Audioattribute;
  return DecoderCommand(device,decmd);
}

// Set decoding stream number for SPU data
// dec_stream, bit 0 thru 5: 0 thru 31, or 63 for SPU disable
// dec_stream, bit 6: 1=display SPU stream, 0=decode, but don't display
// returns 0 on success, 1 on error
int DecoderSetSubPictureStream(unsigned int dec_stream) {
  if (device==NULL) return 1;
  decmd.cmd=Decoder_SPU;
  decmd.param1=dec_stream&0x001F;
  decmd.param2=(dec_stream&0x0040);
  return DecoderCommand(device,decmd);
}

// Set palette information for SPU
// data has to have colors*3 bytes: y,cr,cb
// returns 0 on success, 1 on error
int DecoderSetSubPicturePalette(int colors, unsigned char *data) {
  if (device==NULL) return 1;
  decmd.cmd=Decoder_SPU_Palette;
  decmd.param1=colors*3;
  decmd.data1=data;
  return DecoderCommand(device,decmd);
}


// Set karaoke mode
// mixmask: bitmap to set source to mix into left/right channel
//     Bit 2:  0=don't mix, 1=mix Ch2 to Ch1 (Melody? to Right?)
//     Bit 3:  0=don't mix, 1=mix Ch3 to Ch1 (Voice 1? to Right?)
//     Bit 4:  0=don't mix, 1=mix Ch4 to Ch1 (Voice 2? to Right?)
//     Bit 10: 0=don't mix, 1=mix Ch2 to Ch0 (Melody? to Left?)
//     Bit 11: 0=don't mix, 1=mix Ch3 to Ch0 (Voice 1? to Left?)
//     Bit 12: 0=don't mix, 1=mix Ch4 to Ch0 (Voice 2? to Left?)
// returns 0 on success, 1 on error
int DecoderSetKaraokeMix(unsigned int mixmask) {
  if (device==NULL) return -1;
  decmd.cmd=Decoder_SetKaraoke;
  decmd.param1=((mixmask&0x0404)?1:0);
  decmd.param2=((mixmask&0x0808)?1:0);
  decmd.param3=((mixmask&0x1010)?1:0);
  return DecoderCommand(device,decmd);
}

// highlighting SPU button
// active:
//   1=show highlight, 0=hide highlight
// coli: color information(SL_COLI or AC_COLI) MSB first 
//   bits:  descr.
//   31-28  Emphasis pixel-2 color
//   27-24  Emphasis pixel-1 color
//   23-20  Pattern pixel color
//   19-16  Background pixel color
//   15-12  Emphasis pixel-2 contrast
//   11- 8  Emphasis pixel-1 contrast
//    7- 4  Pattern pixel contrast
//    3- 0  Background pixel contrast
// posi: button position(BTN_POSI) MSB first
//   bits:  descr.
//   47-46  button color number
//   45-36  start x
//   33-24  end x
//   23-22  auto action mode
//   21-12  start y
//    9- 0  end y
// returns 0 on success, 1 on error
int DecoderHighlight(int active, unsigned char *coli, unsigned char *posi) {
  if (device==NULL) return 1;
  decmd.cmd=Decoder_Highlight;  // set highlight information
  decmd.param1=active;  // show highlight information?
  decmd.data1=coli;  // color & contrast
  decmd.data2=posi;  // where to highlight?
  return DecoderCommand(device,decmd);
}

// put decoder into or out of pause
// cmd: 0=pause off, 1=pause on, 2=pause toggle
// returns 0 on success, 1 on error
int DecoderPause(int cmd) {
  if (device==NULL) return 1;
  decmd.cmd=Decoder_Pause;
  decmd.param1=cmd;
  return DecoderCommand(device,decmd);
}

// retreive navigation pack parsed from the stream
// data will be filled with 1024 bytes of DSI or PCI data
// returns 1024 on success, 0 on error
int DecoderGetNaviPack(unsigned char *data) {
  if (device==NULL) return 0;
  decmd.cmd=Decoder_GetNavi;  // retreive navi pack from the channel buffer
  decmd.data1=data;
  return DecoderCommand(device,decmd);
}

// Bypass CSS module, use data as-is
// returns 0 on success, -1 on error
int DecoderCSSBypass(void) {
  if (device==NULL) return -1;
  decmd.param1=0;  // DVD Decoder API: bypass and disable CSS module
  decmd.cmd=Decoder_CSS;
  return DecoderCommand(device,decmd);
}  

// Retreive host challenge for drive
// disc: 1=disc key auth, 0=title key auth
// data: will be filled with 10 bytes key data
// returns 0 on success, -1 on error
int DecoderCSSDriveAuthChallenge(int disc, char *data) {
  if (device==NULL) return -1;
  decmd.param1=((disc)?1:5);  // DVD Decoder API: retreive drive challenge
  decmd.data1=data;
  decmd.cmd=Decoder_CSS;
  return DecoderCommand(device,decmd);
}  

// Send drive response to host
// disc: 1=disc key auth, 0=title key auth
// data: 5 bytes key data
// returns 0 on success, -1 on error
int DecoderCSSDriveAuthResponse(int disc, char *data) {
  if (device==NULL) return -1;
  decmd.param1=((disc)?2:6);  // DVD Decoder API: post drive response
  decmd.data1=data;
  decmd.cmd=Decoder_CSS;
  return DecoderCommand(device,decmd);
}  

// Send drive challenge to host
// and retreive host response for drive
// disc: 1=disc key auth, 0=title key auth
// data: 10 bytes key data, will be filled with 5 bytes key data
// returns 0 on success, -1 on error
int DecoderCSSHostAuth(int disc, char *data) {
  if (device==NULL) return -1;
  decmd.param1=((disc)?3:7);  // DVD Decoder API: post card challenge
  decmd.data1=data;
  decmd.cmd=Decoder_CSS;
  return DecoderCommand(device,decmd);
}  

// Send disc key to host
// data: 2048 bytes key data
// returns 0 on success, -1 on error
int DecoderCSSDiscKey(char *data) {
  if (device==NULL) return -1;
  decmd.param1=4;  // DVD Decoder API: post disc key
  decmd.data1=data;
  decmd.cmd=Decoder_CSS;
  return DecoderCommand(device,decmd);
}  

// Send title key to host
// data: 5 bytes key data
// returns 0 on success, -1 on error
int DecoderCSSTitleKey(char *data) {
  if (device==NULL) return -1;
  decmd.param1=8;  // DVD Decoder API: post title key
  decmd.data1=data;
  decmd.cmd=Decoder_CSS;
  return DecoderCommand(device,decmd);
}  

// sends one Logical Block to the DVD Decoder
// data: pointer to data to be sent to the decoder
// size: amount of bytes in data
// initial: 1 if first block of a stream
// returns 0 on success, 1 on error
int DecoderWrite(unsigned char *data, int size, int initial) {
  if (device==NULL) return 1;
  decmd.cmd=Decoder_WriteBlock;
  decmd.param1=size;
  decmd.param2=initial;
  decmd.param3=initial;
  decmd.data1=data;
  if (DecoderCommand(device,decmd)) return 1;
  return 0;
}

// Set the path to the device that displays the DVD data
// This file has to accept the I/O-Controls defined in this
// convergence DVD Decoder API: http://linuxtv.org/dvd/api/
// returns 0 on success, 1 on error
int DecoderSetDevice(char *Path) {
  if (strlen(Path)==0) {
    if ((device=fopen(DEFAULTDEVICE,"w"))==NULL) return 1;
  } else {
    if ((device=fopen(Path,"w"))==NULL) return 1;
  }
  decmd.cmd=Decoder_Set_Streamtype;
  decmd.param1=stream_DVD;  // DVD stream with SPU/Navi/etc.
  DecoderCommand(device,decmd);
  return 0;
}


