/*************************************************************/
/*  DVDPlayerAPI.h - Linux DVD Player Application Interface  */
/*    in accordance with Annex J of the DVD specifications   */
/*    Written 1999/07/28 by Christian Wolff of               */
/*    convergence integrated media GmbH, Berlin              */
/*************************************************************/

// As of 07/September/1999

#ifndef _DVD_PLAYER_API_H_
#define _DVD_PLAYER_API_H_

#define DVD_PLAYER_API_VERSION 1

#define Time2TimeCode(hour,min,sec,frame,rate) (\
  ((((hour) /10)<<28)|(((hour) %10)<<24)&0xFF000000L)|\
  ((((min)  /10)<<20)|(((min)  %10)<<16)&0x00FF0000L)|\
  ((((sec)  /10)<<12)|(((sec)  %10)<<8 )&0x0000FF00L)|\
  ((((frame)/10)<<4 )|( (frame)%10)     &0x0000003FL)|\
  (((rate)<<6)&0x000000C0L)\
)

#define TimeCode2Time(bcdtime,hour,min,sec,frame,rate) {\
  hour =((bcdtime>>28)&0x0F)*10+((bcdtime>>24)&0x0F);\
  min  =((bcdtime>>20)&0x0F)*10+((bcdtime>>16)&0x0F);\
  sec  =((bcdtime>>12)&0x0F)*10+((bcdtime>>8 )&0x0F);\
  frame=((bcdtime>>4 )&0x03)*10+( bcdtime     &0x0F);\
  rate = (bcdtime>>6 )&0x03; /* 1=25fps, 3=30fps */\
}

typedef enum DVDDomain {
  DVDDomainVideoManagerMenu=0x01,
  DVDDomainFirstPlay=0x02,
  DVDDomainVideoTitleSetMenu=0x04,
  DVDDomainTitle=0x08,
  DVDDomainStop=0x10
} DVDDomainID;
     
typedef enum MenuCall {
  MenuIDNone=0,
  MenuIDTitle=2,
  MenuIDRoot=3,
  MenuIDSubPicture=4,
  MenuIDAudio=5,
  MenuIDAngle=6,
  MenuIDPart=7
} MenuCallID;

#define DVD_PLAYER_OK 0              /* Successful */
#define DVD_PLAYER_ERROR 1           /* Unknown or general error */
#define DVD_PLAYER_INVALID 2         /* Invalid parameter */
#define DVD_PLAYER_NOT_AVAILABLE 3   /* Feature or function not available */
#define DVD_PLAYER_WRONG_DOMAIN 4    /* Player is in wrong state for this command */
#define DVD_PLAYER_NOT_FOUND 5       /* Device, path, or file not found */



/* Command functions */

/* Play the specified title of the DVD */
/*   Title:  1..99 */
int TitlePlay(int Title);

/* Play the specified title of the DVD, 
   from the specified part on */
/*   Title:  1..99 */
/*   Part:   1..999 */
int PartPlay(int Title, int Part);

/* Play the specified amount of parts 
   of the specified title of the DVD, 
   from the specified part on */
/*   Title:  1..99 */
/*   Part:   1..999 */
/*   PartsToPlay:   1..999 */
int PartPlayAutoStop(int Title, int Part, int PartsToPlay);

/* Play the specified title of the DVD, 
   from the specified time on */
/*   Title:  1..99 */
/*   Time: hours, minutes, seconds, and frames */
int TimePlay(int Title, unsigned long int TimeCode);


/* Stop playing current title */
int Stop(void);

/* Stop playing current title and 
   play the "GoUp"-program chain */
int GoUp(void);

/* Stop playing current title and 
   play current title from the specified timecode on */
int TimeSearch(unsigned long int TimeCode);

/* Stop playing current title and 
   play current title from the specified part on */
/*   Part:   1..999 */
int PartSearch(int Part);

/* Stop playing current title and 
   play current title from the previous program on */
int PrevPGSearch(void);

/* Stop playing current title and 
   play current title from the first program on */
int TopPGSearch(void);

/* Stop playing current title and 
   play current title from the next program on */
int NextPGSearch(void);


/* Scan play forward at specified speed */
/*   0 < speed < 1 : slow motion
     speed > 1 : fast forward play */
int ForwardScan(float Speed);

/* Scan play backward at specified speed */
/*   0 < speed < 1 : slow motion
     speed > 1 : fast reverse play */
int BackwardScan(float Speed);

/* proposal: 
   make the function to accept negative values 
   for backward scan - more nifty ;-) */
#define ScanPlay(Speed) ((Speed<0)?BackwardScan(-1*Speed):ForwardScan(Speed))

/* Call specified Menu */
/*   MenuIDTitle:      call title menu
     MenuIDRoot:       call root menu
     MenuIDAudio:      call audio menu
     MenuIDSubPicture: call sub picture menu
     MenuIDAngle:      call angle menu
     MenuIDPart:       call title part menu */
int MenuCall(MenuCallID MenuID);

/* End menu mode*/
int Resume(void);

/* Menu action of user arrow keys, 
   moving highlight to next button */
int UpperButtonSelect(void);
int LowerButtonSelect(void);
int LeftButtonSelect(void);
int RightButtonSelect(void);

/* Activate (press) highlighted button */
int ButtonActivate(void);

/* Activate (press) specified button */
/*   Button: 1..36 */
int ButtonSelectAndActivate(int Button);

/* Select button at specified coordinates */
int MouseSelect(int x, int y);

/* Activate (press) button at specified coordinates */
int MouseActivate(int x, int y);


/* Release still mode */
int StillOff(void);

/* Enter pause mode and display still frame*/
int PauseOn(void);

/* Release pause mode */
int PauseOff(void);


/* Set specified language for the menus */
/*   LanguageCode: 2 characters ISO 639 Language Code */
int MenuLanguageSelect(char *LanguageCode);


/* Select specified audio stream */
/*   Audio: 0..7 */
int AudioStreamChange(int Audio);

/* Select specified sub picture stream and
   set if sub picture should be displayed or not*/
/*   Subpicture: 0..31 or 63 (dummy stream) */
/*   Display: 0 or 1 */
int SubpictureStreamChange(int Subpicture, int Display);

/* Select specified angle */
/*   Angle: 1..9 */
int AngleChange(int Angle);

/* Set the level of the parental management system */
/*   ParentalLevel: 0 (disabled) or 1..8 */
int ParentalLevelSelect(int ParentalLevel);

/* Set the country of the parental management system */
/*   CountryCode: 2 characters ISO 3166 Alpha 2 Country Code*/
int ParentalCountrySelect(char *CountryCode);

/*   MixMode: bitmap to set source to mix into left/right channel
     Bit 2:  0=don't mix, 1=mix Ch2 to Ch1 (Melody? to Right?)
     Bit 3:  0=don't mix, 1=mix Ch3 to Ch1 (Voice 1? to Right?)
     Bit 4:  0=don't mix, 1=mix Ch4 to Ch1 (Voice 2? to Right?)
     Bit 10: 0=don't mix, 1=mix Ch2 to Ch0 (Melody? to Left?)
     Bit 11: 0=don't mix, 1=mix Ch3 to Ch0 (Voice 1? to Left?)
     Bit 12: 0=don't mix, 1=mix Ch4 to Ch0 (Voice 2? to Left?) */
int KaraokeAudioPresentationModeChange(int MixMode);

/* Set the Video display presentation mode */
/*   InitialDisplay (...aspect ratio): 0=4:3, 1=not specified 3=16:9
     CurrentDisplay (...mode): 0=Normal (4:3 or 16:9), 1=Pan-Scan, 2=Letterbox */
int VideoPresentationModeChange(int InitialDisplay, int CurrentDisplay);

/* Set the path to the UDF mounted DVD disc */
int SetRoot(char *Path);

/* Set the path to the device that displays the DVD data */
/* This file has to accept the I/O-Controls defined in this */
/* convergence DVD Device API http://linuxtv.org/dvd/api/   */
int SetDevice(char *Path);




/* Informational functions */

/* retrieves content of GPRM[0] thru GPRM[15] in RegisterArray */
/* see DVD-Video Specifications for format */
int GetAllGPRMs(unsigned int *RegisterArray);

/* retrieves content of SPRM[0] thru SPRM[20] in RegisterArray */
/* see DVD-Video Specifications for format */
int GetAllSPRMs(unsigned int *RegisterArray);


/* retrieves 2 bytes video attributes of the current title */
/* */ 
int GetCurrentVideoAttributes(unsigned char *ATR);


/* retrieves 3 characters in LanguageCode: 2 characters ISO 639 or 0xFFXX, */
/* one character extension */
int GetAudioLanguage(int Stream, char *LanguageCode);

/* retrieves number of audio streams in this title and the currently selected one */ 
int GetCurrentAudio(int *StreamsAvailable, int *CurrentStream);

/* retrieves 8 bytes audio attributes of the current title */
/* */ 
int GetCurrentAudioAttributes(unsigned char *ATR);


/* retrieves number of audio streams in this title and the currently selected one */ 
/* IsDisabled is set to 1 if display of stream is disabled, otherwise to 0 */
int GetCurrentSubpicture(int *StreamsAvailable, int *CurrentStream, int *IsDisabled);

/* retrieves 6 bytes sub-picture attributes of the current title */
/* */ 
int GetCurrentSubpictureAttributes(unsigned char *ATR);

/* retrieves 3 characters in LanguageCode: */
/* 2 characters ISO 639 or 0xFFXX, */
/* one character extension */
int GetSubpictureLanguage(int Stream, char *LanguageCode);


/* retrieves number of angles in this title and the currently selected one */ 
int GetCurrentAngle(int *AnglesAvailable, int *CurrentAngle);

/* retrieves number of buttons in this section and the currently selected one */ 
int GetCurrentButton(int *ButtonsAvailable, int *CurrentButton);

/* retrieves the current domain the player is in */
int GetCurrentDomain(DVDDomainID *Domain);

/* retrieves current title and part number, as well as timecode of current play location */
/* TimeCode: hours, minutes, seconds, and frames
   bit 31-24 contain the hour value as 2 digit BCD,
   bit 23-16 contain the minute value as 2 digit BCD,
   bit 15-8 contain the second value as 2 digit BCD,
   bit 5-0 contain the frame value as 1.5 digit BCD,
   bit 7-6 contain the frame rate: 01=25fps, 11=30fps */
int GetCurrentLocation(int *TitleNum, int *PartNum, unsigned long int *TimeCode);


/* retrieves bit table of prohibited user operations */
/* if a bit is set, the corresponding action is prohibited */
/* Bit: Flag:
    0   Title Or Time Play
    1   Part Search Or Play
    2   Title Play
    3   Stop
    4   GoUp
    5   Time Or Part Search
    6   Prev Or Top PG Search
    7   Next PG Search
    8   Forward Scan
    9   Backward Scan
   10   Title Menu Call
   11   Root Menu Call
   12   SubPic Menu Call
   13   Audio Menu Call
   14   Angle Menu Call
   15   Part Menu Call
   16   Resume
   17   Button Select Or Activate
   18   Still Off
   19   Pause On
   20   Audio Stream Change
   21   SubPic Stream Change
   22   Angle Change
   23   Karaoke Audio Pres Mode Change
   24   Video Pres Mode Change */
int GetCurrentUOPS(unsigned long int *UOP);


/* retrieves information about the currently inserted disc */
/* NumOfVol: Total number of volumes in this volume set */
/* ThisVolNum: Volume number of this disc */
/* Side: 1=Side A, 2=Side B */
/* NumOfTitles: number of titles on this disc */
int GetCurrentVolumeInfo(int *NumOfVol, int *ThisVolNum, int *Side, int *NumOfTitles);

/* retreives the txtdt_mg structure of the disc. */
/* if the buffersize, indicated by BufSize, is too small, DVD_PLAYER_INVALID is returned 
   and the required amount of bytes is put into ActualSize */
int GetDVDTextInfo(char *TextManager, int BufSize, int *ActualSize);

/* retrieves the number of Parts of a specified title */
/* Title: 1..99 */
int GetNumberOfParts(int Title, int *NumberOfParts);

/* retrieves the parental settings of the player */
/* ParentalLevel: parental level the player is set to */
/* CountryCode: 2 bytes ISO 639 country code of the player, or 0xFFFF if not set */
int GetPlayerParentalLevel(unsigned int *ParentalLevel, unsigned int *CountryCode);

/* retrieves the combined parental settings of the title */
/* Title: 1..99 */
int GetTitleParentalLevels(int Title, unsigned int *ParentalLevels);

/* retreives 768 bytes (256 trough 1023) of the VMGI_MAT of the current disc */
/* see DVD-Video Specifications for format */
int GetVMGAttributes(unsigned int *ATR);

/* retreives 768 bytes (256 trough 1023) of the VTSI_MAT of the specified title */
/* Title: -1 for current title, or 1..99 */
/* see DVD-Video Specifications for format */
int GetTitleAttributes(int Title, unsigned int *ATR);

/* retreives playback time of the current title */
/* TotalTime: hours, minutes, seconds, and frames
   bit 31-24 contain the hour value as 2 digit BCD,
   bit 23-16 contain the minute value as 2 digit BCD,
   bit 15-8 contain the second value as 2 digit BCD,
   bit 5-0 contain the frame value as 1.5 digit BCD,
   bit 7-6 contain the frame rate: 01=25fps, 11=30fps */
int GetTotalTitleTime(unsigned long int *TotalTime);

/* retreives current path to the DVD disc */
int GetRoot(char *Root, int BufSize, int *ActualSize);


#endif  /* _DVD_PLAYER_API_H_ */
